/*+*****************************************************************************
*                                                                              *
* File: iconn.c                                                                *
*                                                                              *
* Description: incoming connection handling                                    *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char iconn_vers[] = "@(#)iconn.c	1.11	02/01/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "iconn.h"
#include "util.h"
#include "io.h"
#include "host.h"
#include "xs.h"
#include "xmd.h"
#include "ident.h"

/******* Constants ************************************************************/

/*
 * the following limits are mainly to prevent denial of service attacks
 */
#define MAX_ICONN	10	/* maximum pending ICONN allowed */
#define MAX_HOST_ICONN	3	/* maximum pending ICONN allowed per host */

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

static int iconn_count = 0;	/* number of pending ICONN */
static ICONN *iconn_first = NULL; /* list of pending ICONN */

/*
 * create an ICONN for an incoming connection that has been detected
 */
ICONN *iconn_create(int local, int ident)
{
  int newfd, count;
  HOST *newhost;
  ICONN *ic, *c;

  /* accept */
  if (local) newfd = xs_accept_local();
  else       newfd = xs_accept(&newhost);
  /* the error has been already reported by xs_accept* so simply ... */
  if (newfd < 0) return(NULL);

  /* create the structure */
  ic = NEW(ICONN);
  ic->fd = newfd;
  if (local) {
    ic->host  = localhost;
    ic->name  = NULL;
    ic->local = 1;
  } else {
    ic->host  = newhost;
    ic->name  = ident ? ident_name(newfd) : NULL;
    ic->local = 0;
  }
  ic->icf = 0;
  iconn_count++;
  /* add it at the beginning of the linked list */
  if (iconn_first) iconn_first->prev = ic;
  ic->next = iconn_first;
  ic->prev = NULL;
  iconn_first = ic;

  /* check the total limit */
  if (iconn_count > MAX_ICONN) {
    iconn_refuse(ic, "Too many pending incoming connections");
    return(NULL);
  }

  /* check the per-host limit */
  count = 0;
  for (c=iconn_first; c; c=c->next)
    if (!strcmp(ic->host->ip_name, c->host->ip_name)) count++;
  if (count > MAX_HOST_ICONN) {
    iconn_refuse(ic, "Too many per-host pending incoming connections");
    return(NULL);
  }

  /* OK */
  return(ic);
}

/*
 * destroy an ICONN but does not kill the connection
 * (it is also called when the connection is accepted...)
 */
void iconn_destroy(ICONN *ic)
{
  /* destroy data */
  if (ic->name) str_free(ic->name);
  /* update the linked list */
  if (ic->next) ic->next->prev = ic->prev;
  if (ic->prev) ic->prev->next = ic->next;
  else iconn_first = ic->next;
  /* don't close fd or destroy host here */
  OLD(ic);
  iconn_count--;
}

/*
 * send a forged connection refused packet
 */
static void iconn_send_refuse(int fd, char *reason)
{
  char buf[IO_BUFSIZE];
  U8BITS sex;
  int got, len;

  /* check reason length */
  len = strlen(reason);
  if (len > 255) die("reason too long in iconn_send_refuse: %d bytes", len);

  /* read data from client */
  /* Xproto.h/xConnClientPrefix:
     CARD8	byteOrder;
     BYTE	pad;
     CARD16	majorVersion B16, minorVersion B16;
     CARD16	nbytesAuthProto B16;
     CARD16	nbytesAuthString B16;
     CARD16	pad2 B16;
     */
  got = io_timeout_read(1, fd, buf, IO_BUFSIZE);
  if (got < 12) return;

  /* forge a server answer */
  /* Xproto.h/xConnSetupPrefix:
     CARD8          success;
     BYTE           lengthReason;
     CARD16         majorVersion B16, minorVersion B16;
     CARD16         length B16;
     */
  sex = getCARD8(buf, 0);
  putCARD8(buf, 0, 0);
  putCARD8(buf, 1, (U8BITS)(len & 0xFF));
  /* we leave intact the version numbers */
  len = PAD4(len);
  putCARD16(sex, &buf[6], len >> 2);
  strcpy(&buf[8], reason);
  io_timeout_write(3, fd, buf, 8 + len);
}

/*
 * destroy an ICONN and kill the connection
 */
void iconn_refuse(ICONN *ic, char *why)
{
#ifdef DEBUG
  if (DEBUGGING(DBG_CONN))
    dprintf("conn", "refused connection from %s: %s", ic->host->name, why);
#endif
  iconn_send_refuse(ic->fd, why);
  close(ic->fd);
  if (!ic->local) host_destroy(ic->host);
  iconn_destroy(ic);
}
