/*+*****************************************************************************
*                                                                              *
* File: host.c                                                                 *
*                                                                              *
* Description: manipulation of host names and addresses                        *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char host_vers[] = "@(#)host.c	1.13	02/19/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "host.h"
#include "util.h"
#include "xmd.h"
#include "sb.h"
#include "err.h"

#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <ctype.h>

#ifdef SOLARIS
#include <sys/systeminfo.h>
#define gethostname(buf,size)	sysinfo(SI_HOSTNAME, buf, size)
#endif /* SOLARIS */

#if !defined(INADDR_NONE) && defined(INADDR_BROADCAST)
#define INADDR_NONE INADDR_BROADCAST
#endif

/******* Constants ************************************************************/

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

HOST *localhost = NULL;

/******* Local Stuff **********************************************************/

static char *empty_string = "";

/*
 * return a printable host name from a 32-bits address
 * shouldn't use U32BITS here but in_addr_t is not standard...
 */
char *host_name(U32BITS addr)
{
  struct in_addr in;

  in.s_addr = addr;
  return(inet_ntoa(in));
}

/*
 * return the DNS domain name from /etc/resolv.conf
 */
static char *host_domain(void)
{
  FILE *fp;
  char *cp, *found;

  fp = fopen("/etc/resolv.conf", "r");
  if (!fp) return(NULL);
  found = NULL;
  while (fgets(sb_line, SB_LINE_LENGTH, fp)) {
    if (!strncasecmp(sb_line, "domain", 6) ||
	!strncasecmp(sb_line, "search", 6)) {
      cp = sb_line + 6;
      /* skip spaces */
      while (*cp && isspace((int)*cp)) cp++;
      if (*cp) {
	/* got something after the spaces, skip it */
	found = cp;
	while (*cp && !isspace((int)*cp)) {
	  *cp = tolower((int)*cp);
	  cp++;
	}
	/* and truncate */
	*cp = '\0';
	break;
      }
    }
  }
  fclose(fp);
  return(found ? str_copy(found) : NULL);
}

/*
 * create a HOST from a struct hostent
 */
static HOST *host_create_by_ent(struct hostent *hep, char *hname)
{
  char *cp;
  HOST *h;
#ifndef DEBUG
  hname = hname; /* unused parameter */
#endif

  /* check what we're given */
  if (hep == NULL) {
    err_report(NULL, "unknown host: ", hname, 0);
    return((HOST *)NULL);
  }
  if ((hep->h_addrtype != AF_INET) ||
      (hep->h_length != sizeof(struct in_addr))) {
    err_report(NULL, hname, " on unsupported network", 0);
    return((HOST *)NULL);
  }

  /* create a HOST */
  h = NEW(HOST);
  h->short_name = str_copy((char *)hep->h_name);
  memmove(&h->addr.s_addr, hep->h_addr, sizeof(h->addr.s_addr));

  /* copy the IP name in static buffer */
  strncpy(h->ip_name, inet_ntoa(h->addr), 15);
  h->ip_name[15] = '\0';

  /* what about a domain name? */
  cp = strchr(h->short_name, '.');
  if (cp == NULL) {
    /* not fully qualified so it must be local */
    if (localhost && localhost->domain != empty_string) {
      /* we can get the fully qualified name using localhost */
      h->fq_name = safe_malloc(strlen(h->short_name) + strlen(localhost->domain) + 2);
      sprintf(h->fq_name, "%s.%s", h->short_name, localhost->domain);
      h->domain = localhost->domain;
    } else {
      /* oops, can't get the fully qualified name */
      h->fq_name = h->short_name;
      h->domain = empty_string;
    }
  } else {
    /* fully qualified */
    h->fq_name = str_copy(h->short_name);
    *cp = '\0';
    h->domain  = ++cp;
  }

  /* "meaningful" name guessing */
  if (localhost == NULL)
    h->name = h->short_name;
  else if ((localhost->domain != empty_string) && !strcasecmp(h->domain, localhost->domain))
    h->name = h->short_name;
  else
    h->name = h->fq_name;

  return(h);
}

/*
 * create a HOST from a struct in_addr
 */
HOST *host_create_by_addr(struct in_addr *addr)
{
  struct hostent *hep;
  HOST *h;
  char *cp;

  hep = gethostbyaddr((const char *)addr, sizeof(struct in_addr), AF_INET);
  h = host_create_by_ent(hep, inet_ntoa(*addr));
  if (h == NULL) return(NULL);

  /* reverse name lookup to be safe */
  hep = gethostbyname(h->fq_name);
  if (hep == NULL) {
    err_report(NULL, "reverse name lookup failed for ", h->fq_name,
	       " [", inet_ntoa(*addr), "]", 0);
    host_destroy(h);
    return(NULL);
  }
  cp = hep->h_addr_list[0];
  while (cp != NULL) {
    if (memcmp(cp, (char *)addr, sizeof(*addr)) == 0)
      return(h);
    hep->h_addr_list++;
    cp = hep->h_addr_list[0];
  }
  err_report(NULL, "reverse name lookup mismatch for ", h->fq_name,
	     " [", inet_ntoa(*addr), "]", 0);
  host_destroy(h);
  return(NULL);
}

/*
 * create a HOST from a name
 */
HOST *host_create_by_name(char *name)
{
  struct in_addr ia;
  struct hostent *hep;

  /* special case for numerical address */
  ia.s_addr = inet_addr(name);
  if (ia.s_addr != INADDR_NONE) return(host_create_by_addr(&ia));

  /* normal case */
  hep = gethostbyname(name);
  return(host_create_by_ent(hep, name));
}

/*
 * destroy a HOST if it's not localhost
 */
void host_destroy(HOST *h)
{
  if (h == localhost) return;
  str_free(h->short_name);
  if (h->short_name != h->fq_name) str_free(h->fq_name);
  OLD(h);
}

/*
 * setup this module, find localhost name
 */
void host_setup(void)
{
  int res;

  /* create localhost */
  res = gethostname(sb_line, SB_LINE_LENGTH);
  if (res < 0) die("can't get local hostname: %s", ERROR);
  localhost = host_create_by_name(sb_line);
  if (!localhost) die("can't create localhost");
  /* try hard to figure out the domain name */
  if (localhost->domain == empty_string) {
    char *domain = host_domain();
#ifdef DEBUG
    if (DEBUGGING(DBG_MISC))
      dprintf("misc", "guessed domain is %s", (domain ? domain : "unknown"));
#endif
    if (domain) {
      int len = strlen(localhost->name);
      localhost->fq_name = (char *)safe_malloc(len + strlen(domain) + 2);
      sprintf(localhost->fq_name, "%s.%s", localhost->name, domain);
      localhost->domain = localhost->fq_name + len + 1;
      str_free(domain);
    }
  }
#ifdef DEBUG
  if (DEBUGGING(DBG_MISC))
    dprintf("misc", "localhost=%s, fq=%s, domain=%s, ip=%s",
	    localhost->name, localhost->fq_name,
	    localhost->domain, localhost->ip_name);
#endif
}
