/*+*****************************************************************************
*                                                                              *
* File: conn.c                                                                 *
*                                                                              *
* Description: connection handling                                             *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char conn_vers[] = "@(#)conn.c	1.11	03/16/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "conn.h"
#include "util.h"
#include "auth.h"
#include "xp.h"
#include "xmd.h"
#include "iconn.h"

#include <time.h>

/******* Constants ************************************************************/

/*
 * client I/O state
 */
#define CS_SETUPH	'i'	/* setup header */
#define CS_SETUP	's'	/* setup */
#define CS_REQH		'h'	/* request header */
#define CS_REQ		'r'	/* request */
#define CS_REQV		'v'	/* verified request */
#define CS_COPY		'c'	/* copy */
#define CS_DISCARD	'd'	/* discard */

/*
 * server I/O state
 */
#define SS_SETUPH	'I'	/* setup header */
#define SS_SETUP	'S'	/* setup */
#define SS_PKTH		'H'	/* packet header */
#define SS_PKT		'P'	/* packet */
#define SS_COPY		'C'	/* copy */
#define SS_DISCARD	'D'	/* discard */

/******* Macros ***************************************************************/

/*
 * handy macros to extract packet information
 */
#define BUF16(pos)       getCARD16(cop->xinfo.sex, &buf[pos])
#define BUF32(pos)       getCARD32(cop->xinfo.sex, &buf[pos])

/*
 * print the X packet if debugging X
 */
#ifdef DEBUG
#define PRINTX(type)	if (DEBUGGING(DBG_X)) xp_print(type, buf, iop->tag, &cop->xinfo)
#else
#define PRINTX(type)	
#endif

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

static CONN *conn_first = NULL, *conn_last = NULL;
static int   conn_id = 1; /* almost uniq id within 0 - 32767 */
static char *conn_id_chars = "0123456789ABCDEFGHJKMNPRSTUVWXYZ";
#ifdef DEBUG
static int conn_count = 0;
static int conn_alerts = 0;
static float conn_elapsed = 0;
static float conn_pcount = 0;
static float conn_psize = 0;
#endif

/*
 * initialise a CONN structure
 */
void conn_init(CONN *cop, ICONN *icop, int server_fd)
{
  /* init */
  cop->host = icop->host;
  io_init(&cop->client, icop->fd, server_fd);
  io_needs(&cop->client, CS_SETUPH, 12); /* need 12 bytes for the setup header */
  io_init(&cop->server, server_fd, icop->fd);
  io_needs(&cop->server, SS_SETUPH, 8);  /* need  8 bytes for the setup header */
  ui_init(&cop->ui);
  cop->start = cop->idle_since = time((time_t *)NULL);
  cop->flags = icop->icf;
  xp_init(&cop->xinfo);
  /* add at the end */
  cop->prev = conn_last;
  cop->next = NULL;
  if (conn_last) conn_last->next = cop;
  else conn_first = cop;
  conn_last = cop;
  /* set id, tags & name */
  cop->id = &cop->tag[1];
  cop->id[0] = conn_id_chars[(conn_id & 0x7C00)>>10];
  cop->id[1] = conn_id_chars[(conn_id & 0x03E0)>>5];
  cop->id[2] = conn_id_chars[(conn_id & 0x001F)];
  cop->id[3] = '\0';
  cop->tag[0]        = '=';
  cop->client.tag[0] = 'c'; strcpy(&cop->client.tag[1], cop->id);
  cop->server.tag[0] = 's'; strcpy(&cop->server.tag[1], cop->id);
  if (icop->name) {
    cop->name = (char *)safe_malloc(2 + strlen(icop->name) + strlen(cop->host->name));
    sprintf(cop->name, "%s@%s", icop->name, cop->host->name);
  } else {
    cop->name = cop->host->name;
  }
  /* increment id and check overflow */
  conn_id++;
  if (conn_id > 0x7FFF) conn_id = 0;
}

/*
 * close a CONN structure
 */
void conn_close(CONN *cop)
{
#ifdef DEBUG
  if (DEBUGGING(DBG_STATS)) {
    float elapsed = (float)(time((time_t *)NULL) - cop->start + 1);
    /* per connection */
    dprintf("stat", "conn %s: %d alerts, %.0fs elapsed, "
	    PRId32BITS " pkts (%g p/m), "
	    PRId32BITS " KB (%g B/s), %.0f B/pkt",
	    cop->id, cop->xinfo.alert_count, elapsed,
	    cop->xinfo.pkt_count, ((float)cop->xinfo.pkt_count/elapsed*60),
	    (cop->xinfo.pkt_size >> 10), ((float)cop->xinfo.pkt_size/elapsed),
	    ((float)cop->xinfo.pkt_size/(float)cop->xinfo.pkt_count));
    /* update global */
    conn_count++;
    conn_alerts  += cop->xinfo.alert_count;
    conn_elapsed += elapsed;
    conn_pcount  += (float)cop->xinfo.pkt_count;
    conn_psize   += (float)cop->xinfo.pkt_size;
    /* global */
    dprintf("stat", "total %d: %d alerts, %.0fs elapsed, "
	    "%.0f pkts (%g p/m), "
	    "%.0f KB (%g B/s), %.0f B/pkt",
	    conn_count, conn_alerts, conn_elapsed,
	    conn_pcount, (conn_pcount/conn_elapsed*60),
	    (conn_psize/1024), (conn_psize/conn_elapsed),
	    (conn_psize/conn_pcount));
  }
#endif
  if (cop->name != cop->host->name) str_free(cop->name);
  host_destroy(cop->host);
  io_close(&cop->client);
  io_close(&cop->server);
  ui_close(&cop->ui);
  /* update list */
  if (cop->next) cop->next->prev = cop->prev;
  else conn_last = cop->prev;
  if (cop->prev) cop->prev->next = cop->next;
  else conn_first = cop->next;
}

/*
 * position of a given connection, result is 1..n, 0 if error
 */
int conn_position(CONN *cop)
{
  CONN *p;
  int pos;

  p = conn_first;
  pos = 1;
  while (p && p != cop) {
    p = p->next;
    pos++;
  }
  if (p) return(pos);
  else return(0);
}

/*
 * connection at a given position, result is NULL if error
 */
CONN *conn_at_position(int pos)
{
  CONN *p;

  p = conn_first;
  while (p && pos != 1) {
    p = p->next;
    pos--;
  }
  return(p);
}

/*
 * try to write from client to server
 */
void conn_client_write(CONN *cop)
{
  IO *iop = &cop->client;
  if (BITTST(iop->flags, IO_WRITE)) io_write(iop);
}

/*
 * try to write from server to client
 */
void conn_server_write(CONN *cop)
{
  IO *iop = &cop->server;
  if (BITTST(iop->flags, IO_WRITE)) io_write(iop);
}

/*
 * try to read from client to server
 */
void conn_client_read(CONN *cop)
{
  IO *iop = &cop->client;
  char *buf;
  int len;

  /* try to do some I/O first */
  if (BITTST(iop->flags, IO_READ)) io_read(iop);
  if (BITTST(iop->flags, IO_READ|IO_ERROR)) return;

  /* work while we can */
#ifdef DEBUG
  if (DEBUGGING(DBG_FSA))
    dprintf(cop->tag, "client state=%c needs=%d has=%d",
	    iop->state, iop->needs, iop->has);
#endif
  while (!BITTST(iop->flags, IO_READ)) {
    buf = &iop->buf[iop->pos];
    switch (iop->state) {
    /******************************************************************************/
    case CS_SETUPH:
      /* setup header:
	 CARD8	byteOrder;
	 BYTE	pad;
	 CARD16	majorVersion B16, minorVersion B16;
	 CARD16	nbytesAuthProto B16;
	 CARD16	nbytesAuthString B16;
	 CARD16	pad2;
	 */
      cop->xinfo.sex = getCARD8(buf, 0);
      len = 12 + PAD4(BUF16(6)) + PAD4(BUF16(8));
      cop->xinfo.pkt_count++;
      cop->xinfo.pkt_size += len;
      io_needs(iop, CS_SETUP, len);
      break;
    /******************************************************************************/
    case CS_SETUP:
      /* (hopefully) complete setup */
      PRINTX(XP_CSETUP);
      /* unblock I/O */
      io_unblock(iop->fd_in);
      io_unblock(iop->fd_out);
      /* forge the setup and send it */
      io_queue(iop, auth_forge(buf));
      /* go into discard mode */
      iop->state = CS_DISCARD;
      break;
    /******************************************************************************/
    case CS_REQH:
      /* request header:
	 CARD8 reqType;
	 CARD8 data;
	 CARD16 length B16;
       */
      len = 4 * BUF16(2);
      cop->xinfo.seq++;
      cop->xinfo.pkt_count++;
      cop->xinfo.pkt_size += len;
      io_needs(iop, CS_REQ, len);
      break;
    /******************************************************************************/
    case CS_REQ:
      /* (maybe incomplete if too big) request: print it first */
      PRINTX(XP_REQUEST);
      /* keep trace of the traffic if we check this connection */
      if (BITTST(cop->flags, CF_CHECKED) || BITTST(cop->flags, CF_SAFE))
	xp_trace(XP_REQUEST, buf, iop->tag, &cop->xinfo);
      /* go into verify mode */
      iop->state = CS_REQV;
      /* maybe check the request */
      if (BITTST(cop->flags, CF_CHECKED) || BITTST(cop->flags, CF_SAFE)) {
	/* really check the request */
	xp_check(buf, &cop->xinfo);
	if (cop->xinfo.alert) {
	  /* alert detected */
#ifdef DEBUG
	  if (DEBUGGING(DBG_LOG) && cop->xinfo.alert_level != AL_INTERNAL)
	    dprintf(timetag, "alert for %s: %s", cop->id, cop->xinfo.alert);
#endif
	  if (BITTST(cop->flags, CF_CHECKED)) {
	    if (BITTST(cop->flags, CF_SAFE)) {
	      /* CHECKED and SAFE: always replace by noop */
	      BITSET(cop->flags, CF_NOOP);
	    } else {
	      /* CHECKED only: prompt or noop if an internal alert */
	      if (cop->xinfo.alert_level == AL_INTERNAL)
		BITSET(cop->flags, CF_NOOP);
	      else
		goto the_end;
	    }
	  } else {
	    /* SAFE only: replace by noop if not an internal alert */
	    if (cop->xinfo.alert_level != AL_INTERNAL)
	      BITSET(cop->flags, CF_NOOP);
	  }
	}
      }
      break;
    /******************************************************************************/
    case CS_REQV:
      /* (maybe incomplete if too big) verified request: handle the user answer */
      /* by default we go into copy mode */
      iop->state = CS_COPY;
      if (cop->xinfo.alert) {
	/* there was indeed an alert */
	cop->xinfo.alert = NULL;
	if (BITTST(cop->flags, CF_NOOP)) {
	  /* we replace this request with a NoOp */
	  BITCLR(cop->flags, CF_NOOP);
	  xp_noop(buf, &cop->xinfo);
#ifdef DEBUG
	  if (DEBUGGING(DBG_X))
	    xp_print(XP_REQUEST, cop->xinfo.fake_request->base, iop->tag, &cop->xinfo);
#endif
	  io_queue(iop, cop->xinfo.fake_request);
	  cop->xinfo.fake_request = NULL;
	  /* go into discard mode now */
	  iop->state = CS_DISCARD;
	}
      }
      break;
    /******************************************************************************/
    case CS_COPY:
    case CS_DISCARD:
      /* copy or discard */
      if (iop->state == CS_COPY) io_flush(iop);
      else io_discard(iop);
      if (iop->needs > 0) {
	/* we still need more bytes */
	BITSET(iop->flags, IO_READ);
      } else {
	/* we now need a new header */
	io_needs(iop, CS_REQH, 4);
      }
      break;
    default:
      die("bad state in conn_client_read: %c", iop->state);
    }
#ifdef DEBUG
    if (DEBUGGING(DBG_FSA))
      dprintf(cop->tag, "client new state=%c needs=%d has=%d",
	      iop->state, iop->needs, iop->has);
#endif
  }
the_end:
  cop->idle_since = time((time_t *)NULL);
}

/*
 * try to read from server to client
 */
void conn_server_read(CONN *cop)
{
  IO *iop = &cop->server;
  char *buf;
  int len;
  U8BITS type;
  U16BITS seq, fseq;
  MC *next;

  /* try to do some I/O first */
  if (BITTST(iop->flags, IO_READ)) io_read(iop);
  if (BITTST(iop->flags, IO_READ|IO_ERROR)) return;

  /* work while we can */
#ifdef DEBUG
  if (DEBUGGING(DBG_FSA))
    dprintf(cop->tag, "server state=%c needs=%d has=%d",
	    iop->state, iop->needs, iop->has);
#endif
  while (!BITTST(iop->flags, IO_READ)) {
    buf = &iop->buf[iop->pos];
    switch (iop->state) {
    /******************************************************************************/
    case SS_SETUPH:
      /* setup header:
	 BOOL           success;
	 BYTE           lengthReason;
	 CARD16         majorVersion B16, minorVersion B16;
	 CARD16         length B16;
       */
      len = 8 + 4 * BUF16(6);
      cop->xinfo.pkt_count++;
      cop->xinfo.pkt_size += len;
      io_needs(iop, SS_SETUP, len);
      break;
    /******************************************************************************/
    case SS_SETUP:
      /* (hopefully) complete setup (after the header):
	 CARD32         release B32;
	 CARD32         ridBase B32, ridMask B32;
	 CARD32         motionBufferSize B32;
	 CARD16         nbytesVendor B16;
	 CARD16         maxRequestSize B16;
	 CARD8          numRoots;
	 CARD8          numFormats;
	 CARD8          imageByteOrder;
	 CARD8          bitmapBitOrder;
	 CARD8          bitmapScanlineUnit, bitmapScanlinePad;
	 KeyCode        minKeyCode, maxKeyCode;
	 CARD32         pad2 B32;
	 */
      PRINTX(XP_SSETUP);
      /* record some info */
      cop->xinfo.resid_base = BUF32(12);
      cop->xinfo.resid_mask = BUF32(16);
      /* go into copy mode */
      iop->state = SS_COPY;
      break;
    /******************************************************************************/
    case SS_PKTH:
      /* packet header:
	 BYTE type;
	 BYTE data1;
	 CARD16 sequenceNumber B16;
	 CARD32 length B32;
	 ...
       */
      type = getCARD8(buf, 0);
      len = 32 + ((type == 1) ? 4 * BUF32(4) : 0);
      cop->xinfo.pkt_count++;
      cop->xinfo.pkt_size += len;
      io_needs(iop, SS_PKT, len);
      break;
    /******************************************************************************/
    case SS_PKT:
      /* (maybe incomplete if too big) packet: print it first */
      PRINTX(XP_PACKET);
      /* keep trace of the traffic if we check this connection */
      if (BITTST(cop->flags, CF_CHECKED) || BITTST(cop->flags, CF_SAFE))
	xp_trace(XP_PACKET, buf, iop->tag, &cop->xinfo);
      /* go into copy mode by default */
      iop->state = SS_COPY;
      /* do we have fake reply packet replacing this one? */
      type = getCARD8(buf, 0);
      if (type == 1) {
	/* this is a reply, we should have a closer look */
	seq = BUF16(2);
	while (cop->xinfo.fake_packet) {
	  /* we have fake replies in stock */
	  fseq = getCARD16(cop->xinfo.sex, &cop->xinfo.fake_packet->base[2]);
	  next = cop->xinfo.fake_packet->next;
	  /* what about sequence numbers? */
	  if (seq == fseq) {
	    /* bingo! we have to replace this packet */
	    cop->xinfo.fake_packet->next = NULL; /* unlink */
#ifdef DEBUG
	    if (DEBUGGING(DBG_X))
	      xp_print(XP_PACKET, cop->xinfo.fake_packet->base, iop->tag, &cop->xinfo);
#endif
	    io_queue(iop, cop->xinfo.fake_packet);
	    cop->xinfo.fake_packet = next;
	    /* go into discard mode now */
	    iop->state = SS_DISCARD;
	    /* we stop checking ourfake packets */
	    break;
	  } else if (seq > fseq) {
	    /* oops! the server is now ahead of us */
#ifdef DEBUG
	    if (DEBUGGING(DBG_X) && DEBUGGING(DBG_WARNING))
	      dprintf(cop->tag, "discarded fake reply seq=%d", fseq);
#endif
	    /* discard this packet and check next one */
	    mc_old(cop->xinfo.fake_packet);
	    cop->xinfo.fake_packet = next;
	  } else {
	    /* time has not come yet... */
	    break;
	  }
	}
      }
      break;
    /******************************************************************************/
    case SS_COPY:
    case SS_DISCARD:
      /* copy or discard */
      if (iop->state == SS_COPY) io_flush(iop);
      else io_discard(iop);
      if (iop->needs > 0) {
	/* we still need more bytes */
	BITSET(iop->flags, IO_READ);
      } else {
	/* we now need a new header */
	io_needs(iop, SS_PKTH, 32);
      }
      break;
    default:
      die("bad state in conn_server_read: %c", iop->state);
    }
#ifdef DEBUG
    if (DEBUGGING(DBG_FSA))
      dprintf(cop->tag, "server new state=%c needs=%d has=%d",
	      iop->state, iop->needs, iop->has);
#endif
  }
  cop->idle_since = time((time_t *)NULL);
}

/*
 * return the connection with the oldest alert
 */
CONN *conn_oldest_alert(void)
{
  CONN *cop, *res;
  time_t oldest;

  res = NULL;
  oldest = 0;
  for (cop=conn_first; cop; cop=cop->next) {
    if (cop->ui.alert) {
      if ((oldest == 0) || (cop->idle_since < oldest)) {
	res = cop;
	oldest = cop->idle_since;
      }
    }
  }
  return(res);
}
