/*+*****************************************************************************
*                                                                              *
* File: cfg.c                                                                  *
*                                                                              *
* Description: configuration file handling                                     *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char cfg_vers[] = "@(#)cfg.c	1.7	02/19/99	Written by Lionel Cons";
#endif /* __lint */

/*
 * each line of the configuration file should contain
 *  host [action] [flag...]
 *
 * host can be a name (foo.bar.org), a name regexp (*.bar.org),
 * an IP name (128.132.17.45), an IP regexp (128.132.*.2??),
 * an address/mask (128.132.0.0/255.255.0.0) or any of these with
 * a user name or name regexp in front (joe@foo or r??t@128.*)
 *
 * action can be 'ask' (default), 'allow' or 'deny'
 *
 * flag can be 'none', 'frozen', 'checked' or 'safe';
 * the flags are ORed, except 'none' which resets the flags to 0
 * 
 * empty lines and lines starting with # are ignored
 */

/******* Include Files ********************************************************/

#include "cfg.h"
#include "util.h"
#include "host.h"
#include "conn.h"
#include "xmd.h"
#include "sb.h"

#include <ctype.h>

/******* Constants ************************************************************/

/******* Macros ***************************************************************/

#define BYTE_CHECK(x)	if ((x < 0) || (x > 255)) return(0)

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

/*
 * simple minded pattern matching with support for ? and *
 * since it's used only for host and user names, it's case insensitive
 */
static int cfg_regexp_match(char *string, char *regexp)
{
  int res;

  /* empty regexp */
  if (*regexp == '\0') {
    if (*string == '\0') return(1);
    return(0);
  }
  /* empty string */
  if (*string == '\0') {
    if (*regexp == '*') return(cfg_regexp_match(string, &regexp[1]));
    return(0);
  }
  /* normal case */
  switch (*regexp) {
  case '?':
    res = cfg_regexp_match(&string[1], &regexp[1]);
    break;
  case '*':
    res = cfg_regexp_match(string, &regexp[1]);
    if (!res) res = cfg_regexp_match(&string[1], regexp);
    break;
  default:
    if (strncasecmp(regexp, string, 1) == 0)
      res = cfg_regexp_match(&string[1], &regexp[1]);
    else
      res = 0;
  }
  return(res);
}

/*
 * test if a user and host matche a pattern
 */
static int cfg_match(char *name, HOST *host, char *pattern)
{
  char *cp;
  int res;

  /*
   * check for a @ in the pattern
   */
  cp = strchr(pattern, '@');
  if (cp) {
    *cp = '\0';
    if (!name) return(0);
    res = cfg_regexp_match(name, pattern);
#ifdef DEBUG
    if (DEBUGGING(DBG_CONFIG))
      dprintf("cfg ", "config name %s match %s: %d", name, pattern, res);
#endif
    if (!res) return(0);
    pattern = ++cp;
  }

  /*
   * try address/mask
   */
  if (strchr(pattern, '/')) {
    U32BITS addr, mask;
    int a1, a2, a3, a4, m1, m2, m3, m4;
    
    res = sscanf(pattern, "%d.%d.%d.%d/%d.%d.%d.%d",
		 &a1, &a2, &a3, &a4, &m1, &m2, &m3, &m4);
    if (res != 8) return(0);
    BYTE_CHECK(a1);
    BYTE_CHECK(a2);
    BYTE_CHECK(a3);
    BYTE_CHECK(a4);
    addr = ((a1 << 24) | (a2 << 16) | (a3 << 8) | a4);
    BYTE_CHECK(m1);
    BYTE_CHECK(m2);
    BYTE_CHECK(m3);
    BYTE_CHECK(m4);
    mask = ((m1 << 24) | (m2 << 16) | (m3 << 8) | m4);
    res = ((host->addr.s_addr & mask) == (addr & mask));
#ifdef DEBUG
    if (DEBUGGING(DBG_CONFIG))
      dprintf("cfg ", "config host %s mask %s: %d", host->ip_name, pattern, res);
#endif
    return(res);
  }
 
  /*
   * try regexp against IP name or fully qualified name
   */
  res = 1;
  for (cp=pattern; *cp; cp++)
    if (!isdigit((int)*cp) && (*cp != '.') && (*cp != '?') && (*cp != '*')) res = 0;
  cp = (res ? host->ip_name : host->fq_name);
  res = cfg_regexp_match(cp, pattern);
#ifdef DEBUG
  if (DEBUGGING(DBG_CONFIG))
    dprintf("cfg ", "config host %s match %s: %d", cp, pattern, res);
#endif
  return(res);
}

/*
 * skip some leading spaces if any and then a word
 * return the new position and the word
 */
static char *cfg_skip_word(char *string, char **word)
{
  char *cp;

  cp = string;
  /* skip spaces */
  while (*cp && isspace((int)*cp)) cp++;
  *word = cp;
  /* skip word */
  while (*cp && !isspace((int)*cp)) cp++;
  /* end the word and move forward one character */
  *cp = '\0';
  cp++;
  return(cp);
}

/*
 * parse the given configuration file and return the action
 * corresponding to the first line matching the given host
 * also return the connection flags
 */
int cfg_action(char *path, char *name, HOST *host, int *cf)
{
  FILE *fp;
  char *cp, *start;
  int res;

  /*
   * open the file
   */
  fp = fopen(path, "r");
  if (fp) {
#ifdef DEBUG
    if (DEBUGGING(DBG_CONFIG))
      dprintf("cfg ", "reading config file %s...", path);
#endif
  } else {
#ifdef DEBUG
    if (DEBUGGING(DBG_CONFIG))
      dprintf("cfg ", "can't read config file %s: %s", path, ERROR);
#endif
    return(CFG_ACTION_ERROR);
  }

  /*
   * parse it
   */
  res = CFG_ACTION_UNKNOWN;
  while (fgets(sb_line, SB_LINE_LENGTH, fp)) {
    /* skip comments */
    if (sb_line[0] == '#') continue;
    /* get first word */
    cp = cfg_skip_word(sb_line, &start);
    /* skip empty line */
    if (*start == '\0') continue;
    /* skip line unless it matches host */
    if (!cfg_match(name, host, start)) continue;
    /* found a matching host, set defaults and check flags */
    /* we assume that *cf already holds the default flags */
    res = CFG_ACTION_ASK;
    cp = cfg_skip_word(cp, &start);
    while (*start) {
      if (!strcasecmp(start, "allow")) {
	res = CFG_ACTION_ALLOW;
      } else if (!strcasecmp(start, "ask")) {
	res = CFG_ACTION_ASK;
      } else if (!strcasecmp(start, "deny")) {
	res = CFG_ACTION_DENY;
      } else if (!strcasecmp(start, "none")) {
	*cf = 0;
      } else if (!strcasecmp(start, "frozen")) {
	BITSET(*cf, CF_FROZEN);
      } else if (!strcasecmp(start, "checked")) {
	BITSET(*cf, CF_CHECKED);
      } else if (!strcasecmp(start, "safe")) {
	BITSET(*cf, CF_SAFE);
      } else {
#ifdef DEBUG
	if (DEBUGGING(DBG_CONFIG))
	  dprintf("cfg ", "bad config keyword ignored: %s", start);
#endif
      }
      cp = cfg_skip_word(cp, &start);
    }      
    break;
  }

  /*
   * security enforcement
   */
  if (res == CFG_ACTION_ALLOW && *cf == 0) {
    /* forbid blind allow with no X traffic inspection */
    BITSET(*cf, CF_SAFE);
  }
  
  /*
   * close it
   */
  fclose(fp);
  return(res);
}
