/*+*****************************************************************************
*                                                                              *
* File: auth.c                                                                 *
*                                                                              *
* Description: X authorization handling                                        *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char auth_vers[] = "@(#)auth.c	1.6	02/01/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "auth.h"
#include "util.h"
#include "io.h"
#include "host.h"
#include "mc.h"
#include "xmd.h"

#include <netinet/in.h>
#include <X11/X.h>
#include <X11/Xauth.h>

/******* Constants ************************************************************/

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

static Xauth *auth_data;	/* static copy of the authorization data */

/*
 * find the authorization data for the given display
 * loosely based on ConnDis.c/GetAuthorization
 */
void auth_setup(HOST *shost, int idisplay)
{
  int local;
  unsigned short family, saddrlen;
  char dpynumbuf[40], *saddr;

  /* the following works, see xs.c */
  local = (shost == localhost);
#ifdef DEBUG
  if (DEBUGGING(DBG_MISC))
    dprintf("auth", "looking for%s authorization for %s:%d",
	    (local ? " local" : ""), shost->name, idisplay);
#endif
  if (local) {
    family = FamilyLocal;
    saddr = shost->name;
    saddrlen = strlen(saddr);
  } else {
    family = FamilyInternet;
    saddr = (char *)&shost->addr;
    saddrlen = sizeof(shost->addr);
  }
  sprintf(dpynumbuf, "%d", idisplay);
try_again:
  auth_data = XauGetAuthByAddr(family, saddrlen, saddr,
			       (unsigned short) strlen(dpynumbuf), dpynumbuf,
			       (unsigned short) 0, (char *) 0);
  if (auth_data == NULL && local && saddr != shost->fq_name) {
    /* try again with fully qualified name */
#ifdef DEBUG
    if (DEBUGGING(DBG_MISC))
      dprintf("auth", "no authorization found, trying again with %s", shost->fq_name);
#endif
    saddr = shost->fq_name;
    saddrlen = strlen(saddr);
    goto try_again;
  }
  if (auth_data == NULL) {
    static Xauth auth_none;
    auth_none.name_length = 0;
    auth_none.name = NULL;
    auth_none.data_length = 0;
    auth_none.data = NULL;
    auth_data = &auth_none;
#ifdef DEBUG
    if (DEBUGGING(DBG_MISC)) dprintf("auth", "no authorization found");
  } else {
    if (DEBUGGING(DBG_MISC)) {
      int i, len = auth_data->data_length;
      char *data = (char *)safe_malloc(2*len+1);
      char *cp = str_ncopy(auth_data->name, auth_data->name_length);
      for (i=0; i<len; i++) sprintf(&data[2*i], "%02x", auth_data->data[i]);
      data[2*len] = '\0';
      dprintf("auth", "found %s authorization of %d bytes: %s", cp, len, data);
      str_free(cp);
      free(data);
    }
#endif
  }
}

/*
 * forge a client connection setup with the authorization found by auth_setup
 * note that buf must contain a valid xConnClientPrefix
 */
MC *auth_forge(char *buf)
{
  MC *mcp;
  U8BITS sex;
  char *cp, *pad = "\0\0\0\0";
  int len, padlen;

  /* Xproto.h/xConnClientPrefix:
     CARD8       byteOrder;
     BYTE        pad;
     CARD16      majorVersion B16, minorVersion B16;
     CARD16      nbytesAuthProto B16;
     CARD16      nbytesAuthString B16;
     CARD16      pad2 B16;
     */
  sex = getCARD8(buf, 0);
  len = 12 + PAD4(auth_data->name_length) + PAD4(auth_data->data_length);
  mcp = mc_new(len);
  /* copy the header and replace the length fields (the rest is intact) */
  memmove(mcp->base, buf, 12);
  putCARD16(sex, &mcp->base[6], auth_data->name_length);
  putCARD16(sex, &mcp->base[8], auth_data->data_length);
  cp = &mcp->base[12];
  /* copy the authorization name */
  len = auth_data->name_length;
  if (len > 0) {
    memmove(cp, auth_data->name, len);
    cp += len;
    padlen = PAD4(len) - len;
    memmove(cp, pad, padlen);
    cp += padlen;
  }
  /* copy the authorization data */
  len = auth_data->data_length;
  if (len > 0) {
    memmove(cp, auth_data->data, len);
    cp += len;
    padlen = PAD4(len) - len;
    memmove(cp, pad, padlen);
    cp += padlen;
  }
  return(mcp);
}
