/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 uiterminal.cpp  -  User Interface code for the terminal/commandline.

    begin                : Thu Jan 31 2002
    copyright            : (C) 2002 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>	/* waitpid () */
#include <sys/wait.h>	/* waitpid () */
#include <unistd.h>	/* fnctl () */
#include <fcntl.h>	/* fnctl () */

#include "mutella.h"
#include "structures.h"

#include "controller.h"
#include "mui.h"
#include "uiterminal.h"
#include "property.h"
#include "preferences.h"
#include "mprintf.h"
#include "ansicolor.h"
#include "event.h"
#include "messages.h"
#include "lineinput.h"
#include "conversions.h"
#include "common.h"
#include "term_help.h"
#include "gnumarkedfiles.h"
#include "uitextmode.h"

#define _isblank isspace

struct SUIColors {
	// list/results
	char header[32];
	char id[32];
	char size[32];
	char speed[32];
	char percent[32];
	char time[32];
	char name[32];
	char extra[32];
	char sha1[32];
	char known[32];
	char flag1[32];
	char flag2[32];
	char num[32];
	char ip[32];
	char status[32];
	// general purpose
	char deflt[32];
	char message[32];
	char error[32];
	// set
	char variable[32];
	char value[32];
};

/////////////////////////////////////////////////////////////////////////////
// Paged output with color support

class MPrintfColor : public MPrintfMore
{
public:
	MPrintfColor(int nCPL = -1, int nLPP = -1) : MPrintfMore(nCPL, nLPP), m_bEnableColor(true) {}
protected:
	virtual void PrePrint(CString& s){
		// search for '??' combinations which are suposed to define 'style'
		// listed in m_propClr;
		int nStart = 0;
		int nPos;
		int nPos1;
		int n,i;
		bool bStyled = false;
		LPCSTR szPropName;
		int nNameLen;
		while ( -1 != (nPos = s.find("??",nStart)) )
		{
			CString sStyle = s.substr(nPos+2,16); // style names are normally short
			bool bFound = false;
			nPos1 = -1;
			// unfortunately sStyle.find('>'); wont work
			n = sStyle.length();
			for(i = 0; i<n; ++i)
				if (sStyle[i] == '>')
				{
					nPos1 = i;
					break;
				}
				else if (sStyle[i]<'0' || sStyle[i]>'z' || (sStyle[i]>'9' && sStyle[i]<'A'))
				{
					// non-alphanum
					break;
				}
			if (nPos1>=0)
			{
				//found '>' -- we have to cut out &&...> and replace it with the ANSI combination
				sStyle = sStyle.substr(0,nPos1);
				CString sTmp = s.substr(nPos+2+nPos1+1); // '+1' is for spacer after
				s.cut(nPos); // cut without memory realloc
				s += ANSI_NORMAL;
				bStyled = false;
				//
				if (m_bEnableColor && sStyle.size())
				{
					MProperty* pP = m_propClr.FindProperty(sStyle.c_str());
					if (pP)
					{
						if (strlen(pP->GetStringValue()))
						{
							s += ESC_CHAR;
							s += pP->GetStringValue();
							bStyled = true;
						}
					}
				}
				nStart = s.length();
				s += sTmp;
			}
			else
			{
				// '??' case -- just replace it with 'ANSI_NORMAL'
				CString sTmp = s.substr(nPos+2);
				s.cut(nPos); // cut without memory realloc
				s += ANSI_NORMAL;
				nStart = s.length();
				s += sTmp;
				bStyled = false;
			}
		}
		if (bStyled)
			s+= ANSI_NORMAL;
	}
public:
	MPropertyContainer m_propClr;
	bool m_bEnableColor;
};

//////////////////////////////////////////////////////////////////////
// Re-define line input to enable completion

class MyLineInput : public MLineInput
{
public:
	MyLineInput(MUITermPriv* pMaster, MController* pController);
	virtual bool CompleteWord(const CString& part, list<CString>& listCompletions);
protected:
	MController* m_pController;
	MUITermPriv* m_pMaster;
};

////////////////////////////////////////////////////////////////////////
// event display

class MEventPrinter : public MSyncEventReceiver {
public:
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		return p->GetSeverity()>=ES_IMPORTANT;
	}
	virtual void OnEvent(MEvent* p, bool){
			ASSERT(p);
			printf("IMPORTANT EVENT:\n"
#ifdef _DEBUG
			             "  type: %d\n  severity: %d\n  id: %d\n"
			             ,
			             p->GetType(), p->GetSeverity(), p->GetID()
#endif
			             );
			if (p->GetID())
				printf("  %s\n"
				       "  %s\n",
				       GetMessageString(p->GetID()).c_str(),
				       p->Format().c_str());
			else
				printf("  %s\n", p->Format().c_str());
	}
};

class MEventCache : public MAsyncEventReceiver {
public:
	MEventCache() : MAsyncEventReceiver(INT_MAX, 0x4000 /*16k*/) {}
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		return p->GetSeverity()>=ES_JUSTINCASE;
	}
};

////////////////////////////////////////////////////////////////////////////
// MUITermPriv

class MUITermPriv : MUITextModePriv {
public:
	MUITermPriv(MController*);
	~MUITermPriv();
	// basic functionality
	virtual bool attach();
	virtual bool init();
	virtual void detach();
	virtual void ui_loop();
	virtual void stop(bool bForce);

	virtual bool execute_command(ComEntry *command, LPSTR word);

	virtual bool vcom_color(char * arg);
	virtual bool on_realtime();
//protected:
	SUIColors     m_colors;
	MPrintfColor  m_more;
	MyLineInput   m_input;
	MEventPrinter m_evPrinter;
	MEventCache   m_evCache;
	bool m_bContinue;
};

////////////////////////////////////////////////////////////////////////////
// UI implementation

// command completion
MyLineInput::MyLineInput(MUITermPriv* pMaster, MController* pController) : m_pMaster(pMaster), m_pController(pController)
{
}

bool MyLineInput::CompleteWord(const CString& part, list<CString>& listCompletions)
{
	ASSERT(m_pMaster);
	ASSERT(m_pController);
	// parse the string to isolate the last word
	if (part[part.size()-1]==' ')
		return false; // sort of quick workaround for situation like command<SPACE>
	CString s = StripWhite(part);
	int nPos = s.rfind(" ");
	// look up the commands if its the first word
	if (nPos<0) // no spaces
	{
		for (int i = 0; MUITextModePriv::commands[i].name; i++)
			if (strncmp (s.c_str(), MUITextModePriv::commands[i].name, s.length()) == 0)
			{
				listCompletions.push_front(MUITextModePriv::commands[i].name);
			}
		return listCompletions.size();
	}
	// otherwise try variables or colors if the first word is set or color command
	CString sCom = StripWhite(s.substr(0, nPos));
	if (sCom=="set" || sCom=="set+" || sCom=="set-" ||sCom=="help")
	{
		CString sPart = s.substr(nPos+1);
		// first check against commands if we have "help" command
		if (sCom=="help")
		{
			for (int i = 0; MUITextModePriv::commands[i].name; i++)
				if (strncmp (sPart.c_str(), MUITextModePriv::commands[i].name, sPart.length()) == 0)
				{
					listCompletions.push_front(MUITextModePriv::commands[i].name);
				}
		}
		// now load properties
		MPropertyContainer* pPC = m_pController->GetPropertyContainer();
		MPropertyContainer::iterator it;
		for (it = pPC->begin(); it!= pPC->end(); ++it)
			if (strncmp (sPart.c_str(), pPC->GetPropertyName(it), sPart.length()) == 0)
			{
				if ( (sCom!="set+" && sCom!="set-") || pPC->GetProperty(it)->IsSet() )
					listCompletions.push_front(pPC->GetPropertyName(it));
			}
		return listCompletions.size();
	} // otherwise try variables or colors if the first word is set or color command
	else if (sCom=="color")
	{
		CString sVar = s.substr(nPos+1);
		//
		MPropertyContainer* pPC = &m_pMaster->m_more.m_propClr;
		MPropertyContainer::iterator it;
		for (it = pPC->begin(); it!= pPC->end(); ++it)
			if (strncmp (sVar.c_str(), pPC->GetPropertyName(it), sVar.length()) == 0)
			{
				listCompletions.push_front(pPC->GetPropertyName(it));
			}
		return listCompletions.size();
	}
	return false;
}

////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////

MUITermPriv::MUITermPriv(MController* pC) : MUITextModePriv (pC), m_more(80,24), m_input(this, pC)
{
	m_pOutput = &m_more;
	m_bContinue = true;
}

MUITermPriv::~MUITermPriv()
{
}

bool MUITermPriv::attach()
{
	//
	ED().AddReceiver(&m_evPrinter);
	ED().AddReceiver(&m_evCache);
	//
	MProperty* pP;
	MPropertyContainer* pPC = m_pController->GetPropertyContainer();
	pPC->AddSection("TerminalUI");
	pP = pPC->AddProperty("MaxResultsDisplayed", &m_nMaxResultsDispl,    1000);
	pP->SetPropertyDescription("Maximum number of results displayed by `results'",
		"Limits the total number of results printed out by the `results' command");
	pP = pPC->AddProperty("ShowGroups",          &m_bShowGroups,         true);
	pP->SetPropertyDescription("Shows grouped results",
		"Controls whether the contents of the grouped results needs to be shown");
	pP = pPC->AddProperty("StartupScript",       m_szStartUpScript,      1024, "~/.mutella/termrc");
	pP->SetPropertyDescription("The path to the start-up script",
		"Path to the Mutella's start-uup script. Probably not that important these days");
	pP = pPC->AddProperty("UseANSIColor",        &m_more.m_bEnableColor, true);
	pP->SetPropertyDescription("Enables color in the terminal",
		"Controls, whether or not Mutella should send ANSI color codes");
	pP = pPC->AddProperty("ColorScheme",         m_szColorScheme,        1024, "~/.mutella/termclr");
	pP->SetPropertyDescription("Sets the file name for the color scheme",
		"You only need that if you like to switch between several color schemes time to time");
	pP = pPC->AddProperty("TerminalCols",   &m_more.m_nCPL,         MPrintfMore::GuessTermCols());
	if (pP) pP->SetPersistance(false);
	pP->SetPropertyDescription("Number of columns in the terminal",
		"Defined automatically at the client start. Non-persistent. If automatic "
		"detection fails you can set typical number of columns in the start-up "
		"script (~/.mutella/termrc)");
	pP = pPC->AddProperty("TerminalLines",  &m_more.m_nLPP,         MPrintfMore::GuessTermLines());
	if (pP) pP->SetPersistance(false);
	pP->SetPropertyDescription("Number of lines in the terminal",
		"Defined automatically at the client start. Non-persistent. If automatic "
		"detection fails you can set typical number of lines in the start-up "
		"script (~/.mutella/termrc)");
	pP = pPC->AddProperty("Paginate",            &m_more.m_bBreak,       true);
	pP->SetPropertyDescription("Stop between pages",
		"When set to `true' terminal output will stop after diplaying each `TerminalLines' lines");
	pPC->SetCurrentSection(NULL);
	return true;
}

bool MUITermPriv::init()
{
	//
	if (!m_input.Init(stdin, stdout))
	{
		// TODO: post event
		TRACE("Failed to init line-input object");
		return false;
	}
	// prepare property container for the color scheme
	m_more.m_propClr.AddProperty("header",   m_colors.header,  32, "[1m"); //bold
	m_more.m_propClr.AddProperty("id",       m_colors.id,      32, "[1m"); //bold
	m_more.m_propClr.AddProperty("size",     m_colors.size,    32, "[34m"); //blue
	m_more.m_propClr.AddProperty("speed",    m_colors.speed,   32, "[32m"); //green
	m_more.m_propClr.AddProperty("percent",  m_colors.percent, 32, "[33m"); //yellow
	m_more.m_propClr.AddProperty("time",     m_colors.time,    32, "[35m"); //magenta
	m_more.m_propClr.AddProperty("name",     m_colors.name,    32, "[31m"); //red
	m_more.m_propClr.AddProperty("extra",    m_colors.extra,   32, "[32m");
	m_more.m_propClr.AddProperty("sha1",     m_colors.sha1,    32, "[33m");
	m_more.m_propClr.AddProperty("known",    m_colors.known,   32, "[35m");
	m_more.m_propClr.AddProperty("flag1",    m_colors.flag1,   32, "[34m");
	m_more.m_propClr.AddProperty("flag2",    m_colors.flag2,   32, "[33m");
	m_more.m_propClr.AddProperty("num",      m_colors.num,     32, "[36m");
	m_more.m_propClr.AddProperty("ip",       m_colors.ip,      32, "[35m");
	m_more.m_propClr.AddProperty("status",   m_colors.status,  32, "[34m");
	// general purpose
	//m_more.m_propClr.AddProperty("default",  m_colors.deflt,   32, "");
	m_more.m_propClr.AddProperty("message",  m_colors.message, 32, "[34m");//blue
	m_more.m_propClr.AddProperty("error",    m_colors.error,   32, "[31m");//red
	// set
	m_more.m_propClr.AddProperty("variable", m_colors.variable,32, "[35m");
	m_more.m_propClr.AddProperty("value",    m_colors.value,   32, "[34m");
	// load the color scheme
	if (strlen(m_szColorScheme))
		m_more.m_propClr.Read(m_szColorScheme);
	return true;
}

void MUITermPriv::detach()
{
	m_input.Close();
	if (strlen(m_szColorScheme) && m_more.m_bEnableColor)
	{
		TRACE("Saving color scheme...");
		m_more.m_propClr.Write(m_szColorScheme);
	}
	// remove properties from m_pController
	// TODO that
	//
	ED().RemoveReceiver(&m_evPrinter);
	ED().RemoveReceiver(&m_evCache);
}

void MUITermPriv::ui_loop()
{
	char *line, *s;
	char tmp[1024];
	CString sLastCommand;
	// startup script
	if (strlen(m_szStartUpScript))
	{
		if (!FileExists(m_szStartUpScript))
		{
			printf("\nCreating default '%s' file\n", m_szStartUpScript);
			printf("Edit it to modify Mutella startup behaviour\n\n");
			FILE* f;
			if (f=fopen(ExpandPath(m_szStartUpScript).c_str(),"w"))
			{
				fprintf(f,"# mutella terminal-mode initialisation script\n");
				fprintf(f,"# edit it to make mutella smarter during startup\n");
				fprintf(f,"\n# for example set terminal size\n");
				fprintf(f,"# (if automatic detection fails), or just do nothing\n");
				fprintf(f,"\nversion\n\n");

				fclose(f);
			}
		}
		if (FileExists(m_szStartUpScript))
		{
			strncpy(tmp,m_szStartUpScript,1024);
			tmp[1023]='\0';
			com_load(tmp);
		}
	}
	/* Loop reading and executing lines until the user quits. */
	while ( m_bContinue )
	{
		if (m_command_queue.size())
		{
			strncpy(tmp,m_command_queue.front().c_str(),1024);
			tmp[1023]='\0';
			m_command_queue.pop();
			//
			execute_line(tmp);
		}
		else
		{
			if (m_evCache.HaveEvents())
			{
				if (m_evCache.MissedEvents())
				{
					m_pOutput->print("??error>You have missed %d events\n", m_evCache.MissedEvents());
					m_evCache.ResetMissedEvents();
				}
				m_evCache.LockFront();
				while (m_evCache.HaveEvents())
				{
					MEvent* p = m_evCache.UnsafeFront();
					ASSERT(p);
					m_pOutput->print("??header>Event %x:??\n"
					             "  received at: ??time>%s??\n"
#ifdef _DEBUG
					             "  type: %d\n  severity: %d\n  id: %d\n"
#endif
					             ,
					             p,
					             FormatAbsTime(p->GetTime()).c_str()
#ifdef _DEBUG
					             ,
					             p->GetType(), p->GetSeverity(), p->GetID()
#endif
					             );
					if (p->GetID())
						m_pOutput->print("  ??message>%s??\n"
						             "  ??message>%s??\n",
						             GetMessageString(p->GetID()).c_str(),
						             p->Format().c_str());
					else
						m_pOutput->print("  ??message>%s??\n", p->Format().c_str());
					//
					m_evCache.Pop();
				}
				m_evCache.UnlockFront();
				m_more.Flush();
			}
			// Read line of input,
			// remove leading and trailing whitespaces from the line.
			// read_key(30);
			printf("> "); fflush(stdout);
			while (!wait_key(-1)){
				//TRACE("waiting...");
			}
			printf("\r \r"); fflush(stdout);
			//TRACE("Approaching InputLine()");
			CString sLine = StripWhite( m_input.InputLine("> ") );
			//TRACE("InputLine() returned");
			if (sLine.empty())
				continue;
			// Then, if there is anything left, execute it.
			execute_line(sLine);
		}
	}
}

void MUITermPriv::stop(bool bForce)
{
	if (!m_bContinue)
		return;
	m_bContinue = false;
	if (!bForce)
		return;
	// a trick to wake up the readline library
	// backup stdin
	int hOldStdin;
	if ((hOldStdin = dup (STDIN_FILENO)) == -1) {
		printf ("MUITermPriv::stop() : Failed to backup stdin\n");
		return;
	}

	// Redirect stdin to a socket from the pair
	int sockets[2];
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, sockets) == -1) {
		printf ("MUITermPriv::stop() : Failed to create a socket pair\n");
		return;
	}
	if (dup2(sockets[0], STDIN_FILENO) == -1) {
		printf ("MUITermPriv::stop() : Failed to redirect stdin\n");
		return;
	}
	// send end-of-line to another socket
	write(sockets[1], "\n\n", 1);
	// give them a chance
	struct timespec rqtp, rmtp;
	rqtp.tv_sec = 1;
	rqtp.tv_nsec = rmtp.tv_sec = rmtp.tv_nsec = 0;
	safe_nanosleep(&rqtp, &rmtp);

	// restore everything
	if (dup2(hOldStdin, STDIN_FILENO) == -1) {
		printf ("MUITermPriv::stop() : Failed to restore stdint\n");
	}
	close(sockets[0]);
	close(sockets[1]);
}

bool MUITermPriv::execute_command(ComEntry *command, LPSTR word)
{
	int 	pipes [2],
			old_stdout,
			fpid = 0,
			flags;
	bool 	bOldPageBreak;
	char 	*args [4], *ptr;

	/* Pipe output to a system command */
	if (command->handler != &MUITextModePriv::com_system && // system command with pipes causes massive problems 
		(ptr = strchr (word, '|'))) {
		*ptr = 0;
		if (pipe (pipes) == -1) {
			printf ("Failed to create a pipe\n");
			return false;
		}

		if ((fpid = fork ()) == -1) {
			printf ("Failed to create child process\n");
			return false;
		}

		if (! fpid) {
			/* child */
			close (pipes [1]);

			/* redirect stdin from the pipe */
			if (dup2 (pipes [0], STDIN_FILENO) == -1) {
				printf ("Failed to redirect stdin\n");
				exit (1);
				return false;
			}

			args [0] = "sh";
			args [1] = "-c";
			args [2] = ++ptr;
			args [3] = 0;
			
			/* Set close_on_exec to 0 */
			flags = fcntl (STDIN_FILENO, F_GETFD, 0);
			flags &= ~FD_CLOEXEC;
			fcntl (STDIN_FILENO, F_SETFD, flags);

			if (execve ("/bin/sh", args, NULL) == -1) {
				printf ("execve() failed\n");
				exit (1);
			}
		} else {
			/* parent */
			close (pipes [0]);

			/* Backup stdout */
			if ((old_stdout = dup (STDOUT_FILENO)) == -1) {
				printf ("Failed to backup stdout\n");
				return false;
			}

			/* Redirect stdout to the pipe */
			if (dup2 (pipes [1], STDOUT_FILENO) == -1) {
				printf ("Failed to redirect stdout\n");
				return false;
		 	}

			/* set max lines to avoid paging */
			bOldPageBreak = m_more.m_bBreak;
			m_more.m_bBreak = false;

			close (pipes [1]);
		}
	}

	/* Call the function. */
	bool bRes = ((*this).*(command->handler))(word);//(this->(*(command->handler)) (word));

	m_more.Flush();

	/* Wait for piped process to exit and restore stdout / term lines */
	if (fpid>0) {
		close (STDOUT_FILENO);
		waitpid (fpid, NULL, 0);
		if (dup2(old_stdout, STDOUT_FILENO)==-1)
			printf("Failed to set stdout back\n");
		close (old_stdout);
		m_more.m_bBreak = bOldPageBreak;
	}
	return bRes;
}

bool MUITermPriv::vcom_color(char * arg)
{
	return set_helper(&m_more.m_propClr, SET_SET, arg);
}

bool MUITermPriv::on_realtime()
{
	bool bRealtime = true;
	bool bOldPageBreak = m_more.m_bBreak;
	m_more.m_bBreak = false;
	m_more.print("--hit any key to stop looping--");
	m_more.Flush();
	fflush(stdout);
	m_more.m_bBreak = bOldPageBreak;
	if (read_key(2))
		bRealtime = false;
	printf("\r                               \r");
	return bRealtime;
}

////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////

MUITerminal::MUITerminal()
{
	m_pPriv = NULL;
}

MUITerminal::~MUITerminal()
{
	delete m_pPriv;
}

bool MUITerminal::Attach(MController* pC)
{

	m_pPriv = new MUITermPriv(pC);
	ASSERT(m_pPriv);
	return m_pPriv->attach();
}

bool MUITerminal::Init()
{
	if (m_pPriv)
		return m_pPriv->init();
	return false;
}

void MUITerminal::Detach()
{
	if (m_pPriv)
		m_pPriv->detach();
}

void MUITerminal::Do()
{
	if (m_pPriv)
		m_pPriv->ui_loop();
}

void MUITerminal::Stop()
{
	if (m_pPriv)
		m_pPriv->stop(true);
}


