/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnudirector.cpp  -  Central point of control for all connections and transfers

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"
#include "event.h"
#include "messages.h"

#include "gnuhash.h"
#include "gnusock.h"
#include "gnunode.h"
#include "gnuupload.h"
#include "gnudownload.h"
#include "gnudirector.h"
#include "gnucache.h"
#include "gnushare.h"
#include "gnusearch.h"
#include "gnuwebcache.h"
#include "gnuwordhash.h"
#include "controller.h"
#include "conversions.h"
#include "common.h"
#include "property.h"
#include "preferences.h"
#include "asyncfile.h"
#include "asyncdns.h"
#include "gnumarkedfiles.h"

/////////////////////////////////////////////////////////////////////////////
// MGnuDirector

MGnuDirector::MGnuDirector(MController* pController) : m_listmutex(true) // resurcive
{
	m_pController = pController;
	m_pController->Attach(this);
	m_pPrefs = pController->GetPrefs();
	
	m_pHostCatcher  = new MGnuCache(this);
	m_pUltraCatcher = new MGnuCache(this);
	m_pHostStore    = new MGnuCache(this);

	m_pKnownFiles = new MGnuMarkedFiles();
	
	m_pShare = NULL;
	VERIFY(new MGnuShare(this));	

	CreateGuid(&m_ClientID);
	m_dCurrentReceiveLimit = 0;
	m_dDownloadRateLimit = 0;
	m_dwReSearchID = 0;
	m_nReSearchTimer = 0;

	m_nPort = 0;
	
	m_nQGood = 0;
	m_nQRepeat = 0;
	m_nQError = 0;
	
	m_dwConnBytesRecv = 0;
	m_dwConnBytesSend = 0;
	m_dwConnRecvRate = 0;
	m_dwConnSendRate = 0;
	
	m_dDownloadRate = 0;
	m_dUploadRate = 0;
	
	m_nLastConnCount = 0;

	m_bForcedSupernode = false;
	m_nHostMode = CM_LEAF; // Ultrapeer or Leaf
	m_nHostModeChangeTime = xtime();
	m_nUltrapeerCredit = 0;
	m_bLastIsShieldedLeaf = false;
	m_bLastEnableSuperNode = false;
	m_nLastNormalConnects = 0;
	m_pWordHashTable = new MGnuWordHash(this);
	m_nExtPongBytes = 0;
	m_nUltraPongBytes = 0;
	m_nStartTime = xtime();
	m_bReceivedIncomingConnections = false;
	
	m_anInitialWebCacheRequests[0] = m_anInitialWebCacheRequests[1] = m_anInitialWebCacheRequests[2] = 0;
	m_bNoValidWebCacheUrlsMessageSent = false;

	m_pAsyncDNS = NULL;
}

MGnuDirector::~MGnuDirector()
{
	MLock lock(m_listmutex);
	//
	map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.begin();
	while( itNode != m_NodeMap.end())
	{
		delete itNode->second;
		m_NodeMap.erase(itNode);
		itNode = m_NodeMap.begin();
	}
	//
	while( m_UploadList.size() )
	{
		delete m_UploadList.back();
		m_UploadList.pop_back();
	}

	while( m_DownloadList.size() )
	{
		delete m_DownloadList.back();
		m_DownloadList.pop_back();
	}

	while( m_WebCacheList.size() )
	{
		delete m_WebCacheList.back();
		m_WebCacheList.pop_back();
	}
	
	m_PendingQueries.clear();
	map<DWORD, MGnuSearch*>::iterator itSearch = m_SearchMap.begin();
	while( itSearch != m_SearchMap.end())
	{
		delete itSearch->second;
		m_SearchMap.erase(itSearch);
		itSearch = m_SearchMap.begin();
	}

	if (m_pHostCatcher)
		delete m_pHostCatcher;
	if (m_pUltraCatcher)
		delete m_pUltraCatcher;
	if (m_pHostStore)
		delete m_pHostStore;
	if (m_pShare)
		delete m_pShare;
	if (m_pWordHashTable)
		delete m_pWordHashTable;

	if (m_pKnownFiles)
		delete m_pKnownFiles;

	if (m_pAsyncDNS)
		m_pAsyncDNS->Release();
	
	m_pController->Detach(this);
}

void MGnuDirector::AttachShare(MGnuShare* pShare)
{
	ASSERT(pShare);
	ASSERT(m_pShare==NULL);
	m_pShare = pShare;
}

void MGnuDirector::DetachShare(MGnuShare* pShare)
{
	ASSERT(m_pShare==pShare);
	m_pShare = NULL;
}

bool MGnuDirector::IsOkForDirectConnect(const IP& host)
{
/*
192.168/16,
172.16/12,
10/8,
169.254/16,
0/8,
240/4
*/
	static IPNET PrivateSubnets[] = {
		MakeIPNET(192,168,0,0,16),
		MakeIPNET(172,16,0,0,16), /* Im not sure if 12 will work */
		MakeIPNET(169,254,0,0,16),
		MakeIPNET(127,0,0,0,8),
		MakeIPNET(10,0,0,0,8),
		MakeIPNET(0,0,0,0,8),
		MakeIPNET(240,0,0,0,8) /* same about 4 */
	};
	static int n = sizeof(PrivateSubnets) / sizeof(IPNET);
	for (int i=0; i<n; ++i)
		if (IsIpInSubnet(host, PrivateSubnets[i]))
			return false;
	return true;
}

bool MGnuDirector::NotLocal(Node TestNode)
{
	IP RemoteIP = TestNode.Host, LocalIP=MakeIP(127,0,0,1);

	UINT LocalPort  = m_nPort;//(m_pPrefs->m_ForcedPort) ? m_pPrefs->m_ForcedPort : m_pPrefs->m_LocalPort;
	UINT RemotePort = TestNode.Port;

	if(LocalPort == RemotePort)
	{
		if(RemoteIP.S_addr == LocalIP.S_addr)
			return false;

		LocalIP.S_addr = (m_pPrefs->m_ipForcedHost.S_addr) ? m_pPrefs->m_ipForcedHost.S_addr : m_pPrefs->m_ipLocalHost.S_addr;

		if(RemoteIP.S_addr == LocalIP.S_addr)
			return false;
	}
	return true;
}

bool MGnuDirector::BehindFirewall()
{
	ASSERT(m_pPrefs);
	return m_pPrefs->m_bBehindFirewall;
}

/////////////////////////////////////////////////////////////////////////////
// LISTEN CONTROL

bool MGnuDirector::StartListening()
{
	m_nPort = m_pPrefs->m_dwLocalPort;

	bool Success  = false;
	int	 Attempts = 0;

	while(!Success && Attempts < 10)
	{
		StopListening();

		if(Create(m_nPort, SOCK_STREAM, FD_ACCEPT))
		{
			if(Listen())
			{
				Success = true;
				m_pPrefs->m_dwLocalPort = m_nPort;
				return true;
			}
			else
			{
				POST_EVENT( MStringEvent(
					ET_ERROR,
					ES_IMPORTANT,
					CString("error: ") + strerror(GetLastError()),
					MSG_FAILED_TO_LISTEN,
					MSGSRC_DIRECTOR
				));
			}
		}
		else
		{
			POST_EVENT( MStringEvent(
				ET_MESSAGE,
				ES_IMPORTANT,
				CString("error: ") + strerror(GetLastError()),
				MSG_FAILED_TO_CREATE_LISTEN_SOCKET,
				MSGSRC_DIRECTOR
			));
		}
		m_nPort += 1;//rand() % 99 + 0;
		POST_MESSAGE(ES_IMPORTANT, CString("Trying port ") + DWrdtoStr(m_nPort));
		Attempts++;
	}

	return false;
}

void MGnuDirector::StopListening()
{
	Close();
}


/////////////////////////////////////////////////////////////////////////////
// CAsyncSocket OVERRIDES

void MGnuDirector::OnAccept(int nErrorCode)
{
	// maybe i should start off with a cgnusock.. ? no idea
	MAsyncSocket New;
	bool Safe = Accept(New);

	if(Safe)
	{
		//
		if (m_pPrefs->m_bBehindFirewall)
		{
			m_pPrefs->m_bBehindFirewall = false; //we can accept
			POST_MESSAGE(ES_UNIMPORTANT, "Accepted incoming connection -- resetting 'Firewall' flag");
		}
		//
		if(AllowIncomingConnections())
		{
			MGnuSock* pIncomming = new MGnuSock(this);
			ASSERT(pIncomming);
			
			SOCKET Free = New.Detach();
			if (Free != INVALID_SOCKET)
			{
				pIncomming->Attach(Free, FD_READ | FD_CLOSE);
				AddSock(pIncomming);
				return;
			}
			else
				delete pIncomming;
		}
		New.Close();
	}
	else
	{
		POST_ERROR(ES_UNIMPORTANT, CString("Node Accept Error: ") + strerror(GetLastError()));
	}
		
	MAsyncSocket::OnAccept(nErrorCode);
}


/////////////////////////////////////////////////////////////////////////////
// TRAFFIC CONTROL

void MGnuDirector::Post_QueryHit(QueryComp* pQuery, const BYTE* pHit, DWORD ReplyLength, BYTE ReplyCount)
{
	/*int packetLength = sizeof(packet_QueryHit) + ReplyLength + sizeof(packet_QueryHitEx) + sizeof(GUID);
    char* pPacket = new char[packetLength];
    ASSERT(pPacket);
	packet_QueryHit*  pQueryHit = (packet_QueryHit*)pPacket;
	packet_QueryHitEx* pQHD = (packet_QueryHitEx*) (pPacket+sizeof(packet_QueryHit)+ReplyLength);
	GUID* pClientID = (GUID*) (pPacket+sizeof(packet_QueryHit)+ReplyLength+sizeof(packet_QueryHitEx));
	//
	memcpy(pPacket+sizeof(packet_QueryHit), pHit, ReplyLength);
	// Build Query Packet
	pQueryHit->Header.Guid	    = pQuery->QueryGuid;
	pQueryHit->Header.Function  = 0x81;
	pQueryHit->Header.TTL	    = pQuery->nHops;
	//TRACE2("Posting QueryHit with TTL = ", (int) pQueryHit->Header.TTL);
	pQueryHit->Header.Hops	    = 0;
	pQueryHit->Header.le_Payload = packetLength - sizeof(packet_Header);

	pQueryHit->TotalHits		    = ReplyCount;
	pQueryHit->le_Port		    = (WORD) GetPublicPort();
	pQueryHit->Host			    = GetPublicIP();
	pQueryHit->le_Speed		    = m_pPrefs->m_dwSpeedStat ? m_pPrefs->m_dwSpeedStat : m_pPrefs->m_dwSpeedDyn;

	// Add Query Hit Descriptor
	bool Busy = m_pPrefs->m_nMaxUploads >= 0 && CountUploads() >= m_pPrefs->m_nMaxUploads;
	
	memcpy(pQHD->VendorID, "MUTE", 4);
	pQHD->Length		= 2;
	pQHD->FlagPush		= 0;//true;
	pQHD->FlagBad		= 1;//true;
	pQHD->FlagBusy		= 1;//true;
	pQHD->FlagStable	= 1;//true;
	pQHD->FlagSpeed		= 1;//true;
	pQHD->FlagTrash		= 0;
	pQHD->Push			= BehindFirewall();
	pQHD->Bad			= 0;//false;
	pQHD->Busy			= 0;//false;//Busy;
	pQHD->Stable		= 1;//false;//true;//m_pDirector->HaveUploaded();
	pQHD->Speed			= 0;//false;//true;//m_pDirector->GetRealSpeed() ? true : false;
	pQHD->Trash			= 0;
	// Add our ID
	memcpy(pClientID, GetClientID(), sizeof(GUID));
	// post the packet
	QueuedHit qh;
	qh.size = packetLength;
	qh.pPacket = pPacket;
	qh.dwNodeID = pQuery->OriginID;
	ASSERT(qh.dwNodeID);
	m_hitqueuemutex.lock();
	m_hitQueue.push(qh);
	m_hitqueuemutex.unlock();
	{
		MLock lock(m_listmutex);
		for(int i = 0; i < m_NodeList.size(); i++)	
		{
			MGnuNode *p = m_NodeList[i];
			if(p == pQuery->Origin)
			{
				p->Send(pPacket, packetLength);
				delete [] pPacket;
				return;
			}
		}
	}*/

	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(pQuery->OriginID);
	if( itNode != m_NodeMap.end() &&
		itNode->second->IsConnected() )
			itNode->second->Send_QueryHit(*pQuery, pHit, ReplyLength, ReplyCount, "");
}

void MGnuDirector::Broadcast_Ping(packet_Ping* Ping, int length, MGnuNode* exception)
{
	if(m_bLastIsShieldedLeaf)
		return;

	MGnuNode * pNode;
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		pNode = itNode->second;
		ASSERT(pNode);
		if( m_nHostMode == CM_ULTRAPEER     &&
			pNode->GetPeerMode() == CM_LEAF ) // it could have crashed here
				continue;
		if( pNode != exception   &&
			pNode->IsConnected() )
				pNode->SendPacket(Ping, length, PACKET_PING, false);
	}
}

void MGnuDirector::Send_ForwardQuery(const QueryComp& Query, const list<DWORD>& NodeIDs )
{
	if(m_bLastIsShieldedLeaf)
		return;

	MLock lock(m_listmutex);
	for (list<DWORD>::const_iterator itID = NodeIDs.begin(); itID != NodeIDs.end(); ++itID)
	{
		if (Query.OriginID == *itID)
			continue;
		map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(*itID);
		if (itNode != m_NodeMap.end()     &&
			itNode->second->IsConnected() )
				itNode->second->Send_ForwardQuery(Query);
	}
}

void MGnuDirector::Broadcast_Query(packet_Query* Query, int length, MGnuNode* exception)
{
	if(m_bLastIsShieldedLeaf)
		return;

	MGnuNode * pNode;
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		pNode = itNode->second;
		ASSERT(pNode);
		if( m_nHostMode == CM_ULTRAPEER     &&
			pNode->GetPeerMode() == CM_LEAF ) // it could have crashed here
				continue;
		if( pNode != exception   &&
			pNode->IsConnected() )
				pNode->SendPacket(Query, length, PACKET_QUERY, false);
	}
}

GUID MGnuDirector::Broadcast_LocalQuery(BYTE* Packet, int length)
{
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
	{
		POST_ERROR(ES_UNIMPORTANT, "Failed to create a GUID to broadcast query");
		return Guid;
	}

	packet_Query* Query = (packet_Query*) Packet;

	Query->Header.Guid = Guid;
	Query->Header.Function = 0x80;
	Query->Header.Hops = 0;
	Query->Header.TTL = MAX_TTL_SELF;
	Query->Header.le_Payload = length - 23;
	Query->le_Speed = 0;

	m_TableLocal.Insert(&Guid, 0ul);

	{
		MLock lock(m_listmutex);
		map<DWORD, MGnuNode*>::iterator itNode;
		for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		{
			if(itNode->second->IsConnected())
			{
#ifdef _PACKET_DEBUG
				POST_EVENT( MPacketEvent(
					(packet_Header*)Packet,
					length,
					itNode->second->GetHost(),
					MSG_PACKET_SENT
				));
#endif
				itNode->second->SendPacket(Packet, length, PACKET_QUERY, false); /* was true here! */
			}
		}
	}
	return Guid;
}

/*void MGnuDirector::SendAllLocalQueries(MGnuNode* pNode)
{
	ASSERT(pNode);
	MLock lock(m_listmutex);
	for(int i = 0; i < m_SearchList.size(); i++)
	{
		MGnuSearch *pS = m_SearchList[i];
		if (pS->m_Packet != NULL && pS->m_nPacketLength != 0)
			pNode->Send(pS->m_Packet, pS->m_nPacketLength);
    }
}*/

/*void MGnuDirector::Route_Pong(packet_Pong* Pong, int length, key_Value* key)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_NodeList.size(); i++)
	{
		MGnuNode *p = m_NodeList[i];
		if(p == key->Origin && p->IsConnected())
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*)Pong,
				length,
				p->GetHost(),
				MSG_PACKET_SENT
			));
#endif
			p->Send((BYTE*) Pong, length);
		}
	}
}

void MGnuDirector::Route_QueryHit(packet_QueryHit* QueryHit, DWORD length, key_Value* key)
{
	MLock lock(m_listmutex);
	//
	DWORD BytestoRead = length;
	for(int i = 0; i < m_NodeList.size(); i++)
	{
		MGnuNode *p = m_NodeList[i];

		if(p == key->Origin && p->IsConnected())
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*)QueryHit,
				length,
				p->GetHost(),
				MSG_PACKET_SENT
			));
#endif
			p->Send((BYTE*) QueryHit, length);
			return;
		}
	}
}

void MGnuDirector::Route_Push(packet_Push* Push, int length, key_Value* key)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_NodeList.size(); i++)
	{
		MGnuNode *p = m_NodeList[i];
		if(p == key->Origin && p->IsConnected())
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*)Push,
				length,
				p->GetHost(),
				MSG_PACKET_SENT
			));
#endif
			p->Send((BYTE*) Push, length);
			return;
		}
	}
}

bool MGnuDirector::Route_LocalPush(const Result& Download)
{
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
	{
		POST_ERROR(ES_UNIMPORTANT, "Failed to create a GUID to route local push");
		return false;
	}

	// Create packet
	packet_Push Push;

	Push.Header.Guid		= Guid;
	Push.Header.Function	= 0x40;
	Push.Header.TTL			= Download.Distance;
	Push.Header.Hops		= 0;
	Push.Header.le_Payload	= 26;
	Push.ServerID			= Download.PushID;
	Push.le_Index			= Download.FileIndex;
	Push.Host				= GetPublicIP();
	Push.le_Port			= (WORD) GetPublicPort();

	{
		MLock lock(m_listmutex);
		// Send Push
		for(int i = 0; i < m_NodeList.size(); i++)
		{
			MGnuNode *p = m_NodeList[i];

			if(p == Download.Origin && p->IsConnected())
			{
#ifdef _PACKET_DEBUG
				POST_EVENT( MPacketEvent(
					(packet_Header*)&Push,
					49,
					p->GetHost(),
					MSG_PACKET_SENT
				));
#endif
				p->Send(&Push, 49);
				return true;
			}
		}
		return false;
	}
}*/

bool MGnuDirector::Route_LocalPush(const Result& Download)
{
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
	{
		POST_ERROR(ES_UNIMPORTANT, "Failed to create a GUID to route local push");
		return false;
	}

	// Create packet
	packet_Push Push;

	Push.Header.Guid		= Guid;
	Push.Header.Function	= 0x40;
	Push.Header.TTL			= Download.Distance;
	Push.Header.Hops		= 0;
	Push.Header.le_Payload	= 26;
	Push.ServerID			= Download.PushID;
	Push.le_Index			= Download.FileIndex;
	Push.Host				= GetPublicIP();
	Push.le_Port			= (WORD) GetPublicPort();

	{
		// Send Push
        MLock lock(m_listmutex);
        map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(Download.OriginID);
        if (itNode != m_NodeMap.end() && itNode->second->IsConnected())
        {
            m_TableLocal.Insert(&Guid, Download.OriginID);
            itNode->second->SendPacket(&Push, 49, PACKET_PUSH, true);
            return true;
        }
    }
    return false;
}

////////////////////////////////////////////////////////////////////////////
// Node control

void MGnuDirector::AddNode(IP ipHost, UINT Port, bool bForceV4)
{
	if(!bForceV4 && FindNode(ipHost, Port) != NULL)
		return;
	//if (!IsOkForDirectConnect(ipHost))
	//	return;

	MGnuNode* pNode = new MGnuNode(this, ipHost, Port, false, bForceV4);
	ASSERT(pNode);

	//Attempt to connect to node
	if(!pNode->Create(0, SOCK_STREAM, FD_READ | FD_CONNECT | FD_CLOSE))
	{
		POST_ERROR(ES_IMPORTANT, CString("Node Create Error: ") + DWrdtoStr(pNode->GetLastError()));
		delete pNode;
		return;
	}

	if( !pNode->Connect(Ip2Str(ipHost).c_str(), Port) )
	{
		if (pNode->GetLastError() != EINPROGRESS)
		{
			delete pNode;
			return;
		}
    }
	// Add node to list
	{
		MLock lock(m_listmutex);
		//
		pNode->m_dwID = GenID();
		m_NodeMap[pNode->m_dwID] = pNode;
	}
}

void MGnuDirector::RemoveNode(MGnuNode* pNode)
{
	MLock lock(m_listmutex);
	//
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if(itNode->second == pNode)
		{
			pNode->ForceDisconnect();
			m_NodeMap.erase(itNode);
			delete pNode;
			return;
		}
}

void MGnuDirector::RemoveNode(DWORD dwID)
{
	MLock lock(m_listmutex);
	//
	map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(dwID);
	if (itNode == m_NodeMap.end())
		return;
	// Make sure socket is closed
	itNode->second->ForceDisconnect();
	m_NodeMap.erase(itNode);
	delete itNode->second;
}

bool MGnuDirector::IsInTheList(MGnuNode* pNode)
{
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if(itNode->second == pNode)
			return true;
	return false;
}

MGnuNode* MGnuDirector::FindNode(IP ipHost, UINT nPort)
{
	MLock lock(m_listmutex);
	//
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if(itNode->second->GetHost().S_addr == ipHost.S_addr)
			return itNode->second;
	return NULL;
}

/////////////////////////////////////////////////////////////////////////////
// TRANSFER CONTROL

int MGnuDirector::CountConnections()
{
	int nConnected = 0;
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if(itNode->second->IsConnected())
			nConnected++;
	return nConnected;
}

int MGnuDirector::CountIncomingConns()
{
	int nConnected = 0;
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if(itNode->second->IsConnected() && itNode->second->IsIncoming())
			nConnected++;
	return nConnected;
}

int MGnuDirector::CountConnPerSubnet(int SubnetBits, IP ipHost)
{
	u_long ulSubnetMask = (1<<SubnetBits) - 1; // OK for little-endian mashines
	MByteOrderTest bot;
	if (bot.IsBigEndian())
		ulSubnetMask = VBO(ulSubnetMask, true);
	u_long ulSubnet = ulSubnetMask & ipHost.S_addr;
	//
	int nConnected = 0;
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if(itNode->second->IsConnected() &&
		    ((itNode->second->GetSGnuNode().m_ipHost.S_addr & ulSubnetMask) == ulSubnet) )
			nConnected++;

	return nConnected;
}

int MGnuDirector::CountUploads()
{
	MLock lock(m_listmutex);
	//
	int NumUploading = 0;
	for(int i = 0; i < m_UploadList.size(); i++)
	{
		MGnuUpload *p = m_UploadList[i];

		if(p->m_nStatus == TRANSFER_SENDING)
			NumUploading++;
	}

	return NumUploading;
}

int MGnuDirector::CountTotalDownloads()
{
	MLock lock(m_listmutex);
	return m_DownloadList.size();
}

int MGnuDirector::CountActiveDownloads()
{
	MLock lock(m_listmutex);
	//
	int NumDownloading = 0;
	for(int i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload *p = m_DownloadList[i];
		NumDownloading += p->CountActiveConnections(NULL);
	}
	//
	return NumDownloading;
}

int MGnuDirector::CountActiveDownloads(const HOST& from, MGnuDownload* pExclude)
{
	MLock lock(m_listmutex);
	int nCount = 0;
	//
	for(int i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload *p = m_DownloadList[i];
		if(p != pExclude)
			nCount += p->CountActiveConnections(&from);
	}
	return nCount;
}

int MGnuDirector::CountUploads(const IP& host, MGnuUpload* pExcept)
{
	MLock lock(m_listmutex);
	//
	int nCount = 0;
	for(int i = 0; i < m_UploadList.size(); i++)
		if( (m_UploadList[i]->m_nStatus == TRANSFER_CONNECTED ||
		     m_UploadList[i]->m_nStatus == TRANSFER_SENDING ) &&
		     m_UploadList[i] != pExcept                       &&
		     m_UploadList[i]->m_ipHost.S_addr == host.S_addr     )
			++nCount;
	return nCount;
}

int MGnuDirector::CountUploads(int nFileIndex, MGnuUpload* pExcept)
{
	MLock lock(m_listmutex);
	//
	int nCount = 0;
	for(int i = 0; i < m_UploadList.size(); i++)
		if( (m_UploadList[i]->m_nStatus == TRANSFER_CONNECTED ||
		     m_UploadList[i]->m_nStatus == TRANSFER_SENDING ) &&
		     m_UploadList[i] != pExcept                       &&
		     m_UploadList[i]->m_nFileIndex == nFileIndex         )
			++nCount;
	return nCount;
}

MGnuDownload* MGnuDirector::LookUpDownload(IP ipHost, CString sFileName, int nIndex)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload *p = m_DownloadList[i];
		if (p->LookUp(ipHost, sFileName, nIndex)>=0)
			return p;
	}
	return NULL;
}


bool MGnuDirector::CheckIfPushung(packet_Push* Push)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_UploadList.size(); i++)
	{
		MGnuUpload* p = m_UploadList[i];
		if(p->m_ipHost.S_addr == Push->Host.S_addr &&
		   p->m_nPort		== Push->le_Port &&
		   p->m_nFileIndex	== Push->le_Index )
			return true;
	}
	return false;
}

void MGnuDirector::AddSock(MGnuSock* pSock)
{
	MLock lock(m_listmutex);
	//
	m_SockList.push_back(pSock);
}

void MGnuDirector::AddNode(MGnuNode* pNode)
{
	MLock lock(m_listmutex);
	//
	pNode->m_dwID = GenID();
	m_NodeMap[pNode->m_dwID] = pNode;
}

void MGnuDirector::AddUpload(MGnuUpload* pUpload)
{
	MLock lock(m_listmutex);
	//
	pUpload->m_dwID = GenID();
	pUpload->SetMaxRate(4096); // set rather low speed for the first second -- it will be updated later
	m_UploadList.push_back(pUpload);
}

bool MGnuDirector::AddDownload(const ResultVec& results)
{
	if (!results.size())
		return false;
	// check whether we are downloading any of those already and
	// at the same time get the shortest filename
	int nShortestNameIndex = 0;
	for (int i=0; i<results.size();++i)
	{
		// check if we already downloading this
		if ( LookUpDownload(results[i].Host, results[i].Name, results[i].FileIndex) )
		{
			return false; // todo: something more appropriate
		}
		if (results[i].Name.length()<results[nShortestNameIndex].Name.length())
			nShortestNameIndex = i;
	}
	// get free file handle 
	long nFileHandle = m_pKnownFiles->GetFreeFileHandle(results[0].Size);
	// create the download object
	MGnuDownload* pD =  new MGnuDownload(this, results, results[nShortestNameIndex].Name, nFileHandle);
	ASSERT(pD);
	// add the file to the known files DB
	CString sCommonSearch = MakeSearchOfFilename(results[nShortestNameIndex].Name);
	for (int i=0; i<results.size();++i)
	{
		CString s = MakeSearchOfFilename(results[i].Name);
		m_pKnownFiles->AddFile(MFT_Attempted, results[i].Size, results[i].Sha1, s, nFileHandle);
		if (s != sCommonSearch)
			m_pKnownFiles->AddFile(MFT_Attempted, results[i].Size, results[i].Sha1, sCommonSearch, nFileHandle);
	}
	// add to download queue
	AddAndStartDownload(pD);	
	return true;
}

bool MGnuDirector::RenameDownload(DWORD dwID, LPCSTR szNewFilename)
{
	if (dwID == 0)
		return false;
	MLock lock(m_listmutex);
	for (vector<MGnuDownload*>::iterator itd = m_DownloadList.begin(); itd != m_DownloadList.end(); ++itd)
	{
		if ((*itd)->GetID()==dwID)
		{
			return (*itd)->Rename(szNewFilename);
		}
	}
	return false;
}

void MGnuDirector::AddAndStartDownload(MGnuDownload* pDownload)
{
	m_listmutex.lock();
	//
	pDownload->GenerateID();
	m_DownloadList.push_back(pDownload);
	m_listmutex.unlock();
	pDownload->Start();
}

MGnuSearch* MGnuDirector::LookUpSearch(const CString& search, int size)
{
	MLock lock(m_listmutex);
	map<DWORD, MGnuSearch*>::iterator itS;
	for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); ++itS)
		if ( LIMIT_EXACTLY == itS->second->GetSizeFilterMode() &&
		     size == itS->second->GetSizeFilterValue()         &&
		     search == itS->second->GetSearchString()          )
		{
			return itS->second;
		}
	return NULL;
}

MGnuSearch* MGnuDirector::GetSearchByFile(const CString& name, int size)
{
	MLock lock(m_listmutex);
	map<DWORD, MGnuSearch*>::iterator itS;
	for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); ++itS)
		if ( LIMIT_EXACTLY == itS->second->GetSizeFilterMode() &&
		     size == itS->second->GetSizeFilterValue()         &&
		     name == itS->second->GetFilename()                )
		{
			return itS->second;
		}
	return NULL;
}

MGnuSearch* MGnuDirector::AddSearch(LPCSTR szSearch, int type, int size, int sizeMode, bool bAutoGet /*=false*/ )
{
	// check for repeating or overlaping
	MLock lock(m_listmutex);
	if (m_SearchMap.size()>=m_pPrefs->m_nMaxSearches)
		return NULL;
	MGnuSearch* pS = NULL;
	if (strlen(szSearch)==37 &&
		0 == memcmp(szSearch, "sha1:", 4))
	{
		// we seem to get SHA1 search
		SHA1Hash sha1;
		// check for Base32 decoding of the hash
		if (sha1.fromStr(szSearch+5))
		{
			map<DWORD, MGnuSearch*>::iterator itS;
			for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); ++itS)
			{
				if ( itS->second->IsInList(sha1) )
					return NULL;
			}
			pS = new MGnuSearch(this, "", sha1, type, size, sizeMode);
		}
	}
	else
	{
		// ordinary string search
		map<DWORD, MGnuSearch*>::iterator itS;
		for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); ++itS)
			if ( sizeMode == itS->second->GetSizeFilterMode()         &&
				 size == itS->second->GetSizeFilterValue()            &&
				 QueryMatch(szSearch, itS->second->GetSearchString()) &&
				 QueryMatch(itS->second->GetSearchString(), szSearch) )
				 return NULL; // mutualy matching searches are redundant
		 // TODO: above check conflicts slightly with exclusive searches -- fix it

		pS = new MGnuSearch(this, szSearch, SHA1Hash(), type, size, sizeMode);
	}
	ASSERT(pS);
	pS->SetAutoget(bAutoGet);
	// add to the list
	pS->SetID(GenID());
	ASSERT(m_SearchMap.find(pS->GetID()) == m_SearchMap.end());
	m_SearchMap[pS->GetID()] = pS;
	// queue sends and do them on timer
	m_PendingQueries.push_back(pS->GetID());
	return pS;
}

bool MGnuDirector::AddSearch(MGnuSearch* pSearch)
{
	// check for repeating or overlaping
	MLock lock(m_listmutex);
	if (m_SearchMap.size()>=m_pPrefs->m_nMaxSearches)
		return false;
	// add to the list
	pSearch->SetID(GenID());
	ASSERT(m_SearchMap.find(pSearch->GetID()) == m_SearchMap.end());
	m_SearchMap[pSearch->GetID()] = pSearch;
	// queue sends and do them on timer
	m_PendingQueries.push_back(pSearch->GetID());
	return true;
}

bool MGnuDirector::ModifySearch(DWORD dwID, LPCSTR szSearch)
{
	if (dwID == 0)
		return false;
	if (strlen(szSearch)<4)
		return false;
	m_listmutex.lock();
	map<DWORD, MGnuSearch*>::iterator itS = m_SearchMap.find(dwID);
	if (itS != m_SearchMap.end())
	{
		ASSERT(dwID == itS->second->GetID());
		// so now we have the search, let's modify it!
		itS->second->SetSearchString(szSearch);
		// TODO: check if this may produce some interference with Download
		m_listmutex.unlock();
		return true;
	}
	m_listmutex.unlock();
	return false;
}

void MGnuDirector::ForEachSearch(void* pData, tEachSearch callback)
{
	MLock lock(m_listmutex);
	SGnuSearch gs;
	map<DWORD, MGnuSearch*>::iterator itS;
	for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); ++itS)
	{
		itS->second->GetSGnuSearch(gs);
		if (!(*callback)(pData, &gs))
			return;
	}
}

int MGnuDirector::GetSearchesCount()
{
	MLock lock(m_listmutex);
	return m_SearchMap.size();
}

MGnuSearch* MGnuDirector::GetSearchByID(DWORD dwID)
{
	if (dwID == 0)
		return NULL;
	m_listmutex.lock();
	map<DWORD, MGnuSearch*>::iterator itS = m_SearchMap.find(dwID);
	if (itS != m_SearchMap.end())
	{
		MGnuSearch* pSearch = itS->second;
		m_listmutex.unlock();
		return pSearch;
	}
	m_listmutex.unlock();
	return NULL;
}

bool MGnuDirector::GetSearchByID(DWORD dwID, SGnuSearch& gs, vector<Result>& rv, vector<ResultGroup>& gv)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	map<DWORD, MGnuSearch*>::iterator itS = m_SearchMap.find(dwID);
	if (itS != m_SearchMap.end())
	{
		MGnuSearch* pSearch = itS->second;
		pSearch->GetAllResults(&gs, &rv, &gv);
		/*
		pSearch->m_mutex.lock();
		gs = *pSearch;
		// now it's time to package results into a vector and create a map if IDs to indeces
		map<DWORD,DWORD> mapIndex;
		int nIndex = 0;
		rv.reserve(pSearch->m_mapResults.size());
		for (map<DWORD,Result>::iterator itr = pSearch->m_mapResults.begin(); itr != pSearch->m_mapResults.end(); ++itr)
		{
			rv.push_back(itr->second);
			ASSERT(itr->first == itr->second.dwID);
			mapIndex[itr->second.dwID] = nIndex;
			nIndex++;
		}
		// copy groups into a vector and re-encode ResultSet
		gv.reserve(pSearch->m_listGroups.size());
		for (list<ResultGroup>::iterator itg = pSearch->m_listGroups.begin(); itg != pSearch->m_listGroups.end(); ++itg)
		{
			gv.push_back(*itg);
			gv.back().ResultSet.clear();
			for (set<DWORD>::iterator iti = itg->ResultSet.begin(); iti != itg->ResultSet.end(); ++iti)
				gv.back().ResultSet.insert(mapIndex[*iti]);
		}
		//
		pSearch->m_mutex.unlock();
		*/
		m_listmutex.unlock();
		return true;
	}
	m_listmutex.unlock();
	return false;
}

bool MGnuDirector::GetResultsByID(DWORD dwID, vector<Result>& rv)
{
	if (dwID == 0)
		return false;
	rv.clear();
	m_listmutex.lock();
	map<DWORD, MGnuSearch*>::iterator itS;
	for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); ++itS)
	{
		vector<Result> rvl;
		vector<ResultGroup> gvl;
		MGnuSearch* pSearch = itS->second;
		pSearch->GetAllResults(NULL, &rvl, &gvl);
		for (vector<ResultGroup>::iterator itg = gvl.begin() ; itg!=gvl.end(); ++itg)
		{
			if ( (*itg).dwID == dwID )
			{
				if ( 0 == (*itg).ResultSet.size() )
				{
					// should never happen
					m_listmutex.unlock();
					return false;
				}
				rv.reserve((*itg).ResultSet.size());
				for (set<int>::iterator iti = (*itg).ResultSet.begin(); iti!=(*itg).ResultSet.end(); ++iti)
					rv.push_back(rvl[*iti]);
				//
				m_listmutex.unlock();
				return true;
			}
		}
		for (vector<Result>::iterator itr = rvl.begin() ; itr!=rvl.end(); ++itr)
		{
			if ( (*itr).dwID == dwID )
			{
				rv.push_back(*itr);
				//
				m_listmutex.unlock();
				return true;
			}
		}
		/*
		pSearch->m_mutex.lock();
		for (list<ResultGroup>::iterator itg = pSearch->m_listGroups.begin() ; itg!=pSearch->m_listGroups.end(); ++itg)
		{
			if ( (*itg).dwID == dwID )
			{
				if ( 0 == (*itg).ResultSet.size() )
				{
					// should never happen
					pSearch->m_mutex.unlock();
					m_listmutex.unlock();
					return false;
				}
				rv.reserve((*itg).ResultSet.size());
				for (set<DWORD>::iterator iti = (*itg).ResultSet.begin(); iti!=(*itg).ResultSet.end(); ++iti)
					rv.push_back(pSearch->m_mapResults[*iti]);
				//
				pSearch->m_mutex.unlock();
				m_listmutex.unlock();
				return true;
			}
		}
		map<DWORD,Result>::iterator itr = pSearch->m_mapResults.find(dwID);
		if (itr != pSearch->m_mapResults.end())
		{
			ASSERT(itr->second.dwID == dwID);
			rv.push_back(itr->second);
			//
			pSearch->m_mutex.unlock();
			m_listmutex.unlock();
			return true;
		}
		pSearch->m_mutex.unlock();
		*/
	}
	m_listmutex.unlock();
	return false;
}

bool MGnuDirector::ClearSearchByID(DWORD dwID)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	map<DWORD, MGnuSearch*>::iterator itS = m_SearchMap.find(dwID);
	if (itS != m_SearchMap.end())
	{
		itS->second->Clear();
		m_listmutex.unlock();
		return true;
	}
	m_listmutex.unlock();
	return false;
}

bool MGnuDirector::RemoveSearchByID(DWORD dwID, bool bRemovePartial)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	map<DWORD, MGnuSearch*>::iterator itS = m_SearchMap.find(dwID);
	if (itS != m_SearchMap.end())
	{
		MGnuSearch* pSearch = itS->second;
		m_SearchMap.erase(itS);
		m_listmutex.unlock();
		if (bRemovePartial &&
			pSearch->GetFilename().length() &&
			pSearch->GetSizeFilterMode() == LIMIT_EXACTLY )
		{
			CString sPartPath;
			BuildDownloadPaths(pSearch->GetFilename(), pSearch->GetSizeFilterValue(), NULL, NULL, &sPartPath);
			if (!IsDownloading(sPartPath))
			{
				// non-blocking file deletion
				MAsyncFile* pDelFile = new MAsyncFile(AFM_READWRITE);
				MRequestDelete* pDel = new MRequestDelete(sPartPath);
				ASSERT(pDel);
				pDelFile->CustomRequest(pDel);
				pDel->Release();
				delete pDelFile;
			}
		}
		delete pSearch;
		return true;
	}
	m_listmutex.unlock();
	return false;
}

void MGnuDirector::RemoveSearchesByType(int nType)
{
	m_listmutex.lock();
	map<DWORD, MGnuSearch*>::iterator itS;
	for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); )
	{
		if (itS->second->GetType()==nType)
			m_SearchMap.erase(itS++);
		else
			++itS;
	}
	m_listmutex.unlock();
}

bool MGnuDirector::IsDownloading(const CString& sPartPath)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload *p = m_DownloadList[i];
		if (p->IsPartOf(sPartPath))
			return true;
	}
	return false;
}

void MGnuDirector::BuildDownloadPaths(CString sFileName, int nFileSize, CString* psFileName, CString* psFilePath, CString* psPartPath)
{
	ReplaceSubStr(sFileName,"\\", "/");
	ReplaceSubStr(sFileName,"`", " ");
	ReplaceSubStr(sFileName,"'", " ");
	ReplaceSubStr(sFileName,"\"", " ");
	// remove path
	sFileName = sFileName.substr( sFileName.rfind("/") + 1);
	// makes life much easier
	MakeLower(sFileName);
	//
	if (psFileName)
		*psFileName = sFileName;
	
	// Set where the file will go upon completion
	if (psFilePath)
	{
		*psFilePath = ExpandPath(m_pPrefs->m_szDownloadPath) + "/" + sFileName;
		ReplaceSubStr(*psFilePath,"//", "/");
	}
	// Add tail to it so we can support partial downloads
	if (psPartPath)
	{
		sFileName += ".[[" + DWrdtoStr(nFileSize) + "]]";
		// put the file to the partial directory
		*psPartPath = ExpandPath(m_pPrefs->m_szDownloadPath) + "/part" + "/" + sFileName;
		ReplaceSubStr(*psPartPath, "//", "/");
	}
}

void MGnuDirector::ForEachConnection(void* pData, tEachConnection callback)
{
	MLock lock(m_listmutex);
	SGnuNode gn;
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		gn = itNode->second->GetSGnuNode();
		if (!(*callback)(pData, &gn))
			return;
	}
}

bool MGnuDirector::CloseConnectionByID(DWORD dwID)
{
	if (dwID == 0)
		return false;
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(dwID);
	if (itNode == m_NodeMap.end())
		return false;
	ASSERT(itNode->second->GetID() == dwID);
	itNode->second->ForceDisconnect(true);
	return true;
}

void MGnuDirector::ForEachUpload(void* pData, tEachUpload callback)
{
	MLock lock(m_listmutex);
	SGnuUpload gu;
	for (int i = 0; i<m_UploadList.size(); ++i)
	{
		m_UploadList[i]->m_mutex.lock();
		gu = *m_UploadList[i];
		m_UploadList[i]->m_mutex.unlock();
		if (!(*callback)(pData, &gu))
			return;
	}
}

void MGnuDirector::ForEachDownload(void* pData, tEachDownload callback)
{
	MLock lock(m_listmutex);
	SGnuDownload gd;
	for (int i = 0; i<m_DownloadList.size(); ++i)
	{
		m_DownloadList[i]->CopyTo(gd);
		//ASSERT(gd.m_nQueuePos < gd.m_Queue.size());
		if (!(*callback)(pData, &gd))
			return;
	}
}

bool MGnuDirector::RemoveTransferByID(DWORD dwID, bool bDelPart)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	for (int i = 0; i<m_DownloadList.size(); ++i)
		if ( m_DownloadList[i]->GetID() == dwID )
		{
			MGnuDownload* pD = m_DownloadList[i];
			pD->Stop(bDelPart,bDelPart);
			m_listmutex.unlock();
			return true;
		}
	//
	for (int i = 0; i<m_UploadList.size(); ++i)
		if ( m_UploadList[i]->m_dwID == dwID )
		{
			MGnuUpload* pU = m_UploadList[i];
			pU->m_mutex.lock();
			pU->ForceDisconnect();
			pU->m_mutex.unlock();
			m_listmutex.unlock();
			return true;
		}
	m_listmutex.unlock();
	return false;
}


// original decode
static CString
GGEP_decode(BYTE *Packet, int Length)
{
	char Flags;
	char ID[16], Buffer[Length + 256];
	int IDLen, BufferLen = 0, i = 0, j;

	BufferLen = sprintf(Buffer, "GGEP Block: ");
	while (i < Length)
	{
		Flags = Packet[i++];
		IDLen = Flags & 0x0f;
		for (j = 0; j < IDLen && i < Length; ++j)
			ID[j] = Packet[i++];
		ID[j] = '\0';
		BufferLen += sprintf(&Buffer[BufferLen], "ID='%s' ", ID);

		BYTE Byte;
		int Amount = 0;
		for (j = 0; j < 3 && i < Length; ++j)
		{
			Byte = Packet[i++];
			Amount |= Byte & 0x3f;
			if (Byte & 0x40)
				break;
			Amount <<= 6;
		}

		if (Flags & 0x20)
			BufferLen += sprintf(&Buffer[BufferLen], "DATA=<compressed (not yet supported)> ");
		else
		{
			if (Flags & 0x40)
				BufferLen += sprintf(&Buffer[BufferLen], "DATA=<COBS encoded (not yet supported)> ");
			else
			{
				if (IDLen == 1 && ID[0] == 'H')
					BufferLen += sprintf(&Buffer[BufferLen], "DATA=<sha1 moved>ggep:sha1:%s", &Packet[i]);
				else
					BufferLen += sprintf(&Buffer[BufferLen], "DATA=<discarded (unknown extension)> ");
			}
		}
		i += Amount;

		if (Flags & 0x80)
			break;
	}
	Buffer[BufferLen++] = '\0';
	return CString(Buffer);
}

// modified by MZ /* crashes */
/*
static CString
GGEP_decode(BYTE *Packet, int Length)
{
	char Flags;
	char ID[16];
	CString sBuffer, sTmp;
	sBuffer.reserve(Length + 256); // why 256?
	int IDLen, i = 0, j;

	sBuffer = "GGEP Block: ";
	while (i < Length)
	{
		Flags = Packet[i++];
		IDLen = Flags & 0x0f;
		for (j = 0; j < IDLen && i < Length; ++j)
			ID[j] = Packet[i++];
		ID[j] = '\0';
		sTmp.format("ID='%s' ", ID);
		sBuffer += sTmp;

		BYTE Byte;
		int Amount = 0;
		for (j = 0; j < 3 && i < Length; ++j)
		{
			Byte = Packet[i++];
			Amount |= Byte & 0x3f;
			if (Byte & 0x40)
				break;
			Amount <<= 6;
		}

		if (Flags & 0x20)
			sBuffer += "DATA=<compressed (not yet supported)> ";
		else
		{
			if (Flags & 0x40)
				sBuffer += "DATA=<COBS encoded (not yet supported)> ";
			else
			{
				if (IDLen == 1 && ID[0] == 'H')
				{
					sTmp.format("DATA=<sha1 moved>ggep:sha1:%s", &Packet[i]);
					sBuffer += sTmp;
				}
				else
					sBuffer += "DATA=<discarded (unknown extension)> ";
			}
		}
		i += Amount;

		if (Flags & 0x80)
			break;
	}
	return sBuffer;
}*/


void MGnuDirector::OnQueryHit(packet_QueryHit* QueryHit, MGnuNode* pOrigin, bool bOur)
{
	if (!bOur && m_pPrefs->m_bStrictSearch)
		return;
	if (QueryHit->Header.Function != 0x81)
		return;
	if (!m_pPrefs->m_bReachableForPush && !IsOkForDirectConnect(QueryHit->Host))
		return;
	if (m_pPrefs->m_bBlackListActive && m_pPrefs->m_ipnetsetBlackList.IsIn(QueryHit->Host))
		return;

	BYTE*			 Packet   = (BYTE*) QueryHit;

	bool ExtendedPacket = false;
	bool Firewall		= false;
	bool Busy			= false;
	bool Stable			= false;
	bool ActualSpeed	= false;

	int    HitsLeft = QueryHit->TotalHits, i, j;
	DWORD  NextPos  = 34;
	DWORD  Length   = QueryHit->Header.le_Payload + 23;

	// Find start of QHD
	int ItemCount = 0;
	bool bNoZero = true;
	for(i = 42; i < Length - 16; i++)
	{
		if(Packet[i] == 0)
			if (bNoZero)
				bNoZero = false;
			else
			{
				ItemCount++;
				bNoZero = true;
				if(ItemCount != QueryHit->TotalHits)
					i += 8;
				else
					break;
			}
	}

	packet_QueryHitEx* QHD = (packet_QueryHitEx*) (Packet+i+1);
	CString Vendor( (char*) QHD->VendorID, 4);
	if( ValidVendor(Vendor) )
	{
		ExtendedPacket = true;

		if(QHD->Length == 1)
			if(QHD->Push == 1)
				Firewall = true;

		if(QHD->Length == 2)
		{
			if(QHD->FlagPush)
				Firewall = QHD->Push;

			if(QHD->FlagBusy)
				Busy = QHD->Busy;

			if(QHD->FlagStable)
				Stable = QHD->Stable;

			if(QHD->FlagSpeed)
				ActualSpeed = QHD->Speed;
		}
	}

	// Extract results from the packet
	while(HitsLeft > 0 && NextPos < Length - 16)
	{
		Result Item;
		packet_QueryHitItem* pQHI = (packet_QueryHitItem*) (Packet+NextPos);

		Item.Size = pQHI->le_Size;
		Item.Port = QueryHit->le_Port;
		if ( (m_pPrefs->m_nMinFileKSize<=0 || Item.Size >= m_pPrefs->m_nMinFileKSize*1024) &&
			 (m_pPrefs->m_nMaxFileKSize<=0 || Item.Size <= m_pPrefs->m_nMaxFileKSize*1024) &&
			 Item.Port != 99 ) // spam protection
		{
			Item.FileIndex = pQHI->le_Index;
			Item.Host      = QueryHit->Host;
			Item.Speed	   = QueryHit->le_Speed;

			Item.Firewall	 = Firewall;
			Item.Busy		 = Busy;
			Item.Stable		 = Stable;
			Item.ActualSpeed = ActualSpeed;
			Item.Vendor = GetVendor(Vendor);

			Item.OriginID = pOrigin->GetID();
			memcpy(&Item.PushID, Packet+(Length - 16), 16);
			Item.Distance = QueryHit->Header.Hops;

			// Get Filename
			for(i = NextPos + 8; Packet[i] != '\0' && i < Length - 16; ++i);

			Item.Name = CString(((char*)Packet)+NextPos+8, 0, i-NextPos-8);
			for (int idx = 0; Packet[i+1] != '\0' && i<Length-16; idx++)
			{
				// we have extra info at the hit tail
				if (Packet[i+1] == 0xC3)
				{
					// GGEP extension...
					int j = i;
					for( ; Packet[j+1] != '\0' && j < Length - 16; ++j);
					Item.Extra.push_back(GGEP_decode(&Packet[i+2], j-i-1));
					i = j;
					// try to get Sha1 out of Extra
					int nShaPos = Item.Extra[idx].find("ggep:sha1:");
					if (nShaPos >= 0)
					{
						Item.Sha1.fromStr(Item.Extra[idx].substr(nShaPos+10, 32));
						Item.Extra[idx] = StripWhite(Item.Extra[idx].substr(0, nShaPos) + Item.Extra[idx].substr(nShaPos+43));
					}
				}
				else
				{
					// HUGE or plain text extension...
					int j = i;
					bool NonPrintable = false;
					for( ; Packet[j+1] != 0x1C && Packet[j+1] != '\0' && j < Length - 16; ++j)
					{
						if (!isprint(Packet[j+1]))
							NonPrintable = true;
					}
					Item.Extra.push_back(CString(((char*)Packet)+i+1, 0, j-i));
					i = j;
					// try to get Sha1 out of Extra
					int nShaPos = Item.Extra[idx].find("urn:sha1:");
					if (nShaPos >= 0)
					{
						Item.Sha1.fromStr(Item.Extra[idx].substr(nShaPos+9, 32));
						Item.Extra[idx] = StripWhite(Item.Extra[idx].substr(0, nShaPos) + Item.Extra[idx].substr(nShaPos+42));
					}
					if (NonPrintable)
						Item.Extra[idx] = CString("data contains non-printable characters");
				}
				if (Packet[i+1] == 0x1C)
					i++;
			}
			/*// TODO: get rid of this rubbush
			for(i = NextPos + 8; !(Packet[i] == '\0' && Packet[i + 1] == '\0'); i++)
				if(i < Length - 16)
					Item.Name += (char) Packet[i];
				else
					break;*/

			// now check if this result matches some of the queries
			//cout << "query hit: " << Item.NameLower << endl;
			m_listmutex.lock();
			map<DWORD, MGnuSearch*>::iterator itS;
			for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); ++itS)
				itS->second->CheckAgainstResult(Item);
			m_listmutex.unlock();
		}
		else
		{
			// skip the result
			for(i = NextPos + 8; Packet[i] != '\0' && i < Length - 16; ++i);
			for( ; Packet[i+1] != '\0' && i < Length - 16; ++i);
		}

		// Check for end of reply packet
		if(i + 2 >= Length - 16)
			HitsLeft = 0;
		else
		{
			HitsLeft--;
			NextPos = i + 2;
		}
	}
}

void MGnuDirector::OnTimer(int nTimeStamp) // called once per second
{
	//POST_DMESSAGE("MGnuDirector::OnTimer()");
	//double BytesReceived = 0,  BytesSent = 0;
	int    nNumUploads   = CountUploads();

	if (nTimeStamp%60 == 0)
	{
		m_TableQuery.ClearUpTable(180);  // 3 min is enough to route packets
		m_TablePing.ClearUpTable(180);   // 3 min is enough to route packets
		m_TablePush.ClearUpTable(1200);  // these tables doesnt seem to overflow
		m_TableLocal.ClearUpTable(600);  //
		m_TableRouting.ClearUpTable(1200);
	}

	// download queuing: expire old busy hosts
	for (map <IP, int>::iterator itBH = m_mapBusyHosts.begin(); itBH != m_mapBusyHosts.end(); )
	{
		if (itBH->second <= nTimeStamp)
		{
			// remove element
			map <IP, int>::iterator itRemove = itBH;
			++itBH;
			m_mapBusyHosts.erase(itRemove);
		}
		else
			++itBH;
	}
	// limit m_pPrefs->m_nRetryWait from below
	if (m_pPrefs->m_nRetryWait < 3)
		m_pPrefs->m_nRetryWait = 3;

	// Null socks - used to interogate incomming connects
	// nobody has access to this list, it's not necesarry to use mutex
	int i;
	for(i = 0; i<m_SockList.size(); i++)
	{
		MGnuSock *pSock = m_SockList[i];
    	if(pSock->m_nSecsAlive > 30 || pSock->m_bDestroy)
		{
			delete pSock;
			m_SockList.erase(m_SockList.begin()+i);
			i--;
		}
		else
			pSock->OnTimer();
	}

	// GnuWebCache connections
	for(i = 0; i<m_WebCacheList.size(); i++)
	{
		MGnuWebCache *pWC = m_WebCacheList[i];
		if (!pWC->IsValid())
		{
			delete pWC;
			m_WebCacheList.erase(m_WebCacheList.begin()+i);
			i--;
		}
		else
			pWC->OnTimer(nTimeStamp);
	}

	// Conection bandwidth
	m_dwConnRecvRate *= 15;
	m_dwConnRecvRate += m_dwConnBytesRecv;
	m_dwConnRecvRate >>= 4;
	m_dwConnBytesRecv = 0;
	m_dwConnSendRate *= 15;
	m_dwConnSendRate += m_dwConnBytesSend;
	m_dwConnSendRate >>= 4;
	m_dwConnBytesSend = 0;

	// recalculate receive limit per connection
	float BandwidthLimit = -1;
	if(m_pPrefs->m_dBandwidthTotal >= 0 && m_pPrefs->m_dBandwidthConnects >= 0)
		BandwidthLimit = min(m_pPrefs->m_dBandwidthTotal,m_pPrefs->m_dBandwidthConnects);
	else if(m_pPrefs->m_dBandwidthTotal >= 0)
		BandwidthLimit = m_pPrefs->m_dBandwidthTotal;
	else if(m_pPrefs->m_dBandwidthConnects >= 0)
		BandwidthLimit = m_pPrefs->m_dBandwidthConnects;
	if (BandwidthLimit > 0)
	{
        BandwidthLimit *= 1024; // put it into bytes
		if (m_dwConnRecvRate+m_dwConnSendRate > BandwidthLimit)
		{
			m_dCurrentReceiveLimit -= (m_dwConnRecvRate + m_dwConnSendRate - BandwidthLimit)/1.5; // '/2' would be more exact,
			                                                                                      // but we are limiting the bandwidth
			                                                                                      // so it's better to overdo...
			if (m_dCurrentReceiveLimit < 0.1*BandwidthLimit)
				m_dCurrentReceiveLimit = 0.1*BandwidthLimit; // dont let it drop to 0
		}
		else
		{
			// limit is possibly too tough we shall make it more relaxed
			m_dCurrentReceiveLimit += (BandwidthLimit - m_dwConnRecvRate - m_dwConnSendRate) / 3; // '/2' whould be more exact,
			                                                                                      // but I'd like it not to make
			                                                                                      // sudden moves
			if (m_dCurrentReceiveLimit > 2*BandwidthLimit)
				m_dCurrentReceiveLimit = 2*BandwidthLimit; // dont let it grow really far
		}
		//cout << "m_dCurrentReceiveLimit = " << m_dCurrentReceiveLimit << endl;
	}
	else
		m_dCurrentReceiveLimit = 0; // means 'no limit'

    //
	ManageConnects(nTimeStamp);
	//cout << "connection state: connections " << nNumConnected << "  bandwidth " << BytesReceived + BytesSent << endl;

	long BytesPerUpload = -1;
	if (nNumUploads)
	{
		if(m_pPrefs->m_dBandwidthTotal >= 0 && nNumUploads)
		{
			BytesPerUpload = (int) ((m_pPrefs->m_dBandwidthTotal * 1024 - (m_dwConnRecvRate+m_dwConnSendRate)) / nNumUploads);

			if(BytesPerUpload < 0)
				BytesPerUpload = 0;
		}
		if ( m_pPrefs->m_dBandwidthTransfer >=0 &&
		     ( BytesPerUpload < 0 ||
		       BytesPerUpload > (m_pPrefs->m_dBandwidthTransfer * 1024)/nNumUploads) )
		{
			BytesPerUpload = long((m_pPrefs->m_dBandwidthTransfer * 1024)/nNumUploads);
		}
	}

	m_listmutex.lock();
	// Uploads
	vector<MGnuUpload*>::iterator itUpload;
	m_dUploadRate = 0;
	for(i = 0; i < m_UploadList.size(); i++)
	{
		MGnuUpload *pUp = m_UploadList[i];

		if(pUp->m_nStatus == TRANSFER_CLOSED)
		{
			delete pUp;
			m_UploadList.erase(m_UploadList.begin()+i);
			i--;
		}
		else
		{
			pUp->BandwidthTimer();

			if(TRANSFER_SENDING == pUp->m_nStatus)
			{
				m_dUploadRate += pUp->m_dRate;
			}
			pUp->SetMaxRate(BytesPerUpload);
		}
	}

	// Downloads
	m_dDownloadRate = 0;
	int nReceivingDownloadSocks = 0;
	int nRC;
	for(i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload *pDown = m_DownloadList[i];
		pDown->OnTimer(nTimeStamp);
		nRC = pDown->CountReceivingConnections(NULL);
		if(nRC)
		{
			m_dDownloadRate += pDown->GetRate();
			nReceivingDownloadSocks += nRC;
		}
	}
	// recalculate download rate limits
	if (m_pPrefs->m_dBandwidthDownl > 0)
	{
		if (nReceivingDownloadSocks)
		{
			register double dBand = m_pPrefs->m_dBandwidthDownl * 1024.0;
			if (m_dDownloadRate > dBand)
			{
				m_dDownloadRateLimit -= (m_dDownloadRate - dBand)*0.7; // 0.9 is here to make it all more smooth
				if (m_dDownloadRateLimit < dBand * 0.1)
					m_dDownloadRateLimit = dBand * 0.1; // dont let it drop to 0
			}
			else
			{
				// limit is possibly too high we shall make it more relaxed
				m_dDownloadRateLimit += (dBand - m_dDownloadRateLimit) * 0.5; // 0.5 reflects our intention not to make sudden moves
				if (m_dDownloadRateLimit > 3.0*dBand)
					m_dDownloadRateLimit = 3.0*dBand; // dont let it grow really high
			}
			//cout << "m_dDownloadRateLimit = " << m_dDownloadRateLimit << endl;
		}
		else
			m_dDownloadRateLimit =  m_pPrefs->m_dBandwidthDownl * 1024.0;
	}
	else
		m_dDownloadRateLimit = 0; // means 'no limit'
	// update downloads rate limits
	for(i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload *pDown = m_DownloadList[i];
		pDown->SetRatePerSocket((int) (m_dDownloadRateLimit / (nReceivingDownloadSocks>=1 ? nReceivingDownloadSocks : 1)));
	}
	
	// track total bandwidth
	int BitSpeed = (int) ((m_dwConnRecvRate+m_dwConnSendRate+m_dDownloadRate+m_dUploadRate) / 1024 * 8);
	if(BitSpeed > m_pPrefs->m_dwSpeedDyn)
		m_pPrefs->m_dwSpeedDyn = BitSpeed;

	// renew searches
	if (m_pPrefs->m_nResubmitSearches > 0 && m_SearchMap.size())
	{
		m_nReSearchTimer++;
		if ( m_nReSearchTimer >= (2 /*let the min time be 2 sec*/ + m_pPrefs->m_nResubmitSearches * 60 / m_SearchMap.size()) )
		{
			m_nReSearchTimer = 0;
			map<DWORD, MGnuSearch*>::iterator itS = m_SearchMap.upper_bound(m_dwReSearchID);
			//
			if (itS == m_SearchMap.end())
				itS = m_SearchMap.begin();
			m_dwReSearchID = itS->first;
			if (!itS->second->IsFull()) //food protection for searches like ".mp3"
				m_PendingQueries.push_back(m_dwReSearchID);
			// sanity check against "non-cooperative users"
			if (m_pPrefs->m_nResubmitSearches < 5)
				m_pPrefs->m_nResubmitSearches = 30;
		}
	}

    // clean up downloads list
	for( i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload* pDown = m_DownloadList[i];
		if (pDown->GetStatus()==TRANSFER_CLOSED)
		{
			// add auto-getting search if appropriate
			if (pDown->DisconnReason() == REASON_TRANSFER_FAILED)
			{
				POST_EVENT( MStringEvent(
					ET_MESSAGE,
					ES_JUSTINCASE,
					CString("`") + pDown->GetFileName() + "'\n" + DWrdtoStr(pDown->GetSizeReceived()) + " bytes received in this effort",
					MSG_DOWNLOAD_TO_AUTOGET,
					MSGSRC_DIRECTOR
				));
				pDown->AddSearch(true);
			}
			else if (pDown->ShouldDeleteSearch() && pDown->GetSearchID())
			{
				// remove "autoget" search
				// m_listmutex is locked here
				map<DWORD, MGnuSearch*>::iterator itS = m_SearchMap.find(pDown->GetSearchID());
				if (itS != m_SearchMap.end())
				{
					delete itS->second;
					m_SearchMap.erase(itS);
				}
			}
			//
			m_DownloadList.erase(m_DownloadList.begin()+i);
			i--;
			delete pDown;
		}
	}

	// auto-get support
	map<DWORD, MGnuSearch*>::iterator itS;
	for (itS = m_SearchMap.begin(); itS != m_SearchMap.end(); ++itS)
	{
		MGnuSearch* pS = itS->second;
		if (pS->IsAutoget() && pS->IsUpdated() && pS->GetResultsCount())
		{
			POST_EVENT( MStringEvent(
				ET_MESSAGE,
				ES_JUSTINCASE,
				pS->GetFilename().size() ? ("`" + pS->GetFilename() + "'") : ("for user autoget search `" + pS->GetSearchString() + "'"),
				MSG_AUTOGET,
				MSGSRC_DIRECTOR
			));
			//
			pS->SetUpdated(false);
			bool bAlready = false;
			// build up the vector of results
			ResultVec rv;
			pS->GetResults(rv);
			for (vector<Result>::iterator itr = rv.begin(); itr != rv.end(); ++itr )
			{
				// check if we already downloading this
				if ( LookUpDownload(itr->Host, itr->Name, itr->FileIndex) )
				{
					bAlready = true;
					break;
				}
			}
			if (!bAlready)
			{
				MGnuDownload* pD = new MGnuDownload(this, rv, pS->GetFilename(), pS->GetFileHandle());
				ASSERT(pD);
				if (pS->GetSizeFilterMode() == LIMIT_EXACTLY)
					pD->SetSearchID(pS->GetID());
				// add to download queue
				AddAndStartDownload(pD);
			}
			//
			pS->SetAutoget(false);
		}
	}

	// Gnutella Web Cache
	if (!m_WebCachesUrlQueue.empty())
	{
		if (nTimeStamp % 900 == 0) // once in 15 minutes
		{
			// update hosts list
			// TODO: get a random URL
			CString sURL = m_WebCachesUrlQueue.front();
			// it will come back
			RemoveWebCacheUrl(sURL);
			// Add client connection
			OpenWebCacheConnection(sURL, MGnuWebCache::HostFile, false);
		}
		else if (nTimeStamp % 900 == 450) // once in 15 minutes
		{
			// update URL list
			// TODO: get a random URL
			CString sURL = m_WebCachesUrlQueue.front();
			// it will come back
			RemoveWebCacheUrl(sURL);
			// Add client connection
			OpenWebCacheConnection(sURL, MGnuWebCache::UrlFile, false);
		}
		else if ( IsUltraPeer() && nTimeStamp % 1800 == 1799 ) // once in 30 minutes
		{
			// m_pPrefs->m_nMaxConnects < 0 || GetLastConnCount() <= m_pPrefs->m_nMaxConnects * 0.8
			// inform the world about us
			// TODO: get a random URL
			CString sURL = m_WebCachesUrlQueue.front();
			// it will come back
			RemoveWebCacheUrl(sURL);
			// Add client connection
			OpenWebCacheConnection(sURL, MGnuWebCache::Update, false);
		}
	}
	else
	{
		if (nTimeStamp % 30 == 0) // maximum once in 30 secs
		{
			// check if our initial servers have not been updated
			if (m_asInitialWebCaches[0] != m_pPrefs->m_szGWebCache1)
			{
				m_asInitialWebCaches[0] = m_pPrefs->m_szGWebCache1;
				m_anInitialWebCacheRequests[0] = 0;
				m_bNoValidWebCacheUrlsMessageSent = false; // we reset the flag if user changes something
			}
			if (m_asInitialWebCaches[1] != m_pPrefs->m_szGWebCache2)
			{
				m_asInitialWebCaches[1] = m_pPrefs->m_szGWebCache2;
				m_anInitialWebCacheRequests[1] = 0;
				m_bNoValidWebCacheUrlsMessageSent = false; // we reset the flag if user changes something
			}
			if (m_asInitialWebCaches[2] != m_pPrefs->m_szGWebCache3)
			{
				m_asInitialWebCaches[2] = m_pPrefs->m_szGWebCache3;
				m_anInitialWebCacheRequests[2] = 0;
				m_bNoValidWebCacheUrlsMessageSent = false; // we reset the flag if user changes something
			}
			// check if our initial servers are valid
			bool bHaveValidUrls = false;
			// request urls from the initial connect servers
			for (int i=0; i<3; ++i)
				if (m_anInitialWebCacheRequests[i] < 5 && m_asInitialWebCaches[i].length())
				{
					OpenWebCacheConnection( m_asInitialWebCaches[i], MGnuWebCache::UrlFile, false);
					++m_anInitialWebCacheRequests[i];
					bHaveValidUrls = true;
				}
			// inform user if we have no valid urls	
			if (!bHaveValidUrls)
			{
				if (!m_bNoValidWebCacheUrlsMessageSent)
				{
					POST_ERROR(
						ES_IMPORTANT,
						"All initial GWebCaches appear to be invalid or unreacheable. Supply working Gnutella web cache URLs or verify your internet connection. Currently it is imposible to initiate connection to the Gnutella network."
					);
					m_bNoValidWebCacheUrlsMessageSent = true;
				}
			}
			else
				m_bNoValidWebCacheUrlsMessageSent = false; // we reset the flag if there were valid URLs
		}
	}

	// send pending queries
	while (!m_PendingQueries.empty())
	{
		itS = m_SearchMap.find(m_PendingQueries.front());
		if (itS != m_SearchMap.end())
			itS->second->SendQuery();
		m_PendingQueries.pop_front();
	}
	// carefull, potential dead-lock
	m_hitqueuemutex.lock();
	while (!m_hitQueue.empty())
	{
		QueuedHit qh = m_hitQueue.front();
		m_hitQueue.pop();
		map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(qh.dwNodeID);
		if (itNode != m_NodeMap.end())
		{
			ASSERT(itNode->second->GetID() == qh.dwNodeID);
			itNode->second->Send(qh.pPacket, qh.size); // TODO: get rig of data copiing and memory re-allocation
		}
		delete [] qh.pPacket;
	}
	m_hitqueuemutex.unlock();
	m_listmutex.unlock();
	// update statictics cache
	m_nLastConnCount = CountConnections();

	// supernode options support
	if (!m_bForcedSupernode && m_bLastEnableSuperNode != m_pPrefs->m_bEnableSuperNodeMode)
	{
		m_bLastEnableSuperNode = m_pPrefs->m_bEnableSuperNodeMode;
		// either upgrade or downgrade client
		if (m_bLastEnableSuperNode)
			EnableSupernodePromotion();
		else
			DisableSupernodePromotion();
	}

	if (m_bForcedSupernode != m_pPrefs->m_bForceSuperNodeMode)
	{
		m_bForcedSupernode = m_pPrefs->m_bForceSuperNodeMode;
		if (m_bForcedSupernode)
			EnableSupernodePromotion(true);
		else
			if (m_bLastEnableSuperNode)
				EnableSupernodePromotion();
			else
				DisableSupernodePromotion();
	}

	// once per hour check if we can upgrade to superpeer
	if (!m_bForcedSupernode && nTimeStamp % 3600 == 3000) // first check will occur 50 minutes after start
	{
		if (m_bLastEnableSuperNode)
			EnableSupernodePromotion();
	}

	// support for DynDNS Firewall
	if (strlen(m_pPrefs->m_szDynDnsFirewall))
	{
		if ( m_pAsyncDNS == NULL)
		{
			if (nTimeStamp % 600 == 60) // check each 10 min, first check one minute from start
				m_pAsyncDNS = new MAsyncDns(m_pPrefs->m_szDynDnsFirewall);
		}
		else
		{
			if (m_pAsyncDNS->IsFinished())
			{
				if (m_pAsyncDNS->IsSuccess())
					m_pPrefs->m_ipForcedHost.S_addr = m_pAsyncDNS->GetHostS_addr();
				m_pAsyncDNS->Release();
				m_pAsyncDNS = NULL;
			}
		}
	}

	// cache LeafMode flag
	m_bLastIsShieldedLeaf = IsShieldedLeaf();

	// broadcast TIMER message to the world
	ED().PostEvent(new MIntEvent(ET_MESSAGE, ES_MINIMAL, nTimeStamp, MSG_TIMER));
}

bool MGnuDirector::AllowMakingConnections()
{
	return !m_pPrefs->m_bQuietMode;
}

void MGnuDirector::ManageConnects(int nTimeStamp)
{
	// Clean out dead nodes
	{
		std::queue<DWORD> queueDeadIDs;
		MLock lock (m_listmutex);
		map<DWORD, MGnuNode*>::iterator itNode;
		for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		{
			ASSERT(itNode->second);
			ASSERT(itNode->second->GetID() == itNode->first);
			if(itNode->second->GetStatus() == SOCK_CLOSED)
				queueDeadIDs.push(itNode->first);
		}
		// TODO: move this to the very end... to collect all the deads
		while (queueDeadIDs.size())
		{
			itNode = m_NodeMap.find(queueDeadIDs.front());
			if (itNode != m_NodeMap.end())
			{
				delete itNode->second;
				m_NodeMap.erase(itNode);
			}
			queueDeadIDs.pop();
		}
	}
	// call do some accounting and then call OnTimer
	int nNormalConnects = 0;
	int nLeafConnects = 0;
	int nUltraConnects = 0;
	int nNodeMapSize = 0;
	{
		MLock lock (m_listmutex);
		nNodeMapSize = m_NodeMap.size();
		map<DWORD, MGnuNode*>::iterator itNode;
		for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		{
			ASSERT(itNode->second);
			if(itNode->second->GetStatus() == SOCK_CONNECTED)
			{
				if( m_nHostMode == CM_ULTRAPEER )
				{
					if(itNode->second->GetPeerMode() == CM_LEAF)
						nLeafConnects++;
					else
					{
						nNormalConnects++;
						if(itNode->second->GetPeerMode() == CM_ULTRAPEER)
							nUltraConnects++;
					}
				}
				else
					nNormalConnects++;
			}
		}
		m_nLastNormalConnects = nNormalConnects;

		for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		{
			itNode->second->SetMaxInRate( (DWORD) (m_dCurrentReceiveLimit / (nNormalConnects + nLeafConnects)) );
			itNode->second->OnTimer();
		}
	}
	// I dont really understand so far what is m_nExtPongBytes
	m_nExtPongBytes = 0;

	if(IsLeaf())
	{
		if(nNormalConnects > m_pPrefs->m_nLeafModeConnects)
			DropSomeNode(nTimeStamp);
	}
	else
	{
		if(m_pPrefs->m_nMaxConnects > 0 && nNormalConnects > m_pPrefs->m_nMaxConnects)
			DropSomeNode(nTimeStamp);
		if(nLeafConnects > m_pPrefs->m_nMaxLeaves)
			DropSomeLeaf(nTimeStamp);
	}


	/*if(m_pPrefs->m_nMaxConnects > 0 && nCurrent >= m_pPrefs->m_nMaxConnects)
	{
		RemoveNode_MaxDrop(nTimeStamp);
	}*/
	//if(nTimeStamp%180 == 150)
	//	RemoveNode_MaxDrop(nTimeStamp);
	if(nTimeStamp%60 == 0)
		RemoveNode_LeastReplyRate(nTimeStamp);
	//
	if(AllowMakingConnections())
	{
		if(IsLeaf())
		{
			if(nNormalConnects < m_pPrefs->m_nLeafModeConnects && nNodeMapSize < m_pPrefs->m_nLeafModeConnects * 2)
			{
				AddConnect(true);
				// MZ: add more connects at once -- we dont seem to find
				//     proper ultrapeers quickly
				AddConnect(true, true);
			}
		}
		else
		{
			if( m_pPrefs->m_nMinConnects                   &&
				nNormalConnects < m_pPrefs->m_nMinConnects &&
				(m_pPrefs->m_nMaxConnects<0 || nNormalConnects < m_pPrefs->m_nMaxConnects) )
					AddConnect(false);
			// Keep a 2/3rds connect to ultrapeers
			if(m_nHostMode == CM_ULTRAPEER && nNormalConnects)
				if(nUltraConnects * 100 / nNormalConnects < 66)
					AddConnect(true);
		}
	}
}

// Node management functions
void MGnuDirector::AddConnect(bool bSuperpeer, bool bUseStore /*=false*/)
{
	if (bUseStore)
		bUseStore = !m_pHostStore->IsHalfEmpty();
	//
	Node node;
	if ( !bUseStore && (bSuperpeer && m_pUltraCatcher->GetRandomNode(node)) ||
		 !bUseStore && m_pHostCatcher->GetRandomNode(node)                  ||
		               m_pHostStore->GetRandomNode(node)                    )
	{
		if (IsOkForDirectConnect(node.Host) && NotLocal(node))
			AddNode( node.Host, node.Port );
	}
	else
	{
		// Gnutella Web Cache
		if (!m_WebCachesUrlQueue.empty())
		{
			// TODO: get a random URL
			CString sURL = m_WebCachesUrlQueue.front();
			// it will come back if the connection is successfull
			RemoveWebCacheUrl(sURL);
			// Add client connection
			OpenWebCacheConnection(sURL, MGnuWebCache::HostFile, false);
		}
	}
}

void MGnuDirector::AddWebCacheUrl(const CString& sURL, bool bChecked /*=false*/)
{
	if (sURL.empty())
		return;
	// we maintain both set and queue to make searching easier
	if (m_WebCachesUrlSet.find(sURL) == m_WebCachesUrlSet.end())
	{
		m_WebCachesUrlSet.insert(sURL);
	}
	else
	{
		// remove it from the queue
		std::deque<CString>::iterator itURL = find(m_WebCachesUrlQueue.begin(), m_WebCachesUrlQueue.end(), sURL);
		if (itURL == m_WebCachesUrlQueue.end())
		{
			// this should never happen!
			m_WebCachesUrlSet.erase(sURL);
			return;
		}
		m_WebCachesUrlQueue.erase(itURL);
	}
	m_WebCachesUrlQueue.push_back(sURL);
	// remove the oldest URLs if we have too much
	while (m_WebCachesUrlQueue.size()>50)
	{
		m_WebCachesUrlSet.erase(m_WebCachesUrlQueue.front());
		m_WebCachesUrlQueue.pop_front();
	}
	// keep track of the last good URL
	if (bChecked)
		m_sLastGoodWebCacheUrl = sURL;
}

void MGnuDirector::OnWebCacheReply(const CString& sURL)
{
	AddWebCacheUrl(sURL, true);
	if (sURL == m_pPrefs->m_szGWebCache1)
		m_anInitialWebCacheRequests[0] = 0;
	else if (sURL == m_pPrefs->m_szGWebCache2)
		m_anInitialWebCacheRequests[1] = 0;
	else if (sURL == m_pPrefs->m_szGWebCache3)
		m_anInitialWebCacheRequests[2] = 0;
}

bool MGnuDirector::SaveWebCaches(const CString sPath)
{
	if( m_WebCachesUrlSet.empty() )
		return true; // all OK, we just have nothing to write
	FILE * pFile;
	if ( (pFile = fopen(ExpandPath(sPath).c_str(), "w")) )
	{
		fprintf(pFile, "# this is mutella's GWebCache list file\n");
		fprintf(pFile, "# it is created automatically at the end\n");
		fprintf(pFile, "# of mutella session and updated periodically\n");
		fprintf(pFile, "# during the session\n");
		fprintf(pFile, "# \n");
		for ( set<CString>::iterator it = m_WebCachesUrlSet.begin(); it != m_WebCachesUrlSet.end(); ++it)
			fprintf(pFile, "%s\n", it->c_str());
		fclose(pFile);
	}
	return NULL!=pFile;
}

bool MGnuDirector::LoadWebCaches(const CString sPath)
{
	FILE * pFile;
	if ( (pFile = fopen(ExpandPath(sPath).c_str(), "r")) )
	{
		CString sURL;
		char buf[1024];
		while ( !feof(pFile) )
		{
			sURL = fgets(buf,1024,pFile);
			if ( sURL.length()==0 || buf[0]=='#' )
				continue;
			AddWebCacheUrl(sURL);
		}
		fclose(pFile);
	}
	return NULL!=pFile;
}

void MGnuDirector::RemoveWebCacheUrl(const CString& sURL)
{
	m_WebCachesUrlSet.erase(sURL);
	std::deque<CString>::iterator itURL = find(m_WebCachesUrlQueue.begin(), m_WebCachesUrlQueue.end(), sURL);
	if (itURL != m_WebCachesUrlQueue.end())
		m_WebCachesUrlQueue.erase(itURL);
}

void MGnuDirector::OpenWebCacheConnection(const CString& sURL, int nRequestType, bool bLock /*=true*/)
{
	//cout << nRequestType << " request to " << sURL << "...";
	MGnuWebCache* pWC = new MGnuWebCache(this);
	if (pWC->Connect(sURL, (MGnuWebCache::RequestType) nRequestType))
	{
		MLock lock(m_listmutex, bLock);
		m_WebCacheList.push_back(pWC);
		//cout << " succeeded!\n";
	}
	else
	{
		delete pWC;
		//cout << " failed!\n";
	}
}


void MGnuDirector::DropSomeNode(int nTimeStamp)
{
	MGnuNode* pDeadNode           = NULL;
	MGnuNode* pDeadUltrapeer      = NULL;
	DWORD dwLowestRating          = 100;
	DWORD dwLowestRatingUltrapeer = 100;

	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode, itRemove;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		ASSERT(itNode->second);
		if( itNode->second->GetStatus() == SOCK_CONNECTED )
		{
			if( itNode->second->GetPeerMode() == CM_NORMAL        &&
				itNode->second->GetEfficeincy() <= dwLowestRating )
			{
				pDeadNode      = itNode->second;
				dwLowestRating = itNode->second->GetEfficeincy();
			}
			else if(itNode->second->GetPeerMode() == CM_ULTRAPEER              &&
					itNode->second->GetEfficeincy() <= dwLowestRatingUltrapeer )
			{
				pDeadUltrapeer          = itNode->second;
				dwLowestRatingUltrapeer = itNode->second->GetEfficeincy();
			}
		}
	}
	// we prefer disconnecting normal nodes
	if(pDeadNode)
		pDeadNode->ForceDisconnect();
	else if(pDeadUltrapeer)
		pDeadUltrapeer->ForceDisconnect();
}

void MGnuDirector::DropSomeLeaf(int nTimeStamp)
{
	MGnuNode* pDeadNode = NULL;
	DWORD dwLowestRating = 100;

	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode, itRemove;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		ASSERT(itNode->second);
		if( itNode->second->IsConnected()                     &&
			itNode->second->GetPeerMode() == CM_LEAF          &&
			itNode->second->GetEfficeincy() <= dwLowestRating )
		{
			pDeadNode      = itNode->second;
			dwLowestRating = itNode->second->GetEfficeincy();
		}
	}

	if(pDeadNode)
		pDeadNode->ForceDisconnect();
}

//
void MGnuDirector::RemoveNode_MaxDrop(int nTimeStamp)
{
	u_long nMaxDrop  = 0;
	DWORD dwRemoveID = 0;
	MLock lock(m_listmutex);
	if(!m_NodeMap.size())
		return;

	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		if(itNode->second->GetSGnuNode().m_nAvgDroppedPackets > nMaxDrop)
		{
			dwRemoveID = itNode->second->GetID();
			nMaxDrop = itNode->second->GetSGnuNode().m_nAvgDroppedPackets;
		}
	}

	if (nMaxDrop>0)
	{
		lock.unlock();
		RemoveNode(dwRemoveID);
	}
}

bool sortNodesByReply(const SGnuNode* n1, const SGnuNode* n2)
{
	// sort accending by reply rate than decending by time
	if (n1->m_dStatisticsRate[NR_QREPLY_IN]<n2->m_dStatisticsRate[NR_QREPLY_IN])
		return true;
	if (n1->m_dStatisticsRate[NR_QREPLY_IN]>n2->m_dStatisticsRate[NR_QREPLY_IN])
		return false;
	if (n1->m_nUptime>n2->m_nUptime)
		return true;
	// equivalent elements return false
	return false;
}

// we dont need connections which does not send us quiery-replies
void MGnuDirector::RemoveNode_LeastReplyRate(int nTimeStamp)
{
	MLock lock(m_listmutex);
	double LeastRate = -1;
	if(m_NodeMap.size()<3)
		return;
	// do it. search for median speed and remove the one which is much slower than that
	vector<MGnuNode*> nodes;
	nodes.reserve(m_NodeMap.size());
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		ASSERT(itNode->second);
		if( itNode->second->IsConnected()                                &&
			itNode->second->GetPeerMode() != CM_LEAF                     &&  // do not kill leaves, they are not supposed to be too good anyway
			(nTimeStamp - itNode->second->GetSGnuNode().m_nUptime) > 180 )   // be kind to new connections, give them 3 minutes
				nodes.push_back(itNode->second);
	}
	//
	if(nodes.size()<3)
		return;
	// sort by reply rate
	sort(nodes.begin(), nodes.end(), sortNodesByReply);
	double dMedianRate = nodes[nodes.size()/2]->GetSGnuNode().m_dStatisticsRate[NR_QREPLY_IN];

	// remove the slowes one if it is much slower than that median
	if ( nodes[0]->GetSGnuNode().m_dStatisticsRate[NR_QREPLY_IN] < 0.15*dMedianRate )
	{
		DWORD dwID = nodes[0]->GetID();
		lock.unlock();
		RemoveNode(dwID);
	}
}

// to remove the connection which floods us
/*void MGnuDirector::RemoveNode_MostRecv(int nTimeStamp)
{
	MLock lock(m_listmutex);
	double MostBytes = -1;
	if(!m_NodeMap.size())
		return;

	map<DWORD, MGnuNode*>::iterator itNode, itRemove;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		if(itNode->second->GetSGnuNode().m_dByteRate[0] > MostBytes)
		{
			itRemove  = itNode;
			MostBytes = itNode->second->GetSGnuNode().m_dByteRate[0];
		}
	}

	if (MostBytes > 0)
	{
		// this is kinda hack, but in principle it is right thing to do
		if (m_dwConnRecvRate > itRemove->second->GetSGnuNode().m_dByteRate[0])
			m_dwConnRecvRate -= (DWORD) itRemove->second->GetSGnuNode().m_dByteRate[0];
		if (m_dwConnSendRate > itRemove->second->GetSGnuNode().m_dByteRate[1])
			m_dwConnSendRate -= (DWORD) itRemove->second->GetSGnuNode().m_dByteRate[1];
		// remove
		lock.unlock();
		RemoveNode(itRemove->second->GetID());
	}
}*/

void MGnuDirector::GetNetStats(int& nHosts, int& nSharingHosts, int& nFiles, int& nSize)
{
	nHosts = 0;
	nSharingHosts = 0;
	nFiles = 0;
	nSize = 0;
	
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if(itNode->second->IsConnected())
		{
			nHosts++;
			nHosts        += itNode->second->GetSGnuNode().m_dwFriendsTotal;
			nSharingHosts += itNode->second->GetSGnuNode().m_dwSharingTotal;
			nSize         += itNode->second->GetSGnuNode().m_llLibraryTotal/1024;
			nFiles        += itNode->second->GetSGnuNode().m_dwFilesTotal;
		}
}

void MGnuDirector::GetCacheStats(int& nCatcherHosts, int& nUltraCatcherHosts, int& nStoreHosts, int& nWebCaches)
{
	MLock lock(m_listmutex);
	nWebCaches = m_WebCachesUrlSet.size();
	nCatcherHosts = m_pHostCatcher->GetSize();
	nUltraCatcherHosts = m_pUltraCatcher->GetSize();
	nStoreHosts = m_pHostStore->GetSize();
}

bool MGnuDirector::RegisterHttpFolder(MUI* pUI, const CString& sFolder)
{
	MLock lock(m_listmutex);
	if (!pUI || m_mapWebFolders.end() != m_mapWebFolders.find(sFolder))
		return false;
	m_mapWebFolders[sFolder] = pUI;
	return true;
}

bool MGnuDirector::UnregisterHttpFolder(MUI* pUI, const CString& sFolder)
{
	MLock lock(m_listmutex);
	map<CString, MUI*>::iterator it = m_mapWebFolders.find(sFolder);
	if (it == m_mapWebFolders.end() || it->second!= pUI)
		return false;
	m_mapWebFolders.erase(it);
	return true;
}

bool MGnuDirector::OnExternalHttpRequest(const IP& ipRemoteHost, LPCSTR szFolder, LPCSTR szHandshake, const char* pBuffRest, HANDLE hSocket)
{
	MLock lock(m_listmutex);
	int n;
	// keep removing sub-folders untill we find it
	for (CString s = szFolder; s.length();)
	{
		map<CString, MUI*>::iterator it = m_mapWebFolders.find(s);
		if (it != m_mapWebFolders.end())
		{
			// found it!
			MUI* pUI = it->second;
			lock.unlock();
			return m_pController->OnExternalHttpRequest(ipRemoteHost, szFolder, szHandshake, pBuffRest, hSocket, pUI);
		}
		// remove the last element of the path
		n = s.rfind('/');
		if (n<=0)
			break;
		// cuts the string at the given location without memory re-allocations
		s.cut(n);
	}
	// did not work out
	return false;
}

// returns 'true' if socket was attached
bool MGnuDirector::OnIncomingPush(const CString& sFileName, int nIndex, SOCKET hPushSock)
{
	MLock lock(m_listmutex);
	for( vector<MGnuDownload*>::iterator it = m_DownloadList.begin(); it != m_DownloadList.end(); ++it)
	{
		if ((*it)->OnIncomingPush(sFileName, nIndex, hPushSock))
			return true;
	}
	return false;
}

IP MGnuDirector::GetPublicIP()
{
	ASSERT(m_pPrefs);
	return m_pPrefs->m_ipForcedHost.S_addr ? m_pPrefs->m_ipForcedHost : m_pPrefs->m_ipLocalHost;
}

DWORD MGnuDirector::GetPublicPort()
{
	ASSERT(m_pPrefs);
	return m_pPrefs->m_dwForcedPort ? m_pPrefs->m_dwForcedPort : GetLocalPort() /*m_pPrefs->m_dwLocalPort*/;
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

DWORD MGnuDirector::GetLocalSharedFiles()
{
	ASSERT(m_pShare);
	return m_pShare->GetFileCount();
}
DWORD MGnuDirector::GetLocalSharedSize()
{
	ASSERT(m_pShare);
	return m_pShare->GetFileSize();
}
DWORD MGnuDirector::GetUltrapeerSizeMarker()
{
	ASSERT(m_pShare);
	return m_pShare->GetUltrapeerSizeMarker();
}

double MGnuDirector::FreeUltraCapacity(int nRemoteTotal)
{
	int nChildCount = CountUltrapeerLeaves();
	int nFreeSlots = m_pPrefs->m_nMaxLeaves - nChildCount;
	if(nFreeSlots < 0)
		nFreeSlots = 0;

	return ((double) nFreeSlots) / nRemoteTotal * 100.0;
}

double MGnuDirector::RunningUltraCapacity(int nRemoteTotal)
{
	int nChildCount = CountUltrapeerLeaves();
	return ((double)nChildCount) / nRemoteTotal * 100.0;
}

int MGnuDirector::CountUltrapeerLeaves()
{
	int nChildCount = 0;
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		ASSERT(itNode->second);
		if( itNode->second->IsConnected()            &&
			itNode->second->GetPeerMode() == CM_LEAF )
				nChildCount++;
	}
	return nChildCount;
}

int MGnuDirector::CountNormalConnects()
{
	int nNormalConnects = 0;
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		ASSERT(itNode->second);
		if( itNode->second->IsConnected() )
		{
			if(m_nHostMode == CM_ULTRAPEER)
			{
				if(itNode->second->GetPeerMode() != CM_LEAF)
					nNormalConnects++;
			}
			else
				nNormalConnects++;
		}
	}
	m_nLastNormalConnects = nNormalConnects;
	return nNormalConnects;
}

int MGnuDirector::GetNormalConnectsApprox()
{
	return m_nLastNormalConnects;
}

void MGnuDirector::CloseAllOtherNodes(MGnuNode* pExcept)
{
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if( itNode->second != pExcept )
			itNode->second->ForceDisconnect();
}

void MGnuDirector::Route_Pong(packet_Pong* pPong, int nLength, DWORD dwRouteID)
{
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(dwRouteID);
	if( itNode != m_NodeMap.end() &&
		itNode->second->GetStatus() == SOCK_CONNECTED )
		itNode->second->SendPacket(pPong, nLength, PACKET_PONG, false);
}

void MGnuDirector::Route_UltraPong(packet_Pong* pPong, int nLength, DWORD dwRouteID)
{
	if(m_nHostMode != CM_ULTRAPEER)
		return;
	// Send a max of 2KB in pongs a sec to children
	if(m_nUltraPongBytes > 2048)
		return;

	bool bHostsFound = false;
	MLock lock(m_listmutex);
	// Send pongs to children
	map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(dwRouteID);
	if(itNode != m_NodeMap.end())
	{
		MGnuNode* p = itNode->second;
		ASSERT(p);
		if( p->GetPeerMode() == CM_LEAF &&  // it could have crashed here
			p->IsConnected()            &&
			!p->m_UltraPongSent         )
		{
			p->SendPacket(pPong, nLength, PACKET_PONG, false);
			m_nUltraPongBytes += nLength;
			p->m_UltraPongSent = true;
			bHostsFound = true;
		}
	}
	// If all children sent ultra-pongs, reset list
	if(!bHostsFound)
		for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
			itNode->second->m_UltraPongSent = false;
}

void MGnuDirector::Route_Push(packet_Push* pPush, int nLength, DWORD dwRouteID)
{
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(dwRouteID);
	if( itNode != m_NodeMap.end() &&
		itNode->second->GetStatus() == SOCK_CONNECTED)
			itNode->second->SendPacket(pPush, nLength, PACKET_PUSH, false);
}

void MGnuDirector::Route_QueryHit(packet_QueryHit* pQueryHit, int nLength, DWORD dwRouteID)
{
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode = m_NodeMap.find(dwRouteID);
	if( itNode != m_NodeMap.end() &&
		itNode->second->GetStatus() == SOCK_CONNECTED)
			itNode->second->SendPacket(pQueryHit, nLength, PACKET_QUERYHIT, false);
}

void MGnuDirector::EnableSupernodePromotion(bool bForce /*=false*/ )
{
	// this function analyses the situation and changes the node capability if appropriate
	if (m_nHostMode == CM_ULTRAPEER || (!bForce && m_nHostMode == CM_ULTRALEAF))
		return;
	// Cant be behind firewall
	if (!bForce                                                                     &&
		(!m_bReceivedIncomingConnections                                            ||
		(m_pPrefs->m_dBandwidthConnects > 0 && m_pPrefs->m_dBandwidthConnects < 32) ||
		m_pPrefs->m_dwSpeedDyn < 256                                                ))
			return;

	if (bForce)
	{
		// upgrade straight to utrapeer mode
		m_nHostMode = CM_ULTRAPEER;

		// Disconenct any leaves from us in hope they will reconnect to our supernode
		MLock lock(m_listmutex);
		map<DWORD, MGnuNode*>::iterator itNode;
		for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		{
			ASSERT(itNode->second);
			if( itNode->second->IsConnected()            &&
				itNode->second->GetPeerMode() == CM_LEAF )
					itNode->second->ForceDisconnect();
		}
	}
	else
		// upgrade to utrapeer-capable leaf
		m_nHostMode = CM_ULTRALEAF;
}

void MGnuDirector::DisableSupernodePromotion()
{
	if (m_nHostMode == CM_LEAF)
		return;
	if(m_nHostMode == CM_ULTRALEAF)
	{
		m_nHostMode = CM_LEAF;
		return;
	}

	m_nHostMode = CM_LEAF;
	
	// disconnect all the leaves
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		ASSERT(itNode->second);
		if( itNode->second->IsConnected() && itNode->second->GetPeerMode() == CM_LEAF )
			itNode->second->ForceDisconnect();
	}
}

bool MGnuDirector::IsLeaf()
{
	return m_nHostMode == CM_LEAF || m_nHostMode == CM_ULTRALEAF;
}

bool MGnuDirector::IsUltraPeer()
{
	return m_nHostMode == CM_ULTRAPEER;
}

bool MGnuDirector::IsShieldedLeaf()
{
	if (!IsLeaf())
		return false;
	MLock lock();
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
	{
		ASSERT(itNode->second);
		if( itNode->second->IsConnected() && itNode->second->GetPeerMode() == CM_ULTRAPEER )
			return true;
	}
	return false;
}

void MGnuDirector::SetHostMode(int nMode)
{
	m_nHostMode = nMode;
	m_nHostModeChangeTime = xtime();
	m_nUltrapeerCredit = 1;
}

int  MGnuDirector::HostModeValidUntil()
{
	return m_nHostModeChangeTime + 1800; // only allow to change modes once in half an hour
}

void MGnuDirector::OnShareChanged()
{
	// Tell nodes to re-apply their patches to word hash table
	MLock lock(m_listmutex);
	map<DWORD, MGnuNode*>::iterator itNode;
	for(itNode = m_NodeMap.begin(); itNode != m_NodeMap.end(); itNode++)
		if( itNode->second->IsConnected() )
			itNode->second->m_PatchUpdateNeeded = true;
}

void MGnuDirector::OnDownloadHostBusy(const IP& ip, int nNextValidTime)
{
	m_mapBusyHosts[ip] = nNextValidTime;
}

bool MGnuDirector::IsDownloadHostBusy(const IP& ip)
{
	return m_mapBusyHosts.find(ip) != m_mapBusyHosts.end();
}

