/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 property.h  -  Property classes for holding preferences in.
                             -------------------
    begin                : Sat Jul 7 2001
    copyright            : (C) 2001 by 
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifndef PROPERTY_H
#define PROPERTY_H

#include "string.h"
#include "basicstruct.h"

enum propert_type{
	PT_NONE    = 0,
	PT_BOOL    = 1,
	PT_INT     = 2,
	PT_DWORD   = 3,
	PT_DOUBLE  = 4,
	PT_STRING  = 5,
	PT_IP      = 6,
	PT_IPNET   = 7,
	PT_SET     = 0x20000,
	PT_BYREF   = 0x10000
};

#define PNAME_SIZE 31
#define DEF_STR_SIZE 256

class MIniFile;
class MPropertySet;

struct Property {
	char name[PNAME_SIZE+1];
	int  type;
	int  size;
	union {
	    // direct value
		bool   bVal; // size == 1
		int    nVal; // size == 4
		DWORD  dwVal; // size == 4
		double dVal; // size == 8
		char * lpstrVal; // size == length of current buffer
		IP     ipVal;
		IPNET  ipnVal;
		// reference (pointer) to the value
		bool   * pbVal; // size == 4
		int    * pnVal; // size == 4
		DWORD  * pdwVal; // size == 4
		double * pdVal; // size == 4
		IP     * pipVal;
		IPNET  * pipnVal;
		MPropertySet *
		         pSet;
	};		
};

class MProperty : protected Property {
public:
	MProperty(LPCSTR szName){
		strncpy(name,szName,PNAME_SIZE);
		name[PNAME_SIZE] = '\0';
		type = PT_NONE;
		size = 0;
		m_bPersistent = true;
	}
	int GetPropType(){return type & 0xFFFF;}
	LPCSTR GetPropertyName(){return name;}
	bool   GetBoolValue(){ASSERT(type & PT_BOOL); return (type & PT_BYREF) ? *pbVal : bVal;}
	void   SetBoolValue(bool b){ASSERT(type & PT_BOOL); if (type & PT_BYREF) *pbVal = b; else bVal = b; }
	int    GetIntValue(){ASSERT(type & PT_INT); return (type & PT_BYREF) ? *pnVal : nVal;}
	void   SetIntValue(int n){ASSERT(type & PT_INT); if (type & PT_BYREF) *pnVal = n; else nVal = n; }
	DWORD  GetDwordValue(){ASSERT(type & PT_DWORD); return (type & PT_BYREF) ? *pdwVal : dwVal;}
	void   SetDwordValue(DWORD dw){ASSERT(type & PT_DWORD); if (type & PT_BYREF) *pdwVal = dw; else dwVal = dw; }
	double GetDoubleValue(){ASSERT(type & PT_DOUBLE); return (type & PT_BYREF) ? *pdVal : dVal;}
	void   SetDoubleValue(double d){ASSERT(type & PT_DOUBLE); if (type & PT_BYREF) *pdVal = d; else dVal = d; }
	IP     GetIpValue(){ASSERT(type & PT_IP); return (type & PT_BYREF) ? *pipVal : ipVal;}
	void   SetIpValue(const IP& ip){ASSERT(type & PT_IP); if (type & PT_BYREF) *pipVal = ip; else ipVal = ip; }
	IPNET  GetIpnetValue(){ASSERT(type & PT_IPNET); return (type & PT_BYREF) ? *pipnVal : ipnVal;}
	void   SetIpnetValue(const IPNET& ipn){ASSERT(type & PT_IPNET); if (type & PT_BYREF) *pipnVal = ipn; else ipnVal = ipn; }
	LPCSTR GetStringValue(){ASSERT(type & PT_STRING); return lpstrVal;}
	bool   SetStringValue(LPCSTR str){
		ASSERT(type & PT_STRING);
		// to make it thread safe we never re-allocate the mem
		strncpy(lpstrVal, str, size-1);// the string can be turnicated
		lpstrVal[size-1] = '\0';
		return strlen(str)<size;		
	}
	bool IsSet(){return type & PT_SET;}
	MPropertySet* GetSet(){ASSERT(type & PT_SET); return pSet;}
	
	// save/load to ini file
	bool Read(MIniFile* pIniFile);
	bool Write(MIniFile* pIniFile);
	// useful parser
	virtual bool SetFromStr(char * buf);
	// output as string
	virtual CString Format();
	// mode
	bool IsPersistent(){return m_bPersistent;}
	void SetPersistance(bool bSet){m_bPersistent = bSet;}
	
	virtual ~MProperty(){};
protected:
	// overloaded access functions -- required by templates;
	inline void setval(bool   b){bVal = b;}
	inline void setval(int    n){nVal = n;}
	inline void setval(DWORD dw){dwVal = dw;}
	inline void setval(double d){dVal = d;}
	inline void setval(const IP& ip){ipVal = ip;}
	inline void setval(const IPNET& ipn){ipnVal = ipn;}
	inline void setptr(bool   * pb){pbVal = pb;}
	inline void setptr(int    * pn){pnVal = pn;}
	inline void setptr(DWORD  * pdw){pdwVal = pdw;}
	inline void setptr(double * pd){pdVal = pd;}
	inline void setptr(IP     * pip){pipVal = pip;}
	inline void setptr(IPNET  * pipn){pipnVal = pipn;}
	inline void setptr(MPropertySet * pSt){pSet = pSt;}
	// data members
	bool m_bPersistent;
	//bool m_bModified;
private:
	MProperty(); // no implementation
	MProperty(const MProperty&); // no implementation
	const MProperty& operator=(const MProperty&); // no implementation
};

// declare link between PT_ constants and types
// using template traits
template<class T>
struct TPropTraitFwd{
};
template<int PT>
struct TPropTraitRev{
};
#define LINK_PT(_type, _number)\
template<> struct TPropTraitFwd<_type> {\
	enum{ PT_TYPE = _number };\
};\
template<> struct TPropTraitRev<_number> {\
	typedef _type T_type;\
};
// declare particular mappings
LINK_PT(bool,   PT_BOOL)
LINK_PT(int,    PT_INT)
LINK_PT(DWORD,  PT_DWORD)
LINK_PT(double, PT_DOUBLE)
LINK_PT(IP,     PT_IP)
LINK_PT(IPNET,  PT_IPNET)
//LINK_PT(LPCSTR, PT_STRING) // I'm not really sure of this mapping

template<class T, int PT = TPropTraitFwd<T>::PT_TYPE>
class TProperty : public MProperty {
public: 
	TProperty(LPCSTR szName, T def = 0) : MProperty(szName) {
		type = PT;
		size = sizeof(T);
		setval(def);
	}
	TProperty(LPCSTR szName, T * pdef) : MProperty(szName) {
		ASSERT(pdef);
		type = PT | PT_BYREF;
		size = sizeof(T);
		setptr(pdef);
	}
	TProperty(LPCSTR szName, T * pdef, const T& init) : MProperty(szName) {
		ASSERT(pdef);
		type = PT | PT_BYREF;
		size = sizeof(T);
		setptr(pdef);
		*pdef = init;
	}
};

typedef TProperty<bool>   MPropBool;
typedef TProperty<int>    MPropInt;
typedef TProperty<DWORD>  MPropDword;
typedef TProperty<double> MPropDbl;
typedef TProperty<IP>     MPropIP;
typedef TProperty<IPNET>  MPropIPNET;

class MPropString : public MProperty {
public:
	MPropString(LPCSTR szName, int nStrSize = DEF_STR_SIZE) : MProperty(szName) {
		ASSERT(nStrSize>0);
		type = PT_STRING;
		size = nStrSize;
		lpstrVal = new char[nStrSize];
		lpstrVal[0] = '\0';
	}
	MPropString(LPCSTR szName, LPSTR szStr, int nStrSize) : MProperty(szName) {
		ASSERT(szStr);
		ASSERT(nStrSize>0);
		type = PT_STRING | PT_BYREF;
		size = nStrSize;
		lpstrVal = szStr;
	}
	MPropString(LPCSTR szName, LPSTR szStr, int nStrSize, LPCSTR szInit) : MProperty(szName) {
		ASSERT(szStr);
		ASSERT(nStrSize>0);
		type = PT_STRING | PT_BYREF;
		size = nStrSize;
		lpstrVal = szStr;
		strncpy(lpstrVal, szInit, nStrSize-1);
	}
	~MPropString(){if ( (type&PT_BYREF) == 0) delete [] lpstrVal;}
};

class MPropertySection{
	friend class MPropertyContainer;
public:
	typedef map<CString, MProperty*>::iterator prop_iterator;
	//
	static inline MProperty* GetProperty(const prop_iterator& it){return it->second;}
	static inline LPCSTR     GetPropertyName(const prop_iterator& it){return it->first.c_str();}
	inline prop_iterator begin(){return m_props.begin();}
	inline prop_iterator end(){return m_props.end();}
protected:
	map<CString, MProperty*> m_props;
	CString m_sName;
};

class MPropertySet{
public:
	MPropertySet() {}
	virtual ~MPropertySet() {}
	//
	virtual bool InsertStr(const CString&)  = 0;
	virtual bool RemoveStr(const CString&)  = 0;
	virtual bool IsInStr(const CString&)    = 0;
	//
	virtual bool SetFromStr(const CString&) = 0;
	virtual CString Format()                = 0;
};

//////////////////////////////////////////////////////////////////////////////////////////////

template<class t_type>

class TPropertySet : public MPropertySet{
public:
	typedef typename set<t_type>::iterator iterator;
	typedef typename set<t_type>::reverse_iterator reverse_iterator;
	//
	TPropertySet(MMutex* pMutex) : m_pMutexSet(pMutex) {}
	//
	virtual bool SetFromStr(const CString&); // this five functions will rely on
	virtual CString Format();                // the implementation of SetFromStr()
	virtual bool InsertStr(const CString&);  // and Format() of TProperty<t_type>
	virtual bool RemoveStr(const CString&);
	virtual bool IsInStr(const CString&);
	//
	void Insert(const t_type&);
	bool Remove(const t_type&);
	bool IsIn(const t_type&);
	void CopyTo(set<t_type>&);
	template<class t_item_type>
	bool IsIn(const t_item_type&);
protected:
	set<t_type> m_set;
	MMutex* m_pMutexSet;
	//
	void Compact(); // needed by certain sets
};

// declare partial specialisations
template<>
bool TPropertySet<CString>::SetFromStr(const CString& str);
template<>
CString TPropertySet<CString>::Format();
template<>
bool TPropertySet<CString>::InsertStr(const CString& str);
template<>
bool TPropertySet<CString>::RemoveStr(const CString& str);
template<>
bool TPropertySet<CString>::IsInStr(const CString& str);

// provide common implementation

template<class t_type>
bool TPropertySet<t_type>::SetFromStr(const CString& str)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	// search for '{' than locate next '}' and pass
	// what is in between to SetFromStr...
	t_type tmp;
	TProperty<t_type> prop("tmp", &tmp);
	int nPosStart = 0;
	int nPosOpen, nPosClose;
	char* szTmp = NULL;
	bool bResult = false; // be optimistic, return true if any succedded
	m_set.clear();
	while ( (nPosOpen = str.find('{', nPosStart)) >= 0  &&
			(nPosClose = str.find('}',nPosOpen+1)) >= 0  )
	{
		szTmp = (char*) realloc(szTmp, nPosClose-nPosOpen);
		strcpy(szTmp,str.substr(nPosOpen+1, nPosClose-nPosOpen-1).c_str());
		if (prop.SetFromStr(szTmp))
		{
			m_set.insert(tmp);
			bResult = true;
		}
		nPosStart = nPosClose+1;
	}
	// we might need to compact the set...
	Compact();
	//
	if (m_pMutexSet)
		m_pMutexSet->unlock();
	free(szTmp);
	return bResult;
}

template<class t_type>
CString TPropertySet<t_type>::Format()
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	CString sTmp;
	for (iterator it=m_set.begin(); it!=m_set.end(); ++it)
	{
		TProperty<t_type> prop("tmp", const_cast<t_type*>(&(*it)));
		sTmp += "{" + prop.Format() + "},";
	}
	if (sTmp.size())
		sTmp.cut(sTmp.size()-1); // remove the trailing ','
	if (m_pMutexSet)
		m_pMutexSet->unlock();
	return sTmp;
}

template<class t_type>
bool TPropertySet<t_type>::InsertStr(const CString& str)
{
	t_type tmp;
	TProperty<t_type> prop("tmp", &tmp);
	char* szTmp = (char*) alloca(str.length()+1);
	strcpy(szTmp,str.c_str());
	if(!prop.SetFromStr(szTmp))
		return false;
	Insert(tmp);
	return true;
}

template<class t_type>
bool TPropertySet<t_type>::RemoveStr(const CString& str)
{
	t_type tmp;
	TProperty<t_type> prop("tmp", &tmp);
	char* szTmp = (char*) alloca(str.length()+1);
	strcpy(szTmp,str.c_str());
	if(!prop.SetFromStr(szTmp))
		return false;
	return Remove(tmp);
}

template<class t_type>
bool TPropertySet<t_type>::IsInStr(const CString& str)
{
	t_type tmp;
	TProperty<t_type> prop("tmp", &tmp);
	char* szTmp = (char*) alloca(str.length()+1);
	strcpy(szTmp,str.c_str());
	if(!prop.SetFromStr(szTmp))
		return false;
	return IsIn(tmp);
}

template<>
void TPropertySet<IPNET>::Insert(const IPNET& item);

template<class t_type>
void TPropertySet<t_type>::Insert(const t_type& item)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	m_set.insert(item);
	if (m_pMutexSet)
		m_pMutexSet->unlock();
}

template<class t_type>
bool TPropertySet<t_type>::Remove(const t_type& item)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	iterator it = m_set.find(item);
	bool bFound = it != m_set.end();
	if (bFound)
		m_set.erase(it);
	if (m_pMutexSet)
		m_pMutexSet->unlock();
	return bFound;
}

template<class t_type>
bool TPropertySet<t_type>::IsIn(const t_type& item)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	iterator it = m_set.find(item);
	bool bFound = it != m_set.end();
	if (m_pMutexSet)
		m_pMutexSet->unlock();
	return bFound;
}

/*template<class t_type>
template<class t_item_type>
bool TPropertySet<t_type>::IsIn(const t_item_type& item)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	iterator it = m_set.find(item);
	bool bFound = it != m_set.end();
	if (m_pMutexSet)
		m_pMutexSet->unlock();
	return bFound;
}*/

template<class t_type>
void TPropertySet<t_type>::CopyTo(set<t_type>& copy_set)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	copy_set = m_set;
	if (m_pMutexSet)
		m_pMutexSet->unlock();
}

template<>
void TPropertySet<IPNET>::Compact();

template<class t_type>
void TPropertySet<t_type>::Compact()
{
}

///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////

class MPropertyContainer {
public:
    typedef map<CString, MProperty*>::iterator prop_iterator;
    typedef map<CString, MPropertySection*>::iterator sec_iterator;
    typedef prop_iterator iterator;

	MPropertyContainer();
	~MPropertyContainer();
	
	int Transfer(MPropertyContainer* pPC);
	inline int Count(){return m_props.size();}
	
	MProperty* AddNewProperty(LPCSTR szName, int nType, int nStrSize = DEF_STR_SIZE);
	MProperty* AddProperty(LPCSTR szName, bool * pb);
	MProperty* AddProperty(LPCSTR szName, bool * pb, bool bInit);
	MProperty* AddProperty(LPCSTR szName, int * pn);
	MProperty* AddProperty(LPCSTR szName, int * pn, int nInit);
	MProperty* AddProperty(LPCSTR szName, DWORD * pdw);
	MProperty* AddProperty(LPCSTR szName, DWORD * pdw, DWORD dwInit);
	MProperty* AddProperty(LPCSTR szName, double * pd);
	MProperty* AddProperty(LPCSTR szName, double * pd, double dInit);
	MProperty* AddProperty(LPCSTR szName, LPSTR pstr, int nStrSize);
	MProperty* AddProperty(LPCSTR szName, LPSTR pstr, int nStrSize, LPCSTR szInit);
	MProperty* AddProperty(LPCSTR szName, IP * pip);
	MProperty* AddProperty(LPCSTR szName, IP * pip, const IP& ipInit);
	MProperty* AddProperty(LPCSTR szName, IPNET * pnet);
	MProperty* AddProperty(LPCSTR szName, IPNET * pnet, const IPNET& netInit);
	// set functions
	bool AddSet(LPCSTR szName, MPropertySet* pSet);
	
	// general service	
	MProperty* FindProperty(LPCSTR szName);
	
	bool AddSection(LPCSTR szName); // also sets section as current
	bool SetCurrentSection(LPCSTR szName); // AddProperty "context"
	MPropertySection* FindSection(LPCSTR szName);
	
	// properties
	static inline MProperty* GetProperty(const iterator& it){return it->second;}
	static inline LPCSTR     GetPropertyName(const iterator& it){return it->first.c_str();}
	inline iterator begin(){return m_props.begin();}
	inline iterator end(){return m_props.end();}
	// sections
	bool HasSections(){return m_secs.size();}
	static inline MPropertySection* GetSection(const sec_iterator& it){return it->second;}
	static inline LPCSTR            GetSectionName(const sec_iterator& it){return it->first.c_str();}
	inline sec_iterator sec_begin(){return m_secs.begin();}
	inline sec_iterator sec_end(){return m_secs.end();}
	// save/load
	bool Read(LPCSTR szName);
	bool Write(LPCSTR szName);
protected:
	map<CString, MProperty*> m_props;
	map<CString, MPropertySection*> m_secs;
	MPropertySection* m_pCurSec;
};

#endif

