#define _LARGEFILE64_SOURCE	/* required for GLIBC to enable stat64 and friends */
#include <sys/types.h>
#include <errno.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <regex.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/wait.h>
#include <signal.h>
#include <fcntl.h>
#if defined(sun) || defined(__sun)
#include <sys/loadavg.h>
#endif

#include "version.h"
#include "error.h"
#include "mt.h"

extern int path_max;

int myrand(int max)
{
	return (int)(((double)max * (double)rand()) / (double)RAND_MAX);
}

#ifdef _DEBUG

extern buffer *lb;
extern int nfd;

typedef struct
{
	void *p;
	char *descr;
	int size;
} memlist;
memlist *pm = NULL;
int n_pm = 0;

void dump_mem(int sig)
{
	int loop;

	signal(SIGHUP, dump_mem);

	if (sig != SIGHUP)
		error_exit("unexpected signal %d for dump_mem\n", sig);

	LOG("%d elements of memory used\n", n_pm);
	for(loop=0; loop<n_pm; loop++)
	{
		LOG("%06d] %p %d (%s)\n", loop, pm[loop].p, pm[loop].size, pm[loop].descr);
	}
	LOG("--- finished memory dump\n");
	LOG("# windows: %d\n", nfd);
	for(loop=0; loop<nfd; loop++)
	{
		LOG("%02d] %d %d\n", loop, lb[loop].maxnlines, lb[loop].curpos);
	}
	LOG("--- finished lb dump\n");
}
void remove_mem_element(void *p)
{
	if (p)
	{
		int loop;
		for(loop=0; loop<n_pm; loop++)
		{
			if (pm[loop].p == p)
			{
				int n_to_move;

				LOG("myfree(%s): %p (%d bytes)\n", pm[loop].descr, pm[loop].p, pm[loop].size);

				n_to_move = (n_pm - loop) - 1;
				memmove(&pm[loop], &pm[loop + 1], n_to_move * sizeof(memlist));
				n_pm--;
				loop=-1;

				break;
			}
		}

		if (loop != -1)
		{
			LOG("myfree: pointer %p not found\n", p);
		}

		pm = (memlist *)realloc(pm, sizeof(memlist) * n_pm);
		if (!pm) error_exit("failed to shrink memorylist to %d elements\n", n_pm);
	}
}
void add_mem_element(void *p, int size, char *what)
{
	pm = (memlist *)realloc(pm, sizeof(memlist) * (n_pm + 1));
	if (!pm) error_exit("failed to grow memorylist from %d elements\n", n_pm);
	pm[n_pm].p = p;
	pm[n_pm].size = size;
	pm[n_pm].descr = what;
	n_pm++;
}
#endif

void myfree(void *p)
{
#ifdef _DEBUG
	remove_mem_element(p);
#endif
	free(p);
}

void * myrealloc(void *oldp, int newsize, char *what)
{
#ifdef _DEBUG
	void *dummy;

	LOG("myrealloc(%s): %p (%d bytes)\n", what, oldp, newsize);
	dummy = realloc(oldp, newsize);
	if (!dummy)
		error_exit("failed to reallocate to %d bytes for %s\n", newsize, what);
	remove_mem_element(oldp);
	add_mem_element(dummy, newsize, what);
	signal(SIGHUP, dump_mem);
#else
	/* ----------------------------------------------------
	 * add code for repeatingly retry? -> then configurable
	 * via configurationfile with number of retries and/or
	 * sleep
	 * ----------------------------------------------------
	 */
	void *dummy = realloc(oldp, newsize);
	if (!dummy)
		error_exit("failed to reallocate to %d bytes for %s\n", newsize, what);
#endif

	return dummy;
}

void * mymalloc(int size, char *what)
{
	return myrealloc(NULL, size, what);
}

char * mystrdup(char *in, char *what)
{
	int len = strlen(in);
	char *dummy = (char *)mymalloc(len + 1, what);

	memcpy(dummy, in, len);
	dummy[len] = 0x00;

	return dummy;
}

ssize_t WRITE(int fd, char *whereto, size_t len)
{
	ssize_t cnt=0;

	while(len>0)
	{
		ssize_t rc;

		rc = write(fd, whereto, len);

		if (rc == -1)
		{
			if (errno != EINTR && errno != EAGAIN)
				error_exit("Problem writing to filedescriptor\n");
		}
		else if (rc == 0)
		{
			break;
		}
		else
		{
			whereto += rc;
			len -= rc;
			cnt += rc;
		}
	}

	return cnt;
}

/** get_load
 * - in:      nothing
 * - returns: char *
 * this function returns a string containing the current load of the system
 * multitail is running on. format: %1.2f %1.2f %1.2f
 * the string is malloc()ed so the caller should free() things
 */
char *get_load(void)
{
#if !defined(__UCLIBC__) && (defined(__FreeBSD__) || defined(linux) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__APPLE__) || defined(__GNU__) || defined(sun) || defined(__sun))
	double loadavg[3];
	char *str = (char *)mymalloc(LOADAVG_STR_LEN, "loadavg string");

#if defined(__GLIBC__) && ( __GLIBC__ < 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ < 2))
	/* Older glibc doesn't have getloadavg() - use sysinfo() */
	/* thanks to Ville Herva for this code! */
	double scale = 1 << SI_LOAD_SHIFT;
	struct sysinfo si;

	if (sysinfo(&si) == -1)
	{
		/* let's exit: if these kind of system-
		 * calls start to fail, something must be
		 * really wrong
		 */
		error_exit("sysinfo() failed");
	}

	loadavg[0] = (double)si.loads[0] / scale;
	loadavg[1] = (double)si.loads[1] / scale;
	loadavg[2] = (double)si.loads[2] / scale;
#else
	if (getloadavg(loadavg, 3) == -1)
	{
		/* see comment on sysinfo() */
		error_exit("getloadavg() failed");
	}
#endif
	snprintf(str, LOADAVG_STR_LEN, "%1.2f %1.2f %1.2f", loadavg[0], loadavg[1], loadavg[2]);

	return str;
#else
	return mystrdup("", "getloadavg");
#endif
}

int get_vmsize(pid_t pid)
{
	int vmsize = -1;
#if defined(linux)
	FILE *fh;
	char *path = mymalloc(path_max, "path");

	sprintf(path, "/proc/%d/stat", pid);

	fh = fopen(path, "r");
	if (fh)
	{
		char *dummystr = mymalloc(path_max, "temp string");
		char dummychar;
		int dummy;

		fscanf(fh, "%d %s %c %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d", &dummy, dummystr, &dummychar, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &vmsize);

		fclose(fh);

		myfree(dummystr);
	}

	myfree(path);
#endif

	return vmsize;
}

/** stop_process
 * - in:      int pid  pid of process
 * - returns: nothing
 * this function sends a TERM-signal to the given process, sleeps for 1009 microseconds
 * and then sends a KILL-signal to the given process if it still exists. the TERM signal
 * is send so the process gets the possibility to gracefully exit. if it doesn't do that
 * in 100 microseconds, it is terminated
 */
void stop_process(int pid)
{
	if (kill(pid, SIGTERM) == -1)
	{
		if (errno != ESRCH)
			error_exit("problem stopping child-process (%d)!\n", pid);
	}

	usleep(1000);

	/* process still exists? */
	if (kill(pid, SIGTERM) == 0)
	{
		/* sleep for a millisecond... */
		usleep(1000);

		/* ...and then really terminate the process */
		if (kill(pid, SIGKILL) == -1)
		{
			if (errno != ESRCH)
				error_exit("problem killing child-process (%d)!\n", pid);
		}
	}
	else if (errno != ESRCH)
		error_exit("problem stopping child-process (%d)!\n", pid);

	/* wait for the last remainder of the exited process to go away,
	 * otherwhise we'll find zombies on our way
	 */
	if (waitpid(pid, NULL, WNOHANG | WUNTRACED) == -1)
	{
		if (errno != ECHILD)
			error_exit("waitpid failed\n");
	}
}

/** delete_array
 * - in:      char **list array of strings to free
 *            int n       number of strings in this array
 * - returns: nothing
 * this function frees an array of strings: all strings are freed and
 * also the pointer-list itself is freed
 */
void delete_array(char **list, int n)
{
	int loop;

	for(loop=n-1; loop>=0; loop--)
		myfree(list[loop]);

	myfree(list);
}

int find_char_offset(char *str, char what)
{
	char *pnt = strchr(str, what);
	if (!pnt)
		return -1;

	return (int)(pnt - str);
}

int file_info(char *filename, off64_t *file_size, time_t *ts)
{
	struct stat64 buf;

	if (stat64(filename, &buf) == -1)
	{
		if (errno != ENOENT)
			error_exit("error while doing stat() on file %s\n", filename);

		return -1;
	}

	*file_size = buf.st_size;

	*ts = buf.st_mtime;

	return 0;
}

int file_exist(char *filename)
{
	struct stat64 buf;

	return stat64(filename, &buf);
}

char * convert_regexp_error(int error, const regex_t *preg)
{
	/* errors are specified not to be longer then 256 characters */
	char *multitail_string = "MultiTail warning: regular expression failed, reason: %s\n";
	int len = strlen(multitail_string);
	char *error_out = NULL;
	const int max_err_len = 256;

	if (error != 0 && error != REG_NOMATCH)
	{
		error_out = (char *)mymalloc(max_err_len + len + 1, "regexp error string");

		/* convert string */
		regerror(error, preg, &error_out[len], max_err_len);
	}

	return error_out;
}

/* I'm sorry for this: */
char check_date(void)
{
	time_t now;
	struct tm *ptm;

	time(&now);
	ptm = localtime(&now);

	if (ptm -> tm_mon == 3 && ptm -> tm_mday == 2) /* April the 2nd? */
	{
		return 1;
	}

	return 0;
}

char * amount_to_str(long long int amount)
{
	char *out = mymalloc(AMOUNT_STR_LEN, "converted amount string");	/* ...XB\0 */

	if (amount >= M_GB)	/* GB */
		snprintf(out, AMOUNT_STR_LEN, "%dGB", (int)((amount + M_GB - 1) / M_GB));
	else if (amount >= M_MB)	/* MB */
		snprintf(out, AMOUNT_STR_LEN, "%dMB", (int)((amount + M_MB - 1) / M_MB));
	else if (amount >= M_KB)	/* KB */
		snprintf(out, AMOUNT_STR_LEN, "%dKB", (int)((amount + M_KB - 1) / M_KB));
	else
		snprintf(out, AMOUNT_STR_LEN, "%d", (int)(amount));

	return out;
}

char * replace_string(char *in, int pos_start, int pos_end, char *with_what)
{
	int str_len_in = strlen(in);
	int str_len_ww = strlen(with_what);
	int n_remove = (pos_end - pos_start) + 1; /* +1 => including pos_end! */
	int new_len = str_len_in + str_len_ww - n_remove;
	char *out = mymalloc(new_len + 1, "replace_string new string"); /* +1 => 0x00 */

	memcpy(out, in, pos_start);
	memcpy(&out[pos_start], with_what, str_len_ww);
	memcpy(&out[pos_start + str_len_ww], &in[pos_end + 1], str_len_in - (pos_end + 1));
	out[new_len] = 0x00;

	return out;
}

struct passwd *getuserinfo(void)
{
	struct passwd *pp = getpwuid(geteuid());
	if (!pp)
		error_exit("failed to get passwdstructure for effective user id %d", geteuid());

	return pp;
}

/* these are there because AIX/IRIX can return EINTR for those */
int myopen(char *path, int mode)
{
	int fd;

	for(;;)
	{
		fd = open64(path, mode);
		if (fd == -1)
		{
			if (errno == EINTR || errno == EAGAIN)
				continue;

			return fd;
		}

		break;
	}

	return fd;
}

int myclose(int fd)
{
	for(;;)
	{
		if (close(fd) == -1)
		{
			if (errno == EINTR || errno == EAGAIN)
				continue;

			return -1;
		}

		return 0;
	}
}
