#define _LARGEFILE64_SOURCE	/* required for GLIBC to enable stat64 and friends */
#include <ctype.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <regex.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <pwd.h>
#include <stdarg.h>

#include "mt.h"
#include "colors.h"
#include "error.h"
#include "term.h"
#include "utils.h"
#include "help.h"

extern char use_colors;
extern int n_colors;
extern char mail;
extern char do_refresh;
extern int max_y, max_x;
extern char bright_colors;

int term_type = TERM_MISC;
extern help_t *help;

void wrong_key(void)
{
	flash();
	flushinp();
}

int ask_yes_no(int what_help, NEWWIN *popup)
{
	for(;;)
	{
		int c = toupper(wait_for_keypress(what_help, 0, popup, 0));

		switch(c)
		{
		case 'Y':
		case 'J':
			return 1;

		case 'N':
			return 0;

		case 7:
		case 'Q':
			return -1;
		}

		wrong_key();
	}
}

void color_on(NEWWIN *win, int index)
{
#ifdef N_CURSES
	if (use_colors)
		wattron(win -> win, COLOR_PAIR(index));
#endif
}

void color_off(NEWWIN *win, int index)
{
#ifdef N_CURSES
	if (use_colors)
		wattroff(win -> win, COLOR_PAIR(index));
#endif
}

void inverse_on(NEWWIN *win)
{
	wattron(win -> win, A_REVERSE);
}

void inverse_off(NEWWIN *win)
{
	wattroff(win -> win, A_REVERSE);
}

void draw_line(NEWWIN *win, char where)
{
	int loop;
	int mx = getmaxx(win -> win), my = getmaxy(win -> win);

	getmaxyx(win -> win, my, mx);

	inverse_on(win);

	if (where == LINE_LEFT)
		mvwvline(win -> win, 0, 0, ' ', my);
	else if (where == LINE_RIGHT)
		mvwvline(win -> win, 0, mx-1, ' ', my);
	else if (where == LINE_TOP)
		mvwhline(win -> win, 0, 0, ' ', mx);
	else if (where == LINE_BOTTOM)
	{
		if (check_date())
		{
			for(loop=0; loop<mx; loop++)
			{
				color_on(win, loop % n_colors);
				mvwprintw(win -> win, my-1, loop, " ");
				color_off(win, loop % n_colors);
			}
		}
		else
		{
			mvwhline(win -> win, my-1, 0, ' ', mx);
		}
	}

	inverse_off(win);
}

char * edit_string(NEWWIN *win, int y, int win_width, int max_width, char numbers_only, char *input_string, int what_help)
{
	char *string = (char *)mymalloc(max_width + 1, "edit buffer");
	int str_pos = 0, x = 0;
	int line_width = win_width - 4;

	inverse_on(win);
	mvwprintw(win -> win, y, 1, ">");
	mvwprintw(win -> win, y, win_width - 2, "<");
	inverse_off(win);

	if (input_string)
	{
		int input_string_len = strlen(input_string), copy_len = min(max_width, input_string_len);
		int dummy = max(0, str_pos - line_width);

		memcpy(string, input_string, copy_len);
		string[copy_len] = 0x00;

		str_pos = dummy;
		mvwprintw(win -> win, y, 2, &string[dummy]);
		x = strlen(string) - dummy;
	}
	else
	{
		string[0] = 0x00;
	}
	wmove(win -> win, y, 2 + x);

	mydoupdate();

	for(;;)
	{
		char force_redraw = 0;
		int prev_str_pos = str_pos;
		int c = wait_for_keypress(what_help, 0, NULL, 1);

		/* confirm */
		if (c == KEY_ENTER || c == 13 || c == 10 )
			break;

		/* abort */
		if (c == 7 || c == 17 || c == 24) /* ^g / ^q / ^x */
		{
			string[0] = 0x00;
			break;
		}

		switch(c)
		{
		case 1:			/* ^A */
			str_pos = x = 0;
			break;
		case 5:			/* ^E */
			{
				int dummy = strlen(string);

				if (dummy > line_width)
				{
					str_pos = dummy - (line_width / 2);
					x = (line_width / 2);
				}
				else
				{
					str_pos = 0;
					x = dummy;
				}
			}
			break;
		case 9:			/* tab (filename completion) */
			if (numbers_only)
			{
				wrong_key();
			}
			else
			{
				int dummy;
				char *file = select_file(string, -1);

				if (file)
				{
					strncpy(string, file, max_width);
					string[max_width] = 0x00;
					myfree(file);
					str_pos = 0;
				}

				dummy = strlen(string);
				if (dummy > line_width)
				{
					str_pos = dummy - (line_width / 2);
					x = (line_width / 2);
				}
				else
				{
					str_pos = 0;
					x = dummy;
				}

				force_redraw = 1;
			}
			break;
		case 23:		/* ^W */
			string[0] = 0x00;
			str_pos = x = 0;
			break;
		case 127:		/* DEL */
		case KEY_BACKSPACE:
			if ((str_pos + x) > 0)
			{
				memmove(&string[str_pos + x - 1], &string[str_pos + x], (max_width - (str_pos + x)) + 1);
				if (x > 0)
				{
					x--;
				}
				else
				{
					str_pos--;
				}

				force_redraw = 1;
			}
			break;
		case KEY_LEFT:
			if (x > 0)
			{
				x--;
			}
			else if (str_pos > 0)
			{
				str_pos--;
			}
			break;
		case KEY_RIGHT:
			if ((x + str_pos) < strlen(string))
			{
				if (x < line_width)
					x++;
				else
					str_pos++;
			}
			else
			{
				wrong_key();
			}
			break;
		default:
			{
				int len = strlen(string);

				/* only allow valid ASCII */
				if (c < 32)
				{
					wrong_key();
					break;
				}

				if (numbers_only && (c < '0' || c > '9'))
				{
					wrong_key();
					break;
				}

				if (len == max_width)
				{
					wrong_key();
					break;
				}

				/* cursor at end of string? */
				if (str_pos == len)
				{
					string[str_pos + x] = c;
					string[str_pos + x + 1] = 0x00;
					waddch(win -> win, c);
				}
				else /* add character to somewhere IN the string */
				{
					memmove(&string[str_pos + x + 1], &string[str_pos + x], strlen(&string[str_pos + x]) + 1);
					string[str_pos + x] = c;
					force_redraw = 1;
				}

				if ((x + str_pos) < max_width)
				{
					if (x < line_width)
						x++;
					else
						str_pos++;
				}
				else
				{
					wrong_key();
				}
			}
			break;
		}

		if (str_pos != prev_str_pos || force_redraw)
		{
			int loop;
			char *dummy = mystrdup(&string[str_pos], "edit_string");
			dummy[min(strlen(dummy), line_width)] = 0x00;
			for(loop=strlen(dummy); loop<line_width; loop++)
				mvwprintw(win -> win, y, 2 + loop, " ");
			mvwprintw(win -> win, y, 2, dummy);
			myfree(dummy);
			force_redraw = 0;
		}
		wmove(win -> win, y, 2 + x);
		mydoupdate();
	}

	if (strlen(string) == 0)
	{
		myfree(string);
		string = NULL;
	}

	return string;
}

NEWWIN * create_popup(int n_lines, int n_colls)
{
	NEWWIN *newwin = NULL;
        int ocols  = (max_x/2) - (n_colls/2);
        int olines = (max_y/2) - (n_lines/2);

	/* create new window */
	newwin = mynewwin(n_lines, n_colls, olines, ocols);

	werase(newwin -> win);
	box(newwin -> win, 0, 0);

	show_panel(newwin -> pwin);

	return newwin;
}

void delete_popup(NEWWIN *mywin)
{
	mydelwin(mywin);

	update_panels();
	doupdate();

	myfree(mywin);
}

void draw_marker_line(NEWWIN *win)
{
	int mx = getmaxx(win -> win);
	int loop;

	inverse_on(win);

	if (check_date())
	{
		for(loop=0; loop<mx; loop++)
		{
			color_on(win, loop % n_colors);
			waddch(win -> win, '-');
			color_off(win, loop % n_colors);
		}
	}
	else
	{
		color_on(win, MY_RED);
		for(loop=0; loop<mx; loop++)
				waddch(win -> win, '-');
		color_off(win, MY_RED);
	}

	inverse_off(win);
}

void mydelwin(NEWWIN *win)
{
	bottom_panel(win -> pwin);

	if (ERR == del_panel(win -> pwin))
		error_exit("del_panel() failed\n");

	if (ERR == delwin(win -> win))
		error_exit("delwin() failed\n");
}

void mydoupdate()
{
	update_panels();
	doupdate();
}

NEWWIN * mynewwin(int nlines, int ncols, int begin_y, int begin_x)
{
	NEWWIN *nwin = (NEWWIN *)mymalloc(sizeof(NEWWIN), "window");
/*	nwin -> win = subwin(stdscr, nlines, ncols, begin_y, begin_x); */
	nwin -> win = newwin(nlines, ncols, begin_y, begin_x);
	if (!nwin -> win)
		error_exit("failed to create window (subwin) with dimensions %d-%d at offset %d,%d (terminal size: %d,%d)\n", ncols, nlines, begin_x, begin_y, COLS, LINES);

	nwin -> pwin = new_panel(nwin -> win);
	if (!nwin -> pwin)
		error_exit("failed to create panel for new window");

	nwin -> x_off = begin_x;
	nwin -> y_off = begin_y;
	nwin -> width = ncols;
	nwin -> height = nlines;

	if (bright_colors)
		wattron(nwin -> win, A_BOLD);

	return nwin;
}

void escape_print(NEWWIN *win, int y, int x, char *str)
{
	int loop, index = 0, len = strlen(str);
	char inv = 0, ul = 0, bold = 0;

	for(loop=0; loop<len; loop++)
	{
		if (str[loop] == '^') /* ^^ is ^ */
		{
			if (str[loop + 1 ] == '^')
			{
				/* just print a _ */
				mvwprintw(win -> win, y, x + index++, "^");
				loop++;
			}
			else
			{
				if (!inv)
					inverse_on(win);
				else
					inverse_off(win);

				inv = 1 - inv;
			}
		}
		else if (str[loop] == '_')
		{
			if (str[loop + 1] == '_')	/* __ is _ */
			{
				/* just print a _ */
				mvwprintw(win -> win, y, x + index++, "_");
				loop++;
			}
			else
			{
				if (!ul)
					wattron(win -> win, A_UNDERLINE);
				else
					wattroff(win -> win, A_UNDERLINE);

				ul = 1 - ul;
			}
		}
		else if (str[loop] == '*')
		{
			if (str[loop + 1] == '*')
			{
				/* just print a * */
				mvwprintw(win -> win, y, x + index++, "*");
				loop++;
			}
			else
			{
				if (!bold)
					wattron(win -> win, A_BOLD);
				else
					wattroff(win -> win, A_BOLD);

				bold = 1 - bold;
			}
		}
		else
		{
			mvwprintw(win -> win, y, x + index++, "%c", str[loop]);
		}
	}

	if (inv) inverse_off(win);
	if (ul)  wattroff(win -> win, A_UNDERLINE);
	if (bold)  wattroff(win -> win, A_BOLD);
}

void win_header(NEWWIN *win, char *str)
{
	wattron(win -> win, A_BOLD);
	mvwprintw(win -> win, 1, 2, str);
	wattroff(win -> win, A_BOLD);
}

void gui_window_header(char *string)
{
	if (term_type == TERM_XTERM)
	{
		/* \033]0;%s\007 */
		putp("\033]0;");
		putp(string);
		putp("\007");
	}
}

void error_popup(char *title, int help, char *message, ...)
{
	va_list ap;
	NEWWIN *mywin;
	char buffer[4096];
	int len;

	va_start(ap, message);
	vsnprintf(buffer, sizeof(buffer), message, ap);
	va_end(ap);

	len = strlen(buffer);

	mywin = create_popup(7, max(33, len) + 4);

	win_header(mywin, title);
	color_on(mywin, MY_RED);
	wattron(mywin -> win, A_BLINK);
	mvwprintw(mywin -> win, 3, 2, buffer);
	wattroff(mywin -> win, A_BLINK);
	color_off(mywin, MY_RED);
	mvwprintw(mywin -> win, 5, 2, "Press any key to exit this screen");
	mydoupdate();

	wrong_key();

	wait_for_keypress(help, 0, mywin, 0);

	delete_popup(mywin);
}
