/* $Id: mt.c,v 2.5.2.1 2003/05/11 19:12:01 folkert Exp folkert $
 */
#include "mt.h"
#include "version.h"

#include <ctype.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <regex.h>
#ifndef __APPLE__
#include <search.h>
#endif
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#ifndef AIX
#include <sys/termios.h> /* needed on Solaris 8 */
#endif
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>

#include "error.h"
#include "my_pty.h"
#include "utils.h"
#include "colors.h"


/* #define KEYB_DEBUG */

#define min(x, y) ((x)<(y)?(x):(y))
#define max(x, y) ((x)>(y)?(x):(y))

#define LINE_LEFT	0
#define LINE_RIGHT	1
#define LINE_TOP	2
#define LINE_BOTTOM	3

#define SEL_WIN		0
#define SEL_SUBWIN	1
#define SEL_FILES	2
#define SEL_CSCHEME	3

typedef struct
{
        char *regex_str;
        regex_t regex;
        char invert_regex;
	char use_regex;
} re;

typedef struct _subwindow_
{
	char *filename;
	char is_command;
	int fd;
	pid_t pid;

	char line_wrap;
	int line_wrap_offset;

	/* repeatingly start a program */
	int restart;
	char first;
	char do_diff;
	char **bcur, **bprev;
	int ncur, nprev;

	char colorize;
	char field_nr;
	char *field_del;
	int color_scheme;

	char hidden;
	char follow_filename;
	char retry_open;

	WINDOW *status;
	WINDOW *data;

	int n_re;
	re *pre;

	struct _subwindow_ *next;
} proginfo;

typedef struct
{
	char **Blines;
	proginfo **pi;
	int curpos;
	char markset;
	int maxnlines;
} buffer;

typedef struct
{
	char *name;
	int n;
	int *color;
        regex_t *regex;
} color_scheme;

proginfo *pi = NULL;
buffer *lb = NULL;
int nfd = 0;
int max_y, max_x;
char terminal_changed = 0;
char split = 0;
char banner = 1;
WINDOW *splitline = NULL;
char mode_statusline = 1;
char warn_closed = 1;
int n_colors = -1;
char use_colors = 0;
int min_n_bufferlines;
int path_max = 0;
int heartbeat = 0;
color_scheme *cschemes = NULL;
int n_cschemes = 0;

//#if 1
void LOG(char *s, ...)
{
        va_list ap;
	FILE *fh = fopen("log.log", "a+");
	if (!fh)
	{
		endwin();
		printf("error\n");
	}

        va_start(ap, s);
        vfprintf(fh, s, ap);
        va_end(ap);

	fclose(fh);
}
//#else
//#define LOG(x)
//#endif

char *keys[] = {
		"Keys while the program runs:",
		"	q / x	exit program",
		"	r	redraw",
		"	e	edit regular expression for a given window",
		"	d	delete window",
		"	a	add window",
		"	s	swap windows",
		"	c	toggle colors",
		"	v	toggle vertical screen split",
		"	z	hide/unhide window",
		"	w	write commandline",
		"	m	set mark",
		"	n	delete mark",
		"	b	scrollback",
		"	p	pause",
		"	i	info",
		"	h	help",
		NULL
		};

/** get_load
 * - in:      nothing
 * - returns: char *
 * this function returns a string containing the current load of the system
 * multitail is running on. format: %1.2f %1.2f %1.2f
 * the string is malloc()ed so the caller should free() things
 */
#if defined(__FreeBSD__) || defined(linux) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__APPLE__) || defined(sun)
char *get_load(void)
{
	double loadavg[3];
	char *str = (char *)mymalloc(20, "loadavg string");

	if (getloadavg(loadavg, 3) == -1)
	{
		free(str);
		return NULL;
	}

	sprintf(str, "%1.2f %1.2f %1.2f", loadavg[0], loadavg[1], loadavg[2]);

	return str;
}
#endif

/** stop_process
 * - in:      int pid  pid of process
 * - returns: nothing
 * this function sends a TERM-signal to the given process, sleeps for 1009 microseconds
 * and then sends a KILL-signal to the given process if it still exists. the TERM signal
 * is send so the process gets the possibility to gracefully exit. if it doesn't do that
 * in 100 microseconds, it is terminated
 */
void stop_process(int pid)
{
	if (kill(pid, SIGTERM) == -1)
	{
		if (errno != ESRCH)
			error_exit("problem stopping child-process (%d)!\n", pid);
	}

	usleep(1000);

	/* process still exists? */
	if (kill(pid, SIGTERM) == 0)
	{
		/* sleep for a millisecond... */
		usleep(1000);

		/* ...and then really terminate the process */
		if (kill(pid, SIGKILL) == -1)
		{
			if (errno != ESRCH)
				error_exit("problem killing child-process (%d)!\n", pid);
		}
	}
	else if (errno != ESRCH)
		error_exit("problem stopping child-process (%d)!\n", pid);
}

/** do_exit
 * - in:      int sig
 * - returns: nothing (doesn't return!)
 * this function is called when MultiTail receives the TERM-signal. it is also
 * called by, for example, wait_for_keypress when ^c is pressed. it stops all
 * child-processes, ends the curses-library and exits with SUCCESS status
 */
void do_exit(int sig)
{
	proginfo *cur;
	int loop;

	/* kill tail processes */
	for(loop=0; loop<nfd; loop++)
	{
		cur = &pi[loop];

		do
		{
			stop_process(cur -> pid);
			cur = cur -> next;
		}
		while(cur);
	}

	endwin();

	exit(EXIT_SUCCESS);
}

/** create_subwindow_list
 * - in:      int f_index     window number
 *            char ***swlist  pointer to an array of strings
 * - returns: int             number of elements in the array of strings
 * this function creates for a given window (f_index) a list of subwindows:
 * subwindows are created when you're merging the output of several files/
 * commands
 */
int create_subwindow_list(int f_index, char ***swlist)
{
	char **list = NULL;
	int n = 0;
	proginfo *cur = &pi[f_index];

	do
	{
		list = (char **)myrealloc(list, (n + 1) * sizeof(char *), "subwindow list");

		list[n] = mystrdup(cur -> filename);

		n++;

		cur = cur -> next;
	}
	while(cur);

	*swlist = list;

	return n;
}

/** delete_array
 * - in:      char **list array of strings to free
 *            int n       number of strings in this array
 * - returns: nothing
 * this function frees an array of strings: all strings are freed and
 * also the pointer-list itself is freed
 */
void delete_array(char **list, int n)
{
	int loop;

	for(loop=n-1; loop>=0; loop--)
		free(list[loop]);

	free(list);
}

void free_re(re *cur_re)
{
	free(cur_re -> regex_str);
	if (cur_re -> use_regex)
		regfree(&cur_re -> regex);
}

void free_subentry(proginfo *entry)
{
	int loop;

	/* free all those allocated memory blocks */
	free(entry -> filename);
	free(entry -> field_del);
	if (entry -> status)
		mydelwin(entry -> status);
	if (entry -> data)
		mydelwin(entry -> data);

	/* free buffers for diff (if any) */
	if (entry -> bcur)
		delete_array(entry -> bcur, entry -> ncur);
	if (entry -> bprev)
		delete_array(entry -> bprev, entry -> nprev);

	/* delete regular expressions */
	for(loop=0; loop<entry -> n_re; loop++)
		free_re(&(entry -> pre)[loop]);
	free(entry -> pre);

	/* stop process */
	stop_process(entry -> pid);
	if (waitpid(entry -> pid, NULL, WNOHANG | WUNTRACED) == -1)
	{
		if (errno != ECHILD)
			error_exit("waitpid failed\n");
	}
}

void store_for_diff(proginfo *cur, char *string)
{
	cur -> bcur = (char **)myrealloc(cur -> bcur, sizeof(char *) * (cur -> ncur + 1), "difference list");

	(cur -> bcur)[cur -> ncur] = mystrdup(string);

	cur -> ncur++;
}

void buffer_replace_pi_pointers(int f_index, proginfo *org, proginfo *new)
{
	int loop;

	for(loop=0; loop<lb[f_index].curpos; loop++)
	{
		if ((lb[f_index].pi)[loop] == org)
		{
			if (!new)
			{
				free((lb[f_index].Blines)[loop]);
				(lb[f_index].Blines)[loop] = NULL;
				(lb[f_index].pi)[loop] = NULL;
			}
			else
			{
				(lb[f_index].pi)[loop] = new;
			}
		}
	}
}

char delete_entry(int f_index, proginfo *sub)
{
	char delete_all = 0;

	/* no children? then sub must be pointing to current */
	if (pi[f_index].next == NULL)
	{
		sub = NULL;
	}

	/* stop the process(es) we're watching ('tail' most of the time) */
	if (sub == NULL) /* delete all? */
	{
		proginfo *cur = &pi[f_index];

		do
		{
			free_subentry(cur);
			cur = cur -> next;
		}
		while(cur);

		/* free the subwindows (if any) */
		cur = pi[f_index].next;
		while(cur)
		{
			proginfo *dummy = cur -> next;
			free(cur);
			cur = dummy;
		}

		delete_all = 1;
	}
	else
	{
		free_subentry(sub);
	}

	/* remove entry from array */
	if (sub == NULL) /* delete entry in the main array */
	{
		int n_to_move = (nfd - f_index) - 1;

	        /* free buffers */
		delete_array(lb[f_index].Blines, lb[f_index].curpos);

		if (n_to_move > 0)
		{
			int loop;

			/* update buffer proginfo-pointers */
			for(loop=f_index + 1; loop<nfd; loop++)
			{
				// replace lb[f_index].pi -> pi[loop] met pi[loop-1]
				buffer_replace_pi_pointers(loop, &pi[loop], &pi[loop - 1]);
			}

			/* prog info */
			memmove(&pi[f_index], &pi[f_index+1], sizeof(proginfo) * n_to_move);
			/* buffers */
			memmove(&lb[f_index], &lb[f_index+1], sizeof(buffer) * n_to_move);
		}

		nfd--;

		/*  shrink array */
		if (nfd == 0)	/* empty? */
		{
			free(pi);
			pi = NULL;
			free(lb);
			lb = NULL;
		}
		else		/* not empty, just shrink */
		{
			pi = (proginfo *)myrealloc(pi, nfd * sizeof(proginfo), "proginfo list");
			lb = (buffer *)myrealloc(lb, nfd * sizeof(buffer), "nfd list");
		}
	}
	else		/* delete sub */
	{
		if (sub != &pi[f_index])	/* not in main array? */
		{
			proginfo *parent = &pi[f_index];

			/* find parent of 'sub' */
			while (parent -> next != sub)
				parent = parent -> next;

			parent -> next = sub -> next;

			buffer_replace_pi_pointers(f_index, sub, NULL);

			free(sub);
		}
		else				/* main array, find next */
		{
			proginfo *oldnext = pi[f_index].next;
			/* first, delete the entries of that entry (which is in the array) */
			buffer_replace_pi_pointers(f_index, &pi[f_index], NULL);
			/* then, 'rename' the pointers (original address -> address in array)... */
			buffer_replace_pi_pointers(f_index, oldnext, &pi[f_index]);
			/* and move the object to the array */
			memmove(&pi[f_index], oldnext, sizeof(proginfo));
			/* free the obsolete entry */
			free(oldnext);
		}
	}

	return delete_all;
}

int start_tail(proginfo *cur, int initial_tail)
{
	if (cur -> is_command)
	{
		int fd_master, fd_slave;

		/* allocate pseudo-tty & fork*/
		cur -> pid = get_pty_and_fork(&fd_master, &fd_slave);
		if (-1 == cur -> pid) error_exit("failed to create pseudo-tty & fork\n");

		/* child? */
		if (cur -> pid == 0)
		{
			/* reset signal handler for SIGTERM*/
			signal(SIGTERM, SIG_DFL);

			/* sleep if requested and only when 2nd or 3d (etc.) execution time */
			if (cur -> restart && cur -> first == 0)
				sleep(cur -> restart);

			/* connect slave-fd to stdin/out/err */
			if (-1 == close(0)) error_exit("close failed\n");
			if (-1 == close(1)) error_exit("close failed\n");
			if (-1 == close(2)) error_exit("close failed\n");
			if (-1 == dup(fd_slave)) error_exit("dup failed\n");
			if (-1 == dup(fd_slave)) error_exit("dup failed\n");
			if (-1 == dup(fd_slave)) error_exit("dup failed\n");

			/* set terminal */
			if (putenv("TERM=dumb") == -1)
			{
				fprintf(stderr, "Could not set TERM environment-variable\n");
			}

			/* start process */
			if (-1 == execlp("/bin/sh", "/bin/sh", "-c", cur -> filename, (void *)NULL)) error_exit("execlp of %s failed\n", cur -> filename);

			/* if execlp returns, an error occured */
			error_exit("error while starting process!\n");
		}

		/* remember master-fd (we'll read from that one) */
		cur -> fd = fd_master;

		/* next time, sleep */
		cur -> first = 0;
	}
	else
	{
		int pipefd[2];

		/* create a pipe, will be to child-process */
		if (-1 == pipe(pipefd)) error_exit("error creating pipe\n");

		/* start child process */
		if ((cur -> pid = fork()) == 0)
		{
			char par1[32] = { 0 };
			char par2[32] = { 0 };

			if (-1 == close(1)) error_exit("close failed\n");	/* close stdout */
			if (-1 == close(2)) error_exit("close failed\n");	/* close stderr */
			if (-1 == dup(pipefd[1])) error_exit("dup failed\n");	/* pipe-write is connected to stdout */
			if (-1 == dup(pipefd[1])) error_exit("dup failed\n");	/* pipe-write is connected to stderr */

			/* Linux' tail has the --retry option, but not all
			 * other UNIX'es have this, so I implemented it
			 * myself
			 */
			if (cur -> retry_open)
			{
				struct stat buf;

				for(;;)
				{
					int rc = stat(cur -> filename, &buf);
					if (rc == -1)
					{
						if (errno != ENOENT)
						{
							fprintf(stderr, "Error while looking for file %s: %d\n", cur -> filename, errno);
							exit(EXIT_FAILURE);
						}
					}
					else if (rc == 0)
						break;

					usleep(WAIT_FOR_FILE_DELAY * 1000);
				}
			}

			/* create command for take last n lines & follow and start tail */
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__) || defined(linux) || defined(__CYGWIN__)
			if (cur -> follow_filename)
			{
	#if defined(linux) || defined(__CYGWIN__)
				sprintf(par1, "--follow=name");
	#else
				sprintf(par1, "-F");
	#endif
				sprintf(par2, "-n %d", initial_tail);
			}
			else
#endif
			{
				sprintf(par1, "-%dlf", initial_tail);
			}

			/* run tail! */
			if (strlen(par2))
			{
				if (-1 == execlp("tail", "tail", par1, par2, cur -> filename, (void *)NULL)) error_exit("execlp of tail failed");
			}
			else
			{
				if (-1 == execlp("tail", "tail", par1, cur -> filename, (void *)NULL)) error_exit("execlp of tail failed");
			}

			/* if execlp returns, an error occured */
			error_exit("error while starting process!\n");
		}

		/* parent will only read from pipe, no write to */
		if (-1 == close(pipefd[1])) error_exit("close failed\n");

		cur -> fd = pipefd[0];
	}

	if (cur -> pid == -1)
		return -1;

	return 0;
}

void init_curses(void)
{
	initscr();
	if (use_colors)
	{
		start_color(); /* don't care if this one failes */
	}
	keypad(stdscr, TRUE);
	cbreak();
	intrflush(stdscr, FALSE);
	leaveok(stdscr, FALSE);
	noecho();
	nonl();
	refresh();
	nodelay(stdscr, FALSE);
	meta(stdscr, TRUE);	/* enable 8-bit input */
	raw();	/* to be able to catch ctrl+c */

#ifdef N_CURSES
	if (use_colors)
	{
		init_pair(MY_RED, COLOR_RED, COLOR_BLACK);
		init_pair(MY_GREEN, COLOR_GREEN, COLOR_BLACK);
		init_pair(MY_YELLOW, COLOR_YELLOW, COLOR_BLACK);
		init_pair(MY_BLUE, COLOR_BLUE, COLOR_BLACK);
		init_pair(MY_MAGENTA, COLOR_MAGENTA, COLOR_BLACK);
		init_pair(MY_CYAN, COLOR_CYAN, COLOR_BLACK);
		init_pair(MY_WHITE, COLOR_WHITE, COLOR_BLACK);
	}
	n_colors = 8;
#endif

	max_y = LINES;
	max_x = COLS;
}

int wait_for_keypress()
{
	fd_set rfds;
	int rc, c;

	FD_ZERO(&rfds);
	FD_SET(0, &rfds);

	for(;;)
	{
		rc = select(1, &rfds, NULL, NULL, NULL);
		if (rc == -1)
		{
			if (errno != EINTR)
				error_exit("select failed");
		}
		else
			break;
	}

	c = getch();
	if (c == 3)	/* ^C */
	{
		do_exit(-1);
		error_exit("this should not be reached\n");
	}
	else if (c == 26)/* ^Z */
	{
		if (kill(getpid(), SIGTSTP) == -1)
			error_exit("problem suspending\n");
	}

	return c;
}

void wrong_key(void)
{
	flash();
	flushinp();
}

char ask_yes_no(void)
{
	for(;;)
	{
		int c = toupper(wait_for_keypress());

		if (c == 'Y' || c == 'J')
		{
			return 1;
		}
		else if (c == 'N')
		{
			return 0;
		}

		wrong_key();
	}
}

char gen_color(char *start, char *end)
{
	char *loop;
	char chk = 0;

	for(loop=start; loop<end; loop++)
	{
		chk ^= *loop;
	}

	return abs(chk) % n_colors;
}

void color_on(WINDOW *win, int index)
{
#ifdef N_CURSES
	if (use_colors)
		wattron(win, COLOR_PAIR(index));
#endif
}

void color_off(WINDOW *win, int index)
{
#ifdef N_CURSES
	if (use_colors)
		wattroff(win, COLOR_PAIR(index));
#endif
}

void draw_line(WINDOW *win, char where)
{
	int loop;
	int mx = getmaxx(win), my = getmaxy(win);

	getmaxyx(win, my, mx);

	wattron(win, A_REVERSE);

	if (where == LINE_LEFT)
		for(loop=0; loop<my; loop++)
			mvwprintw(win, loop, 0, " ");
	else if (where == LINE_RIGHT)
		for(loop=0; loop<my; loop++)
			mvwprintw(win, loop, mx-1, " ");
	else if (where == LINE_TOP)
		for(loop=0; loop<mx; loop++)
			mvwprintw(win, 0, loop, " ");
	else if (where == LINE_BOTTOM)
	{
		/* I'm sorry for this: */
		time_t now;
		struct tm *ptm;

		time(&now);
		ptm = localtime(&now);

		if (ptm -> tm_mon == 3 && ptm -> tm_mday == 2) /* April the 2nd? */
		{
			for(loop=0; loop<mx; loop++)
			{
				color_on(win, loop % n_colors);
				mvwprintw(win, my-1, loop, " ");
				color_off(win, loop % n_colors);
			}
		}
		else
		{
			for(loop=0; loop<mx; loop++)
				mvwprintw(win, my-1, loop, " ");
		}
	}

	wattroff(win, A_REVERSE);
}

WINDOW * create_popup(int n_lines, int n_colls)
{
	WINDOW *win = mysubwin(stdscr, n_lines, n_colls, (max_y/2) - (n_lines/2), (max_x/2) - (n_colls/2));

	wclear(win);

	box(win, 0, 0);

	return win;
}

int compare_filenames(const void *arg1, const void *arg2)
{
	return strcmp(*(char **)arg1, *(char **)arg2);
}

int match_files(char *search_for, char **path, char ***found)
{
	DIR *dir;
	struct dirent *entry;
	char *fname;
	char **list = NULL;
	int nfound = 0;
	size_t fname_size;
	char *slash = strrchr(search_for, '/');
	if (slash)
	{
		fname = mystrdup(slash + 1);
		*(slash + 1) = 0x00;
		*path = search_for;
	}
	else
	{
		*path = "./";
		fname = mystrdup(search_for);
	}
	fname_size = strlen(fname);

	dir = opendir(*path);
	if (!dir)
	{
		return 0;
	}

	while((entry = readdir(dir)) != NULL)
	{
		if ((fname_size == 0 || strncmp(entry -> d_name, fname, fname_size) == 0) && strcmp(entry -> d_name, ".") != 0 &&
											     strcmp(entry -> d_name, "..") != 0)
		{
			list = (char **)myrealloc(list, (nfound + 1) * sizeof(char *), "directory list");

			list[nfound] = mystrdup(entry -> d_name);
			nfound++;
		}
	}

	if (closedir(dir) == -1)
		error_exit("closedir failed");

	qsort( (void *)list, (size_t)nfound, sizeof(char *), compare_filenames);

	*found = list;

	free(fname);

	return nfound;
}

void escape_print(WINDOW *win, int y, int x, char *str)
{
	int loop, index = 0, len = strlen(str);
	char inv = 0;

	for(loop=0; loop<len; loop++)
	{
		if (str[loop] == '^')
		{
			if (!inv)
				wattron(win, A_REVERSE);
			else
				wattroff(win, A_REVERSE);

			inv = 1 - inv;
		}
		else
		{
			mvwprintw(win, y, x + index++, "%c", str[loop]);
		}
	}

	if (inv) wattroff(win, A_REVERSE);
}

void win_header(WINDOW *win, char *str)
{
	wattron(win, A_REVERSE);
	mvwprintw(win, 1, 2, str);
	wattroff(win, A_REVERSE);
}

char generate_string(char *whereto, void **list, char type, int wcols, int index)
{
	char invert = 0;

	if (type == SEL_WIN)
	{
		snprintf(whereto, wcols + 1, "%02d %s", index, ((proginfo *)pi)[index].filename);
		invert = pi[index].hidden;
	}
	else if (type == SEL_SUBWIN || type == SEL_FILES)
		strncpy(whereto, ((char **)list)[index], min(strlen(((char **)list)[index]), wcols) + 1);
	else if (type == SEL_CSCHEME)
		strncpy(whereto, ((color_scheme *)list)[index].name, min(strlen(((color_scheme *)list)[index].name), wcols) + 1);
	else
		error_exit("internal error (%d is an unknown selectionbox type)\n", type);

	whereto[min(strlen(whereto), wcols)] = 0x00;

	return invert;
}

int selection_box(void **list, int nlines, char type)
{
	WINDOW *win;
	int wlines = min(nlines, (max_y - 1) - 3);
	int total_win_size = wlines + 3;
	int wcols = (max_x / 4) - 4;
	int pos = 0, ppos = -1, offs = 0, poffs = -1;
	int loop, sel = -1;
	char invert;
	char *dummy = (char *)mymalloc(wcols + 1, "windowname string");

	win = mysubwin(stdscr, total_win_size, max_x/4, (max_y / 2) - (total_win_size / 2), (max_x/4) * 3);

	for(;;)
	{
		int c;

		/* draw list */
		if (pos != ppos)
		{
			int entries_left = (nlines - pos);

			wclear(win);

			draw_line(win, LINE_TOP);
			draw_line(win, LINE_BOTTOM);

			win_header(win, "Select file");

			for(loop=0; loop<min(entries_left, wlines); loop++)
			{
				invert = generate_string(dummy, list, type, wcols, loop + pos);
				if (loop == offs)
					wattron(win, A_REVERSE);
				if (invert)
					color_on(win, 3);
				mvwprintw(win, loop + 2, 2, "%s", dummy);
				if (invert)
					color_off(win, 3);
				if (loop == offs)
					wattroff(win, A_REVERSE);
			}

			draw_line(win, LINE_LEFT);
			draw_line(win, LINE_RIGHT);

			mydoupdate(win);

			ppos = pos;
			poffs = offs;
		}
		else if (poffs != offs)
		{
			generate_string(dummy, list, type, wcols, poffs + pos);
			mvwprintw(win, poffs + 2, 2, "%s", dummy);

			invert = generate_string(dummy, list, type, wcols, offs + pos);

			wattron(win, A_REVERSE);
			if (invert)
				color_on(win, 3);
			mvwprintw(win, offs + 2, 2, "%s", dummy);
			if (invert)
				color_off(win, 3);
			wattroff(win, A_REVERSE);

			mydoupdate(win);

			poffs = offs;
		}

		c = wait_for_keypress();

		if (c == KEY_UP)
		{
			if ((offs + pos) > 0)
			{
				if (offs)
					offs--;
				else
					pos--;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_DOWN)
		{
			if ((pos + offs) < (nlines-1))
			{
				if (offs < (wlines-1))
					offs++;
				else
					pos++;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_NPAGE)
		{
			if ((pos + offs) < (nlines - 1))
			{
				pos += min(wlines, (nfd - 1) - (pos + offs));
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_PPAGE)
		{
			if ((pos + offs - wlines) >= 0)
			{
				if (pos > wlines)
				{
					pos -= wlines;
				}
				else
				{
					pos -= (wlines - offs);
					offs = 0;
				}
			}
			else if (offs > 0)
			{
				offs = 0;
			}
			else if (pos > 0)
			{
				pos = 0;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_ENTER || c == 13 || c == 10)
		{
			sel = pos + offs;
			break;
		}
		else if (toupper(c) == 'Q' || toupper(c) == 'X')
		{
			break;
		}
		else
		{
			wrong_key();
		}
	}

	wclear(win);
	mydoupdate(win);

	free(dummy);

	return sel;
}

int select_window(void)
{
	return selection_box((void **)pi, nfd, SEL_WIN);
}

proginfo * select_subwindow(int f_index)
{
	proginfo *cur;
	char **list;
	int list_n, index, loop;
	
	if (f_index == -1)
		return NULL;

	list_n = create_subwindow_list(f_index, &list);

	index = selection_box((void **)list, list_n, SEL_SUBWIN);

	cur = &pi[f_index];
	for(loop=0; loop<index; loop++)
	{
		cur = cur -> next;
	}

	delete_array(list, list_n);

	return cur;
}

char * select_file(char *input)
{
	char **list = NULL, *path;
	char *new_fname = NULL;
	struct stat statbuf;
	int list_n, index;

	list_n = match_files(input, &path, &list);
	if (list_n == 0)
	{
		flash();
		return NULL;
	}

	index = selection_box((void **)list, list_n, SEL_FILES);
	if (index != -1)
	{
		new_fname = (char *)mymalloc(path_max + 1, "new filename");

		sprintf(new_fname, "%s%s", path, list[index]);

		if (stat(new_fname, &statbuf) == -1)
		{
			free(new_fname);
			new_fname = NULL;
			flash();
		}
		else
		{
			if (S_ISDIR(statbuf.st_mode))
			{
				strcat(new_fname, "/");
			}
		}
	}

	delete_array(list, list_n);

	return new_fname;
}

char * edit_string(WINDOW *win, int y, int win_width, int max_width, char numbers_only, char *input_string)
{
	char *string = (char *)mymalloc(max_width + 1, "edit buffer");
	int str_pos = 0, x = 0;
	int line_width = win_width - 4;

	wattron(win, A_REVERSE);
	mvwprintw(win, y, 1, ">");
	mvwprintw(win, y, win_width - 2, "<");
	wattroff(win, A_REVERSE);

	if (input_string)
	{
		int dummy = max(0, str_pos - line_width);
		strcpy(string, input_string);
		str_pos = dummy;
		mvwprintw(win, y, 2, &string[dummy]);
		x = strlen(string) - dummy;
	}
	else
	{
		string[0] = 0x00;
	}
	wmove(win, y, 2 + x);

	mydoupdate(win);

	for(;;)
	{
		char force_redraw = 0;
		int prev_str_pos = str_pos;
		int c = wait_for_keypress();

		/* confirm */
		if (c == KEY_ENTER || c == 13 || c == 10 )
			break;

		/* abort */
		if (c == 17 || c == 24) /* ^q / ^x */
		{
			string[0] = 0x00;
			break;
		}

		switch(c)
		{
		case 1:			/* ^A */
			str_pos = x = 0;
			break;
		case 5:			/* ^E */
			{
				int dummy = strlen(string);

				if (dummy > line_width)
				{
					str_pos = dummy - (line_width / 2);
					x = (line_width / 2);
				}
				else
				{
					str_pos = 0;
					x = dummy;
				}
			}
			break;
		case 9:			/* tab (filename completion) */
			if (numbers_only)
			{
				wrong_key();
			}
			else
			{
				int dummy = strlen(string);
				char *file = select_file(string);

				if (file)
				{
					strcpy(string, file);
					free(file);
					str_pos = 0;
				}

				dummy = strlen(string);
				if (dummy > line_width)
				{
					str_pos = dummy - (line_width / 2);
					x = (line_width / 2);
				}
				else
				{
					str_pos = 0;
					x = dummy;
				}

				force_redraw = 1;
			}
			break;
		case 23:		/* ^W */
			string[0] = 0x00;
			str_pos = x = 0;
			break;
		case 127:		/* DEL */
		case KEY_BACKSPACE:
			if ((str_pos + x) > 0)
			{
				memmove(&string[str_pos + x - 1], &string[str_pos + x], (max_width - (str_pos + x)) + 1);
				if (x > 0)
				{
					x--;
				}
				else
				{
					str_pos--;
				}

				force_redraw = 1;
			}
			break;
		case KEY_LEFT:
			if (x > 0)
			{
				x--;
			}
			else if (str_pos > 0)
			{
				str_pos--;
			}
			break;
		case KEY_RIGHT:
			if ((x + str_pos) < strlen(string))
			{
				if (x < line_width)
					x++;
				else
					str_pos++;
			}
			else
			{
				wrong_key();
			}
			break;
		default:
			{
				int len = strlen(string);

				/* only allow valid ASCII */
				if (c < 32)
				{
					wrong_key();
					break;
				}

				if (numbers_only && (c < '0' || c > '9'))
				{
					wrong_key();
					break;
				}

				if (len == max_width)
				{
					wrong_key();
					break;
				}

				/* cursor at end of string? */
				if (str_pos == len)
				{
					string[str_pos + x] = c;
					string[str_pos + x + 1] = 0x00;
					waddch(win, c);
				}
				else /* add character to somewhere IN the string */
				{
					memmove(&string[str_pos + x + 1], &string[str_pos + x], strlen(&string[str_pos + x]) + 1);
					string[str_pos + x] = c;
					force_redraw = 1;
				}

				if ((x + str_pos) < max_width)
				{
					if (x < line_width)
						x++;
					else
						str_pos++;
				}
				else
				{
					wrong_key();
				}
			}
			break;
		}

		if (str_pos != prev_str_pos || force_redraw)
		{
			int loop;
			char *dummy = mystrdup(&string[str_pos]);
			dummy[min(strlen(dummy), line_width)] = 0x00;
			for(loop=strlen(dummy); loop<line_width; loop++)
				mvwprintw(win, y, 2 + loop, " ");
			mvwprintw(win, y, 2, dummy);
			free(dummy);
			force_redraw = 0;
		}
		wmove(win, y, 2 + x);
		mydoupdate(win);
	}

	if (strlen(string) == 0)
	{
		free(string);
		string = NULL;
	}

	return string;
}

int find_char_offset(char *str, char what)
{
	int loop, len = strlen(str);

	for(loop=0; loop<len; loop++)
	{
		if (str[loop] == what)
			return loop;
	}

	return -1;
}

void do_color_print(WINDOW *win, proginfo *cur, unsigned char *string, regmatch_t *matches, int matching_regex)
{
	char reverse = 0;
	int loop, nbytes = strlen(string);
	char color = -1;
	int prt_start = 0, prt_end = nbytes;
	int mx = getmaxx(win);
	char use_regex = 0;
	int cmatches[MAX_COLORS_PER_LINE][3];	/* max. 80 colors per line */
	int n_cmatches = 0;

	/* find color */
	if (use_colors && cur -> colorize) /* no need to do this for b/w terminals */
	{
		if (cur -> colorize == 's')
		{
			char *sp = NULL;

			if (nbytes >= 16)
			{
				sp = strchr(&string[16], ' ');
			}

			while(sp && *sp == ' ') sp++;

			if (sp)
			{
				char *end1 = strchr(sp, '[');
				char *end2 = strchr(sp, ' ');
				char *end3 = strchr(sp, ':');
				char *end = NULL;

				end = end1;
				if ((end2 && end2 < end) || (end == NULL))
					end = end2;
				if ((end3 && end3 < end) || (end == NULL))
					end = end3;

				if (end)
					color = gen_color(sp, end);
				else
					color = gen_color(sp, &sp[strlen(sp)]);
			}
			else
			{
				color = 0;
			}
		}
		else if (cur -> colorize == 'f')
		{
			char *sp = string, *dummy = NULL;
			int nlen = strlen(cur -> field_del);

			for(loop=0; loop<cur -> field_nr; loop++)
			{
				sp = strstr(sp, cur -> field_del);
				while(sp)
				{
					if (strncmp(sp, cur -> field_del, nlen) == 0)
					{
						sp += nlen;
					}
					else
					{
						break;
					}
				}

				if (!sp)
					break;
			}

			if (sp)
				dummy = strstr(sp, cur -> field_del);

			if (dummy != NULL && sp != NULL)
				color = gen_color(sp, dummy);
			else if (sp)
				color = gen_color(sp, &sp[strlen(sp)]);
			else
				color = 0;
		}
		else if (cur -> colorize == 'm')
		{
			color = gen_color(string, &string[nbytes]);
		}

		if (cur -> colorize != 'S')
			color_on(win, color);
	}

	/* generate list of offsets for colors */
	if (cur -> colorize == 'S')
	{
		regmatch_t matches[MAX_N_RE_MATCHES];
		int loop;

		for(loop=0; loop<cschemes[cur -> color_scheme].n; loop++)
		{
			int offset = 0;
			char match = 1;

			while(offset < strlen(string) && match)
			{
				match = 0;

				if (regexec(&cschemes[cur -> color_scheme].regex[loop], &string[offset], MAX_N_RE_MATCHES, matches, offset?REG_NOTBOL:0) == 0)
				{
					int loop2;

					match = 1;

					for(loop2=0; loop2<MAX_N_RE_MATCHES; loop2++)
					{
						if (matches[loop2].rm_so == -1)
							break;

						cmatches[n_cmatches][0] = matches[loop2].rm_so + offset;
						cmatches[n_cmatches][1] = matches[loop2].rm_eo + offset;
						cmatches[n_cmatches][2] = cschemes[cur -> color_scheme].color[loop];

						offset = max(offset, matches[loop2].rm_eo + offset);

						if (++n_cmatches == MAX_COLORS_PER_LINE)
							break;
					}
				}

				if (n_cmatches == MAX_COLORS_PER_LINE)
					break;
			}

			if (n_cmatches == MAX_COLORS_PER_LINE)
				break;
		}
	}

	/* figure out what to display (because of linewraps and such) */
	if (cur -> line_wrap == 'a')
	{
		/* default is everything */
	}
	else if (cur -> line_wrap == 'l')
	{
		prt_end = min(nbytes, mx);
	}
	else if (cur -> line_wrap == 'r')
	{
		prt_start = max(0, prt_end - mx);
	}
	else if (cur -> line_wrap == 'S')
	{
		prt_start = find_char_offset(string, ':');
		if (prt_start == -1)
			prt_start = 0;
	}
	else if (cur -> line_wrap == 's')
	{
		prt_start = find_char_offset(&string[16], ' ');
		if (prt_start == -1)
			prt_start = 0;
	}
	else if (cur -> line_wrap == 'o')
	{
		prt_start = min(cur -> line_wrap_offset, nbytes);
	}

	if (matching_regex != -1)
	{
		use_regex = (cur -> pre)[matching_regex].use_regex;
	}

	/* print text */
	for(loop=prt_start; loop<prt_end; loop++)
	{
		char re_inv = 0;

		/* find things to invert */
		if (matches != NULL && (use_regex == 'c' || use_regex == 'C' || use_regex == 'b'))
		{
			int loop2;

			for(loop2=0; loop2<MAX_N_RE_MATCHES; loop2++)
			{
				if (loop >= matches[loop2].rm_so && loop < matches[loop2].rm_eo)
				{
					re_inv = 1;
					break;
				}
			}
		}

		/* find things to colorize */
		if (cur -> colorize == 'S')
		{
			int loop2, new_color = -1;

			for(loop2=0; loop2<n_cmatches; loop2++)
			{
				if (loop >= cmatches[loop2][0] && loop < cmatches[loop2][1])
				{
					new_color = cmatches[loop2][2];
					break;
				}
			}

			if (color != new_color)
			{
				color_off(win, color);
				color = -1;
			}

			if (new_color != -1 && color != new_color)
			{
				color = new_color;
				color_on(win, color);
			}
		}

		if (string[loop] < 32 && string[loop] != 10)
		{
			re_inv = 1;
		}

		if (re_inv)
		{
			if (!reverse)
			{
				wattron(win, A_REVERSE);
				reverse = 1;
			}
		}

		if (string[loop] < 32 && string[loop] != 10)
		{
			waddch(win, '.');
		}
		else
		{
			waddch(win, string[loop]);
		}

		if (reverse)
		{
			wattroff(win, A_REVERSE);
			reverse = 0;
		}
	}

	if (cur -> colorize && color != -1)
	{
		color_off(win, color);
	}
}

char check_filter(proginfo *cur, char *string, regmatch_t **pmatch, char **error, int *matching_regex, char do_bell)
{
	int loop;
	char ok = 1;

	*error = NULL;
	*matching_regex = -1;

	/* do this regular expression? */
	for(loop=0; loop<cur -> n_re; loop++)
	{
		if ((cur -> pre)[loop].use_regex)
		{
			int rc;

			ok = 0;

			if (pmatch)
			{
				*pmatch = (regmatch_t *)mymalloc(sizeof(regmatch_t) * MAX_N_RE_MATCHES, "matching regular expressions");

				rc = regexec(&(cur -> pre)[loop].regex, string, MAX_N_RE_MATCHES, *pmatch, 0);
			}
			else
			{
				rc = regexec(&(cur -> pre)[loop].regex, string, 0, NULL, 0);
			}

			/* matches? */
			if ((rc == 0 && (cur -> pre)[loop].invert_regex == 0) || (rc == REG_NOMATCH && (cur -> pre)[loop].invert_regex == 1) || (cur -> pre)[loop].use_regex == 'C' || (cur -> pre)[loop].use_regex == 'b')
			{
				if (!(rc == 0 && (cur -> pre)[loop].invert_regex == 0))
				{
					if (pmatch)
					{
						free(*pmatch);
						*pmatch = NULL;
					}
				}

				/* sound bell when matches? */
				if (do_bell && rc == 0 && toupper((cur -> pre)[loop].use_regex) == 'B')
					beep();

				*matching_regex = loop;

				return 1;
			}
			/* no match: error actually! */
			else if (rc != REG_NOMATCH && rc != 0)
			{
				char reerror[256];
				regerror(rc, &(cur -> pre)[loop].regex, reerror, sizeof(reerror));

				*error = (char *)mymalloc(512, "regexp error string");

				snprintf(*error, 511, "multitail warning: regex match failed! reason: %s\n", reerror);
			}
		}
	}

	return ok;
}

void color_print(int f_index, proginfo *cur, char *string)
{
	regmatch_t *pmatch = NULL;
	int matching_regex = 0;
	char ok;
	char *error = NULL;
	WINDOW *win = pi[f_index].data;
	if (!win)
	{
		return;
	}

	/* check filter */
	ok = check_filter(cur, string, &pmatch, &error, &matching_regex, 1);
	if (ok)
	{
		do_color_print(win, cur, (unsigned char *)string, pmatch, matching_regex);
	}
	else if (error)
	{
		do_color_print(win, cur, error, NULL, -1);
		free(error);
	}

	/* remember string */
	if (ok || lb[f_index].markset == 'a')
	{
		int curline;

		if (lb[f_index].curpos < lb[f_index].maxnlines || lb[f_index].maxnlines == 0)
		{
			lb[f_index].Blines = (char **)myrealloc(lb[f_index].Blines, sizeof(char *) *
				(lb[f_index].curpos + 1), "lines buffer");

			lb[f_index].pi    = (proginfo **)myrealloc(lb[f_index].pi, sizeof(proginfo *) *
				(lb[f_index].curpos + 1), "lines buffer progrinfo");

			curline = lb[f_index].curpos;
			lb[f_index].curpos++;
		}
		else
		{
			/* delete oldest */
			free((lb[f_index].Blines)[0]);
			curline = lb[f_index].maxnlines-1;
			memmove(&(lb[f_index].Blines)[0], &(lb[f_index].Blines)[1], (lb[f_index].maxnlines-1) * sizeof(char *));
			memmove(&(lb[f_index].pi)[0], &(lb[f_index].pi)[1], (lb[f_index].maxnlines-1) * sizeof(proginfo *));
		}

		(lb[f_index].Blines)[curline] = mystrdup(string);
		(lb[f_index].pi   )[curline] = cur;
	}
}

void create_window_set(int startx, int width, int nwindows, int indexoffset)
{
	int loop;
	int n_window_lines, add_lines;
	int n_not_hidden = 0, win_counter=0;
	int added = 0;

	for(loop=0; loop<nwindows; loop++)
	{
		if (pi[loop + indexoffset].hidden == 0)
			n_not_hidden++;
	}

	if (n_not_hidden == 0)
		return;

	n_window_lines = max_y / n_not_hidden;
	if (n_window_lines < 3)
	{
		error_exit("display too small! (is: %d, must be at least %d lines)\n", max_y, n_not_hidden * 3);
	}
	add_lines = max_y - (n_window_lines * n_not_hidden);

	/* re-create windows */
	for(loop=0; loop<nwindows; loop++)
	{
		char add = 0;
		int cur_index = loop + indexoffset;

		if (pi[cur_index].hidden == 0)
		{
			int cur_win_length;
			int loop2;
			regmatch_t *pmatch = NULL;
			char *error = NULL, ok;

			if (add_lines > 0)
			{
				add = 1;
				add_lines--;
			}

			cur_win_length = n_window_lines - (mode_statusline >= 0?1:0) + add;

			/* creat data window, clear en set scroll ok */
			pi[cur_index].data   = mysubwin(stdscr, cur_win_length, width, win_counter * n_window_lines + added, startx);
			wclear(pi[cur_index].data);

			scrollok(pi[cur_index].data, TRUE);/* supposed to always return OK, according to the manpage */

			/* create status window */
			if (mode_statusline >= 0)
			{
				pi[cur_index].status = mysubwin(stdscr, 1, width, win_counter * n_window_lines + n_window_lines - 1 + add + added, startx);
				wclear(pi[cur_index].status);

				color_on(pi[cur_index].status, 0);

				/* create status-line */
				draw_line(pi[cur_index].status, LINE_BOTTOM);

				wattron(pi[cur_index].status, A_REVERSE);
				mvwprintw(pi[cur_index].status, 0, 0, "%02d] %s", cur_index, pi[cur_index].filename);
				wattroff(pi[cur_index].status, A_REVERSE);

				wnoutrefresh(pi[cur_index].status);
			}

			/* display old strings */
			for(loop2=max(0, lb[cur_index].curpos - cur_win_length); loop2<lb[cur_index].curpos; loop2++)
			{
				if ((lb[cur_index].Blines)[loop2])
				{
					int matching_regex = -1;

					/* check filter */
					ok = check_filter(lb[cur_index].pi[loop2], lb[cur_index].Blines[loop2], &pmatch, &error, &matching_regex, 0);
					if (ok)
					{
						do_color_print(pi[cur_index].data, (lb[cur_index].pi)[loop2], (lb[cur_index].Blines)[loop2], pmatch, matching_regex);
					}
					else if (error)
					{
						do_color_print(pi[cur_index].data, (lb[cur_index].pi)[loop2], error, NULL, -1);
						free(error);
					}
					free(pmatch);
				}
			}
			wnoutrefresh(pi[cur_index].data);

			if (add)
			{
				added++;
			}

			win_counter++;
		}
	}
}

void create_windows(void)
{
	int loop;

	/* close windows */
	for(loop=0; loop<nfd; loop++)
	{
		if (pi[loop].status)
		{
			mydelwin(pi[loop].status);
			pi[loop].status = NULL;
		}
		if (pi[loop].data)
		{
			mydelwin(pi[loop].data);
			pi[loop].data = NULL;
		}
	}

	if (splitline)
	{
		mydelwin(splitline);
		splitline = NULL;
	}

	if (split && nfd > 1)
	{
		int half_nfd = nfd / 2;
		int left_n = nfd - half_nfd;
		int n_not_hidden = 0;

		for(loop=left_n; loop<nfd; loop++)
		{
			if (pi[loop].hidden == 0)
				n_not_hidden++;
		}

		if (n_not_hidden > 0)
		{
			create_window_set(0, max_x/2, left_n, 0);
			create_window_set(max_x/2 + 1, max_x/2 - 1, half_nfd, left_n);

			splitline = mysubwin(stdscr, max_y, 1, 0, max_x/2);

			draw_line(splitline, LINE_LEFT);
			wnoutrefresh(splitline);
		}
		else
		{
			create_window_set(0, max_x, nfd, 0);
		}
	}
	else if (nfd != 0)
	{
		create_window_set(0, max_x, nfd, 0);
	}

	doupdate();
}

void do_resize(int s)
{
        struct winsize size;

        if (ioctl(STDIN_FILENO, TIOCGWINSZ, &size) == 0)
	{
		terminal_changed = 1;

		max_y = size.ws_row;
		max_x = size.ws_col;
	}
	else
	{
		error_exit("ioctl failed\n");
	}

	signal(SIGWINCH, do_resize);
}

int file_size(char *filename, char exit_error)
{
	struct stat buf;

	if (stat(filename, &buf) == -1)
	{
		if (exit_error == 1)
		{
			error_exit("error %d while accessing file %s\n", errno, filename);
		}
	}

	return buf.st_size;
}

void swap_window(void)
{
	WINDOW *win = create_popup(11, 40);

	win_header(win, "Swap windows");

	if (nfd > 1)
	{
		int f1_index, f2_index;

		mvwprintw(win, 2, 2, "Window 1:");
		wnoutrefresh(win);
		f1_index = select_window();
		if (f1_index != -1)
		{
			mvwprintw(win, 3, 2, "%s", pi[f1_index].filename);
			mvwprintw(win, 4, 2, "Window 2:");
			wnoutrefresh(win);

			f2_index = select_window();
			if (f2_index != -1)
			{
				proginfo dummy;
				buffer dummy2;

				buffer_replace_pi_pointers(f1_index, &pi[f1_index], &pi[f2_index]);
				buffer_replace_pi_pointers(f2_index, &pi[f2_index], &pi[f1_index]);

				memmove(&dummy, &pi[f1_index], sizeof(proginfo));
				memmove(&pi[f1_index], &pi[f2_index], sizeof(proginfo));
				memmove(&pi[f2_index], &dummy, sizeof(proginfo));

				memmove(&dummy2, &lb[f1_index], sizeof(buffer));
				memmove(&lb[f1_index], &lb[f2_index], sizeof(buffer));
				memmove(&lb[f2_index], &dummy2, sizeof(buffer));
			}
		}
	}
	else
	{
		mvwprintw(win, 2, 2, "Only 1 window!");
		mvwprintw(win, max_y/2 - 2, 1, "Press any key to exit this screen");
		mydoupdate(win);
		wait_for_keypress();
	}

	mydelwin(win);
}

void delete_window(void)
{
	int f_index = 0;
	WINDOW *win = create_popup(7, 24);

	win_header(win, "Delete window");
	wnoutrefresh(win);

	/* only popup selectionbox when more then one window */
	if (nfd > 1)
		f_index = select_window();

	/* user did not press q/x? */
	if (f_index != -1)
	{
		char all = 1;

		/* multiple files for this window? then ask if delete all
		 * or just one
		 */
		if (pi[f_index].next)
		{
			escape_print(win, 3, 2, "Delete all? (^y^/^n^)");
			mydoupdate(win);

			if (ask_yes_no() == 0)
				all = 0;
		}

		if (all)
		{
			delete_entry(f_index, NULL);
		}
		else
		{
			proginfo *cur = select_subwindow(f_index);

			if (cur)
				delete_entry(f_index, cur);
		}
	}

	mydelwin(win);
}

void hide_window(void)
{
	int f_index = -1;
	WINDOW *win = create_popup(11, 40);

	win_header(win, "Hide/unhide window");
	wnoutrefresh(win);

	f_index = select_window();
	if (f_index != -1)
	{
		pi[f_index].hidden = 1 - pi[f_index].hidden;
	}
}

void info(void)
{
	WINDOW *win = create_popup(13, 40);
	char *dummy;

	mvwprintw(win, 1, 2, "-=* multitail *=-");
	mvwprintw(win, 3, 2, "Written by folkert@vanheusden.com");
	mvwprintw(win, 4, 2, "Website:");
	mvwprintw(win, 5, 2, "http://www.vanheusden.com/multitail/");

#if defined(__FreeBSD__) || defined(linux) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__APPLE__) || defined(sun)
	mvwprintw(win, 7, 2, "Current load of system:");
	dummy = get_load();
	mvwprintw(win, 8, 2, "%s", dummy);
	free(dummy);
#endif
	if (!use_colors)
	{
		mvwprintw(win, 9, 2, "Your terminal doesn't support colors");
	}

	mvwprintw(win, 11, 1, "Press any key to exit this screen");

	mydoupdate(win);

	wait_for_keypress();

	mydelwin(win);
}

char ask_negate_regexp(WINDOW *win, int line)
{
	escape_print(win, line, 2, "Negate regular expression? (^y^/^n^)");
	mydoupdate(win);

	return ask_yes_no();
}

char ask_colors(WINDOW *win, int line, char cur, char *fieldnr, char **fielddel, int *cscheme_index)
{
	char ok = 0;

	escape_print(win, line, 1, "Colors? (^s^yslog/^m^isc/^f^ield/^n^one,^S^cheme)");

	mydoupdate(win);
	for(;;)
	{
		int c = wait_for_keypress();

		switch(c)
		{
		case 'S':
			{
				int dummy = selection_box((void *)cschemes, n_cschemes, SEL_CSCHEME);
				if (dummy != -1)
				{
					*cscheme_index = dummy;
					return 'S';
				}
			}

		case 's':
		case 'm':
			return 'm';

		case 'N':
			return 0;

		case 'f':
			{
				int loop, fld_dummy;
				char *dummy, nr[5]; /* xxx\0 and a little :-) */

				sprintf(nr, "%d", *fieldnr);

				mvwprintw(win, line + 2, 1, "Enter field number: ");
				dummy = edit_string(win, line + 3, 42, 39, 1, nr);
				if (dummy)
				{
					fld_dummy = atoi(dummy);
					free(dummy);

					mvwprintw(win, line + 4, 1, "Enter delimiter (eg a space):");
					dummy = edit_string(win, line + 5, 42, 39, 0, *fielddel);

					if (dummy)
					{
						free(*fielddel);
						*fielddel = dummy;
						*fieldnr = fld_dummy;
						ok = 1;
					}
				}

				for(loop=0; loop<4; loop++)
					mvwprintw(win, line + 1 + loop, 1, "                              ");
			}

			if (ok)
				return 'f';
			break;

		case 13:
			if (cur != -1)
				return cur;
		}

		wrong_key();
	}
}

char ask_regexp_type(WINDOW *win, int line)
{
	escape_print(win, line+0, 2, "Usage of regexp? (^m^atch, match +");
	escape_print(win, line+1, 2, "^c^olor, ^C^olor, ^B^ell, ^b^ell + colorize)");
	mydoupdate(win);

	for(;;)
	{
		int c = wait_for_keypress();

		if (c == 'm' || toupper(c) == 'C' || toupper(c) == 'B')
			return c;

		wrong_key();
	}
}

void will_not_fit(void)
{
	WINDOW *win = create_popup(3, 40);

	color_on(win, 1);
	box(win, 0, 0);
	mvwprintw(win, 1, 2, "I'm sorry, that won't fit!");
	color_off(win, 1);

	mydoupdate(win);
	wrong_key();
	wait_for_keypress();

	mydelwin(win);
}

void add_window(void)
{
	char *fname, *expr;
	WINDOW *win = create_popup(11, 45);
	int col, fc;
	proginfo *cur = NULL;
	char ask_add_to;

	if (nfd)
		ask_add_to = 1;
	else
		ask_add_to = 0;

	for(;;)
	{
		int cscheme = -1;
		char field_nr = 0;
		char *field_del = NULL;
		char invert_regex = 0;

		win_header(win, "Add window");

		if (ask_add_to)
		{
			ask_add_to = 0;
			mvwprintw(win, 2, 2, "Add (merge) to existing window?");
			mydoupdate(win);

			if (ask_yes_no())
			{
				int index = 0;

				if (nfd > 1)
					index = select_window();

				if (index == -1)
					break;

				cur = &pi[index];
			}
		}

		/* check if this extra window will still fit */
		if (!cur)
		{
			if ((max_y / (nfd + 1)) < 3)
			{
				will_not_fit();
				break;
			}
		}

		escape_print(win, 2, 2, "File or command? (^f^/^c^)         ");
		mydoupdate(win);
		for(;;)
		{
			fc = toupper(wait_for_keypress());

			if (fc == 'F' || fc == 'C' || fc == 'Q')
				break;

			wrong_key();
		}

		if (fc == 'Q')
			break;
		else if (fc == 'F')
			mvwprintw(win, 2, 2, "Enter filename:       ");
		else
			mvwprintw(win, 2, 2, "Enter command:        ");

		fname = edit_string(win, 3, 41, path_max, 0, NULL);
		if (!fname)
			break;

		col = ask_colors(win, 4, -1, &field_nr, &field_del, &cscheme);

		if (cur == NULL)
		{
			pi = (proginfo *)myrealloc(pi, (nfd + 1) * sizeof(proginfo), "proginfo list");

			lb = (buffer *)myrealloc(lb, (nfd + 1) * sizeof(buffer), "proginfo list");

			memset(&lb[nfd], 0x00, sizeof(buffer));
			lb[nfd].maxnlines = min_n_bufferlines;
			lb[nfd].markset = 'm';

			cur = &pi[nfd++];
		}
		else
		{
			/* skip to end of chain */
			while (cur -> next)
				cur = cur -> next;

			/* allocate new entry */
			cur -> next = (proginfo *)mymalloc(sizeof(proginfo), "proginfo");
			/* and set pointer to the newly allocated entry */
			cur = cur -> next;
		}
		memset(cur, 0x00, sizeof(proginfo));

		cur -> restart = -1;
		if (fc == 'C')
		{
			char *dummy;

			mvwprintw(win, 5, 2, "Repeat command interval: (empty for don't)");
			dummy = edit_string(win, 6, 10, 10, 1, NULL);
			if (dummy)
			{
				cur -> restart = atoi(dummy);
				free(dummy);
			}
		}
		cur -> filename = fname;
		cur -> status = cur -> data = NULL;
		cur -> is_command = fc == 'F' ? 0 : 1;
		cur -> retry_open = cur -> follow_filename = 0;
		cur -> next = NULL;
		cur -> first = 1;
		cur -> line_wrap = 'a';

		cur -> colorize = col;
		cur -> field_nr = field_nr;
		cur -> field_del = field_del;
		cur -> color_scheme = cscheme;

		/* start tail-process */
		if (start_tail(cur, max_y / (nfd + 1)) == -1)
		{
			mvwprintw(win, 7, 2, "error opening file!");
			mydoupdate(win);
			wait_for_keypress();
			nfd--;
			break;
		}

		mvwprintw(win, 9, 2, "Add another to this new window?");
		mydoupdate(win);
		if (ask_yes_no() == 0)
			break;

		wclear(win);
		box(win, 0, 0);
	}

	mydelwin(win);
}

void toggle_colors(void)
{
	WINDOW *win = create_popup(11, 42);
	int f_index = 0;

	win_header(win, "Toggle colors");
	wnoutrefresh(win);

	if (nfd > 1)
		f_index = select_window();

	if (f_index != -1)
	{
		char *dummy = mystrdup(pi[f_index].filename);
		dummy[min(strlen(dummy), 40)] = 0x00;
		mvwprintw(win, 3, 1, dummy);
		free(dummy);

		pi[f_index].colorize = ask_colors(win, 4, pi[f_index].colorize, &pi[f_index].field_nr, &pi[f_index].field_del, &pi[f_index].color_scheme);
	}

	mydelwin(win);
}

char zerotomin(char c)
{
	if (c == 0)
		return 'n';
	if (c == 1)
		return 'y';

	return c;
}

void enter_regexp(void)
{
	WINDOW *win;
	proginfo *cur;
	int f_index = 0;
	int cur_re = 0;

	/* select window */
	if (nfd > 1)
		f_index = select_window();

	if (f_index == -1)	/* user pressed Q/X */
		return;

	/* select subwindow */
	cur = &pi[f_index];
	if (cur -> next)
		cur = select_subwindow(f_index);
	if (!cur)
		return;

	/* create window */
	win = create_popup(21, 40);
	win_header(win, "Edit reg.exp.");
	mvwprintw(win, 2, 2, "%s", cur -> filename);
	escape_print(win, 3, 2, "^A^dd, ^E^dit, ^D^elete, e^X^it");

	for(;;)
	{
		int c, key;
		int loop;
		char buffer[38 + 1];
		buffer[38] = 0x00;

		/* clear */
		memset(buffer, ' ', 38);
		for(loop=4; loop<20; loop++)
			mvwprintw(win, loop, 1, buffer);

		/* display them lines */
		for(loop=0; loop<cur -> n_re; loop++)
		{
			strncpy(buffer, (cur -> pre)[loop].regex_str, 34);
			if (loop == cur_re)
				wattron(win, A_REVERSE);
			mvwprintw(win, 4 + loop, 1, "%c%c %s", 
					zerotomin((cur -> pre)[loop].invert_regex),
					zerotomin((cur -> pre)[loop].use_regex),
					buffer);
			if (loop == cur_re)
				wattroff(win, A_REVERSE);
		}
		mydoupdate(win);

		/* wait for key */
		for(;;)
		{
			key = wait_for_keypress();
			c = toupper(key);

			/* convert return to 'E'dit */
			if (key == 13)
				key = c = 'E';

			/* any valid keys? */
			if (c == 'Q' || c == 'X' || c == 'A' || c == 'E' || c == 'D' || key == KEY_DOWN || key == KEY_UP)
				break;

			wrong_key();
		}
		/* exit this screen? */
		if (c == 'Q' || c == 'X')
			break;

		if (key == KEY_UP)
		{
			if (cur_re > 0)
				cur_re--;
			else
				wrong_key();
		}
		else if (key == KEY_DOWN)
		{
			if (cur_re < (cur -> n_re -1))
				cur_re++;
			else
				wrong_key();
		}

		/* add or edit */
		if (c == 'A' || c == 'E')
		{
			char *str;

			/* max. 10 regular expressions */
			if (c == 'A' && cur -> n_re == 10)
			{
				wrong_key();
				continue;
			}
			if (c == 'E' && cur -> n_re == 0)
			{
				wrong_key();
				continue;
			}

			/* ask new reg exp */
			mvwprintw(win, 15, 2, "Edit regular expression:");
			if (c == 'E')
				str = edit_string(win, 16, 38, 128, 0, (cur -> pre)[cur_re].regex_str);
			else
				str = edit_string(win, 16, 38, 128, 0, NULL);

			/* user did not abort edit? */
			if (str == NULL)
				continue;

			/* edit: free previous */
			if (c == 'E')
			{
				regfree(&(cur -> pre)[cur_re].regex);
				free((cur -> pre)[cur_re].regex_str);
			}
			/* add: allocate new */
			else
			{
				cur_re = cur -> n_re++;
				cur -> pre = (re *)myrealloc(cur -> pre, cur -> n_re * sizeof(re), "list of regular expressions");
				memset(&(cur -> pre)[cur_re], 0x00, sizeof(re));
			}

			/* wether to negate this expression or not */
			(cur -> pre)[cur_re].invert_regex = ask_negate_regexp(win, 17);

			/* compile */
			if (regcomp(&(cur -> pre)[cur_re].regex, str, REG_EXTENDED))
			{
				mvwprintw(win, 17, 2, "Failed to compile regular expression!");
				mvwprintw(win, 18, 2, "Press any key...");
				mydoupdate(win);
				wait_for_keypress();
				free(str);

				if (c == 'A')
				{
					cur -> n_re--;
					if (cur -> n_re == cur_re)
						cur_re--;
				}
			}
			else
			{
				/* compilation went will, remember everything */
				(cur -> pre)[cur_re].use_regex = ask_regexp_type(win, 18);
				(cur -> pre)[cur_re].regex_str = str;
			}
		}

		/* delete entry */
		if (c == 'D')
		{
			/* delete entry */
			free_re(&(cur -> pre)[cur_re]);

			/* update administration */
			if (cur -> n_re == 1)
			{
				free(cur -> pre);
				cur -> pre = NULL;
			}
			else
			{
				int n_to_move = (cur -> n_re - cur_re) - 1;

				if (n_to_move > 0)
					memmove(&(cur -> pre)[cur_re], &(cur -> pre)[cur_re+1], sizeof(re) * n_to_move);
			}
			cur -> n_re--;

			/* move cursor */
			if (cur_re > 0 && cur_re == cur -> n_re)
			{
				cur_re--;
			}
		}
	}

	mydelwin(win);
}

void toggle_vertical_split(void)
{
	if (split)
	{
		if ((max_y / nfd) < 3)
		{
			will_not_fit();
		}
		else
			split = 0;
	}
	else
		split = 1;
}

void show_help(void)
{
	WINDOW *win = create_popup(15, 40);

	escape_print(win, 1, 2, "^x^/^q^    exit");
	escape_print(win, 2, 2, "^r^      redraw the screen");
	escape_print(win, 3, 2, "^e^      enter/edit regular expression");
	escape_print(win, 4, 2, "^d^      delete window");
	escape_print(win, 5, 2, "^a^      add window");
	escape_print(win, 6, 2, "^s^      swap windows");
	escape_print(win, 7, 2, "^v^      toggle vertical split");
	escape_print(win, 8, 2, "^w^      write startupscript");
	escape_print(win, 9, 2, "^z^      hide/unhide window");
	escape_print(win, 10, 2, "^m^      set mark  ^n^ delete mark");
	escape_print(win, 11, 2, "^b^      scrollback");
	escape_print(win, 12, 2, "^i^      info      ^h^ this screen");
	mvwprintw(win, 13, 1, "Press any key to exit this screen");
	mydoupdate(win);

	wait_for_keypress();

	mydelwin(win);
}

void do_set_mark(int f_index, char store_what_lines, int maxnlines)
{
	lb[f_index].markset = store_what_lines;
	lb[f_index].maxnlines = maxnlines;

	delete_array(lb[f_index].Blines, lb[f_index].curpos);
	lb[f_index].Blines = NULL;

	lb[f_index].curpos = 0;
}

int find_string(int f_index, char *find, int offset)
{
	int loop, index = -1;
        regex_t regex;

	/* compile the searchstring (which can be a regular expression) */
	if (regcomp(&regex, find, REG_EXTENDED))
		return -1;	/* failed -> not found */

	for(loop=offset; loop<lb[f_index].curpos; loop++)
	{
		if (regexec(&regex, lb[f_index].Blines[loop], 0, NULL, 0) == 0)
		{
			index = loop;
			break;
		}
	}

       	regfree(&regex);

	return index;
}

void scrollback_help(void)
{
	WINDOW *win = create_popup(9, 44);

	escape_print(win, 1, 2, "^x^/^q^    exit");
	escape_print(win, 2, 2, "^f^/^/^    search for string");
	escape_print(win, 3, 2, "^n^      repeat search");
	escape_print(win, 4, 2, "^e^      edit filter (regular expression)");
	escape_print(win, 5, 2, "^c^      set colors");
	escape_print(win, 6, 2, "^s^      save to file");
	mydoupdate(win);

	wait_for_keypress();

	mydelwin(win);
}

void scrollback_savefile(int window)
{
	char *file = NULL;
	char sel;
	WINDOW *win = create_popup(8, 40);

	win_header(win, "Save buffer to file");

	if (lb[window].markset == 'a')
	{
		escape_print(win, 3, 2, "Write all (^a^) or filtered (^m^)?");
		mydoupdate(win);

		for(;;)
		{
			sel = tolower(wait_for_keypress());

			if (sel == 'a' || sel == 'm' || sel == 'q')
				break;

			wrong_key();
		}
	}
	else
	{
		sel = 'a'; /* no need to filter again, it is already filtered */
	}

	if (sel != 'q')
	{
		mvwprintw(win, 4, 2, "Select file");
		file = edit_string(win, 5, 40, path_max, 0, NULL);
	}
	if (file)
	{
		FILE *fh = fopen(file, "w");
		if (fh)
		{
			int loop;

			for(loop=0; loop<lb[window].curpos; loop++)
			{
				if (lb[window].Blines[loop])
				{
					char *error;
					int dummy = -1;

					/* check filter */
					if (sel == 'a' || check_filter(lb[window].pi[loop], lb[window].Blines[loop], NULL, &error, &dummy, 0))
					{
						fprintf(fh, "%s", lb[window].Blines[loop]);
					}
					else if (error)
					{
						fprintf(fh, "---> Error: %s\n", error);
						free(error);
					}
				}
			}

			fclose(fh);
		}
		else
		{
			mvwprintw(win, 6, 2, "Cannot write to file!");
			mydoupdate(win);

			wait_for_keypress();
		}
	}

	mydelwin(win);
}

void scrollback(void)
{
	int window = 0;
	WINDOW *win1 = create_popup(7, 22);

	win_header(win1, "Scrollback");

	if (nfd > 1)
	{
		window = select_window();
	}

	if (window != -1)
	{
		if (lb[window].markset == 0)
		{
			mvwprintw(win1, 3, 2, "Cannot scrollback:");
			mvwprintw(win1, 4, 2, "no mark was set!");
			mydoupdate(win1);
			wait_for_keypress();
		}
	}

	mydelwin(win1);

	if (window != -1 && lb[window].markset != 0)
	{
		char *find = NULL;
		WINDOW *win2;
		int nlines = max_y - 6, ncols = max_x - 6;
		int offset = max(0, lb[window].curpos - nlines), poffset=-1;

		win1 = create_popup(max_y - 4, max_x - 4);
		wattron(win1, A_REVERSE);
		mvwprintw(win1, max_y - 5, 1, "%02d] %s - %d buffered lines", window, pi[window].filename, lb[window].curpos);
		wattroff(win1, A_REVERSE);
		wnoutrefresh(win1);

		win2 = create_popup(nlines, ncols);
		scrollok(win2, FALSE); /* supposed to always return OK, according to the manpage */

		for(;;)
		{
			int c;

			if (offset != poffset)
			{
				int loop, nlinesdisplayed = 0;

				poffset = offset;

				wclear(win2);
				for(loop=0; loop<lb[window].curpos; loop++)
				{
					int takes;
					int line = offset + loop;

					if (line >= lb[window].curpos)
						break;

					/* anything in buffer? or empty line? */
					if (lb[window].Blines[line])
					{
						char ok;
						regmatch_t *pmatch = NULL;
						char *error = NULL;
						int matching_regex = -1;

						/* calculate the number of lines this string takes */
						takes = (strlen((lb[window].Blines)[line]) + (ncols - 1)) / ncols;

						/* do not print more then what fits into the window */
						if (nlinesdisplayed + takes > nlines)
						{
							do_color_print(win2, lb[window].pi[line], "...", NULL, -1);
							break;
						}

						/* check filter */
						ok = check_filter(lb[window].pi[line], lb[window].Blines[line], &pmatch, &error, &matching_regex, 0);
						if (ok)
						{
							do_color_print(win2, lb[window].pi[line], lb[window].Blines[line], pmatch, matching_regex);
						}
						else if (error)
						{
							do_color_print(win2, lb[window].pi[line], error, NULL, -1);
							free(error);
						}
						/* did not match? do not print a thing */
						if (!ok)
						{
							takes = 0;
						}

						free(pmatch);
					}
					else
					{
						wprintw(win2, "\n");
						takes = 1;
					}
					nlinesdisplayed += takes;

					if (nlinesdisplayed >= nlines)
						break;
				}

				mydoupdate(win2);
			}

			c = wait_for_keypress();

			if (toupper(c) == 'Q' || toupper(c) == 'X')
				break;
			else if (toupper(c) == 'E')
			{
				poffset = -1;
				enter_regexp();
			}
			else if (c == KEY_UP)
			{
				if (offset > 0)
					offset--;
				else
				{
					wrong_key();
				}
			}
			else if (c == KEY_DOWN)
			{
				if (offset < (lb[window].curpos - 1))
					offset++;
				else
				{
					wrong_key();
				}
			}
			else if (c == KEY_NPAGE)
			{
				if (offset < (lb[window].curpos - 1))
					offset = min(lb[window].curpos - 1, offset + nlines);
				else
					wrong_key();
			}
			else if (c == KEY_PPAGE)
			{
				if (offset)
					offset = max(0, offset - nlines);
				else
					wrong_key();
			}
			else if (c == KEY_HOME)
			{
				if (offset)
					offset = 0;
				else
					wrong_key();
			}
#ifdef N_CURSES
			else if (c == KEY_END)
			{
				if (offset < (lb[window].curpos - 1))
					offset = lb[window].curpos - 1;
				else
					wrong_key();
			}
#endif
			else if (toupper(c) == 'F' || c == '/')
			{
				char *dummy;
				int new_f_index;
				WINDOW *win = create_popup(5, 40);

				win_header(win, "Find");

				dummy = edit_string(win, 3, 40, 80, 0, find);
				free(find);
				find = dummy;

				poffset = -1; /* force redraw */

				new_f_index = find_string(window, find, 0);
				if (new_f_index == -1)
					wrong_key();
				else
					offset = new_f_index;
			}
			else if (toupper(c) == 'N')
			{
				if (find)
				{
					int new_f_index = find_string(window, find, offset + 1);
					if (new_f_index == -1)
						wrong_key();
					else
						offset = new_f_index;
				}
				else
					wrong_key();
			}
			else if (toupper(c) == 'S')
			{
				scrollback_savefile(window);
				poffset = -1;	/* force redraw */
			}
			else if (toupper(c) == 'H')
			{
				scrollback_help();
				poffset = -1;	/* force redraw */
			}
			else if (toupper(c) == 'C')
			{
				toggle_colors();
				poffset = -1;	/* force redraw */
			}
			else
			{
				wrong_key();
			}
		}

		mydelwin(win2);
		mydelwin(win1);

		free(find);
	}
}

void delete_mark(void)
{
	int window = 0;
	WINDOW *win = create_popup(13, 40);

	win_header(win, "Delete mark");

	if (nfd > 1)
	{
		window = select_window();
	}

	if (window != -1)
	{
		if (lb[window].markset == 0)
		{
			mvwprintw(win, 3, 2, "No mark was set!");
			mydoupdate(win);
			wait_for_keypress();
		}
		else
		{
			lb[window].markset = 0;
			lb[window].maxnlines = min_n_bufferlines;

			delete_array(lb[window].Blines, lb[window].curpos);
			lb[window].Blines = NULL;

			lb[window].curpos = 0;
		}
	}

	mydelwin(win);
}

void do_pause(void)
{
	WINDOW *win = create_popup(3, 8);

	color_on(win, 1);
	box(win, 0, 0);
	mvwprintw(win, 1, 1, "Paused");
	color_off(win, 1);

	wmove(win, 1, 2);
	mydoupdate(win);

	wait_for_keypress();
}

void set_mark(void)
{
	char winchoice, whatlines, maxnlines = 0;
	WINDOW *win = create_popup(13, 40);

	win_header(win, "Set mark");

	if (nfd > 1)
	{
		escape_print(win, 3, 2, "Set on ^a^ll or ^o^ne");
		mvwprintw(win, 4, 2, "window(s)?");
		mydoupdate(win);

		for(;;)
		{
			winchoice = toupper(wait_for_keypress());

			if (winchoice == 'Q' || winchoice == 'X')
			{
				winchoice = 'Q';
				break;
			}

			if (winchoice == 'A' || winchoice == 'O')
				break;

			wrong_key();
		}
	}
	else
	{
		winchoice = 'A';
	}

	/* ask wether to store all lines or just the ones matchine to the
	 * regular expression (if any)
	 */
	if (winchoice != 'Q')
	{
		escape_print(win, 5, 2, "Store ^a^ll or ^m^atching regular");
		mvwprintw(win, 6, 2, "expressions (if any)?");
		mydoupdate(win);
		for(;;)
		{
			whatlines = toupper(wait_for_keypress());
			if (whatlines == 'Q' || whatlines == 'X')
			{
				winchoice = 'Q';
				break;
			}

			if (whatlines == 'A' || whatlines == 'M')
				break;

			wrong_key();
		}
	}

	/* get number of lines to store */
	if (winchoice != 'Q')
	{
		char *dummy;

		mvwprintw(win, 7, 2, "Store how many lines? (0 for all)");

		dummy = edit_string(win, 8, 40, 7, 1, NULL);

		if (!dummy)
		{
			winchoice = 'Q';
		}
		else
		{
			maxnlines = atoi(dummy);

			free(dummy);
		}
	}

	/* do set mark */
	if (winchoice != 'Q')
	{
		if (winchoice == 'A')
		{
			int loop;

			for(loop=0; loop<nfd; loop++)
			{
				do_set_mark(loop, whatlines == 'A' ? 'a' : 'm', maxnlines);
			}
		}
		else
		{
			int window = select_window();

			if (window != -1)
			{
				do_set_mark(window, whatlines == 'A' ? 'a' : 'm', maxnlines);
			}
		}
	}

	mydelwin(win);
}

void update_statusline(WINDOW *status, proginfo *cur)
{
	if (mode_statusline > 0)
	{
		char *dummy;
		time_t now;
		int dx, dy;

		time(&now);
		dummy = ctime(&now);

		getmaxyx(status, dy, dx);
		if (dx > (strlen(cur -> filename) + strlen(dummy) + 11))
		{
			wattron(status, A_REVERSE);

			if (cur -> is_command)
			{
				mvwprintw(status, 0, dx - strlen(dummy), "%s", dummy);
			}
			else
			{
				char exit_on_error = 0;
	
				if (cur -> retry_open || cur -> follow_filename)
					exit_on_error = 1;

				mvwprintw(status, 0, dx - strlen(dummy) - 11, "%8d - %s", file_size(cur -> filename, exit_on_error), dummy);
			}

			wattroff(status, A_REVERSE);

			wnoutrefresh(status);
		}
	}
}

void generate_diff(int winnr, proginfo *cur)
{
	int loop;

	/* calculate & print difference */
	if (cur -> nprev == 0 && cur -> ncur > 0)
	{
		for(loop=0; loop<cur -> ncur; loop++)
		{
			color_print(winnr, cur, cur -> bcur[loop]);
		}
	}
	else if (cur -> nprev > 0 && cur -> ncur > 0)
	{
		int curprev = 0;

		for(loop=0; loop<cur -> ncur; loop++)
		{
			int indexfound = -1, loop2;

			for(loop2=curprev; loop2<cur -> nprev; loop2++)
			{
				if (strcmp((cur -> bprev)[loop2], (cur -> bcur)[loop]) == 0)
				{
					indexfound = loop2;
					break;
				}
			}

			if (indexfound == -1)
			{
				// output cur[loop]
				color_print(winnr, cur, (cur -> bcur)[loop]);
			}
			else
			{
				curprev = indexfound + 1;
			}
		}
	}
	wnoutrefresh(pi[winnr].data);

	/* free up previous */
	delete_array(cur -> bprev, cur -> nprev);

	/* remember current */
	cur -> bprev = cur -> bcur;
	cur -> nprev = cur -> ncur;
	cur -> bcur = NULL;
	cur -> ncur = 0;

	/* update statusline */
	update_statusline(pi[winnr].status, cur);
}

char close_window(int winnr, proginfo *cur)
{
	/* make sure it is really gone */
	stop_process(cur -> pid);

	/* wait for the last remainder of the exited process to go away,
	 * otherwhise we'll find zombies on our way
	 */
	if (waitpid(cur -> pid, NULL, WNOHANG | WUNTRACED) == -1)
	{
		if (errno != ECHILD)
			error_exit("waitpid failed\n");
	}

	/* restart window? */
	if (cur -> restart >= 0)
	{
		/* close old fd */
		if (close(cur -> fd) == -1) error_exit("closing filedescriptor failed\n");

		/* do diff */
		if (cur -> do_diff)
			generate_diff(winnr, cur);

                /* restart tail-process */
                if (start_tail(cur, max_y / (nfd + 1)) != -1)
			return -1;

		/* ...and if that fails, go on with the closing */
	}

	if (warn_closed)
	{
		char buffer[128];
		WINDOW *win;

		sprintf(buffer, " Window %d closed ", winnr);
		win = mysubwin(stdscr, 3, strlen(buffer)+4, max_y/2, ((max_x/2) - (strlen(buffer)/2)) + 3);

		color_on(win, 1);
		box(win, 0, 0);
		mvwprintw(win, 1, 2, "%s", buffer);
		color_off(win, 1);

		wmove(win, 1, 2);
		mydoupdate(win);

		wait_for_keypress();
	}

	/* file no longer exists (or so) */
	return delete_entry(winnr, cur);
}

void usage(void)
{
	int index = 0;

	printf("%s\n\n", version);
	printf("multitail [-cs|-Cs|-c-] [-i] inputfile [-i anotherinputfile] [...]\n");
	printf("-l	parameter is a command to be executed\n");
	printf("-L	see -l but add to the previous window\n");
	printf("-r interval\n");
	printf("	restart the command when it exited after `interval' seconds\n");
	printf("-R interval\n");
	printf("	same as -r, only with this one only the difference is displayed\n");
	printf("-i	the following parameter is a filename (in case it starts with a dash)\n");
	printf("-I	see -i but add to the previous window\n");
	printf("-s	vertical split screen\n");
	printf("-f	follow the following filename, not the descriptor\n");
	printf("--retry	keep trying to open the following file if it is inaccessible\n");
	printf("-e	use regular expression on following file\n");
	printf("-ec	use regular expression but display the matches inverted on following file\n");
	printf("-eC	use regexp, display everything but matches inverted on following file\n");
	printf("-E	use regular expression on following files\n");
	printf("-Ec	use regular expression but display the matches inverted on following files\n");
	printf("-EC	use regexp, display everything but matches inverted on following files\n");
	printf("-v      invert next regular expression\n");
	printf("-cs	colorize current with syslog-scheme\n");
	printf("-c	colorize current\n");
	printf("-cs scheme\n");
	printf("	use colorscheme 'scheme' (as defined in multitail.conf)\n");
	printf("-Cs	colorize all following files with syslog-scheme\n");
	printf("-C	colorize all following files\n");
	printf("-Cf/-cf field delimiter\n");
	printf("        colorize next/all file(s) depending on the given field number. fields\n");
	printf("        are delimited with the given field-delimiter\n");
	printf("-c-	do NOT colorize the following file\n");
	printf("-C-	do NOT colorize the following files\n");
	printf("-d	do NOT update the status-line\n");
	printf("-D	do not display a status-line at all\n");
	printf("-z	do not \"window closed\" windows\n");
	printf("-w	do not use colors\n");
	printf("-m nlines\n");
	printf("	immediately set mark\n");
	printf("-u	set update interval (for slow links)\n");
	printf("-H interval\n");
	printf("	visual heart-beat (usefull when you want to keep your session alive)\n");
	printf("-p x [y]\n");
	printf("	set linewrap (l=left/a=all/r=right/s=syslog,S=syslog w/o procname, o=offset -> 'y')\n");
	printf("-h	this help\n");
	printf("\n");
	printf("You can have multiple regular expressions per file/command. Be warned: if\n");
	printf("you define multiple and one of them is specified with '-E' (=for every\n");
	printf("following file), _all_ of the current regular expressions are for all\n");
	printf("following files!\n");
	printf("\n");

	do
	{
		printf("%s\n", keys[index++]);
	} while (keys[index]);
}

void write_escape_str(FILE *fh, char *string)
{
	int loop, len = strlen(string);

	fprintf(fh, "\"");
	for(loop=0; loop<len; loop++)
	{
		if (string[loop] == '\"')
			fprintf(fh, "\\");
		fprintf(fh, "%c", string[loop]);
	}
	fprintf(fh, "\"");
}

void write_script(void)
{
	char *fname;
	WINDOW *win = create_popup(12, 42);
	if (!win)
		error_exit("cannot create window!\n");

	win_header(win, "Write script");

	mvwprintw(win, 2, 2, "Enter filename:");

	fname = edit_string(win, 3, 41, path_max, 0, NULL);
	if (fname)
	{
		FILE *fh = fopen(fname, "w");
		if (fh)
		{
			int loop;

			fprintf(fh, "#!/bin/sh\n\n");

			fprintf(fh, "multitail");

			if (heartbeat)
				fprintf(fh, " -H %d", heartbeat);

			for(loop=0; loop<nfd; loop++)
			{
				proginfo *cur = &pi[loop];
				char first = 1;

				do
				{
					if (cur -> line_wrap != 'a')
					{
						fprintf(fh, " -p %c", cur -> line_wrap);
					}
#if 0
					if (cur -> regex_str)
					{
						if (cur -> invert_regex)
							fprintf(fh, " -v");
						if (cur -> use_regex == 'm')
							fprintf(fh, " -e ");
						else if (cur -> use_regex)
							fprintf(fh, " -e%c ", cur -> use_regex);
						write_escape_str(fh, cur -> regex_str);
					}
#endif

					if (cur -> colorize)
					{
						if (cur -> colorize == 's')
							fprintf(fh, " -cs");
						else if (cur -> colorize == 'm')
							fprintf(fh, " -c");
						else if (cur -> colorize == 'f')
						{
							fprintf(fh, " -cf %d ", cur -> field_nr);
							write_escape_str(fh, cur -> field_del);
						}
					}

					if (first)
					{
						first = 0;

						if (cur -> is_command)
						{
							if (cur -> restart != -1)
								fprintf(fh, " -%c %d", cur -> do_diff?'R':'r', cur -> restart);

							fprintf(fh, " -l ");
						}
						else if ((cur -> filename)[0] == '-')
							fprintf(fh, " -i ");
						else
							fprintf(fh, " ");
					}
					else
					{
						if (cur -> is_command)
							fprintf(fh, " -L ");
						else
							fprintf(fh, " -I ");
					}

					write_escape_str(fh, cur -> filename);

					cur = cur -> next;
				}
				while(cur);
			}
			if (split)
				fprintf(fh, " -s");

			if (fchmod(fileno(fh), S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) == -1)
			{
				error_exit("error setting mode-bits on file");
			}

			fprintf(fh, "\n");

			fclose(fh);
		}
		else
		{
			mvwprintw(win, 4, 2, "Error creating file!");
			mydoupdate(win);

			wait_for_keypress();
		}

		free(fname);
	}

	mydelwin(win);
}

int find_colorscheme(char *name)
{
	int loop;

	for(loop=0; loop<n_cschemes; loop++)
	{
		if (strcasecmp(cschemes[loop].name, name) == 0)
			return loop;
	}

	return -1;
}

/* returns the default color scheme or -1 if none */
int load_config(char *file)
{
	int fd = -1;
	char *cur_scheme = NULL;
	char *defaultcscheme = NULL;
	int defaultcindex = -1;

	/* given file */
	if (file)
	{
		fd = open(file, O_RDONLY);
		if (fd == -1)
			error_exit("Could not open configfile %s!\n", file);
	}

	/* file in current path */
	if (fd == -1)
		fd = open("multitail.conf", O_RDONLY);

	/* file in /etc */
	if (fd == -1)
		fd = open("/etc/multitail.conf", O_RDONLY);

	/* no configfiles found? */
	if (fd == -1)
		return -1;

	for(;;)
	{
		char *dummy, *par;
		char *cmd = read_line_fd(fd);
		if (!cmd)
			break;

		/* skip comments */
		if (cmd[0] == '#' || cmd[0] == ';')
			continue;

		/* lines are in the format of command:parameter */
		dummy = strchr(cmd, ':');
		if (!dummy)
			error_exit("Malformed configline found: %s (command delimiter (:) missing)\n", cmd);
		*dummy = 0x00;
		par = dummy + 1; 

		if (strcasecmp(cmd, "defaultcscheme") == 0)	/* default colorscheme */
		{
			defaultcscheme = mystrdup(par);
		}
		else if (strcasecmp(cmd, "colorscheme") == 0)	/* name for colorscheme */
		{
			if (cur_scheme)
				free(cur_scheme);

			cur_scheme = mystrdup(par);
		}
		else if (strcasecmp(cmd, "cs_re") == 0)		/* ColorScheme_RegularExpression */
		{
			char *re;
			int sn;

			dummy = strchr(par, ':');		/* format: cs_re:color:regular expression */
			if (!dummy)
				error_exit("cs_re-entry malformed: color or regular expression missing (%s:%s)\n", cmd, par);
			*dummy = 0x00;
			re = dummy + 1;

			/* find colorscheme */
			sn = find_colorscheme(cur_scheme);
			if (sn == -1)
			{
				sn = n_cschemes++;
				cschemes = (color_scheme *)myrealloc(cschemes, n_cschemes * sizeof(color_scheme), "list of colorschemes");

				cschemes[sn].name = mystrdup(cur_scheme);
				cschemes[sn].n = 0;
				cschemes[sn].color = NULL;
				cschemes[sn].regex = NULL;
			}

			/* add to list */
			cschemes[sn].color = (int *)myrealloc(cschemes[sn].color, (cschemes[sn].n + 1) * sizeof(int), "color for regexp");
			cschemes[sn].regex = (regex_t *)myrealloc(cschemes[sn].regex, (cschemes[sn].n + 1) * sizeof(regex_t), "regexp for colorscheme");
			if (strcasecmp(par, "red") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_RED;
			else if (strcasecmp(par, "green") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_GREEN;
			else if (strcasecmp(par, "yellow") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_YELLOW;
			else if (strcasecmp(par, "blue") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_BLUE;
			else if (strcasecmp(par, "magenta") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_MAGENTA;
			else if (strcasecmp(par, "cyan") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_CYAN;
			else if (strcasecmp(par, "white") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_WHITE;
			else
				error_exit("%s is not a known color\n", par);

			/* compile regular expression */
			if (regcomp(&cschemes[sn].regex[cschemes[sn].n], re, REG_EXTENDED))
				error_exit("While loading configurationfile: failed to compile regular expression: %s\n", re);

			cschemes[sn].n++;
		}

		free(cmd);
	}

	close(fd);

	if (defaultcscheme)
	{
		defaultcindex = find_colorscheme(defaultcscheme);

		if (defaultcindex == -1)
			error_exit("Default colorscheme '%s' is not defined! Check multitail.conf\n", defaultcscheme);
	}

	return defaultcindex;
}

int main(int argc, char *argv[])
{
	int loop;
	char curcolor = 'n', allcolor = 'n';
	char regex_mode = 0;
	char fi = 0;
	char *fd = NULL;
	char follow_filename = 0, retry = 0, invert_regexp = 0;
	char *expr = NULL, exprall = 0;
	time_t lastupdate = 0;
	int update_interval = 0;
	char do_refresh = 0;
	char *nsubwindows = NULL;
	int maxlines = 0;
	char setmark = 0, allmark = 0;
	int restart = -1;
	char do_diff = 0;
	int hb_x = 0, hb_y = 0, hb_dx = 1, hb_dy = 1;
	time_t hb_time;
	char line_wrap = 'a';
	int line_wrap_offset = 0;
	re *pre = NULL;
	int n_re = 0;
	char config_loaded = 0;
	char *config_file = NULL;
	int cur_color_scheme = -1;
	int default_color_scheme = -1;

	if (SIG_ERR == signal(SIGTERM, do_exit)) error_exit("signal failed");

	/* determine PATH_MAX */
	path_max = pathconf("/", _PC_PATH_MAX);
	if (path_max == -1)
	{
		if (errno) error_exit("pathconf(_PC_PATH_MAX) failed\n");

		path_max = 255;
	}
	else
	{
		path_max++; /* since its relative to root */
	}

	/* calc. buffer length (at least a complete terminal screen) */
	initscr();
	min_n_bufferlines = max(MIN_N_BUFFERLINES, LINES);
#ifdef N_CURSES
	if (has_colors())
	{
		use_colors = 1;
	}
#endif
	endwin();

	/* verify size of terminal window */
	if (LINES < 24 || COLS < 80)
		error_exit("Your terminal(-window) is %dx%d. That is too small for MultiTail (at least 80x24 is required).\n");

	/* init random thing */
	time(&hb_time);
	srandom((unsigned int)hb_time);

	/* parse commandline */
	for(loop=1; loop<argc; loop++)
	{
		if (strcmp(argv[loop], "-V") == 0)
		{
			printf("%s\n\n", version);
			printf("Thank you for using MultiTail.\n");
			printf("If you have any suggestion on how I can improve this program,\n");
			printf("do not hesitate to contact me at folkert@vanheusden.com\n");
			printf("Website is available at: http://www.vanheusden.com/multitail/\n\n\n");

			return 0;
		}
		else if (strcmp(argv[loop], "-p") == 0)
		{
			line_wrap = argv[++loop][0];
			if (line_wrap == 'o')
				line_wrap_offset = atoi(argv[++loop]);
		}
		else if (strcmp(argv[loop], "--retry") == 0)
		{
			retry = 1;
		}
		else if (strcmp(argv[loop], "-u") == 0)
		{
			update_interval = atoi(argv[++loop]);
		}
		else if (strcasecmp(argv[loop], "-r") == 0)
		{
			if (argv[loop][1] == 'R')
				do_diff = 1;

			restart = atoi(argv[++loop]);
		}
		else if (strcmp(argv[loop], "-s") == 0)
		{
			split = 1;
		}
		else if (argv[loop][0] == '-' && toupper(argv[loop][1]) == 'E')
		{
			/* get expression */
			expr = argv[++loop];

			/* -e => only for this file, -E => for all following files */
			if (argv[loop][1] == 'E')
				exprall = 1;

			/* c/C/m define colors */
			if (toupper(argv[loop][2]) == 'C')
				regex_mode = argv[loop][2];
			else if (toupper(argv[loop][2]) == 'B')
				regex_mode = argv[loop][2];
			else
				regex_mode = 'm';	/* m = match, only print when matches */

			/* compile & set expression */
			/* allocate new structure */
			pre = (re *)myrealloc(pre, sizeof(re) * (n_re + 1), "list of regular expressions");

			/* initialize structure */
			memset(&pre[n_re], 0x00, sizeof(re));

			/* compile */
			if (regcomp(&pre[n_re].regex, expr, REG_EXTENDED))
				error_exit("failed to compile regular expression \"%s\"!\n", expr);

			/* remember string for later edit */
			pre[n_re].regex_str = mystrdup(expr);

			/* set flag on current file */
			pre[n_re].use_regex = regex_mode;
			if (exprall == 0)
				regex_mode = 0;

			/* wether to invert the reg exp or not */
			pre[n_re].invert_regex = invert_regexp;
			if (exprall == 0)
				invert_regexp = 0;

			n_re++;

		}
		else if (strcmp(argv[loop], "-v") == 0)
		{
			invert_regexp = 1;
		}
		else if (argv[loop][0] == '-' && toupper(argv[loop][1]) == 'C')
		{
			char dummy = -1, doall = 0;

			if (argv[loop][1] == 'C')
				doall = 1;

			if (argv[loop][2] == 's')	/* syslog-file coloring? */
				dummy = 's';
			else if (argv[loop][2] == 'S')	/* use colorscheme */
			{
				char *cur_cscheme = argv[++loop];

				if (config_loaded == 0)
				{
					/* load configurationfile (if any) */
					default_color_scheme = load_config(config_file);

					config_loaded = 1;
				}

				cur_color_scheme = find_colorscheme(cur_cscheme);
				if (cur_color_scheme == -1)
					error_exit("Color scheme %s not found! Check your configfile (%s)\n", cur_cscheme, config_file);
				dummy = 'S';
			}
			else if (argv[loop][2] == '-')	/* do not color current */
				dummy = 'n';
			else if (argv[loop][2] == 'f')	/* select field for coloring */
			{
				dummy = 'f';
				fi = atoi(argv[++loop]);
				fd = argv[++loop];
			}
			else				/* use complete line for coloring */
				dummy = 'm';

			if (doall)
				allcolor = dummy;
			else
				curcolor = dummy;
		}
		else if (strcmp(argv[loop], "-f") == 0)
		{
			follow_filename = 1;
		}
		else if (strcmp(argv[loop], "-w") == 0)
		{
			use_colors = 0;
		}
		else if (strcmp(argv[loop], "-m") == 0 || strcmp(argv[loop], "-M") == 0)
		{
			setmark = 1;
			if (argv[loop][1] == 'M')
				allmark = 1;
			maxlines = atoi(argv[++loop]);
		}
		else if (strcasecmp(argv[loop], "-i") == 0 || argv[loop][0] != '-' ||
			 strcasecmp(argv[loop], "-l") == 0)
		{
			struct stat buf;
			char *dummy;
			char is_cmd = 0;
			char is_sub = 0;
			proginfo *cur;

			if (config_loaded == 0)
			{
				/* load configurationfile (if any) */
				default_color_scheme = load_config(config_file);

				config_loaded = 1;
			}

			if (strcasecmp(argv[loop], "-l") == 0)
			{
				is_cmd = 1;
			}

			if (strcmp(argv[loop], "-L") == 0 || strcmp(argv[loop], "-I") == 0)
			{
				is_sub = 1;
			}

			if (argv[loop][0] == '-')
			{
				loop++;
			}

			dummy = argv[loop];

			if (is_sub == 1 && nfd > 0)
			{
				cur = &pi[nfd - 1];

				while(cur -> next)
				{
					cur = cur -> next;
				}

				cur -> next = (proginfo *)mymalloc(sizeof(proginfo), "proginfo");

				cur = cur -> next;

				nsubwindows[nfd-1]++;
			}
			else
			{
				pi = (proginfo *)myrealloc(pi, (nfd + 1) * sizeof(proginfo), "proginfo");

				lb = (buffer *)myrealloc(lb, (nfd + 1) * sizeof(buffer), "buffers");

				nsubwindows = (char *)myrealloc(nsubwindows, (nfd + 1) * sizeof(char), "subwindows");
				if (!nsubwindows) error_exit("out of memory!\n");

				nsubwindows[nfd] = 1;
				memset(&lb[nfd], 0x00, sizeof(buffer));
				lb[nfd].maxnlines = min_n_bufferlines;
				lb[nfd].markset = 'm';

				cur = &pi[nfd];
				nfd++;
			}
			memset(cur, 0x00, sizeof(proginfo));

			/* see if file exists */
			if (is_cmd == 0 && retry == 0 && stat(dummy, &buf) == -1)
			{
				error_exit("error opening %s (%d)\n", dummy, errno);
			}

			/* init. struct. for this file */
			cur -> filename = mystrdup(dummy);
			cur -> is_command = is_cmd;

			/* store regular expression(s) */
			cur -> n_re = n_re;
			cur -> pre = pre;
			if (exprall == 0)
			{
				n_re = 0;
				pre = NULL;
			}

			/* hide this window? */
			cur -> hidden = 0;

			/* line wrap */
			cur -> line_wrap = line_wrap;
			line_wrap = 'a';
			cur -> line_wrap_offset = line_wrap_offset;

			cur -> retry_open = retry;
			cur -> follow_filename = follow_filename;

			/* 'watch' functionality configuration (more or less) */
			cur -> restart = restart;
			restart = -1; /* invalidate for next parameter */
			cur -> first = 1;
			cur -> do_diff = do_diff;
			do_diff = 0;

			if (cur_color_scheme == -1)
				cur_color_scheme = default_color_scheme;

			cur -> color_scheme = cur_color_scheme;
			if (curcolor != 'n' && curcolor != 0)
			{
				cur -> colorize = curcolor;
			}
			else
			{
				if (allcolor == 'n' && curcolor == 'n' && default_color_scheme != -1)
				{
					cur -> colorize = 'S';
				}
				else if (allcolor == 'n' || curcolor == 'n')
				{
					cur -> colorize = 0;
				}
				else
				{
					cur -> colorize = allcolor;
				}
			}
			cur -> field_nr = fi;
			if (fd)
			{
				cur -> field_del = mystrdup(fd);
			}
			else
			{
				cur -> field_del = NULL;
			}

			curcolor = 'n';
			retry = follow_filename = 0;
		}
		else if (strcmp(argv[loop], "-Cs") == 0)
		{
			allcolor = 's';
		}
		else if (strcmp(argv[loop], "-d") == 0)
		{
			mode_statusline = 0;
		}
		else if (strcmp(argv[loop], "-D") == 0)
		{
			mode_statusline = -1;
		}
		else if (strcmp(argv[loop], "-z") == 0)
		{
			warn_closed = 0;
		}
		else if (strcmp(argv[loop], "-H") == 0)
		{
			heartbeat = atoi(argv[++loop]);
		}
		else
		{
			if (strcmp(argv[loop], "-h") != 0)
			{
				fprintf(stderr, "unknown parameter '%s'\n", argv[loop]);
			}

			usage();

			return 1;
		}
	}

	if (config_loaded == 0)
	{
		/* load configurationfile (if any) */
		default_color_scheme = load_config(config_file);

		config_loaded = 1;
	}

	/* start curses library */
	init_curses();

	/* start processes */
	for(loop=0; loop<nfd; loop++)
	{
		proginfo *cur = &pi[loop];
		int cur_win_size = min_n_bufferlines / nsubwindows[loop];

		do
		{
			if (start_tail(cur, cur_win_size) == -1)
				error_exit("failed to start process! (%s)\n", cur -> filename);

			cur = cur -> next;
		} while(cur);
	}
	free(nsubwindows);

	/* set signalhandler for terminal resize */
	if (SIG_ERR ==  signal(SIGWINCH, do_resize)) error_exit("signal failed");

	/* create windows */
	do_refresh = 2;

	for(;;)
	{
		int last_fd = 0, rc;
		fd_set rfds;
		struct timeval tv;

		tv.tv_sec = 0;
		tv.tv_usec = 100000;

		FD_ZERO(&rfds);

		/* add stdin to fd-set: needed for monitoring key-presses */
		FD_SET(0, &rfds);

		/* add fd's of pipes to fd-set */
		for(loop=0; loop<nfd; loop++)
		{
			proginfo *cur = &pi[loop];

			do
			{
				if (cur -> fd != -1)
				{
					FD_SET(cur -> fd, &rfds);
					last_fd = max(last_fd, cur -> fd);
				}

				cur = cur -> next;
			}
			while(cur);
		}

		/* heartbeat? */
		if (heartbeat)
		{
			time_t now;

			time(&now);

			if ((now - hb_time) >= heartbeat)
			{
				int delta = (int)((random() % 3) + 1);

				move(hb_y, hb_x);

				hb_y += hb_dy;
				hb_x += hb_dx;

				if (hb_y >= max_y - 1)
				{
					hb_y = max_y - 1;
					hb_dy = -delta;
				}
				else if (hb_y < 0)
				{
					hb_y = 0;
					hb_dy = delta;
				}
				if (hb_x >= max_x - 1)
				{
					hb_x = max_x - 1;
					hb_dx = -delta;
				}
				else if (hb_x < 0)
				{
					hb_x = 0;
					hb_dx = delta;
				}

				hb_time = now;

				if (do_refresh == 0)
					do_refresh = 1;
			}
		}

		/* update screen? */
		if (do_refresh)
		{
			time_t now;

			time(&now);

			if (do_refresh == 2)
			{
				/* no windows? display list of keys */
				if (nfd == 0)
				{
					int index = 0;

					wclear(stdscr);

					wprintw(stdscr, "%s\n\n", version);

					do
					{
						wprintw(stdscr, "%s\n", keys[index++]);
					} while (keys[index]);

					mydoupdate(stdscr);
				}
				else
				{
					create_windows();
				}
			}

			if ((now - lastupdate) >= update_interval)
			{
				do_refresh = 0;
				lastupdate = now;
				refresh();
			}
		}

		/* wait for any data or key press */
		if ((rc = select(last_fd + 1, &rfds, NULL, NULL, &tv)) == -1)
		{
			if (errno != EINTR)
			{
				error_exit("select returned an error! (%d)\n", errno);
			}

			continue;
		}

		if (terminal_changed)
		{
			terminal_changed = 0;

#ifdef N_CURSES
			if (ERR == resizeterm(max_y, max_x)) error_exit("problem resizeing terminal\n");
#endif

			touchwin(stdscr);
			endwin();
			refresh();

			create_windows();
		}

		/* any fd's set? */
		if (rc == 0)
		{
			/* verify if any of the processes exited */
			for(loop=0; loop<nfd; loop++)
			{
				char deleted_entry_in_array = 0;
				proginfo *cur = &pi[loop];

				do
				{
					/* see if the process exited */
					pid_t rc = waitpid(cur -> pid, NULL, WNOHANG | WUNTRACED);

					/* error while waiting? */
					if (rc == -1 && errno != ECHILD)
			                        error_exit("waitpid failed\n");

					/* did it exit? */
					if (rc != 0) /* equal to: rc == cur -> pid */
					{
						deleted_entry_in_array = close_window(loop, cur);
						/* is an entry deleted? (>=0) or restarted? (==-1) */
						if (deleted_entry_in_array >= 0)
						{
							do_refresh = 2;
							break;
						}
						else if (cur -> do_diff && do_refresh == 0)
						{
							do_refresh = 1;
						}
					}

					cur = cur -> next;
				}
				while(cur);

				if (deleted_entry_in_array > 0)
					break;
			}

			/* and since no fd's were set, redo the select()-call */
			continue;
		}

		/* any key pressed? */
		if (FD_ISSET(0, &rfds))
		{
			int c = wait_for_keypress();
			int uc = toupper(c);

			do_refresh = 2;
			
      			if (uc == 'Q' || uc == 'X')
			{
				break;
			}
			else if (uc == 'A')
			{
				add_window();
				continue;
			}
			else if (uc == 'H' || uc == '?')
			{
				show_help();
				continue;
			}
			else if (uc == 'I')
			{
				info();
				continue;
			}

			if (nfd == 0)
			{
				wrong_key();
				do_refresh = 0;
				continue;
			}

#ifdef N_CURSES
			if (c == KEY_RESIZE || uc == 'R')
#else
			if (uc == 'R')
#endif
			{
			}
			else if (uc == 'E' || uc == '\\')
			{
				enter_regexp();
			}
			else if (uc == 'D')
			{
				delete_window();
				continue;
			}
			else if (uc == 'V')
			{
				toggle_vertical_split();
			}
			else if (uc == 'C' && use_colors)
			{
				toggle_colors();
			}
			else if (uc == 'S')
			{
				swap_window();
				continue;
			}
			else if (uc == 'Z')
			{
				hide_window();
			}
			else if (uc == 'W')
			{
				write_script();
			}
			else if (uc == 'M')
			{
				set_mark();
			}
			else if (uc == 'N')
			{
				delete_mark();
			}
			else if (uc == 'B')
			{
				scrollback();
			}
			else if (uc == 'P')
			{
				do_pause();
			}
			else
			{
				wrong_key();
				do_refresh = 0;
			}
		}

		/* go through all fds */
		for(loop=0; loop<nfd; loop++)
		{
			char deleted_entry_in_array = 0;
			proginfo *cur = &pi[loop];
			WINDOW *data = pi[loop].data, *status = pi[loop].status;

			do
			{
				if (cur -> fd == -1)
					continue;

				if (FD_ISSET(cur -> fd, &rfds))
				{
					char buffer[65536];
					int nbytes;

					nbytes = read(cur -> fd, buffer, 65535);
					if (nbytes == -1)
					{
						if (errno != EINTR)
						{
							fprintf(stderr, "read-error on file!\n");
							deleted_entry_in_array = close_window(loop, cur);
							if (deleted_entry_in_array >= 0)
							{
								do_refresh = 2;
								break;
							}
							else if (cur -> do_diff && do_refresh == 0)
							{
								do_refresh = 1;
							}
						}
					}
					else if (nbytes == 0)	/* tail exited? */
					{
						deleted_entry_in_array = close_window(loop, cur);
						if (deleted_entry_in_array >= 0)
						{
							do_refresh = 2;
							break;
						}
						else if (cur -> do_diff && do_refresh == 0)
						{
							do_refresh = 1;
						}
					}
					else
					{
						char *pnt = buffer;
						buffer[nbytes] = 0x00;

						for(;;)
						{
							char org;
							char *end = strchr(pnt, '\n');
							if (end)
							{
								org = end[1];
								end[1] = 0x00;
							}

							/* is this the output of a program which we should diff and such? */
							if (cur -> do_diff)
							{
								store_for_diff(cur, pnt);
							}
							else /* no, just output */
							{
								if (do_refresh == 0)
									do_refresh = 1;	/* after update interval, update screen */

								/* display statusline? */
								update_statusline(status, cur);

								/* output new text */
								color_print(loop, cur, pnt);
							}

							if (end)
							{
								end[1] = org;
								pnt = end + 1;
							}
							else
								break;
						}

						if (do_refresh);
							wnoutrefresh(data);
					}
				}

				cur = cur -> next;
			}
			while(cur);

			if (deleted_entry_in_array > 0)
				break;
		}
	}

	endwin();

	/* kill tail processes */
	do_exit(0);

	return 0;
}
