 /*Copyright (C) Martin Robinson 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *----------------------------------------------------------------------*/
#include "mterm.h"
int main (int argc, char *argv[]) {

	/* Certain Xlib variables that shoud be kept in a cage */
	XGCValues gc_val;
	XTextProperty name;
	XWMHints wmhints;	
	XSizeHints  sizehints;
	unsigned long gc_mask;
	char *geometry = "";
	int gravity;
	int color_black, color_white;
	KeySym ks;
	Atom hints_atom;
	MotifWmHints mw_hints;
	Pixmap mask;
	/* Other variables that I wish I didn't have to put here */
	char buf[1];
	int i;

	/* Parse options first, so they can be overridden by command line */
	parse_options();

	/* Take care of this command line stuff */
	for (i = 1; i < argc; i++) {
		char *arg = argv[i];
	
		if (arg[0] == '-') {
			switch(arg[1]) {
				case 'a':
					opt_autokill = 1;
					continue;
				case 'i':
					opt_icon_win = 1;
					continue;
				case 'l':
					if (argc <= ++i) usage();
					opt_window_chars = atoi(argv[i]) - 1;
					continue;
				case 'n':
					if (argc <= ++i) usage();
					opt_hist_size = atoi(argv[i]);
					continue;
				case 'o':
					opt_spec_win = 0;
					continue;
				case 'p':
					if (argc <= ++i) usage(); 
					geometry = argv[i];
					continue;
				case 'r':
					opt_rem_hist = 1;
					continue;
				case 's':
					opt_shaped = 1;	
					continue;
				case 'w':
					opt_withdrawn = 0;
					continue;
				default: 
					usage();
				
			}
		}
	}
	/* Default behavior: turn off special window when not withdrawn */
	/* This is currently like this, because long windows flicker when
	 * updated constantly, which is, unfortunately, what we have to do.*/
	if (opt_withdrawn == 0) {
		opt_spec_win = 0;
		opt_shaped = 0;
	}

	/* The mterm_hist file writes history - initialize it before reading*/
	create_history();

	memset(command, 0, COMMAND_SIZE);

	/* Read history file */
	if (opt_rem_hist) parse_history_file();

	/* Whats in a name */
	if (XStringListToTextProperty(&c_name, 1, &name) ==0) {
        fprintf(stderr, "%s: can't allocate window name\n", c_name);
        quit_program(1);
    }

	/* Lets open this puppy up */
	if ( !(display = XOpenDisplay(NULL)) ) {
		fprintf(stderr, "%s: can not open display %s", c_name,
		XDisplayName(NULL));
		quit_program(1);
	}

	/* Start initializing the window */
	screen = DefaultScreen(display);
	root = RootWindow(display, screen);

	/* Prepare the pixmaps -- this is important because it is relied upon
	 * for the size parameters of the window							*/
	prepare_pixmaps();


	/* Really, I swear its Xlib's fault */
	color_black = BlackPixel(display, screen);
	color_white = WhitePixel(display, screen);	

	/* Size of up the whole window mess thing */
	sizehints.flags = USSize|USPosition;
	sizehints.x = 0;
	sizehints.y = 0;

	XWMGeometry(display, screen, geometry, NULL, 1, &sizehints,
				&sizehints.x, &sizehints.y, &sizehints.width,
				&sizehints.height, &gravity);

	/* Calculate max chars and dead space */
	max_chars = (TEXT_WIDTH/CHAR_WIDTH);
	if  (TEXT_WIDTH%CHAR_WIDTH > 0 ) max_chars--;

	if (opt_withdrawn) {
		sizehints.width = pix_main.attr.width;
		sizehints.height = pix_main.attr.height;
	} else {
		if ( opt_window_chars < max_chars) opt_window_chars = max_chars;
		sizehints.width = ((opt_window_chars+1)*CHAR_WIDTH)+(2*TEXT_X_OFFSET);
		sizehints.height = CHAR_HEIGHT + (2*TEXT_Y_OFFSET);
	}

	win = XCreateSimpleWindow(display, root, sizehints.x, sizehints.y,
							  sizehints.width, sizehints.height, 1,
							  color_black, color_black);
	if (opt_icon_win)
		iconwin = XCreateSimpleWindow(display, root, sizehints.x, sizehints.y,
								  sizehints.width, sizehints.height, 1,
								  color_black, color_black);

	act_win = win;

	/* Set up the shape of the window */
	if (opt_shaped) {
			mask = XCreateBitmapFromData(display, win, mask_bits, mask_width, 
										 mask_height);

			XShapeCombineMask(display,win,ShapeBounding,0,0,mask,ShapeSet);
			if (opt_icon_win)
			XShapeCombineMask(display,iconwin,ShapeBounding,0,0,mask,ShapeSet);
	}
	
	/* Set up ye olde window manager hints */
	wmhints.flags = StateHint;

	if (opt_icon_win) {
		wmhints.flags = wmhints.flags | IconWindowHint | 
						IconPositionHint | WindowGroupHint;
		wmhints.icon_window = iconwin;
		wmhints.icon_x = sizehints.x;
		wmhints.icon_y = sizehints.y;
		wmhints.window_group = win;
	}

	if (opt_withdrawn) 
		wmhints.initial_state = WithdrawnState;
	else
		wmhints.initial_state = NormalState;

	XSetWMHints(display, win, &wmhints);

	/* The special long window for long commands */
	sizehints.width = SPEC_WIN_SIZE * CHAR_WIDTH + 2;
	sizehints.height = CHAR_HEIGHT + 2;
	sizehints.x = sizehints.x + pix_main.attr.width + TEXT_X_OFFSET;
	sizehints.y = TEXT_Y_OFFSET;

	swin = XCreateSimpleWindow(display, root, sizehints.x, sizehints.y,
							  sizehints.width, sizehints.height, 1,
							  color_white, color_black);
	hints_atom = XInternAtom(display, "_MOTIF_WM_HINTS", 0);
	mw_hints.flags = MWM_HINTS_DECORATIONS;
	mw_hints.decorations = 0;
	XChangeProperty(display, swin, hints_atom, hints_atom, 32,
					PropModeReplace, (unsigned char*)&mw_hints, 5);


	XSetWMNormalHints(display, win, &sizehints);
	XSetWMNormalHints(display, swin, &sizehints);

	XSetWMName(display, win, &name);

	XSelectInput(display, win, WM_EVENTS);
	if (opt_icon_win) 
		XSelectInput(display, iconwin, WM_EVENTS);

	/* Stateless eh? */ 
	gc_mask = GCForeground|GCBackground|GCGraphicsExposures;
	gc_val.foreground = color_white;
	gc_val.background = color_black;
	gc_val.graphics_exposures = FALSE;
	main_gc = XCreateGC(display, root, gc_mask, &gc_val);
		
	XSetCommand(display, win, argv, argc);
	XMapWindow(display, win);

	init_tab_complete();

	while(1) {
		while(XPending(display)) {
			XEvent event;
			XNextEvent(display, &event);
		
			switch(event.type) {
				case Expose: /* Exposure */
					if(event.xexpose.count == 0) {
						act_win = event.xexpose.window;
						/* We don't want to map this window until we know
 						* that the first is all set. If its not done like
 						* this, manual place Window Managers crash (twm) */
						if (opt_spec_win) XMapWindow(display, swin);
					}
					redraw_window();
					break;

				case ButtonPress:
					if (opt_withdrawn)
						XSetInputFocus(display,event.xbutton.window,
									   RevertToNone,CurrentTime);
					if (event.xbutton.button == Button2) {
						paste_selection(event.xkey.time, event.xkey.x,
										event.xkey.y);
						redraw_window();
					}
					break;

				case KeyPress:
					XLookupString(&event.xkey,buf, 1, &ks, NULL); 

					key_pressed(buf[0], ks, event.xkey.state);
					redraw_window();

					break;

				case DestroyNotify:
					XCloseDisplay(display);
					quit_program(0);
				}
			}
			
		/* Redraw the window after each cycle */
		if ( (spec_map < 5) && (opt_spec_win)) { /*  XXX BIG BIG BIG Hack XXX */
			redraw_window(); 
		}

		XFlush(display); /* Flush out the display */
		usleep(50000L);  /* Sleep for 5/100 of a second */
			
	}

	return 0;
}
	
/* Usage */
void usage() {
	char **cmesg;

	fprintf(stderr, "%s (mini terminal) version %s\n", c_name, c_ver);
	fprintf(stderr, "usage: %s [-options ...] \n", c_name);

	for (cmesg = help_msg; *cmesg; cmesg++) 
		fprintf(stderr, "%s\n", *cmesg);
	
	fprintf(stderr, "\n");
	
	exit(1);
}

/* Subroutinish thing that takes the raw xpm data (#included) and converts
 * all of them to Xpixmap format through prepare_pixmaps(...) function*/
int prepare_pixmaps() {

	static char **main_xpm; /* This is needed to distinguish between masked */
							/* and unmasked xpm's */

	/* If opt_shaped then we want to use the masked version */
	main_xpm = opt_shaped ? mask_xpm: mterm_xpm;

	/* Creation of the main pixmap */
	pixmap_from_data(main_xpm, &pix_main);

	/* The text pixmap */
	pixmap_from_data(alpha_xpm, &pix_alpha);

		return 1;
}

/* This function takes the raw xpm data (which is really just a pointer to
 * some pointers to characters) and transforms it magically with the xpm
 * library into something that can be shown on the screen				*/
int pixmap_from_data(char **data, xpixmap* icon) {

	icon->attr.valuemask |= (XpmReturnPixels | XpmReturnExtensions|
							 XpmExactColors | XpmCloseness);
	icon->attr.exactColors = False;
	icon->attr.closeness = 40000;


	if ( (XpmCreatePixmapFromData(display, root, data, &(icon->pixmap),
		  &(icon->mask), &(icon->attr)) ) != XpmSuccess) {
		
			fprintf(stderr, "%s: Couldn't create pixmaps.", c_name);
			quit_program(1);
	}
	return 1;
}

/* Another subroutine type thing that copies the main pixmap into the
 * window. It also copies the characters into place. A beast.			*/
int redraw_window() {
	int i, first_char;
	Drawable drawable;

	pos_spec_window();
	
	if (opt_withdrawn) {
		drawable = pix_main.pixmap;
	} else {
		max_chars = opt_window_chars;
		drawable = win;
	}


	if (opt_spec_win) {
		if (strlen(command) > max_chars + 1) { 
			if (spec_map < 5) { /* XXX BIG BIG BIG Hack - necessary evil XXX */
				XMapWindow(display, swin);
				spec_map++;
	
				/* Some window managers need this (wmaker) */
				XSetInputFocus(display, act_win, RevertToNone, CurrentTime);
			}
 		} else {
			XUnmapWindow(display, swin);
			spec_map = 0;


		}
	}
	
	/* Clear out the window by copying a sufficient number of space imgs */
	/* The 0,0 is the place in the alpha img where the space is			 */
	/* For the special long window*/
	if (opt_spec_win) 
		for (i = 0; i <= SPEC_WIN_SIZE; i++) 
			XCopyArea(display, pix_alpha.pixmap, swin, main_gc, 0,0,
					 CHAR_WIDTH, CHAR_HEIGHT, (i*CHAR_WIDTH) + 1,1);

	/* For the normal window */
	for (i = 0; i <= max_chars; i++) { 
		XCopyArea(display, pix_alpha.pixmap, drawable, main_gc, 0,0,
				  CHAR_WIDTH, CHAR_HEIGHT, TEXT_X_OFFSET+(i*CHAR_WIDTH),
				  TEXT_Y_OFFSET);

		/* This is for clearing the cursor */
		if (i == max_chars)
			XCopyArea(display, pix_alpha.pixmap, drawable, main_gc, 0,0,
					  CHAR_WIDTH, CHAR_HEIGHT, 2+TEXT_X_OFFSET+(i*CHAR_WIDTH),
					  TEXT_Y_OFFSET);
	}
		

	XFlush(display);
	
	
	/* Copy command to the window, one char at a time */
	first_char = strlen(command) - max_chars - 1;
	if (first_char < 0) first_char = 0;
	
	for (i = first_char; i < strlen(command); i++)  
		XCopyArea(display, pix_alpha.pixmap, drawable, main_gc,
		 		 (command[i] - ' ')*8,0, CHAR_WIDTH, CHAR_HEIGHT, 
				 ((i - first_char)*8) + TEXT_X_OFFSET, TEXT_Y_OFFSET);

	if (opt_spec_win) 
		for (i=0; i < strlen(command); i++)  {
			flush_expose(swin);
			XCopyArea(display, pix_alpha.pixmap, swin, main_gc,
					 (command[i]-' ')*8, 0, CHAR_WIDTH, CHAR_HEIGHT,
					 (i*8) + 1, 1);
		}
	

	/* Draw the cursor */
	if (strlen(command))
		XDrawLine(display, drawable, main_gc,
				 (strlen(command)-first_char)*CHAR_WIDTH+TEXT_X_OFFSET+1,
				 TEXT_Y_OFFSET,
				 (strlen(command)-first_char)*CHAR_WIDTH+TEXT_X_OFFSET+1,
				 CHAR_HEIGHT);

	/* Copy the background pixmap to the window if in withdrawn mode*/
	if (opt_withdrawn) {
		flush_expose(win);
		XCopyArea(display, pix_main.pixmap, win, main_gc, 0, 0,
				  pix_main.attr.width, pix_main.attr.height, 0,0);
	}

	if (opt_icon_win) {
		flush_expose(iconwin);
		XCopyArea(display, pix_main.pixmap, iconwin, main_gc, 0, 0,
				  pix_main.attr.width, pix_main.attr.height, 0,0);
	}
	return 1;
}
			  
int flush_expose (Window w) {
  XEvent dummy;
  int i=0;
  
  while (XCheckTypedWindowEvent (display, w, Expose, &dummy)) i++;
  return i;
}   

/* This function decides what to do when a key is pressed. It will either
 * add it to command, execute command, other varied and
 * hackish things, or be ignored. Returns 0 if ignored			*/
int key_pressed(int key, KeySym ks, unsigned int state) {
	char exec[275];
	static int tabpos = 0;
	static KeySym lastkey = XK_Return;
	
	if (ks == XK_Return) {

		if (strlen(command)) {
			/* contstruct the command */
			if (state & OUTPUT_MODIFIER) {
				sprintf(exec, "xterm -e %s", command);
			} else {
				sprintf(exec, "%s & ", command);
			}
			
			system(exec);
			add_to_history(command);

			memset(command, 0, COMMAND_SIZE);	
			XSetInputFocus(display, root, RevertToNone, CurrentTime);
			
			/* Write to the history if we gotta do that. */
			write_history();

			/* Quit the program if we gotta do that.  */
			if (opt_autokill) quit_program(0);
		}

	} else if (ks == XK_Up) {
		if (strings_filled() > hist_loc) hist_loc++;
		if (strings_filled() >= hist_loc) {	
			memset(command, 0, COMMAND_SIZE);
			strcpy(command, history_index(hist_loc));
		}
	} else if (ks == XK_Down) {
		if (hist_loc <= 1) { 
			memset(command, 0, COMMAND_SIZE);
			hist_loc = 0;
		}

		if (hist_loc > 1 && hist_loc <= opt_hist_size) { 
			memset(command, 0, COMMAND_SIZE);
			strcpy(command, history_index(--hist_loc));
		}

	} else if (ks == XK_Delete || ks == XK_BackSpace) {
		command[strlen(command)-1] = 0;
		
	} else if (ks == XK_Escape) {
			XSetInputFocus(display, root, RevertToNone, CurrentTime);
		
	} else if (ks == XK_Tab) {
		if (strlen(command)) {
			if (lastkey != XK_Tab)
				tabpos = strlen(command);
			tab_complete(tabpos, lastkey);
		}
	} else if (key >= ' ' && key <= '~') {/* Your going to need ASCII */
		if (strlen(command) < COMMAND_SIZE - 1) { 
			command[strlen(command)] = key;
			command[strlen(command)] = 0; /* Make sure the last character */
		} else return 0;
	
	}
	
	lastkey = ks;
	size_spec_window();
	return 0;
}

void traverse_dirs(const char *dirs, void (*tfunc) (const char *)) {

	char *dir_list, *dir;
	int numfiles, i;
	struct dirent **namelist;

	/* copy the dir listing, because we are going to destroy it */
	dir_list = malloc(strlen(dirs) + 1);
	strncpy(dir_list, dirs, strlen(dirs) + 1);
	dir = strtok(dir_list, ":");

	/* for each dir in the listing */
	while (dir) {

		/* get the number of files */
		numfiles = get_files(dir, &namelist);

		for (i = 0; i < numfiles; i++) { /* for all files */

			if (is_exec(dir, namelist[i]->d_name)) /* if executable */
				(*tfunc)(namelist[i]->d_name); /* call desired func */

		}

		dir = strtok(NULL, ":"); /* chop up the string */
	}

	free(dir_list); /* get rid of copy */
}

void inc_totalexec(const char *file) {
	totalexec++;
	return;
}

void add_to_exec_list(const char *file) {
	static int execidx = 0;

	/* dump executable name into char * array*/
	exec_files[execidx] = malloc(strlen(file) + 1);
	strncpy(exec_files[execidx++], file, strlen(file) + 1);

	return;
}


/* This function builds the list of executables for TAB completion.
 * The executable list is built from the PATH environment variable and then
 * sorted.  */
void init_tab_complete(void) {
	char *path;
	
	/* Check for an empty PATH environment variable. Maybe we should
	 * provide a default PATH.  */
	if (!(path = getenv("PATH")))
		return;

	/* Count the number of executables so we can malloc enough space
	 * for them all.  */
	traverse_dirs(path, inc_totalexec);
	exec_files = malloc(totalexec * sizeof(char *));

	/* Now, loop through the PATH again, and add the executables
	 * to the list that we've malloc'ed space for.  */
	traverse_dirs(path, add_to_exec_list);

	/* sort the exec list alphanumerically */
	qsort(exec_files, totalexec, sizeof(char *), vcompstr);
}

int vcompstr(const void *one, const void *two) {
	return strcmp(*(const char **) one, *(const char **) two);
}

/* Searches for a command based on the substring in the command field at the
   time that the TAB key is pressed.  The first match that is found is copied
   into the command field.  Multiple presses of the TAB key cycle through all
   the matches * in order.  												*/
void tab_complete(int tabpos, KeySym ks) {
	int i, lastpos;
	char tab_cmd[COMMAND_SIZE];
	static int searchpos = 0;

	memset(tab_cmd, 0, COMMAND_SIZE);
	strncpy(tab_cmd, command, tabpos);

	/* If the last key pressed was not a TAB, search the list from
	 * the beginning.  Otherwise search where we left off (to allow
	 * multiple TAB presses to cycle through the list).
	 */

	if (ks != XK_Tab)
		searchpos = 0;

	if (searchpos)
		lastpos = searchpos - 1;
	else
		lastpos = totalexec;

	for (i = searchpos; i != lastpos; i++) {

		/* Wrap to the beginning of the list */
		if (i == totalexec) 
			i = 0;

		/* When a match is found, move the search position forward
		 * for subsequent TABs, and copy the matched command into 
		 * the mterm text field.
		 */
		/*if (is_match(exec_files[i], tab_cmd)) {*/
		if (!strncmp(exec_files[i], tab_cmd, strlen(tab_cmd))) {
			searchpos = i+ 1;
			memset(command, 0, COMMAND_SIZE);
			strncpy(command, exec_files[i], strlen(exec_files[i]));
			return;
		}
	} 
}

/* Returns a sorted list of files in namelist (gathered from path), and the 
 * number of files as a return value.  0 is returned on error (i.e. the 
 * directory path cannot be read).
 */
int get_files(char *path, struct dirent ***namelist) {
	DIR *dp;
	int numfiles;

	if ((dp = opendir(path)) == NULL) {
		DEBUG_PRINT((stderr, "Could not access directory: %s\n", path));
		return(0);
	}

	if ((numfiles = scandir(path, namelist, 0, alphasort)) == -1) {
		DEBUG_PRINT((stderr, "Could not scan directory: %s\n", path));
		return(0);
	}

	return(numfiles);
}

/* Returns true if a file is executable. The full pathname is constructed
 * from directory and filename. The file is then stat'ed, and we verify
 * that the file is a regular file and we have execute permissions on it.
 */
int is_exec(char *directory, char *filename) {
	char *fullpath;
	int pathlen, executable = 0;
	struct stat filestat;

	pathlen = strlen(directory) + 1 + strlen(filename);
	fullpath = malloc(pathlen + 1);
	memset(fullpath, 0, pathlen + 1);
	snprintf(fullpath, pathlen+1, "%s/%s", directory, filename);

	/* if we can stat/access this file and it is regular */
	if ( !stat(fullpath, &filestat) && S_ISREG(filestat.st_mode)
		 && !access(fullpath, X_OK)) {

		executable = 1; /* it is executable */

	} else { 

		DEBUG_PRINT((stderr, "Could not stat directory: %s\n", fullpath));
		executable = 0; /* it is not executable */

	}

	free(fullpath);
	return(executable);
}

/* Returns the number of non-empty char arrays in the history linked  
 * list. This is used when finding when to rotate the history and other 
 * places      											*/
short int strings_filled() {
	short int i = 0;
	hist_node *node_loc = root_node;

	while(node_loc != NULL) {
		if (strlen(node_loc->command) <= 0) return i;
		node_loc = node_loc->next;
		i++;
	}

	return opt_hist_size; /* All filled */
}

/* Does what it says by pushing each item in the list down one and writing
 * the first one. *Should* handle empty spaces correctly			*/
short int add_to_history(char *exec) {
	hist_node *node_loc = last_node;

	/* Reset hist_loc, because it needs to be done virtually anytime this
	 * is called													*/
	hist_loc = 0;


    memset(node_loc->command, 0, COMMAND_SIZE);
	while(node_loc->prev != NULL) { /* Note - this loops operates on  */
		node_loc = node_loc->prev;  /* opt_hist_size-1 to the first item */
		strcpy(node_loc->next->command, node_loc->command);
        memset(node_loc->command, 0, COMMAND_SIZE);
	}
	strcpy(node_loc->command, exec);
	
	return 1;
}

/* Initialization function - sets up the static size and linking of the
 * linked list and calls clear_history to clear out the char arrays. */
void create_history() {
	int i;
	hist_node *new_node;

	new_node = root_node = malloc(sizeof(hist_node));
	new_node->prev = NULL;
	
	for (i = 2; i <= opt_hist_size; i++) {/* After this point the  */

		/* linked list generally floats  */
		new_node->next = malloc(sizeof(hist_node)); 
		new_node->next->prev = new_node;/* independenty of opt_hist_size */
		new_node = new_node->next;
	}
	last_node = new_node;
	last_node->next = NULL;
	
	clear_history();

}

void clear_history() {
	hist_node *node_loc = root_node;

	while (node_loc != NULL) { /* Traverse the list */
		memset(node_loc->command, 0, COMMAND_SIZE);
		node_loc = node_loc->next;
	}

	/* There is no history, so we don't want to be floudering in the middle
	 * of nothing.														*/
	hist_loc = 0;
}

char *history_index(int index) {
	char *ret_string;
	int i;
	hist_node *node_loc = root_node;

	if ( (ret_string = malloc(COMMAND_SIZE)) == NULL) {
		fprintf(stderr, "Could not allocate memory at line %i\n", __LINE__);
		quit_program(1);
	}

	memset(ret_string, 0, COMMAND_SIZE);

	if (strings_filled() < index) { /*  Don't let a bad index slip through */
		fprintf(stderr, "%s: history index out of bounds, ignoring.\n",
				c_name);
		return ret_string; /*  Return nothing  */
	}

	for(i = 1; i < index; i++) {/* Travel the list until 'index' */
		node_loc = node_loc->next;
	}
	
	strcpy(ret_string, node_loc->command); /* Copy the command over */

	return ret_string;
}

/* This function resizes the special window. It is called quite a bit */
void size_spec_window() {
	if (opt_spec_win) {
		XResizeWindow(display, swin, (CHAR_WIDTH*strlen(command)) + 2,
					  CHAR_HEIGHT+2);
		spec_map=1;
	}
}

/* This function positions the special window. It is called quite a bit */
void pos_spec_window() {
	int x=0, y =0;
	static int spec_x, spec_y; /* keep track of special window locations */
	XWindowAttributes attr;	
	Window blah;

	if (opt_spec_win) {
		XGetWindowAttributes(display, root, &attr);
		XTranslateCoordinates(display, act_win, root, 0, 0, &x, &y, &blah);

		spec_y = TEXT_Y_OFFSET + y;
	
		if (x > (attr.width / 2)) 
			spec_x = x - (strlen(command)*CHAR_WIDTH) - TEXT_X_OFFSET;
		else
			spec_x = x + pix_main.attr.width + TEXT_X_OFFSET;

		XMoveWindow(display, swin, spec_x, spec_y);

	}
}

/* XXX This function is somewhat broken. It does not find the correct
 * selection all the time. 											*/
void paste_selection (Time time, int x, int y) {
	int bytes;
	char *sel="";
	Window owner;
	Atom prop;

	if (x<0 || x>= pix_main.attr.width || y<0 || y>=pix_main.attr.height)
		return;

	XFlush(display);

	owner = XGetSelectionOwner(display, XA_PRIMARY);
	if (owner == None) owner = root;

	prop = XInternAtom(display, "PRIMARY", False);

	XConvertSelection(display, XA_PRIMARY, XA_STRING, XA_CUT_BUFFER0, 
						  act_win, time);

	sel = XFetchBytes(display, &bytes);

	if (strlen(command) + strlen(sel) < COMMAND_SIZE) {
		strcat(command, sel);
		command[strlen(command)] = 0; /* Last character is null */
		size_spec_window();
	} else {
	   fprintf(stderr, "%s: Pasted selection is too long, ignoring.\n", c_name);
	}

	XFree(sel);	
	
}

/* This function reads the .mtermrc file and sets the options therein   */
int parse_options () {
	FILE *rc_file;	
	char rc_file_name[256];
	char cline[256];
	char pline[256];
	int e;

	sprintf(rc_file_name, "%s%s", getenv("HOME"), "/.mtermrc");
	
	if ( (rc_file = fopen(rc_file_name, "r")) != NULL) {
		while( (fgets(cline, 255, rc_file)) != NULL) {
			e = 0;
			
		/* Find where the spaces end and the graphable characters start */
			while( !(isgraph(cline[e])) ) e++;	
	
			sscanf(cline, "%s %i", pline, &e);
 			
			/* textboxlength (for windowed mode) */
			if (pline[0] == 't') {
				opt_window_chars = e - 1;
				continue;	
			} else if (pline[0] == 'h') { /* historysize  */
				opt_hist_size = e;
				continue;
			}
			
			/* These need to be true||false values */
			if (e != 0 && e != 1) continue;

			switch (pline[0]) {
				case 'a': /*  autokill: */
					opt_autokill = e;
					break;
				case 's': /*  shaped: */
					opt_shaped = e;
					break;
				case 'h': /*  historysize: */
					opt_hist_size = e;
					break;
				case 'i': /*  iconwindow: */
					opt_icon_win = e;
					break;
				case 'w':/*  windowed */
					opt_withdrawn = !e;
					break;
				case 'r': /*  rememberhist  */
					opt_rem_hist = e;
					break;
				}
		
			memset(cline, 0, 256);
			memset(pline, 0, 256);
		}

		fclose(rc_file);
		return 1;

	} else return 0;
}

/* This functions adds the commands in the .mterm_hist file to the
 * history linked list											*/
int parse_history_file() {
	FILE *hist_file;	
	char hist_file_name[256];
	char cline[256];

	sprintf(hist_file_name, "%s%s", getenv("HOME"), "/.mterm_hist");
	
	if ( (hist_file = fopen(hist_file_name, "r")) != NULL) {
		while( (fgets(cline, 255, hist_file)) != NULL) {

			/* Get rid of the newline */
			sscanf(cline, "%[^\n]", cline);
			
			/* Add to history */
			if (strlen(cline) <= COMMAND_SIZE && strlen(cline) > 0 )  {
				add_to_history(cline);
			}
		}
		fclose(hist_file);
	}
	
	return 0;
}
	
/* This function  writes to the history. Called after every command
 * execution and in program shutdown									*/
int write_history() {
	FILE *hist_file;
	char hist_file_name[256];
	char output[257];
	hist_node *node_loc = last_node;

	if (opt_rem_hist) {
		sprintf(hist_file_name, "%s%s", getenv("HOME"), "/.mterm_hist");

    	if ((hist_file=fopen(hist_file_name,"w")) != NULL) {
			while(node_loc != NULL) {
				if (strlen(node_loc->command) > 0) {
					memset(output, 0, COMMAND_SIZE+1);
					sprintf(output,"%s\n", node_loc->command);
					fputs(output, hist_file);
				}
				node_loc = node_loc->prev;
			}
				
			fclose(hist_file);
			return 1;
		}
	}
	return 0;
}
/* This function does some cleaning up when mterm shuts down - saves the
 * history to the .mterm_hist file and closes the XDisplay				*/
void quit_program(int exit_code) {
	hist_node *node_loc = last_node;

	/* Free all this memory stuff */
	while(node_loc->prev != NULL) {
		node_loc = node_loc->prev;
		free(node_loc->next);
	}
	free(node_loc);

 	write_history();

	XCloseDisplay(display);
	exit(exit_code);
}
