/* 
 * $Id: simple_bgp.c,v 1.13 1999/04/15 04:24:55 binkertn Exp $ 
 */

/* sbgp: Simple BGP4 speaker and listner. Provides way to dump
 * BGP routing stream to other MRT tools, as well as to save stream
 * to file. Can also inject BGP information into peering with real
 * router */

#include <mrt.h>
#include <bgp.h>

/*
 * GLOBALS
 */
io_t *IO;
trace_t *default_trace;
mtimer_t *timer;
long last = 0;			/* last time we sent a packet */
int bgp4plus_version = DEFAULT_BGP4PLUS_VERSION;
int nothing = 1;
char *usage = 
    "Usage: sbgp [-01av] [-i binary_data_in_file] [-o binary_data_out_file]\n"
    "\t    [-l log_file] [-f config_file] [-c port] [-d port] [-s src_addr]\n"
    "\t    [ASmy_as] [peer_ip ASpeer_as]...\n";

/*
static void sbgp_process_update (bgp_peer_t * peer, u_char * cp, u_int len);
static int sbgp_peer_established (bgp_peer_t * peer);
static int sbgp_process_input ();
static int process_command_line_args (int argc, char *argv[]);
static void io_timer (mtimer_t * timer);
*/


/* 
 *  Just write update as MRT output mesage
 */
static void
sbgp_process_update (bgp_peer_t * peer, u_char * buf, u_int len)
{
    u_char tmp[MAX_MSG_SIZE], *cp;

    cp = tmp;

    /* from */
    BGP_PUT_SHORT (peer->gateway->AS, cp);

#ifdef HAVE_IPV6
    if (peer->gateway->prefix->family == AF_INET6) {
	memcpy (cp, prefix_tochar (peer->gateway->prefix), 16);
	cp += 16;
    }
    else
#endif /* HAVE_IPV6 */
	BGP_PUT_NETLONG (prefix_tolong (peer->gateway->prefix), cp);

    /* to */
#ifndef BGP_MULTI
    BGP_PUT_SHORT (BGP->my_as, cp);
#else /* BGP_MULTI */
    BGP_PUT_SHORT (peer->local_bgp->this_as, cp);
#endif /* BGP_MULTI */

#ifdef HAVE_IPV6
    if (peer->local_addr->family == AF_INET6) {
	memcpy (cp, prefix_tochar (peer->local_addr), 16);
	cp += 16;
    }
    else
#endif /* HAVE_IPV6 */
	BGP_PUT_NETLONG (prefix_tolong (peer->local_addr), cp);

    memcpy (cp, buf, len);
    len += (cp - tmp);

#ifdef HAVE_IPV6
    if (peer->gateway->prefix->family == AF_INET6) {
	io_write (IO, 0,
		  (bgp4plus_version == 0) ? MSG_PROTOCOL_BGP4PLUS :
		  MSG_PROTOCOL_BGP4PLUS_01,
		  MSG_BGP_UPDATE, len, tmp);
    }
    else
#endif /* HAVE_IPV6 */
	io_write (IO, 0, MSG_PROTOCOL_BGP, MSG_BGP_UPDATE, len, tmp);
}


#ifdef notdef
static int
sbgp_peer_down (bgp_peer_t * peer)
{
    Timer_Turn_ON (peer->timer_Start);
    select_delete_fd (IO->in.fd);	/* this should really be in IO routines ... */
    return (1);
}
#endif


/*
 * called by IO. Read a packet and send it on its way via BGP
 */
static void
sbgp_process_input (mrt_msg_t * msg)
{
    bgp_peer_t *peer;
    u_char *buf;

    /*while ((msg = (mrt_msg_t *) io_read (IO)) != NULL) { */
    if ((msg = (mrt_msg_t *) io_read (IO)) == NULL)
	return;

    if (msg->subtype != MSG_BGP_UPDATE) {
	select_enable_fd (IO->in.fd);
	return;
    }

    LL_Iterate (BGP->ll_bgp_peers, peer) {
	u_char *cp = (u_char *) msg->value;
	int length = msg->length;

	if (peer->state != BGPSTATE_ESTABLISHED) {
	    continue;
	}

	if (msg->type == MSG_PROTOCOL_BGP4PLUS ||
	    msg->type == MSG_PROTOCOL_BGP4PLUS_01) {
	    length -= (2 + 16 + 2 + 16);
	    cp += (2 + 16 + 2 + 16);
	}
	else {
	    length -= (2 + 4 + 2 + 4);
	    cp += (2 + 4 + 2 + 4);
	}

	buf = NewArray (u_char, length);
	memcpy (buf, cp, length);

	bgp_send_update (peer, length, (u_char *) buf);
    }

    if (last > 0) {
	/* another packet at same second */
	if (msg->tstamp == last) {
	    select_enable_fd (IO->in.fd);
	}
	else {
	    Timer_Set_Time (timer, msg->tstamp - last);
	    Timer_Turn_ON (timer);
	    trace (NORM, BGP->trace, "IO waiting for %d seconds\n",
		   msg->tstamp - last);
	}

    }
    else {
	select_enable_fd (IO->in.fd);
    }
    last = msg->tstamp;

    /* free memory here */
    Delete (msg);
}


static int
sbgp_peer_established (bgp_peer_t * peer)
{
    if (IO->io_input_type != IO_NONE)
        io_set_notify (IO, 1, sbgp_process_input);
    return (1);
}


static int
process_command_line_args (int argc, char *argv[])
{
    char c;
    extern char *optarg;	/* getopt stuff */
    extern int optind;		/* getopt stuff */
    prefix_t *prefix;
    prefix_t *src_addr = NULL;
    int errors = 0;
    int as = /* DEFAULT_MY_AS */ 0 /* must be set */;

#ifndef BGP_MULTI
    u_long id = htonl (DEFAULT_MY_ID);
#else /* BGP_MULTI */
    u_long id = htonl (DEFAULT_ROUTER_ID);
#endif /* BGP_MULTI */

    int aflag = 0;

    while ((c = getopt (argc, argv, "ar:w:hvo:i:l:s:c:d:")) != -1)
	switch (c) {
	case '0':
	    bgp4plus_version = 0;
	    break;
	case '1':
	    bgp4plus_version = 1;
	    break;
	case 'a':		/* accept all incoming connections */
	    set_BGP (BGP_ACCEPT_ALL_PEERS, 1, 0);
	    aflag++;
	    break;
	case 's':
	    src_addr = ascii2prefix (0, optarg);
	    break; 
	case 'd':		/* port for BGP daemon to listen on */
	    BGP->lport = atol (optarg);
	    break;
	case 'c':		/* port for BGP to connect to  */
	    BGP->cport = atol (optarg);
	    break;
	case 'o':		/* out data file name */
	    if (io_set (IO, IO_OUTFILE, optarg, NULL) < 0) {
		fprintf (stderr, "Error setting outfile %s\n", optarg);
		errors++;
	    }
	    break;
	case 'w':		/* out message key name */
	    if (io_set (IO, IO_OUTMSGQ, optarg, NULL) < 0) {
		fprintf (stderr, "Error setting outfile %s\n", optarg);
		errors++;
	    }
	    break;
	case 'i':		/* in data file name */
	    if (io_set (IO, IO_INFILE, optarg, NULL) < 0) {
		fprintf (stderr, "Error setting infile %s\n", optarg);
		exit (0);
	    }
	    break;
	case 'r':		/* in message key */
	    if (io_set (IO, IO_INMSGQ, optarg, NULL) < 0) {
		fprintf (stderr, "Error setting inmsg %s\n", optarg);
		errors++;
	    }
	    break;
	case 'l':		/* log file name */
	    set_trace (default_trace, TRACE_LOGFILE, optarg, NULL);
	    break;
	case 'v':		/* verbose */
	    set_trace (BGP->trace, TRACE_FLAGS, TR_ALL,
		       TRACE_LOGFILE, "stdout", NULL);
	    break;
	case 'f':		/* config file */
	    fprintf (stdout, "Config file option not yet supported");
	    errors++;
	    break;
	case 'h':
	default:
	    errors++;
	    break;
	}

    if (errors) {
	fprintf (stderr, usage);
	fprintf (stderr, "MRT version (%s) compiled on %s\n",
		MRT_VERSION, __DATE__);
	exit (1);
    }

#ifndef BGP_MULTI
    set_BGP (BGP_MY_ID, id, 0);
#endif /* BGP_MULTI */

    /* set my AS number */
    if ((optind < argc) && (!strncasecmp ("AS", argv[optind], 2)))
#ifndef BGP_MULTI
        set_BGP (BGP_MY_AS, atoi (2 + argv[optind++]), 0);
#else /* BGP_MULTI */
        as = atoi (2 + argv[optind++]);
#endif /* BGP_MULTI */

#ifdef BGP_MULTI
    if (BGP->router_bgp[as] == NULL) {
      bgp_local_t *local = New (bgp_local_t);
      init_bgp_local(local);
      BGP->router_bgp[as] = local;
      BGP->router_bgp[as]->this_as = as;
      BGP->router_bgp[as]->this_id =
        prefix_tolong (INTERFACE_MASTER->default_interface->primary->prefix);
      BGP->router_bgp[as]->view_no = BGP->view_count++;
      /* This is not good for IPv6.  -- binkertn */
      BGP->views[BGP->router_bgp[as]->view_no] = New_View(32);
      
      LL_Add (BGP->ll_bgp_locals, BGP->router_bgp[as]);
    }
    
    set_BGP (BGP_CURRENT_BGP, BGP->router_bgp[as], NULL);
#endif /* BGP_MULTI */

    /* add peers: peer_ip [peer_AS] */
    if ((optind + 1 < argc) && (!strncasecmp ("AS", argv[optind + 1], 2))) {
	bgp_peer_t *peer;

	prefix = ascii2prefix (0, argv[optind++]);
	if (prefix == NULL) {
	    fprintf (stderr, "Unknown host %s\n", argv[optind - 1]);
	    exit (0);
	}
	peer = Add_BGP_Peer (NULL, prefix, src_addr, NULL, 
			     atoi (2 + argv[optind++]), BGP->trace);
	start_bgp_peer (peer);
	nothing = 0;
    }
    else if (aflag) {
	init_BGP_listen (src_addr, NULL);
	nothing = 0;
    }

    return (1);
}


/*
 * renable timer after x seconds so we can recreate time-based events
 */
static void
io_timer (mtimer_t * timer)
{
    select_enable_fd (IO->in.fd);
}


int
main (int argc, char *argv[])
{
    default_trace = New_Trace ();
/*
   set_trace (default_trace, TRACE_FLAGS, TR_ALL, 
   TRACE_LOGFILE, "stdout",
   NULL);
 */

    init_mrt (default_trace);
    init_BGP (default_trace);

    IO = New_IO (default_trace);
    timer = New_Timer (io_timer, 0, "IO timer", NULL);

    set_BGP (BGP_PEER_ESTABLISHED_FN, sbgp_peer_established,
	     BGP_RECV_UPDATE_FN, sbgp_process_update,
	     NULL);

    process_command_line_args (argc, argv);
    if (nothing) {
	fprintf (stderr, usage);
	fprintf (stderr, "MRT version (%s) compiled on %s\n",
		MRT_VERSION, __DATE__);
	exit (1);
    }
    start_bgp ();
    mrt_main_loop ();
    return (1);
}
