/*
 * $Id: ripng.c,v 1.13 1999/04/29 19:02:58 masaki Exp $
 */

#include <config.h>
#ifdef HAVE_IPV6
#include <sys/fcntl.h>
#include <sys/uio.h>

#include <mrt.h>
#include <api6.h>
#include <rip.h>

/* NRL_IPV6 is for the obsolete NRL IPV6 implemenation */
#ifdef NRL_IPV6
/*
 * NRL's IPV6 use IPPROTO_IP instead of IPPROTO_IPV6 for socket options
 */
#undef IPPROTO_IPV6
#define IPPROTO_IPV6 IPPROTO_IP
#endif /* NRL_IPV6 */


static struct in6_addr all_ripng_routers;

/* 
 * Given 1) the gateway we received the packet from, 
 *       2) a pointer to the packet, and
 *       3) the length (num bytes) of the packet, 
 * munge the packet making a linked list of prefix and attribute tuples
 */

static void
ripng_process_packet_response (gateway_t * gateway, u_char * update, int bytes,
			       LINKED_LIST * ll_prefixes, LINKED_LIST * ll_attr,
			       int pref)
{
    u_char *cp;
    prefix_t *prefix;
    nexthop_t *nexthop = ref_nexthop (gateway);
    rip_attr_t *p_attr;
    time_t now;

#define RIPNG_RTELEN 20

    if ((bytes % RIPNG_RTELEN) != 0) {
	trace (TR_WARN, RIPNG->trace, "invalid RTE size %d\n", bytes);
    }

    time (&now);
    cp = update;

    while (cp < update + bytes) {
	struct in6_addr *addr6;
	char tmp6[INET6_ADDRSTRLEN];
	int tag, prefixlen, metric;

	if ((update + bytes) - cp < RIPNG_RTELEN)
	    break;
	addr6 = (struct in6_addr *) cp;
	cp += 16;

/* UTIL_GET_XXX requires type of argument is equal to XXX */

	BGP_GET_SHORT (tag, cp);
	BGP_GET_BYTE (prefixlen, cp);
	BGP_GET_BYTE (metric, cp);

#define RIPNG_NEXT_HOP 0xff
	if (metric == RIPNG_NEXT_HOP) {
	    if (IN6_IS_ADDR_UNSPECIFIED (addr6)) {
		trace (TR_PACKET, RIPNG->trace,
		       "nexthop is the originator itself %a\n",
		       gateway->prefix);
		continue;
	    }
	    if (!IN6_IS_ADDR_LINKLOCAL (addr6)) {
		trace (TR_WARN, RIPNG->trace,
		       "nexthop %s but not link-local address from %a\n",
		       inet_ntop (AF_INET6, addr6, tmp6, sizeof tmp6),
		       gateway->prefix);
		continue;
	    }
	    trace (TR_PACKET, RIPNG->trace,
		   "  nexthop %s\n",
		   inet_ntop (AF_INET6, addr6, tmp6, sizeof tmp6));
	    if (prefixlen != 0 && tag != 0)
		trace (TR_WARN, RIPNG->trace,
		       "non-zero prefixlen (%d) or tag (%d) "
		       "in specifying nexthop %s from %a\n",
		       prefixlen, tag,
		       inet_ntop (AF_INET6, addr6, tmp6, sizeof tmp6),
		       gateway->prefix);
	    deref_nexthop (nexthop);
	    prefix = New_Prefix (AF_INET6, addr6, 128);
	    nexthop = add_nexthop (prefix, gateway->interface);
	    Deref_Prefix (prefix);
	}
	else {
	    if (prefixlen > 128) {
	        trace (TR_WARN, RIPNG->trace,
		       "too large prefixlen %d\n", prefixlen);
		continue;
	    }
	    prefix = New_Prefix (AF_INET6, addr6, prefixlen);
	    p_attr = rip_new_attr (RIPNG, metric);
	    p_attr->gateway = gateway;
	    p_attr->nexthop = ref_nexthop (nexthop);
	    p_attr->metric = metric;
	    p_attr->tag = tag;
	    p_attr->utime = now;
	    p_attr->pref = pref;
	    LL_Add (ll_prefixes, prefix);
	    LL_Add (ll_attr, p_attr);
	}
    }
    deref_nexthop (nexthop);
}


/* 
 * Given 1) the gateway we received the packet from, 
 *       2) a pointer to the packet, and
 *       3) the length (num bytes) of the packet, 
 * fill up metrics if exists
 *
 * return value >0: ordinal request (number of entries)
 *              -1: whole-table request
 */

static int
ripng_process_packet_request (struct sockaddr_in6 *from, u_char * update,
			      int bytes, rip_interface_t * rip_interface)
{
    u_char *cp = update;
    prefix_t *prefix;
    rip_route_t *route;
    struct in6_addr *addr6;
    int n = 0;

    prefix = New_Prefix (AF_INET6, &from->sin6_addr, 128);

    /* it may be a little bit expensive to lock the route table
       we may not need to do it */

    while (cp < update + bytes) {
	int tag, prefixlen, metric;

	assert ((update + bytes) - cp >= RIPNG_RTELEN);
	addr6 = (struct in6_addr *) cp;
	cp += 16;

	BGP_GET_SHORT (tag, cp);
	BGP_GET_BYTE (prefixlen, cp);
	BGP_GET_BYTE (metric, cp);

	if (cp == update + bytes && metric == RIP_METRIC_INFINITY &&
	    prefixlen == 0 &&	/* tag == 0 && */
	    IN6_IS_ADDR_UNSPECIFIED (addr6)) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  whole-table request, responding\n");
	    Deref_Prefix (prefix);
	    return (-1);
	}

	prefix = Change_Prefix (AF_INET6, addr6, 128, prefix);

	if ((route = HASH_Lookup (RIPNG->hash, prefix))) {
	    rip_attr_t *attr = route->active;

	    tag = attr->tag;

	    if (ntohs (from->sin6_port) == RIPNG->port
	    /* && IN6_IS_ADDR_LINKLOCAL (&from->sin6_addr) */ ) {
		/* router's query, apply policy for that interface */
		if ((metric = rip_policy (RIPNG, prefix, attr,
					  rip_interface)) < 0) {
		    metric = RIP_METRIC_INFINITY;
		}
	    }
	    else {
		trace (TR_PACKET, RIPNG->trace, "  o %p metric %d\n",
		       prefix, metric);
	    }
	}
	else {
	    metric = RIP_METRIC_INFINITY;
	    tag = 0;		/* ID doesn't specify -- masaki */
	    trace (TR_PACKET, RIPNG->trace, "  x %p metric %d (no exist)\n",
		   prefix, metric);
	}
	cp -= 4;
	BGP_PUT_SHORT (tag, cp);
	BGP_PUT_BYTE (prefixlen, cp);
	BGP_PUT_BYTE (metric, cp);
	n++;
    }
    Deref_Prefix (prefix);
    return (n);
}


void
ripng_init (void)
{
    /* initialize multicast address to send and join
       defined in the spec */
    all_ripng_routers.s6_addr[0] = 0xff;
    all_ripng_routers.s6_addr[1] = 0x02;
    all_ripng_routers.s6_addr[15] = 0x09;
}


/* 
 * start listening for broadcast ripng updates and request
 */
int
ripng_init_listen (interface_t *interface)
{
    struct sockaddr_in6 ripng;
     /* INRIA accepts both char and int but returns int */
     /* KAME accepts u_int only */
    int one = 1, zero = 0, ttl = 255;
    int size, sockfd;

    if ((sockfd = socket (PF_INET6, SOCK_DGRAM, 0)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "socket: %m\n");
	return (-1);
    }

    setsockopt (sockfd, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof (one));
#ifdef SO_REUSEPORT
    setsockopt (sockfd, SOL_SOCKET, SO_REUSEPORT, (char *)&one, sizeof (one));
#endif /* SO_REUSEPORT */

#ifdef SO_RCVBUF
    size = RIPNG_MIN_BUFSIZE;
    if (setsockopt (sockfd, SOL_SOCKET, SO_RCVBUF, (char *) &size,
		    sizeof (size)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt SO_RCVBUF (%s): %m\n", size);
    }
#endif /* SO_RCVBUF */

#ifndef SOLARIS_IPV6
    /* Some may work, but it's unknown how to get info later on Solaris */
#ifdef IPV6_RECVPKTINFO
    /* XXX -- This is required by INRIA IPV6 to get pktinfo
       IPV6_PKTINFO does not work for setsockopt as RFC 2292 defines. */
    /*
     *  Receive destination address and interface from socket
     */
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_RECVPKTINFO, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt IPV6_RECVPKTINFO: %m\n");
    }
#else /* IPV6_RECVPKTINFO */
#ifdef IPV6_PKTINFO
    /*
     *  Receive the interface info from socket (RFC 2292)
     */
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_PKTINFO, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt IPV6_PKTINFO: %m\n");
    }
#else /* IPV6_PKTINFO */
#ifdef IPV6_RECVIF
    /* old INRIA IPV6 way */
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_RECVIF, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt IPV6_RECVIF: %m\n");
    }
#else /* IPV6_RECVIF */
#ifdef IP_RECVINTERFACE
    /* old INRIA IPv6 requires this */
    if (setsockopt (sockfd, IPPROTO_IPV6, IP_RECVINTERFACE, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt IP_RECVINTERFACE: %m\n");
    }
#endif /* IP_RECVINTERFACE */
#endif /* IPV6_RECVIF */
#ifdef IPV6_RECVDSTADDR
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_RECVDSTADDR, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt IPV6_RECVDSTADDR: %m\n");
    }
#else /* IPV6_RECVDSTADDR */
#ifdef IP_RECVDSTADDR
    if (setsockopt (sockfd, IPPROTO_IPV6, IP_RECVDSTADDR, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt IP_RECVDSTADDR: %m\n");
    }
#endif /* IP_RECVDSTADDR */
#endif /* IPV6_RECVDSTADDR */
#endif /* IPV6_PKTINFO */
#endif /* IPV6_RECVPKTINFO */

#ifdef IPV6_HOPLIMIT
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_HOPLIMIT, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt IPV6_HOPLIMIT: %m\n");
    }
#else /* IPV6_HOPLIMIT */
#ifdef IPV6_RECVHOPS
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_RECVHOPS, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "setsockopt IPV6_RECVHOPS: %m\n");
    }
#endif /* IPV6_RECVHOPS */
#endif /* IPV6_HOPLIMIT */
#endif /* SOLARIS_IPV6 */

#ifdef IPV6_UNICAST_HOPS
    /* I'm not sure that unicast ttl should be 255 when responding 
       routing table request, but anyway ... */
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_UNICAST_HOPS, (char *) &ttl,
		    sizeof (ttl)) < 0) {
	trace (TR_ERROR, RIPNG->trace,
	       "setsockopt IPV6_UNICAST_HOPS: %m\n");
    }
#endif /* IPV6_UNICAST_HOPS */

    {
#ifdef NRL_IPV6
/*
 * On NRL's IPV6, IPV6_MULTICAST_LOOP and IPV6_MULTICAST_HOPS
 * require a pointer to one byte
 */
	u_char zero = 0, ttl = 255;
#endif /* NRL_IPV6 */
#ifdef IPV6_MULTICAST_LOOP
	if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_MULTICAST_LOOP,
			(char *) &zero, sizeof (zero)) < 0) {
	    trace (TR_ERROR, RIPNG->trace,
		   "setsockopt IPV6_MULTICAST_LOOP: %m\n");
	}
#endif /* IPV6_MULTICAST_LOOP */

#ifdef IPV6_MULTICAST_HOPS
	if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_MULTICAST_HOPS,
			(char *) &ttl, sizeof (ttl)) < 0) {
	    trace (TR_ERROR, RIPNG->trace,
		   "setsockopt IPV6_MULTICAST_HOPS: %m\n");
	}
#endif /* IPV6_MULTICAST_HOPS */
    }

    memset (&ripng, 0, sizeof (ripng));
    ripng.sin6_port = htons (RIPNG->port);
    ripng.sin6_family = AF_INET6;
    ripng.sin6_flowinfo = htonl (RIPNG_IPV6_PRIORITY);
    
#ifdef SIN6_LEN
    ripng.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */
    if (interface) {
	assert (interface->link_local);
        memcpy (&ripng.sin6_addr, 
		prefix_toaddr6 (interface->link_local->prefix), 16);
#ifdef __KAME__
	ripng.sin6_addr.s6_addr8[2] = (interface->index >> 8) & 0xff;
	ripng.sin6_addr.s6_addr8[3] = interface->index & 0xff;
#endif /* __KAME__ */
    }

    if (bind (sockfd, (struct sockaddr *) &ripng, sizeof (ripng)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "bind to port %d: %m\n",
	       RIPNG->port);
	close (sockfd);
	return (-1);
    }

    trace (TR_TRACE, RIPNG->trace, "listening port %d at %s on %s\n",
	   ntohs (ripng.sin6_port), 
	   (interface)? prefix_toa (interface->link_local->prefix): "*",
	   (interface)? interface->name: "?");
    /* RIPNG->sockfd = sockfd; */
    return (sockfd);
}


/*
 * turn on the interface
 */
void
ripng_interface (rip_interface_t *rip_interface, int on)
{
    struct ipv6_mreq mreq;
    char tmp6[INET6_ADDRSTRLEN];
    int cmd = IPV6_DROP_MEMBERSHIP;
    char *s_cmd = "IPV6_DROP_MEMBERSHIP";
    interface_t *interface = rip_interface->interface;

    /* 
     * Join/Leave the specified multicast address
     */

    memset (&mreq, 0, sizeof (mreq));
    IN6_ADDR_COPY (&mreq.ipv6mr_multiaddr, &all_ripng_routers);

#ifdef __KAME__
	/* On KAME IPv6, it is required to modify the destination address
	   as to include the outgoing interface's index */
	/* This is unnessesary when responding to a unicast address which
	   includes an index */

	mreq.ipv6mr_multiaddr.s6_addr8[2] = (interface->index >> 8) & 0xff;
	mreq.ipv6mr_multiaddr.s6_addr8[3] = interface->index & 0xff;
#endif /* __KAME__ */

#if defined(NRL_IPV6) || defined(SOLARIS_IPV6)
	IN6_ADDR_COPY (&mreq.ipv6mr_interface,
		       prefix_toaddr6 (interface->primary6->prefix));
#else
	mreq.ipv6mr_interface = interface->index;
#endif /* NRL_IPV6 && SOLARIS_IPV6 */

    if (on) {
        assert (rip_interface->sockfd < 0);
#ifdef SOLARIS_IPV6
	/* to enforce the kernel to use a link-local address */
        /* use sockfd for each interface */
        if ((rip_interface->sockfd = ripng_init_listen (interface)) >= 0) {
	    if (setsockopt (rip_interface->sockfd, 
			    IPPROTO_IPV6, IPV6_MULTICAST_IF,
	    /* This way is the same with one used in IPv4 */
		 (const char *) prefix_tochar (interface->primary6->prefix),
			    sizeof (struct in6_addr)) < 0) {
	        trace (TR_ERROR, RIPNG->trace,
		       "setsockopt IPV6_MULTICAST_IF for %s: %m\n",
		       interface->name);
	    }
	}
#endif /* SOLARIS_IPV6 */
        cmd = IPV6_ADD_MEMBERSHIP;
        s_cmd = "IPV6_ADD_MEMBERSHIP";
    }

	if (rip_interface->sockfd >= 0) {
	    if (setsockopt (rip_interface->sockfd, IPPROTO_IPV6, cmd,
			(char *) &mreq, sizeof (mreq)) >= 0) {
	        /* this doesn't work. we can not use a bound socket 
	           for a multicast receiption */
	        select_add_fd_event ("ripng_receive_update", 
			     rip_interface->sockfd, SELECT_READ, 1,
                             RIPNG->schedule, ripng_receive_update,
			     1, rip_interface);
	    }
	}
	if (setsockopt (RIPNG->sockfd, IPPROTO_IPV6, cmd,
			(char *) &mreq, sizeof (mreq)) >= 0) {
	    trace (TR_TRACE, RIPNG->trace,
		   "%s %s on %s\n", s_cmd,
	    inet_ntop (AF_INET6, &mreq.ipv6mr_multiaddr, tmp6, sizeof tmp6),
		   interface->name);
	}
	else {
	    trace (TR_ERROR, RIPNG->trace,
		   "setsockopt %s %s for %s: %m\n", s_cmd,
	    inet_ntop (AF_INET6, &mreq.ipv6mr_multiaddr, tmp6, sizeof tmp6),
		   interface->name);
	    return;
	}

    if (on) {
        BITX_SET (&RIPNG->interface_mask, interface->index);
        /* send initial request to routers */
        ripng_send_request (rip_interface, NULL);
    }
    else {
        BITX_RESET (&RIPNG->interface_mask, interface->index);
        if (rip_interface->sockfd >= 0) {
	    trace (TR_INFO, RIPNG->trace, "Closing scoket %d\n",
                   rip_interface->sockfd);
            select_delete_fd (rip_interface->sockfd);
            rip_interface->sockfd = -1;
	}
    }
}


static int
ripng_sendmsgto (rip_interface_t *rip_interface, 
		 u_char * buffer, u_int buflen, u_int flag,
	  	 struct sockaddr_in6 *addr, u_int addrlen)
{
    int is_mc = IN6_IS_ADDR_MULTICAST (&addr->sin6_addr);
    int is_ll = IN6_IS_ADDR_LINKLOCAL (&addr->sin6_addr);
    int rc;
    char tmp6[INET6_ADDRSTRLEN];
    interface_t *interface = rip_interface->interface;
    int sockfd = (rip_interface->sockfd >= 0)?
			rip_interface->sockfd: RIPNG->sockfd;

    if (!is_ll && !is_mc) {
	if ((rc = sendto (sockfd, (char *) buffer, buflen, flag,
			  (struct sockaddr *) addr, addrlen)) < 0) {
	    trace (TR_ERROR, RIPNG->trace, "sendto: %m\n");
	}
    }
    else {

#ifdef IPV6_PKTINFO
	/* RFC 2292 */
	struct msghdr mhdr;
	struct cmsghdr *cmsg;
	struct in6_pktinfo *pkt_info;
	struct iovec iov;
	char chdr[sizeof (struct cmsghdr) + sizeof (struct in6_pktinfo)];

#ifdef __KAME__
	/* I'm not sure the recent KAME requires this.
	   I'm not sure in which ways this is required */
	struct sockaddr_in6 sin6;
	if (is_mc) {

	    /* On KAME IPv6, it is required to modify the destination address
	       as to include the outgoing interface's index */
	    /* This is unnessesary when responding to a unicast address which
	       includes an index */

	    memcpy (&sin6, addr, sizeof (sin6));
	    addr = &sin6;
	    addr->sin6_addr.s6_addr8[2] = (interface->index >> 8) & 0xff;
	    addr->sin6_addr.s6_addr8[3] = interface->index & 0xff;
	}
#endif /* __KAME__ */
	mhdr.msg_name = (char *) addr;
	mhdr.msg_namelen = addrlen;
	mhdr.msg_iov = &iov;
	mhdr.msg_iovlen = 1;
	mhdr.msg_controllen = sizeof (chdr);
	iov.iov_base = buffer;
	iov.iov_len = buflen;
	cmsg = (struct cmsghdr *) chdr;
	cmsg->cmsg_len = sizeof (chdr);
	cmsg->cmsg_level = IPPROTO_IPV6;
	cmsg->cmsg_type = IPV6_PKTINFO;
	mhdr.msg_control = (void *) cmsg;
	pkt_info = (struct in6_pktinfo *) CMSG_DATA (cmsg);
	memset (pkt_info, 0, sizeof (struct in6_pktinfo));
	pkt_info->ipi6_ifindex = interface->index;

	if (!interface->link_local) {
	    trace (TR_ERROR, RIPNG->trace,
		   "No link-local address for %s\n", interface->name);
	}
	else {
	    trace (TR_PACKET, RIPNG->trace,
		   "source address should be %a\n",
		   interface->link_local->prefix);
	    IN6_ADDR_COPY (&pkt_info->ipi6_addr,
			   prefix_toaddr6 (interface->link_local->prefix));
	}
	if ((rc = sendmsg (sockfd, &mhdr, flag)) < 0) {
	    trace (TR_ERROR, RIPNG->trace, "sendmsg: %m\n");
	}
#else /* IPV6_PKTINFO */
	if (is_mc && sockfd == RIPNG->sockfd) {
	    /*
	     * choose outgoing intertface for multicast
	     */
/* INRIA and KAME requires an interface index instead */
	    if (setsockopt (sockfd, 
			    IPPROTO_IPV6, IPV6_MULTICAST_IF,
	    /* This way is the same with one used in IPv4 */
		 (const char *) prefix_tochar (interface->primary6->prefix),
			    sizeof (struct in6_addr)) < 0) {
		trace (TR_ERROR, RIPNG->trace,
		       "setsockopt IPV6_MULTICAST_IF for %s: %m\n",
		       interface->name);
	    }
	}

	if ((rc = sendto (sockfd, 
			  (char *) buffer, buflen, flag,
			  (struct sockaddr *) addr, addrlen)) < 0) {
	    trace (TR_ERROR, RIPNG->trace, "sendto: %m\n");
	}

#endif /* IPV6_PKTINFO */
    }
    trace (TR_PACKET, RIPNG->trace, "send %d bytes to %s on %s\n",
	   buflen, inet_ntop (AF_INET6, &addr->sin6_addr, tmp6, sizeof tmp6),
	   interface->name);
    return (rc);
}


/* 
 * use multicast for multicast-capable interfaces 
 * use unicast for p-to-p non-multicast-capable interfaces (non standard way)
 * if ll_prefixes is NULL, send a request for complete routing table 
 *    ll_prefixes ... not implemented yet and usual daemon doesn't need it
 */
void
ripng_send_request (rip_interface_t *rip_interface, LINKED_LIST * ll_prefixes)
{
#define RIPNG_HDRLEN 4
#define RIPNG_RTELEN 20
    u_char buffer[RIPNG_HDRLEN + RIPNG_RTELEN];
    u_char *cp, *start;
    struct sockaddr_in6 ripng;
    char tmp6[INET6_ADDRSTRLEN];
    interface_t *interface = rip_interface->interface;

    assert (ll_prefixes == NULL);	/* XXX not yet implemented */
    start = buffer;

    memset (&ripng, 0, sizeof (ripng));
    ripng.sin6_port = htons (RIPNG->port);
    ripng.sin6_family = AF_INET6;
    ripng.sin6_flowinfo = htonl (RIPNG_IPV6_PRIORITY);
#ifdef SIN6_LEN
    ripng.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */

    cp = buffer;
    memset (buffer, 0, sizeof (buffer));

    BGP_PUT_BYTE (RIPNG_REQUEST, cp);
    BGP_PUT_BYTE (RIPNG_VERSION, cp);

    BGP_PUT_SHORT (0, cp);	/* must be 0 */
    memset (cp, 0, 16);
    cp += 16;

    BGP_PUT_SHORT (0, cp);	/* must be 0 */
    BGP_PUT_BYTE (0, cp);	/* length 0 or 16?? spec is not clear */
    BGP_PUT_BYTE (RIP_METRIC_INFINITY, cp);	/* infinity metric */

	if (!BIT_TEST (interface->flags, IFF_MULTICAST)) {
	    struct in6_addr *dest;
	    /*
	     * The RIPng draft onlys say about multicast,
	     * so this way may work but non standard
	     */
	    if (!BIT_TEST (interface->flags, IFF_POINTOPOINT))
		return;

	    if ((dest = prefix_toaddr6 (
			       interface->link_local->broadcast)) == NULL) {
		/* shouldn't use global address, though */
		if ((dest = prefix_toaddr6 (
				    interface->primary->broadcast)) == NULL)
		    return;
	    }
	    IN6_ADDR_COPY (&ripng.sin6_addr, dest);
	}
	else {
	    IN6_ADDR_COPY (&ripng.sin6_addr, &all_ripng_routers);
	}

	trace (TR_TRACE, RIPNG->trace, "send request to %s on %s\n",
	       inet_ntop (AF_INET6, &ripng.sin6_addr, tmp6, sizeof tmp6),
	       interface->name);

	ripng_sendmsgto (rip_interface, buffer, (cp - start), 0,
			 &ripng, sizeof (ripng));
/*
    }
*/
}


/*
 * given an interface, broadcast ripng routes (according to policy)
 * we do this every RIPNG_UPDATE_INTERVAL
 * if change flag is set (==1) then only send routes marked as
 * changed
 */
void
ripng_send_routes (rip_interface_t * rip_interface, struct sockaddr_in6 *host,
		   int change)
{
    rip_route_t *route;
    u_char *data, *cp;
    struct sockaddr_in6 ripng;
    u_int routes = 0;		/* count of routes */
    u_int pdulen;
    int metric;			/* should be signed */
    char tag[MAXLINE];
    char tmp6[INET6_ADDRSTRLEN];
    LINKED_LIST *ll_tuple;
    tuple_t *tuple;
    interface_t *interface = rip_interface->interface;

    assert (interface);

    if (host) {
	memcpy (&ripng, host, sizeof (ripng));
#ifdef notdef
	char stmp[INET6_ADDRSTRLEN];
	trace (TR_TRACE, RIPNG->trace, "preparing for %s\n",
	       inet_ntop (AF_INET6, &host->sin6_addr, stmp, sizeof stmp));
#endif
    }
    else {
#ifdef notdef
	trace (TR_TRACE, RIPNG->trace, "preparing for %s\n", interface->name);
#endif
	memset (&ripng, 0, sizeof (ripng));
	ripng.sin6_port = htons (RIPNG->port);
	ripng.sin6_family = AF_INET6;
	ripng.sin6_flowinfo = htonl (RIPNG_IPV6_PRIORITY);
#ifdef SIN6_LEN
	ripng.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */

	if (BIT_TEST (interface->flags, IFF_MULTICAST)) {
	    IN6_ADDR_COPY (&ripng.sin6_addr, &all_ripng_routers);
	}
	else if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
	    struct in6_addr *dest;

	    if ((dest = prefix_toaddr6 (
			       interface->link_local->broadcast)) == NULL) {
		/* shouldn't use global address, though */
		if ((dest = prefix_toaddr6 (
				  interface->primary->broadcast)) == NULL) {
		    return;
		}
	    }
	    IN6_ADDR_COPY (&ripng.sin6_addr, dest);
	}
	else {
	    return;
	}
    }

#define IPV6_HDRLEN 40
#define UDP_HDRLEN 8
    pdulen = interface->mtu - IPV6_HDRLEN - UDP_HDRLEN;
    assert (pdulen >= RIPNG_HDRLEN + RIPNG_RTELEN);
    data = NewArray (u_char, pdulen);

    cp = data;
    memset (data, 0, pdulen);

    BGP_PUT_BYTE (RIPNG_RESPONSE, cp);
    BGP_PUT_BYTE (RIPNG_VERSION, cp);
    BGP_PUT_SHORT (0, cp);

    trace (TR_TRACE, RIPNG->trace, "send response to %s on %s\n",
	   inet_ntop (AF_INET6, &ripng.sin6_addr, tmp6, sizeof tmp6),
	   interface->name);

    /* make a list first and then send them to avoid waiting on sendto() 
       with locking the route table */
    ll_tuple = LL_Create (0);

    /* on flashing update, locking should be done outer side
       because of keeping change flags and reset them after sending out */

    HASH_Iterate (RIPNG->hash, route) {

	prefix_t *prefix = route->prefix;
	rip_attr_t *attr = route->active;

	/* doing ouput processing and only sending changed routes */
	if (change && (!BIT_TEST (route->flags, RT_RIP_CHANGE))) {
#ifdef notdef
	    trace (TR_PACKET, RIPNG->trace, "  x %p metric %d (unchange)\n",
		   prefix, attr->metric);
#endif
	    continue;
	}

	if ((metric = rip_policy (RIPNG, prefix, attr, rip_interface)) < 0)
	    continue;

	routes++;

	if (attr->tag) {
	    sprintf (tag, " tag 0x%lx", attr->tag);
	}
	else {
	    tag[0] = '\0';
	}
	trace (TR_PACKET, RIPNG->trace, "  o %p metric %d%s\n",
	       prefix, metric, tag);

	BGP_PUT_ADDR6 (prefix_tochar (prefix), cp);
	BGP_PUT_SHORT (attr->tag, cp);
	BGP_PUT_BYTE (prefix->bitlen, cp);
	BGP_PUT_BYTE (metric, cp);

	/* 
	 * see if we have filled up the buffer. If so, send packet and 
	 * create new buffer 
	 */
	if (cp - data > pdulen - RIPNG_RTELEN) {
	    tuple = New (tuple_t);
	    tuple->len = cp - data;
	    tuple->data = data;
	    LL_Add (ll_tuple, tuple);
	    data = NewArray (u_char, pdulen);
	    cp = data;
	    memset (data, 0, pdulen);
	    BGP_PUT_BYTE (RIPNG_RESPONSE, cp);
	    BGP_PUT_BYTE (RIPNG_VERSION, cp);
	    BGP_PUT_SHORT (0, cp);
	    routes = 0;
	}
    }

    /* okay, send packet (assuming we have one) */
    if (routes > 0) {
	tuple = New (tuple_t);
	tuple->len = cp - data;
	tuple->data = data;
	LL_Add (ll_tuple, tuple);
    }
    else {
	/* delete the last buffer created but no routes */
	Delete (data);
    }

    LL_Iterate (ll_tuple, tuple) {
	ripng_sendmsgto (rip_interface, tuple->data, tuple->len, 0,
			 &ripng, sizeof (ripng));
	Delete (tuple->data);
	Delete (tuple);
    }

    LL_Destroy (ll_tuple);
/*  Delete (tuple); */
}


static int
ripng_recvmsgfrom (int sockfd, u_char * buffer, int buflen, u_int flag,
	     struct sockaddr_in6 *from, int * fromlen, interface_t ** ifp)
{
    int rc;
    prefix_t *prefix;
    interface_t *interface = NULL;
    u_int index = 0;
    struct in6_addr dstaddr;
    int limit = 0;
    char tmp6[INET6_ADDRSTRLEN];

#ifndef SOLARIS_IPV6
    struct msghdr mhdr;
    struct cmsghdr *cmsg;
#ifdef IPV6_PKTINFO
    struct in6_pktinfo *pkt_info;
#endif
    struct iovec iov;
    char chdr[1024];

    mhdr.msg_name = (void *) from;
    mhdr.msg_namelen = *fromlen;
    memset (mhdr.msg_name, 0, mhdr.msg_namelen);
    mhdr.msg_iov = &iov;
    mhdr.msg_iovlen = 1;
    mhdr.msg_control = (void *) chdr;
    mhdr.msg_controllen = sizeof (chdr);
    memset (mhdr.msg_control, 0, mhdr.msg_controllen);
    iov.iov_base = buffer;
    assert (buflen >= INTERFACE_MASTER->max_mtu);
    iov.iov_len = buflen;

    rc = recvmsg (sockfd, &mhdr, flag);
    select_enable_fd (sockfd);
    if (rc <= 0) {
	/* zero is error? XXX */
	trace (TR_ERROR, RIPNG->trace, "recvmsg: %m\n");
	return (-1);
    }
    else {
	memset (&dstaddr, 0, sizeof (dstaddr));
/* fprintf(stderr, "mhdr.msg_control = 0x%x mhdr.msg_controllen = %d\n", 
   mhdr.msg_control, mhdr.msg_controllen); */
#ifdef __GLIBC__
/* There is a bug in GNU LIBC 2.0, so I define another one. */
#undef CMSG_NXTHDR
#define CMSG_NXTHDR(mhdr, cmsg) \
    (((unsigned char *)(cmsg) + (cmsg)->cmsg_len + sizeof(struct cmsghdr) > \
	(unsigned char *)(mhdr)->msg_control + (mhdr)->msg_controllen) ? \
	(struct cmsghdr *)NULL : \
        (struct cmsghdr *)((unsigned char *)(cmsg) \
         + (((cmsg)->cmsg_len + sizeof(long int)-1) & ~(sizeof(long int)-1))))
#endif /* __GLIBC__ */
	if (mhdr.msg_controllen > 0)
	    for (cmsg = CMSG_FIRSTHDR (&mhdr);
		 cmsg;
		 cmsg = CMSG_NXTHDR (&mhdr, cmsg)) {
/* fprintf(stderr, 
   "cmg=0x%x cmsg->cmsg_level = %d cmsg->cmsg_type = %d cmsg->cmsg_len = %d\n", 
   cmsg, cmsg->cmsg_level, cmsg->cmsg_type, cmsg->cmsg_len); */
		switch (cmsg->cmsg_type) {
#ifdef IPV6_PKTINFO
		case IPV6_PKTINFO:
		    pkt_info = (struct in6_pktinfo *) CMSG_DATA (cmsg);
		    index = pkt_info->ipi6_ifindex;
		    dstaddr = pkt_info->ipi6_addr;	/* struct copy */
		    trace (TR_PACKET, RIPNG->trace,
			   "recv IPV6_PKTINFO dest = %s index = %d\n",
			   inet_ntop (AF_INET6, (struct in6_addr *) &dstaddr,
				      tmp6, sizeof tmp6), index);
		    break;
#endif /* IPV6_PKTINFO */
#ifdef IPV6_RECVIF
		case IPV6_RECVIF:
		    index = *(u_short *) CMSG_DATA (cmsg);
		    trace (TR_PACKET, RIPNG->trace,
			   "recv IPV6_RECVIF index = %d\n", index);
		    break;
#else /* IPV6_RECVIF */
#ifdef IPV6_RECVINTERFACE
		case IPV6_RECVINTERFACE:
		    index = *(u_short *) CMSG_DATA (cmsg);
		    trace (TR_PACKET, RIPNG->trace,
			   "recv IPV6_RECVINTERFACE index = %d\n", index);
		    break;
#endif /* IPV6_RECVINTERFACE */
#ifdef IP_RECVINTERFACE
		case IP_RECVINTERFACE:
		    index = *(u_short *) CMSG_DATA (cmsg);
		    trace (TR_PACKET, RIPNG->trace,
			   "recv IP_RECVINTERFACE index = %d\n", index);
		    break;
#endif /* IP_RECVINTERFACE */
#endif /* IPV6_RECVIF */
		    /*
		     * XXX destination address of packet should be checked 
		     * RIPNG request may not be addressed with multicast addess
		     * it may be possible later
		     */
#ifdef IPV6_RECVDSTADDR
		case IPV6_RECVDSTADDR:
		    IN6_ADDR_COPY (&dstaddr,
				   (struct in6_addr *) CMSG_DATA (cmsg));
		    trace (TR_PACKET, RIPNG->trace,
			   "recv IPV6_RECVDSTADDR dest = %s\n",
			 inet_ntop (AF_INET6, &dstaddr, tmp6, sizeof tmp6));
		    break;
#else /* IPV6_RECVDSTADDR */
#ifdef IP_RECVDSTADDR
		case IP_RECVDSTADDR:
		    IN6_ADDR_COPY (&dstaddr,
				   (struct in6_addr *) CMSG_DATA (cmsg));
		    trace (TR_PACKET, RIPNG->trace,
			   "recv IP_RECVDSTADDR dest = %s\n",
			 inet_ntop (AF_INET6, &dstaddr, tmp6, sizeof tmp6));
		    break;
#endif /* IP_RECVDSTADDR */
#endif /* IPV6_RECVDSTADDR */
		    /*
		     * XXX it may be possible later checking hop limit
		     */
#ifdef IPV6_HOPLIMIT
		case IPV6_HOPLIMIT:
		    limit = *(u_char *) CMSG_DATA (cmsg);
		    trace (TR_PACKET, RIPNG->trace,
			   "recv IPV6_HOPLIMIT = %d\n", limit);
		    break;
#endif /* IPV6_HOPLIMIT */
#ifdef IPV6_UNICAST_HOPS
		case IPV6_UNICAST_HOPS:
		    limit = *(u_int *) CMSG_DATA (cmsg);
		    trace (TR_PACKET, RIPNG->trace,
			   "recv IPV6_HOPLIMIT = %d\n", limit);
		    break;
#endif /* IPV6_UNICAST_HOPS */
		default:
		    trace (TR_PACKET, RIPNG->trace,
			"recv unknown cmsg level = %d type = %d len = %d\n",
			 cmsg->cmsg_level, cmsg->cmsg_type, cmsg->cmsg_len);
		    break;
		}
	    }
    }
#else /* SOLARIS_IPV6 */
/* Solaris has IP_RECVDSTADDR, but differs. No way to know the interface */
    if ((rc = recvfrom (sockfd, (char *) buffer, buflen, flag,
			(struct sockaddr *) from, (int *) fromlen)) <= 0) {
	trace (TR_ERROR, RIPNG->trace, "recvfrom: %m\n");
        select_enable_fd (sockfd);
	return (-1);
    }
    select_enable_fd (sockfd);
#endif /* SOLARIS_IPV6 */

    prefix = New_Prefix (AF_INET6, &from->sin6_addr, 128);

    if (index > 0) {
	interface = find_interface_byindex (index);
	if (interface == NULL) {
	    trace (TR_PACKET, RIPNG->trace, "received response "
		   "on unknown interface from %a, ignore!\n", prefix);
	    Deref_Prefix (prefix);
	    return (-1);
	}
    }
    else {
	gateway_t *gateway;
	/*
	 * if the system doesn't supply the incoming interface index,
	 * try to find among configured interfaces and gateways registered,
	 */
	if ((interface = find_interface (prefix))) {
	    trace (TR_PACKET, RIPNG->trace,
		   "guesses %a on %s (find_interface)\n",
		   prefix, interface->name);
	}
	else if ((gateway = find_gateway (prefix)) &&
		 gateway->interface) {

	    interface = gateway->interface;
	    trace (TR_PACKET, RIPNG->trace,
		   "guesses %a on %s (find_gateway)\n",
		   prefix, interface->name);
	}
	else {
	    trace (TR_PACKET, RIPNG->trace,
		   "discard packet from %a (interface unknown)\n", prefix);
    	    rc = -1;
	}
    }
    *ifp = interface;
    Deref_Prefix (prefix);
    return (rc);
}


/* 
 * read and process and RIPNG update packet recieved on our interface
 */
void
ripng_receive_update (rip_interface_t *rip_interface)
{
    u_char *buffer, *cp = buffer;
    int n, fromlen;
    int command, version, zero;
    struct sockaddr_in6 from;
    prefix_t *prefix = NULL;
    gateway_t *gateway = NULL;
    interface_t *interface;
    int sockfd;

    sockfd = (rip_interface && rip_interface->sockfd >= 0)? 
		rip_interface->sockfd: RIPNG->sockfd;
    fromlen = sizeof (from);
    cp = buffer = NewArray (u_char, RIPNG_MIN_BUFSIZE);
    if ((n = ripng_recvmsgfrom (sockfd, 
			    buffer, RIPNG_MIN_BUFSIZE,
			    O_NONBLOCK, &from, &fromlen, &interface)) < 0) {
	goto ignore;
    }

    assert (interface);

    if (fromlen != sizeof (from) || from.sin6_family != AF_INET6) {
	trace (TR_ERROR, RIPNG->trace, "Error reading on socket\n");
	goto ignore;
    }

    /*
     * get command first for later check
     */
    BGP_GET_BYTE (command, cp);
    BGP_GET_BYTE (version, cp);
    BGP_GET_SHORT (zero, cp);

    prefix = New_Prefix (AF_INET6, &from.sin6_addr, 128);

    if (rip_interface == NULL) {
	trace (TR_TRACE, RIPNG->trace,
               "packet from %a at RIPNG socket %d\n", prefix, sockfd);
        rip_interface = RIPNG->rip_interfaces[interface->index];
	assert (rip_interface);
    }
    else if (rip_interface != RIPNG->rip_interfaces[interface->index]) {
	trace (TR_ERROR, RIPNG->trace,
               "confusion: from %a on %s but must be on %s\n",
               prefix, interface->name, rip_interface->interface->name);
        interface = rip_interface->interface;
    }

    if (!BITX_TEST (&RIPNG->interface_mask, interface->index)) {
	trace (TR_WARN, RIPNG->trace,
	       "packet from %a on disabled interface %s\n",
	       prefix, interface->name);
	goto ignore;
    }

    /*
     * common check for all commands
     */
    if (version != RIPNG_VERSION) {
	trace (TR_WARN, RIPNG->trace,
	       "unsupported version %d from %a on %s\n",
	       version, prefix, interface->name);
/* XXX  goto ignore; */
    }

    if (zero) {
	trace (TR_WARN, RIPNG->trace,
	       "non-zero pad field (value 0x%x) from %a on %s\n",
	       zero, prefix, interface->name);
/* XXX  goto ignore; */
    }

    if (command != RIPNG_RESPONSE && command != RIPNG_REQUEST) {
	trace (TR_WARN, RIPNG->trace,
	       "unsupported command %d from %a on %s, ignore!\n",
	       command, prefix, interface->name);
	goto ignore;
    }

    if (command == RIPNG_RESPONSE) {

	/* register the gateway */
	gateway = add_gateway (prefix, 0, interface);

	/* don't listen to things broadcast from our own interface */
	if (find_interface_local (prefix)) {
	    trace (TR_PACKET, RIPNG->trace,
		   "recv ignore own response from %a on %s, ignore!\n",
		   prefix, interface->name);
	    goto ignore;
	}

	if (IN6_IS_ADDR_V4MAPPED (&from.sin6_addr)) {
	    trace (TR_WARN, RIPNG->trace, "received response "
		   "with ipv4 source from %a on %s, ignore!\n",
		   prefix, interface->name);
	    goto ignore;
	}

	if (!IN6_IS_ADDR_LINKLOCAL (&from.sin6_addr)) {
	    trace (TR_WARN, RIPNG->trace,
		   "received response with non link-local source "
		   "from %a on %s, but continue\n", prefix, interface->name);
	}

	/* check if this interface is configured for RIPNG */
	if (!BITX_TEST (&RIPNG->interface_mask, interface->index)) {
	    trace (TR_PACKET, RIPNG->trace, "received response "
		   "from %a on unconfigured interface %s, ignore!\n",
		   gateway->prefix, interface->name);
	    goto ignore;
	}

	/*
	 * ID also suggests to check if the hop count is 255
	 */

	if (ntohs (from.sin6_port) != RIPNG->port) {
	    trace (TR_INFO, RIPNG->trace,
		   "non-standard source port %d from %a on %s, "
		   "ignored!\n", ntohs (from.sin6_port),
		   gateway->prefix, interface->name);
	    goto ignore;
	}

	if (interface->mtu < n + IPV6_HDRLEN + UDP_HDRLEN) {
	    trace (TR_WARN, RIPNG->trace,
		   "received packet size %d (+%d hdrs) exceeds mtu %d, "
		   "from %a on %s\n",
		   n, IPV6_HDRLEN + UDP_HDRLEN, interface->mtu,
		   gateway->prefix, interface->name);
	}
	else {
	    trace (TR_TRACE, RIPNG->trace,
		   "recv response %d bytes from %a on %s\n",
		   n, gateway->prefix, interface->name);
	}

	if (n - (cp - buffer)) {
	    LINKED_LIST *ll_prefixes, *ll_attr;

	    ll_prefixes = LL_Create (LL_DestroyFunction, Deref_Prefix, 0);
	    ll_attr = LL_Create (0);

	    /* munge the ripng packet 
	       and return list of prefixes and attributes */

	    ripng_process_packet_response (gateway, cp, n - (cp - buffer),
					   ll_prefixes, ll_attr,
		(rip_interface->default_pref >= 0)?
                        rip_interface->default_pref: RIPNG_PREF);

	    /* update our tables */
	    rip_process_update (RIPNG, ll_prefixes, ll_attr);

	    LL_Destroy (ll_prefixes);
	    LL_Destroy (ll_attr);
	}
    }
    else if (command == RIPNG_REQUEST) {

	/* register the gateway */
	if (ntohs (from.sin6_port) == RIPNG->port)
	    gateway = add_gateway (prefix, 0, interface);

	/* don't listen to things broadcast from our own interface
	   except for a query */
	if (find_interface_local (prefix) &&
	    ntohs (from.sin6_port) == RIPNG->port) {
	    goto ignore;
	}

	trace (TR_TRACE, RIPNG->trace,
	       "recv request %d bytes port %d from %a on %s\n",
	       n, ntohs (from.sin6_port), prefix, interface->name);

#ifdef notdef
	/* request from over a router will be rejected for now */
	if (!IN6_IS_ADDR_LINKLOCAL (&from.sin6_addr)) {
	    trace (TR_WARN, RIPNG->trace,
		   "received request "
		   "with non link-local source from %a on %s, ignore!\n",
		   prefix, interface->name);
	    goto ignore;
	}
#endif

	if (n - (cp - buffer) > 0) {
	    /* more than 1 entry */
	    if (ripng_process_packet_request (&from, cp, n - (cp - buffer),
					      rip_interface) >= 0) {
		cp = buffer;
		BGP_PUT_BYTE (RIPNG_RESPONSE, cp);
		BGP_PUT_BYTE (RIPNG_VERSION, cp);
		BGP_PUT_SHORT (0, cp);

		ripng_sendmsgto (rip_interface, buffer, n, 0, &from,
				 sizeof (from));
	    }
	    else {
		ripng_send_routes (rip_interface, &from, 0);
	    }
	    trace (TR_PACKET, RIPNG->trace,
		   "recv request answered to %a port %d on %s\n",
		   prefix, ntohs (from.sin6_port), interface->name);
	}
	else {
	    trace (TR_WARN, RIPNG->trace,
		   "recv request no entry from %a on %s, discard it!\n",
		   prefix, interface->name);
	}
    }

  ignore:
    if (prefix)
	Deref_Prefix (prefix);
    Delete (buffer);
}

#endif /* HAVE_IPV6 */
