/*
 * $Id: mrt.c,v 1.28 1999/04/29 02:26:27 masaki Exp $
 */

#include <string.h>
#include <time.h>
#include <signal.h>

#include <mrt.h>
#include <select.h>
#include <trace.h>
#include <timer.h>

mrt_t *MRT;


void
mrt_exit (status)
    int status;
{
    trace (INFO, MRT->trace, "exit (%d)\n", status);
    exit (status);
}
  

/* mrt_thread_create
 * a wrapper around pthread create, but also adds
 * thread to bookkeeping structures
 * Note: not locking down structure because (for moment) only main
 *       thread calls this routine. Maybe this will change.
 */

/* I'm changing a way to handle pthread_attr being passed to pthread_create ().
   man page doesn't say if I can destroy it right after pthread_create (),
   so I'm saving it into mrt_thread_t structure to delete it when exiting. */

mrt_thread_t *mrt_thread_create (char *name, schedule_t *schedule,
				 thread_fn_t call_fn, void *arg) 
{
  mrt_thread_t *mrt_thread;
  pthread_t thread;
#ifndef HAVE_LIBPTHREAD
  pthread_t save;
#endif /* HAVE_LIBPTHREAD */
  pthread_attr_t tattr;

  /* Locking is required to avoid a confusion 
     in case a created thread executes mrt_thread_exit()
     before adding a mrt_thread structure. */
  pthread_mutex_lock (&MRT->mutex_lock);

  pthread_attr_init (&tattr);
#ifdef HAVE_LIBPTHREAD
#ifndef __linux__
  /* as long as I remember, the following attributes are required
     to reuse a thread id. */
  pthread_attr_setscope (&tattr, PTHREAD_SCOPE_SYSTEM);
  pthread_attr_setdetachstate (&tattr, PTHREAD_CREATE_DETACHED);
#endif /* __linux__ */

  if (pthread_create (&thread, &tattr, (thread_fn_t) call_fn, arg) < 0) {
    trace (TR_ERROR, MRT->trace, "pthread_create %s (%m)\n", name);
    pthread_mutex_unlock (&MRT->mutex_lock);
    pthread_attr_destroy (&tattr);
    return (NULL);
  }

#else
  thread = ++MRT->threadn;
  /* a call should be delayed at the end */
#endif /* HAVE_LIBPTHREAD */

  mrt_thread = New (mrt_thread_t);
  /* this expects New () clears the memory */
  mrt_thread->name = (name)? strdup (name): NULL;
  mrt_thread->thread = thread;
  mrt_thread->schedule = schedule;
  mrt_thread->attr = tattr;
  LL_Add (MRT->ll_threads, mrt_thread);
  pthread_mutex_unlock (&MRT->mutex_lock);

  if (schedule)
     schedule->self = thread;
#ifndef HAVE_LIBPTHREAD
  save = set_thread_id (thread);
  (*call_fn) (arg);
  set_thread_id (save);
#endif /* HAVE_LIBPTHREAD */
  return (mrt_thread);
}


static void
mrt_thread_process_schedule (event_t *event)
{
    schedule_t *schedule = event->args[1];

    init_mrt_thread_signals ();
    if (event->call_fn)
	event->call_fn (event->args[0]);

    Deref_Event (event);
    if (schedule == NULL)
	mrt_thread_exit ();
#ifdef HAVE_LIBPTHREAD
    while (1)
        schedule_wait_for_event (schedule);
    /* NOT REACHED */
#else
    /* We should put the schedule into the global linked list here
       so that they will be processed by the main program sequentially.
       But, they are already on the list when it was created. */
#endif /* HAVE_LIBPTHREAD */
}


mrt_thread_t *
mrt_thread_create2 (char *name, schedule_t *schedule,
		    thread_fn_t init_fn, void *arg) 
{
    event_t *event;

    /* here I borrowed event structure 
       since pthread_create can pass only one argument */
    event = New_Event (2);
    event->call_fn = (event_fn_t) init_fn;
    event->args[0] = arg;
    event->args[1] = schedule;
    return (mrt_thread_create (name, schedule, 
			       (thread_fn_t) mrt_thread_process_schedule, 
			       event));
}


/*
 * mrt_thread_exit
 */
void mrt_thread_exit () {
  mrt_thread_t *mrt_thread;
  pthread_t thread = pthread_self ();

  assert (thread > 0);
  pthread_mutex_lock (&MRT->mutex_lock);
  LL_Iterate (MRT->ll_threads, mrt_thread) {
     if (mrt_thread->thread == thread) {
        LL_Remove (MRT->ll_threads, mrt_thread);
#ifdef HAVE_LIBPTHREAD
  	pthread_attr_destroy (&mrt_thread->attr);
#endif /* HAVE_LIBPTHREAD */
        Delete (mrt_thread->name);
        Delete (mrt_thread);
	break;
     }
  }
  pthread_mutex_unlock (&MRT->mutex_lock);

#ifdef HAVE_LIBPTHREAD
  pthread_exit (NULL);
#endif /* HAVE_LIBPTHREAD */
}


#ifndef HAVE_LIBPTHREAD
/* use real assert */
#undef assert
#include </usr/include/assert.h>
int
pthread_mutex_init (pthread_mutex_t *mutex, const pthread_mutexattr_t *attr)
{
    assert (mutex);
    *mutex = 0;
    return (0);
}

int
pthread_mutex_lock (pthread_mutex_t *mutex)
{
    assert (mutex);
    while (*mutex == 1)
        mrt_busy_loop (mutex, 1);
    *mutex = 1;
    return (0);
}

int
pthread_mutex_trylock (pthread_mutex_t *mutex)
{
    assert (mutex);
    if (*mutex == 0) {
	*mutex = 1;
	return (0);
    }
    return (1);
}

int
pthread_mutex_unlock (pthread_mutex_t *mutex)
{
    assert (mutex);
    /* assert (*mutex == 1); */
    *mutex = 0;
    return (0);
}

int
pthread_mutex_destroy (pthread_mutex_t *mutex)
{
    assert (mutex);
    return (0);
}

int 
pthread_cond_init (pthread_cond_t *cond, const pthread_condattr_t *attr)
{
    assert (cond);
    *cond = 0;
    return (0);
}

int 
pthread_cond_wait (pthread_cond_t *cond, pthread_mutex_t *mutex)
{
    assert (cond);
    assert (mutex);
    assert (*mutex == 1);
    while (*cond == 0) {
	pthread_mutex_unlock (mutex);
        mrt_busy_loop (cond, 0);
	pthread_mutex_lock (mutex);
    }
    return (0);
}

int 
pthread_cond_signal (pthread_cond_t *cond)
{
    assert (cond);
    /* assert (*cond == 0); */
    *cond = 1;
    return (0);
}

int 
pthread_cond_destroy (pthread_cond_t *cond)
{
    assert (cond);
    return (0);
}

#endif /* HAVE_LIBPTHREAD */


#ifndef HAVE_LIBPTHREAD
void mrt_busy_loop (int *force_exit_flag, int ok) {
  while (MRT->force_exit_flag == 0 && 
	    (force_exit_flag == NULL || *force_exit_flag == ok)) {
    /* process all first */
    while (MRT->force_exit_flag == 0 && 
		(force_exit_flag == NULL || *force_exit_flag == ok) &&
		process_all_schedules ());
    if (MRT->force_exit_flag == 0 &&
            (force_exit_flag == NULL || *force_exit_flag == ok))
        mrt_select ();
    if (MRT->force_exit_flag == 0 &&
            (force_exit_flag == NULL || *force_exit_flag == ok))
        mrt_alarm ();
  }
  if (MRT->force_exit_flag != 0) {
    trace (TR_WARN, MRT->trace, "MRT signal (%d) received\n", 
	   MRT->force_exit_flag);
  }
}
#endif /* HAVE_LIBPTHREAD */


int
mrt_update_pid (void)
{
    int pid = getpid ();
    if (MRT->pid != pid) {
	trace (TR_INFO, MRT->trace, "PID updated (%d -> %d)\n",
	      MRT->pid, pid);
	MRT->pid = pid;
    }
    return (pid);
}


void mrt_main_loop () {
    /* may someone forget this */
    mrt_update_pid ();
#ifndef HAVE_LIBPTHREAD
  mrt_busy_loop (NULL, 0);
#else
   /* starts select thread seperately */
   start_select ();
  /* loop */
  /* XXX we need to implement a mechanism to wait mrt threds */
  while (MRT->force_exit_flag == 0) {
    /* main thread catches alarm interrupts by sigwait() */
    mrt_alarm ();
  }
  if (MRT->force_exit_flag != 0) {
    trace (TR_WARN, MRT->trace, "MRT signal (%d) received\n", 
	   MRT->force_exit_flag);
  }
#endif /* HAVE_LIBPTHREAD */
}


/* init_mrt
 * initialize MRT bookeeping structures. *ALL* MRT programs must call this 
 * routine.
 */
#ifdef HAVE_THR_SETCONCURRENCY
#include <thread.h>
#endif /* HAVE_THR_SETCONCURRENCY */

int init_mrt (trace_t *tr) {
   mrt_thread_t *mrt_thread;

   assert (MRT == NULL);
#ifdef HAVE_THR_SETCONCURRENCY
   /* NOTE: Solaris Thread, not Pthread */
   thr_setconcurrency (40);
#endif /* HAVE_THR_SETCONCURRENCY */

   signal (SIGPIPE, mrt_process_signal);
   signal (SIGHUP, mrt_process_signal);
   signal (SIGINT, mrt_process_signal);

   MRT = New (mrt_t);
   MRT->start_time = time (NULL);
   MRT->ll_threads = LL_Create (0);
   MRT->ll_trace = LL_Create (0);
   MRT->ll_signal_call_fn = LL_Create (0);
   MRT->ll_gateways = LL_Create (0);
   MRT->trace = tr;
   MRT->config_file_name = NULL;
   MRT->pid = getpid ();
   MRT->version = MRT_VERSION;
   MRT->date = __DATE__;

   pthread_mutex_init (&MRT->mutex_lock, NULL);
   init_schedules (tr);

   mrt_thread = New (mrt_thread_t);
   mrt_thread->name = strdup ("MAIN thread");
   mrt_thread->schedule = NULL;
   mrt_thread->thread = pthread_self ();
   LL_Add (MRT->ll_threads, mrt_thread);

   /* only used on non-thread capable machines */
   MRT->threadn = pthread_self ();

   /* init_uii (tr); */
   init_timer (tr);
   init_select (tr);
#ifdef notdef
   /* some commands don't need information on own interfaces */
   init_interfaces (tr);
#endif

  return (1);
}
