/*
 * $Id: solaris.c,v 1.18 1999/04/30 09:56:37 masaki Exp $
 */

#include <fcntl.h>
#include <stropts.h>
#include <ctype.h>

#include <sys/stream.h>
#include <sys/tihdr.h>
#include <sys/tiuser.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <netinet/in.h>
#include <net/if.h>
#include <net/route.h>
#include <inet/common.h>
#include <inet/mib2.h>
#include <inet/ip.h>

#include <mrt.h>
#include <interface.h>

extern int getpagesize (void);

static int sockfd = -1;
#ifdef HAVE_IPV6
static int sockfd6 = -1;
#endif /* HAVE_IPV6 */

/* get interface configuration */
static interface_t *
ifstatus (char *name)
{
    struct ifreq ifr;
    int s;
    u_long flags, mtu;
    char *cp;
    interface_t *interface;

    safestrncpy (ifr.ifr_name, name, sizeof (ifr.ifr_name));

    if ((cp = strrchr (ifr.ifr_name, ':')))
	*cp = '\0';		/* Remove the :n extension from the name */

    if ((cp = strrchr (ifr.ifr_name, '#')))
	*cp = '\0';		/* Remove the # extension from the name */

    if ((interface = find_interface_byname (ifr.ifr_name)))
	return (interface);

    if ((s = sockfd) < 0)
	return (NULL);

    if (ioctl (s, SIOCGIFFLAGS, (caddr_t) & ifr) < 0) {
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFFLAGS for % (%s)\n",
	       name, strerror (errno));
	return (NULL);
    }
    assert (sizeof (ifr.ifr_flags) == 2);
    flags = ifr.ifr_flags & 0x0000ffff;		/* short */

    if (ioctl (s, SIOCGIFMTU, (caddr_t) & ifr) < 0) {
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "SIOCSIFMTU for % (%s)\n", name, strerror (errno));
	return (NULL);
    }

    if (strncmp (ifr.ifr_name, "ip", 2) == 0 && isdigit (ifr.ifr_name[2]))
	flags |= IFF_TUNNEL;
#define ifr_mtu ifr_metric
    mtu = ifr.ifr_mtu;
    name = ifr.ifr_name;

    interface = new_interface (name, flags, mtu, 0);
    return (interface);
}


/* get interface configuration for IPv4 */
static void
ifstatus_v4 (interface_t * interface, char *name)
{
    struct ifreq ifr;
    struct sockaddr_in addr, mask, dest;
    int s;
    u_long flags;

    if ((s = sockfd) < 0)
	return;

    safestrncpy (ifr.ifr_name, name, sizeof (ifr.ifr_name));
    flags = interface->flags;

    if (ioctl (s, SIOCGIFADDR, (caddr_t) & ifr) < 0) {
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFADDR for % (%s)\n",
	       name, strerror (errno));
	return;
    }
    memcpy (&addr, &ifr.ifr_addr, sizeof (addr));

    if (addr.sin_family != AF_INET || addr.sin_addr.s_addr == INADDR_ANY) {
#if 0
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFADDR returns strange address (family=%d)\n",
	       addr.sin_family);
#endif
	return;
    }

    if (ioctl (s, SIOCGIFNETMASK, (caddr_t) & ifr) < 0) {
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFNETMASK for % (%s)\n",
	       name, strerror (errno));
	/* sometimes, no netmask */
	memset (&ifr.ifr_addr, -1, sizeof (ifr.ifr_addr));
    }
    memcpy (&mask, &ifr.ifr_addr, sizeof (mask));

    if (BIT_TEST (flags, IFF_POINTOPOINT)) {
	if (ioctl (s, SIOCGIFDSTADDR, (caddr_t) & ifr) < 0) {
	    trace (TR_ERROR, INTERFACE_MASTER->trace,
		   "SIOCGIFDSTADDR for % (%s)\n",
		   name, strerror (errno));
	    /* sometimes, no destination address */
	    memset (&ifr.ifr_addr, 0, sizeof (ifr.ifr_addr));
	}
    }
    else if (BIT_TEST (flags, IFF_BROADCAST)) {
	if (ioctl (s, SIOCGIFBRDADDR, (caddr_t) & ifr) < 0) {
	    trace (TR_ERROR, INTERFACE_MASTER->trace,
		   "SIOCGIFBRDADDR for % (%s)\n",
		   name, strerror (errno));
	    /* sometimes, no broadcast address ??? */
	    memset (&ifr.ifr_addr, 0, sizeof (ifr.ifr_addr));
	}
    }
    memcpy (&dest, &ifr.ifr_addr, sizeof (dest));

    add_addr_to_interface (interface, AF_INET,
			   &addr.sin_addr.s_addr,
			   mask2len (&mask.sin_addr.s_addr, 4),
			   &dest.sin_addr.s_addr);
}


#ifdef HAVE_IPV6
/* get interface configuration for IPv6 */
static void
ifstatus_v6 (interface_t * interface, char *name)
{
    struct ifreq ifr;
    struct in6_addr addr, sin_zero;
    struct in6_addr sin, *dest = NULL;
    struct v6addrreq v6addrreq;
    struct v6maskreq v6maskreq;
    u_long flags, mask;
    int s;

    if ((s = sockfd6) < 0)
	return;

    memcpy (ifr.ifr_name, name, sizeof (ifr.ifr_name));
    flags = interface->flags;
    memcpy (v6addrreq.v6ar_name, name, sizeof (v6addrreq.v6ar_name));

    if (ioctl (s, SIOCGIFV6ADDR, (caddr_t) & v6addrreq) < 0) {
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFV6ADDR for % (%s)\n",
	       name, strerror (errno));
	return;
    }
    memcpy (&addr, &v6addrreq.v6ar_addr, sizeof (addr));

    /* not really an IPv6 address/interface */
    memset (&sin_zero, 0, sizeof (sin_zero));
    if (!memcmp (&addr, &sin_zero, sizeof (sin_zero))) {
#if 0
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "Interface %s doesn't have IPv6 address\n", name);
#endif
	return;
    }

    memcpy (v6maskreq.v6mr_name, name, sizeof (v6maskreq.v6mr_name));
    if (ioctl (s, SIOCGIFV6MASK, (caddr_t) & v6maskreq) < 0) {
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFV6MASK for % (%s)\n",
	       name, strerror (errno));
	return;
    }
    mask = v6maskreq.v6mr_mask;

    if (BIT_TEST (flags, IFF_POINTOPOINT)) {
	if (ioctl (s, SIOCGIFV6DST, (caddr_t) & v6addrreq) >= 0) {
	    dest = &sin;
	    memcpy (dest, &v6addrreq.v6ar_addr, sizeof (*dest));
	}
	else {
	    trace (TR_ERROR, INTERFACE_MASTER->trace,
		   "No destination? for %s\n", name);
	}
    }

    add_addr_to_interface (interface, AF_INET6,
			   (char *) &addr, mask, (char *) dest);
}

#endif /* HAVE_IPV6 */


static int
read_interface4 (void)
{
    struct ifconf ifc;
    struct ifreq *ifptr, *end;
    int num;
    char *name, *buffer;
    interface_t *interface;

    if ((sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "socket for AF_INET (%s)\n", strerror (errno));
	return (-1);
    }

    for (num = 64;; num = (num << 1)) {
	buffer = (char *) NewArray (struct ifreq, num);
	ifc.ifc_len = num * sizeof (struct ifreq);
	ifc.ifc_buf = buffer;

	if (ioctl (sockfd, SIOCGIFCONF, (char *) &ifc) < 0) {
	    trace (TR_ERROR, INTERFACE_MASTER->trace,
		   "SIOCGIFCONF (%s)\n", strerror (errno));
	    return (-1);
	}
	if (ifc.ifc_len < num * sizeof (struct ifreq))
	      break;
	Delete (buffer);
    }

    end = (struct ifreq *) (ifc.ifc_buf + ifc.ifc_len);
    ifptr = ifc.ifc_req;

    while (ifptr < end) {
	name = ifptr->ifr_name;
	ifptr++;
	interface = ifstatus (name);
	if (interface == NULL)
	    continue;
	ifstatus_v4 (interface, name);
    }
    Delete (buffer);
    return (1);
}


#ifdef HAVE_IPV6
static int
read_interface6 (void)
{
    struct ifconf ifc;
    struct v6conf *ifptr, *end;
    int num;
    char *name, *buffer;
    interface_t *interface;

    if ((sockfd6 = socket (AF_INET6, SOCK_DGRAM, 0)) < 0) {
	trace (TR_WARN, INTERFACE_MASTER->trace,
	       "socket for AF_INET6 (%s), IPv4 only\n",
	       strerror (errno));
	return (-1);
    }

    for (num = 64;; num = (num << 1)) {
	buffer = (char *) NewArray (struct v6conf, num);
	ifc.ifc_len = num * sizeof (struct v6conf);
	ifc.ifc_buf = buffer;

	if (ioctl (sockfd6, SIOCGIFV6CONF, (char *) &ifc) < 0) {
	    trace (TR_ERROR, INTERFACE_MASTER->trace,
		   "SIOCGIFV6CONF (%s)\n", strerror (errno));
	    Delete (buffer);
	    return (-1);
	}
	if (ifc.ifc_len < num * sizeof (struct v6conf))
	      break;
	Delete (buffer);
    }

    end = (struct v6conf *) (ifc.ifc_buf + ifc.ifc_len);
    ifptr = (struct v6conf *) ifc.ifc_req;

    while (ifptr < end) {
	name = ifptr->v6cf_name;
	ifptr++;
	interface = ifstatus (name);
	if (interface == NULL)
	    continue;
	ifstatus_v6 (interface, name);
    }
    Delete (buffer);
    return (1);
}
#endif /* HAVE_IPV6 */


int
read_interfaces (void)
{
    if (read_interface4 () < 0)
	return (-1);
#ifdef HAVE_IPV6
    read_interface6 ();
    /* don't care */
#endif /* HAVE_IPV6 */
    return (1);
}


int
sys_kernel_update_route (prefix_t * dest, prefix_t * next_hop,
		         prefix_t * old_hop, int index, int oldindex)
{
    int s;
    struct rtentry rt;
    struct sockaddr_in *dst = (struct sockaddr_in *) &rt.rt_dst;
    struct sockaddr_in *gateway = (struct sockaddr_in *) &rt.rt_gateway;
#ifdef HAVE_IPV6
    struct v6rtreq rt6;
#endif /* HAVE_IPV6 */
    int op;

    if (next_hop && old_hop) {
	sys_kernel_update_route (dest, NULL, old_hop, 0, oldindex);
	return (sys_kernel_update_route (dest, next_hop, NULL, index, 0));
    }

    if (dest->family == AF_INET) {

	if ((s = sockfd) < 0)
	    return (-1);

        if (next_hop) {
	    op = SIOCADDRT;
        }
        else if (old_hop) {
	    next_hop = old_hop;
	    index = oldindex;
	    op = SIOCDELRT;
        }

	memset (&rt, 0, sizeof (rt));

	dst->sin_family = AF_INET;
	memcpy (&dst->sin_addr, prefix_tochar (dest), sizeof (dst->sin_addr));

	gateway->sin_family = AF_INET;
	memcpy (&gateway->sin_addr, prefix_tochar (next_hop),
		sizeof (gateway->sin_addr));

	if (dest->bitlen == 32)
	    rt.rt_flags |= RTF_HOST;

	rt.rt_flags |= RTF_UP;
	if (gateway->sin_addr.s_addr != INADDR_ANY) {
	    rt.rt_flags |= RTF_GATEWAY;
	}
#ifdef notdef
	if (cmd == KERNEL_ROUTE_ADD) {
	    /* I'm not sure this does work -- masaki */
	    interface = find_interface_byindex (index);
	    inf.if_name = interface->name;
	    rt.rt_ifp = &inf;
	}
#endif

	if (ioctl (s, op, &rt) < 0) {
	    trace (TR_ERROR, MRT->trace, "%s (%s)\n",
		   (op == SIOCADDRT) ? "SIOCADDRT" : "SIOCDELRT",
		   strerror (errno));
	    return (-1);
	}

    }
#ifdef HAVE_IPV6
    else if (dest->family == AF_INET6) {

	if ((s = sockfd6) < 0)
	    return (-1);

        if (next_hop) {
	    op = SIOCADDV6RT;
        }
        else if (old_hop) {
	    next_hop = old_hop;
	    index = oldindex;
	    op = SIOCDELV6RT;
        }
	memset (&rt6, 0, sizeof (rt6));

	memcpy (&rt6.v6rt_dst, prefix_tochar (dest), sizeof (rt6.v6rt_dst));
	memcpy (&rt6.v6rt_gw, prefix_tochar (next_hop), sizeof (rt6.v6rt_gw));
	rt6.v6rt_mask = dest->bitlen;
	/* Solaris IPv6 seems to require this */
	netmasking (dest->family, &rt6.v6rt_dst, dest->bitlen);

	if (dest->bitlen == 128)
	    rt6.v6rt_flags |= RTF_HOST;

	rt6.v6rt_flags |= RTF_UP;
	if (!ipv6_any_addr (&rt6.v6rt_gw)) {
	    rt6.v6rt_flags |= RTF_GATEWAY;
	}

	if (op == SIOCADDV6RT) {
	    interface_t *interface;
	    /* I'm not sure this does work -- masaki */
	    interface = find_interface_byindex (index);
	    assert (interface);
	    safestrncpy (rt6.v6rt_ifname, interface->name,
		         sizeof (rt6.v6rt_ifname));
	    /* XXX dirty way! */
	    if (strchr (rt6.v6rt_ifname, '#') == NULL) {
		strcat (rt6.v6rt_ifname, "#v6");
	    }
	}

	if (ioctl (s, op, &rt6) < 0) {
	    trace (TR_ERROR, MRT->trace, "%s (%s)\n",
		   (op == SIOCADDV6RT) ? "SIOCADDV6RT" : "SIOCDELV6RT",
		   strerror (errno));
	    return (-1);
	}
    }
#endif /* HAVE_IPV6 */
    else {
	assert (0);		/* not a family we know about */
    }

    return (1);
}


#define DEV_IP "/dev/ip"
static int route_sockfd = -1;

int
kernel_init (void)
{
    if ((route_sockfd = open (DEV_IP, O_RDWR)) < 0) {
	trace (TR_ERROR, INTERFACE_MASTER->trace,
	       "open for %s (%s)\n", DEV_IP, strerror (errno));
	return (-1);
    }
    return (0);
}


#ifdef IRE_DEFAULT		/* This means Solaris 5.6 */
/* I'm not sure if they are compatible, though -- masaki */
#define IRE_GATEWAY IRE_DEFAULT
#define IRE_NET IRE_PREFIX
#define IRE_ROUTE IRE_CACHE
#define IRE_SUBNET IRE_IF_NORESOLVER
#define IRE_RESOLVER IRE_IF_RESOLVER
#define IRE_ROUTE_ASSOC IRE_HOST
#define IRE_ROUTE_REDIRECT IRE_HOST_REDIRECT
#endif /* IRE_DEFAULT */

static int
kernel_read_rt_table_v4 (void)
{
    int sd, flags;
    struct strbuf strbuf;
    struct T_optmgmt_req *tor;
    struct T_optmgmt_ack *toa;
    struct T_error_ack *tea;
    struct opthdr *req;
    int rc = -1;
    char *cp;

    if ((sd = route_sockfd) < 0)
	return (-1);

    strbuf.maxlen = getpagesize ();
    strbuf.buf = (char *) malloc (strbuf.maxlen);
    if (strbuf.buf == NULL) {
	goto finish;
    }
    tor = (struct T_optmgmt_req *) strbuf.buf;
    toa = (struct T_optmgmt_ack *) strbuf.buf;
    tea = (struct T_error_ack *) strbuf.buf;

    tor->PRIM_type = T_OPTMGMT_REQ;
    tor->OPT_offset = sizeof (struct T_optmgmt_req);
    tor->OPT_length = sizeof (struct opthdr);
#ifndef MI_T_CURRENT
/* I don't know what this means -- masaki */
#define MI_T_CURRENT    0x100
#endif /* MI_T_CURRENT */
    tor->MGMT_flags = MI_T_CURRENT;

    req = (struct opthdr *) (tor + 1);
    req->level = MIB2_IP;
    req->name = 0;
    req->len = 0;

    strbuf.len = tor->OPT_length + tor->OPT_offset;
    flags = 0;
    rc = putmsg (sd, &strbuf, (struct strbuf *) 0, flags);
    if (rc < 0)
	goto finish;

    req = (struct opthdr *) (toa + 1);

    for (;;) {
	flags = 0;
	rc = getmsg (sd, &strbuf, (struct strbuf *) 0, &flags);
	if (rc < 0)
	    goto finish;	/* this is EOD msg */
	if (rc == 0
	    && strbuf.len >= sizeof (struct T_optmgmt_ack)
	    && toa->PRIM_type == T_OPTMGMT_ACK
	    && toa->MGMT_flags == T_SUCCESS
	    && req->len == 0) {
	    rc = 1;
	    goto finish;	/* this is EOD msg */
	}
	if (strbuf.len >= sizeof (struct T_error_ack)
	    && tea->PRIM_type == T_ERROR_ACK) {
	    rc = -1;
	    goto finish;
	}
	if (rc != MOREDATA
	    || strbuf.len < sizeof (struct T_optmgmt_ack)
	    || toa->PRIM_type != T_OPTMGMT_ACK
	    || toa->MGMT_flags != T_SUCCESS) {
	    rc = -1;
	    goto finish;
	}
	if (req->level != MIB2_IP || req->name != MIB2_IP_21) {
	    do {
		rc = getmsg (sd, (struct strbuf *) 0, &strbuf, &flags);
	    } while (rc == MOREDATA);
	    continue;
	}
	strbuf.maxlen = (getpagesize () / sizeof (mib2_ipRouteEntry_t)) *
	    sizeof (mib2_ipRouteEntry_t);
	strbuf.len = 0;
	flags = 0;
	do {
	    rc = getmsg (sd, (struct strbuf *) 0, &strbuf, &flags);
	    if (rc < 0)
		goto finish;
	    if (rc == 0 || rc == MOREDATA) {
		mib2_ipRouteEntry_t *rp = (mib2_ipRouteEntry_t *) strbuf.buf;
		mib2_ipRouteEntry_t *lp =
		(mib2_ipRouteEntry_t *) (strbuf.buf + strbuf.len);

		do {
    		    interface_t *interface = NULL;
		    int proto = PROTO_KERNEL;
#ifdef notdef
fprintf(stderr, "re_ire_type=%x\n", rp->ipRouteInfo.re_ire_type);
fprintf(stderr, "dest=0x%x\n", rp->ipRouteDest);
fprintf(stderr, "nexthop=0x%x\n", rp->ipRouteNextHop);
fprintf(stderr, "mask=%d\n", mask2len (&rp->ipRouteMask, 4));
#endif
		    if (BIT_TEST (rp->ipRouteInfo.re_ire_type, 
			    IRE_BROADCAST|IRE_ROUTE_REDIRECT
			    |IRE_LOCAL|IRE_ROUTE /* I'm not sure */
			    /*|IRE_INTERFACE|IRE_LOOPBACK*/)) {
			continue;
/* loopback route is differnt from one by interface */
		    }
#ifdef notdef
		    /* net or host routes */
		    if (!BIT_TEST (rp->ipRouteInfo.re_ire_type,
				   IRE_NET | IRE_ROUTE_ASSOC)) {
			continue;
		    }
#endif /* notdef */
		    /* On Solaris, even an interface route have an next hop */
		    if (rp->ipRouteNextHop == INADDR_ANY)
			continue;

		    if (rp->ipRouteIfIndex.o_length > 0) {
		        /* XXX dirty way! */
		        if ((cp = strrchr (rp->ipRouteIfIndex.o_bytes, ':'))) {
			    *cp = '\0';
		        }
		        interface = find_interface_byname (
					       rp->ipRouteIfIndex.o_bytes);
		    }
		    if (interface == NULL) {
			prefix_t *prefix;
			prefix = New_Prefix (AF_INET, &rp->ipRouteNextHop, 32);
			interface = find_interface (prefix);
			Deref_Prefix (prefix);
		    }
		    if (interface == NULL) {
			char tmp4[64], tmp5[64];
			trace (TR_ERROR, INTERFACE_MASTER->trace,
	       		       "interface unknown %s gw %s on %s type %x\n", 
				inet_ntop (AF_INET, &rp->ipRouteDest, 
					   tmp4, sizeof (tmp4)),
				mask2len (&rp->ipRouteMask, 4),
				inet_ntop (AF_INET, &rp->ipRouteNextHop, 
					   tmp5, sizeof (tmp5)),
				(rp->ipRouteIfIndex.o_length > 0)?
				    rp->ipRouteIfIndex.o_bytes:"?",
				rp->ipRouteInfo.re_ire_type);
			continue;
		    }
		    /* to do the same thing with other platforms,
		       clear the next hop in case of interface routes */
		    if (!BIT_TEST (rp->ipRouteInfo.re_ire_type, IRE_GATEWAY) ||
		         BIT_TEST (rp->ipRouteInfo.re_ire_type, IRE_LOOPBACK)) {
			rp->ipRouteNextHop = INADDR_ANY;
			proto = PROTO_CONNECTED;
		    }
		    update_kernel_route ('A', AF_INET, &rp->ipRouteDest,
				      &rp->ipRouteNextHop,
				      mask2len (&rp->ipRouteMask, 4), 
				      interface->index, proto);
		} while (++rp < lp);
	    }
	} while (rc == MOREDATA);
	rc = 1;
    }
  finish:
    if (strbuf.buf)
	free (strbuf.buf);
    return (rc);
}


#ifdef HAVE_IPV6
static int
kernel_read_rt_table_v6 (void)
{
    int sd, flags;
    struct strbuf strbuf;
    struct T_optmgmt_req *tor;
    struct T_optmgmt_ack *toa;
    struct T_error_ack *tea;
    struct opthdr *req;
    int rc = -1;
    interface_t *interface;
    char *cp;

    if ((sd = route_sockfd) < 0)
	return (-1);

    strbuf.maxlen = getpagesize ();
    strbuf.buf = (char *) malloc (strbuf.maxlen);
    if (strbuf.buf == NULL) {
	goto finish;
    }
    tor = (struct T_optmgmt_req *) strbuf.buf;
    toa = (struct T_optmgmt_ack *) strbuf.buf;
    tea = (struct T_error_ack *) strbuf.buf;

    tor->PRIM_type = T_OPTMGMT_REQ;
    tor->OPT_offset = sizeof (struct T_optmgmt_req);
    tor->OPT_length = sizeof (struct opthdr);
    tor->MGMT_flags = MI_T_CURRENT;

    req = (struct opthdr *) (tor + 1);
    req->level = MIB2_IP6;
    req->name = 0;
    req->len = 0;

    strbuf.len = tor->OPT_length + tor->OPT_offset;
    flags = 0;
    rc = putmsg (sd, &strbuf, (struct strbuf *) 0, flags);
    if (rc < 0)
	goto finish;

    req = (struct opthdr *) (toa + 1);

    for (;;) {
	flags = 0;
	rc = getmsg (sd, &strbuf, (struct strbuf *) 0, &flags);
	if (rc < 0)
	    goto finish;	/* this is EOD msg */
	if (rc == 0
	    && strbuf.len >= sizeof (struct T_optmgmt_ack)
	    && toa->PRIM_type == T_OPTMGMT_ACK
	    && toa->MGMT_flags == T_SUCCESS
	    && req->len == 0) {
	    rc = 1;
	    goto finish;	/* this is EOD msg */
	}
	if (strbuf.len >= sizeof (struct T_error_ack)
	    && tea->PRIM_type == T_ERROR_ACK) {
	    rc = -1;
	    goto finish;
	}
	if (rc != MOREDATA
	    || strbuf.len < sizeof (struct T_optmgmt_ack)
	    || toa->PRIM_type != T_OPTMGMT_ACK
	    || toa->MGMT_flags != T_SUCCESS) {
	    rc = -1;
	    goto finish;
	}
	if (req->level != MIB2_IP6 || req->name != MIB2_IP_23) {
	    do {
		rc = getmsg (sd, (struct strbuf *) 0, &strbuf, &flags);
	    } while (rc == MOREDATA);
	    continue;
	}
	strbuf.maxlen = (getpagesize () / sizeof (mib2_ip6RouteEntry_t)) *
	    sizeof (mib2_ip6RouteEntry_t);
	strbuf.len = 0;
	flags = 0;
	do {
	    rc = getmsg (sd, (struct strbuf *) 0, &strbuf, &flags);
	    if (rc < 0)
		goto finish;
	    if (rc == 0 || rc == MOREDATA) {
		mib2_ip6RouteEntry_t *rp = (mib2_ip6RouteEntry_t *) strbuf.buf;
		mib2_ip6RouteEntry_t *lp =
		(mib2_ip6RouteEntry_t *) (strbuf.buf + strbuf.len);
		do {
		    int proto = PROTO_KERNEL;

		    if (BIT_TEST (rp->ip6RouteInfo.re_ire_type, 
			    IRE_BROADCAST|IRE_ROUTE_REDIRECT
			    |IRE_LOCAL|IRE_ROUTE /* I'm not sure */
			    /* |IRE_INTERFACE|IRE_LOOPBACK */)) {
			continue;
		    }
#ifdef notdef
fprintf(stderr, "re_ire_type=%x\n", rp->ip6RouteInfo.re_ire_type);
		    /* net or host routes */
		    if (!BIT_TEST (rp->ip6RouteInfo.re_ire_type,
				   IRE_NET | IRE_ROUTE_ASSOC)) {
			continue;
		    }
#endif
		    if (IN6_IS_ADDR_UNSPECIFIED (&rp->ip6RouteNextHop))
			continue;

		    /* There are strange ::/0 routes for resolving ll addrs */
		    if (IN6_IS_ADDR_UNSPECIFIED (&rp->ip6RouteDest) &&
			    mask2len ((char *) &rp->ip6RouteMask, 16) == 0)
			continue;

		    if (rp->ip6RouteIfIndex.o_length > 0) {
		        /* XXX dirty way! */
		        if ((cp = strrchr (rp->ip6RouteIfIndex.o_bytes, ':'))) {
			    *cp = '\0';
		        }
		        if ((cp = strrchr (rp->ip6RouteIfIndex.o_bytes, '#'))) {
			    *cp = '\0';
		        }
		        interface = find_interface_byname (
					       rp->ip6RouteIfIndex.o_bytes);
		    }
		    if (interface == NULL) {
			prefix_t *prefix;
			prefix = New_Prefix (AF_INET6, &rp->ip6RouteNextHop, 
					     128);
			interface = find_interface (prefix);
			Deref_Prefix (prefix);
		    }
		    if (interface == NULL) {
			char tmp4[64], tmp5[64];
			trace (TR_ERROR, INTERFACE_MASTER->trace,
	       		       "interface unknown %s gw %s on %s type %x\n", 
			       	inet_ntop (AF_INET6, &rp->ip6RouteDest, 
					  tmp4, sizeof (tmp4)),
				mask2len (&rp->ip6RouteMask, 4),
				inet_ntop (AF_INET6, &rp->ip6RouteNextHop, 
					  tmp5, sizeof (tmp5)),
					  (rp->ip6RouteIfIndex.o_length > 0)?
					      rp->ip6RouteIfIndex.o_bytes:"?",
				rp->ip6RouteInfo.re_ire_type);
			continue;
		    }
		    /* IRE_GATEWAY doesn't seem to be used in IPv6 */
		    if (BIT_TEST (rp->ip6RouteInfo.re_ire_type, IRE_RESOLVER) ||
		        BIT_TEST (rp->ip6RouteInfo.re_ire_type, IRE_LOOPBACK)) {
			memset (&rp->ip6RouteNextHop, 0, 16);
			proto = PROTO_CONNECTED;
		    }

		    update_kernel_route ('A', AF_INET6, &rp->ip6RouteDest,
				      &rp->ip6RouteNextHop,
				  mask2len ((char *) &rp->ip6RouteMask, 16),
				      interface->index, proto);
		} while (++rp < lp);
	    }
	} while (rc == MOREDATA);
    }
  finish:
    if (strbuf.buf)
	free (strbuf.buf);
    return (rc);
}
#endif /* HAVE_IPV6 */


int
sys_kernel_read_rt_table (void)
{
    kernel_read_rt_table_v4 ();
#ifdef HAVE_IPV6
    kernel_read_rt_table_v6 ();
#endif /* HAVE_IPV6 */
    return (1);
}
