/* 
 * $Id: commconf.c,v 1.13 1999/04/30 23:48:11 masaki Exp $
 */

#include <ctype.h>
#include <mrt.h>
#include <rip.h>
#include <config_file.h>
#include <protoconf.h>

config_mrtd_t *CONFIG_MRTD;
int BGPSIM_TRANSPARENT = 0; /* should be zero */


static void
get_config_ip_route (static_route_t * static_route)
{
    char options[MAXLINE], *cp = options;

    strcpy (cp, "");
    if (static_route->pref >= 0) {
	sprintf (cp, " %d", static_route->pref);
	cp += strlen (cp);
    }
    if (static_route->interface) {
	sprintf (cp, " %s", static_route->interface->name);
	cp += strlen (cp);
    }
    config_add_output ("route %s %s%s\n",
		       prefix_toax (static_route->prefix),
		       prefix_toa (static_route->nexthop),
		       options);
}


/* 
 * config a static ip route
 */
static int
config_ip_route (uii_connection_t * uii, prefix_t * prefix,
		 prefix_t * nexthop, char *options)
{
    static_route_t *static_route;
    interface_t *interface = NULL;
    char name[MAXLINE];
    int pref = -1;
    generic_attr_t *attr;

#ifdef notdef
    LL_Iterate (CONFIG_MRTD->ll_static_routes, static_route) {
	if (prefix_compare (prefix, static_route->prefix)) {
    	    config_notice (TR_PARSE, uii,
		   	    "static route %s already defined\n",
		   	    prefix_toax (prefix));
	    Deref_Prefix (prefix);
	    Deref_Prefix (nexthop);
	    if (options) Delete (options);
	    return (-1);
	}
    }
#endif

    strcpy (name, "");
    if (options) {
	if (parse_line (options, "%n %d", name, &pref) <= 0) {
	    if (parse_line (options, "%d %n", &pref, name) <= 0) {
	        Deref_Prefix (prefix);
	        Deref_Prefix (nexthop);
		Delete (options);
	    	return (-1);
	    }
	}
	if (name[0]) {
	    interface = find_interface_byname (name);
	    if (interface == NULL) {
    	        config_notice (TR_ERROR, uii, "interface %s not found\n", name);
	        Deref_Prefix (prefix);
	        Deref_Prefix (nexthop);
		Delete (options);
		return (-1);
	    }
	}
    }

    if (prefix_is_unspecified (nexthop) && interface == NULL) {
        config_notice (TR_ERROR, uii,
		   "null nexthop require interface name\n");
	Deref_Prefix (prefix);
        Deref_Prefix (nexthop);
	if (options) Delete (options);
	return (-1);
    }
#ifdef HAVE_IPV6
    if (prefix_is_linklocal (nexthop) && interface == NULL) {
        config_notice (TR_ERROR, uii,
		   "link_local nexthop require interface name\n");
	Deref_Prefix (prefix);
        Deref_Prefix (nexthop);
	if (options) Delete (options);
	return (-1);
    }
#endif /* HAVE_IPV6 */

    if (name[0])
        config_notice (TR_PARSE, uii,
		   "static route %s via %s pref %d on %s\n",
		   prefix_toax (prefix), prefix_toa (nexthop),
		   (pref >= 0) ? pref : STATIC_PREF, name);
    else
        config_notice (TR_PARSE, uii,
		   "static route %s via %s pref %d\n",
		   prefix_toax (prefix), prefix_toa (nexthop),
		   (pref >= 0) ? pref : STATIC_PREF);

    if (interface == NULL)
	interface = find_interface (nexthop);
    attr = New_Generic_Attr (PROTO_STATIC);
    attr->nexthop = add_nexthop (nexthop, interface);
    attr->gateway = add_gateway (nexthop, 0, interface);
    if (MRT->rib_update_route)
        MRT->rib_update_route (prefix, attr, NULL,
                               (pref >= 0)? pref: STATIC_PREF, 0);
    Deref_Generic_Attr (attr);

    /* XXX error handling ? */

    static_route = New (static_route_t);
    static_route->prefix = prefix;
    static_route->nexthop = nexthop;
    static_route->interface = (name[0])?interface:NULL;
    static_route->pref = pref;
    LL_Add (CONFIG_MRTD->ll_static_routes, static_route);
    config_add_module (0, "route", get_config_ip_route, static_route);
    if (options) Delete (options);

    return (1);
}


static int
no_config_ip_route (uii_connection_t * uii, prefix_t * prefix)
{
    generic_attr_t *attr;
    static_route_t *static_route;

    if (CONFIG_MRTD->ll_static_routes == NULL) {
	config_notice (TR_PARSE, uii, "No matching route to delete for %s\n",
		       prefix_toax (prefix));
	Deref_Prefix (prefix);
	return (0);
    }

    LL_Iterate (CONFIG_MRTD->ll_static_routes, static_route) {
	if (prefix_compare (static_route->prefix, prefix)) {

    	    attr = New_Generic_Attr (PROTO_STATIC);
    	    attr->nexthop = add_nexthop (static_route->nexthop, 
					 static_route->interface);
    	    attr->gateway = add_gateway (static_route->nexthop, 0, 
					 static_route->interface);
	    if (MRT->rib_update_route)
    	        MRT->rib_update_route (prefix, NULL, attr, 0, 0);
    	    Deref_Generic_Attr (attr);
	    /* XXX error handling ? */

	    config_del_module (0, "route", get_config_ip_route,
			       static_route);
	    config_notice (TR_PARSE, uii,
			   "CONFIG static route %s via %s being deleted\n",
			   prefix_toax (prefix),
			   prefix_toa (static_route->nexthop));
	    Deref_Prefix (static_route->prefix);
	    Deref_Prefix (static_route->nexthop);
	    Delete (static_route);
	    LL_Remove (CONFIG_MRTD->ll_static_routes, static_route);
	    Deref_Prefix (prefix);
	    return (1);
	}
    }
    config_notice (TR_PARSE, uii, "No matching route to delete");
    return (0);
}


/*
 * start interface configuration 
 */
static void
get_config_interface (interface_t * interface)
{
    config_add_output ("interface %s\n", interface->name);
}


static int
config_interface (uii_connection_t * uii, char *name)
{
    interface_t *interface, *ip;

    if ((interface = find_interface_byname (name)) == NULL) {
	config_notice (TR_ERROR, uii,
		       "INTERFACE %s not found\n", name);
	Delete (name);
	return (-1);
    }
    config_notice (TR_TRACE, uii, "INTERFACE %s\n", name);
    CONFIG_MRTD->interface = interface;
    uii->previous[++uii->prev_level] = uii->state;
    uii->state = UII_CONFIG_INTERFACE;
    Delete (name);

    LL_Iterate (CONFIG_MRTD->ll_interfaces, ip) {
	if (ip == interface)
	    return (0);
    }

    LL_Add (CONFIG_MRTD->ll_interfaces, interface);
    /* this is a memory to avoid duplication of config_add_module */
    config_add_module (CF_DELIM, "interface", get_config_interface, interface);
    return (1);
}


static void
get_config_interface_gateway (prefix_t * prefix)
{
    config_add_output ("  gateway %s\n", prefix_toa (prefix));
}


static int
config_interface_gateway (uii_connection_t * uii, prefix_t * prefix)
{
    config_notice (TR_TRACE, uii,
		   "gateway %s\n", prefix_toa (prefix));
    add_gateway (prefix, 0, CONFIG_MRTD->interface);
    config_add_module (0, "gateway", get_config_interface_gateway, prefix);
    return (1);
}


extern int config_bgp_router_id (uii_connection_t * uii, prefix_t *prefix);

static int
config_router_id (uii_connection_t * uii, prefix_t *prefix)
{
    /* this command has to be obsolete since it requires BGP module */
    return (config_bgp_router_id (uii, prefix));
}


static void
get_config_gateway (gateway_t *gateway)
{
    config_add_output ("gateway %s on %s\n", prefix_toa (gateway->prefix),
		       gateway->interface->name);
}


static int
config_gateway (uii_connection_t * uii, prefix_t *prefix, char *name)
{
    interface_t *interface;
    gateway_t *gateway;

    interface = find_interface_byname (name);
    if (interface == NULL) {
        config_notice (TR_ERROR, uii,
		       "interface %s not found\n", name);
	Delete (name);
	return (-1);
    }
    config_notice (TR_TRACE, uii, "gateway %s on %s\n", prefix_toa (prefix),
		   interface->name);
    gateway = add_gateway (prefix, 0, interface);
    config_add_module (0, "gateway", get_config_gateway, gateway);
    Delete (name);
    Deref_Prefix (prefix);
    return (1);
}


#ifdef notdef
int
config_router_network_weight (uii_connection_t * uii, char *name_or_prefix,
			      int weight)
{
    prefix_t *prefix;
    char name[MAXLINE];
    interface_t *interface;
    LINKED_LIST *ll;

    if (CONFIG_MRTD->protocol != PROTO_RIP && 
	CONFIG_MRTD->protocol != PROTO_RIPNG &&
	CONFIG_MRTD->protocol != PROTO_BGP) {
	Delete (name_or_prefix);
	return (-1);
    }

    if (parse_line (name_or_prefix, "%m", &prefix) >= 1) {

	Delete (name_or_prefix);

	if (CONFIG_MRTD->protocol == PROTO_BGP)
	    LL_Add (BGP->ll_networks, Ref_Prefix (prefix));
	if (CONFIG_MRTD->protocol == PROTO_RIP)
	    LL_Add (RIP->ll_networks, Ref_Prefix (prefix));
#ifdef HAVE_IPV6
	if (CONFIG_MRTD->protocol == PROTO_RIPNG)
	    LL_Add (RIPNG->ll_networks, Ref_Prefix (prefix));
#endif /* HAVE_IPV6 */
	if (CONFIG_MRTD->protocol == PROTO_BGP) {
	    Deref_Prefix (prefix);
	    return (1);
	}

	if ((ll = find_network (prefix)) != NULL) {
	    LL_Iterate (ll, interface) {
		switch (CONFIG_MRTD->protocol) {
		case PROTO_RIP:
		    if (BITX_TEST (&RIP->interface_mask, interface->index))
			break;
		    BITX_SET (&RIP->interface_mask, interface->index);
		    interface->default_pref = weight;
		    config_notice (TR_TRACE, uii,
				   "CONFIG RIP on interface %s weight %d\n",
				   interface->name, weight);
		    break;
#ifdef HAVE_IPV6
		case PROTO_RIPNG:
		    if (BITX_TEST (&RIPNG->interface_mask, interface->index))
			break;
		    if (!BIT_TEST (interface->flags, IFF_MULTICAST) &&
			BIT_TEST (interface->flags, IFF_POINTOPOINT) &&
			(interface->link_local == NULL ||
			 interface->link_local->broadcast == NULL)) {
			config_notice (TR_ERROR, uii,
				     "RIPNG on interface %s ignored "
			    "due to no multicast or link-local dest addr\n",
				       interface->name);
			break;
		    }
		    BITX_SET (&RIPNG->interface_mask, interface->index);
		    interface->default_pref = weight;
		    config_notice (TR_TRACE, uii,
				 "CONFIG RIPNG on interface %s weight %d\n",
				   interface->name, weight);
		    break;
#endif /* HAVE_IPV6 */
		default:
		    assert (0);
		    break;
		}
	    }
	    LL_Destroy (ll);
	    Deref_Prefix (prefix);
	    return (1);
	}
	/* else {
	    Deref_Prefix (prefix);
	    return (-1);
	} */
    }
    else if (parse_line (name_or_prefix, "%s", name) >= 1) {

	Delete (name_or_prefix);
/*
 * This part must be the last otherwise some ipv6 address will be mistook 
 */
	if ((interface = find_interface_byname (name)) == NULL) {
	    config_notice (TR_ERROR, uii,
		   "could not find interface %s\n", name);
	    return (-1);
	}

	if (CONFIG_MRTD->protocol == PROTO_RIP) {
	    BITX_SET (&RIP->interface_mask, interface->index);
	    interface->default_pref = weight;
	    config_notice (TR_TRACE, uii,
			   "CONFIG RIP on interface %s\n", interface->name);
	    LL_Add (RIP->ll_networks, Ref_Prefix (interface->primary->prefix));
	    return (1);
	}
#ifdef HAVE_IPV6
	else if (CONFIG_MRTD->protocol == PROTO_RIPNG) {
	    if (!BITX_TEST (&RIPNG->interface_mask, interface->index)) {
		if (!BIT_TEST (interface->flags, IFF_MULTICAST) &&
		    BIT_TEST (interface->flags, IFF_POINTOPOINT) &&
		    (interface->link_local == NULL ||
		     interface->link_local->broadcast == NULL)) {
		    config_notice (TR_ERROR, uii,
				   "RIPNG on interface %s ignored "
			    "due to no multicast or link-local dest addr\n",
				   interface->name);
		}
		else {
	    	    BITX_SET (&RIPNG->interface_mask, interface->index);
		    interface->default_pref = weight;
		    config_notice (TR_TRACE, uii,
				   "CONFIG RIPNG on interface %s\n",
				   interface->name);
	            LL_Add (RIPNG->ll_networks, Ref_Prefix (interface->primary6->prefix));
		}
	    }
	    return (1);
	}
#endif /* HAVE_IPV6 */
	else if (CONFIG_MRTD->protocol == PROTO_BGP) {
	    /* not supported */
	    return (-1);
	}
	else {
	    /* should not happen */
	    assert (0);
	}
    }
    else {
	Delete (name_or_prefix);
	return (-1);
    }
    assert (0);
    /* NOT REACHED */
    return (-1); /* shut up the compiler */
}
#endif


typedef struct _string2proto_table_t {
    char *name;
    int proto;
} string2proto_table_t;

string2proto_table_t string2proto_table[] = {
    { "kernel", PROTO_KERNEL},
    {"static", PROTO_STATIC},
    {"direct", PROTO_CONNECTED}, /* for compatibility */
    {"connected", PROTO_CONNECTED},
    {"rip", PROTO_RIP},
    {"ospf", PROTO_OSPF},
    {"bgp", PROTO_BGP},
    {"ripng", PROTO_RIPNG},
    {NULL, 0},
};


int
string2proto (char *proto_string)
{
    string2proto_table_t *table;

    for (table = string2proto_table; table->name; table++) {
	if (strcasecmp (table->name, proto_string) == 0)
	    return (table->proto);
    }
    return (-1);
}


static int
config_mrt_reboot (uii_connection_t *uii)
{
#if defined (__linux__) && defined (HAVE_LIBPTHREAD)
    /* I couldn't figure out how I can do this */
    config_notice (TR_ERROR, uii,
        "This command is unavailable on Linux with Pthread\n");
    return (0);
#else
    uii_send_data (uii, "Are you sure (yes/no)? ");
    if (uii_yes_no (uii)) {
        mrt_reboot ();
        /* never returns */
    }
    return (1);
#endif
}


int
init_mrtd_config (trace_t * trace)
{
    CONFIG_MRTD = New (config_mrtd_t);
    CONFIG_MRTD->ll_static_routes = NULL;
    CONFIG_MRTD->interface = NULL;
    CONFIG_MRTD->ll_interfaces = LL_Create (0);
    CONFIG_MRTD->ll_static_routes = LL_Create (0);

    set_uii (UII, UII_PROMPT, UII_UNPREV, "Password: ", 0);
    set_uii (UII, UII_PROMPT, UII_NORMAL, "MRTd> ", 0);
    set_uii (UII, UII_PROMPT, UII_ENABLE, "MRTd# ", 0);
    set_uii (UII, UII_PROMPT, UII_CONFIG, "Config> ", 0);
    set_uii (UII, UII_PROMPT, UII_CONFIG_INTERFACE, "Interface> ", 0);
    set_uii (UII, UII_PROMPT, UII_CONFIG_LINE, "Line> ", 0);

    /* Interactive commands */

    uii_add_command2 (UII_NORMAL, COMMAND_NODISPLAY, "show ip", 
		      show_ip_routes,
		      "Show the central routing table");
    uii_add_command2 (UII_NORMAL, 0, "show ip routes", 
		      show_ip_routes,
		      "Show the central routing table");
    uii_add_command2 (UII_NORMAL, COMMAND_NODISPLAY, "show rib routes", 
		      show_ip_routes, "Show the central routing table");
    uii_add_command2 (UII_NORMAL, COMMAND_NODISPLAY, "show rib", 
		      show_rib_status,
		      "Show the central routing status");

    uii_add_command2 (UII_NORMAL, 0, "show interfaces",
		      show_interfaces, "Show all interfaces available");

    uii_add_command2 (UII_ENABLE, 0, "reboot", config_mrt_reboot,
		      "Reboot MRTd");

#ifdef HAVE_IPV6
    uii_add_command2 (UII_NORMAL, COMMAND_NODISPLAY, "show ipv6 routes", 
		      show_ipv6_routes,
		      "Show the central routing status");
    uii_add_command2 (UII_NORMAL, COMMAND_NODISPLAY, "show rib6", 
		      show_rib_status,
		      "Show the central routing status");
    uii_add_command2 (UII_NORMAL, COMMAND_NODISPLAY, "show ipv6", 
		      show_ipv6_routes,
		      "Show the central routing status");
    uii_add_command2 (UII_NORMAL, COMMAND_NODISPLAY, "show rib6 routes", 
		      show_ipv6_routes, "Show the central routing status");
#endif /* HAVE_IPV6 */

    uii_add_command2 (UII_CONFIG, 0, "route %m %M %S",
		      config_ip_route, "Add a static route");
    uii_add_command2 (UII_CONFIG, 0, "ip route %m %M %S",
		      config_ip_route, "Add a static route");
    uii_add_command2 (UII_CONFIG, 0, "no route %m %M",
		      no_config_ip_route, "Delete static route");
    uii_add_command2 (UII_CONFIG, 0, "no ip route %m %M",
		      no_config_ip_route, "Delete static route");
#ifdef HAVE_IPV6
    uii_add_command2 (UII_CONFIG, 0, "ipv6 route %P %A %S",
		      config_ip_route, "Add a static route");
    uii_add_command2 (UII_CONFIG, 0, "no ipv6 route %P %A",
		      no_config_ip_route, "Delete static route");
#endif /* HAVE_IPV6 */

    uii_add_command2 (UII_CONFIG, 0, "interface %s",
		      config_interface, "Define interface");
    uii_add_command2 (UII_CONFIG_INTERFACE, 0, "gateway %M",
		      config_interface_gateway, 
		      "Define gateways on the interface");

#ifndef HAVE_IPV6
    uii_add_command2 (UII_ENABLE, 0, "trace rib (*|inet)", 
		      trace_rib, "Enable trace rib");

    uii_add_command2 (UII_ENABLE, 0, "no trace rib (*|inet)", 
		      no_trace_rib, "Disable trace rib");
#else
    uii_add_command2 (UII_ENABLE, 0, "trace rib (*|inet|inet6)", 
		      trace_rib, "Enable trace rib");
    uii_add_command2 (UII_ENABLE, 0, "no trace rib (*|inet|inet6)", 
		      no_trace_rib, "Disable trace rib");
#endif /* HAVE_IPV6 */
    uii_add_command2 (UII_ENABLE, 0, "trace ip rib", 
		      trace_ip_rib, "Enable trace rib");
    uii_add_command2 (UII_ENABLE, 0, "no trace ip rib", 
		      trace_ip_rib, "Disable trace rib");
#ifdef HAVE_IPV6
    uii_add_command2 (UII_ENABLE, 0, "trace ipv6 rib", 
		      trace_ipv6_rib, "Enable trace rib");
    uii_add_command2 (UII_ENABLE, 0, "no trace ipv6 rib", 
		      trace_ipv6_rib, "Disable trace rib");
#endif /* HAVE_IPV6 */

#ifdef BGP_MULTI
    /* I don't necessarily like moving the router_id config into the bgp config
       because this will probably screw up other protocols. --binkertn */
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, COMMAND_NODISPLAY, "router_id %a",
	    	      config_router_id, "Set router id");
#else /* BGP_MULTI */
    uii_add_command2 (UII_CONFIG, COMMAND_NODISPLAY, "router_id %a",
	    	      config_router_id, "Set router id");
#endif /* BGP_MULTI */

    uii_add_command2 (UII_CONFIG, 0, "gateway %M on %n",
	    	      config_gateway, "Define gateway on interface");

    return (1);
}

