/* 
 * $Id: bgpconf.c,v 1.16 1999/04/30 23:48:11 masaki Exp $
 */

#include <ctype.h>
#include <stdarg.h>
#include <mrt.h>
#include <bgp.h>
#include <config_file.h>
#include <protoconf.h>


static void
get_config_aspath_filter (int num)
{
    char *cp = as_access_list_toa (num);
    if (cp) {
	config_add_output ("%s", cp);
	Delete (cp);
    }
}


static int
config_ip_as_filter (uii_connection_t * uii, int num, char *permit_or_deny,
		     char *re)
{
    int permit = 0;

    if (strcasecmp ("permit", permit_or_deny) == 0)
	permit = 1;

    if (num < 0 && num >= MAX_AS_ALIST) {
	config_notice (TR_ERROR, uii,
		   "invalid as-path access-list number (%d)\n", num);
	Delete (permit_or_deny);
	Delete (re);
	return (-1);
    }

    if (uii->negative) {
        if (remove_as_access_list (num, re, permit) > 0) {
	    if (count_as_access_list (num) <= 0) {
	        config_del_module (0, "as-path access-list", 
			           get_config_aspath_filter, (void *) num);
	    }
    	    Delete (permit_or_deny);
    	    Delete (re);
    	    return (1);
        }
        else {
            Delete (permit_or_deny);
            Delete (re);
            return (-1);
	}
    }

    if (add_as_access_list (num, re, permit) > 0) {
	if (count_as_access_list (num) == 1) {
	    config_add_module (0, "as-path access-list", 
			       get_config_aspath_filter, (void *) num);
	}
    }
    else {
        Delete (permit_or_deny);
        Delete (re);
        return (-1);
    }
    Delete (permit_or_deny);
    Delete (re);
    return (1);
}


static void
print_aggregate_address (prefix_t * prefix, u_long option, int view)
{
    char buff[MAXLINE];

    if (prefix->family == AF_INET && view != 0)
	sprintf (buff, " view %d", view);
#ifdef HAVE_IPV6
    else if (prefix->family == AF_INET6 && view != 1)
	sprintf (buff, " view %d", view);
#endif /* HAVE_IPV6 */
    else
	strcpy (buff, "");
    config_add_output ("  aggregate-address %s%s%s%s\n",
		       prefix_toax (prefix),
		       BIT_TEST (option, BGP_AGGOPT_AS_SET) ? " as-set" : "",
	  BIT_TEST (option, BGP_AGGOPT_SUMMARY_ONLY) ? " summary-only" : "",
		       buff);
}


static void
get_config_bgp_peer (bgp_peer_t *peer)
{
	char name[MAXLINE];

        pthread_mutex_lock (&peer->mutex_lock);
	strcpy (name, (peer->name)? peer->name: 
		prefix_toa (peer->gateway->prefix));

	if (peer->name) {
	    char option[MAXLINE], *cp = option;
	    *cp = '\0';
	    if (peer->bind_addr) {
	        sprintf (cp, " src %s", prefix_toa (peer->bind_addr));
		cp += strlen (cp);
	    }
	    if (peer->bind_if) {
		sprintf (cp, " iface %s", peer->bind_if->name);
		cp += strlen (cp);
	    }
	    config_add_output ("  neighbor %s peer %s%s\n", 
		peer->name, prefix_toa (peer->gateway->prefix), option);
	}
	else
	    config_add_output ("  neighbor %s remote-as %d\n", name,
		               peer->gateway->AS);

	if (peer->description)
	    config_add_output ("  neighbor %s description %s\n", name,
			       peer->description);

	if (peer->name == NULL) {
	    if (peer->bind_addr) {
	        config_add_output ("  neighbor %s update-source %s\n", name,
	        		   prefix_toa (peer->bind_addr));
	    }
	    if (peer->bind_if) {
	        config_add_output ("  neighbor %s update-source %s\n", name,
	        		   peer->bind_if->name);
	    }
	}
	if (peer->dlist_in >= 0)
	    config_add_output ("  neighbor %s distribute-list %d in\n",
			name, peer->dlist_in);
	if (peer->dlist_out >= 0)
	    config_add_output ("  neighbor %s distribute-list %d out\n",
		       name, peer->dlist_out);
	if (peer->flist_in >= 0)
	    config_add_output ("  neighbor %s filter-list %d in\n",
			name, peer->flist_in);
	if (peer->flist_out >= 0)
	    config_add_output ("  neighbor %s filter-list %d out\n",
		       name, peer->flist_out);
	if (peer->route_map_in >= 0)
	    config_add_output ("  neighbor %s route-map %d in\n",
			name, peer->route_map_in);
	if (peer->route_map_out >= 0)
	    config_add_output ("  neighbor %s route-map %d out\n",
		       name, peer->route_map_out);

	if (peer->maximum_prefix > 0)
	    config_add_output ("  neighbor %s maximum-prefix %d\n",
		       name, peer->maximum_prefix);

	if (BIT_TEST (peer->options, BGP_CONNECT_PASSIVE))
	    config_add_output ("  neighbor %s passive\n",
			       name);
	if (BIT_TEST (peer->options, BGP_TRANSPARENT_AS))
	    config_add_output ("  neighbor %s transparent-as\n", name);
	if (BIT_TEST (peer->options, BGP_TRANSPARENT_NEXTHOP))
	    config_add_output ("  neighbor %s transparent-nexthop\n", name);
	if (BIT_TEST (peer->options, BGP_NEXTHOP_SELF))
	    config_add_output ("  neighbor %s next-hop-self\n", name);
	if (BIT_TEST (peer->options, BGP_BGP4PLUS_01))
	    config_add_output ("  neighbor %s bgp4+ 1\n", name);
	if (BIT_TEST (peer->options, BGP_BGP4PLUS_AUTO))
	    config_add_output ("  neighbor %s bgp4+ auto\n", name);
	if (BIT_TEST (peer->options, BGP_ROUTE_REFLECTOR_CLIENT))
	    config_add_output ("  neighbor %s route-reflector-client\n", name);

	if (peer->HoldTime_Interval >= 0)
	    config_add_output ("  neighbor %s holdtime %d\n",
		               name, peer->HoldTime_Interval);
	if (peer->KeepAlive_Interval >= 0)
	    config_add_output ("  neighbor %s keepalive %d\n",
		               name, peer->KeepAlive_Interval);
	if (peer->ConnectRetry_Interval >= 0)
	    config_add_output ("  neighbor %s connectretry %d\n",
		               name, peer->ConnectRetry_Interval);
	if (peer->Start_Interval >= 0)
	    config_add_output ("  neighbor %s starttime %d\n",
		               name, peer->Start_Interval);

        pthread_mutex_unlock (&peer->mutex_lock);
}


static void
get_config_router_bgp ()
{
    bgp_peer_t *peer;
    int i;
    prefix_t *prefix;
    char stmp[64];
    u_long router_id;
    u_long cluster_id;

#ifndef BGP_MULTI
    config_add_output ("router bgp %d\n", BGP->my_as);
#else /* BGP_MULTI */
    config_add_output ("router bgp %d\n", BGP->current_bgp->this_as);
#endif /* BGP_MULTI */

#ifndef BGP_MULTI
    router_id = BGP->my_id;
    cluster_id = BGP->cluster_id;
#else /* BGP_MULTI */
    router_id = BGP->current_bgp->this_id;
    cluster_id = BGP->current_bgp->cluster_id;
#endif /* BGP_MULTI */
    /* there is no flags if or not it is explicitly defined */
    if (router_id != MRT->default_id)
        config_add_output ("  bgp router-id %s\n", 
			inet_ntop (AF_INET, &router_id, stmp, sizeof stmp));
    if (cluster_id != MRT->default_id)
        config_add_output ("  bgp cluster_id %s\n", 
			inet_ntop (AF_INET, &cluster_id, stmp, sizeof stmp));

    /* I think this is thread-safe... we're just reading data */
    /* NO. someone may change the peer's list, and it may break here */

    pthread_mutex_lock (&BGP->mutex_lock);
    LL_Iterate (BGP->ll_networks, prefix) {
	config_add_output ("  network %s\n", prefix_toax (prefix));
    }
    pthread_mutex_unlock (&BGP->mutex_lock);

    pthread_mutex_lock (&BGP->mutex_lock);
    for (i = 0; i < MAX_BGP_VIEWS; i++) {
	view_print_aggregate (i, print_aggregate_address);
    }
    pthread_mutex_unlock (&BGP->mutex_lock);

    pthread_mutex_lock (&MRT->mutex_lock);
    for (i = 0; i <= PROTO_MAX; i++) {
        if (BGP4_BIT_TEST (BGP->redistribute_mask, i))
	    config_add_output ("  redistribute %s\n", proto2string (i));
    }
    pthread_mutex_unlock (&MRT->mutex_lock);

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    /* we have to have a machanism like shcedule all and then wait all */
    /* waiting inside a lock will cause a deadlock when the same peer
       tries to get the lock in bgp_process_changes. get_config_bgp_peer
       scheduled can not go ahead unless the currect task finishes
       if they are in the same schedule queue. */
    LL_Iterate (BGP->ll_bgp_peers, peer) {
	/* if peer's being blocked on connect(), it doesn't run
	   so call it from this thread directly -- masaki */
	get_config_bgp_peer (peer);
/*
	schedule_event_and_wait ("get_config_bgp_peer", peer->schedule,
				  get_config_bgp_peer, 1, peer);
*/
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
}


static int
config_router_bgp (uii_connection_t * uii, int as)
{
    if (uii->negative) {
        if (!BGP4_BIT_TEST (MRT->protocols, PROTO_BGP))
	    return (0);

        config_del_module (CF_DELIM, "router bgp", get_config_router_bgp, NULL);

        stop_bgp ();
        /* probably OK without locking */
        pthread_mutex_lock (&MRT->mutex_lock);
        BGP4_BIT_RESET (MRT->protocols, PROTO_BGP);
        pthread_mutex_unlock (&MRT->mutex_lock);
#ifndef BGP_MULTI
        set_BGP (BGP_MY_AS, 0, 0);
#else /* BGP_MULTI */
        set_BGP (BGP_CURRENT_BGP, NULL, 0);
#endif /* BGP_MULTI */
        return (1);
    }

#ifndef BGP_MULTI
    /* The key change to add multiple bgp instances is really right here.
       There are some checks that need to be made better than what is going
       on right now.  -- binkertn */
    if (as <= 0 || (BGP->my_as != 0 && BGP->my_as != as)) {
	config_notice (TR_TRACE, uii,
		       "BGP is already configured; AS is %d\n", BGP->my_as);
	return (-1);
    }
#endif /* BGP_MULTI */

#ifdef BGP_MULTI
    /* I am adding a change to force that AS numbers be between 0 and
       (64k - 1)  -- binkertn */
    if (as < 0 || as > MAX_AS_NUMBER) {
	config_notice (TR_TRACE, uii,
		       "AS %d is out of bounds.  An AS number is only 16-bits\n", as);
	return (-1);
    }

#endif /* BGP_MULTI */

    CONFIG_MRTD->protocol = PROTO_BGP;
    uii->previous[++uii->prev_level] = uii->state;
    uii->state = UII_CONFIG_ROUTER_BGP;

#ifndef BGP_MULTI
    if (BGP4_BIT_TEST (MRT->protocols, PROTO_BGP))
	return (0);
#endif /* BGP_MULTI */

#ifndef BGP_MULTI
    set_BGP (BGP_MY_AS, as, NULL);
#else /* BGP_MULTI */
    if (BGP->router_bgp[as] == NULL) {
      bgp_local_t *local = New (bgp_local_t);
      /* This is not good for IPv6.  -- binkertn */
      view_t *view = New_View(32);
      int view_no = BGP->view_count++;
      char str[MAXLINE];

      init_bgp_local(local);
      local->this_as = as;
      local->this_id = MRT->default_id;
      local->cluster_id = MRT->default_id;
      local->view_no = view_no;

      BGP->router_bgp[as] = local;

      /* This is not good for IPv6.  -- binkertn */
      view->viewno = view_no;
      view->trace = trace_copy (BGP->trace);
      sprintf (str, "BGP view %d", view_no);
      set_trace (view->trace, TRACE_PREPEND_STRING, str, 0);
      BGP->views[view_no] = view;
      
      LL_Add (BGP->ll_bgp_locals, BGP->router_bgp[as]);
    }
    
    set_BGP (BGP_CURRENT_BGP, BGP->router_bgp[as], NULL);

#endif /* BGP_MULTI */

    config_add_module (CF_DELIM, "router bgp", get_config_router_bgp, NULL);

    /* probably OK without locking */
    pthread_mutex_lock (&MRT->mutex_lock);
    BGP4_BIT_SET (MRT->protocols, PROTO_BGP);
    pthread_mutex_unlock (&MRT->mutex_lock);
    start_bgp ();
    return (1);
}

#ifdef BGP_MULTI
static int
config_router_bgp_bind_interface_only(uii_connection_t * uii)
{
  if (BGP->current_bgp->bind_interface_only == 1) {
    config_notice (TR_TRACE, uii, "Interfaces are already bound.\n");
    return (1);
  }

  BGP->current_bgp->bind_interface_only = 1;
  config_notice (TR_TRACE, uii, "Interfaces are bound.\n");
  return (1);
}

static int
config_router_bgp_add_interface(uii_connection_t * uii, prefix_t *prefix)
{
  BGP->current_bgp->num_interfaces++;
  LL_Add(BGP->current_bgp->ll_interfaces, Ref_Prefix (prefix));
  return (1);
}

static int
config_router_bgp_delete_interface(uii_connection_t * uii, prefix_t *prefix)
{
  prefix_t *p;
  p = LL_Find(BGP->current_bgp->ll_interfaces, prefix);
  if (p) {
    LL_Remove(BGP->current_bgp->ll_interfaces, p);
    Deref_Prefix(p);
    return (1);
  }

  config_notice(TR_TRACE, uii, "Prefix not found.\n");
  return (0);
}
#endif /* BGP_MULTI */

/* config_router_neighbor_remoteas
 * neighbor %p remote-as %d
 */
static int
config_router_neighbor_remoteas (uii_connection_t * uii,
				 prefix_t * prefix, int as)
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    if ((peer = Find_BGP_Peer_ByPrefix (prefix)) == NULL) {
        /* I am only one except for automatic addition that adds a peer */
        peer = Add_BGP_Peer (NULL, prefix, NULL, NULL, as, BGP->trace);
	assert (peer);
	if (BGPSIM_TRANSPARENT) {
	    /* BGPSIM should behave as transparent */
	    /*
            pthread_mutex_lock (&peer->mutex_lock);
	    BIT_SET (peer->options, BGP_TRANSPARENT_AS);
	    BIT_SET (peer->options, BGP_TRANSPARENT_NEXTHOP);
            pthread_mutex_unlock (&peer->mutex_lock);
	    */
	}
    }
    else {
	if (peer->gateway->AS != as) {
	    config_notice (TR_TRACE, uii, "Peer AS is changing to %d\n", as);
	    if (peer->state == BGPSTATE_IDLE ||
	            peer->state == BGPSTATE_ACTIVE ||
	            peer->state == BGPSTATE_CONNECT)
		/* should I once down the peer? */
		peer_set_as (peer, as);
	    else
	        peer->new_as = as;
	}
    	Deref_Prefix (prefix);
        pthread_mutex_unlock (&BGP->peers_mutex_lock);
    	return (-1);
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);

    if (BIT_TEST (peer->options, BGP_INTERNAL)) {
	trace (TR_PARSE, BGP->trace, "neighbor %s AS%d (iBGP)\n",
	       prefix_toa (prefix), as);
    }
    else {
	trace (TR_PARSE, BGP->trace, "neighbor %s AS%d (eBGP)\n",
	       prefix_toa (prefix), as);
    }
    Deref_Prefix (prefix);
    start_bgp_peer (peer);
    return (1);
}


/* no_config_router_neighbor_remoteas
 * no neighbor %p remote-as %d
 */
static int
no_config_router_neighbor_remoteas (uii_connection_t * uii,
				    prefix_t * prefix, int as)
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    if ((peer = Find_BGP_Peer_ByPrefix (prefix)) == NULL) {
	config_notice (TR_ERROR, uii, "Peer does not exist\n");
	Deref_Prefix (prefix);
        pthread_mutex_unlock (&BGP->peers_mutex_lock);
	return (-1);
    }
    /* XXX */

    Destroy_BGP_Peer (peer);
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    config_notice (TR_TRACE, uii, "Peer deleted\n");
    Deref_Prefix (prefix);
    return (1);
}


static int
config_router_neighbor_n_peer (uii_connection_t * uii, char *name,
			prefix_t * prefix, char *option)
{
    bgp_peer_t *peer;
    prefix_t *usrc = NULL;
    char ifname[MAXLINE];
    interface_t *interface = NULL;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    if ((peer = Find_BGP_Peer_ByPrefix (prefix)) != NULL) {
	config_notice (TR_ERROR, uii, "Peer %s is already defined\n",
			prefix_toa (prefix));
        pthread_mutex_unlock (&BGP->peers_mutex_lock);
    	Delete (name);
    	Deref_Prefix (prefix);
    	Delete (option);
    	return (-1);
    }

    if ((peer = Find_BGP_Peer_ByID (name)) != NULL) {
	config_notice (TR_ERROR, uii, "Peer %s is already defined\n", name);
        pthread_mutex_unlock (&BGP->peers_mutex_lock);
    	Delete (name);
    	Deref_Prefix (prefix);
    	Delete (option);
    	return (-1);
    }

    strcpy (ifname, "");
    if (option) {
        if (parse_line (option, "src %M iface %s", &usrc, ifname) <= 0 &&
            parse_line (option, "iface %s src %M", ifname, &usrc) <= 0) {
            pthread_mutex_unlock (&BGP->peers_mutex_lock);
    	    Delete (name);
    	    Deref_Prefix (prefix);
    	    Delete (option);
    	    return (-1);
	}
	Delete (option);
    }

    /* XXX usrc should be checked against interface addresses */

    if (ifname[0] && (interface = find_interface_byname (ifname)) == NULL) {
	config_notice (TR_ERROR, uii, "Interface %s is not found\n", ifname);
        pthread_mutex_unlock (&BGP->peers_mutex_lock);
    	Delete (name);
    	Deref_Prefix (prefix);
    	if (usrc) Deref_Prefix (usrc);
    	return (-1);
    }

    /* I am only one except for automatic addition that adds a peer */
    peer = Add_BGP_Peer (name, prefix, usrc, interface, 0, BGP->trace);
    assert (peer);

	if (BGPSIM_TRANSPARENT) {
	    /* BGPSIM should behave as transparent */
	    /*
            pthread_mutex_lock (&peer->mutex_lock);
	    BIT_SET (peer->options, BGP_TRANSPARENT_AS);
	    BIT_SET (peer->options, BGP_TRANSPARENT_NEXTHOP);
            pthread_mutex_unlock (&peer->mutex_lock);
	    */
	}

    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    Delete (name);
    Deref_Prefix (prefix);
    if (usrc) Deref_Prefix (usrc);
    return (1);
}


static int
config_router_neighbor_n_remoteas (uii_connection_t * uii, char *name, int as)
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    if ((peer = Find_BGP_Peer_ByID (name)) == NULL) {
        pthread_mutex_unlock (&BGP->peers_mutex_lock);
	config_notice (TR_ERROR, uii, "Peer %s is not defined\n", name);
    	Delete (name);
    	return (-1);
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);

#ifndef BGP_MULTI
    if (BGP->my_as == as) {
#else /* BGP_MULTI */
    if (BGP->current_bgp->this_as == as) {
#endif /* BGP_MULTI */
	trace (TR_PARSE, BGP->trace,
	       "neighbor %s AS%d as iBGP\n", name, as);
    }
    else {
	trace (TR_PARSE, BGP->trace,
	       "neighbor %s AS%d as eBGP\n", name, as);
    }
    Delete (name);
    start_bgp_peer (peer);
    return (1);
}


static int
no_config_router_neighbor_n (uii_connection_t * uii, char *name)
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    if ((peer = Find_BGP_Peer_ByID (name)) == NULL) {
        pthread_mutex_unlock (&BGP->peers_mutex_lock);
	config_notice (TR_ERROR, uii, "Peer %s does not exist\n", name);
	Delete (name);
	return (-1);
    }
    /* XXX */

    Destroy_BGP_Peer (peer);
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    config_notice (TR_INFO, uii, "Peer %s deleted\n", name);
    Delete (name);
    return (1);
}


static int
config_router_neighbor_list (uii_connection_t *uii, char *name, int alist)
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    peer = Find_BGP_Peer_ByID (name);

    if (uii->negative) {
	if (peer == NULL) {
    	    pthread_mutex_unlock (&BGP->peers_mutex_lock);
	    config_notice (TR_ERROR, uii, "Peer %s is not defined\n", name);
    	    Delete (name);
    	    return (-1);
	}
	Destroy_BGP_Peer (peer);
    }
    else {
	if (peer != NULL) {
    	    pthread_mutex_unlock (&BGP->peers_mutex_lock);
	    config_notice (TR_ERROR, uii, "Peer %s is already defined\n", name);
    	    Delete (name);
    	    return (-1);
	}
	peer = Add_BGP_Peer (name, NULL, NULL, NULL, 0, BGP->trace);
	peer->neighbor_list = alist;
    }

    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    Delete (name);
    return (1);
}


static bgp_peer_t *
name2peer (char *name)
{
    bgp_peer_t *peer;
    prefix_t *prefix;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    if ((prefix = ascii2prefix (0, name)) != NULL) {
        peer = Find_BGP_Peer_ByPrefix (prefix);
	Deref_Prefix (prefix);
    }
    else {
        peer = Find_BGP_Peer_ByID (name);
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    return (peer);
}


static int
config_router_neighbor_update_source (uii_connection_t *uii, char *name, 
				      char *s)
{
    bgp_peer_t *peer;
    prefix_t *prefix;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
    	Delete (s);
	return (-1);
    }
    Delete (name);

    pthread_mutex_lock (&peer->mutex_lock);
    if (uii->negative) {
        prefix = ascii2prefix (0, s);
    	Delete (s);
	if (prefix) {
	    if (peer->bind_addr && prefix_compare (prefix, peer->bind_addr)) {
	        Deref_Prefix (peer->bind_addr);
	        peer->bind_addr = NULL;
    		pthread_mutex_unlock (&peer->mutex_lock);
        	return (1);
	    }
	    Deref_Prefix (prefix);
	}
	else {
	    if (peer->bind_if && strcasecmp (peer->bind_if->name, s) == 0) {
		peer->bind_if = NULL;
    		pthread_mutex_unlock (&peer->mutex_lock);
        	return (1);
	    }
	}
    	pthread_mutex_unlock (&peer->mutex_lock);
        return (0);
    }
    else {
        prefix = ascii2prefix (0, s);
    	Delete (s);
	if (prefix) {
	    if (find_interface_local (prefix) == NULL) {
		config_notice (TR_ERROR, uii, 
			       "Address %s is not found\n", 
				prefix_toa (prefix));
    	        pthread_mutex_unlock (&peer->mutex_lock);
		return (-1);
	    }
	    if (peer->bind_addr) {
	        Deref_Prefix (peer->bind_addr);
	    }
	    peer->bind_addr = prefix;
    	    pthread_mutex_unlock (&peer->mutex_lock);
    	    return (1);
	}
	else {
	    interface_t *interface;

	    interface = find_interface_byname (s);
	    if (interface) {
	        peer->bind_if = interface;
    	        pthread_mutex_unlock (&peer->mutex_lock);
    	        return (1);
	    }
	    else {
		config_notice (TR_ERROR, uii, 
			       "Interface %s is not found\n", s);
    	        pthread_mutex_unlock (&peer->mutex_lock);
		return (-1);
	    }
	}
    }
    /* NOT REACHED */
}



static void
set_bgp_peer (bgp_peer_t *peer, int first, ...)
{
    va_list ap;
    enum BGP_PEER_ATTR attr;
    int re_eval = -1;
    int establish = -1;
    char str[MAXLINE];

    assert (peer);
    pthread_mutex_lock (&peer->mutex_lock);
    va_start (ap, first);
    for (attr = (enum BGP_PEER_ATTR) first; attr; 
				     attr = va_arg (ap, enum BGP_PEER_ATTR)) {
	switch (attr) {
	case BGP_PEER_DESCRIPTION:
	    if (peer->description)
		Delete (peer->description);
	    peer->description = va_arg (ap, char *); /* don't need to dup */
	    if (peer->description) {
#ifdef HAVE_IPV6
		if (peer->gateway && peer->gateway->prefix->family == AF_INET6)
		    sprintf (str, "BGP4+ %s", peer->description);
		else
#endif /* HAVE_IPV6 */
		sprintf (str, "BGP %s", peer->description);
	    }
	    else {
#ifdef HAVE_IPV6
		if (peer->gateway && peer->gateway->prefix->family == AF_INET6)
		    sprintf (str, "BGP4+ %s", peer->name?peer->name:
					prefix_toa (peer->gateway->prefix));
		else
#endif /* HAVE_IPV6 */
		sprintf (str, "BGP %s", peer->name?peer->name:
                                        prefix_toa (peer->gateway->prefix));
	    }
	    set_trace (peer->trace, TRACE_PREPEND_STRING, str, 0);
	    break;
	case BGP_PEER_VIEW:
	    peer->view = va_arg (ap, int);
	    break;
	case BGP_PEER_WEIGHT:
    	    peer->default_weight = va_arg (ap, int);
	    if (peer->state == BGPSTATE_ESTABLISHED)
		re_eval = 1;
	    break;
	case BGP_PEER_ALIAS:
    	    if (peer->aliases == NULL)
	        peer->aliases = LL_Create (0);
    	    LL_Add (peer->aliases, va_arg (ap, prefix_t *));
	    break;
	case BGP_PEER_MAXPREF:
    	    peer->maximum_prefix = va_arg (ap, int);
	    break;
	case BGP_PEER_SETOPT:
            BIT_SET (peer->options, va_arg (ap, u_long));
	    break;
	case BGP_PEER_RESETOPT:
            BIT_RESET (peer->options, va_arg (ap, u_long));
	    break;
	case BGP_PEER_DLIST_IN:
    	    peer->dlist_in = va_arg (ap, int);
	    if (peer->state == BGPSTATE_ESTABLISHED)
    		re_eval = 1;
	    break;
	case BGP_PEER_DLIST_OUT:
    	    peer->dlist_out = va_arg (ap, int);
	    if (peer->state == BGPSTATE_ESTABLISHED)
		establish = 0;
	    break;
	case BGP_PEER_FLIST_IN:
    	    peer->flist_in = va_arg (ap, int);
	    if (peer->state == BGPSTATE_ESTABLISHED)
    		re_eval = 1;
	    break;
	case BGP_PEER_FLIST_OUT:
    	    peer->flist_out = va_arg (ap, int);
	    if (peer->state == BGPSTATE_ESTABLISHED)
		establish = 0;
	    break;
	case BGP_PEER_RTMAP_IN:
    	    peer->route_map_in = va_arg (ap, int);
	    if (peer->state == BGPSTATE_ESTABLISHED)
    		re_eval = 1;
	    break;
	case BGP_PEER_RTMAP_OUT:
    	    peer->route_map_out = va_arg (ap, int);
	    if (peer->state == BGPSTATE_ESTABLISHED)
		establish = 1;
	    break;
	case BGP_PEER_HOLDTIME:
            peer->HoldTime_Interval = va_arg (ap, int);
	    if (peer->HoldTime_Interval >= 0)
	        Timer_Set_Time (peer->timer_HoldTime, peer->HoldTime_Interval);
	    break;
	case BGP_PEER_KEEPALIVE:
            peer->KeepAlive_Interval = va_arg (ap, int);
	    if (peer->KeepAlive_Interval >= 0)
	        Timer_Set_Time (peer->timer_KeepAlive, 
				peer->KeepAlive_Interval);
	    break;
	case BGP_PEER_CONNECTRETRY:
            peer->ConnectRetry_Interval = va_arg (ap, int);
	    if (peer->ConnectRetry_Interval >= 0)
	        Timer_Set_Time (peer->timer_ConnectRetry, 
				peer->ConnectRetry_Interval);
	    break;
	case BGP_PEER_START:
            peer->Start_Interval = va_arg (ap, int);
	    if (peer->Start_Interval >= 0)
	        Timer_Set_Time (peer->timer_Start, 
				peer->Start_Interval);
	    break;
	default:
	    assert (0);
    	    break;
	}
    }
    va_end (ap);
    pthread_mutex_unlock (&peer->mutex_lock);
    if (re_eval >= 0)
        bgp_re_evaluate_in (peer);
    if (establish >= 0)
	bgp_establish_peer (peer, establish);
}


/* 
 * config_router_neighbor_view
 * just used by BGP to configure peer (at least for now)
 */
static int
config_router_neighbor_view (uii_connection_t * uii, char * name, int view)
{
    bgp_peer_t *peer;
    char str[MAXLINE];

    if (view >= MAX_BGP_VIEWS || view < 0) {
	config_notice (TR_ERROR, uii,
		       "Invalid view %d for BGP neighbor %s [0-%d]\n",
		       view, name, MAX_BGP_VIEWS - 1);
	Delete (name);
	return (-1);
    }

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	return (-1);
    }
    Delete (name);

    if (BGP->views[view]) {
	if (BGP->views[view]->family != peer->gateway->prefix->family) {
	    config_notice (TR_ERROR, uii, "view %d already defined for %s\n",
			   view, family2string (BGP->views[view]->family));
	    return (-1);
	}

	trace (TR_PARSE, BGP->trace, "neighbor %s view %d\n",
	       (peer->name)? peer->name: prefix_toa (peer->gateway->prefix), 
	       view);

        pthread_mutex_lock (&MRT->mutex_lock);
	if (BGP->views[view] == NULL) {
	    /* a new view */
#ifdef HAVE_IPV6
	    if (peer->gateway->prefix->family == AF_INET6) {
		BGP->views[view] = New_View (128);
    	        sprintf (str, "BGP4+ view %d", BGP->views[view]->viewno);
	    }
	    else
#endif /* HAVE_IPV6 */
	    {
		BGP->views[view] = New_View (32);
    	        sprintf (str, "BGP view %d", BGP->views[view]->viewno);
	    }

	    BGP->views[view]->viewno = view;
	    BGP->views[view]->trace = trace_copy (BGP->trace);
    	    set_trace (BGP->views[view]->trace, TRACE_PREPEND_STRING, str, 0);

	}
        pthread_mutex_unlock (&MRT->mutex_lock);
	schedule_event2 ("set_bgp_peer",
                         peer->schedule, (event_fn_t) set_bgp_peer, 4,
                         peer, BGP_PEER_VIEW, view, 0);
    }
    return (1);
}


static int
config_router_neighbor_weight (uii_connection_t * uii,
			       char * name, int weight)
{
    bgp_peer_t *peer;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	return (-1);
    }
    Delete (name);

    if (uii->negative)
	weight = -1;

    schedule_event2 ("set_bgp_peer",
                     peer->schedule, (event_fn_t) set_bgp_peer, 4,
                     peer, BGP_PEER_WEIGHT, weight, 0);

    return (1);
}


static int
config_router_neighbor_description (uii_connection_t * uii,
			            char * name, char *description)
{
    bgp_peer_t *peer;
    char *s;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	Delete (description);
	return (-1);
    }
    Delete (name);

    if (uii->negative)
	s = NULL;
    else {
	s = description;
    }

    schedule_event2 ("set_bgp_peer",
                     peer->schedule, (event_fn_t) set_bgp_peer, 4,
                     peer, BGP_PEER_DESCRIPTION, s, 0);

    return (1);
}


static int
config_router_neighbor_maximum_prefix (uii_connection_t * uii,
			                 char * name, int num)
{
    bgp_peer_t *peer;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	return (-1);
    }
    Delete (name);

    if (uii->negative)
	num = -1;

    schedule_event2 ("set_bgp_peer",
                     peer->schedule, (event_fn_t) set_bgp_peer, 4,
                     peer, BGP_PEER_MAXPREF, num, 0);

    return (1);
}


static int
config_router_neighbor_alias (uii_connection_t * uii, char *name,
			      prefix_t * prefix_alias)
{
    bgp_peer_t *peer;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	return (-1);
    }
    Delete (name);

    schedule_event2 ("set_bgp_peer", peer->schedule, 
		     (event_fn_t) set_bgp_peer, 4,
                     peer, BGP_PEER_ALIAS, prefix_alias, 0);
    return (1);
}


static int
config_router_neighbor_option (uii_connection_t * uii, char *name,
			       char *option)
{
    bgp_peer_t *peer;
    u_long opt = 0;
    enum BGP_PEER_ATTR attr = BGP_PEER_SETOPT;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	Delete (option);
	return (-1);
    }
    Delete (name);

    if (uii->negative)
	attr = BGP_PEER_RESETOPT;

    if (strcasecmp (option, "transparent-as") == 0)
        opt = BGP_TRANSPARENT_AS;
    else if (strcasecmp (option, "transparent-nexthop") == 0)
        opt = BGP_TRANSPARENT_NEXTHOP;
    else if (strcasecmp (option, "passive") == 0)
        opt = BGP_CONNECT_PASSIVE;
    else if (strcasecmp (option, "next-hop-self") == 0)
        opt = BGP_NEXTHOP_SELF;
    else if (strcasecmp (option, "route-reflector-client") == 0)
        opt = BGP_ROUTE_REFLECTOR_CLIENT;
    else {
	assert (0);
    }

    schedule_event2 ("set_bgp_peer", peer->schedule, 
		     (event_fn_t) set_bgp_peer, 4, peer, attr, opt, 0);
    Delete (option);
    return (1);
}


static int
config_router_neighbor_list_in_out (uii_connection_t * uii, char *name, 
				      char *op, int num, char *in_or_out)
{
    bgp_peer_t *peer;
    enum BGP_PEER_ATTR attr = 0;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	Delete (op);
	Delete (in_or_out);
	return (-1);
    }
    Delete (name);

    if (uii->negative)
	num = -1;

    if (strcasecmp (op, "distribute-list") == 0) {
	if (strcasecmp (in_or_out, "in") == 0)
            attr = BGP_PEER_DLIST_IN;
	else
            attr = BGP_PEER_DLIST_OUT;
    }
    else if (strcasecmp (op, "filter-list") == 0) {
	if (strcasecmp (in_or_out, "in") == 0)
            attr = BGP_PEER_FLIST_IN;
	else
            attr = BGP_PEER_FLIST_OUT;
    }
    else {
	assert (0);
    }
    schedule_event2 ("set_bgp_peer", peer->schedule, 
		     (event_fn_t) set_bgp_peer, 4, peer, attr, num, 0);
    Delete (op);
    Delete (in_or_out);
    return (1);
}


static int
config_router_neighbor_routemap_in_out (uii_connection_t * uii,
				       char * name, int num, char *in_or_out)
{
    bgp_peer_t *peer;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	Delete (in_or_out);
	return (-1);
    }
    Delete (name);

    if (uii->negative)
	num = -1;

    if (strcasecmp (in_or_out, "in") == 0)
        schedule_event2 ("set_bgp_peer", peer->schedule, 
		         (event_fn_t) set_bgp_peer, 
			  4, peer, BGP_PEER_RTMAP_IN, num, 0);
    else
        schedule_event2 ("set_bgp_peer", peer->schedule, 
		         (event_fn_t) set_bgp_peer, 
			  4, peer, BGP_PEER_RTMAP_OUT, num, 0);
    Delete (in_or_out);
    return (1);
}


#ifdef HAVE_IPV6
static int
config_router_neighbor_bgp4plus (uii_connection_t * uii,
				   char * name, char * version)
{
    bgp_peer_t *peer;
    u_long opt = BGP_BGP4PLUS;
    enum BGP_PEER_ATTR cmd = BGP_PEER_SETOPT;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
        Delete (version);
	return (-1);
    }
    Delete (name);

    if (uii->negative)
	cmd = BGP_PEER_RESETOPT;

    if (strcasecmp (version, "1") == 0 || strcasecmp (version, "new") == 0 ||
        strcasecmp (version, "rfc") == 0)
	opt |= BGP_BGP4PLUS_01;
    else if (strcasecmp (version, "auto") == 0)
        opt |= BGP_BGP4PLUS_AUTO;

    schedule_event2 ("set_bgp_peer", peer->schedule, 
		     (event_fn_t) set_bgp_peer, 4, peer, cmd, opt, 0);

    Delete (version);
    return (1);
}
#endif /* HAVE_IPV6 */


static int
config_router_neighbor_time (uii_connection_t * uii,
			     char * name, char *timer, int num)
{
    bgp_peer_t *peer;
    enum BGP_PEER_ATTR cmd = 0;
    int num2;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	return (-1);
    }
    Delete (name);

    /* since the argument num may not be available on the stack */
    num2 = (uii->negative)? -1: num;

    if (strcasecmp (timer, "holdtime") == 0) {
	cmd = BGP_PEER_HOLDTIME;
    }
    else if (strcasecmp (timer, "keepalive") == 0) {
	cmd = BGP_PEER_KEEPALIVE;
    }
    else if (strcasecmp (timer, "connectretry") == 0) {
	cmd = BGP_PEER_CONNECTRETRY;
    }
    else if (strcasecmp (timer, "starttime") == 0) {
	cmd = BGP_PEER_START;
    }
    else {
	assert (0);
    }

    schedule_event2 ("set_bgp_peer", peer->schedule, 
		     (event_fn_t) set_bgp_peer, 4, peer, cmd, num2, 0);

    Delete (timer);
    return (1);
}


static int
trace_bgp_neighbor (uii_connection_t * uii, char *name)
{
    bgp_peer_t *peer;

    if ((peer = name2peer (name)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", name);
	Delete (name);
	return (-1);
    }
    Delete (name);

    if (uii->negative)
        set_trace (peer->trace, TRACE_DEL_FLAGS, TR_ALL, NULL);
    else
        set_trace (peer->trace, TRACE_ADD_FLAGS, TR_ALL, NULL);
    return (1);
}


static int
config_router_aggregate (uii_connection_t * uii, prefix_t * prefix, 
			 char *rest)
{
    u_long option;
    char opt[4][MAXLINE];
    int view = 0, n;

#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6)
	view = 1;
#endif /* HAVE_IPV6 */

    opt[0][0] = opt[1][0] = opt[2][0] = opt[3][0] = '\0';
    option = 0;
    if (rest && (n = parse_line (rest, "%s %s %s %s",
				 opt[0], opt[1], opt[2], opt[3])) >= 1) {
	int i;

	for (i = 0; i < n; i++) {

	    if (strcasecmp (opt[i], "as-set") == 0) {
	        BIT_SET (option, BGP_AGGOPT_AS_SET);
	    }
	    else if (strcasecmp (opt[i], "summary-only") == 0) {
	        BIT_SET (option, BGP_AGGOPT_SUMMARY_ONLY);
	    }
	    else if (i < n - 1 && strcasecmp (opt[i], "view") == 0
	        && isdigit (opt[i + 1][0])) {
	        view = atoi (opt[i + 1]);
		i++;
	    }
	}
    }

    if (rest)
	Delete (rest);

    if (BGP->views[view] == NULL) {
	config_notice (TR_TRACE, uii,
		    "CONFIG aggregate-address view %d not defined\n", view);
	Deref_Prefix (prefix);
	return (-1);
    }
    if (BGP->views[view]->family != prefix->family) {
	config_notice (TR_TRACE, uii,
		"CONFIG aggregate-address view %d family %s not %s\n", view,
		       family2string (BGP->views[view]->family),
		       family2string (prefix->family));
	Deref_Prefix (prefix);
	return (-1);
    }
    view_open (BGP->views[view]);
    view_add_aggregate (BGP->views[view], prefix, option);
    view_close (BGP->views[view]);
    config_notice (TR_TRACE, uii,
		   "CONFIG aggregate-address %s%s%s\n",
		   prefix_toax (prefix),
		   BIT_TEST (option, BGP_AGGOPT_AS_SET) ? " as-set" : "",
	 BIT_TEST (option, BGP_AGGOPT_SUMMARY_ONLY) ? " summary-only" : "");
    Deref_Prefix (prefix);
    return (1);
}


static int
no_config_router_aggregate (uii_connection_t * uii, prefix_t * prefix, 
			    char *rest)
{
    int view = 0;

#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6)
	view = 1;
#endif
    if (rest) {
	parse_line (rest, "view %d", &view);
	Delete (rest);
    }

    if (BGP->views[view] == NULL) {
	config_notice (TR_ERROR, uii,
		 "no aggregate-address view %d not defined\n", view);
	Deref_Prefix (prefix);
	return (-1);
    }
    if (BGP->views[view]->family != prefix->family) {
	config_notice (TR_ERROR, uii,
		   "no aggregate-address view %d family %s not %s\n",
		       view,
		       family2string (BGP->views[view]->family),
		       family2string (prefix->family));
	Deref_Prefix (prefix);
	return (-1);
    }
    view_open (BGP->views[view]);
    view_del_aggregate (BGP->views[view], prefix);
    view_close (BGP->views[view]);
    config_notice (TR_TRACE, uii,
		   "CONFIG no aggregate-address %s view %d\n",
		   prefix_toa (prefix), view);
    Deref_Prefix (prefix);
    return (1);
}


static int
config_dump_bgp_common (uii_connection_t * uii, char *filename,
			char *unit, time_t * time)
{
    if (unit[0] && isdigit (unit[0])) {
	int i;
	u_int u;
	u = atoi (unit);
	for (i = 0; i < sizeof (unit) && unit[i] && isdigit (unit[i]); i++);
	if (i < sizeof (unit) && isalpha (unit[i])) {
	    if (strncasecmp (unit + i, "m", 1) == 0)
		u *= 60;
	    else if (strncasecmp (unit + i, "h", 1) == 0)
		u *= (60 * 60);
	}
	*time = u;
    }
    return (1);
}


static char *
time_to_unit (time_t t)
{
    char *buff;

    THREAD_SPECIFIC_STRAGE (buff);
    if ((t % 3600) == 0)
	sprintf (buff, "%ldh", t / 3600);
    else if ((t % 60) == 0)
	sprintf (buff, "%ldm", t / 60);
    else
	sprintf (buff, "%ld", t);
    return (buff);

}

static void
get_config_dump_bgp_updates (u_long type_mask)
{
    char *type = "updates";

    if (type_mask == ~0)
	type = "all";
    config_add_output ("dump bgp %s %s %s\n", type, BGP->dump_update_form,
		       time_to_unit (BGP->dump_update_interval));
}


static void
get_config_dump_bgp_view (int id)
{
    config_add_output ("dump bgp view %d %s %s\n", id,
			   BGP->dump_route_form[id],
			   time_to_unit (BGP->dump_route_interval[id]));
}


static int
config_dump_bgp_updates (uii_connection_t * uii, char *type, char *filename, 
			 char *unit)
{
    time_t t = 0;
    u_long type_mask = 0;

    BGP4_BIT_SET (type_mask, BGP_UPDATE);
    if (strcasecmp (type, "all") == 0)
	type_mask = ~0;

    if (uii->negative) {
	set_BGP (BGP_DUMP_UPDATE_FORM, NULL, 0);
	config_del_module (0, "dump bgp", get_config_dump_bgp_updates,
			   (void *) type_mask);
	Delete (type);
	return (1);
    }
    if (config_dump_bgp_common (uii, filename, unit, &t) >= 0 && filename[0]) {
	set_BGP (BGP_DUMP_UPDATE_FORM, filename, t, type_mask, 0, 0);
	config_notice (TR_TRACE, uii,
		       "CONFIG dump bgp %s %s %d\n", type, filename, t);
	config_add_module (0, "dump bgp", get_config_dump_bgp_updates, 
			   (void *) type_mask);
	Delete (type);
	Delete (filename);
	Delete (unit);
	return (1);
    }
    Delete (type);
    Delete (filename);
    Delete (unit);
    return (-1);
}


static int
config_dump_bgp_view (uii_connection_t * uii,
		      int id, char *filename, char *unit)
{
    time_t t = 0;

    if (config_dump_bgp_common (uii, filename, unit, &t) >= 0 && filename[0]) {
        set_BGP (BGP_DUMP_ROUTE_FORM, id, filename, t, DUMP_ASCII, 0);
	config_notice (TR_TRACE, uii,
		       "CONFIG dump bgp view %d %s %d\n", id, filename, t);
	config_add_module (0, "dump bgp", get_config_dump_bgp_view, (void *)id);
	Delete (filename);
	Delete (unit);
	return (1);
    }
    Delete (filename);
    Delete (unit);
    return (-1);
}


static void
get_config_dump_f_bgp (char *type)
{
    config_add_output ("dump %s bgp %s %s %s\n", (
			BGP->dump_update_family == AF_INET6)?"ipv6":"ip",
			type, BGP->dump_update_form,
		        time_to_unit (BGP->dump_update_interval));
}


static int
config_dump_f_bgp_updates (uii_connection_t * uii, int family,
		     char *type, char *filename, char *unit)
{
    time_t t = 0;
    u_long type_mask = 0;

    BGP4_BIT_SET (type_mask, BGP_UPDATE);
    if (strcasecmp (type, "all") == 0)
	type_mask = ~0;

    if (uii->negative) {
	set_BGP (BGP_DUMP_UPDATE_FORM, NULL, 0);
	config_del_module (0, "dump f bgp", 
			   get_config_dump_f_bgp, (void *) type_mask);
	Delete (type);
	return (1);
    }

    if (config_dump_bgp_common (uii, filename, unit, &t) >= 0 && filename[0]) {
	set_BGP (BGP_DUMP_UPDATE_FORM, filename, t, type_mask, family, 0);
	config_add_module (0, "dump f bgp", 
			   get_config_dump_f_bgp, (void *) type_mask);
	Delete (type);
	Delete (filename);
	Delete (unit);
	return (1);
    }
    Delete (type);
    Delete (filename);
    Delete (unit);
    return (-1);
}


static int
config_dump_ip_bgp_updates (uii_connection_t * uii,
		    char *type, char *filename, char *unit)
{
    return (config_dump_f_bgp_updates (uii, AF_INET, type, filename, unit));
}


#ifdef HAVE_IPV6
static int
config_dump_ipv6_bgp_updates (uii_connection_t * uii,
		      char *type, char *filename, char *unit)
{
    return (config_dump_f_bgp_updates (uii, AF_INET6, type, filename, unit));
}
#endif /* HAVE_IPV6 */


static void
get_config_dump_f_bgp_routes (int id)
{
    config_add_output ("dump %s bgp routes %s %s\n", (id == 1)?"ipv6":"ip",
			   BGP->dump_route_form[id],
			   time_to_unit (BGP->dump_route_interval[id]));
}


static int
config_dump_f_bgp_routes (uii_connection_t * uii, int family,
	 	   	  char *filename, char *unit)
{
    int id = 0;
    time_t t = 0;

#ifdef HAVE_IPV6
    if (family == AF_INET6)
	id = 1;
#endif /* HAVE_IPV6 */

    if (uii->negative) {
        set_BGP (BGP_DUMP_ROUTE_FORM, id, NULL, 0);
	config_del_module (0, "dump bgp routes", get_config_dump_f_bgp_routes, 
			   (void *)id);
	return (1);
    }

    if (config_dump_bgp_common (uii, filename, unit, &t) >= 0 && filename[0]) {
        set_BGP (BGP_DUMP_ROUTE_FORM, id, filename, t, DUMP_ASCII, 0);
	config_notice (TR_TRACE, uii,
		       "CONFIG dump bgp view %d %s %d\n", id, filename, t);
	config_add_module (0, "dump bgp routes", get_config_dump_f_bgp_routes, 
			   (void *)id);
	Delete (filename);
	Delete (unit);
	return (1);
    }
    Delete (filename);
    Delete (unit);
    return (-1);
}


static int
config_dump_ip_bgp_routes (uii_connection_t * uii,
	 	   	   char *filename, char *unit)
{
    return (config_dump_f_bgp_routes (uii, AF_INET, filename, unit));
}


static int
dump_f_bgp_routes (uii_connection_t * uii, int family, char *filename)
{
    int viewno = 0;
    char *cp;
    char *name;

    if (family == AF_INET6)
	viewno = 1;

    if ((cp = strrchr (filename, '/')))
      cp = cp + 1;
    else
      cp = filename;

    if (UII->redirect == NULL) {
	uii_send_data (uii,
                   "No redirection allowed! Use redirect in configuration\n");
	Delete (filename);
	return (0);
    }

    name = NewArray (char, strlen (UII->redirect) + strlen (cp) + 1 + 1);
    sprintf (name, "%s/%s", UII->redirect, cp);
    if (dump_view_bgp_routes (viewno, name, DUMP_BINARY) >= 0) {
	uii_send_data (uii, "[%s]\n", name);
    }
    Delete (name);
    Delete(filename);
    return (1);
}


static int
dump_ip_bgp_routes (uii_connection_t * uii, char *filename)
{
    return (dump_f_bgp_routes (uii, AF_INET, filename));
}


#ifdef HAVE_IPV6
static int
dump_ipv6_bgp_routes (uii_connection_t * uii, char *filename)
{
    return (dump_f_bgp_routes (uii, AF_INET6, filename));
}
#endif /* HAVE_IPV6 */


#ifdef HAVE_IPV6
static int
config_dump_ipv6_bgp_routes (uii_connection_t * uii,
	 	   	   char *filename, char *unit)
{
    return (config_dump_f_bgp_routes (uii, AF_INET6, filename, unit));
}
#endif /* HAVE_IPV6 */


static void
get_config_dump_bgp_routes ()
{
  if (BGP->dump_route_form[0]) {
      if (BGP->dump_route_type[0] == DUMP_ASCII) 
        config_add_output ("dump bgp routes %s %s\n", BGP->dump_route_form[0],
		           time_to_unit (BGP->dump_route_interval[0]));
      else
        config_add_output ("dump-binary bgp routes %s %s\n", 
		           BGP->dump_route_form[0],
		           time_to_unit (BGP->dump_route_interval[0]));
  }
}


static int
_config_dump_bgp_routes (uii_connection_t * uii, char *filename, 
			 char *unit, int dump_type)
{
    time_t t = 0;

    if (uii->negative) {
	set_BGP (BGP_DUMP_ROUTE_FORM, 0, NULL, 0);
#ifdef HAVE_IPV6
	set_BGP (BGP_DUMP_ROUTE_FORM, 1, NULL, 0);
#endif /* HAVE_IPV6 */
	config_del_module (0, "bgp dump", get_config_dump_bgp_routes, NULL);
	return (1);
    }

    if (config_dump_bgp_common (uii, filename, unit, &t) >= 0 && filename[0]) {
	set_BGP (BGP_DUMP_ROUTE_FORM, 0, filename, t, dump_type, 0);
#ifdef HAVE_IPV6
	set_BGP (BGP_DUMP_ROUTE_FORM, 1, filename, t, dump_type, 0);
#endif /* HAVE_IPV6 */
	config_notice (TR_TRACE, uii,
		       "CONFIG dump bgp routes %s %d\n", filename, t);
	config_add_module (0, "bgp dump", get_config_dump_bgp_routes, NULL);
	Delete (filename);
	Delete (unit);
	return (1);
    }
    Delete (filename);
    Delete (unit);
    return (-1);
}


static int
config_dump_bgp_routes (uii_connection_t * uii, char *filename, char *unit) {
  
  if (uii->negative)
      return (_config_dump_bgp_routes (uii, NULL, NULL, DUMP_ASCII));
  else
      return (_config_dump_bgp_routes (uii, filename, unit, DUMP_ASCII));
}


static int
config_dump_bgp_routes_binary (uii_connection_t * uii, char *filename, 
			       char *unit) {
  if (uii->negative)
    return (_config_dump_bgp_routes (uii, NULL, NULL, DUMP_BINARY));
  else
    return (_config_dump_bgp_routes (uii, filename, unit, DUMP_BINARY));
}


static int
kill_peer (uii_connection_t * uii, char *whom)
{
    bgp_peer_t *peer;

    if (strcmp (whom, "*") == 0) {
	bgp_kill_all ();
	return (1);
    }
    if ((peer = name2peer (whom)) == NULL) {
	config_notice (TR_ERROR, uii, "No peer %s\n", whom);
	return (-1);
    }
    bgp_stop_peer (peer);
    return (1);
}


static int
config_router_bgp_network_prefix (uii_connection_t * uii, prefix_t *prefix)
{
    prefix_t *network;

    pthread_mutex_lock (&BGP->mutex_lock);
    LL_Iterate (BGP->ll_networks, network) {
	if (prefix_compare (prefix, network))
	    break;
    }

    if (uii->negative) {
	if (network == NULL) {
            Deref_Prefix (prefix);
    	    pthread_mutex_unlock (&BGP->mutex_lock);
	    return (0);
	}
	if (MRT->rib_redistribute_network)
            MRT->rib_redistribute_network (CONFIG_MRTD->protocol, prefix, 0);
        LL_Remove (BGP->ll_networks, network);
        Deref_Prefix (network);
        Deref_Prefix (prefix);
    	pthread_mutex_unlock (&BGP->mutex_lock);
        return (1);
    }

    if (network != NULL) {
	Deref_Prefix (prefix);
    	pthread_mutex_unlock (&BGP->mutex_lock);
	return (0);
    }

    if (MRT->rib_redistribute_network)
        MRT->rib_redistribute_network (CONFIG_MRTD->protocol, prefix, 1);
    LL_Add (BGP->ll_networks, Ref_Prefix (prefix));
    Deref_Prefix (prefix);
    pthread_mutex_unlock (&BGP->mutex_lock);
    return (1);
}


/*
 * Redistribute a given protocol into another protocol 
 */
static int
config_router_bgp_redistribute (uii_connection_t * uii, char *proto_string)
{
    int proto;

    if ((proto = string2proto (proto_string)) < 0) {
	Delete (proto_string);
	return (-1);
    }

    if (CONFIG_MRTD->protocol == proto) {
	config_notice (TR_ERROR, uii,
		 "%s redistribute %s -- the same protocols!\n",
		  proto2string (CONFIG_MRTD->protocol), proto2string (proto));
	Delete (proto_string);
	return (-1);
    }

    pthread_mutex_lock (&BGP->mutex_lock);
    if (uii->negative) {
	if (!BGP4_BIT_TEST (BGP->redistribute_mask, proto)) {
	    Delete (proto_string);
	    return (0);
	}
        BGP4_BIT_RESET (BGP->redistribute_mask, proto);
        if (MRT->rib_redistribute_request)
            MRT->rib_redistribute_request (CONFIG_MRTD->protocol, proto, 0);
        Delete (proto_string);
        pthread_mutex_unlock (&BGP->mutex_lock);
        return (1);
    }

    if (BGP4_BIT_TEST (BGP->redistribute_mask, proto)) {
	Delete (proto_string);
        pthread_mutex_unlock (&BGP->mutex_lock);
	return (0);
    }
    BGP4_BIT_SET (BGP->redistribute_mask, proto);
    if (MRT->rib_redistribute_request)
        MRT->rib_redistribute_request (CONFIG_MRTD->protocol, proto, 1);
    Delete (proto_string);
    pthread_mutex_unlock (&BGP->mutex_lock);
    return (1);
}


static int 
show_bgp (uii_connection_t * uii)
{
    return (show_f_bgp_summary (uii, 0, FALSE));
}


static int 
show_bgp_summary (uii_connection_t * uii)
{
    return (show_f_bgp_summary (uii, 0, TRUE));
}

static int 
show_ip_bgp_summary (uii_connection_t * uii)
{
    return (show_f_bgp_summary (uii, AF_INET, TRUE));
}


static int 
show_ip_bgp_neighbors (uii_connection_t * uii)
{
    return (show_f_bgp_summary (uii, AF_INET, FALSE));
}


#ifdef HAVE_IPV6
static int 
show_ipv6_bgp_summary (uii_connection_t * uii)
{
    return (show_f_bgp_summary (uii, AF_INET6, TRUE));
}


static int 
show_ipv6_bgp_neighbors (uii_connection_t * uii)
{
    return (show_f_bgp_summary (uii, AF_INET6, FALSE));
}
#endif /* HAVE_IPV6 */


static int
show_bgp_neighbors_errors (uii_connection_t * uii, char *peer_or_star)
{
    return (show_f_bgp_neighbors_errors (uii, 0, peer_or_star));
}


static int
show_ip_bgp_neighbors_errors (uii_connection_t * uii, char *peer_or_star)
{
    return (show_f_bgp_neighbors_errors (uii, AF_INET, peer_or_star));
}


#ifdef HAVE_IPV6
static int
show_ipv6_bgp_neighbors_errors (uii_connection_t * uii, char *peer_or_star)
{
    return (show_f_bgp_neighbors_errors (uii, AF_INET6, peer_or_star));
}
#endif /* HAVE_IPV6 */


static int
show_bgp_neighbors_routes (uii_connection_t * uii, char *peer_or_star)
{
    return (show_f_bgp_neighbors_routes (uii, 0, peer_or_star));
}


static int
show_ip_bgp_neighbors_routes (uii_connection_t * uii, char *peer_or_star)
{
    return (show_f_bgp_neighbors_routes (uii, AF_INET, peer_or_star));
}


#ifdef HAVE_IPV6
static int
show_ipv6_bgp_neighbors_routes (uii_connection_t * uii, char *peer_or_star)
{
    return (show_f_bgp_neighbors_routes (uii, AF_INET6, peer_or_star));
}
#endif /* HAVE_IPV6 */


static int 
show_bgp_rt_regexp (uii_connection_t * uii, char *expr)
{
    return (show_f_bgp_rt_regexp (uii, 0, expr));
}


static int 
show_ip_bgp (uii_connection_t * uii)
{
    return (show_f_bgp_rt_regexp (uii, AF_INET, NULL));
}


static int 
show_ip_bgp_regexp (uii_connection_t * uii, char *exp)
{
    return (show_f_bgp_rt_regexp (uii, AF_INET, exp));
}


#ifdef HAVE_IPV6
static int 
show_ipv6_bgp (uii_connection_t * uii)
{
    return (show_f_bgp_rt_regexp (uii, AF_INET6, NULL));
}


static int 
show_ipv6_bgp_regexp (uii_connection_t * uii, char *exp)
{
    return (show_f_bgp_rt_regexp (uii, AF_INET6, exp));
}
#endif /* HAVE_IPV6 */

    
/* show_bgp_routing table 
 * dump BGP routes to socket. usually called by UII
 */ 
static int
show_bgp_routing_table (uii_connection_t * uii)
{   
    return (show_f_bgp_rt_regexp (uii, 0, NULL));
}


static int 
trace_ip_bgp (uii_connection_t * uii)
{
    return (trace_f_bgp (uii, AF_INET));
}

#ifdef HAVE_IPV6
static int 
trace_ipv6_bgp (uii_connection_t * uii)
{
    return (trace_f_bgp (uii, AF_INET6));
}
#endif /* HAVE_IPV6 */


static int
load_bgp_routes (uii_connection_t * uii, char *filename)
{
    int count;
    count = load_f_bgp_routes (filename, 0);
    if (count < 0)
	user_notice (TR_ERROR, UII->trace, uii, "load from %s failed\n",
		     filename);
    else
	uii_send_data (uii, "load %d routes from %s\n", count, filename);
    return (count);
}

static int
load_ip_bgp_routes (uii_connection_t * uii, char *filename)
{
    int count;
    count = load_f_bgp_routes (filename, AF_INET);
    if (count < 0)
	user_notice (TR_ERROR, UII->trace, uii, "load from %s failed\n",
		     filename);
    else
	uii_send_data (uii, "load %d routes from %s\n", count, filename);
    return (count);
}


#ifdef HAVE_IPV6
static int
load_ipv6_bgp_routes (uii_connection_t * uii, char *filename)
{
    int count;
    count = load_f_bgp_routes (filename, AF_INET6);
    if (count < 0)
	user_notice (TR_ERROR, UII->trace, uii, "load from %s failed\n",
		     filename);
    else
	uii_send_data (uii, "load %d routes from %s\n", count, filename);
    return (count);
}
#endif /* HAVE_IPV6 */


/*
 * router id %a 
 */
int
config_bgp_router_id (uii_connection_t * uii, prefix_t *prefix)
{
    if (uii->negative) {
#ifndef BGP_MULTI
        set_BGP (BGP_MY_ID, MRT->default_id, NULL);
#else /* BGP_MULTI */
        BGP->current_bgp->this_id = MRT->default_id;
#endif /* BGP_MULTI */
	return (1);
    }
#ifndef BGP_MULTI
    set_BGP (BGP_MY_ID, prefix_tolong (prefix), NULL);
#else /* BGP_MULTI */
    /* This is a rather painful kludge.  I need to get some of this global
       configuration stuff away from being global.  Maybe a global BGP
       with an array (hash) of bgp router instances in it. -- binkertn */
    BGP->current_bgp->this_id = prefix_tolong (prefix);
    /*    set_BGP (BGP_CURRENT_ID, prefix_tolong (prefix), NULL); */
#endif /* BGP_MULTI */
    trace (TR_TRACE, MRT->trace, "router-id %s\n", prefix_toa (prefix));
    Deref_Prefix (prefix);
    return (1);
}


static int
config_bgp_cluster_id (uii_connection_t * uii, prefix_t *prefix)
{
    if (uii->negative) {
#ifndef BGP_MULTI
	BGP->cluster_id = MRT->default_id;
#else /* BGP_MULTI */
        BGP->current_bgp->cluster_id = MRT->default_id;
#endif /* BGP_MULTI */
	return (1);
    }
#ifndef BGP_MULTI
    BGP->cluster_id = prefix_tolong (prefix);
#else /* BGP_MULTI */
    BGP->current_bgp->cluster_id = prefix_tolong (prefix);
#endif /* BGP_MULTI */
    trace (TR_TRACE, MRT->trace, "cluster-id %s\n", prefix_toa (prefix));
    Deref_Prefix (prefix);
    return (1);
}


void
config_bgp_init (void)
{
    set_uii (UII, UII_PROMPT, UII_CONFIG_ROUTER_BGP, "Router BGP> ", 0);

    uii_add_command2 (UII_NORMAL, 0, "show ip bgp summary", 
		      show_ip_bgp_summary, 
		      "Show BGP peers and their status");
    uii_add_command2 (UII_NORMAL, 0, "show ip bgp neighbors", 
		      show_ip_bgp_neighbors,
		      "Show BGP peers and their status");
    uii_add_command2 (UII_NORMAL, 0, "show ip bgp", 
		      show_ip_bgp, "Show BGP routing table");
    uii_add_command2 (UII_NORMAL, 0, "show ip bgp regexp %S",
		      show_ip_bgp_regexp,
		      "Show BGP routes matching the regular expression");
    uii_add_command2 (UII_NORMAL, 0, "show ip bgp %m %S",
		      show_bgp_routes,
		      "Show BGP routes matching the expression");
    uii_add_command2 (UII_NORMAL, 0, 
		      "show ip bgp neighbors (%M|%n|*) errors",
		      show_ip_bgp_neighbors_errors,
		      "Show BGP recent errors [with the peer]");
    uii_add_command2 (UII_NORMAL, 0, 
		      "show ip bgp neighbors (%M|%n|*) routes",
		      show_ip_bgp_neighbors_routes,
		      "Show BGP routes sent [to the peer]");
    uii_add_command2 (UII_ENABLE, 0, 
		      "clear ip bgp (%M|%n|*)", kill_peer,
		      "Close BGP peering session");
    uii_add_command2 (UII_CONFIG, 0,
		      "ip as-path access-list %d (permit|deny) %S",
		      config_ip_as_filter, "Defines AS path filter");
    uii_add_command2 (UII_CONFIG, 0,
		      "no ip as-path access-list %d (permit|deny) %S",
		      config_ip_as_filter, "Deletes AS path filter");

    uii_add_command2 (UII_ENABLE, 0, 
		      "trace ip bgp neighbor (%M|%n)", 
		      trace_bgp_neighbor,
		      "Enable trace bgp neighbor");
    uii_add_command2 (UII_ENABLE, 0, 
		      "no trace ip bgp neighbor (%M|%n)", 
		      trace_bgp_neighbor, "Disable trace bgp neighbor");
    uii_add_command2 (UII_ENABLE, 0, "trace ip bgp", 
		      trace_ip_bgp, "BGP information");
    uii_add_command2 (UII_ENABLE, 0, "no trace ip bgp", 
		      trace_ip_bgp, "BGP information");
    uii_add_command2 (UII_ENABLE, 0, "dump ip bgp routes %s",
		      dump_ip_bgp_routes, "Dumps BGP routing table");
    uii_add_command2 (UII_ENABLE, 0, "load ip bgp routes %s",
		      load_ip_bgp_routes, "Loads BGP routing table");
    uii_add_command2 (UII_ENABLE, 0, "load bgp routes %s",
		      load_bgp_routes, "Loads BGP routing table");

    uii_add_command2 (UII_CONFIG, 0, "dump ip bgp routes %s %s",
		      config_dump_ip_bgp_routes, "Dumps BGP routing table");
    uii_add_command2 (UII_CONFIG, 0, "no dump ip bgp routes",
		      config_dump_ip_bgp_routes, 
		      "Stops dumping BGP routing table");
    uii_add_command2 (UII_CONFIG, 0, 
		      "dump ip bgp (updates|all) %s %s",
		      config_dump_ip_bgp_updates, 
		      "Dumps BGP updates and state changes");
    uii_add_command2 (UII_CONFIG, 0, 
		      "no dump ip bgp (updates|all) %s %s",
		      config_dump_ip_bgp_updates, 
		      "Stops dumping BGP updates and state changes");

#ifdef HAVE_IPV6
    uii_add_command2 (UII_NORMAL, 0, "show ipv6 bgp summary", 
		      show_ipv6_bgp_summary, 
		      "Show IPv6 BGP peers and their status");
    uii_add_command2 (UII_NORMAL, 0, "show ipv6 bgp neighbors", 
		      show_ipv6_bgp_neighbors,
		      "Show IPv6 BGP peers and their status");
    uii_add_command2 (UII_NORMAL, 0, "show ipv6 bgp", 
		      show_ipv6_bgp, "Show IPv6 BGP routing table");
    uii_add_command2 (UII_NORMAL, 0, "show ipv6 bgp regexp %S",
		      show_ipv6_bgp_regexp,
		      "Show IPv6 BGP routes matching the regular expression");
    uii_add_command2 (UII_NORMAL, 0, "show ipv6 bgp %P %S",
		      show_bgp_routes,
		      "Show IPv6 BGP routes matching the expression");
    uii_add_command2 (UII_NORMAL, 0, 
		      "show ipv6 bgp neighbors (%A|%n|*) errors",
		      show_ipv6_bgp_neighbors_errors,
		      "Show IPv6 BGP recent errors [with the peer]");
    uii_add_command2 (UII_NORMAL, 0, 
		      "show ipv6 bgp neighbors (%A|%n|*) routes",
		      show_ipv6_bgp_neighbors_routes,
		      "Show IPv6 BGP routes sent [to the peer]");
    uii_add_command2 (UII_ENABLE, 0, 
		      "clear ipv6 bgp (%A|%n|*)", kill_peer,
		      "Close IPv6 BGP peering session");
    uii_add_command2 (UII_CONFIG, 0,
		      "ipv6 as-path access-list %d (permit|deny) %S",
		      config_ip_as_filter, "Defines AS path filter");
    uii_add_command2 (UII_CONFIG, 0,
		      "no ipv6 as-path access-list %d (permit|deny) %S",
		      config_ip_as_filter, "Deletes AS path filter");

    uii_add_command2 (UII_ENABLE, 0, 
		      "trace ipv6 bgp neighbor (%A|%n)", 
		      trace_bgp_neighbor,
		      "Enable trace IPv6 BGP neighbor");
    uii_add_command2 (UII_ENABLE, 0, 
		      "no trace ipv6 bgp neighbor (%A|%n)", 
		      trace_bgp_neighbor, 
		      "Disable trace IPv6 BGP neighbor");

    uii_add_command2 (UII_ENABLE, 0, "trace ipv6 bgp", 
		      trace_ipv6_bgp, "Traces IPv6 BGP information");
    uii_add_command2 (UII_ENABLE, 0, "no trace ipv6 bgp", 
		      trace_ipv6_bgp, "Untraces IPv6 BGP information");
    uii_add_command2 (UII_ENABLE, 0, "dump ipv6 bgp routes %s",
		      dump_ipv6_bgp_routes, 
		      "Dumps IPv6 BGP routing table");
    uii_add_command2 (UII_ENABLE, 0, "load ipv6 bgp routes %s",
		      load_ipv6_bgp_routes, 
		      "Loads IPv6 BGP routing table");

    uii_add_command2 (UII_CONFIG, 0, "dump ipv6 bgp routes %s %s",
		      config_dump_ipv6_bgp_routes, 
		      "Dumps IPv6 BGP routing table");
    uii_add_command2 (UII_CONFIG, 0, "no dump ipv6 bgp routes",
		      config_dump_ipv6_bgp_routes, 
		      "Stops dumping IPv6 BGP routing table");
    uii_add_command2 (UII_CONFIG, 0, 
		      "dump ipv6 bgp (updates|all) %s %s",
		      config_dump_ipv6_bgp_updates,
		      "Dumps IPv6 BGP updates and state changes");
    uii_add_command2 (UII_CONFIG, 0, 
		      "no dump ipv6 bgp (updates|all) %s %s",
		      config_dump_ipv6_bgp_updates, 
		      "Stops dumping IPv6 BGP updates and state changes");
#endif /* HAVE_IPV6 */

    uii_add_command2 (UII_NORMAL, 0, "show bgp", show_bgp,
		      "Show BGP peers and their status");
    uii_add_command2 (UII_NORMAL, 0, "show bgp summary", show_bgp_summary,
		      "Show BGP peers and their status");
#ifdef BGP_MULTI
    uii_add_command2 (UII_NORMAL, 0, "show bgp local", show_bgp_local,
		      "Show locally configured ASes");
#endif /* BGP_MULTI */
    uii_add_command2 (UII_NORMAL, 0, "show bgp neighbors", show_bgp,
		      "Show BGP peers and their status");
    uii_add_command2 (UII_NORMAL, COMMAND_NODISPLAY, "show view %d", show_view,
		      "Show the BGP routing table for the view");
    uii_add_command2 (UII_NORMAL, 0, "show bgp routes",
		      show_bgp_routing_table, "Show BGP routing table");

#ifdef BGP_MULTI
    uii_add_command2 (UII_NORMAL, 0, "show bgp routes as %d",
		      show_bgp_rt_as, "Show BGP routing table");
    uii_add_command2 (UII_NORMAL, 0, "show bgp routes view %d",
		      show_bgp_rt_view_no, "Show BGP routing table");
#endif /* BGP_MULTI */
    uii_add_command2 (UII_NORMAL, 0, "show bgp regexp %S",
		      show_bgp_rt_regexp,
		      "Show BGP routes matching the regular expression");
    uii_add_command2 (UII_NORMAL, 0, "show bgp %m %S",
		      show_bgp_routes,
		      "Show BGP routes matching the expression");
    uii_add_command2 (UII_NORMAL, 0, 
		      "show bgp neighbors (%M|%n|*) errors",
		      show_bgp_neighbors_errors,
		      "Show BGP recent errors [with the peer]");
    uii_add_command2 (UII_NORMAL, 0, 
		      "show bgp neighbors (%M|%n|*) routes",
		      show_bgp_neighbors_routes,
		      "Show BGP routes sent [to the peer]");

    uii_add_command2 (UII_ENABLE, 0, 
		      "clear bgp (%M|%n|*)", kill_peer,
		      "Close BGP peering session");

    uii_add_command2 (UII_CONFIG, 0, "router bgp %d",
		      config_router_bgp,
		      "Enables BGP routing protocol with AS number");
    uii_add_command2 (UII_CONFIG, 0, "no router bgp",
		      config_router_bgp,
		      "Disables BGP routing protocol");
    uii_add_command2 (UII_CONFIG, 0, "bgp router-id %a",
		      config_bgp_router_id, "Set BGP router-id");
    uii_add_command2 (UII_CONFIG, 0, "no bgp router-id",
		      config_bgp_router_id, "Unset BGP router-id");
    uii_add_command2 (UII_CONFIG, 0, "bgp cluster-id %a",
		      config_bgp_cluster_id, "Set BGP cluster-id");
    uii_add_command2 (UII_CONFIG, 0, "no bgp cluster-id",
		      config_bgp_cluster_id, "Unset BGP cluster-id");

    uii_add_command2 (UII_CONFIG, 0,
		      "as-path access-list %d (permit|deny) %S",
		      config_ip_as_filter, "Defines AS path filter");

    uii_add_command2 (UII_ENABLE, 0, 
		      "trace bgp neighbor (%M|%n)", trace_bgp_neighbor,
		      "Enable trace bgp neighbor");
    uii_add_command2 (UII_ENABLE, 0, 
		      "no trace bgp neighbor (%M|%n)", 
		      trace_bgp_neighbor, "Disable trace bgp neighbor");
    uii_add_command2 (UII_ENABLE, COMMAND_NODISPLAY, 
		      "trace bgp view (*|inet|inet6|%d)", 
		      trace_bgp_view, "Enable trace bgp view");
    uii_add_command2 (UII_ENABLE, COMMAND_NODISPLAY, 
		      "no trace bgp view (*|inet|inet6|%d)", 
		      no_trace_bgp_view, "Disable trace bgp view");

    uii_add_command2 (UII_ENABLE, 0, "trace bgp", trace_bgp,
		      "Enable trace BGP");
    uii_add_command2 (UII_ENABLE, 0, "no trace bgp", no_trace_bgp,
		      "Disable trace BGP");

    uii_add_command2 (UII_CONFIG, 0, 
		"dump bgp view %dview_number %sFilename %sInterval(e.g._1m)",
		 config_dump_bgp_view, "Dumps BGP routing table for the view");
    uii_add_command2 (UII_CONFIG, 0, "dump bgp routes %s %s",
		      config_dump_bgp_routes, "Dumps BGP routing table");
    uii_add_command2 (UII_CONFIG, 0, "no dump bgp routes",
		      config_dump_bgp_routes, 
		      "Stops dumping BGP routing table");
    uii_add_command2 (UII_CONFIG, 0, 
		      "dump-binary bgp routes %sFilename %sInterval",
		      config_dump_bgp_routes_binary, 
		      "Dumps Binary BGP routing table");
    uii_add_command2 (UII_CONFIG, 0, 
		      "no dump-binary bgp routes",
		      config_dump_bgp_routes_binary, 
		      "Stops dumping Binary BGP routing table");
    uii_add_command2 (UII_CONFIG, 0, 
		      "dump bgp (updates|all) %sFilename %sInterval(e.g._1m)",
		      config_dump_bgp_updates, 
		      "Dumps BGP updates and state changes");
    uii_add_command2 (UII_CONFIG, 0, 
		      "no dump bgp (updates|all)",
		      config_dump_bgp_updates, 
		      "Stops dumping BGP updates and state changes");

#ifdef BGP_MULTI
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0, "add interface %m",
		      config_router_bgp_add_interface,
		      "Adds a useable interface to the active bgp config");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0, "delete interface %m",
		      config_router_bgp_delete_interface,
		      "Removes an interface to the active bgp config");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0, "bind interface only",
		      config_router_bgp_bind_interface_only,
		      "Allows connectons only over bound interfaces");
#endif /* BGP_MULTI */

    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0, "network %m",
		      config_router_bgp_network_prefix, 
		      "Originates routes in BGP");

    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
	"redistribute (static|ospf|rip|ripng|direct|connected|kernel)",
		      config_router_bgp_redistribute,
		      "Redistribute route from the protocol");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
	"no redistribute (static|ospf|rip|ripng|direct|connected|kernel)",
		      config_router_bgp_redistribute,
		      "Not redistribute route from the protocol");

    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor %M remote-as %d",
		   config_router_neighbor_remoteas, "Adds an BGP neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor %M remote-as %d",
	     no_config_router_neighbor_remoteas, "Deletes an BGP neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor (%M|%n) description %s",
		       config_router_neighbor_description, 
		      "Describes BGP neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor (%M|%n) description",
	     	      config_router_neighbor_description, 
		      "Undescribe BGP neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor (%M|%n) update-source (%M|%n)",
		   config_router_neighbor_update_source, 
		   "Specify the source address");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor (%M|%n) update-source (%M|%n)",
		   config_router_neighbor_update_source, 
		   "Unspecify the source address");

    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor (%M|%n) view %d", config_router_neighbor_view,
		      "Assigns a view for the neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor (%M|%n) weight %d", 
		       config_router_neighbor_weight,
		      "Sets a weight for the neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor (%M|%n) weight %d", 
		       config_router_neighbor_weight,
		      "Resets a weight for the neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor (%M|%n) maximum-prefix %d", 
		       config_router_neighbor_maximum_prefix,
		      "Sets max number of prefixes for the neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor (%M|%n) maximum-prefix %d", 
		       config_router_neighbor_maximum_prefix,
		      "Resets max number of prefixes for the neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor (%M|%n) alias %n",
		      config_router_neighbor_alias, "Obsolete");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
	"neighbor (%M|%n) "
	"(transparent-as|transparent-nexthop|passive|next-hop-self|"
	"route-reflector-client)",
		      config_router_neighbor_option,
		      "Sets an option for the neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
	"no neighbor (%M|%n) "
	"(transparent-as|transparent-nexthop|passive|next-hop-self|"
	"route-reflector-client)",
		      config_router_neighbor_option,
		      "Resets an option for the neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
	"neighbor (%M|%n) (distribute-list|filter-list) %d (in|out)",
		      config_router_neighbor_list_in_out,
		      "Applies the filter for incoming or outgoing");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
	"no neighbor (%M|%n) (distribute-list|filter-list) %d (in|out)",
		      config_router_neighbor_list_in_out,
		      "Removes the filter for incoming or outgoing");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor (%M|%n) route-map %d (in|out)",
		      config_router_neighbor_routemap_in_out,
		      "Applies the route map for incoming or outgoing");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor (%M|%n) route-map %d (in|out)",
		      config_router_neighbor_routemap_in_out,
		      "Removes the route map for incoming or outgoing");
#ifdef HAVE_IPV6
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor (%M|%n) bgp4+ (0|1|old|new|rfc|auto)",
		      config_router_neighbor_bgp4plus,
		      "Specifies BGP4+ packet format");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor (%M|%n) bgp4+ (0|1|old|new|rfc|auto)",
		      config_router_neighbor_bgp4plus,
		      "Unspecifies BGP4+ packet format");
#endif /* HAVE_IPV6 */
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
	"neighbor (%M|%n) (holdtime|keepalive|connectretry|starttime) %d",
		      config_router_neighbor_time,
		      "Sets timer for neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
	"no neighbor (%M|%n) (holdtime|keepalive|connectretry|starttime)",
		      config_router_neighbor_time,
		      "Resets timer for neighbor to the default");

    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor %n peer %M %S",
		   config_router_neighbor_n_peer, "Adds an BGP neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor %n peer",
	     no_config_router_neighbor_n, "Deletes an BGP neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor %n remote-as %d",
		       config_router_neighbor_n_remoteas, 
		      "Sets AS number to the neighbor");

    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "neighbor %n neighbor-list %d",
		       config_router_neighbor_list, 
		      "Allows anonymous neighbor peers");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no neighbor %n neighbor-list",
		       config_router_neighbor_list, 
		      "Deletes anonymous neighbor peers");

    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0, 
		      "neighbor (%M|%n) trace", trace_bgp_neighbor,
		      "Enable trace bgp neighbor");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0, 
		      "no neighbor (%M|%n) trace",
		      trace_bgp_neighbor, "Disable trace bgp neighbor");

    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "aggregate-address %m %S", config_router_aggregate,
		      "Creates an aggregate entry to prefix");
    uii_add_command2 (UII_CONFIG_ROUTER_BGP, 0,
		      "no aggregate-address %m %S", 
		       no_config_router_aggregate,
		      "Deletes an aggregate entry to prefix");
}
