/*
 * $Id: bgp.h,v 1.35 1999/04/30 23:46:53 masaki Exp $
 */

#ifndef _BGP_H
#define _BGP_H

#include <mrt.h>
#include <timer.h>
#include <trace.h>
#include <io.h>
#include <schedule.h>
#include <hash.h>

#include <rib.h>
#include <bgp_proto.h>

struct _bgp_attr_t;
#define bgp_attr_t struct _bgp_attr_t

/* various settable attributes */
#define DEFAULT_PEER_NAME	"glasshouse.merit.edu"
#define DEFAULT_PEER_AS		185
#ifndef BGP_MULTI
#define DEFAULT_MY_AS		6503
#define DEFAULT_MY_ID		0xc66c3cb0
#else /* BGP_MULTI */
#define DEFAULT_LOCAL_AS        6503
#define DEFAULT_ROUTER_ID       0xc66c3cb0

#define MAX_AS_NUMBER           65535
#endif /* BGP_MULTI */

#define DEFAULT_DATAFILE	"/tmp/bgp.data"

#define DEFAULT_BGP_VERSION	4
#define DEFAULT_BGP4PLUS_VERSION 0

#define DEFAULT_CONNET_RETRY_INTERVAL	120 /* was 60 */
#define DEFAULT_START_INTERVAL		10 /* RFC suggests 60 */
#define DEFAULT_KEEPALIVE_INTERVAL	30
#define DEFAULT_HOLDTIME_INTERVAL	90 /* was 180 */
#define BGP_OPEN_TIMEOUT		240

#define BGP_MIN_HOLDTIME        20      /* What we'll accept */

/* pdu types */
#define BGP_OPEN		1
#define BGP_UPDATE		2
#define BGP_NOTIFICATION	3
#define BGP_KEEPALIVE		4

/* saving routing table to disk (mrtd) */
#define DUMP_ASCII		1
#define DUMP_BINARY		2

#define DUMP_DIR_RECV	0x01
#define DUMP_DIR_SEND	0x01
#define DUMP_DIR_BOTH   (DUMP_DIR_RECV|DUMP_DIR_SEND)

extern char *sbgp_states[];
extern char *sbgp_pdus[];
extern char *sbgp_events[];
extern char *s_bgp_error[];
extern char *s_bgp_error_header[];
extern char *s_bgp_error_open[];
extern char *s_bgp_error_update[];


#define BGP_TRANSPARENT_AS	0x0001
#define BGP_TRANSPARENT_NEXTHOP	0x0002
#define BGP_CONNECT_PASSIVE	0x0004	/* don't try to connect */
#define BGP_BGP4PLUS		0x0008
#define BGP_BGP4PLUS_00	BGP_BGP4PLUS	/* draft 00 */
#define BGP_BGP4PLUS_01		0x0010	/* draft 01 */
#define BGP_BGP4PLUS_AUTO	0x0020	/* auto detection */
#define BGP_EBGP_MULTIHOP	0x0040
#define BGP_NEXTHOP_SELF	0x0080
#define BGP_INTERNAL		0x0100
#define BGP_ROUTE_REFLECTOR_CLIENT 0x0200
#define BGP_PEER_IS_MULTIHOP	0x0400


typedef struct listen_socket_t {
    int sockfd;
    int ref_count;
    prefix_t *prefix;
    interface_t *interface;
    pthread_mutex_t mutex_lock;
} listen_socket_t;

#ifdef BGP_MULTI
typedef struct _bgp_local_t {
    int this_as;

  /* must fix ID stuff when adding interface things.  Currently everything
     gets the same router_id which is unnacceptable.  -- binkertn */
    u_long this_id;  
    u_long cluster_id;  
    int view_no;
    pthread_mutex_t mutex_lock;

    int bind_interface_only;
    int num_interfaces;
    LINKED_LIST *ll_interfaces;
} bgp_local_t;
#endif /* BGP_MULTI */

typedef struct _bgp_peer_t {
    char *name;
    char *description;
    interface_t *bind_if;
    prefix_t *bind_addr;
    prefix_t *local_addr;	/* local address for the peer */
    /* XXX redundant? interface's address */
    /* prefix_t *peer_addr; */	/* peer address only */
    /* XXX redundant? gateway's prefix */
    /* int peer_port; */	/* port number in case it isn't 179 */
    /* u_long peer_id; */	/* peer's router id */
    gateway_t *gateway;
    nexthop_t *nexthop; 	/* immediate next hop */
    int new_as;			/* this will be used at the next start */
    LINKED_LIST *aliases;	/* peer's aliases */
    int sockfd;			/* sockect connected to peer */
    int sockfd_in;		/* sockect connected to peer (incoming) */
    listen_socket_t *listen_socket;	/* sockect for listening incoming connection */
    int view;			/* RIB to use when sending routes */
    u_long index;		/* index (used by view in mask) */
    u_long options;		/* see options above */
#ifdef BGP_MULTI
    int my_as;			/* host byte order */
    u_long my_id;		/* network byte order */
    bgp_local_t *local_bgp;     /* local bgp session */
#endif /* BGP_MULTI */
    int neighbor_list;		/* if > 0, this is a neighbor list */
    LINKED_LIST *children;
    struct _bgp_peer_t *parent;

    int dlist_in;		/* list num for input filtering */
    int dlist_out;		/* list num for output filtering */
    int flist_in;		/* list num for input aspath filtering */
    int flist_out;		/* list num for output aspath filtering */
    int route_map_in;		/* num for input route-map */
    int route_map_out;		/* num for output route-map */

    int state;
    int code;
    int subcode;

    pthread_mutex_t mutex_lock;
    pthread_t self;		/* my thread ID */
    schedule_t *schedule;	/* event processing */

    /* packet parsing storage */
    u_char buffer[BGPMAXPACKETSIZE * 2];
    u_char *read_ptr;		/* current ptr of read in buffer */
    u_char *start_ptr;		/* current ptr of start of packet */
    u_char *packet;		/* pointer to the packet */

    u_char buffer_in[BGPMAXPACKETSIZE];
    u_char *read_ptr_in;	/* current ptr of read in buffer */

    LINKED_LIST *send_queue;
    pthread_mutex_t send_mutex_lock;

    /* timers */
    mtimer_t *timer_Start;
    mtimer_t *timer_ConnectRetry;
    mtimer_t *timer_KeepAlive;
    mtimer_t *timer_HoldTime;

    /* time value of intervals in seconds */
    int Start_Interval;
    int ConnectRetry_Interval;
    int KeepAlive_Interval;
    int HoldTime_Interval;

    /* tracing stuff */
    trace_t *trace;

    time_t time;
    int default_weight;
    int maximum_prefix;
    int pending_in_recv_open;

    LINKED_LIST *ll_announce;
    LINKED_LIST *ll_withdraw;
    bgp_attr_t *attr;

    LINKED_LIST *ll_update_out;
    pthread_mutex_t update_mutex_lock;

    /* statistics */
    u_long num_packets_recv;
    u_long num_notifications_recv;
    u_long num_updates_recv;
    u_long num_packets_sent;
    u_long num_notifications_sent;
    u_long num_updates_sent;
    u_long num_connections_established;
    u_long num_connections_dropped;

} bgp_peer_t;

#include <view.h>

typedef struct _bgp_t {
#ifndef BGP_MULTI
    int my_as;			/* host byte order */
    u_long my_id;		/* network byte order */
    u_long cluster_id;		/* network byte order */
#endif /* BGP_MULTI */
    int cport;			/* port to send out open on */
    int lport;			/* port to listen for open on */

#if 0
    int sockfd;			/* socket we are listening on */
    int sockfd_count;           /* number of peers listening on this socket */
#endif

#ifndef BGP_MULTI
    view_t *view;		/* view from which RIB gets routes */
#else /* BGP_MULTI */
    int view_count;             /* count of current views */
    bgp_local_t *router_bgp[MAX_AS_NUMBER+1];
    bgp_local_t *current_bgp;   /* current bgp session in active configuration */
    LINKED_LIST *ll_bgp_locals;
    pthread_mutex_t locals_mutex_lock; /* a lock for ll_bgp_config */
#endif /* BGP_MULTI */

    pthread_mutex_t peers_mutex_lock; /* a lock for ll_bgp_peers */
    LINKED_LIST *ll_bgp_peers;
    int num_peers;		/* number of peers */
    int num_ipv4_peers;		/* number of IPv4 peers */

    int accept_all_peers;
    trace_t *trace;
    LINKED_LIST *ll_listen_sockets;

    view_t *views[MAX_BGP_VIEWS];
    HASH_TABLE *attr_hash;	/* hash of bgp attribute blocks */

    schedule_t *schedule;	/* event processing */

    void (*peer_down_call_fn) ();
    int (*update_call_fn) ();
    void (*state_change_fn) ();
    void (*peer_established_call_fn) ();
    int (*send_update_call_fn) ();

    /* Since view is created dynamically,
       there may be a dump request without a real view */
    char *dump_route_form[MAX_BGP_VIEWS];
    time_t dump_route_interval[MAX_BGP_VIEWS];
    time_t dump_route_time[MAX_BGP_VIEWS];
    int	dump_route_type[MAX_BGP_VIEWS];	/* DUMP_ASCII or DUMP_BINARY */
    int	dump_new_format;	/* bool */
    int	dump_direction;		/* 1 -- receiving, 2 -- sending, 3 -- both */

    char *dump_update_form;
    time_t dump_update_interval;
    time_t dump_update_time;
    u_long dump_update_types;
    int dump_update_family;

    pthread_mutex_t mutex_lock;
    LINKED_LIST *ll_networks;

    bgp_bitset_t bits_assigned;	/* record the peers' bits */

    u_int bgp_num_active_route_head;
    u_int bgp_num_active_route_node;

#define DEFAULT_LOCAL_PREF 100
    int default_local_pref;

    time_t Default_Start_Interval;
    time_t Default_ConnectRetry_Interval;
    time_t Default_KeepAlive_Interval;
    time_t Default_HoldTime_Interval;

    u_long redistribute_mask;
} bgp_t;


extern bgp_t *BGP;

enum BGP_ATTR {
    BGP_NULL = 0,
#ifndef BGP_MULTI
    BGP_MY_AS,
    BGP_MY_ID,
#else /* BGP_MULTI */
    BGP_CURRENT_BGP,
#endif /* BGP_MULTI */
    BGP_TRACE_STRUCT,
    BGP_PEER_DOWN_FN,
    BGP_PEER_ESTABLISHED_FN,
    BGP_RECV_UPDATE_FN,
    BGP_STATE_CHANGE_FN,
    BGP_SEND_UPDATE_FN, 	/* call this routine when sending updates */
    BGP_ACCEPT_ALL_PEERS,	/* start peering sesssion with anyone */
    BGP_LPORT,			/* port to listen for connections on */
    BGP_CPORT,			/* port to attempt connections to */
    BGP_USE_PORTSERVER,		/* use portserver library for listening */
    BGP_DUMP_ROUTE_FORM,	/* a file name used in dumping a route table */
    BGP_DUMP_UPDATE_FORM,	/* a file name used in dumping updates */
    BGP_DUMP_DIRECTION,		/* 1 -- receiving, 2 -- sending, 3 -- both */
};


enum BGP_PEER_ATTR {
    BGP_PEER_NULL = 0,
    BGP_PEER_DESCRIPTION,
    BGP_PEER_VIEW,
    BGP_PEER_WEIGHT,
    BGP_PEER_ALIAS,
    BGP_PEER_MAXPREF,
    BGP_PEER_SETOPT,
    BGP_PEER_RESETOPT,
    BGP_PEER_DLIST_IN,
    BGP_PEER_DLIST_OUT,
    BGP_PEER_FLIST_IN,
    BGP_PEER_FLIST_OUT,
    BGP_PEER_RTMAP_IN,
    BGP_PEER_RTMAP_OUT,
    BGP_PEER_HOLDTIME,
    BGP_PEER_KEEPALIVE,
    BGP_PEER_CONNECTRETRY,
    BGP_PEER_START,
};


/* a convenience structure to store all prefixes with same BGP
 * attributes while we are building BGP update packets
 */
typedef struct _bgp_bucket_t {
    LINKED_LIST *ll_prefix;
    bgp_attr_t *attr;
} bgp_bucket_t;


/* in bgp_pdu.c */
int bgp_read (bgp_peer_t * peer, int sockfd, u_char * ptr, int len);
int bgp_get_pdu (bgp_peer_t * peer);
int bgp_flush_queue (bgp_peer_t * peer);
int bgp_process_update (bgp_peer_t * peer);
int bgp_process_open (bgp_peer_t * peer);
int bgp_process_notify (bgp_peer_t * peer);
int bgp_send_open (bgp_peer_t * peer);
int bgp_send_keepalive (bgp_peer_t * peer);
int bgp_send_notification (bgp_peer_t * peer, int code, int subcode);
int bgp_send_notification2 (int sockfd, prefix_t * prefix, int port,
			    int code, int subcode);
int bgp_send_notify_byte (bgp_peer_t * peer, int code, int subcode, int opt);
int bgp_send_notify_word (bgp_peer_t * peer, int code, int subcode, int opt);
int bgp_send_update (bgp_peer_t * peer, int len, u_char * data);

/* in bgp_util.c */
char *bgp_notify_to_string (int code, int subcode);
char *bgpmessage2string (int type);
int bgp_start_transport_connection (bgp_peer_t * peer);
void bgp_release_resources (bgp_peer_t * peer);
bgp_peer_t *Find_BGP_Peer (prefix_t * prefix, prefix_t *usrc, 
			   interface_t *interface);
bgp_peer_t *Find_BGP_Peer_ByID (char *name);
bgp_peer_t *Add_BGP_Peer (char *name, prefix_t * prefix, prefix_t *usrc,
			  interface_t *interface, int as, trace_t * trace);
bgp_peer_t * Remove_BGP_Peer (bgp_peer_t *peer);
void Destroy_BGP_Peer (bgp_peer_t * peer);
void bgp_flush_socket (bgp_peer_t * peer);
void bgp_broken_pipe ();
void bgp_die ();
int bgp_kill_peer (gateway_t * gateway);
void bgp_stop_peer (bgp_peer_t * peer);
void bgp_start_peer (bgp_peer_t * peer);
void bgp_kill_all ();
/* void default_bgp_peer_down_fd (bgp_peer_t * peer); */
void bgp_start_peer_thread (bgp_peer_t * peer);
void bgp_start_main_thread ();
bgp_peer_t *find_bgp_peer (gateway_t * gateway);
void peer_set_as (bgp_peer_t *peer, int as);
#ifdef BGP_MULTI
void init_bgp_local (bgp_local_t *local);
int show_bgp_local (uii_connection_t * uii);
#endif /* BGP_MULTI */
int show_bgp_routes (uii_connection_t * uii, prefix_t *prefix, char *options);
#ifdef BGP_MULTI
int show_bgp_rt_as (uii_connection_t * uii, int as);
int show_bgp_rt_view_no (uii_connection_t * uii, int view_no);
#endif /* BGP_MULTI */
void bgp_peer_down (bgp_peer_t * peer);
void bgp_peer_dead (bgp_peer_t * peer);
void bgp_dump_view (uii_connection_t * uii, view_t *view, 
		    as_regexp_code_t * code, bgp_peer_t *peer, 
		    condition_t *cond);
int set_BGP (int first,...);
void start_bgp (void);
void stop_bgp (void);
void start_bgp_peer (bgp_peer_t * peer);
void bgp_start_listening (bgp_peer_t *peer);
int trace_bgp (uii_connection_t * uii);
int no_trace_bgp (uii_connection_t * uii);
int bgp_check_attr (bgp_peer_t * peer, bgp_attr_t * attr, int as);
int check_bgp_networks (prefix_t *prefix);
int bgp_in_recv_open (bgp_peer_t * peer);
listen_socket_t * init_BGP_listen (prefix_t *bind_addr, interface_t *bind_if);
int show_f_bgp (uii_connection_t * uii, int family);
int show_f_bgp_summary (uii_connection_t * uii, int family, int summary);
int show_f_bgp_rt_regexp (uii_connection_t * uii, int family, char *expr);
int show_f_bgp_neighbors_routes (uii_connection_t * uii, int family,
                             char *peer_or_star);
int show_f_bgp_neighbors_errors (uii_connection_t * uii, int family,
                             char *peer_or_star);

/* in bgp_sm.c */
void bgp_change_state (bgp_peer_t * peer, int state, int event);
#ifdef notdef
void bgp_sm_state_idle (bgp_peer_t * peer, int event);
void bgp_sm_state_active (bgp_peer_t * peer, int event);
void bgp_sm_state_connect (bgp_peer_t * peer, int event);
void bgp_sm_state_opensent (bgp_peer_t * peer, int event);
void bgp_sm_state_openconfirm (bgp_peer_t * peer, int event);
void bgp_sm_state_established (bgp_peer_t * peer, int event);
#endif
void bgp_sm_process_event (bgp_peer_t * peer, int event);

/* in bgp_timer.c */
void bgp_timer_ConnectRetry_fire (mtimer_t * timer, bgp_peer_t * peer);
void bgp_timer_KeepAlive_fire (mtimer_t * timer, bgp_peer_t * peer);
void bgp_timer_HoldTime_fire (mtimer_t * timer, bgp_peer_t * peer);
void bgp_timer_StartTime_fire (mtimer_t * timer, bgp_peer_t * peer);

/* in bgp_thread.c */
void bgp_schedule_timer (mtimer_t * timer, bgp_peer_t * peer);
void bgp_schedule_socket (bgp_peer_t * peer);
void bgp_get_config_neighbor (bgp_peer_t * peer);
void bgp_schedule_get_config_neighbor (bgp_peer_t * peer);

/* public functions */
void init_BGP (trace_t * ltrace);

/* bgp_dump.c */
void bgp_write_mrt_msg (bgp_peer_t * peer, enum MRT_MSG_BGP_TYPES bgptype,
			u_char * buf, int length);
void bgp_write_status_change (bgp_peer_t * peer, u_short state);
int bgp_table_dump_write (int fd, int type, int subtype, 
			  int seq_num, int viewno, u_char *pp, int len);
u_char *bgp_table_dump_entry (u_char *cp, u_char *end, int type, int subtype,
                      int viewno, prefix_t *prefix, int status,
                      time_t originated, bgp_attr_t *attr);
bgp_peer_t * create_fake_peer (void);
int dump_view_bgp_routes (int viewno, char *filename, int dump_type);

#define Find_BGP_Peer_ByPrefix(prefix) Find_BGP_Peer(prefix, NULL, NULL)

#undef bgp_attr_t
#endif /* _BGP_H */
