/*
 * Copyright (C), 2000-2004 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <config.h>

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#ifdef HAVE_SETJMP_H
#include <setjmp.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#ifdef HAVE_TIME_H
#include <time.h>
#endif

#ifndef HAVE_SOL_IP
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#endif

#ifdef HAVE_NETINET_IP_ICMP_H
#include <netinet/ip_icmp.h>
#endif

#include "monitor.h"
#include "alert.h"
#include "event.h"
#include "socket.h"
#include "net.h"
#include "device.h"
#include "process.h"


/**
 *  Implementation of validation engine
 *
 *  @author Jan-Henrik Haukeland, <hauk@tildeslash.com>
 *  @author Olivier Beyssac, <ob@r14.freenix.org> (check_skip)
 *  @author Martin Pala <martinp@tildeslash.com>
 *  @author Christian Hopp <chopp@iei.tu-clausthal.de>
 *
 *  @version \$Id: validate.c,v 1.123 2004/08/16 22:38:33 martinp Exp $
 *  @file
 */


/* -------------------------------------------------------------- Prototypes */


static void check_process_state(Service_T);
static void check_process_resources(Service_T, Resource_T);
static void check_process_connection(Service_T, Port_T);
static void check_device_resources(Service_T, Device_T);
static void check_timestamp(Service_T);
static void check_checksum(Service_T);
static void check_perm(Service_T);
static void check_uid(Service_T);
static void check_gid(Service_T);
static void check_size(Service_T);
static int  compare_value(int, int, int);
static int  check_timeout(Service_T);
static int  check_skip(Service_T);

ProcessTree_T *ptree=NULL;     
int            ptreesize=0;    
ProcessTree_T *oldptree=NULL;  
int            oldptreesize=0; 


/* ---------------------------------------------------------------- Public */


/**
 *  This function contains the main check machinery for  monit. The
 *  validate function check services in the service list to see if
 *  they will pass all defined tests.
 */
void validate() {

  Service_T s;
  sigset_t ns, os;

  if(Run.doprocess)
    initprocesstree(&ptree, &ptreesize, &oldptree, &oldptreesize);

  if(! update_system_load(ptree, ptreesize))
    log("Update of loadavg has failed!\n");
  time(&systeminfo.collected);

  for(s= servicelist; s; s= s->next) {
    if(s->visited)
      continue;
    LOCK(s->mutex)
      set_signal_block(&ns, &os);
      if(s->monitor && !check_skip(s) && !check_timeout(s)) {
        s->check(s);
        s->monitor= MONITOR_YES;
      }
      time(&s->collected);
      unset_signal_block(&os);
    END_LOCK;
  }

  if(Run.doprocess)
    delprocesstree(&oldptree, oldptreesize);

  reset_depend();

}


/**
 * Validate a given process service s. Events are posted according to 
 * its configuration. In case of a fatal event FALSE is returned.
 */
int check_process(Service_T s) {

  pid_t  pid= -1;
  Port_T pp= NULL;
  Resource_T pr= NULL;

  ASSERT(s);

  /* Test for running process */
  if(!(pid= is_process_running(s))) {
    /* Reset the proc info object to prevent false data in the first run */
    reset_procinfo(s);
    Event_post(s, EVENT_NONEXIST, TRUE, s->action_NONEXIST,
      "'%s' process is not running", s->name);
    return FALSE;
  } else {
    Event_post(s, EVENT_NONEXIST, FALSE, s->action_NONEXIST,
      "'%s' process is running with pid %d", s->name, (int)pid);
  }

  s->inf->uptime= get_process_uptime(s->path);

  if(Run.doprocess) {
    if(update_process_data(s, ptree, ptreesize, pid)) {
      check_process_state(s);
      for(pr= s->resourcelist; pr; pr= pr->next) {
        check_process_resources(s, pr);
      }
    } else {
      log("'%s' failed to get service data\n", s->name);
    }

  }

  /* Test each host:port and protocol in the service's portlist */
  if(s->portlist)
    for(pp= s->portlist; pp; pp= pp->next)
      check_process_connection(s, pp);

  return TRUE;
  
}


/**
 * Validate a given device service s. Events are posted according to 
 * its configuration. In case of a fatal event FALSE is returned.
 */
int check_device(Service_T s) {

  Device_T td;
  struct stat stat_buf;

  ASSERT(s);

  if(stat(s->path, &stat_buf) != 0) {
    Event_post(s, EVENT_NONEXIST, TRUE, s->action_NONEXIST,
      "'%s' device doesn't exist", s->name);
    return FALSE;
  } else {
    s->inf->st_mode= stat_buf.st_mode;
    s->inf->st_uid= stat_buf.st_uid;
    s->inf->st_gid= stat_buf.st_gid;
    Event_post(s, EVENT_NONEXIST, FALSE, s->action_NONEXIST,
      "'%s' device exist", s->name);
  }

  if(!device_usage(s->inf, s->path)) {
    Event_post(s, EVENT_DATA, TRUE, s->action_DATA,
      "'%s' unable to read device %s state\n", s->name, s->path);
    return FALSE;
  } else {
    Event_post(s, EVENT_DATA, FALSE, s->action_DATA,
      "'%s' succeeded getting device statistic for %s", s->name, s->path);
  }

  if(s->perm)
    check_perm(s);

  if(s->uid)
    check_uid(s);

  if(s->gid)
    check_gid(s);

  if(s->devicelist)
    for(td= s->devicelist; td; td= td->next)
      check_device_resources(s, td);

  return TRUE;

}


/**
 * Validate a given file service s. Events are posted according to 
 * its configuration. In case of a fatal event FALSE is returned.
 */
int check_file(Service_T s) {

  struct stat stat_buf;

  ASSERT(s);

  if(stat(s->path, &stat_buf) != 0) {
    Event_post(s, EVENT_NONEXIST, TRUE, s->action_NONEXIST,
      "'%s' file doesn't exist", s->name);
    return FALSE;
  } else {
    s->inf->st_mode= stat_buf.st_mode;
    s->inf->st_uid= stat_buf.st_uid;
    s->inf->st_gid= stat_buf.st_gid;
    s->inf->st_size= stat_buf.st_size;
    s->inf->timestamp= MAX(stat_buf.st_mtime, stat_buf.st_ctime);
    Event_post(s, EVENT_NONEXIST, FALSE, s->action_NONEXIST,
      "'%s' file exist", s->name);
  }

  if(!S_ISREG(s->inf->st_mode)) {
    Event_post(s, EVENT_INVALID, TRUE, s->action_INVALID,
      "'%s' is not regular file", s->name);
    return FALSE;
  } else {
    Event_post(s, EVENT_INVALID, FALSE, s->action_INVALID,
      "'%s' is regular file", s->name);
  }

  if(s->checksum)
    check_checksum(s);

  if(s->perm)
    check_perm(s);

  if(s->uid)
    check_uid(s);

  if(s->gid)
    check_gid(s);

  if(s->sizelist)
    check_size(s);

  if(s->timestamplist)
    check_timestamp(s);

  return TRUE;

}


/**
 * Validate a given directory service s. Events are posted according to
 * its configuration. In case of a fatal event FALSE is returned.
 */
int check_directory(Service_T s) {

  struct stat stat_buf;

  ASSERT(s);

  if(stat(s->path, &stat_buf) != 0) {
    Event_post(s, EVENT_NONEXIST, TRUE, s->action_NONEXIST,
      "'%s' directory doesn't exist", s->name);
    return FALSE;
  } else {
    s->inf->st_mode= stat_buf.st_mode;
    s->inf->st_uid= stat_buf.st_uid;
    s->inf->st_gid= stat_buf.st_gid;
    s->inf->timestamp= MAX(stat_buf.st_mtime, stat_buf.st_ctime);
    Event_post(s, EVENT_NONEXIST, FALSE, s->action_NONEXIST,
      "'%s' directory exist", s->name);
  }

  if(!S_ISDIR(s->inf->st_mode)) {
    Event_post(s, EVENT_INVALID, TRUE, s->action_INVALID,
      "'%s' is not directory", s->name);
    return FALSE;
  } else {
    Event_post(s, EVENT_INVALID, FALSE, s->action_INVALID,
      "'%s' is directory", s->name);
  }

  if(s->perm)
    check_perm(s);

  if(s->uid)
    check_uid(s);

  if(s->gid)
    check_gid(s);

  if(s->timestamplist)
    check_timestamp(s);

  return TRUE;

}


/**
 * Validate a remote service.
 * @param s The remote service to validate
 * @return FALSE if there was an error otherwise TRUE
 */
int check_remote_host(Service_T s) {

  Port_T p= NULL;
  Icmp_T icmp;

  ASSERT(s);

  /* Test each icmp type in the service's icmplist */
  if(s->icmplist) {
    for(icmp= s->icmplist; icmp; icmp= icmp->next) {

      switch(icmp->type) {
      case ICMP_ECHO:

        icmp->response= icmp_echo(s->path, icmp->timeout);

        if(icmp->response < 0) {
          icmp->is_available= FALSE;
          DEBUG("'%s' icmp ping failed\n", s->name);
          Event_post(s, EVENT_ICMP, TRUE, icmp->action,
            "'%s' failed ICMP test [%s]", s->name, icmpnames[icmp->type]);
        } else {
          icmp->is_available= TRUE;
          DEBUG("'%s' icmp ping succeeded [response time %.3fs]\n",
            s->name, icmp->response);
          Event_post(s, EVENT_ICMP, FALSE, icmp->action,
            "'%s' passed ICMP test [%s]", s->name, icmpnames[icmp->type]);
        }
        break;

      default:
        log("'%s' error -- unknown ICMP type: [%d]\n", s->name, icmp->type);
        return FALSE;

      }
    }
  }

  /* Test each host:port and protocol in the service's portlist */
  if(s->portlist)
    for(p= s->portlist; p; p= p->next)
      check_process_connection(s, p);

  return TRUE;
  
}


/* --------------------------------------------------------------- Private */


/**
 * Test the connection and protocol
 */
static void check_process_connection(Service_T s, Port_T p) {

  Socket_T socket;
  volatile int rv= TRUE;
  char report[STRLEN]={0};
  struct timeval t1;
  struct timeval t2;

  ASSERT(s && p);

  p->response = -1;

  /* Get time of connection attempt beginning */
  gettimeofday(&t1, NULL);

  /* Open a socket to the destination INET[hostname:port] or UNIX[pathname] */
  socket= socket_create(p);
  if(!socket) {
    snprintf(report, STRLEN,
             "'%s' failed, cannot open a connection to %s",
             s->name, p->address);
    rv= FALSE;
    goto error;
  } else {
    DEBUG("'%s' succeeded connecting to %s\n",
      s->name, p->address);
  }

  /* Verify that the socket is ready for i|o */
  if(! socket_is_ready(socket)) {
    snprintf(report, STRLEN,
             "'%s' failed, the socket at %s is not ready for i|o -- %s",
             s->name, p->address, STRERROR);
    rv= FALSE;
    goto error;
  }

  /* Run the protocol verification routine through the socket */
  if(! p->protocol->check(socket)) {
    snprintf(report, STRLEN,
             "'%s' failed protocol test [%s] at %s.",
             s->name, p->protocol->name, p->address);
    rv= FALSE;
    goto error;
  } else {
    DEBUG("'%s' succeeded testing protocol [%s] at %s\n",
          s->name, p->protocol->name, p->address);
  }

  /* Get time of connection attempt finish */
  gettimeofday(&t2, NULL);

  /* Get the response time */
  p->response= (double)(t2.tv_sec  - t1.tv_sec) +
               (double)(t2.tv_usec - t1.tv_usec)/1000000;

  error:
  if(socket) socket_free(&socket);

  if(!rv) {
    p->is_available= FALSE;
    Event_post(s, EVENT_CONNECTION, TRUE, p->action, report);
  } else {
    p->is_available= TRUE;
    Event_post(s, EVENT_CONNECTION, FALSE, p->action,
      "'%s' connection passed", s->name);
  }
      
}


/**
 * Test process state (e.g. Zombie)
 */
static void check_process_state(Service_T s) {

  ASSERT(s);

  if(s->inf->status_flag & PROCESS_ZOMBIE) {
    /* We do not check the process anymore if it's a zombie
       since such a process is (usually) unmanageable */
    LOCK(s->mutex)
      s->monitor= MONITOR_NOT;
    END_LOCK;
    Event_post(s, EVENT_DATA, TRUE, s->action_DATA,
      "'%s' process with pid %d is a zombie", s->name, s->inf->pid);
  } else {
    DEBUG("'%s' zombie check passed [status_flag=%04x]\n",
      s->name,  s->inf->status_flag);
    Event_post(s, EVENT_DATA, FALSE, s->action_DATA,
      "'%s' check process state passed", s->name);
  }

}


/**
 * Check process resources
 */
static void check_process_resources(Service_T s, Resource_T pr) {

  int okay= TRUE;
  char report[STRLEN]={0};

  ASSERT(s && pr);

  switch(pr->resource_id) {
  case RESOURCE_ID_CPU_PERCENT:
    if(compare_value(pr->operator, s->inf->cpu_percent, pr->limit)) {
      snprintf(report, STRLEN,
        "'%s' cpu usage of %.1f%% matches resource limit [cpu usage%s%.1f%%]",
        s->name, s->inf->cpu_percent/10.0, operatorshortnames[pr->operator],
        pr->limit/10.0);
      okay= FALSE;
    } else {
      DEBUG("'%s' cpu usage check passed [current cpu usage=%.1f%%]\n",
        s->name, s->inf->cpu_percent/10.0);
    }
    break;

  case RESOURCE_ID_MEM_PERCENT:
    if(compare_value(pr->operator, s->inf->mem_percent, pr->limit)) {
      snprintf(report, STRLEN,
        "'%s' mem usage of %.1f%% matches resource limit [mem usage%s%.1f%%]",
        s->name, s->inf->mem_percent/10.0, operatorshortnames[pr->operator],
        pr->limit/10.0);
      okay= FALSE;
    } else {
      DEBUG("'%s' mem usage check passed [current mem usage=%.1f%%]\n",
        s->name, s->inf->mem_percent/10.0);
    }
    break;

  case RESOURCE_ID_MEM_KBYTE:
    if(compare_value(pr->operator, s->inf->mem_kbyte, pr->limit)) {
      snprintf(report, STRLEN,
        "'%s' mem amount of %ldkB matches resource limit [mem amount%s%ldkB]",
        s->name, s->inf->mem_kbyte, operatorshortnames[pr->operator], pr->limit);
      okay= FALSE;
    } else {
      DEBUG("'%s' mem amount check passed [current mem amount=%ldkB]\n",
        s->name, s->inf->mem_kbyte);
    }
    break;

  case RESOURCE_ID_LOAD1:
    if(compare_value(pr->operator,
                     (int)(systeminfo.loadavg[0]*10.0), pr->limit)) {
      snprintf(report, STRLEN,
        "'%s' loadavg(1min) of %.1f matches resource limit "
        "[loadavg(1min)%s%.1f]",
        s->name, systeminfo.loadavg[0], operatorshortnames[pr->operator],
        pr->limit/10.0);
      okay= FALSE;
    } else {
      DEBUG("'%s' loadavg(1min) check passed [current loadavg(1min)=%.1f]\n",
        s->name, systeminfo.loadavg[0]);
    }
    break;

  case RESOURCE_ID_LOAD5:
    if(compare_value(pr->operator,
                     (int)(systeminfo.loadavg[1]*10.0), pr->limit)) {
      snprintf(report, STRLEN,
        "'%s' loadavg(5min) of %.1f matches resource limit "
        "[loadavg(5min)%s%.1f]",
        s->name, systeminfo.loadavg[1], operatorshortnames[pr->operator],
        pr->limit/10.0);
      okay= FALSE;
    } else {
      DEBUG("'%s' loadavg(5min) check passed [current loadavg(5min)=%.1f]\n",
        s->name, systeminfo.loadavg[1]);
    }
    break;

  case RESOURCE_ID_LOAD15:
    if(compare_value(pr->operator,
                     (int)(systeminfo.loadavg[2]*10.0), pr->limit)) {
      snprintf(report, STRLEN,
        "'%s' loadavg(15min) of %.1f matches resource limit "
        "[loadavg(15min)%s%.1f]",
        s->name, systeminfo.loadavg[2], operatorshortnames[pr->operator],
        pr->limit/10.0);
      okay= FALSE;
    } else {
      DEBUG("'%s' loadavg(15min) check passed [current loadavg(15min)=%.1f]\n",
        s->name, systeminfo.loadavg[2]);
    }
    break;

  case RESOURCE_ID_CHILDREN:
    if(compare_value(pr->operator, s->inf->children, pr->limit)) {
      snprintf(report, STRLEN,
        "'%s' children of %i matches resource limit [children%s%ld]",
        s->name, s->inf->children, operatorshortnames[pr->operator], pr->limit);
      okay= FALSE;
    } else {
      DEBUG("'%s' children check passed [current children=%i]\n",
        s->name, s->inf->children);
    }
    break;

  case RESOURCE_ID_TOTAL_MEM_KBYTE:
    if(compare_value(pr->operator, s->inf->total_mem_kbyte, pr->limit)) {
      snprintf(report, STRLEN,
        "'%s' total mem amount of %ldkB matches resource limit"
        " [total mem amount%s%ldkB]",
        s->name, s->inf->total_mem_kbyte, operatorshortnames[pr->operator],
        pr->limit);
      okay= FALSE;
    } else {
      DEBUG("'%s' total mem amount check passed "
        "[current total mem amount=%ldkB]\n", s->name, s->inf->total_mem_kbyte);
    }
    break;

  default:
    log("'%s' error -- unknown resource ID: [%d]\n", s->name, pr->resource_id);
    return;
  }

  if(okay && pr->cycle > 0) {
    pr->cycle--;
  } else if(! okay) {
    pr->cycle++;
  }

  if(pr->cycle >= pr->max_cycle) {
    pr->cycle=0;
    Event_post(s, EVENT_RESOURCE, TRUE, pr->action, report);
  } else {
    Event_post(s, EVENT_RESOURCE, FALSE, pr->action,
      "'%s' resource passed", s->name);
  }
  
}


/**
 * Test for associated path checksum change
 */
static void check_checksum(Service_T s) {

  Checksum_T  cs;
  int         changed;

  ASSERT(s && s->path && s->checksum && s->checksum->hash);

  cs= s->checksum;

  FREE(s->inf->cs_sum);
  s->inf->cs_sum= get_checksum(s->path, cs->type);

  if(s->inf->cs_sum) {

    Event_post(s, EVENT_DATA, FALSE, s->action_DATA,
      "'%s' checksum computed for %s", s->name, s->path);

    switch(cs->type) {
      case HASH_MD5:
        changed= strncmp(cs->hash, s->inf->cs_sum, 32);
        break;
      case HASH_SHA1:
        changed= strncmp(cs->hash, s->inf->cs_sum, 40);
        break;
      default:
        log("'%s' unknown hash type\n", s->name);
        FREE(s->inf->cs_sum);
        return;
    }

    if(changed) {

      /* if we are testing for changes only, the value is variable */
      if(cs->test_changes) {
        Event_post(s, EVENT_CHANGED, TRUE, cs->action,
          "'%s' checksum was changed for %s", s->name, s->path);
        /* reset expected value for next cycle */
        FREE(cs->hash);
        cs->hash= xstrdup(s->inf->cs_sum);
      } else {
        /* we are testing constant value for failed or passed state */
        Event_post(s, EVENT_CHECKSUM, TRUE, cs->action,
          "'%s' checksum test failed for %s", s->name, s->path);
      }

    } else if(cs->test_changes) {

      DEBUG("'%s' checksum has not changed\n", s->name);
      Event_post(s, EVENT_CHANGED, FALSE, cs->action,
        "'%s' checksum has not changed", s->name);

    } else {

      DEBUG("'%s' has valid checksums\n", s->name);
      Event_post(s, EVENT_CHECKSUM, FALSE, cs->action,
        "'%s' checksum passed", s->name);

    }

    return;

  }

  Event_post(s, EVENT_DATA, TRUE, s->action_DATA,
    "'%s' cannot compute checksum for %s", s->name, s->path);

}


/**
 * Test for associated path permission change
 */
static void check_perm(Service_T s) {

  ASSERT(s && s->perm);

  if((s->inf->st_mode & 07777) != s->perm->perm) {
    Event_post(s, EVENT_PERMISSION, TRUE, s->perm->action,
      "'%s' permission test failed for %s -- current permission is %o",
      s->name, s->path, s->inf->st_mode&07777);
  } else {
    DEBUG("'%s' file permission check passed [current permission=%o]\n",
      s->name, s->inf->st_mode&07777);
    Event_post(s, EVENT_PERMISSION, FALSE, s->perm->action,
      "'%s' permission passed", s->name);
  }
}


/**
 * Test for associated path uid change
 */
static void check_uid(Service_T s) {

  ASSERT(s && s->uid);

  if(s->inf->st_uid != s->uid->uid) {
    Event_post(s, EVENT_UID, TRUE, s->uid->action,
      "'%s' uid test failed for %s -- current uid is %d",
      s->name, s->path, (int)s->inf->st_uid);
  } else {
    DEBUG("'%s' device uid check passed [current uid=%d]\n", s->name,
          (int)s->inf->st_uid);
    Event_post(s, EVENT_UID, FALSE, s->uid->action, "'%s' uid passed", s->name);
  }

}


/**
 * Test for associated path gid change
 */
static void check_gid(Service_T s) {

  ASSERT(s && s->gid);

  if(s->inf->st_gid != s->gid->gid ) {
    Event_post(s, EVENT_GID, TRUE, s->gid->action,
      "'%s' gid test failed for %s -- current gid is %d",
      s->name, s->path, (int)s->inf->st_gid);
  } else {
    DEBUG("'%s' device gid check passed [current gid=%d]\n", s->name,
          (int)s->inf->st_gid);
    Event_post(s, EVENT_GID, FALSE, s->gid->action, "'%s' gid passed", s->name);
  }

}


/**
 * Validate timestamps of a service s
 */
static void check_timestamp(Service_T s) {

  Timestamp_T t;
  time_t      now;

  ASSERT(s && s->timestamplist);


  if((int)time(&now) == -1) {
    Event_post(s, EVENT_DATA, TRUE, s->action_DATA,
      "'%s' can't obtain actual system time", s->name);
    return;
  } else {
    Event_post(s, EVENT_DATA, FALSE, s->action_DATA,
      "'%s' actual system time obtained", s->name);
  }

  for(t= s->timestamplist; t; t= t->next) {
    if(t->test_changes) {
    /* if we are testing for changes only, the value is variable */
      if(t->timestamp != s->inf->timestamp) {
        Event_post(s, EVENT_CHANGED, TRUE, t->action,
          "'%s' timestamp was changed for %s", s->name, s->path);
        /* reset expected value for next cycle */
        t->timestamp= s->inf->timestamp;
      } else {
        DEBUG("'%s' timestamp was not changed for %s\n", s->name, s->path);
        Event_post(s, EVENT_CHANGED, FALSE, t->action,
          "'%s' timestamp was not changed for %s", s->name, s->path);
      }
      break;
    } else {
    /* we are testing constant value for failed or passed state */
      if(compare_value(t->operator, (int)(now - s->inf->timestamp), t->time)) {
        Event_post(s, EVENT_TIMESTAMP, TRUE, t->action,
          "'%s' timestamp test failed for %s", s->name, s->path);
      } else {
        DEBUG("'%s' timestamp test passed for %s\n", s->name, s->path); 
        Event_post(s, EVENT_TIMESTAMP, FALSE, t->action,
          "'%s' timestamp passed", s->name);
      }
    }
  }
}


/**
 * Test size
 */
static void check_size(Service_T s) {

  Size_T sl;

  ASSERT(s && s->sizelist);

  for(sl= s->sizelist; sl; sl= sl->next) {

    /* if we are testing for changes only, the value is variable */
    if(sl->test_changes) {
      if(sl->size != s->inf->st_size) {
        Event_post(s, EVENT_CHANGED, TRUE, sl->action,
          "'%s' size was changed for %s", s->name, s->path);
        /* reset expected value for next cycle */
        sl->size= s->inf->st_size;
      } else {
        DEBUG("'%s' size has not changed [current size=%lu B]\n", s->name,
              s->inf->st_size);
        Event_post(s, EVENT_CHANGED, FALSE, sl->action,
          "'%s' size was not changed", s->name, s->path);
      }
      break;
    }

    /* we are testing constant value for failed or passed state */
    if(compare_value(sl->operator, s->inf->st_size, sl->size)) {
      Event_post(s, EVENT_SIZE, TRUE, sl->action,
        "'%s' size test failed for %s -- current size is %lu B",
        s->name, s->path, s->inf->st_size);
    } else {
      DEBUG("'%s' file size check passed [current size=%lu B]\n", s->name,
            s->inf->st_size);
      Event_post(s, EVENT_SIZE, FALSE, sl->action, "'%s' size passed", s->name);
    }
  }
}


/**
 * Device test
 */
static void check_device_resources(Service_T s, Device_T td) {

  ASSERT(s && td);

  if( (td->limit_percent < 0) && (td->limit_absolute < 0) ) {
    log("'%s' error: device limit not set\n", s->name);
    return;
  }

  switch(td->resource) {

  case RESOURCE_ID_INODE:
      if(s->inf->f_files <= 0) {
	DEBUG("'%s' filesystem doesn't support inodes\n", s->name);
	return;
      }

      if(td->limit_percent >= 0) {
	if(compare_value(
	       td->operator,
	       100 * (s->inf->f_files - s->inf->f_filesfree) /
	       s->inf->f_files, td->limit_percent)) {
          Event_post(s, EVENT_RESOURCE, TRUE, td->action,
            "'%s' inode usage %ld%% matches resource limit [inode usage%s%d%%]",
            s->name,
	    100 * (s->inf->f_files - s->inf->f_filesfree) /
	    s->inf->f_files, operatorshortnames[td->operator],
	    td->limit_percent);
	  return;
	}
      } else {
	if(compare_value(td->operator, s->inf->f_files -
			 s->inf->f_filesfree, td->limit_absolute)) {
          Event_post(s, EVENT_RESOURCE, TRUE, td->action,
            "'%s' inode usage %ld matches resource limit [inode usage%s%ld]",
            s->name,
            s->inf->f_files - s->inf->f_filesfree,
            operatorshortnames[td->operator], td->limit_absolute);
	  return;
	}
      }
      DEBUG("'%s' inode usage check passed [current inode usage=%.1f%%]\n",
	    s->name,
	    (float) 100 * (s->inf->f_files - s->inf->f_filesfree) /
	    s->inf->f_files);
      Event_post(s, EVENT_RESOURCE, FALSE, td->action,
        "'%s' device resources passed", s->name);
      return;

  case RESOURCE_ID_SPACE:
      if(td->limit_percent >= 0) {
        if(compare_value(
	       td->operator,
	       100 * (s->inf->f_blocks - s->inf->f_blocksfreetotal) /
	       s->inf->f_blocks, td->limit_percent)) {
          Event_post(s, EVENT_RESOURCE, TRUE, td->action,
	    "'%s' space usage %ld%% matches resource limit [space usage%s%d%%]",
            s->name,
            100*(s->inf->f_blocks - s->inf->f_blocksfreetotal) /
            s->inf->f_blocks, operatorshortnames[td->operator],
            td->limit_percent);
          return;
        }
      } else {
        if(compare_value(td->operator, s->inf->f_blocks -
			 s->inf->f_blocksfreetotal, td->limit_absolute)) {
          Event_post(s, EVENT_RESOURCE, TRUE, td->action,
            "'%s' space usage %ld blocks matches resource limit "
            "[space usage%s%ld blocks]",
            s->name,
            s->inf->f_blocks - s->inf->f_blocksfreetotal,
            operatorshortnames[td->operator], td->limit_absolute);
	  return;
        }
      }
      DEBUG("'%s' space usage check passed [current space usage=%.1f%%]\n",
	    s->name,
	    (float) 100*(s->inf->f_blocks - s->inf->f_blocksfreetotal) /
	    s->inf->f_blocks);
      Event_post(s, EVENT_RESOURCE, FALSE, td->action,
        "'%s' device resources passed", s->name);
      return;
      
  default:
      log("'%s' error -- unknown resource type: [%d]\n", s->name, td->resource);
      return;
  }
  
}


/**
 * Comparison of values. Returns TRUE if comparison matches, otherwise
 * FALSE.
 */
static int compare_value(int operator, int left, int right) {

  switch(operator) {
  case OPERATOR_GREATER:
      if(left > right)
	  return TRUE;
      break;
  case OPERATOR_LESS:
      if(left < right)
	  return TRUE;
      break;
  case OPERATOR_EQUAL:
      if(left == right)
	  return TRUE;
      break;
  case OPERATOR_NOTEQUAL:
      if(left != right)
	  return TRUE;
      break;
  default:
      log("Unknown comparison operator\n");
      return FALSE;
  }

  return FALSE;
    
}


/**
 * Returns TRUE if the service timed out, otherwise FALSE.
 */
static int check_timeout(Service_T s) {

  ASSERT(s);

  if(!s->def_timeout)
    return FALSE;

  /*
   * Start counting cycles
   */
  if(s->nstart > 0)
    s->ncycle++;

  /*
   * Check timeout
   */
  if(s->nstart >= s->to_start && s->ncycle <= s->to_cycle) {
    Event_post(s, EVENT_TIMEOUT, TRUE, s->action_TIMEOUT,
              "'%s' service timed out and will not be checked anymore",
              s->name);
    return TRUE;
  }

  /*
   * Stop counting and reset if the
   * cycle interval is passed
   */
  if(s->ncycle > s->to_cycle) {
    s->ncycle= 0;
    s->nstart= 0;
  }

  return FALSE;

}


/**
 * Returns TRUE if validation should be skiped for
 * this service in this cycle, otherwise FALSE
 */
static int check_skip(Service_T s) {

  ASSERT(s);
  
  if(!s->def_every)
    return FALSE;
  
  if(++s->nevery < s->every)
    return TRUE;

  s->nevery= 0;

  return FALSE;

}


