/*
 * Copyright (C), 2000-2004 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */


#include <config.h>

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef TIME_WITH_SYS_TIME
#include <time.h>

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#else
#include <time.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_ASM_PARAM_H
#include <asm/param.h>
#endif

#ifdef HAVE_ASM_PAGE_H
#include <asm/page.h>
#endif

#ifdef HAVE_GLOB_H
#include <glob.h>
#endif

#include "monitor.h"
#include "process.h"
#include "process_sysdep.h"

/**
 *  System dependent resource gathering code for Linux.
 *
 *  @author Jan-Henrik Haukeland, <hauk@tildeslash.com>
 *  @author Christian Hopp <chopp@iei.tu-clausthal.de>
 *  @author Martin Pala <martinp@tildeslash.com>
 *  @author Arkadiusz Miskiewicz <arekm@pld-linux.org>
 *
 *  @version \$Id: sysdep_LINUX.c,v 1.21 2004/09/11 19:07:35 martinp Exp $
 *
 *  @file
 */


/* ----------------------------------------------------------------- Private */


#define PAGE_TO_KBYTE_SHIFT PAGE_SHIFT-10

#define MEMTOTAL "MemTotal:"
#define MEMFREE  "MemFree:"
#define MEMBUF   "Buffers:"
#define MEMCACHE "Cached:"

static long   old_cpu_user = 0;
static long   old_cpu_syst = 0;
static long   old_cpu_wait = 0;
static double old_time     = 0.0;


/* ------------------------------------------------------------------ Public */


int init_process_info_sysdep(void) {
  char *ptr;
  char  buf[1024];

  if(!read_proc_file(buf, 1024, "meminfo", -1)) 
  {
    return FALSE;
  }
  if(!(ptr= strstr(buf, MEMTOTAL)))
  {
    DEBUG("system statistic error -- cannot get real memory amount\n");
    return FALSE;
  }
  if(sscanf(ptr+strlen(MEMTOTAL), "%ld", &systeminfo.mem_kbyte_max) != 1)
  {
    DEBUG("system statistic error -- cannot get real memory amount\n");
    return FALSE;
  }

  if((num_cpus = sysconf(_SC_NPROCESSORS_CONF)) == -1)
  {
    DEBUG("system statistic error -- cannot get cpu count: %s\n", STRERROR);
    return FALSE;
  }

  return TRUE;
}


/**
 * Read all processes of the proc files system to initilize
 * the process tree (sysdep version... but should work for
 * all procfs based unices) 
 * @param reference  reference of ProcessTree
 * @return treesize>0 if succeeded otherwise =0.
 */
int initprocesstree_sysdep(ProcessTree_T ** reference) {
  int            i;
  int            treesize;
  int            stat_ppid;
  char          *tmp;
  char           buf[4096];
  char           stat_item_state;
  long           stat_item_cutime;
  long           stat_item_cstime;
  long           stat_item_rss;
  glob_t         globbuf;
  unsigned long  stat_item_utime;
  unsigned long  stat_item_stime;
  ProcessTree_T *pt;
#ifndef USE_QUICK_MEM_CALC
  int            statm_item_share;
  int            statm_item_trs;
  int            statm_item_drs;
#endif

  ASSERT(reference);

  /* Find all processes in the /proc directory */
  if(glob("/proc/[0-9]*", GLOB_ONLYDIR, NULL, &globbuf))
  {
    log("system statistic error -- glob failed\n");
    return FALSE;
  } 

  treesize = globbuf.gl_pathc;

  pt = xcalloc(sizeof(ProcessTree_T), treesize);

  /* Insert data from /proc directory */
  for(i = 0; i < treesize; i++)
  {
    pt[i].pid = atoi(globbuf.gl_pathv[i] + strlen("/proc/"));
    
    if(!read_proc_file(buf, 4096, "stat", pt[i].pid))
    {
      continue;
    }

    pt[i].time = get_float_time();

    /* Move along the buffer to get past the process name */
    if(!(tmp = strrchr(buf, ')') + 2))
    {
      log("system statistic error -- file /proc/%d parse error\n", pt[i].pid);
      continue;
    }

    /* This implementation is done by using fs/procfs/array.c as a basis
     * it is also worth looking into the source of the procps utils */
    sscanf(tmp,"%c %d %*d %*d %*d %*d %*u %*u"
           "%*u %*u %*u %lu %lu %ld %ld %*d %*d %*d "
           "%*d %*u %*u %ld %*u %*u %*u %*u %*u "
           "%*u %*u %*u %*u %*u %*u %*u %*u %*d %*d\n",
           &stat_item_state, &stat_ppid,
           &stat_item_utime, &stat_item_stime,
           &stat_item_cutime, &stat_item_cstime, &stat_item_rss);
    
    /* abs to please the compiler... we dont want to shift negatively.
     * why doesn't C understand this??? */
    pt[i].ppid = stat_ppid;
  
    /* jiffies -> seconds = 1 / HZ
     * HZ is defined in "asm/param.h"  and it is usually 1/100s but on
     * alpha system it is 1/1024s */
    pt[i].cputime =
        ((float)(stat_item_utime + stat_item_stime) * 10.0) / HZ;

    /* State is Zombie -> then we are a Zombie ... clear or? (-: */
    if(stat_item_state == 'Z')
    {
      pt[i].status_flag |= PROCESS_ZOMBIE;
    }

#ifdef USE_QUICK_MEM_CALC
    if(PAGE_TO_KBYTE_SHIFT < 0)
    {
      pt[i].mem_kbyte = (stat_item_rss >> abs(PAGE_TO_KBYTE_SHIFT));
    }
    else
    {
      pt[i].mem_kbyte = (stat_item_rss << abs(PAGE_TO_KBYTE_SHIFT));
    }
#else
    if(!read_proc_file(buf, 4096, "statm", pt[i].pid))
    {
      pt[i].mem_kbyte = 0;
    }
    else
    {
      /* This implementation is done by using fs/procfs/array.c as a basis
       * it is also worth looking into the source of the procps utils.
       * Beware then proc man page has a wrong sorting of the entries!  */
      sscanf(buf,"%*d %*d %d %d %*d %d %*d\n",
             &statm_item_share,&statm_item_trs, &statm_item_drs);
      
      if(PAGE_TO_KBYTE_SHIFT < 0)
      {
        pt[i].mem_kbyte =
            (statm_item_trs   >> abs(PAGE_TO_KBYTE_SHIFT)) +
            (statm_item_drs   >> abs(PAGE_TO_KBYTE_SHIFT)) -
            (statm_item_share >> abs(PAGE_TO_KBYTE_SHIFT));
        
      }
      else
      {
        pt[i].mem_kbyte =
            (statm_item_trs   << abs(PAGE_TO_KBYTE_SHIFT)) +
            (statm_item_drs   << abs(PAGE_TO_KBYTE_SHIFT)) -
            (statm_item_share << abs(PAGE_TO_KBYTE_SHIFT));
      }
    }
#endif
  }
  
  *reference = pt;
  globfree(&globbuf);

  return treesize;
}


/**
 * This routine returns 'nelem' double precision floats containing
 * the load averages in 'loadv'; at most 3 values will be returned.
 * @param loadv destination of the load averages
 * @param nelem number of averages
 * @return: 0 if successful, -1 if failed (and all load averages are 0).
 */
int getloadavg_sysdep (double *loadv, int nelem) {
  return getloadavg(loadv, nelem);
}


/**
 * This routine returns kbyte of real memory in use.
 * @return: TRUE if successful, FALSE if failed
 */
int used_system_memory_sysdep(SystemInfo_T *si) {
  char          *ptr;
  char           buf[1024];
  unsigned long  mem_free;
  unsigned long  buffers;
  unsigned long  cached;
  
  if(!read_proc_file(buf, 1024, "meminfo", -1))
  {
    goto error;
  }

  if(!(ptr= strstr(buf, MEMFREE)))
  {
    log("system statistic error -- cannot get real memory free amount\n");
    goto error;
  }
  if(sscanf(ptr + strlen(MEMFREE), "%ld", &mem_free) != 1)
  {
    log("system statistic error -- cannot get real memory free amount\n");
    goto error;
  }

  if(!(ptr= strstr(buf, MEMBUF)))
  {
    log("system statistic error -- cannot get real memory beffers amount\n");
    goto error;
  }
  if(sscanf(ptr + strlen(MEMBUF), "%ld", &buffers) != 1)
  {
    log("system statistic error -- cannot get real memory buffers amount\n");
    goto error;
  }

  if(!(ptr= strstr(buf, MEMCACHE)))
  {
    log("system statistic error -- cannot get real memory cache amount\n");
    goto error;
  }
  if(sscanf(ptr + strlen(MEMCACHE), "%ld", &cached) != 1)
  {
    log("system statistic error -- cannot get real memory cache free amount\n");
    goto error;
  }

  si->total_mem_kbyte = systeminfo.mem_kbyte_max - mem_free - buffers - cached;

  if(si->total_mem_kbyte < 0)
  {
    log("system statistic error -- memory usage statistic error\n");
    goto error;
  }

  return TRUE;

  error:
  log("system statistic error -- memory usage gathering failed\n");
  si->total_mem_kbyte = 0;
  return FALSE;
}


/**
 * This routine returns system/user CPU time in use.
 * @return: TRUE if successful, FALSE if failed (or not available)
 */
int used_system_cpu_sysdep(SystemInfo_T *si) {
  int    rv;
  long   cpu_user;
  long   cpu_syst;
  long   cpu_wait;
  char   buf[1024];
  double curr_time = get_float_time();

  if(!read_proc_file(buf, 1024, "stat", -1))
  {
    goto error;
  }

  rv = sscanf(buf, "cpu %ld %*d %ld %*d %ld", &cpu_user, &cpu_syst, &cpu_wait);
  if(rv < 2 || rv > 3)
  {
    log("system statistic error -- cannot read cpu usage\n");
    goto error;
  }
  else
  {
    cpu_user = (double)cpu_user * 10.0 / HZ;
    cpu_syst = (double)cpu_syst * 10.0 / HZ;
    cpu_wait = (rv == 3)?(double)cpu_wait * 10.0 / HZ:0.0;
  }
  
  if(old_time == 0.0)
  {
    si->total_cpu_user_percent = 0;
    si->total_cpu_syst_percent = 0;
    si->total_cpu_wait_percent = 0;
  }
  else
  {
    si->total_cpu_user_percent =
      (int)(1000 * (double)(cpu_user - old_cpu_user) / (curr_time - old_time));
    si->total_cpu_syst_percent =
      (int)(1000 * (double)(cpu_syst - old_cpu_syst) / (curr_time - old_time));
    si->total_cpu_wait_percent =
      (int)(1000 * (double)(cpu_wait - old_cpu_wait) / (curr_time - old_time));
  }
  old_time = curr_time;

  old_cpu_user = cpu_user;
  old_cpu_syst = cpu_syst;
  old_cpu_wait = cpu_wait;
  return TRUE;

  error:
  si->total_cpu_user_percent = 0;
  si->total_cpu_syst_percent = 0;
  si->total_cpu_wait_percent = 0;
  return FALSE;
}

