/*
 * Copyright (C), 2000-2004 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif


#include "monitor.h"
#include "alert.h"
#include "event.h"
#include "process.h"


/**
 * Implementation of the event interface.
 *
 * @author Jan-Henrik Haukeland, <hauk@tildeslash.com>
 * @author Martin Pala <martinp@tildeslash.com>
 * @version \$Id: event.c,v 1.41 2004/07/31 20:15:07 martinp Exp $
 * @file
 */


/* ------------------------------------------------------------- Definitions */

EventTable_T Event_Table[]= {
  {EVENT_CHANGED,    "Changed",                "Changed not"},
  {EVENT_CHECKSUM,   "Checksum failed",        "Checksum passed"},
  {EVENT_CONNECTION, "Connection failed",      "Connection passed"},
  {EVENT_DATA,       "Data access error",      "Data access succeeded"},
  {EVENT_EXEC,       "Execution failed",       "Execution succeeded"},
  {EVENT_GID,        "GID failed",             "GID passed"},
  {EVENT_ICMP,       "ICMP failed",            "ICMP passed"},
  {EVENT_INVALID,    "Invalid type",           "Type passed"},
  {EVENT_NONEXIST,   "Does not exist",         "Exists"},
  {EVENT_PERMISSION, "Permission failed",      "Permission passed"},
  {EVENT_RESOURCE,   "Resource limit matched", "Resource limit passed"},
  {EVENT_SIZE,       "Size failed",            "Size passed"},
  {EVENT_TIMEOUT,    "Timeout",                "Timeout recovery"},
  {EVENT_TIMESTAMP,  "Timestamp failed",       "Timestamp passed"},
  {EVENT_UID,        "UID failed",             "UID passed"},
  /* Virtual events */
  {EVENT_NULL,       "No Event",               "No Event"},
};


/* -------------------------------------------------------------- Prototypes */


static void handle_event(Event_T);
static int  handle_action(Event_T, Action_T);


/* ------------------------------------------------------------------ Public */


/**
 * Post a new Event
 * @param service The Service the event belongs to
 * @param id The event identification
 * @param state TRUE for failed, FALSE for passed event state
 * @param action Description of the event action
 * @param s Optional message describing the event
 */
void Event_post(Service_T service, long id, short state, EventAction_T action,
  char *s, ...) {

  Event_T e = service->eventlist;

  ASSERT(service);
  ASSERT(action);

  if(e == NULL)
  {
    /* Only first failed event can initialize the queue for given event type,
     * thus passed events are ignored until first error. However, in the case
     * that the error flag is set for the passed event, we will allow it (i.e.
     * event queue was flushed during monit reload and the service was in
     * failed state before reload) */
    if(!state && !(service->error & id))
      return;

    /* Initialize event list and add first event. */
    NEW(e);
    e->id = id;
    e->source = service;
    e->state = state;
    e->state_changed = TRUE;
    e->action = action;
    if(s)
    {
      long l;
      va_list ap;

      va_start(ap, s);
      e->message = format(s, ap, &l);
      va_end(ap);
    }
    pthread_mutex_init(&e->mutex, NULL);
    service->eventlist = e;
  }
  else
  {
    /* Try to find the event with the same origin and type identification.
     * Each service and each test have its own custom actions object, so
     * we share actions object address to identify event source. */
    do
    {
      if(e->action == action && e->id == id)
      {
        LOCK(e->mutex)
          /* update the message */
          if(s)
          {
            long l;
            va_list ap;

            FREE(e->message);
            va_start(ap, s);
            e->message = format(s, ap, &l);
            va_end(ap);
          }

          if(e->state != state)
          {
            e->state = state;
            e->state_changed = TRUE;
          }
        END_LOCK;
	break;
      }
      e = e->next;
    }
    while(e);

    if(!e)
    {
      /* Only first failed event can initialize the queue for given event type,
       * thus passed events are ignored until first error */
      if(!state)
        return;

      /* Event was not found in the pending events list, we will add it. */
      NEW(e);
      e->id = id;
      e->source = service;
      e->state = state;
      e->state_changed = TRUE;
      e->action = action;
      if(s)
      {
        long l;
        va_list ap;

        va_start(ap, s);
        e->message = format(s, ap, &l);
        va_end(ap);
      }
      pthread_mutex_init(&e->mutex, NULL);
      e->next = service->eventlist;
      service->eventlist = e;
    }
  }

  LOCK(e->mutex)
    handle_event(e);
  END_LOCK;

}


/* -------------------------------------------------------------- Properties */


/**
 * Get the Service where the event orginated
 * @param E An event object
 * @return The Service where the event orginated
 */
Service_T Event_get_source(Event_T E) {

  ASSERT(E);

  return E->source;

}


/**
 * Get the Event type
 * @param E An event object
 * @return The Event type
 */
int Event_get_id(Event_T E) {

  ASSERT(E);
  
  return E->id;

}


/**
 * Get the optionally Event message describing why the event was
 * fired.
 * @param E An event object
 * @return The Event message. May be NULL
 */
const char *Event_get_message(Event_T E) {

  ASSERT(E);

  return E->message;

}


/**
 * Get a textual description of actual event type. For instance if the
 * event type is possitive EVENT_TIMESTAMP, the textual description is
 * "Timestamp error". Likewise if the event type is negative EVENT_CHECKSUM
 * the textual description is "Checksum recovery" and so on.
 * @param E An event object
 * @return A string describing the event type in clear text. If the
 * event type is not found NULL is returned.
 */
const char *Event_get_description(Event_T E) {

  EventTable_T *et= Event_Table;

  ASSERT(E);

  while((*et).id)
  {
    if(E->id == (*et).id)
    {
      return E->state?(*et).description_failed:(*et).description_passed;
    }
    et++;
  }
  
  return NULL;

}


/**
 * Get a textual description of actual event action. For instance if the
 * event type is possitive EVENT_NONEXIST, the textual description of
 * failed state related action is "restart". Likewise if the event type is
 * negative EVENT_CHECKSUM the textual description of recovery related action
 * is "alert" and so on.
 * @param E An event object
 * @return A string describing the event type in clear text. If the
 * event type is not found NULL is returned.
 */
const char *Event_get_action_description(Event_T E) {

  int id;
  Action_T A;

  ASSERT(E);

  A = E->state?E->action->failed:E->action->passed;

  /* In the case of passive mode we replace the description of start, stop
   * or restart action for alert action, because these actions are passive in
   * this mode */
  id= (E->source->mode == MODE_PASSIVE &&
       ((A->id == ACTION_START)||
        (A->id == ACTION_STOP) ||
        (A->id == ACTION_RESTART))
      )?ACTION_ALERT:A->id;

  return actionnames[id];

}


/* ----------------------------------------------------------------- Private */


/*
 * Handle the event
 * @param E An event
 */
static void handle_event(Event_T E) {

  ASSERT(E);
  ASSERT(E->action);
  ASSERT(E->action->failed);
  ASSERT(E->action->passed);

  /* We will handle only first passed event, recurrent passed events are
   * ignored. Failed events are handled each time. */
  if(!E->state_changed && !E->state)
  {
    return;
  }

  if(E->message)
  {
    log("%s\n", E->message);
  }

  if(E->state)
  {
    E->source->error |= E->id;
    handle_action(E, E->action->failed);
  }
  else
  {
    E->source->error &= ~E->id;
    handle_action(E, E->action->passed);
  }

  /* Possible event state change was handled so we will reset the flag. */
  E->state_changed = FALSE;

}


static int handle_action(Event_T E, Action_T A) {

  ASSERT(E);
  ASSERT(A);

  if(A->id == ACTION_IGNORE)
  {
    return TRUE;
  }

  /* Alert is common action. It is send on state change only (e.g. once). */
  if(E->state_changed || (E->id == EVENT_CHANGED))
  {
    handle_alert(E);
  }

  switch(A->id)
  {
  case ACTION_ALERT:
      /* Already handled. */
      break;

  case ACTION_EXEC:
      spawn(E->source, A->exec, EVENT_DESCRIPTION(E));
      break;

  case ACTION_RESTART:
      if(E->source->def_timeout)
        E->source->nstart++;
      if((E->source->mode != MODE_PASSIVE))
      {
        control_service(E->source->name, "restart");
      }
      return FALSE;

  case ACTION_START:
      if(E->source->def_timeout)
        E->source->nstart++;
      if((E->source->mode != MODE_PASSIVE))
      {
        control_service(E->source->name, "start");
      } 
      return FALSE;

  case ACTION_STOP:
      if((E->source->mode != MODE_PASSIVE))
      {
        control_service(E->source->name, "stop");
      } 
      return FALSE;

  case ACTION_UNMONITOR:
      control_service(E->source->name, "unmonitor");
      return FALSE;

  default:
      log("'%s' error -- unknown failure action: [%d]\n", E->source->name,
        A->id);
      break;

  }
  return TRUE;
}

