/*
 * Copyright (C), 2000-2003 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <config.h>

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "net.h"
#include "socket.h"
#include "monitor.h"
#include "monit_process.h"
#include "monit_device.h"


/* Private Prototypes */
static void local_status(Service_T);
static int remote_status(Service_T);


/**
 *  Print the status of services in the service list.
 *
 *  @author Jan-Henrik Haukeland, <hauk@tildeslash.com>
 *  @author Christian Hopp, <chopp@iei.tu-clausthal.de>
 *
 *  @version \$Id: status.c,v 1.50 2003/10/25 19:17:11 hauk Exp $
 *
 *  @file
 */

/* ------------------------------------------------------------------ Public */


/**
 * Show all services in the service list
 */
void status() {

  Service_T s;
  int       status=TRUE;
  char *uptime= get_process_uptime(Run.pidfile, " ");

  if(IS(uptime, "")) {
    fprintf(stdout, "The monit daemon is not running\n");
  } else {
    fprintf(stdout, "The monit daemon uptime: %s\n",uptime);
  }
  FREE(uptime);
  
  for(s= servicelist_conf; s; s= s->next_conf) {
    if (status) {
      status&=remote_status(s);
    } else {
      local_status(s);
    }
  }

}


/**
 * Show all services in the group
 * @param G  group name
 */
void status_group(char *G) {

  Service_T s;

  ASSERT(G);
  
  for(s= servicelist_conf; s; s= s->next_conf) {
    if(IS(s->group, G)) {
      remote_status(s);
    }
  }
  
}


/* ----------------------------------------------------------------- Private */


/**
 * Print the service status for the given service
 * @param s  A Service_T object
 */
static void local_status(Service_T s) {
  
  pid_t  pid= -1;
  struct stat stat_buf;

  ASSERT(s);

  switch(s->type) {

  case TYPE_PROCESS:
    if((pid= is_process_running(s))) {

      char *uptime= get_process_uptime(s->path, " ");

      fprintf(stdout,
              "Process '%s' is running with pid [%d]\n\t"
	      "Uptime: %s\n",
	      s->name, (int)pid, uptime);
    
      FREE(uptime);
 
    } else {
    
      fprintf(stdout, "Process '%s' is not running\n",  s->name);
    
    }
    break;

  case TYPE_DEVICE:
    if(!DeviceInfo_Usage(s->devinfo, s->path)||stat(s->path, &stat_buf) != 0 ) {
      fprintf(stdout,
	"Device '%s' is not accessible\n\t"
	"Space: - Inodes: -\n\t"
	"UID: - GID: - Permission: -\n",
	s->name);
    } else {

      if(s->devinfo->f_files > 0) {

        fprintf(stdout,
		"Device '%s' is accessible\n\t"
		"Space: %.1f%% [%.1f MB] Inodes: %.1f%% [%ld objects]\n\t"
		"UID: %d GID: %d Permission: %o\n",
		s->name,
		(float) 100 * (s->devinfo->f_blocks -
			       s->devinfo->f_blocksfreetotal) /
		s->devinfo->f_blocks,
		(float) (s->devinfo->f_blocks - s->devinfo->f_blocksfreetotal) /
		1048576 * s->devinfo->f_bsize,
		(float) 100 * (s->devinfo->f_files - s->devinfo->f_filesfree) /
		s->devinfo->f_files,
		s->devinfo->f_files - s->devinfo->f_filesfree,
		(int)stat_buf.st_uid, (int)stat_buf.st_gid,
	        (int)(stat_buf.st_mode & 07777));
	
      } else {

        fprintf(stdout, "Device '%s' is accessible\n\t"
		"Space: %.1f%% [%.1f MB]\n\t"
		"UID: %d GID: %d Permission: %o\n",
		s->name,
		(float) 100 * (s->devinfo->f_blocks -
			       s->devinfo->f_blocksfreetotal)
		/ s->devinfo->f_blocks,
		(float) (s->devinfo->f_blocks - s->devinfo->f_blocksfreetotal) /
		1048576 * s->devinfo->f_bsize,
		(int)stat_buf.st_uid, (int)stat_buf.st_gid,
	        (int)(stat_buf.st_mode & 07777));
      }

    }
    break;

  case TYPE_FILE:
      if( (stat(s->path, &stat_buf) != 0) || !S_ISREG(stat_buf.st_mode) ) {
	fprintf(stdout, "File '%s' doesn't exist\n",
		s->name);
      } else {
	fprintf(stdout,
		"File '%s' exist\n\t"
		"Size: %lu B UID: %d GID: %d Permission: %o\n",
		s->name,
		(unsigned long)stat_buf.st_size/1048576, (int)stat_buf.st_uid,
		(int)stat_buf.st_gid, (int)(stat_buf.st_mode & 07777));
      }
      break;
      
  case TYPE_DIRECTORY:
      if( (stat(s->path, &stat_buf) != 0) || !S_ISDIR(stat_buf.st_mode) ) {
	fprintf(stdout, "Directory '%s' doesn't exist\n", s->name);
      } else {
	fprintf(stdout,
		"Directory '%s' exist\n\t"
		"UID: %d GID: %d Permission: %o\n",
		s->name,
		(int)stat_buf.st_uid, (int)stat_buf.st_gid,
		(int)(stat_buf.st_mode & 07777));
      }
      break;
      
  case TYPE_REMOTE:
      fprintf(stdout,
	      "Host '%s'\n\tStatus is not available\n", s->name);
      break;
      
  default:
      break;
      
  }

}


/**
 * Get the service status for the given service from a monit server
 * @param p  A Service_T object
 * @return TRUE if the monit server could be connected 
 */
static int remote_status(Service_T s) {

  ASSERT(s);
  
  if(exist_daemon()) {
    
    /* If a monit daemon exist we request status information from the server */
    
    Socket_T sock= socket_new(Run.bind_addr?Run.bind_addr:"localhost",
			      Run.httpdport, SOCKET_TCP, Run.httpdssl);
    if(!sock) {
      
      log("%s: error connecting to the monit daemon\n", prog);
      
      return FALSE;
      
    } else {

      int n;
      char buf[STRLEN];
      char *auth= get_basic_authentication_header();

      if (auth==NULL) {

        socket_free(&sock);
        local_status(s);
        return FALSE;
        
      }

      socket_print(sock, "GET /%s?action=status HTTP/1.0\r\n%s\r\n",
		   s->name, auth);
      FREE(auth);

      while((0 < (n= socket_read(sock, buf, STRLEN)))) {

        if (strncmp("HTTP/1.0 401 Unauthorized", buf, 25)==0) {
          log("Authorization failed!\n");
          socket_free(&sock);
          return FALSE;
        }
        
	buf[n]= 0;
	fprintf(stdout, "%s", buf);
      }
      
      socket_free(&sock);
    
      return TRUE;

    }
    
  } else {
    
    /* No monit daemon exist, just print local status information */
    
    local_status(s);
    
    return FALSE;
  }

}
