/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: mdatatypes.c,v 1.67 2003/02/24 16:54:54 ostborn Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "config.h"
#include "mdatatypes.h"
#include "misc.h"

/* the datatypes */
#include "datatypes/count/datatype.h"
#include "datatypes/visited/datatype.h"
#include "datatypes/webhist/datatype.h"
#include "datatypes/mailhist/datatype.h"
#include "datatypes/state/datatype.h"
#include "datatypes/match/datatype.h"
#include "datatypes/record/datatype.h"
#include "datatypes/sublist/datatype.h"
#include "datatypes/split/datatype.h"
#include "datatypes/brokenlink/datatype.h"
#include "datatypes/traffic/datatype.h"
#include "datatypes/netmask/datatype.h"
#include "datatypes/visit/datatype.h"
#include "datatypes/query/datatype.h"
#include "datatypes/location/datatype.h"
#include "datatypes/string/datatype.h"

#define M_DEBUG_SAX_ENTRY 0
#define M_DEBUG_SAX_DISPATCH 0

typedef struct {
	int type;
	const char *string;
} key_map;

const key_map key_mapping[] = {
	/* 0.8.0+ */
	{ M_DATA_TYPE_COUNT,      "type_count" },
	{ M_DATA_TYPE_VISITED,    "type_visited" },
	{ M_DATA_TYPE_WEBHIST,    "type_webhist" },
	{ M_DATA_TYPE_MAILHIST,   "type_mailhist" },
	{ M_DATA_TYPE_BROKENLINK, "type_brokenlink" },
	{ M_DATA_TYPE_VISIT,      "type_visit" },
	{ M_DATA_TYPE_SUBLIST,    "type_sublist" },
	{ M_DATA_TYPE_LOCATION,   "type_location" },

	/* 0.7.x */
	{ M_DATA_TYPE_COUNT,      "count" },
	{ M_DATA_TYPE_VISITED,    "visited" },
	{ M_DATA_TYPE_WEBHIST,    "webhist" },
	{ M_DATA_TYPE_MAILHIST,   "mailhist" },
	{ M_DATA_TYPE_BROKENLINK, "brokenlink" },
	{ M_DATA_TYPE_VISIT,      "visit" },
	{ M_DATA_TYPE_SUBLIST,    "sublist" },

	{ -1, NULL }
};

/**
 * get a string represenation for the current datatype
 *
 * mainly used by the xml-write code (key='...')
 *
 * creates warnings if the datatype is unknown, not handled or unset
 *
 * @param data the datatype
 * @return the corresponding string or 'unknown' if the datatype is currently not handled
 */
const char *mdata_get_key_from_datatype(mdata *data) {
	int i;
	if (!data) return NULL;

	i = 0;
	for(i = 0; key_mapping[i].string != NULL; i++) {
		if (key_mapping[i].type == data->type) {
			return key_mapping[i].string;
		}
	}
	
	/* if everything went well we shouldn't reach this point */
	fprintf(stderr, "%s.%d: can't %s datatype '%d' (%s)\n",
		__FILE__, __LINE__,
		"get key for",
		data->type, data->key);

	return "unknown";
}

/**
 * get the data for a given string representation
 *
 * mainly used by the xml-read code (key='...')
 *
 * creates warnings if the datatype is unknown, not handled or unset
 *
 * @param data the datatype
 * @return the corresponding string or 'unknown' if the datatype is currently not handled
 */
int mdata_get_datatype_from_key(const char *key) {
	int i;
	if (!key) return M_DATA_TYPE_UNSET;

	for (i = 0; key_mapping[i].string != NULL; i++) {
		if (strcmp(key_mapping[i].string, key) == 0) {
			return key_mapping[i].type;
		}
	}

	/* if everything went well we shouldn't reach this point */
	fprintf(stderr, "%s.%d: can't %s datatype '%s'\n", __FILE__, __LINE__, "get datatype for", key);

	return M_DATA_TYPE_UNSET;
}

/**
 * free all structures of the datatype
 *
 * creates warnings if the datatype is unknown, not handled or unset
 *
 * @param data the to be freed datatype
 */
void mdata_free(mdata *data) {
	if (!data) return;

	switch(data->type) {
	case M_DATA_TYPE_COUNT:    mdata_Count_free(data); break;
	case M_DATA_TYPE_VISITED:  mdata_Visited_free(data); break;
	case M_DATA_TYPE_WEBHIST:  mdata_WebHist_free(data); break;
	case M_DATA_TYPE_MAILHIST: mdata_Mailhist_free(data); break;
	case M_DATA_TYPE_RECORD:   mdata_Record_free(data); break;
	case M_DATA_TYPE_STATE:    mdata_State_free(data); break;
	case M_DATA_TYPE_MATCH:    mdata_Match_free(data); break;
	case M_DATA_TYPE_SUBLIST:  mdata_SubList_free(data); break;
	case M_DATA_TYPE_SPLIT:    mdata_Split_free(data); break;
	case M_DATA_TYPE_BROKENLINK: mdata_BrokenLink_free(data); break;
	case M_DATA_TYPE_TRAFFIC:  mdata_Traffic_free(data); break;
	case M_DATA_TYPE_NETMASK:  mdata_Netmask_free(data); break;
	case M_DATA_TYPE_VISIT:    mdata_Visit_free(data); break;
	case M_DATA_TYPE_QUERY:    mdata_Query_free(data); break;
	case M_DATA_TYPE_LOCATION: mdata_Location_free(data); break;
	case M_DATA_TYPE_STRING:   mdata_String_free(data); break;
		
	case M_DATA_TYPE_UNSET:
		fprintf(stderr, "%s.%d: can't %s unset datatype\n", __FILE__, __LINE__, "free");
		break;
	default:
		fprintf(stderr, "%s.%d: can't %s datatype '%d'\n", __FILE__, __LINE__, "free", data->type);
		break;
	}

	if (data->key)	free(data->key);
#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: freeing ID '%d'\n", __FILE__, __LINE__, data->id);
#endif
	free(data);
}

/**
 * create a new datatype
 *
 * allocates the memory for a new datatype and sets the internal
 * type specifier to UNSET
 *
 * @return a new 'undefined' datatype
 */

mdata *mdata_init () {
	mdata *data;
#if 0
	static int id = 0;
#endif
	
	data = malloc(sizeof(mdata));
	assert(data);

	memset(data,0,sizeof(mdata));

	data->key       = NULL;
	data->type	= M_DATA_TYPE_UNSET;
#if 0
	data->id        = id++;
#endif

	return data;
}

/**
 * convertes the datatype to XML
 *
 * this function is mainly a dispatcher between the different datatypes
 * and just calles their own ..._to_xml functions.
 *
 * creates warnings if the datatype is unknown, not handled or unset
 *
 * @param node a XML node
 * @param data the datatype which
 * @return -1 on error or the return value of the specific _to_xml function
 */

int mdata_datatype_to_xml(gzFile *fd, mdata *data) {
	if (!data) return -1;

	switch (data->type) {
	case M_DATA_TYPE_COUNT:      return mdata_Count_to_xml(fd, data);
	case M_DATA_TYPE_VISITED:    return mdata_Visited_to_xml(fd, data);
	case M_DATA_TYPE_BROKENLINK: return mdata_BrokenLink_to_xml(fd, data);
	case M_DATA_TYPE_WEBHIST:    return mdata_WebHist_to_xml(fd, data);
	case M_DATA_TYPE_MAILHIST:   return mdata_Mailhist_to_xml(fd, data);
	case M_DATA_TYPE_VISIT:      return mdata_Visit_to_xml(fd, data);
	case M_DATA_TYPE_SUBLIST:    return mdata_SubList_to_xml(fd, data);
	case M_DATA_TYPE_LOCATION:   return mdata_Location_to_xml(fd, data);
		
	case M_DATA_TYPE_UNSET:
		fprintf(stderr, "%s.%d: can't %s unset datatype\n", __FILE__, __LINE__, "write");
		break;
	default:
		fprintf(stderr, "%s.%d: can't %s datatype '%d'\n", __FILE__, __LINE__, "write", data->type);
		break;
	}
	return -1;
}

mdata *mdata_datatype_init(int type) {
	switch (type) {
	case M_DATA_TYPE_COUNT:      return mdata_Count_init();
	case M_DATA_TYPE_VISITED:    return mdata_Visited_init();
	case M_DATA_TYPE_BROKENLINK: return mdata_BrokenLink_init();
	case M_DATA_TYPE_WEBHIST:    return mdata_WebHist_init();
	case M_DATA_TYPE_MAILHIST:   return mdata_Mailhist_init();
	case M_DATA_TYPE_VISIT:      return mdata_Visit_init();
	case M_DATA_TYPE_SUBLIST:    return mdata_SubList_init();
	case M_DATA_TYPE_LOCATION:   return mdata_Location_init();
		
	case M_DATA_TYPE_UNSET:
		fprintf(stderr, "%s.%d: can't %s unset datatype\n", __FILE__, __LINE__, "init");
		break;
	default:
		fprintf(stderr, "%s.%d: can't %s datatype '%d'\n", __FILE__, __LINE__, "init", type);
		break;
	}
	return NULL;
}

/**
 * creates a new XML-node in the XML-tree and inserts the XML from the datatypes
 *
 * @param node a XML node
 * @param data the datatype which
 */

int mdata_write(gzFile *fd, mdata *data) {
	/* encode key with url-encoding */
	char *encoded_key;
	
	gzprintf(fd, "<%s key=\"", mdata_get_key_from_datatype(data));
	encoded_key = url_encode(data->key);
	gzwrite(fd, encoded_key, strlen(encoded_key));
	free(encoded_key);
	gzprintf(fd, "\">");

	mdata_datatype_to_xml(fd, data);

	gzprintf(fd, "</%s>\n", mdata_get_key_from_datatype(data));

	return 0;
}

/**
 * combine two elements
 *
 * used if the keys of both elements are equal while a *_insert
 * the actual append function can reject the combine if the internal contrains
 * aren't matched
 *
 * @param dst elements
 * @param src element which
 * @return M_DATA_APPEND - append successfull
 */

mdata_append_return mdata_append(mdata *dst, mdata *src) {
	int ret = 0;
	if (dst->type != src->type) return -1;

	switch(src->type) {
		case M_DATA_TYPE_COUNT:
			ret = mdata_Count_append(dst, src);
			break;
		case M_DATA_TYPE_VISITED:
			ret = mdata_Visited_append(dst, src);
			break;
		case M_DATA_TYPE_VISIT:
			ret = mdata_Visit_append(dst, src);
			break;
		case M_DATA_TYPE_WEBHIST:
			ret = mdata_WebHist_append(dst, src);
			break;
		case M_DATA_TYPE_MAILHIST:
			ret = mdata_Mailhist_append(dst, src);
			break;
		case M_DATA_TYPE_RECORD:
			ret = mdata_Record_append(dst, src);
			break;
		case M_DATA_TYPE_STATE:
			ret = mdata_State_append(dst, src);
			break;
		case M_DATA_TYPE_MATCH:
			ret = mdata_Match_append(dst, src);
			break;
		case M_DATA_TYPE_SUBLIST:
			ret = mdata_SubList_append(dst, src);
			break;
		case M_DATA_TYPE_SPLIT:
			ret = mdata_Split_append(dst, src);
			break;
		case M_DATA_TYPE_BROKENLINK:
			ret = mdata_BrokenLink_append(dst, src);
			break;
		case M_DATA_TYPE_TRAFFIC:
			ret = mdata_Traffic_append(dst, src);
			break;
		case M_DATA_TYPE_NETMASK:
			ret = mdata_Netmask_append(dst, src);
			break;
		case M_DATA_TYPE_LOCATION:
			ret = mdata_Location_append(dst, src);
			break;
		case M_DATA_TYPE_UNSET:
			fprintf(stderr, "%s.%d: can't %s unset datatype\n", __FILE__, __LINE__, "append");
			break;
		default:
			fprintf(stderr, "%s.%d: can't %s datatype '%d'\n", __FILE__, __LINE__, "append", src->type);
			break;
	}

	return ret;
}

/**
 * copy the element
 *
 * @param src to be copied element
 * @return a copy of the provided element
 */

mdata *mdata_copy(mdata *src) {
	mdata *dst = NULL;

	switch(src->type) {
		case M_DATA_TYPE_COUNT:
			dst = mdata_Count_copy(src);
			break;
		case M_DATA_TYPE_VISITED:
			dst = mdata_Visited_copy(src);
			break;
		case M_DATA_TYPE_WEBHIST:
			dst = mdata_WebHist_copy(src);
			break;
		case M_DATA_TYPE_MAILHIST:
			dst = mdata_Mailhist_copy(src);
			break;
		case M_DATA_TYPE_RECORD:
			dst = mdata_Record_copy(src);
			break;
		case M_DATA_TYPE_STATE:
			dst = mdata_State_copy(src);
			break;
		case M_DATA_TYPE_MATCH:
			dst = mdata_Match_copy(src);
			break;
		case M_DATA_TYPE_SUBLIST:
			dst = mdata_SubList_copy(src);
			break;
		case M_DATA_TYPE_SPLIT:
			dst = mdata_Split_copy(src);
			break;
		case M_DATA_TYPE_BROKENLINK:
			dst = mdata_BrokenLink_copy(src);
			break;
		case M_DATA_TYPE_TRAFFIC:
			dst = mdata_Traffic_copy(src);
			break;
		case M_DATA_TYPE_NETMASK:
			dst = mdata_Netmask_copy(src);
			break;
		
		case M_DATA_TYPE_LOCATION:
			dst = mdata_Location_copy(src);
			break;
		case M_DATA_TYPE_UNSET:
			fprintf(stderr, "%s.%d: can't %s unset datatype\n", __FILE__, __LINE__, "copy");
			break;
		default:
			fprintf(stderr, "%s.%d: can't %s datatype '%d'\n", __FILE__, __LINE__, "copy", src->type);
			break;
	}

	return dst;
}

/**
 * convert a xml-node to a 'mdata'
 *
 * spites error messages if a unhandled datatype should be read
 *
 * return 0 - success, -1 error
 *
 */
int mdata_read(void *user_data, m_tag tagtype, const xmlChar *value, const xmlChar **attrs) {
	int datatype = 0;
	mstate_stack *m = user_data;

//	M_WP();

	datatype = m->st[m->st_depth-1].type;

	switch(datatype) {
	case M_DATA_TYPE_COUNT:
		if (mdata_Count_from_xml(user_data, tagtype, value, attrs))
			return -1;
		break;
	case M_DATA_TYPE_VISITED:
		if (mdata_Visited_from_xml(user_data, tagtype, value, attrs))
			return -1;
		break;
	case M_DATA_TYPE_BROKENLINK:
		if (mdata_BrokenLink_from_xml(user_data, tagtype, value, attrs))
			return -1;
		break;
	case M_DATA_TYPE_VISIT:
		if (mdata_Visit_from_xml(user_data, tagtype, value, attrs))
			return -1;
		break;
	case M_DATA_TYPE_WEBHIST:
		if (mdata_WebHist_from_xml(user_data, tagtype, value, attrs))
			return -1;
		break;
	case M_DATA_TYPE_MAILHIST:
		if (mdata_Mailhist_from_xml(user_data, tagtype, value, attrs))
			return -1;
		break;
	case M_DATA_TYPE_SUBLIST:
		if (mdata_SubList_from_xml(user_data, tagtype, value, attrs))
			return -1;
		break;
	case M_DATA_TYPE_LOCATION:
		if (mdata_Location_from_xml(user_data, tagtype, value, attrs))
			return -1;
		break;
	case M_DATA_TYPE_UNSET:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't %s unset datatype\n",
			 "read");
		return -1;
		break;
	default:
		M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't %s xml for field '%s' - datatype: %d\n",
			 "read",
			 value,
			 datatype);
		return -1;
	}

	return 0;
}

/**
 * get the count of the countable elements
 *
 * @param data the element which is queried
 * @return the count value or 0 if the uncountable value is queried
 */

int mdata_get_count(mdata *data) {
	if (data == NULL) return 0;

	switch(data->type) {
	case M_DATA_TYPE_COUNT:      return data->data.count.count;
	case M_DATA_TYPE_VISITED:    return data->data.visited.count;
	case M_DATA_TYPE_BROKENLINK: return data->data.brokenlink.count;
	case M_DATA_TYPE_SUBLIST:    return data->data.sublist.count;
	case M_DATA_TYPE_LOCATION:   return data->data.location->count;
	case M_DATA_TYPE_UNSET:
		fprintf(stderr, "%s.%d: can't %s from unset datatype (%s)\n", __FILE__, __LINE__, "get count", data->key);
		break;
	default:
		fprintf(stderr, "%s.%d: can't %s from datatype '%d'\n", __FILE__, __LINE__, "get count", data->type);
	}
	return 0;
}

/**
 * get the vcount of the elements with vcount (currently visits only)
 *
 * @param data the element which is queried
 * @return the vcount value or 0 if the uncountable value is queried
 */

double mdata_get_vcount(const mdata *data) {
	if (data == NULL) return 0;

	switch(data->type) {
	case M_DATA_TYPE_VISITED:  return data->data.visited.vcount;
	case M_DATA_TYPE_UNSET:
		fprintf(stderr, "%s.%d: can't %s from unset datatype (%s)\n", __FILE__, __LINE__, "get vcount", data->key);
		break;
	default:
		fprintf(stderr, "%s.%d: can't %s from datatype '%d'\n", __FILE__, __LINE__, "get vcount", data->type);
	}
	return 0;
}

/**
 * set the count of countable elements
 *
 * @param data
 * @param count
 * @return -1 error, 0 ok
 */

int mdata_set_count(mdata *data, int count) {
	if (data == NULL) return -1;

	switch(data->type) {
		case M_DATA_TYPE_COUNT:
			data->data.count.count = count;
			return 0;
		case M_DATA_TYPE_VISITED:
			data->data.visited.count = count;
			return 0;
		case M_DATA_TYPE_BROKENLINK:
			data->data.brokenlink.count = count;
			return 0;
		case M_DATA_TYPE_SUBLIST:
			data->data.sublist.count = count;
			return 0;
		case M_DATA_TYPE_LOCATION:
			data->data.location->count = count;
			return 0;
		case M_DATA_TYPE_UNSET:
			fprintf(stderr, "%s.%d: can't %s for unset datatype\n", __FILE__, __LINE__, "set count");
			break;
		default:
			fprintf(stderr, "%s.%d: can't %s for datatype '%d'\n", __FILE__, __LINE__, "set count", data->type);
	}
	return -1;
}

/**
 * transform a xmlleaf to the internals strutures
 *
 * @param dest destination
 * @param type type of the destination
 * @param node the xml leaf
 * @return
 */

int mdata_insert_value(void *user_data, m_tag tagtype, const xmlChar *value, const xmlChar **attrs) {
	mstate_stack *m = user_data;
	int type;
	void *dest;

	type = m->st[m->st_depth-1].type;
	dest = m->st[m->st_depth-1].data;

//	M_WP();

	switch (tagtype) {
	case M_TAG_TEXT:
		switch (type) {
		case M_DATA_FIELDTYPE_LONG: {
			long str = *(long *)(dest);

			if (str != 0) {
				/* the TEXT field was divided in several parts
				 * we have to cat them together again
				 *
				 * the current value of '*dest' is first part of our number
				 */
				char cat_buf[255];
				sprintf(cat_buf, "%ld%s", str, value);
				str = strtol(cat_buf, NULL, 10);
			} else {
				str = strtol(value, NULL, 10);
			}

			*(long *)(dest) = str;
			break;
		}
		case M_DATA_FIELDTYPE_DOUBLE: {
			double str = *(double *)(dest);

			if (str != 0) {
				/* the TEXT field was divided in several parts
				 * we have to cat them together again
				 *
				 * the current value of '*dest' is first part of our number
				 */
				char cat_buf[255];
				sprintf(cat_buf, "%.0f%s", str, value);
				str = strtod(cat_buf, NULL);
			} else {
				str = strtod(value, NULL);
			}

			*(double *)(dest) = str;
			break;
		}
		case M_DATA_FIELDTYPE_STRING: {
			char *str = *(char **)(dest);

			if (str != NULL) {
				char *cat_buf;
				char *dec_str;
				/* the TEXT field was divided in several parts
				 * we have to cat them together again
				 *
				 * the current value of '*dest' is first part of our string
				 *
				 *
				 */


				if (str[strlen(str)-1] == '%') {
					char *str2 = malloc(strlen(value) + 1 + 1);
					strcpy(str2, "%");
					strcat(str2, value);

					dec_str = url_decode(str2);
					free(str2);
				} else {
					dec_str = url_decode(value);
				}
				cat_buf = malloc(strlen(str) + strlen(dec_str) + 1);

				strcpy(cat_buf, str);
				strcat(cat_buf, dec_str);

				*(char **)(dest) = cat_buf;

				free(dec_str);
				free(str);
			} else {
				*(char **)(dest) = url_decode(value);
			}

			break;
		}
		default:
			M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "(text) unhandled tagtype: %d, %s\n",
				 tagtype, value);
			return -1;
		}
		break;

	case M_TAG_BEGIN:
//		M_WP();
		switch (type) {

		case M_DATA_FIELDTYPE_LIST:
		case M_DATA_FIELDTYPE_HASH: {
			mdata *h;
			int type;

			if (!attrs || !(0 == strcmp(attrs[0], "key"))) {
				M_WP();
				return -1;
			}

			type = mdata_get_datatype_from_key(value);
			h = mdata_datatype_init(type);
			assert(h);
			
			h->key = url_decode(attrs[1]);

//			M_WP();

			m->st[m->st_depth].function = mdata_read;
			m->st[m->st_depth].type = mdata_get_datatype_from_key(value);
			m->st[m->st_depth].data = h;

			break;
		}
		case M_DATA_FIELDTYPE_WEB_ARRAY: {
			int i;
			int ndx;
			mdata_values data_values[] = {
				{ "hits", M_DATA_FIELDTYPE_LONG },
				{ "files", M_DATA_FIELDTYPE_LONG },
				{ "pages", M_DATA_FIELDTYPE_LONG },
				{ "visits", M_DATA_FIELDTYPE_LONG },
				{ "hosts", M_DATA_FIELDTYPE_LONG },
				{ "xfersize", M_DATA_FIELDTYPE_DOUBLE },

				{ NULL, M_DATA_FIELDTYPE_UNSET }
			};


//			M_WP();

			if (attrs && attrs[0] && (0 == strcmp(attrs[0], "index"))) {
//				M_WP();

				ndx = strtol(attrs[1], NULL, 10);

				m->st[m->st_depth].function = mdata_insert_value;
				m->st[m->st_depth].type = M_DATA_FIELDTYPE_WEB_ARRAY;
				m->st[m->st_depth].data = &(((marray_web *)(m->st[m->st_depth-1].data))[ndx]);
			} else {

				for (i = 0; data_values[i].string && (0 != strcmp(data_values[i].string, value)); i++) ;
//				M_WP();
				if (!data_values[i].string) {
					M_WP();
					return -1;
				}

//				M_WP();
				switch (i) {
				case 0:
					m->st[m->st_depth].data = &(((marray_web *)(m->st[m->st_depth-1].data))->hits);
					break;
				case 1:
					m->st[m->st_depth].data = &(((marray_web *)(m->st[m->st_depth-1].data))->files);
					break;
				case 2:
					m->st[m->st_depth].data = &(((marray_web *)(m->st[m->st_depth-1].data))->pages);
					break;
				case 3:
					m->st[m->st_depth].data = &(((marray_web *)(m->st[m->st_depth-1].data))->visits);
					break;
				case 4:
					m->st[m->st_depth].data = &(((marray_web *)(m->st[m->st_depth-1].data))->hosts);
					break;
				case 5:
					m->st[m->st_depth].data = &(((marray_web *)(m->st[m->st_depth-1].data))->xfersize);
					break;
				}

				m->st[m->st_depth].function = mdata_insert_value;
				m->st[m->st_depth].type = data_values[i].type;
			}
			break;
		}
		default:
			M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "(begin) unhandled tagtype: %d, %s\n",
				 tagtype, value);
			return -1;
		}
		break;
	case M_TAG_END:
		switch(type) {
		case M_DATA_FIELDTYPE_HASH:
		case M_DATA_FIELDTYPE_WEB_ARRAY:
		case M_DATA_FIELDTYPE_LONG:
			break;
		default:
			break;
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "(end) unhandled tagtype: %d, %s, %d\n",
				 tagtype, value, type);
			return -1;
		}
		break;


	default:
		M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "unknown type: %d, %s\n",
			 tagtype,
			 value);
		return -1;
	}

	return 0;
}

void mdata_array_msort(mdata **md, int *a, int *b, int l, int r, int sortby, int sortdir) {
	int i, j, k, m;

	if (r > l) {
		m = (r + l) / 2;
		mdata_array_msort(md, a, b, l, m, sortby, sortdir);
		mdata_array_msort(md, a, b, m+1, r, sortby, sortdir);
		for (i = m + 1; i > l; i--) b[i-1] = a[i-1];
		for (j = m; j < r; j++) b[r+m-j] = a[j+1];
		for (k = l; k <= r; k++) {
			switch (sortby) {
			case M_SORTBY_KEY:
				if (sortdir == M_SORTDIR_ASC) {
					a[k] = (strcmp(md[b[i]]->key, md[b[j]]->key) < 0) ? b[i++] : b[j--];
				} else {
					a[k] = (strcmp(md[b[i]]->key, md[b[j]]->key) > 0) ? b[i++] : b[j--];
				}
				break;
			case M_SORTBY_COUNT:
				if (sortdir == M_SORTDIR_ASC) {
					a[k] = (mdata_get_count(md[b[i]]) < mdata_get_count(md[b[j]])) ? b[i++] : b[j--];
				} else {
					a[k] = (mdata_get_count(md[b[i]]) > mdata_get_count(md[b[j]])) ? b[i++] : b[j--];
				}
				break;
			case M_SORTBY_VCOUNT:
				if (sortdir == M_SORTDIR_ASC) {
					a[k] = (mdata_get_vcount(md[b[i]]) < mdata_get_vcount(md[b[j]])) ? b[i++] : b[j--];
				} else {
					a[k] = (mdata_get_vcount(md[b[i]]) > mdata_get_vcount(md[b[j]])) ? b[i++] : b[j--];
				}
				break;
			case M_SORTBY_QUOTIENT:
				if (sortdir == M_SORTDIR_ASC) {
					a[k] = (mdata_get_vcount(md[b[i]]) < mdata_get_vcount(md[b[j]])) ? b[i++] : b[j--];
				} else {
					a[k] = ((mdata_get_count(md[b[i]]) / mdata_get_vcount(md[b[i]])) > (mdata_get_count(md[b[j]]) / mdata_get_vcount(md[b[j]]))) ? b[i++] : b[j--];
				}
				break;
			default:
				break;
			}
		}
	}
}

int mdata_is_grouped(mdata *data) {
	switch (data->type) {
	case M_DATA_TYPE_COUNT:
		return (data->data.count.grouped == M_DATA_STATE_GROUPED);
	case M_DATA_TYPE_VISITED:
		return (data->data.visited.grouped == M_DATA_STATE_GROUPED);
	case M_DATA_TYPE_BROKENLINK:
		return (data->data.brokenlink.grouped == M_DATA_STATE_GROUPED);
	default:
		fprintf(stderr, "%s.%d: unhandled datatype for grouped: %d\n", __FILE__, __LINE__, data->type);
		return 0;
	}
}

int mdata_show(const mdata *data) {
	switch (data->type) {
	case M_DATA_TYPE_BROKENLINK:
		mdata_BrokenLink_show(data);
		return 0;
	default:
		fprintf(stderr, "%s.%d: unhandled datatype for show: %d\n", __FILE__, __LINE__, data->type);
		return -1;
	}
}


int mdata_get_key_type(const mdata *data) {
	return M_DATA_KEY_TYPE_STRING;
}

const char *mdata_get_key(mdata *data, void *st) {
	return data->key;
}
