/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: parse.c,v 1.15 2003/04/18 18:40:42 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "misc.h"

#include "plugin_config.h"

/* #define DEBUG_PCRE 0 */

const char *short_month[] = {	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			"Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL};

int find_os (mconfig *ext_conf, char *str) {
	config_input *conf = ext_conf->plugin_conf;
	mlist *l = conf->match_os;
	int str_len;
	if (!str || !l) return 0;

	while (*str == ' ') str++;
	
	str_len = strlen(str);

	for (l = conf->match_os; l; l = l->next) {
		mdata *data = l->data;

		if (data && strmatch(data->data.match.match, NULL, str, str_len)) {
			return 1;
		}
	}

	return 0;
}

int find_ua (mconfig *ext_conf, char *str) {
	config_input *conf = ext_conf->plugin_conf;
	mlist *l = conf->match_ua;
	int str_len;
	
	if (!str || !l) return 0;

	while (*str == ' ') str++;
	
	str_len = strlen(str);

	for (l = conf->match_ua; l; l = l->next) {
		mdata *data = l->data;

		if (data && strmatch(data->data.match.match, NULL, str, str_len)) {
			return 1;
		}
	}

	return 0;
}

int parse_useragent(mconfig *ext_conf,const char *str, mlogrec_web_extclf *record) {
/* get user agent */
	char *pc1 = (char *)str, *pc3, *pc2 = (char *)str, *buf_copy;

	buf_copy = malloc(strlen(str)+1);
	strcpy(buf_copy, str);

	if ((pc3 = strchr(pc1, '(') )) {
		if (strstr(pc3, "compatible")) {
			int finished = 0;

			pc1 = pc2 = (pc3+1);

			while (!finished) {
				while (*pc2 && !(*pc2 == ';' || *pc2 == ')')) pc2++;
				if (!*pc2) {
					M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_VERBOSE,
						"the 'Useragent' field of the logfile is incorrect -> '%s'\n", buf_copy);
					free(buf_copy);
					return -1;
				} else if (*pc2 == ')') {
					finished = 1;
				}

				while (*pc1 == ' ') pc1++;

				*pc2 = '\0';
				if (!record->req_useragent && find_ua(ext_conf, pc1)) {
					buffer_strcpy(record->req_useragent, pc1);
				} else if (!record->req_useros && find_os(ext_conf, pc1)) {
					buffer_strcpy(record->req_useros, pc1);
				}
				pc1 = ++pc2;
			}


		} else {
			int finished = 0;

			pc2 = pc3;

			*pc2 = '\0';

			buffer_strcpy(record->req_useragent, pc1);

			pc1 = pc2 = (pc3+1);

			while (!finished) {
				while (*pc2 && !(*pc2 == ';' || *pc2 == ')')) pc2++;
				if (!*pc2) {
					M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
						"the 'Useragent' field of the logfile is incorrect: %s",buf_copy);
					free(buf_copy);
					return -1;
				} else if (*pc2 == ')') {
					finished = 1;
				}

				while (*pc1 == ' ') pc1++;

				*pc2 = '\0';

				if (!record->req_useros && find_os(ext_conf, pc1)) {
					buffer_strcpy(record->req_useros, pc1);
				}
				pc1 = ++pc2;
			}
		}

	} else {
		buffer_strcpy(record->req_useragent, str);
	}

	free(buf_copy);

	return 0;
}

int parse_timestamp(mconfig *ext_conf, const char *str, mlogrec *record) {
#define N 20 + 1
	int ovector[3 * N], n, i;
	char buf[10];
	struct tm tm;
	config_input *conf = ext_conf->plugin_conf;

	if ((n = pcre_exec(conf->match_timestamp, conf->match_timestamp_extra, str, strlen(str), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: string doesn't match: %s\n", __FILE__, __LINE__, str);
		} else {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
		}
		return -1;
	}

	memset(&tm, 0, sizeof(struct tm));

	/* everything has matched, take the different pieces and be happy :) */
	pcre_copy_substring(str, ovector, n, 1, buf, sizeof(buf));
	tm.tm_mday = strtol(buf, NULL, 10);

	pcre_copy_substring(str, ovector, n, 2, buf, sizeof(buf));
	for (i = 0; short_month[i];i++) {
		if (!strcmp(buf, short_month[i])) {
			tm.tm_mon = i;
		}
	}

	pcre_copy_substring(str, ovector, n, 3, buf, sizeof(buf));
	tm.tm_year = strtol(buf, NULL, 10)-1900;

	pcre_copy_substring(str, ovector, n, 4, buf, sizeof(buf));
	tm.tm_hour = strtol(buf, NULL, 10);
	pcre_copy_substring(str, ovector, n, 5, buf, sizeof(buf));
	tm.tm_min = strtol(buf, NULL, 10);
	pcre_copy_substring(str, ovector, n, 6, buf, sizeof(buf));
	tm.tm_sec = strtol(buf, NULL, 10);

	record->timestamp = mktime (&tm);

	return 0;
#undef  N
}

int parse_url(mconfig *ext_conf,const char *str, mlogrec_web *record) {
#define N 20 + 1
	int ovector[3 * N], n;
#ifdef DEBUG_INPUT
	int i;
#endif
	config_input *conf = ext_conf->plugin_conf;
	const char **list;

	if ((n = pcre_exec(conf->match_url, conf->match_url_extra, str, strlen(str), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: url doesn't match: %s\n", __FILE__, __LINE__, str);
		} else {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
		}
		return -1;
	}

	pcre_get_substring_list(str, ovector, n, &list);
	
	switch (n) {
	case 5:
		buffer_strcpy(record->req_protocol, list[4]);
	case 4:
		if (*list[3]) buffer_strcpy(record->req_getvars, list[3]);
	case 3:
		buffer_strcpy(record->req_method, list[1]);
		buffer_strcpy(record->req_url, list[2]);
		
#ifdef DEBUG_INPUT
		for (i = 0; i < n ; i++) {
			printf("--> %d: %s\n", i, list[i]);
		}

		fprintf(stderr, "%s.%d: %s, %s, %s, %s\n", __FILE__, __LINE__, record->req_method, record->req_url, record->req_getvars, record->req_protocol);
#endif
		free(list);
		break;
	default:	
		fprintf(stderr, "%s.%d: Matched fields below minimum: %d\n", __FILE__, __LINE__, n);
		return -1;
	}

	return 0;
#undef  N
}

int parse_record_pcre(mconfig *ext_conf, mlogrec *record, buffer *b) {
#define N 20 + 1
	const char **list;
	int ovector[3 * N], n;
#ifdef DEBUG_PCRE
	int i;
#endif
	config_input *conf = ext_conf->plugin_conf;
	mlogrec_web *recweb = NULL;
	mlogrec_web_extclf *recext = NULL;
	
	if (record->ext_type != M_RECORD_TYPE_WEB) {
		if (record->ext_type != M_RECORD_TYPE_UNSET) {
			mrecord_free_ext(record);
		}
		
		record->ext_type = M_RECORD_TYPE_WEB;
		record->ext = mrecord_init_web();
	}

	recweb = record->ext;
	
	if (recweb == NULL) return M_RECORD_HARD_ERROR;
	
	recweb->ext = mrecord_init_web_extclf();
	recweb->ext_type = M_RECORD_TYPE_WEB_EXTCLF;

	recext = recweb->ext;

	if (recext == NULL) return -1;

	if ((n = pcre_exec(conf->match_clf, conf->match_clf_extra, b->ptr, b->used - 1, 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: string doesn't match: %s\n", __FILE__, __LINE__, b->ptr);
		} else {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
		}
		return -1;
	}

	if (n > 7) {
		pcre_get_substring_list(b->ptr, ovector, n, &list);
	/*
	   1 - client-ip
	   2 - '-'
	   3 - '-'
	   4 - timestamp
	   5 - request
	   6 - status
	   7 - xfersize
	   8 - client info
	   9 - client uid
	   10 - statistics
	   11-16 - currently unknown
	*/
		buffer_strcpy(recweb->req_host_ip, (char *)list[1]);

		if (parse_timestamp(ext_conf, list[4], record) == -1) {
			free(list);
			return -1;
		}

		if (parse_url(ext_conf, list[5], recweb) == -1) {
			free(list);
			return -1;
		}

		if (parse_useragent(ext_conf, list[8], recext) == -1) {
			free(list);
			return -1;
		}

		recweb->req_status = strtol(list[6], NULL,10);
		recweb->xfersize = strtol(list[7], NULL,10);

		/* logging level > 0 */
		if (n > 15) {
			recext->duration = strtol(list[15], NULL, 10);
		}

		pcre_free(list);
	} else {
		fprintf(stderr, "%s.%d: Matched fields below minimum: %d %s\n", __FILE__, __LINE__, n, b->ptr);
		return -1;
	}

#ifdef DEBUG_PCRE
	pcre_get_substring_list(b->ptr, ovector, n, &list);
	for (i = 0; i < n; i++) {
		printf("%d: %s\n", i, list[i]);
	}
	pcre_free(list);
#endif
	return 0;
#undef  N
}

int mplugins_input_realserver_get_next_record(mconfig *ext_conf, mlogrec *record) {
	int ret = 0;
	config_input *conf = ext_conf->plugin_conf;

	if (record == NULL) return M_RECORD_HARD_ERROR;

	/* fill the line buffer */
	if (NULL == mgets(&(conf->inputfile), conf->buf)) return M_RECORD_EOF;
	
	ret = parse_record_pcre   (ext_conf, record, conf->buf);
	
	if (ret == M_RECORD_CORRUPT) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
			 "affected Record: %s\n",
			 conf->buf->ptr
			 );
	}
	return ret;
}
