/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: datatype.c,v 1.2 2003/01/06 11:11:43 ostborn Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "config.h"
#include "mdatatypes.h"
#include "datatype.h"
#include "mconfig.h"
#include "misc.h"

/* begin of Location */

int mdata_Location_to_xml(gzFile *fd, mdata *data) {
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.location->x, #x);
#define WRS(x,f) \
	if (data->data.location->x != NULL) { \
		char *s = url_encode(data->data.location->x); \
		gzprintf(fd, "<%s>", #x); \
		gzwrite(fd, s, strlen(s)); \
		gzprintf(fd, "</%s>", #x); \
		free(s); \
	} else { \
		gzprintf(fd, "<%s />", #x); \
	}
	
	WR(count, d);

	WRS(city, s);
	WRS(province, s);
	WRS(country, s);
	WRS(provider, s);
	
	return 0;
}

int mdata_Location_free(mdata *data) {
	if (!data) return -1;
	if (data->type != M_DATA_TYPE_LOCATION) return -1;

#define FREE(x) \
	if (data->data.location->x) free(data->data.location->x);
	
	FREE(city);
	FREE(province);
	FREE(country);
	FREE(provider);
#undef FREE
	
	free(data->data.location);
	
	return 0;
}



int mdata_Location_setdata(mdata *data, const char *str, int count, 
			   char *city,
			   char *province, 
			   char *country,
			   char *provider) {

	data->key = strdup(str);
	assert(data->key);

	if (data->type == M_DATA_TYPE_UNSET)
		data->type		= M_DATA_TYPE_LOCATION;

	data->data.location->count	= count;
	data->data.location->city	= strdup(city ? city : "");
	data->data.location->province	= strdup(province ? province : "");
	data->data.location->country	= strdup(country ? country : "");
	data->data.location->provider	= strdup(provider ? provider : "");
	
	return 0;
}

int mdata_Location_from_xml(void *user_data, int tagtype, const xmlChar *value, const xmlChar **attrs) {
	int i;
	const mdata_values data_values[] = {
		{ "count", M_DATA_FIELDTYPE_LONG },
		
		{ "city", M_DATA_FIELDTYPE_STRING },
		{ "province", M_DATA_FIELDTYPE_STRING },
		{ "country", M_DATA_FIELDTYPE_STRING },
		{ "provider", M_DATA_FIELDTYPE_STRING },

		{ NULL, M_DATA_FIELDTYPE_UNSET }
	};

	mstate_stack *m = user_data;

//	M_WP();

	switch(tagtype) {
	case M_TAG_BEGIN:
		for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
			;

		if (data_values[i].string == NULL) {
			M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "unknown tag '%s'\n",
				 value);
			return -1;
		}
#define SET_DATA(x) \
	m->st[m->st_depth].data = &(((mdata *)(m->st[m->st_depth-1].data))->x);
		switch(i) {
		case 0: SET_DATA(data.location->count); break;
		case 1: SET_DATA(data.location->city); break;
		case 2: SET_DATA(data.location->province); break;
		case 3: SET_DATA(data.location->country); break;
		case 4: SET_DATA(data.location->provider); break;
		default:
			return -1;
		}
#undef SET_DATA
		/* the same for all 4 values */
		m->st[m->st_depth].function = mdata_insert_value;
		m->st[m->st_depth].type = data_values[i].type;

		break;
	case M_TAG_END:
		((mdata *)(m->st[m->st_depth-1].data))->type = M_DATA_TYPE_LOCATION;

		switch(m->st[m->st_depth-2].type) {
		case M_DATA_FIELDTYPE_HASH: {
			mhash *h = m->st[m->st_depth-2].data;
			mdata *data = m->st[m->st_depth-1].data;

			mhash_insert_sorted(h, data);
			break;
		}
		default:
			M_WP();
		}

		break;
	case M_TAG_TEXT:
		M_WP();
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't handle tagtype '%d'\n",
			 tagtype);
		return -1;
	}

	return 0;
}



int mdata_Location_append(mdata *dst, mdata *src) {
	dst->data.location->count += src->data.location->count;

	return 0;
}

mdata *mdata_Location_init() {
	mdata *data = mdata_init();
	assert(data);

	data->key		= NULL;
	data->type		= M_DATA_TYPE_LOCATION;
	
	data->data.location        = malloc(sizeof(data_Location));
	memset(data->data.location, 0, sizeof(data_Location));

#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: got %ld (%d)\n", __FILE__, __LINE__, data->id, data->type);
#endif
/* init specifics */

/*
	data->data.location->count		= 0;
	data->data.location->grouped	= M_DATA_STATE_PLAIN;
*/

	return data;
}

mdata *mdata_Location_copy(mdata *src) {
	mdata *data = mdata_Location_init();
	assert(data);

	fprintf(stderr, "%s.%d: (location) abstract function 'copy' !!\n", __FILE__, __LINE__);

/*
	mdata_Location_setdata(data, src->key, src->data.location->count, src->data.location->grouped);
*/
	return data;
}

mdata *mdata_Location_create(const char *str, int count, 
			     char *city,
			     char *province, 
			     char *country,
			     char *provider) {
	
	mdata *data = mdata_Location_init();
	assert(data);
	mdata_Location_setdata(data, str, count, city, province, country, provider);

	return data;
}

/* end of Location */
