/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: plugin_config.c,v 1.46 2003/07/18 07:23:56 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "datatypes/match/datatype.h"
#include "datatypes/sublist/datatype.h"
#include "datatypes/split/datatype.h"
#include "misc.h"

#include "plugin_config.h"

int mplugins_processor_web_dlinit(mconfig *ext_conf) {
	config_processor *conf = NULL;

	if (0 != strcmp(ext_conf->version, VERSION)) {
		M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "version string doesn't match: (mla) %s != (plugin) %s\n", ext_conf->version, VERSION);
		return -1;
	}

	conf = malloc(sizeof(config_processor));
	memset(conf, 0, sizeof(config_processor));

	conf->page_type		= mlist_init();

	conf->ignore_host	= mlist_init();
	conf->ignore_url	= mlist_init();
	conf->ignore_ua		= mlist_init();
	conf->ignore_hostmask	= mlist_init();
	conf->ignore_searchengine= mlist_init();

	conf->hide_url		= mlist_init();
	conf->hide_referrer	= mlist_init();
	conf->hide_host		= mlist_init();
	conf->hide_brokenlinks	= mlist_init();
	conf->hide_brokenlinks_ref	= mlist_init();
	conf->hide_robots	= mlist_init();
	conf->hide_extension	= mlist_init();
	conf->hide_hostmask	= mlist_init();
	conf->match_searchengine	= mlist_init();

	conf->group_os		= mlist_init();
	conf->group_ua		= mlist_init();
	conf->group_hosts	= mlist_init();
	conf->group_brokenlinks	= mlist_init();
	conf->group_referrer	= mlist_init();
	conf->group_url		= mlist_init();
	conf->group_searchstrings = mlist_init();
	conf->group_robots	= mlist_init();
	conf->group_extension	= mlist_init();
	conf->group_searchengine	= mlist_init();

	conf->searchengines	= mlist_init();

	conf->splitby		= mlist_init();
	conf->split_def		= mlist_init();

	conf->host_cache = malloc(sizeof(m_cache));
	conf->host_cache->entry = NULL;
	conf->host_cache->last_tstmp = 0;
	
	conf->grouped           = buffer_init();

	ext_conf->plugin_conf = conf;

	return 0;
}

int mplugins_processor_web_dlclose(mconfig *ext_conf) {
	config_processor *conf = ext_conf->plugin_conf;

	if (!conf) {
		fprintf(stderr, "conf == NULL !\n");
		return -1;
	}

	if (conf->host_cache_max > 0) {
		int i;
		for (i = 0; i < conf->host_cache_max; i++) {
			if (conf->host_cache->entry[i]) {
				if (conf->host_cache->entry[i]->key) free(conf->host_cache->entry[i]->key);
				if (conf->host_cache->entry[i]->value) free(conf->host_cache->entry[i]->value);
				free(conf->host_cache->entry[i]);
			}
		}
		free(conf->host_cache->entry);
	}
	free(conf->host_cache);
	
	buffer_free(conf->grouped);

	mlist_free(conf->page_type);

	mlist_free(conf->ignore_host);
	mlist_free(conf->ignore_url);
	mlist_free(conf->ignore_ua);
	mlist_free(conf->ignore_hostmask);
	mlist_free(conf->ignore_searchengine);

	mlist_free(conf->hide_url);
	mlist_free(conf->hide_referrer);
	mlist_free(conf->hide_host);
	mlist_free(conf->hide_brokenlinks);
	mlist_free(conf->hide_brokenlinks_ref);
	mlist_free(conf->hide_robots);
	mlist_free(conf->hide_extension);
	mlist_free(conf->hide_hostmask);
	mlist_free(conf->match_searchengine);

	mlist_free(conf->group_os);
	mlist_free(conf->group_ua);
	mlist_free(conf->group_hosts);
	mlist_free(conf->group_referrer);
	mlist_free(conf->group_brokenlinks);
	mlist_free(conf->group_url);
	mlist_free(conf->group_searchstrings);
	mlist_free(conf->group_robots);
	mlist_free(conf->group_extension);
	mlist_free(conf->group_searchengine);

	mlist_free(conf->searchengines);
	mlist_free(conf->splitby);
	mlist_free(conf->split_def);

	if (conf->log_ungrouped_searchengines_file) free(conf->log_ungrouped_searchengines_file);
	if (conf->log_ungrouped_FILE) fclose(conf->log_ungrouped_FILE);
	
	if (conf->localizer_db) free(conf->localizer_db);
#ifdef HAVE_LIBLOCALIZER
	if (conf->localizer) localizer_free(conf->localizer);
#endif
	
	free(ext_conf->plugin_conf);
	ext_conf->plugin_conf = NULL;

	return 0;
}

int mplugins_processor_web_parse_config(mconfig *ext_conf, const char *cf, const char *section) {
	config_processor *conf = ext_conf->plugin_conf;

	const mconfig_values config_values[] = {
	/* rewritings */
		{"groupreferrer", M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_referrer)},
		{"groupos",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_os)},
		{"groupua",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_ua)},
		{"grouphosts",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_hosts)},
		{"groupbrokenlinks", M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_brokenlinks)},
		{"groupurl",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_url)},
		{"groupsearchstrings",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_searchstrings)},
		{"grouprobots",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_robots)},
		{"groupextension", M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_extension)},
		{"groupsearchengine", M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_searchengine)},
	/* simple matches */
		{"hideurl",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_url)},
		{"hidereferrer", M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_referrer)},
		{"hidehost",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_host)},
		{"hidebrokenlinks", M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_brokenlinks)},
		{"hidebrokenlinksref", M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_brokenlinks_ref)},
		{"hiderobots",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_robots)},
		{"hideextension", M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_extension)},
		{"hidehostmask", M_CONFIG_TYPE_STRING_LIST,	M_CONFIG_VALUE_APPEND, &(conf->hide_hostmask)},
		{"matchsearchengine", M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->match_searchengine)},
		{"ignorehost",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->ignore_host)},
		{"ignoreurl",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->ignore_url)},
		{"ignoreua",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->ignore_ua)},
		{"ignorehostmask",	M_CONFIG_TYPE_STRING_LIST,	M_CONFIG_VALUE_APPEND, &(conf->ignore_hostmask)},
		{"ignoresearchengine",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->ignore_searchengine)},
		{"pagetype",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->page_type)},
		{"splitby",	M_CONFIG_TYPE_STRING_LIST,	M_CONFIG_VALUE_APPEND, &(conf->splitby)},
	/* strings */
		{"log_ungrouped_searchengines_file", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->log_ungrouped_searchengines_file)},
	/* integers */
		{"debug_searchengines", M_CONFIG_TYPE_INT, M_CONFIG_VALUE_OVERWRITE, &(conf->debug_searchengines)},
		{"debug_visits", M_CONFIG_TYPE_INT,     M_CONFIG_VALUE_OVERWRITE, &(conf->debug_visits)},
		{"visit_timeout", M_CONFIG_TYPE_INT,    M_CONFIG_VALUE_OVERWRITE, &(conf->visit_timeout)},
		{"debug_timing", M_CONFIG_TYPE_INT,    M_CONFIG_VALUE_OVERWRITE, &(conf->debug_timing)},
		{"maxhostcacheentries", M_CONFIG_TYPE_INT,    M_CONFIG_VALUE_OVERWRITE, &(conf->host_cache_max)},
		{"max_hits_per_visit", M_CONFIG_TYPE_INT,    M_CONFIG_VALUE_OVERWRITE, &(conf->max_hits_per_visit)},
		{"log_ungrouped_searchengines", M_CONFIG_TYPE_INT, M_CONFIG_VALUE_OVERWRITE, &(conf->log_ungrouped_searchengines)},
		{"decode_searchstrings", M_CONFIG_TYPE_INT, M_CONFIG_VALUE_OVERWRITE, &(conf->decode_searchstrings)},
		{"localizer_db", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->localizer_db)},
		{NULL, M_CONFIG_TYPE_INT, 0, NULL}
	};

	if (conf == NULL) return -1;

	return mconfig_parse_section(ext_conf, cf, section, config_values);
}

struct field_value {
	char	*string;
	int	type;
};

int mplugins_processor_web_set_defaults(mconfig *ext_conf) {
	config_processor *conf = ext_conf->plugin_conf;

	if (conf->log_ungrouped_searchengines
	    && conf->log_ungrouped_searchengines_file
	    && *conf->log_ungrouped_searchengines_file) {
		char *fn = malloc(strlen(ext_conf->statedir ? ext_conf->statedir : ".")
				  + strlen(conf->log_ungrouped_searchengines_file) + 1 + 1);
		if (fn) {
			if (*conf->log_ungrouped_searchengines_file == '/') {
				/* Absolute path name */
				strcpy(fn, conf->log_ungrouped_searchengines_file);
			} else {
				/* Relative */
				strcpy(fn, ext_conf->statedir ? ext_conf->statedir : ".");
				strcat(fn, "/");
				strcat(fn, conf->log_ungrouped_searchengines_file);
			}

			if (*fn) {
				conf->log_ungrouped_FILE = fopen(fn, "a");
				if (!conf->log_ungrouped_FILE)
					fprintf(stderr, "%s.%d: failed to open '%s': %s\n", __FILE__, __LINE__,
						conf->log_ungrouped_searchengines_file, strerror(errno));
			}
			free(fn);
		}
	}
	
	if (conf->visit_timeout <= 0) {
		conf->visit_timeout = 1800;
	}

	if (conf->host_cache_max < 0) conf->host_cache_max = 0;

	if (conf->host_cache_max > 0) {
		int i;
		conf->host_cache->entry = malloc(sizeof(m_cache_entry *) * conf->host_cache_max);
		for (i = 0; i < conf->host_cache_max; i++) {
			conf->host_cache->entry[i] = malloc(sizeof(m_cache_entry));
			conf->host_cache->entry[i]->key = NULL;
			conf->host_cache->entry[i]->value = NULL;
			conf->host_cache->entry[i]->tstmp = -1;
		}
	}

	if (conf->splitby) {
		mlist *l;
		pcre *match;
		const char *errptr;
		int erroffset = 0;

		if ((match = pcre_compile("^([a-z]+),\\s*\"(.*)\",\\s*(.+)\\s*$",
			0, &errptr, &erroffset, NULL)) == NULL) {
			fprintf(stderr, "%s.%d: rexexp compilation error at %s\n", __FILE__, __LINE__, errptr);
			return -1;
		}

		/* doing some tests with the string */
		for (l = conf->splitby; l; l = l->next) {
			mdata *data = l->data;
#define N 20 + 1
			int ovector[3 * N], n, i;

			if (!data) break;

			/* the dawn string */
			if ((n = pcre_exec(match, NULL, data->key, strlen(data->key), 0, 0, ovector, 3 * N)) < 0) {
				if (n == PCRE_ERROR_NOMATCH) {
					fprintf(stderr, "%s.%d: (splitby) string doesn't match: %s\n", __FILE__, __LINE__, data->key);
				} else {
					fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
				}
				return -1;
			}

#undef N
			if (n >= 3) {
				const char **list;
				/* Split */
				mdata *ins_data = NULL;

				const struct field_value field_values[] = {
				/* local server */
					{ "srvhost",	M_SPLIT_FIELD_SRVHOST },
					{ "srvport",	M_SPLIT_FIELD_SRVPORT },
				/* remote client */
					{ "requser",	M_SPLIT_FIELD_REQUSER },
					{ "requrl",	M_SPLIT_FIELD_REQURL },
					{ "reqhost",	M_SPLIT_FIELD_REQHOST },
				/* client's referrer */
					{ "refurl",	M_SPLIT_FIELD_REFURL },

				/* default */
					{ "default",	M_SPLIT_FIELD_DEFAULT },
					{ NULL,		M_SPLIT_FIELD_UNSET }
				};

				pcre_get_substring_list(data->key, ovector, n, &list);

				/* search split def */
				for (i = 0; field_values[i].string; i++) {
					if (0 == strcmp(field_values[i].string, list[1])) break;
				}

				if (field_values[i].string) {
					ins_data = mdata_Split_create(list[3], field_values[i].type, list[2]);

					if (ext_conf->debug_level > 2)
						fprintf(stderr, "%s.%d: using splitter for \"%s\" type %d\n", __FILE__, __LINE__, list[2], field_values[i].type);

					if (ins_data) {
						mlist_append(conf->split_def, ins_data);
					} else {
						fprintf(stderr, "%s.%d: the definition for the splitter couldn't be created\n", __FILE__, __LINE__);
					}
				} else {
					fprintf(stderr, "%s.%d: the requested key isn't supported: %s\n", __FILE__, __LINE__, list[1]);
				}

				free(list);
			} else {
				fprintf(stderr, "%s.%d: too few fields matched: %d\n", __FILE__, __LINE__, n);
			}
		}

		pcre_free(match);
	}
	
	
#ifdef HAVE_LIBLOCALIZER
	if (conf->localizer_db) {
		if (NULL == (conf->localizer = localizer_init())) {
			return -1;
		}
		if (localizer_read_shm(conf->localizer, conf->localizer_db)) {
			return -1;
		}
	}
#endif

	return 0;
}

int mplugins_processor_web_cleanup(mconfig *ext_conf, mstate *state, const char *subpath) {
	return cleanup_visits(ext_conf, state, state->timestamp);
}

int mplugins_init(mplugin *func) {
	func->dlinit = mplugins_processor_web_dlinit;
	func->dlclose = mplugins_processor_web_dlclose;
	func->parse_config = mplugins_processor_web_parse_config;
	func->set_defaults = mplugins_processor_web_set_defaults;
	func->get_next_record = NULL;
	func->insert_record = mplugins_processor_web_insert_record;
	func->gen_report = mplugins_processor_web_cleanup;
        func->gen_history = NULL;

	return 0;
}
