/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: plugin_config.c,v 1.10 2002/10/23 15:05:36 le_zas Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <limits.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>

#include "mconfig.h"
#include "mstate.h"
#include "mlocale.h"
#include "mhash.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "mplugins.h"

#include "plugin_config.h"

#define M_PLUGIN_NAME "output_skeleton"

int mplugins_output_skeleton_dlinit(mconfig *ext_conf) {
	config_output *conf = NULL;

	if (0 != strcmp(ext_conf->version, VERSION)) {
		M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "version string doesn't match: (mla) %s != (plugin) %s\n", ext_conf->version, VERSION);
		return -1;
	}

	conf = malloc(sizeof(config_output));
	memset(conf, 0, sizeof(config_output));

	ext_conf->plugin_conf = conf;

	return 0;
}

int mplugins_output_skeleton_dlclose(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;

	free(ext_conf->plugin_conf);
	ext_conf->plugin_conf = NULL;

	return 0;
}

int mplugins_output_skeleton_parse_config(mconfig *ext_conf, const char *filename, const char *section) {
	config_output *conf = ext_conf->plugin_conf;

	const mconfig_values config_values[] = {
		{"outputdir", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->outputdir)},

		{NULL, M_CONFIG_TYPE_INT, 0, NULL}
	};

	return mconfig_parse_section(ext_conf, filename, section, config_values);
}

int mplugins_output_skeleton_set_defaults(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	struct stat st;

	if (!mplugins_output_skeleton_patch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not patch config\n");
		return -1;
	}

	if (conf->outputdir == NULL) {
		fprintf(stderr, "ERROR: [%s] no output-directory was set ( outputdir = ... )\n",
			M_PLUGIN_NAME);
		return -1;
	}

	if (stat(conf->outputdir, &st) != 0) {
		fprintf(stderr, "ERROR: [%s] can't check if the output-directory is ok (%s): %s\n",
			M_PLUGIN_NAME,
			conf->outputdir, strerror(errno));
		return -1;
	} else {
		if (S_ISDIR(st.st_mode) &&
			(((st.st_mode & S_IWUSR) && (st.st_mode & S_IRUSR)) ||
			((st.st_mode & S_IWGRP) && (st.st_mode & S_IRGRP)) ||
			((st.st_mode & S_IWOTH) && (st.st_mode & S_IROTH)))
			) {
			/* every thing seems to be ok. */
		} else {
			fprintf(stderr, "ERROR: [%s] the output-directory doesn't have the read and write permissions: %s\n",
				M_PLUGIN_NAME,
				conf->outputdir);
			return -1;
		}
	}

	if (!mplugins_output_skeleton_unpatch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not unpatch config\n");
		return -1;
	}

	return 0;
}

int mplugins_init(mplugin *func) {
	func->dlinit = mplugins_output_skeleton_dlinit;
	func->dlclose = mplugins_output_skeleton_dlclose;
	func->parse_config = mplugins_output_skeleton_parse_config;
	func->set_defaults = mplugins_output_skeleton_set_defaults;
	func->get_next_record = NULL;
	func->insert_record = NULL;
	func->gen_report = NULL;
        func->gen_history = NULL;

	return 0;
}


/**
 * handle the global-variable in the config
 *
 * add a shadow-config which holds the original, unpatched values
 * and include the transformed value in the normal config
 *
 * @see mplugin_output_skeleton_unpatch_config
 */

int mplugins_output_skeleton_patch_config(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf, *orig_conf;

	if (conf->old_conf == NULL) {
		orig_conf = malloc(sizeof(config_output));
		memset(orig_conf, 0, sizeof(config_output));
#define PATCH(x) \
		orig_conf->x = conf->x; conf->x = mconfig_get_value(ext_conf, orig_conf->x);

		PATCH(outputdir);

		conf->old_conf = orig_conf;
#undef PATCH
		return 1;
	} else {
		return 0;
	}
}

/**
 * restore the original onfig
 *
 * @see mplugin_output_skeleton_patch_config
 */

int mplugins_output_skeleton_unpatch_config(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf, *orig_conf;

	if (conf->old_conf != NULL) {
		orig_conf = conf->old_conf;
#define PATCH(x) \
		free(conf->x); conf->x = orig_conf->x;

		PATCH(outputdir);

		free(orig_conf);

		conf->old_conf = NULL;
#undef PATCH
		return 1;
	} else {
		return 0;
	}
}

