/*
 * mod_security (apache 1.x), http://www.modsecurity.org/
 * Copyright (c) 2002-2004 Ivan Ristic <ivanr@webkreator.com>
 *
 * $Id: mod_security.c,v 1.153 2004/11/03 14:29:02 ivanr Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */            

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#if !(defined(WIN32) || defined(NETWARE))
#include <unistd.h>
#endif

#include "httpd.h"
#include "http_config.h"
#include "http_request.h"
#include "http_protocol.h"
#include "http_core.h"
#include "http_main.h"
#include "http_log.h"
#include "util_script.h"

#if defined(NETWARE)
#include <nwsemaph.h>
LONG locking_sem = 0;
#endif

#ifdef WIN32
#include <sys/locking.h>
#endif

#if defined(NETWARE)
#define CREATEMODE ( S_IREAD | S_IWRITE )
#elif defined(WIN32)
#define CREATEMODE ( _S_IREAD | _S_IWRITE )
#else
#define CREATEMODE ( S_IRUSR | S_IWUSR )
#endif

module MODULE_VAR_EXPORT security_module;

#define SMALL_BUF_SIZE 255

#define MODULE_RELEASE "1.8.6"

#define UNICODE_ERROR_CHARACTERS_MISSING    -1
#define UNICODE_ERROR_INVALID_ENCODING      -2
#define UNICODE_ERROR_OVERLONG_CHARACTER    -3

#define MODSEC_SKIP                     -2000

#define NOT_SET                         -1
#define NOT_SET_P                       (void *)-1

#define FILTERING_OFF                   0
#define FILTERING_ON                    1
#define FILTERING_DYNAMIC_ONLY          2

#define AUDITLOG_OFF                    0
#define AUDITLOG_ON                     1
#define AUDITLOG_DYNAMIC_OR_RELEVANT    2
#define AUDITLOG_RELEVANT_ONLY          3

#define ACTION_NONE                 0
#define ACTION_DENY                 1
#define ACTION_REDIRECT             2
#define ACTION_ALLOW                3
#define ACTION_SKIP                 4

#define VAR_ACTION_ALLOW            1
#define VAR_ACTION_DENY             0

#define VAR_UNKNOWN                 0
#define VAR_CUSTOM                  1
#define VAR_HEADER                  2
#define VAR_ENV                     3
#define VAR_ARGS                    4
#define VAR_POST_PAYLOAD            5
#define VAR_ARGS_NAMES              6
#define VAR_ARGS_VALUES             7
#define VAR_ARGS_SELECTIVE          8
#define VAR_OUTPUT                  9
#define VAR_COOKIES_NAMES           10
#define VAR_COOKIES_VALUES          11
#define VAR_COOKIE                  12
#define VAR_RESERVED_5              13
#define VAR_RESERVED_6              14
#define VAR_RESERVED_7              15
#define VAR_RESERVED_8              16
#define VAR_RESERVED_9              17
#define VAR_RESERVED_10             18
#define VAR_RESERVED_11             19
#define VAR_RESERVED_12             20

#define VAR_REMOTE_ADDR             21
#define VAR_REMOTE_HOST             22
#define VAR_REMOTE_USER             23   
#define VAR_REMOTE_IDENT            24
#define VAR_REQUEST_METHOD          25
#define VAR_SCRIPT_FILENAME         26
#define VAR_PATH_INFO               27
#define VAR_QUERY_STRING            28
#define VAR_AUTH_TYPE               29
#define VAR_DOCUMENT_ROOT           30
#define VAR_SERVER_ADMIN            31
#define VAR_SERVER_NAME             32
#define VAR_SERVER_ADDR             33
#define VAR_SERVER_PORT             34
#define VAR_SERVER_PROTOCOL         35
#define VAR_SERVER_SOFTWARE         36
#define VAR_TIME_YEAR               37
#define VAR_TIME_MON                38
#define VAR_TIME_DAY                39
#define VAR_TIME_HOUR               40
#define VAR_TIME_MIN                41
#define VAR_TIME_SEC                42
#define VAR_TIME_WDAY               43
#define VAR_TIME                    44
#define VAR_API_VERSION             45
#define VAR_THE_REQUEST             46
#define VAR_REQUEST_URI             47
#define VAR_REQUEST_FILENAME        48
#define VAR_IS_SUBREQ               49
#define VAR_HANDLER                 50

#define MULTIPART_BUF_SIZE              1024

#define MULTIPART_TMP_FILE_NONE         0
#define MULTIPART_TMP_FILE_CREATE       1
#define MULTIPART_TMP_FILE_CREATE_LEAVE 2

#define MULTIPART_FORMDATA              1
#define MULTIPART_FILE                  2

#define NOTE                        "mod_security-note"
#define NOTE_DYNAMIC                "mod_security-dynamic"
#define NOTE_MESSAGE                "mod_security-message"
#define NOTE_NOAUDITLOG             "mod_security-noauditlog"
#define NOTE_EXECUTED               "mod_security-executed"
#define NOTE_ACTION                 "mod_security-action"
#define NOTE_MSR                    "mod_security-msr"
#define NOTE_ACTED                  "mod_security-relevant"

#define FATAL_ERROR                 "Unable to allocate memory"

#define UNKNOWN_CSID    0
#define MB_CSID         800         /* First multibyte character set */
#define UNI3_CSID       873         /* Unicode 3.x character set ID  */
#define SJIS1_CSID      832         /* SJIS character set ID         */
#define SJIS2_CSID      834         /* SJIS+YEN character set ID     */
#define BIG5_CSID       865         /* BIG5 character set ID         */
#define GBK_CSID        852         /* GBK character set ID          */
#define GB2312_CSID     850         /* GB2312 character set ID       */
#define ZHT32EUC_CSID   860         /* Chinese 4-byte character set  */
#define JEUC1_CSID      830         /* JEUC character set ID         */
#define JEUC2_CSID      831         /* JEUC+YEN character set ID     */
#define JA16VMS_CSID    829         /* VMS 2-byte Japanese           */

static char *all_variables[] = {
    "UNKOWN",
    "CUSTOM",
    "HEADER",
    "ENV",
    "ARGS",
    "POST_PAYLOAD",
    "ARGS_NAMES",
    "ARGS_VALUES",
    "ARGS_SELECTIVE",
    "OUTPUT",
    "COOKIES_NAMES",    /* 10 */
    "COOKIES_VALUES",
    "COOKIE",
    "RESERVED_5",
    "RESERVED_6",
    "RESERVED_7",
    "RESERVED_8",
    "RESERVED_9",
    "RESERVED_10",
    "RESERVED_11",
    "RESERVED_12",      /* 20 */
    "REMOTE_ADDR",
    "REMOTE_HOST",
    "REMOTE_USER",
    "REMOTE_IDENT",
    "REQUEST_METHOD",
    "SCRIPT_FILENAME",
    "PATH_INFO",
    "QUERY_STRING",
    "AUTH_TYPE",
    "DOCUMENT_ROOT",    /* 30 */
    "SERVER_ADMIN",
    "SERVER_NAME",
    "SERVER_ADDR",
    "SERVER_PORT",
    "SERVER_PROTOCOL",
    "SERVER_SOFTWARE",
    "TIME_YEAR",
    "TIME_MON",
    "TIME_DAY",
    "TIME_HOUR",        /* 40 */
    "TIME_MIN",
    "TIME_SEC",
    "TIME_WDAY",
    "TIME",
    "API_VERSION",
    "THE_REQUEST",
    "REQUEST_URI",
    "REQUEST_FILENAME",
    "IS_SUBREQ",
    "HANDLER",          /* 50 */
    NULL
};

typedef struct {
    char *name;
    int type;
    int action;
} variable;

typedef struct {
    request_rec *r;
    char *command;
    char *args;
} exec_data;

typedef struct {
    char *buffer;
    char *sofar;
    long length;
    long remaining;
} request_body;

typedef struct {
    int log;
    int action;
    int status;
    int pause;
    int skip_count;
    int is_chained;
    char *redirect_url;
    int exec;
    char *exec_string;
    char *id;
    char *msg;
} actionset_t;

typedef struct {
    actionset_t *actionset;
    char *pattern;
    regex_t *regex;
    int is_selective;
    int is_negative;
    int is_allow;
    int requires_parsed_args;
    array_header *variables;
} signature;

typedef struct {
    int filter_engine;
    int configuration_helper;
    int scan_post;
    actionset_t *action;
    array_header *signatures;
    char *path;
    int auditlog_flag;
    char *auditlog_name;
    int auditlog_fd;
    int filter_debug_level;
    char *debuglog_name;
    int debuglog_fd;
    int filters_clear;
    int range_start;
    int range_end;
    int check_encoding;
    int check_unicode_encoding;
    char *upload_dir;
    int upload_keep_files;
    char *upload_approve_script;
    int normalize_cookies;
    int check_cookie_format;
    int charset_id;
    int multibyte_replacement_byte;
    pool *p;
} sec_dir_config;

typedef struct {
    int server_response_token;
    char *chroot_dir;
    int chroot_completed;
    char *chroot_lock;
    char *server_signature;
} sec_srv_config;

typedef struct {
    /* part type, can be MULTIPART_FORMDATA or MULTIPART_FILE */
    int type;
    /* the name */
    char *name;
    
    /* variables only, variable the value */
    char *value;
    /* files only, the content type (where available) */
    char *content_type;
    
    /* files only, the name of the temporary file holding data */
    char *tmp_file_name;
    int tmp_file_fd;
    unsigned int tmp_file_size;
    /* files only, filename as supplied by the browser */
    char *filename;
} multipart_part;

typedef struct {
    request_rec *r;
    sec_dir_config *dcfg;
    pool *p;    
    
    /* this array keeps parts */
    array_header *parts;
    
    /* do we need to make a copy of the request
     * in a temporary file
     */
    int create_tmp_file;
    char *tmp_file_name;
    int tmp_file_fd;
    
    /* mime boundary used to detect when
     * parts end and new begin
     */
    char *boundary;
    
    /* internal buffer and other variables
     * used while parsing
     */
    char buf[MULTIPART_BUF_SIZE + 2];
    int buf_contains_line;
    char *bufptr;
    int bufleft;    
    
    /* pointer that keeps track of a part while
     * it is being built
     */
    multipart_part *mpp;
    
    /* part parsing state; 0 means we are reading
     * headers, 1 means we are collecting data
     */
    int mpp_state;
    
    /* because of the way this parsing algorithm
     * works we hold back the last two bytes of
     * each data chunk so that we can discard it
     * later if the next data chunk proves to be
     * a boundary; the first byte is an indicator
     * 0 - no content, 1 - two data bytes available
     */
    char reserve[4];
} multipart_data;

typedef struct {
    request_rec *r;
    char *_the_request;
    char *_post_payload;
    char *_fake_post_payload;
    int should_body_exist;
    int is_body_read;
    int post_payload_dynamic_off;
    sec_dir_config *dcfg;
    sec_srv_config *scfg;
    table *parsed_args;
    table *parsed_cookies;
    char *tmp_message;
    char *tmp_redirect_url;
    int tmp_log_message;
    multipart_data *mpd;
} modsec_rec;

/* -------------- Function prototypes ---------------------------------------- */

static int check_single_signature(modsec_rec *msr, signature *sig);
static int check_sig_against_string(modsec_rec *msr, signature *_sig, const char *s, int var_type);
static int fd_lock(request_rec *r, int fd);
static int fd_unlock(request_rec *r, int fd);
static void sec_debug_log(request_rec *r, int level, const char *text, ...);

static char *normalise_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);
static char *normalise(request_rec *r, sec_dir_config *dcfg, char *_uri, char **error_msg);
static char *normalise_relaxed_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);
static char *normalise_relaxed(request_rec *r, sec_dir_config *dcfg, char *_uri, char **error_msg);
static char *normalise_other_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);
static char *normalise_urlencoding_relaxed_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);
static char *normalise_urlencoding_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg);

static char *remove_binary_content(request_rec *r, char *data);
static int parse_arguments(char *s, table *parsed_args, request_rec *r, sec_dir_config *dcfg, char **error_msg);
static char *unescape_regex_hex_inplace(char *input);
static int detect_unicode_character(request_rec *r, char *p_read);
static unsigned char x2c(unsigned char *what);
static request_rec *find_last_request(request_rec *r);

static int parse_cookies(request_rec *r, table *parsed_cookies, char **error_msg);
static const char *get_env_var(request_rec *r, char *name);
static const char *get_variable(request_rec *r, variable *v, table *parsed_args);
static char *parse_action(char *p2, actionset_t *actionset, pool *_pool);
static void sec_child_init(server_rec *s, pool *p);
static void sec_init(server_rec *s, pool *p);
static void sec_set_dir_defaults(sec_dir_config *dcfg);
static int is_filtering_on_here(request_rec *r, sec_dir_config *dcfg);
static char *get_temp_folder(void);

static int sec_initialize(modsec_rec *msr);
static int sec_check_access(request_rec *r);
static int sec_check_all_signatures(modsec_rec *msr);

static int multipart_init(multipart_data *mpd, request_rec *r);
static int multipart_finish(multipart_data *mpd);
static int multipart_process_chunk(multipart_data *mpd, char *buf, int size);
static char *multipart_construct_filename(multipart_data *mpd);
static int multipart_process_data_chunk(multipart_data *mpd);
static int multipart_process_boundary(multipart_data *mpd);
static int verify_uploaded_files(request_rec *r, multipart_data *mpd, char *approver_script, char **error_msg);
static int multipart_get_variables(multipart_data *mpd, table *parsed_args, sec_dir_config *dcfg, char **error_msg);
static int multipart_contains_files(multipart_data *mpd);
static int create_chroot_lock(server_rec *s, pool *p, char *lockfilename);
static int is_time_to_chroot(server_rec *s, pool *p);
static int change_server_signature(server_rec *s, sec_srv_config *scfg);

static int convert_charset_to_id(char *name);
static char *filter_multibyte_inplace(int cs_id, char replacement_byte, char *outbuf);
static char *filter_multibyte_other(int charset_id, char replacement_byte, char *inptr);
static char *filter_multibyte_unicode(int charset_id, char replacement_byte, char *inptr);

static void sec_sleep(int msec_pause);
static int sec_mkstemp(char *template);
static char *debuglog_escape(pool *p, char *text);
static char *real_debuglog_escape(pool *p, char *text);

static void *sec_create_srv_config(pool *p, server_rec *s);
static void *sec_merge_srv_config(pool *p, void *_parent, void *_child);
static void *sec_create_dir_config(pool *p, char *path);
static void *sec_merge_dir_config(struct pool *p, void *_parent, void *_child);

static int read_post_payload(modsec_rec *msr, char **p);
static int sec_exec_child(void *_ed, child_info *pinfo);
static int sec_logger(request_rec *_r);

static char *current_logtime(request_rec *r);
static char *current_filetime(request_rec *r);

static int perform_action(modsec_rec *msr, actionset_t *actionset);
static char *construct_fake_urlencoded(modsec_rec *msr, table *args);

/* ---------------------------------------------------------------------------- */

char *construct_fake_urlencoded(modsec_rec *msr, table *args) {
    table_entry *te;
    array_header *arr;
    int k;
    char *body;
    unsigned int body_len;
    
    if (args == NULL) return NULL;
    
    /* calculate buffer size */
    body_len = 1;
    arr = ap_table_elts(args);
    te = (table_entry *)arr->elts;
    for(k = 0; k < arr->nelts; k++) {
        body_len += 4;
        body_len += strlen(te[k].key);
        body_len += strlen(te[k].val);
    }
    
    /* allocate the buffer */
    body = ap_palloc(msr->r->pool, body_len + 1);
    if (body == NULL) return NULL;
    *body = 0;
    
    /* loop through remaining variables
     * and create a single string out of them
     */
    arr = ap_table_elts(args);
    te = (table_entry *)arr->elts;
    for(k = 0; k < arr->nelts; k++) {
        if (*body != 0) {
            strncat(body, "&", body_len - strlen(body));
        }
        strncat(body, te[k].key, body_len - strlen(body));
        strncat(body, "=", body_len - strlen(body));
        strncat(body, te[k].val, body_len - strlen(body));
    }
    
    return body;
}

int perform_action(modsec_rec *msr, actionset_t *actionset) {
    request_rec *r = msr->r;
    int log_level = 1;
    int rc = DECLINED;
    char *message = NULL;
    
    if (msr->tmp_message == NULL) {
        msr->tmp_message = "Unknown error";
    }
    
    if (actionset->log == 0) {
        ap_table_setn(msr->r->notes, NOTE_NOAUDITLOG, "noauditlog");
    
        /* log level 2 will log to mod_security log but not
         * to the Apache error log
         */
        log_level = 2;
    }

    /* perform action */
    switch(actionset->action) {
        
        case ACTION_ALLOW :
            message = ap_psprintf(r->pool, "Access allowed. %s", msr->tmp_message);
            rc = DECLINED;
            break;
        
        case ACTION_DENY :
            rc = actionset->status;
            message = ap_psprintf(r->pool, "Access denied with code %i. %s", rc, msr->tmp_message);
            break;
                
        case ACTION_REDIRECT :
            message = ap_psprintf(r->pool, "Access denied with redirect to [%s]. %s", actionset->redirect_url, msr->tmp_message);
            ap_table_setn(r->headers_out, "Location", actionset->redirect_url);
            /* rc = HTTP_MOVED_TEMPORARILY; */
            rc = REDIRECT;
            break;
                
        default :
            message = ap_psprintf(r->pool, "Warning. %s", msr->tmp_message);
            rc = DECLINED;
            break;
    }
    
    sec_debug_log(r, log_level, "%s", message);
    ap_table_setn(r->headers_in, NOTE_MESSAGE, message);
    
    if ((rc != DECLINED)&&(rc != MODSEC_SKIP)) {
        char *action = ap_psprintf(msr->r->pool, "%i", rc);
        ap_table_setn(r->headers_in, NOTE_ACTION, action);
    }
    
    /* execute the external script */
    if (actionset->exec) {
        exec_data *ed = NULL;
        BUFF *p1, *p2, *p3;
        char buf[4097];
            
        ed = ap_pcalloc(r->pool, sizeof(exec_data));
        ed->r = r;
        ed->command = actionset->exec_string;
        ed->args = NULL;
            
        sec_debug_log(r, 1, "Executing command \"%s\"", debuglog_escape(msr->r->pool, ed->command));
            
        ap_table_setn(r->headers_in, NOTE_EXECUTED, ed->command);

        if (!ap_bspawn_child(r->main ? r->main->pool : r->pool, sec_exec_child, (void *)ed, kill_after_timeout, &p1, &p2, &p3)) {
            ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r->server, "mod_security: couldn't spawn child process: %s", actionset->exec_string);
            return HTTP_INTERNAL_SERVER_ERROR;
        }
            
        while(ap_bgets(buf, 4096, p2) > 0) continue;
        while(ap_bgets(buf, 4096, p3) > 0) continue;
    }
        
    if (actionset->pause != 0) {
        /* TODO add uri information to the message, it won't have enough
         * information when it is in the error log
         */
        sec_debug_log(r, log_level, "Pausing for %i ms", actionset->pause);
        sec_sleep(actionset->pause);
    }

    msr->tmp_message = NULL;
    return rc;
}

int sec_mkstemp(char *template) {

    #if !(defined(WIN32)||defined(NETWARE))
    return mkstemp(template);
    #else
    
    if (mktemp(template) == NULL) return -1;
    return open(template, O_WRONLY | O_APPEND | O_CREAT | O_BINARY, CREATEMODE);
    
    #endif
}

#ifdef WIN32

char *strtok_r(char *s, const char *sep, char **lasts) {
    char *sbegin, *send;
    
    /* These two parameters must always be valid */
    if ((sep == NULL)||(lasts == NULL)) return NULL;
    
    /* Either of the following two must not be NULL */
    if ((s == NULL)&&(*lasts == NULL)) return NULL;
    
    sbegin = s ? s : *lasts;
    
    /* Advance through the separator at the beginning */
    sbegin += strspn(sbegin, sep);
    if (*sbegin == '\0') {
        *lasts = NULL;
        return NULL;
    }

    /* Find the next separator */    
    send = strpbrk(sbegin, sep);
    if (send != NULL) *send++ = 0;
    *lasts = send;
    
    return sbegin;
}

#endif

void sec_sleep(int msec_pause) {
    if (msec_pause <= 0) return;
    
    #ifdef WIN32
    _sleep(msec_pause);
    #elif defined NETWARE
    delay(msec_pause);
    #else
    usleep(msec_pause * 1000);
    #endif
}

/*
 * Change the signature of the server if change
 * was requested in configuration. We do this by
 * locating the signature in server memory and
 * writing over it.
 */
int change_server_signature(server_rec *s, sec_srv_config *scfg) {
    char *server_version;
    
    if (scfg->server_signature == NULL) return 0;
    
    server_version = (char *)ap_get_server_version();
    if (server_version == NULL) {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, s, "SecServerSignature: ap_get_server_version returned NULL");
        return -1;
    }
    
    if (strlen(server_version) >= strlen(scfg->server_signature)) {
        strcpy(server_version, scfg->server_signature);
        return 1;
    }
    else {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, s, "SecServerSignature: not enough space to copy new signature");
        return -1;
    }
}

int convert_charset_to_id(char *name) {
    if (strcasecmp(name, "utf-8") == 0) return UNI3_CSID;
    if (strcasecmp(name, "shift-jis") == 0) return SJIS1_CSID;
    if (strcasecmp(name, "shift_jis") == 0) return SJIS2_CSID;
    if (strcasecmp(name, "big5") == 0) return BIG5_CSID;
    if (strcasecmp(name, "gbk") == 0) return GBK_CSID;
    if (strcasecmp(name, "gb2312") == 0) return GB2312_CSID;
    if (strcasecmp(name, "euc-tw") == 0) return ZHT32EUC_CSID;
    if (strcasecmp(name, "euc-jp") == 0) return JEUC1_CSID;
    if (strcasecmp(name, "eucjis") == 0) return JEUC2_CSID;
    if (strcasecmp(name, "jis0208") == 0) return JA16VMS_CSID;
    return -1;
}

char *filter_multibyte_unicode(int charset_id, char replacement_byte, char *inptr) {
    char *outptr = inptr;
    int i, j, k, n;
    
    i = strlen(inptr);
    j = 0;
    
    /* Unicode */
    while(j < i) {
        k = inptr[j] & 0xFF;
        if (k < 0x80) {
            j++;
            *outptr++ = (char)k;
        }
        else if (k < 0xC0) {
            j++;
            *outptr++ = replacement_byte;
        }
        else {
            if (k < 0xE0) n = 2;
            else if (k < 0xF0) n = 3;
            else if (k < 0xF8) n = 4;
            else if (k < 0xFC) n = 5;
            else if (k < 0xFE) n = 6;
            else n = 1;

            if (i - j >= n) {
                j += n;
            }
            else {
                i = j;
            }

            *outptr++ = replacement_byte;
        }
    }
    
    *outptr = 0;
    return inptr;
}

char *filter_multibyte_other(int charset_id, char replacement_byte, char *inptr) {
    char *outptr = inptr;
    int i, j, k, n;
    
    i = strlen(inptr);
    j = 0;
    
    while(j < i) {
        k = inptr[j] & 0xFF;
        if (k < 0x80) {
            j++;
            *outptr++ = (char)k;
        }
        else {
            n = 2;
                
            if ((k == 0x8E)&&(charset_id == ZHT32EUC_CSID)) {
                n = 4;
            }
            else if ((k == 0x8F)&&((charset_id == JEUC1_CSID)||(charset_id == JEUC2_CSID))) {
                n = 3;
            }
            else if ( ((k == 0x80)||(k == 0xFF))
                    && ((charset_id == BIG5_CSID)||(charset_id == GBK_CSID)||(charset_id == GB2312_CSID)) ) {
                n = 1;
            }
            else if ( ((k == 0x80)||((k >= 0xA0) && (k < 0xE0)))
                    && ((charset_id == SJIS1_CSID)||(charset_id == SJIS2_CSID)) ) {
                n = 1;
            }

            if (i - j >= n) {
                j += n;
            }
            else {
                i = j;
            }

            *outptr++ = (n == 1) ? (char)k : replacement_byte;
        }
    }
    
    *outptr = 0;
    return inptr;
}

char *filter_multibyte_inplace(int charset_id, char replacement_byte, char *inptr) {
    if (charset_id < MB_CSID) return inptr; /* not multibyte charset */
    if (charset_id == UNI3_CSID) return filter_multibyte_unicode(charset_id, replacement_byte, inptr);
    else return filter_multibyte_other(charset_id, replacement_byte, inptr);
}

int parse_cookies(request_rec *r, table *parsed_cookies, char **error_msg) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    char *header, *header_copy, *attr_name, *attr_value, *p;
    char *prev_attr_name = NULL;
    int cookie_count = 0;
    
    if (error_msg == NULL) return -1;
    *error_msg = NULL;
    
    header = (char *)ap_table_get(r->headers_in, "Cookie");
    /* Return here if no cookies found */
    if (header == NULL) return 0;
    
    header_copy = ap_pstrdup(r->pool, header);
    if (header_copy == NULL) return -1;
    
    sec_debug_log(r, 6, "Raw cookie header: [%s]", debuglog_escape(r->pool, header));
    
    p = header_copy;
    while(*p != 0) {
        attr_name = NULL;
        attr_value = NULL;
    
        /* attribute name */
        while(isspace(*p)) p++;
        attr_name = p;
        while((*p != 0)&&(*p != '=')) p++;
     
        if (*p == 0) {   
            /* we've reached the end of the string,
             * and the attribute value is missing
             */
            *error_msg = ap_psprintf(r->pool, "Cookie value is missing #1");
            return -1;
        }

        /* terminate the attribute name,
         * writing over the = character
         */
        *p++ = 0;
        
        /* attribute value */
        while((isspace(*p))&&(*p != 0)) p++;
        if (*p == 0) {
            *error_msg = ap_psprintf(r->pool, "Cookie value is missing #2");
            return -1;
        }
        
        if (*p == '"') {
            char *s, *d;
            
            /* quoted string */
            if (*++p == 0) {
                /* invalid formating */
                *error_msg = ap_psprintf(r->pool, "Invalid formatting for cookie value, missing data");
                return -1;
            }
            
            attr_value = p;
            
            for(;;) {
                while((*p != 0)&&(*p != '"')) p++;
                if (*p == 0) break;
                
                /* we allow quotation marks to be escaped
                 * using two methods: double quotation mark
                 * and escaping with a slash
                 */
                 
                if (*(p - 1) == '\\') {
                    /* this quotation mark was escaped */
                    p++;
                    continue;
                }
                
                if (*(p + 1) == '"') {
                    /* the next character is also the quotation mark */
                    p += 2;
                    continue;
                }
                
                /* we've reached the end of the string */
                break;
            }
            
            if (*p == 0) {
                /* we are missing the quotation
                 * mark at the end of the string
                 */
                *error_msg = ap_psprintf(r->pool, "Invalid formatting, missing quotation mark at the end");
                return -1;
            }
            
            /* terminate the attribute value
             * writing over the the quotation mark
             */
            *p++ = 0;
            
            while(isspace(*p)) p++;
            if ((*p != ',')&&(*p != ';')&&(*p != 0)) {
                *error_msg = ap_psprintf(r->pool, "Invalid formatting, expected , or ; as boundary character [got %i]", *p);
                return -1;
            }

            if (*p != 0) {            
                /* move the pointer to the first character
                 * of the next attribute
                 */
                p++;
            }

            /* decode escaped quotation marks */            
            s = d = attr_value;
            while(*s != 0) {
                if ((*s == '"')&&(*(s + 1) == '"')) {
                    *d++ = '"';
                    s += 2;
                    continue;
                }
                
                if ((*s == '\\')&&(*(s + 1) == '"')) {
                    *d++ = '"';
                    s += 2;
                    continue;
                }
                
                *d++ = *s++;
            }
            *d = 0;
            
        } else {
            /* non-quoted string */
            attr_value = p;
            while((*p != 0)&&(*p != ',')&&(*p != ';')) p++;
            
            /* we only increase the pointer if we haven't
             * reached the end of the string, to allow the
             * loop to continue
             */
            if (*p != 0) {
                *p++ = 0;
            }
        }
        
        /* use the attr_name & attr_value */
        if (dcfg->normalize_cookies) {
            char *my_error_msg = NULL;
            
            if (normalise_inplace(r, dcfg, attr_name, &my_error_msg) == NULL) {
                *error_msg = ap_psprintf(r->pool, "Error normalizing cookie name: %s", my_error_msg);
                return -1;   
            }
            if (normalise_inplace(r, dcfg, attr_value, &my_error_msg) == NULL) {
                *error_msg = ap_psprintf(r->pool, "Error normalizing cookie value: %s", my_error_msg);
                return -1;   
            }
        }
        
        if (attr_name[0] == '$') {
            if (strlen(attr_name) == 1) {
                *error_msg = ap_psprintf(r->pool, "Invalid formatting, cookie keyword name empty");
                return -1;
            }
        
            if (prev_attr_name != NULL) {
                /* cookie keyword, we change the name we use
                 * so they can have a unique name in the cookie table
                 */
                attr_name = ap_psprintf(r->pool, "$%s_%s", prev_attr_name, attr_name + 1);
            }
        }
        
        if (strlen(attr_name) == 0) {
            *error_msg = ap_psprintf(r->pool, "Invalid formatting, cookie name empty");
            return -1;
        }
        
        sec_debug_log(r, 4, "Adding cookie: [%s][%s]", debuglog_escape(r->pool, attr_name), debuglog_escape(r->pool, attr_value));
        ap_table_add(parsed_cookies, attr_name, attr_value);
        
        cookie_count++;
        
        /* only keep the cookie names for later */
        if (attr_name[0] != '$') prev_attr_name = attr_name;
    }    
    
    return cookie_count;
}

const char *get_env_var(request_rec *r, char *name) {
    const char *result = ap_table_get(r->notes, name);

    if (result == NULL) {
        result = ap_table_get(r->subprocess_env, name);
    }

    if (result == NULL) {
        result = getenv(name);
    }
    
    return result;
}

const char *get_variable(request_rec *r, variable *v, table *parsed_args) {
    sec_dir_config *dcfg_proper = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    sec_dir_config *dcfg = (sec_dir_config *)ap_pcalloc(r->pool, sizeof(sec_dir_config));
    char *my_error_msg = NULL;
    const char *result = NULL;
    struct tm *tm;
    time_t tc;
    
    memcpy(dcfg, dcfg_proper, sizeof(sec_dir_config));
    dcfg->check_encoding = 0;
    dcfg->check_unicode_encoding = 0;
    dcfg->check_cookie_format = 0;
    dcfg->range_start = 0;
    dcfg->range_end = 255;
    
    switch (v->type) {

        case VAR_CUSTOM:
            if (parsed_args != NULL) {
                /* we don't normalise parameter values becaue
                 * they are stored normalised
                 */
                result = ap_table_get(parsed_args, v->name);
            }
            else {
                sec_debug_log(r, 1, "get_variable: VAR_CUSTOM requested but parsed_args = NULL");
            }
            break;

        case VAR_HEADER:
            result = ap_table_get(r->headers_in, v->name);
            if (result != NULL) result = normalise_relaxed(r, dcfg, (char *)result, &my_error_msg);
            break;

        case VAR_ENV:
            result = ap_table_get(r->notes, v->name);

            if (result == NULL) {
                result = ap_table_get(r->subprocess_env, v->name);
            }

            if (result == NULL) {
                result = getenv(v->name);
            }
            break;

        case VAR_ARGS:
            /* this variable type should have been resolved
             * without calling this function
             */
            sec_debug_log(r, 1, "get_variable: internal error, VAR_ARGS should not be requested from this function");
            break;

        case VAR_REMOTE_ADDR:
            result = r->connection->remote_ip;
            break;

        case VAR_REMOTE_HOST:
            result = ap_get_remote_host(r->connection, r->per_dir_config, REMOTE_NAME);
            break;

        case VAR_REMOTE_USER:
            result = r->connection->user;
            break;

        case VAR_REMOTE_IDENT:
            result = ap_get_remote_logname(r);
            break;

        case VAR_REQUEST_METHOD:
            result = r->method;
            break;

        case VAR_REQUEST_URI:
            result = r->unparsed_uri;
            if (result != NULL) result = normalise(r, dcfg, (char *)result, &my_error_msg);
            break;

        case VAR_AUTH_TYPE:
            result = r->connection->ap_auth_type;
            break;

        case VAR_IS_SUBREQ:
            result = (r->main != NULL ? "true" : "false");
            break;

        case VAR_DOCUMENT_ROOT:
            result = ap_document_root(r);
            break;

        case VAR_SERVER_ADMIN:
            result = r->server->server_admin;
            break;

        case VAR_SERVER_NAME:
            result = ap_get_server_name(r);
            break;

        case VAR_SERVER_ADDR:
            result = r->connection->local_ip;
            break;

        case VAR_SERVER_PORT:
            result = ap_psprintf(r->pool, "%i", (int)ap_get_server_port(r));
            break;

        case VAR_SERVER_PROTOCOL:
            result = r->protocol;
            break;

        case VAR_SERVER_SOFTWARE:
            result = ap_get_server_version();
            break;

        case VAR_API_VERSION:
            result = ap_psprintf(r->pool, "%d:%d", MODULE_MAGIC_NUMBER_MAJOR, MODULE_MAGIC_NUMBER_MINOR);
            break;

        case VAR_TIME_YEAR:
            tc = time(NULL);
            tm = localtime(&tc);
            result = ap_psprintf(r->pool, "%02d%02d", (tm->tm_year / 100) + 19, tm->tm_year % 100);
            break;

        case VAR_TIME:
            tc = time(NULL);
            tm = localtime(&tc);
            result = ap_psprintf(r->pool, "%02d%02d%02d%02d%02d%02d%02d",
                     (tm->tm_year / 100) + 19, (tm->tm_year % 100),
                     tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min,
                     tm->tm_sec);
            break;

        case VAR_TIME_WDAY:
            tc = time(NULL);
            tm = localtime(&tc);
            result = ap_psprintf(r->pool, "%d", tm->tm_wday);
            break;

        case VAR_TIME_SEC:
            tc = time(NULL);
            tm = localtime(&tc);
            result = ap_psprintf(r->pool, "%02d", tm->tm_sec);
            break;

        case VAR_TIME_MIN:
            tc = time(NULL);       
            tm = localtime(&tc);
            result = ap_psprintf(r->pool, "%02d", tm->tm_min);
            break;

        case VAR_TIME_HOUR:
            tc = time(NULL);
            tm = localtime(&tc);
            result = ap_psprintf(r->pool, "%02d", tm->tm_hour);
            break;

        case VAR_TIME_MON:
            tc = time(NULL);
            tm = localtime(&tc);
            result = ap_psprintf(r->pool, "%02d", tm->tm_mon + 1);
            break;

        case VAR_TIME_DAY:
            tc = time(NULL);
            tm = localtime(&tc);
            result = ap_psprintf(r->pool, "%02d", tm->tm_mday);
            break;

        case VAR_SCRIPT_FILENAME:
        case VAR_REQUEST_FILENAME:
            result = r->filename;
            break;

        case VAR_PATH_INFO:
            result = r->path_info;
            if (result != NULL) result = normalise(r, dcfg, (char *)result, &my_error_msg);
            break;

        case VAR_THE_REQUEST:
            result = r->the_request;
            if (result != NULL) result = normalise(r, dcfg, (char *)result, &my_error_msg);
            break;

        case VAR_QUERY_STRING:
            result = r->args;
            if (result != NULL) result = normalise(r, dcfg, (char *)result, &my_error_msg);
            break;
            
        case VAR_HANDLER :
            result = r->handler;
            break;
            
        case VAR_COOKIE:
            /* cookies were escaped earlier */
            if (parsed_args == NULL) {
                sec_debug_log(r, 1, "get_variable: VAR_COOKIE requested but parsed_args is NULL");
            }
            else {
                result = ap_table_get(parsed_args, v->name);
            }
            break;
    }

    if (result == NULL) {
        result = "";
    }

    return result;
}

request_rec *find_last_request(request_rec *r) {
    request_rec *rlast = r;
    sec_debug_log(r, 9, "find_last_request: start with %x [%s]", rlast, debuglog_escape(r->pool, rlast->uri));
    while (rlast->next != NULL) {
        rlast = rlast->next;
        sec_debug_log(r, 9, "find_last_request: found next %x [%s]", rlast, debuglog_escape(r->pool, rlast->uri));
    }
    return rlast;
}

void *sec_create_srv_config(pool *p, server_rec *s) {
    sec_srv_config *scfg = (sec_srv_config *)ap_pcalloc(p, sizeof(sec_srv_config));
    /* fprintf(stderr, "sec_create_srv_config: %s\n", s->server_hostname); */
    if (scfg == NULL) return NULL;
    
    scfg->server_response_token = 0;
    scfg->server_signature = NULL;
    scfg->chroot_dir = NULL;
    scfg->chroot_completed = 0;
    scfg->chroot_lock = ap_server_root_relative(p, "logs/modsec_chroot.lock");

    return scfg;
}

void *sec_merge_srv_config(pool *p, void *_parent, void *_child) {
    sec_srv_config *parent = (sec_srv_config *)_parent;
    sec_srv_config *new = (sec_srv_config *)ap_pcalloc(p, sizeof(sec_srv_config));
    if (new == NULL) return NULL;
    
    /* fprintf(stderr, "sec_merge_srv_config\n"); */
    new->server_signature = parent->server_signature;
    return new;
}

void sec_set_dir_defaults(sec_dir_config *dcfg) {
    if (dcfg == NULL) return;
    
    /* return immediatelly if we've already been here */
    if (dcfg->configuration_helper == 1) return;
    
    dcfg->configuration_helper = 1;
    if (dcfg->filter_engine == NOT_SET) dcfg->filter_engine = 0;
    
    if (dcfg->scan_post == NOT_SET) dcfg->scan_post = 0;
    if (dcfg->auditlog_flag == NOT_SET) dcfg->auditlog_flag = 0;
    if (dcfg->filter_debug_level == NOT_SET) dcfg->filter_debug_level = 0;
    if (dcfg->filters_clear == NOT_SET) dcfg->filters_clear = 0;
    if (dcfg->action == NOT_SET_P) {
        dcfg->action = (actionset_t *)ap_pcalloc(dcfg->p, sizeof(actionset_t));
        dcfg->action->log = 1;
        dcfg->action->action = ACTION_DENY;
        dcfg->action->status = HTTP_FORBIDDEN;
        dcfg->action->skip_count = 1;
    }
    
    if (dcfg->auditlog_name == NOT_SET_P) dcfg->auditlog_name = NULL;
    if (dcfg->debuglog_name == NOT_SET_P) dcfg->debuglog_name = NULL;
    
    if (dcfg->range_start == NOT_SET) dcfg->range_start = 0;
    if (dcfg->range_end == NOT_SET) dcfg->range_end = 255;
    if (dcfg->check_encoding == NOT_SET) dcfg->check_encoding = 0;
    if (dcfg->check_unicode_encoding == NOT_SET) dcfg->check_unicode_encoding = 0;
    if (dcfg->upload_dir == NOT_SET_P) dcfg->upload_dir = NULL;
    if (dcfg->upload_keep_files == NOT_SET) dcfg->upload_keep_files = 0;
    if (dcfg->upload_approve_script == NOT_SET_P) dcfg->upload_approve_script = NULL;
    
    if (dcfg->normalize_cookies == NOT_SET) dcfg->normalize_cookies = 1;
    if (dcfg->check_cookie_format == NOT_SET) dcfg->check_cookie_format = 0;
    
    if (dcfg->charset_id == NOT_SET) dcfg->charset_id = UNKNOWN_CSID;
    if (dcfg->multibyte_replacement_byte == NOT_SET) dcfg->multibyte_replacement_byte = 0x0A;
}

void *sec_create_dir_config(pool *p, char *path) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_pcalloc(p, sizeof(*dcfg));
    if (dcfg == NULL) return NULL;
    
    /* fprintf(stderr, "sec_create_dir_config: %s\n", path); */
    
    dcfg->p = p;
    
    dcfg->configuration_helper = NOT_SET;
    dcfg->filter_engine = NOT_SET;
    dcfg->scan_post = NOT_SET;
    dcfg->action = NOT_SET_P;

    dcfg->signatures = ap_make_array(p, 10, sizeof(signature *));

    if (path == NULL) {
        dcfg->path = ap_pstrdup(p, "(null)");
    }
    else {
        dcfg->path = ap_pstrdup(p, path);
    }

    dcfg->auditlog_flag = NOT_SET;
    dcfg->auditlog_name = NOT_SET_P;
    dcfg->auditlog_fd = NOT_SET;
    
    dcfg->filter_debug_level = NOT_SET;
    dcfg->filters_clear = NOT_SET;
    dcfg->debuglog_name = NOT_SET_P;
    dcfg->debuglog_fd = NOT_SET;
    
    dcfg->range_start = NOT_SET;
    dcfg->range_end = NOT_SET;
    dcfg->check_encoding = NOT_SET;
    dcfg->check_unicode_encoding = NOT_SET;
    
    dcfg->upload_dir = NOT_SET_P;
    dcfg->upload_keep_files = NOT_SET;
    dcfg->upload_approve_script = NOT_SET_P;
    
    dcfg->normalize_cookies = NOT_SET;
    dcfg->check_cookie_format = NOT_SET;
    
    dcfg->charset_id = NOT_SET;
    dcfg->multibyte_replacement_byte = NOT_SET;

    return dcfg;
}

void *sec_merge_dir_config(struct pool *p, void *_parent, void *_child) {
    sec_dir_config *parent = (sec_dir_config *)_parent;
    sec_dir_config *child = (sec_dir_config *)_child;
    
    /* merge child & parent into new */
    
    sec_dir_config *new = (sec_dir_config *)ap_pcalloc(p, sizeof(*new));
    if (new == NULL) return NULL;
    
    /* fprintf(stderr, "sec_merge_dir_config: parent=%s, child=%s\n", parent->path, child->path); */
    
    memcpy(new, child, sizeof(*child));

    new->filter_engine = (child->filter_engine == NOT_SET) ? parent->filter_engine : child->filter_engine;
    new->scan_post = (child->scan_post == NOT_SET) ? parent->scan_post : child->scan_post;
    new->action = (child->action == NOT_SET_P) ? parent->action : child->action;

    /* filters_clear is a special case, the value is not inherited from the
     * parent, they work only where explicitely used
     */
    new->filters_clear = child->filters_clear;
    
    new->signatures = ap_copy_array(p, child->signatures);
    
    /* we copy signatures from the parent only if not told not to in
     * the child configuration. new->filters_clear may be NOT_SET here
     * (defaults are configured on the fly, but this is one decision
     * we must make here.
     */
    if (new->filters_clear != 1) ap_array_cat(new->signatures, parent->signatures);
    
    new->auditlog_flag = (child->auditlog_flag == NOT_SET) ? parent->auditlog_flag : child->auditlog_flag;
    
    if (child->auditlog_fd == NOT_SET) {
        new->auditlog_fd = parent->auditlog_fd;
        new->auditlog_name = parent->auditlog_name;
    }
    else {
        new->auditlog_fd = child->auditlog_fd;
        new->auditlog_name = child->auditlog_name;
    }
    
    new->filter_debug_level = (child->filter_debug_level == NOT_SET) ? parent->filter_debug_level : child->filter_debug_level;
    
    if (child->debuglog_fd == NOT_SET) {
        new->debuglog_fd = parent->debuglog_fd;
        new->debuglog_name = parent->debuglog_name;
    }
    else {
        new->debuglog_fd = child->debuglog_fd;
        new->debuglog_name = child->debuglog_name;
    }
    
    new->range_start = (child->range_start == NOT_SET) ? parent->range_start : child->range_start;
    new->range_end = (child->range_end == NOT_SET) ? parent->range_end : child->range_end;
    new->check_encoding = (child->check_encoding == NOT_SET) ? parent->check_encoding : child->check_encoding;    
    new->check_unicode_encoding = (child->check_unicode_encoding == NOT_SET) ? parent->check_unicode_encoding : child->check_unicode_encoding;
    new->upload_dir = (child->upload_dir == NOT_SET_P) ? parent->upload_dir : child->upload_dir;
    new->upload_keep_files = (child->upload_keep_files == NOT_SET) ? parent->upload_keep_files : child->upload_keep_files;
    new->upload_approve_script = (child->upload_approve_script == NOT_SET_P) ? parent->upload_approve_script : child->upload_approve_script;
    
    new->normalize_cookies = (child->normalize_cookies == NOT_SET) ? parent->normalize_cookies : child->normalize_cookies;
    new->check_cookie_format = (child->check_cookie_format == NOT_SET) ? parent->check_cookie_format : child->check_cookie_format;
    new->charset_id = (child->charset_id == NOT_SET) ? parent->charset_id : child->charset_id;
    new->multibyte_replacement_byte = (child->multibyte_replacement_byte == NOT_SET) ? parent->multibyte_replacement_byte : child->multibyte_replacement_byte;
    
    return new;
}

unsigned char x2c(unsigned char *what) {
    register unsigned char digit;

    digit = (what[0] >= 'A' ? ((what[0] & 0xdf) - 'A') + 10 : (what[0] - '0'));
    digit *= 16;
    digit += (what[1] >= 'A' ? ((what[1] & 0xdf) - 'A') + 10 : (what[1] - '0'));
    
    return(digit);
}

int detect_unicode_character(request_rec *r, char *p_read) {
    int unicode_len = 0;
    unsigned int d = 0;
    unsigned char c;
    
    if (p_read == NULL) return 0;
    
    c = *p_read;
    if (c == 0) return 0;
    
    if ((c & 0xE0) == 0xC0) {
        /* two byte unicode */
        if (*(p_read + 1) == 0) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            unicode_len = 2;
            d = ((c & 0x1F) << 6) | (*(p_read + 1) & 0x3F);
        }
    }
    else if ((c & 0xF0) == 0xE0) {
        /* three byte unicode */
        if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            unicode_len = 3;
            d = ((c & 0x0F) << 12) | ((*(p_read + 1) & 0x3F) << 6) | (*(p_read + 2) & 0x3F);
        }
    }
    else if ((c & 0xF8) == 0xF0) {
        /* four byte unicode */
        if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            d = ((c & 0x07) << 18) | ((*(p_read + 1) & 0x3F) << 12) | ((*(p_read + 2) & 0x3F) < 6) | (*(p_read + 3) & 0x3F);
            unicode_len = 4;
        }
    }
    else if ((c & 0xFC) == 0xF8) {
        /* five byte unicode */
        if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)||(*(p_read + 4) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 4)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            d = ((c & 0x03) << 24) | ((*(p_read + 1) & 0x3F) << 18) | ((*(p_read + 2) & 0x3F) << 12) | ((*(p_read + 3) & 0x3F) << 6) | (*(p_read + 4) & 0x3F);
            unicode_len = 5;
        }
    }
    else if ((c & 0xFE) == 0xFC) {
        /* six byte unicode */
        if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)||(*(p_read + 4) == 0)||(*(p_read + 5) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
        else
        if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 4)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else
        if (((*(p_read + 5)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
        else {
            d = ((c & 0x01) << 30) | ((*(p_read + 1) & 0x3F) << 24) | ((*(p_read + 2) & 0x3F) << 18) | ((*(p_read + 3) & 0x3F) << 12) | ((*(p_read + 4) & 0x3F) << 6) | (*(p_read + 5) & 0x3F);
            unicode_len = 6;
        }
    }
        
    if ((unicode_len > 1)&&((d & 0x7F) == d)) {
        unicode_len = UNICODE_ERROR_OVERLONG_CHARACTER;
    }
        
    return(unicode_len);
}

char *normalise_urlencoding_relaxed_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    unsigned char *p_read, *p_write;
    unsigned char c;
    
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;
    
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    
    while ((c = *p_read) != 0) {
        
        if (c == '%') {
        
            /* see if there are enough bytes available */
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) {
                c = 0;
            }
            else {
                /* here we only decode a %xx combo if it is a valid
                 * encoding, we leave it as is otherwise
                 */
                char c1 = *(p_read + 1), c2 = *(p_read + 2);
                
                if ( (((c1 >= '0')&&(c1 <= '9')) || ((c1 >= 'a')&&(c1 <= 'f')) || ((c1 >= 'A')&&(c1 <= 'F'))) 
                    && (((c2 >= '0')&&(c2 <= '9')) || ((c2 >= 'a')&&(c2 <= 'f')) || ((c2 >= 'A')&&(c2 <= 'F'))) ) {
                    
                    c = x2c(++p_read);
                    p_read++;
                }
            }
        } else {
            /* this check is performed only against the original data
             * and not against the decoded values (we want to
             * avoid false positives)
             */
            if ((c < dcfg->range_start)||(c > dcfg->range_end)) {
                *error_msg = ap_psprintf(r->pool, "Invalid character detected [%i]", c);
                return NULL;
            }
        }

        /* replace null bytes with whitespace */        
        if (c == 0) c = 32;
        
        *p_write++ = c;
        p_read++;
    }
    *p_write = 0;
    
    return uri;
}

char *normalise_urlencoding_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    unsigned char *p_read, *p_write;
    unsigned char c;
    
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;
    
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    
    while ((c = *p_read) != 0) {
        
        /* decode URL decoding */
        if (c == '+') c = 32;
        else
        if (c == '%') {
        
            /* see if there are enough bytes available */
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) {
                if (dcfg->check_encoding) {
                    *error_msg = ap_psprintf(r->pool, "Invalid URL encoding detected: not enough characters");
                    return NULL;
                }
                else c = 0;
            }
            else {
        
                /* move onto the first hexadecimal letter */
                p_read++;
                
                c = *p_read;
                
                if (dcfg->check_encoding) {
                    if (! (((c >= '0')&&(c <= '9')) || ((c >= 'a')&&(c <= 'f')) || ((c >= 'A')&&(c <= 'F'))) ) {
                        *error_msg = ap_psprintf(r->pool, "Invalid URL encoding detected: invalid characters used");
                        return NULL;
                    }
                }
                
                c = *(p_read + 1);
                
                if (dcfg->check_encoding) {
                    if (! (((c >= '0')&&(c <= '9')) || ((c >= 'a')&&(c <= 'f')) || ((c >= 'A')&&(c <= 'F'))) ) {
                        *error_msg = ap_psprintf(r->pool, "Invalid URL encoding detected: invalid characters used");
                        return NULL;
                    }
                }
                
                /* decode two hexadecimal letters into a single byte */
                c = x2c(p_read);
                p_read++;
            }
        }

        if ((c < dcfg->range_start)||(c > dcfg->range_end)) {
            *error_msg = ap_psprintf(r->pool, "Invalid character detected [%i]", c);
            return NULL;
        }

        /* we replace null bytes with whitespace */        
        if (c == 0) c = 32;
        *p_write++ = c;
        p_read++;
    }
    *p_write = 0;
    
    return uri;
}

char *normalise_other_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    unsigned char *p_read, *p_write, *p_slash;
    unsigned char c;
    int count;
    
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;
    
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    p_slash = NULL;
    count = 0;
    while (*p_read != 0) {
        c = *p_read;

        if (dcfg->check_unicode_encoding) {
            int urc = detect_unicode_character(r, (char *)p_read);
            
            switch(urc) {
                case UNICODE_ERROR_CHARACTERS_MISSING :
                    *error_msg = ap_psprintf(r->pool, "Invalid Unicode encoding: not enough bytes");
                    return NULL;
                    break;
                case UNICODE_ERROR_INVALID_ENCODING :
                    *error_msg = ap_psprintf(r->pool, "Invalid Unicode encoding: invalid byte value");
                    return NULL;
                    break;
                case UNICODE_ERROR_OVERLONG_CHARACTER :
                    *error_msg = ap_psprintf(r->pool, "Invalid Unicode encoding: overlong character");
                    return NULL;
                    break;
            }
            
        }
        
        switch (c) {
        
            #ifdef WIN32
            case '\\' :
            #endif
        
            case '/' :
                if (p_slash == NULL) {
    
                    /* remove the occurencies of "./" */
                    if ( (count > 1) && ((*(p_write - 1) == '.') && (*(p_write - 2) == '/')) ) {
                        count -= 2;
                        p_write -= 2;
                    }
                    
                    p_slash = p_read;
                    *p_write++ = '/';
                    p_read++;
                    count++;
                }
                else {
                    /* the previous character was a slash, we
                     * will ignore this one - just increment
                     * the read pointer
                     */
                    p_read++;
                }
                break;

            default:
                /* p_slash is used to detect more than one
                 * slash character in a row
                 */
                p_slash = NULL;
                *p_write++ = c;
                p_read++;
                count++;
                
                break;
            }
    }
    *p_write = 0;
    
    return uri;
}

char *normalise_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;

    if (normalise_urlencoding_inplace(r, dcfg, uri, error_msg) == NULL) {
        /* error_msg already populated */
        return NULL;
    }
    
    if (normalise_other_inplace(r, dcfg, uri, error_msg) == NULL) {
        /* error_msg already populated */
        return NULL;
    }    
    
    return filter_multibyte_inplace(dcfg->charset_id, (char)dcfg->multibyte_replacement_byte, uri);
}

char *normalise_relaxed_inplace(request_rec *r, sec_dir_config *dcfg, char *uri, char **error_msg) {
    if (error_msg == NULL) return NULL;
    *error_msg = NULL;
    if (uri == NULL) return NULL;

    if (normalise_urlencoding_relaxed_inplace(r, dcfg, uri, error_msg) == NULL) {
        /* error_msg already populated */
        return NULL;
    }
    
    if (normalise_other_inplace(r, dcfg, uri, error_msg) == NULL) {
        /* error_msg already populated */
        return NULL;
    }    
    
    return filter_multibyte_inplace(dcfg->charset_id, (char)dcfg->multibyte_replacement_byte, uri);
}

char *normalise_relaxed(request_rec *r, sec_dir_config *dcfg, char *_uri, char **error_msg) {
    char *uri;
    
    if (_uri == NULL) return NULL;
    uri = ap_pstrdup(r->pool, _uri);
    if (uri == NULL) return NULL;
    
    return normalise_relaxed_inplace(r, dcfg, uri, error_msg);
}

char *normalise(request_rec *r, sec_dir_config *dcfg, char *_uri, char **error_msg) {
    char *uri;
    
    if (_uri == NULL) return NULL;
    uri = ap_pstrdup(r->pool, _uri);
    if (uri == NULL) return NULL;
    
    return normalise_inplace(r, dcfg, uri, error_msg);
}

int is_filtering_on_here(request_rec *r, sec_dir_config *dcfg) {
    
    /* refuse to work if the per dir config is null */
    if (dcfg == NULL) {
        sec_debug_log(r, 2, "Filtering off, dcfg is null");
        return 0;
    }
    
    /* refuse to work if not initialised properly */
    if (dcfg->filter_engine == NOT_SET) {
        return 0;
    }
    
    /* refuse to work if this is a subrequest */
    if (!ap_is_initial_req(r)) {
        char *s = NULL;
        if ( ((r->main)&&((s = (char * )ap_table_get(r->main->notes, NOTE_DYNAMIC)) != NULL))
            || ((r->prev)&&((s = (char * )ap_table_get(r->prev->notes, NOTE_DYNAMIC)) != NULL)) ) {
            sec_debug_log(r, 2, "Looking into subrequest because initial request skipped because of DynamicOnly");
        }
        else {
            sec_debug_log(r, 2, "Filtering off for a subrequest");
            return 0;
        }
    }
    
    /* refuse to work if filtering is off */
    if (dcfg->filter_engine == FILTERING_OFF) {
        sec_debug_log(r, 2, "Filtering off, switched off for path [%s]", dcfg->path);
        return 0;
    }
    
    if ((dcfg->filter_engine == FILTERING_DYNAMIC_ONLY)&&(S_ISDIR(r->finfo.st_mode))) {
        sec_debug_log(r, 2, "DynamicOnly setting does not work for folder requests in 1.8.x");
        return 1;
    }
    
    if ((dcfg->filter_engine == FILTERING_DYNAMIC_ONLY)&&(r->handler == NULL)) {
        ap_table_setn(r->notes, NOTE_DYNAMIC, "skipped");
        sec_debug_log(r, 2, "Filtering off for non-dynamic resources (content-type = \"%s\")", debuglog_escape(r->pool, (char *)r->content_type));
        return 0;
    }

    return 1;
}

int read_post_payload(modsec_rec *msr, char **p) {
    request_rec *r = msr->r;
    request_body *rb = ap_pcalloc(r->pool, sizeof(request_body));
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    
    *p = NULL;
    
    if (rb == NULL) {
        msr->tmp_message = ap_psprintf(r->pool, "Failed to allocate %i bytes", sizeof(*rb));
        return -1;
    }

    if (dcfg->scan_post != 1) {
        sec_debug_log(r, 2, "Not looking at POST, feature is disabled");
        return 0;
    }
    
    /* is this a POST request?
     * if it is, we will handle it ourselves
     */
    if ((r->method_number != M_POST)&&(strncmp(r->the_request, r->method, strlen(r->method)) == 0)) {
        sec_debug_log(r, 2, "read_post_payload: skipping a non-POST request");
        return 0;
    }

    {    
        unsigned long bufsize, payload_size = 0;
        char *payload, *t;
        int i, rc;
        
        msr->should_body_exist = 1;

        if ((rc = ap_setup_client_block(r, REQUEST_CHUNKED_ERROR)) != OK) {
            msr->tmp_message = ap_psprintf(r->pool, "ap_setup_client_block failed with %i", rc);
            *p = NULL;
            return -1;
        }

        bufsize = r->remaining;
        if ((bufsize < 0)||(bufsize + 1 <= 0)) {
            msr->tmp_message = ap_psprintf(r->pool, "Invalid content length: %lu", bufsize);
            *p = NULL;
            return -1;
        }
        
        *p = payload = t = ap_palloc(r->pool, bufsize + 1);
        if (payload == NULL) {
            msr->tmp_message = ap_psprintf(r->pool, "read_post_payload: failed to allocate %li bytes", r->remaining + 1);
            *p = NULL;
            return -1;
        }

        ap_hard_timeout("mod_security: receive request data", r);

        if (ap_should_client_block(r)) {
            while ((i = ap_get_client_block(r, t, bufsize)) > 0) {
                payload_size += i;
                bufsize -= i;
                t += i;
                ap_reset_timeout(r);
            }
        }

        ap_kill_timeout(r);

        payload[payload_size] = 0;

        rb->buffer = payload;
        rb->sofar = payload;
        rb->length = payload_size;
        rb->remaining = payload_size;
        
        /* here we alter Apache internal structures
         * to point those pointers to our buffer
         * and to reset counters so that Apache will
         * think that the payload was not read at all
         */
        r->connection->client->inptr = (unsigned char *)payload;
        r->connection->client->incnt = payload_size;
        r->read_length = 0;
        r->remaining = payload_size;
        
        ap_table_setn(r->notes, NOTE, (char *)rb);
        sec_debug_log(r, 9, "Read %i bytes from POST [r=%x]", rb->length, r);
        msr->is_body_read = 1;
    }

    return 1;
}

char *unescape_regex_hex_inplace(char *input) {
    char *p = input;
    char c;
    
    while(*p != 0) {
        /* look for the "\x" combo */
        if ( (*p == '\\') && ((*(p + 1) == 'x')||(*(p + 1) == 'X')) ) {
            /* are there enough bytes */
            if ((*(p + 2) != 0)&&(*(p + 3) != 0)) {
                char *end = p + 4; /* end points to the first character after the \xHH */
                char *t = p;
                
                c = x2c((unsigned char *)(p + 2));
                *t++ = c;
                do {
                    c = *end++;
                    *t++ = c;
                } while(c != 0);
            }               
        }
        p++;
    }
    
    return input;
}

int parse_arguments(char *s, table *parsed_args, request_rec *r, sec_dir_config *dcfg, char **error_msg) {
    long int inputlength, i, j;
    char *my_error_msg = NULL;
    char *value = NULL;
    char *buf;
    int status;
    
    if (error_msg == NULL) return -1;
    *error_msg = NULL;
    
    if (s == NULL) return -1;
    inputlength = strlen(s);
    if (inputlength == 0) return 1;
    if (inputlength + 1 <= 0) return -1;

    buf = (char *)malloc(inputlength + 1);
    if (buf == NULL) {
        *error_msg = ap_psprintf(r->pool, "Failed to allocate %li bytes", inputlength + 1);
        return -1;
    }
    
    i = 0;
    j = 0;
    status = 0;
    while (i < inputlength) {
        if (status == 0) {
            /* parameter name */
            while ((s[i] != '=') && (s[i] != '&') && (i < inputlength)) {
                buf[j] = s[i];
                j++;
                i++;
            }
            buf[j++] = 0;
        } else {
            /* parameter value */
            while ((s[i] != '&') && (i < inputlength)) {
                buf[j] = s[i];
                j++;
                i++;
            }
            buf[j++] = 0;
        }

        if (status == 0) {
            if (normalise_inplace(r, dcfg, buf, &my_error_msg) == NULL) {
                free(buf);
                *error_msg = ap_psprintf(r->pool, "Error normalizing parameter name: %s", my_error_msg);
                return -1;
            }
            
            if (s[i] == '&') {
                /* Empty parameter */
                sec_debug_log(r, 4, "Adding parameter: [%s][]", debuglog_escape(r->pool, buf));
                ap_table_add(parsed_args, buf, "");
                status = 0; /* unchanged */
                j = 0;
            } else {
                status = 1;
                value = &buf[j];
            }
        }
        else {
            if (normalise_inplace(r, dcfg, value, &my_error_msg) == NULL) {
                free(buf);
                *error_msg = ap_psprintf(r->pool, "Error normalizing parameter value: %s", my_error_msg);
                return -1;
            }
            sec_debug_log(r, 4, "Adding parameter: [%s][%s]", debuglog_escape(r->pool, buf), debuglog_escape(r->pool, value));
            ap_table_add(parsed_args, buf, value);
            status = 0;
            j = 0;
        }
        
        i++; /* skip over the separator */
    }

    /* last parameter was empty */
    if (status == 1) {
        sec_debug_log(r, 4, "Adding parameter: [%s][]", debuglog_escape(r->pool, buf));
        ap_table_add(parsed_args, buf, "");
    }

    free(buf);
    return 1;
}

char *remove_binary_content(request_rec *r, char *data) {
    long size = r->remaining;
    char *src, *dst, *newdata;
    
    if (data == NULL) return NULL;
    if ((size < 0)||(size + 1 <= 0)) return NULL;

    /* make a copy of the payload first */
    newdata = ap_palloc(r->pool, size + 1);
    if (newdata == NULL) {
        sec_debug_log(r, 1, "remove_binary_content: failed to allocate %i bytes", size + 1);
        return NULL;
    }

    /* now remove zeros from the new buffer */    
    src = data;
    dst = newdata;
    while(size--) {
        if (*src != 0) *dst++ = *src++;
        else src++;
    }
    *dst = 0;
    
    return newdata;
}

int sec_initialize(modsec_rec *msr) {
    char *my_error_msg = NULL;
    request_rec *r = msr->r;
    int rc;
    const array_header *arr;
    table_entry *te;
    int i;
    
    /* TODO rename the variable below, the name is misleading
     * msr->_the_request
     */

    /* Although we could use the contents of r->unparsed_uri
     * to populate the variable below I have found it to be
     * safer to do it this way.
     */
    if ((r->args == NULL)&&(r->main != NULL)&&(r->main->args != NULL)) {
        /* I have found that the value of r->args is not populated
         * for subrequests although it is for for the main request; therefore
         * here we use the value available in the main request
         */
        msr->_the_request = ap_pstrcat(r->pool, r->uri, "?", r->main->args, NULL);
    }
    else if (r->args == NULL) {
        msr->_the_request = ap_pstrdup(r->pool, r->uri);
    }
    else {
        msr->_the_request = ap_pstrcat(r->pool, r->uri, "?", r->args, NULL);
    }
    
    msr->_the_request = normalise_inplace(r, msr->dcfg, msr->_the_request, &my_error_msg);
    if (msr->_the_request == NULL) {
        msr->tmp_message = ap_psprintf(r->pool, "Error normalizing REQUEST_URI: %s", my_error_msg);
        return perform_action(msr, msr->dcfg->action);
    }
    
    sec_debug_log(r, 4, "Normalised REQUEST_URI: %s", debuglog_escape(r->pool, msr->_the_request));
    sec_debug_log(r, 2, "Parsing arguments...");

    /* parse and validate GET parameters when available */
    if (r->args != NULL) {
        if (parse_arguments(r->args, msr->parsed_args, r, msr->dcfg, &my_error_msg) < 0) {
            msr->tmp_message = ap_psprintf(r->pool, "Invalid parameters: %s", my_error_msg);
            return perform_action(msr, msr->dcfg->action);
        }
    }
    
    arr = ap_table_elts(r->headers_in);
    te = (table_entry *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        if (normalise_relaxed(r, msr->dcfg, te[i].key, &my_error_msg) == NULL) {
            msr->tmp_message = ap_psprintf(r->pool, "Error validating header name: %s", my_error_msg);
            return perform_action(msr, msr->dcfg->action);
        }
        if (normalise_relaxed(r, msr->dcfg, te[i].val, &my_error_msg) == NULL) {
            msr->tmp_message = ap_psprintf(r->pool, "Error validating header value (%s): %s", te[i].key, my_error_msg);
            return perform_action(msr, msr->dcfg->action);
        }
    }

    /* parse, optionally validate cookies */    
    if ((parse_cookies(r, msr->parsed_cookies, &my_error_msg) < 0)&&(msr->dcfg->check_cookie_format == 1)) {
        msr->tmp_message = ap_psprintf(r->pool, "Invalid cookie format: %s", my_error_msg);
        return perform_action(msr, msr->dcfg->action);
    }
    
    if (msr->dcfg->scan_post) {
        char *content_type, *s;
        
        s = (char *)get_env_var(r, "MODSEC_NOPOSTBUFFERING");
        if (s != NULL) {
            msr->post_payload_dynamic_off = 1;
            sec_debug_log(r, 2, "read_post_payload: POST scanning turned off dynamically (MODSEC_NOPOSTBUFFERING=%s)", debuglog_escape(r->pool, s));
        } else {
            rc = read_post_payload(msr, &msr->_post_payload);
            if (rc < 0) {
                /* the error message prepared by read_post_payload */
                return perform_action(msr, msr->dcfg->action);
            }
        }

        if (msr->_post_payload != NULL) {    
            content_type = (char *)ap_table_get(r->headers_in, "Content-Type");
            if (content_type != NULL) sec_debug_log(r, 3, "content-type = \"%s\"", debuglog_escape(r->pool, content_type));
            else sec_debug_log(r, 3, "content-type is NULL");
            
            if ((content_type != NULL) && (strcmp(content_type, "application/x-www-form-urlencoded") == 0)) {
                /* parse variables before normalising the bufffer */
                if (msr->parsed_args != NULL) {
                    sec_debug_log(r, 3, "Parsing variables from POST payload");
                    if (parse_arguments(msr->_post_payload, msr->parsed_args, r, msr->dcfg, &my_error_msg) < 0) {
                        msr->tmp_message = ap_psprintf(r->pool, "Error parsing POST parameters: %s", my_error_msg);
                        return perform_action(msr, msr->dcfg->action);
                    }
                }
                
                msr->_post_payload = normalise(r, msr->dcfg, msr->_post_payload, &my_error_msg);
                if (msr->_post_payload == NULL) {
                    msr->tmp_message = ap_psprintf(r->pool, "Error normalizing POST payload: %s", my_error_msg);
                    return perform_action(msr, msr->dcfg->action);
                }
            }
            else if ((content_type != NULL) && ( strncmp(content_type, "multipart/form-data", 19) == 0)) {
                multipart_data *mpd = (multipart_data *)ap_pcalloc(r->pool, sizeof(*mpd));
                char *boundary = NULL;

                msr->mpd = mpd;                
                mpd->create_tmp_file = MULTIPART_TMP_FILE_CREATE;
                                
                boundary = strstr(content_type, "boundary=");
                if ((boundary != NULL)&&(*(boundary + 9) != 0)) {
                    mpd->boundary = boundary + 9;
                
                    if (multipart_init(mpd, r) < 0) {
                        msr->tmp_message = ap_psprintf(r->pool, "Invalid multipart/form-data format");
                        return perform_action(msr, msr->dcfg->action);
                    }
                    
                    if (multipart_process_chunk(mpd, msr->_post_payload, r->remaining) < 0) {
                        msr->tmp_message = ap_psprintf(r->pool, "Error processing POST data");
                        return perform_action(msr, msr->dcfg->action);
                    }
                    
                    /* get parsed variables */
                    if (multipart_get_variables(mpd, msr->parsed_args, msr->dcfg, &my_error_msg) < 0) {
                        msr->tmp_message = ap_psprintf(r->pool, "Error parsing multipart parameters: %s", my_error_msg);
                        return perform_action(msr, msr->dcfg->action);
                    }
                    
                    if (msr->dcfg->upload_approve_script != NULL) {
                        /* we only accept 1 as a correct result; the function may also return
                         * 0 for verification failed and -1 for an error (e.g. the script cannot
                         * be executed)
                         */
                        if (verify_uploaded_files(r, mpd, msr->dcfg->upload_approve_script, &my_error_msg) != 1) {
                            msr->tmp_message = ap_psprintf(r->pool, "Error verifying files: %s", my_error_msg);
                            return perform_action(msr, msr->dcfg->action);
                        }
                    }
                }
            }
            else {
                /*  remove binary content from the payload */
                sec_debug_log(r, 3, "Removing null bytes from POST payload");
                msr->_post_payload = remove_binary_content(r, msr->_post_payload);
                if (msr->_post_payload == NULL) {
                    msr->tmp_message = ap_psprintf(r->pool, "Error while removing binary content from POST");
                    return perform_action(msr, msr->dcfg->action);
                }
            }
             
            if (msr->_post_payload == NULL) {
                /* TODO will this ever happen? */
                msr->tmp_message = ap_psprintf(r->pool, "Error while removing binary content from POST");
                return perform_action(msr, msr->dcfg->action);
            }
        }
    }    
    
    return DECLINED;
}

int sec_check_all_signatures(modsec_rec *msr) {
    request_rec *r = msr->r;
    signature **signatures;
    int i, mode = 0;
    int skip_count = 0;
    int rc = DECLINED;
    
    /* loop through all signatures */
    signatures = (signature **)msr->dcfg->signatures->elts;
    for (i = 0; i < msr->dcfg->signatures->nelts; i++) {
        
        if (signatures[i]->actionset == NULL) {
            /* VERY IMPORTANT, configure rule action on-the-fly;
             * this cannot be done during the configuration phase
             * because per-dir configuration is also configured
             * on-the-fly.
             */
            signatures[i]->actionset = msr->dcfg->action;
        }
        
        /* check if we need to skip this rule */
        if (skip_count != 0) {
            skip_count--;
            continue;
        }

        /* just clear the flag, we had to use the flag
         * to detect a case when the last rule in the
         * rule chain was marked as chained            
         */
        if (mode == 2) mode = 0;
            
        /* in mode 1, we are looking for the next filter
         * that is next chained, and then skip it to
         * execute the next filter in the chain
         */
        if (mode == 1) {
            if (signatures[i]->actionset->is_chained == 0) mode = 0;
            continue;
        }
            
        msr->tmp_message = NULL;
        msr->tmp_redirect_url = NULL;
        msr->tmp_log_message = 0;
            
        rc = check_single_signature(msr, signatures[i]);
        sec_debug_log(r, 9, "Signature check returned %i", rc);
    
        /* We won't print any messages if the rule belongs
         * to a chain - only the last rule counts
         */
        if (signatures[i]->actionset->is_chained == 0) {
            if (msr->tmp_message != NULL) {
                ap_table_setn(r->headers_in, NOTE_MESSAGE, msr->tmp_message);
                if (msr->tmp_log_message) {
                    sec_debug_log(r, 1, "%s", msr->tmp_message);
                }
                else {
                    sec_debug_log(r, 2, "%s", msr->tmp_message);
                    ap_table_setn(r->notes, NOTE_NOAUDITLOG, "noauditlog");
                }
            } else {
                /* The final message is NULL, we unset any temporary
                 * messages that were present - it may happen when an
                 * external script is executed and rules are chained
                 */
                ap_table_unset(r->headers_in, NOTE_MESSAGE);
            }
        }
            
        /* DECLINED means that an allow action was called,
         * we need to pass the request through            
         */
        if (rc == DECLINED) {
            sec_debug_log(r, 9, "Allow request to pass through");
            return DECLINED;
        }

        /* OK means there was no filter match, we
         * switch to mode 1, processing will continue
         * with the next filter chain
         */
        if (rc == OK) {
            /* we go into mode 1 (looking for the last rule
             * in the chain) if this rule is not it
             */
            if (signatures[i]->actionset->is_chained == 1) {
                sec_debug_log(r, 9, "Chained rule and no match, find the next rule not in chain");
                mode = 1;
            }
            continue;
        }
            
        /* any status greater than zero means there
         * was a filter match, so we either stop execution
         * or proceed to the next rule if this rule was
         * chained
         */
        if (rc > 0) {
            if (signatures[i]->actionset->is_chained == 1) {
                mode = 2;
                sec_debug_log(r, 9, "Chained rule with match, continue in the loop");
                continue;
            }
            else {
rule_match:
                if (msr->tmp_redirect_url != NULL) {
                    ap_table_setn(msr->r->headers_out, "Location", msr->tmp_redirect_url);
                }
            
                sec_debug_log(r, 9, "Rule match, returning code %i", rc);
                return rc;
            }
        }
            
        /* if the return status is zero this
         * means skip some rules, so we skip
         */
        if (rc == MODSEC_SKIP) {
            skip_count = signatures[i]->actionset->skip_count;
            continue;
        }
            
        sec_debug_log(r, 1, "Unprocessed return code [%i]", rc);
        return DECLINED;
    }
    
    /* handle the case where there was a match on the
     * last filter so mode 2 check could not be done        
     * strickly speaking this should be a configuration error
     */
    if (mode == 2) {
        sec_debug_log(r, 1, "Last rule marked as chained - ignoring");
        goto rule_match;
    }

    return DECLINED;
}

int check_single_signature(modsec_rec *msr, signature *sig) {
    int j, rs;

    /* the idea behind non-selective filters is to apply them over
     * raw data, typically the complete first line of the request
     * and the complete POST payload
     */
        
    if (sig->is_selective == 0) {
        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at THE_REQUEST", debuglog_escape(msr->r->pool, sig->pattern));
            
        rs = check_sig_against_string(msr, sig, msr->_the_request, VAR_THE_REQUEST);
        if (rs != OK) return rs;

        if (msr->is_body_read) {
            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at POST_PAYLOAD", debuglog_escape(msr->r->pool, sig->pattern));
            
            if (msr->mpd != NULL) {
                /* multipart/form-data request */
                if (msr->_fake_post_payload == NULL) {
                    msr->_fake_post_payload = construct_fake_urlencoded(msr, msr->parsed_args);
                    if (msr->_fake_post_payload == NULL) {
                        sec_debug_log(msr->r, 1, "Failed during fake POST payload construction");
                        return DECLINED;
                    }
                }
                rs = check_sig_against_string(msr, sig, msr->_fake_post_payload, VAR_POST_PAYLOAD);
                if (rs != OK) return rs;
            } else {
                rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                if (rs != OK) return rs;
            }
        }
    }
    else {
        variable **variables;
        const char *v;

        /* this is a selective signature, this means that we need to
         * check only one part of the request and leave the rest alone
         */
            
        /* selective signatures can be negative and non-negative;
         * non-negative signatures consist of a list of variables
         * that represent parts of the request that need to be
         * checked; negative signatures apply only to request
         * arguments, when you want to exclude an argument from
         * a check            
         */

        if (sig->is_negative == 0) {
                
            /* loop through signature variables and
             * check them
             */

            variables = (variable **)sig->variables->elts;
            for (j = 0; j < sig->variables->nelts; j++) {

                if (variables[j]->type == VAR_ARGS) {
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at QUERY_STRING", debuglog_escape(msr->r->pool, sig->pattern));
            
                    variables[j]->type = VAR_QUERY_STRING;
                    v = get_variable(msr->r, variables[j], msr->parsed_args);
                    variables[j]->type = VAR_ARGS;
                    
                    rs = check_sig_against_string(msr, sig, v, VAR_QUERY_STRING);
                    if (rs != OK) return rs;

                    if (msr->is_body_read) {
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at POST_PAYLOAD", debuglog_escape(msr->r->pool, sig->pattern));
                  
                        if (msr->mpd != NULL) {
                            /* multipart/form-data request */
                            if (msr->_fake_post_payload == NULL) {
                                msr->_fake_post_payload = construct_fake_urlencoded(msr, msr->parsed_args);
                                if (msr->_fake_post_payload == NULL) {
                                    sec_debug_log(msr->r, 1, "Failed during fake POST payload construction");
                                    return DECLINED;
                                }
                            }
                            rs = check_sig_against_string(msr, sig, msr->_fake_post_payload, VAR_POST_PAYLOAD);
                            if (rs != OK) return rs;
                        } else {
                            rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                            if (rs != OK) return rs;
                        }
                    }
                }
                else if (variables[j]->type == VAR_POST_PAYLOAD) {
                    /* Ignore requests without bodies */
                    if (msr->should_body_exist) {
                        /* Note it can happen that a body is available but
                         * _post_payload is NULL.
                         */
                        if (msr->is_body_read == 0) {
                            /* Only complain if body is not available by configuration mistake */
                            if (msr->post_payload_dynamic_off == 0) {
                               sec_debug_log(msr->r, 1, "Filtering against POST payload requested but payload is not available");
                            }
                            return OK;
                        } else {
                            if (msr->mpd == NULL) {
                                if (msr->_post_payload == NULL) {
                                    sec_debug_log(msr->r, 1, "Expected POST payload but got NULL");
                                    return DECLINED;
                                }
                                rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                                if (rs != OK) return rs;
                            } else {
                                /* multipart/form-data request */
                                if (msr->_fake_post_payload == NULL) {
                                    msr->_fake_post_payload = construct_fake_urlencoded(msr, msr->parsed_args);
                                    if (msr->_fake_post_payload == NULL) {
                                        sec_debug_log(msr->r, 1, "Failed during fake POST payload construction");
                                        return DECLINED;
                                    }
                                }
                                rs = check_sig_against_string(msr, sig, msr->_fake_post_payload, VAR_POST_PAYLOAD);
                                if (rs != OK) return rs;
                            }
                        }
                    }
                }
                else if (variables[j]->type == VAR_ARGS_NAMES) {
                    table_entry *te;
                    array_header *arr;
                    int k;
                    
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_NAMES", debuglog_escape(msr->r->pool, sig->pattern));
                    
                    if (msr->parsed_args == NULL) {
                        sec_debug_log(msr->r, 1, "check_single_signature, parsed_args is NULL");
                        return DECLINED;
                    }
                    
                    arr = ap_table_elts(msr->parsed_args);
                    te = (table_entry *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {
                        rs = check_sig_against_string(msr, sig, te[k].key, VAR_ARGS_NAMES);
                        if (rs != OK) return rs;
                    }
                }
                else if (variables[j]->type == VAR_ARGS_VALUES) {
                    table_entry *te;
                    array_header *arr;
                    int k;
                    
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_VALUES", debuglog_escape(msr->r->pool, sig->pattern));
                    
                    if (msr->parsed_args == NULL) {
                        sec_debug_log(msr->r, 1, "check_single_signature, parsed_args is NULL");
                        return DECLINED;
                    }
                    
                    arr = ap_table_elts(msr->parsed_args);
                    te = (table_entry *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {
                        rs = check_sig_against_string(msr, sig, te[k].val, VAR_ARGS_VALUES);
                        if (rs != OK) return rs;
                    }
                }
                else if(variables[j]->type == VAR_COOKIES_NAMES) {
                    table_entry *te;
                    array_header *arr;
                    int k;
                    
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at COOKIES_NAMES", debuglog_escape(msr->r->pool, sig->pattern));
                    
                    if (msr->parsed_cookies == NULL) {
                        sec_debug_log(msr->r, 1, "check_single_signature, parsed_cookies is NULL");
                        return DECLINED;
                    }
                    
                    arr = ap_table_elts(msr->parsed_cookies);
                    te = (table_entry *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {
                        sec_debug_log(msr->r, 5, "Cookie [%s][%s]", debuglog_escape(msr->r->pool, te[k].key), debuglog_escape(msr->r->pool, te[k].val));
                        rs = check_sig_against_string(msr, sig, te[k].key, VAR_COOKIES_NAMES);
                        if (rs != OK) return rs;
                    }
                }
                else if(variables[j]->type == VAR_COOKIES_VALUES) {
                    table_entry *te;
                    array_header *arr;
                    int k;
                    
                    sec_debug_log(msr->r, 2, "Checking signature \"%s\" at COOKIES_VALUES", debuglog_escape(msr->r->pool, sig->pattern));
                    
                    if (msr->parsed_cookies == NULL) {
                        sec_debug_log(msr->r, 1, "check_single_signature, parsed_cookies is NULL");
                        return DECLINED;
                    }
                                                                                        
                    arr = ap_table_elts(msr->parsed_cookies);
                    te = (table_entry *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {
                        sec_debug_log(msr->r, 5, "Cookie [%s][%s]", debuglog_escape(msr->r->pool, te[k].key), debuglog_escape(msr->r->pool, te[k].val));
                        rs = check_sig_against_string(msr, sig, te[k].val, VAR_COOKIES_VALUES);
                        if (rs != OK) return rs;
                    }
                }
                else {
                    /* simple variable, get the value and check it */

                    if (variables[j]->type == VAR_COOKIE) {
                        if (msr->parsed_cookies == NULL) {
                            sec_debug_log(msr->r, 1, "check_single_signature, parsed_cookies is NULL");
                            return DECLINED;
                        }
                        v = get_variable(msr->r, variables[j], msr->parsed_cookies);
                    } else {
                        if (msr->parsed_args == NULL) {
                            sec_debug_log(msr->r, 1, "check_single_signature, parsed_args is NULL");
                            return DECLINED;
                        }
                        v = get_variable(msr->r, variables[j], msr->parsed_args);
                    }
                    
                    if (v != NULL) {
                        if (variables[j]->name != NULL) {
                            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at %s(%s)", debuglog_escape(msr->r->pool, sig->pattern), all_variables[variables[j]->type], variables[j]->name);
                        }
                        else {                        
                            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at %s", debuglog_escape(msr->r->pool, sig->pattern), all_variables[variables[j]->type]);
                        }
                            
                        /* sec_debug_log(msr->r, 5, "Variable value \"%s\"", debuglog_escape(msr->r->pool, (char *)v)); */
                        
                        rs = check_sig_against_string(msr, sig, (char *)v, variables[j]->type);
                        if (rs != OK) return rs;
                    }
                    else {
                        sec_debug_log(msr->r, 1, "Variable not found \"%s\"", debuglog_escape(msr->r->pool, variables[j]->name));
                    }
                }
            }
        }
        else {
            table *our_parsed_args;
            char *fake_body = NULL;

            if (msr->parsed_args == NULL) {
                sec_debug_log(msr->r, 1, "Internal error: arguments are not parsed");
                return DECLINED;
            }
            
            our_parsed_args = ap_copy_table(msr->r->pool, msr->parsed_args);
            
            /* Find the unwanted variable names in the signature
             * data and remove them from the variable list.
             */
            variables = (variable **)sig->variables->elts;
            for (j = 0; j < sig->variables->nelts; j++) {
                if ((variables[j]->type == VAR_CUSTOM) && (variables[j]->action == VAR_ACTION_ALLOW)) {
                    ap_table_unset(our_parsed_args, variables[j]->name);
                }
            }
            
            fake_body = construct_fake_urlencoded(msr, our_parsed_args);
            if (fake_body == NULL) {
                sec_debug_log(msr->r, 1, "Failed with construct_fake_urlencoded");
                return DECLINED;
            }
            
            /* make the check against the compiled string */
            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_SELECTIVE", debuglog_escape(msr->r->pool, sig->pattern));
            rs = check_sig_against_string(msr, sig, (char *)fake_body, VAR_ARGS_SELECTIVE);
            if (rs != OK) return rs;
        }
    }
        
    return OK;
}

int sec_exec_child(void *_ed, child_info *pinfo) {
    char **env = NULL;
    int child_pid;
    exec_data *ed = (exec_data *)_ed;
    request_rec *r = ed->r;
    char *command = ap_pstrdup(r->pool, ed->command);
      
    ap_add_cgi_vars(r);
    ap_add_common_vars(r);
    
    /* TODO add mod_security specific environment variables */
    
    /* Hack to make PHP happy; PHP is often compiled to be
     * run from the command line and from the web server. It
     * analyses environment variables to determine how it is
     * supposed to behave. To make things worse, if it determines
     * that it is being run as a CGI script, there are certain
     * security checks it performs. In most cases, it simply
     * refuses to run when started. The following two lines
     * are a workaround around those security checks. 
     */
    ap_table_add(r->subprocess_env, "PATH_TRANSLATED", ed->command);
    ap_table_add(r->subprocess_env, "REDIRECT_STATUS", "302");
    
    env = (char **)ap_create_environment(r->pool, r->subprocess_env);
    ap_error_log2stderr(r->server);
    
    #ifndef WIN32
    {
        char *p;
        
        /* Suexec will complain if the name of the
         * script starts with a slash. To work around
         * that we chdir to the folder, and then execute
         * the script giving a relative filename. We have
         * already forked so we can do that.
         */
        p = strrchr(command, '/');
        if (p != NULL) {
            r->filename = p + 1;
            *p = 0;
            chdir(command);
        } else {
            r->filename = ed->command;
        }
    }
    #else
        r->filename = ed->command;
    #endif
    
    r->args = ed->args;
    
    ap_cleanup_for_exec();
    
    child_pid = ap_call_exec(r, pinfo, r->filename, env, 0);

#ifdef WIN32
    return(child_pid);
#else
    ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, NULL, "mod_security: exec failed: %s", ed->command); 
    exit(0);
    
    return(0);
#endif
}
                             
int check_sig_against_string(modsec_rec *msr, signature *_sig, const char *s, int var_type) {
    request_rec *r = msr->r;
    BUFF *p1, *p2, *p3;
    int regex_result;
    int rc = OK;
    
    if (_sig->regex == NULL) {
        sec_debug_log(r, 1, "Compiled regex for pattern \"%s\" is NULL!", debuglog_escape(msr->r->pool, _sig->pattern));
        return OK;
    }
    
    sec_debug_log(r, 9, "Checking against \"%s\"", debuglog_escape(msr->r->pool, (char *)s));

    regex_result = ap_regexec(_sig->regex, s, 0, NULL, 0);
    /* regexec returns 0 on success or REG_NOMATCH on failure */
    if ( ((regex_result == 0)&&(_sig->is_allow == 0)) || ((regex_result != 0)&&(_sig->is_allow == 1)) ) {
        actionset_t *actionset = _sig->actionset;
        
        /* perform action */
        switch(actionset->action) {
        
            case ACTION_SKIP :
                sec_debug_log(r, 2, "Skipping %i statements on pattern match \"%s\" at %s", actionset->skip_count, debuglog_escape(msr->r->pool, _sig->pattern), all_variables[var_type]);
                rc = MODSEC_SKIP;
                break;
        
            case ACTION_ALLOW :
                msr->tmp_message = ap_psprintf(r->pool, "Access allowed based on pattern match \"%s\" at %s", debuglog_escape(r->pool, _sig->pattern), all_variables[var_type]);
                /* returning DECLINED will interrupt filter processing but
                 * it won't do anything the the request
                 */
                rc = DECLINED;
                break;
        
            case ACTION_DENY :
                msr->tmp_message = ap_psprintf(r->pool, "Access denied with code %i. Pattern match \"%s\" at %s.", actionset->status, debuglog_escape(r->pool, _sig->pattern), all_variables[var_type]);
                rc = actionset->status;
                break;
                
            case ACTION_REDIRECT :
                msr->tmp_message = ap_psprintf(r->pool, "Access denied with redirect to [%s]. Pattern match \"%s\" at %s.", actionset->redirect_url, debuglog_escape(r->pool, _sig->pattern), all_variables[var_type]);
                msr->tmp_redirect_url = actionset->redirect_url;
                rc = REDIRECT;
                break;
                
            default :
                msr->tmp_message = ap_psprintf(r->pool, "Warning. Pattern match \"%s\" at %s.", debuglog_escape(r->pool, _sig->pattern), all_variables[var_type]);
                break;
        }
        
        /* execute the external script */
        if (actionset->exec) {
            exec_data *ed = NULL;
            char buf[4097];
            
            ed = ap_pcalloc(r->pool, sizeof(exec_data));
            ed->r = r;
            ed->command = actionset->exec_string;
            ed->args = NULL;
            
            sec_debug_log(r, 1, "Executing command \"%s\"", debuglog_escape(msr->r->pool, ed->command));
            
            ap_table_setn(r->headers_in, NOTE_EXECUTED, ed->command);
            if (msr->tmp_message != NULL) {
                ap_table_setn(r->headers_in, NOTE_MESSAGE, msr->tmp_message);
            }
            if ((rc != DECLINED)&&(rc != MODSEC_SKIP)) {
                char *action = ap_psprintf(r->pool, "%i", rc);
                ap_table_setn(r->headers_in, NOTE_ACTION, action);
            }

            if (!ap_bspawn_child(r->main ? r->main->pool : r->pool, sec_exec_child, (void *)ed, kill_after_timeout, &p1, &p2, &p3)) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r->server, "mod_security: couldn't spawn child process: %s", actionset->exec_string);
                return HTTP_INTERNAL_SERVER_ERROR;
            }
            
            while(ap_bgets(buf, 4096, p2) > 0) continue;
            while(ap_bgets(buf, 4096, p3) > 0) continue;
        }
        
        if (actionset->pause != 0) {
            sec_debug_log(r, 1, "Pausing [%s] for %i ms", debuglog_escape(msr->r->pool, r->uri), actionset->pause);
            sec_sleep(actionset->pause);
        }
        
        if ((msr->tmp_message != NULL)&&(actionset->log)) msr->tmp_log_message = 1;
    }

    return rc;
}

char *parse_action(char *p2, actionset_t *actionset, pool *_pool) {
    char *t = ap_pstrdup(_pool, p2);
    char *saveptr = NULL;
    char *p = strtok_r(t, ",", &saveptr);
    int found;
    
    actionset->skip_count = 1;
    actionset->action = ACTION_DENY;
    actionset->status = HTTP_FORBIDDEN;

    while (p != NULL) {
        found = 0;
        
        if (strcmp(p, "log") == 0) {
            found = 1;
            actionset->log = 1;
        }
        else if (strcmp(p, "nolog") == 0) {
            found = 1;
            actionset->log = 0;
        }
        else if (strncmp(p, "status", 6) == 0) {
            found = 1;
            if (strlen(p) > 7) {
                actionset->status = atoi(p + 7);
            }
        }
        else if (strcmp(p, "deny") == 0) {
            found = 1;
            actionset->action = ACTION_DENY;
        }        
        else if (strcmp(p, "pass") == 0) {
            found = 1;
            actionset->action = ACTION_NONE;
        }
        else if (strcmp(p, "allow") == 0) {
            found = 1;
            actionset->action = ACTION_ALLOW;
        }
        else if (strcmp(p, "chain") == 0) {
            found = 1;
            actionset->is_chained = 1;
        }
        else if (strcmp(p, "chained") == 0) {
            found = 1;
            actionset->is_chained = 1;
        }
        else if (strncmp(p, "skipnext", 8) == 0) {
            found = 1;
            actionset->action = ACTION_SKIP;
            if (strlen(p) > 9) {
                actionset->skip_count = atoi(p + 9);
            }
        }
        else if (strncmp(p, "skip", 4) == 0) {
            found = 1;
            actionset->action = ACTION_SKIP;
            if (strlen(p) > 5) {
                actionset->skip_count = atoi(p + 5);
            }
        }
        else if (strncmp(p, "exec", 4) == 0) {
            found = 1;
            actionset->exec = 1;
            if (strlen(p) > 5) {
                actionset->exec_string = ap_pstrdup(_pool, p + 5);
            }
        }
        else if (strncmp(p, "redirect", 8) == 0) {
            found = 1;
            actionset->action = ACTION_REDIRECT;
            if (strlen(p) > 9) {
                actionset->redirect_url = ap_pstrdup(_pool, p + 9);
            }
        }
        else if (strncmp(p, "msg", 3) == 0) {
            found = 1;
            if (strlen(p) > 4) {
                actionset->msg = ap_pstrdup(_pool, p + 4);
            }
        }
        else if (strncmp(p, "id", 2) == 0) {
            found = 1;
            if (strlen(p) > 3) {
                actionset->id = ap_pstrdup(_pool, p + 3);
            }
        }
        else if (strncmp(p, "pause", 5) == 0) {
            found = 1;
            if (strlen(p) > 6) {
                actionset->pause = atoi(p + 6);
            }
        }
        
        if (found == 0) {
            return ap_psprintf(_pool, "Unknown mod_security action \"%s\"", p);
        }

        p = strtok_r(NULL, ",", &saveptr);
    }

    /* Chained rules must always try to deny
     * access in order for chaining to work
     * properly
     */
    if (actionset->is_chained) {
        actionset->action = ACTION_DENY;
        actionset->status = HTTP_FORBIDDEN;
    }

    return NULL;
}

static const char *cmd_filter_engine(cmd_parms *cmd, sec_dir_config* dcfg, char *p1) {
    if (strcasecmp(p1, "On") == 0) dcfg->filter_engine = FILTERING_ON;
    else
    if (strcasecmp(p1, "Off") == 0) dcfg->filter_engine = FILTERING_OFF;
    else
    if (strcasecmp(p1, "DynamicOnly") == 0) dcfg->filter_engine = FILTERING_DYNAMIC_ONLY;
    else
    return ap_psprintf(cmd->pool, "Unrecognized parameter value for SecFilterEngine: %s", p1);
    
    return NULL;
}

static const char *cmd_filter_inheritance(cmd_parms *cmd, sec_dir_config* dcfg, int flag) {
    if (flag) dcfg->filters_clear = 0;
    else dcfg->filters_clear = 1;
    return NULL;
}

static const char *cmd_server_response_token(cmd_parms *cmd, sec_dir_config *dcfg, int flag) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);

    if (cmd->server->is_virtual) {
        return "SecServerResponseToken not allowed in VirtualHost";
    }     
    
    scfg->server_response_token = flag;
    return NULL;
}

static const char *cmd_audit_engine(cmd_parms *cmd, sec_dir_config * dcfg, char *p1) {
    if (strcasecmp(p1, "On") == 0) dcfg->auditlog_flag = AUDITLOG_ON;
    else
    if (strcasecmp(p1, "Off") == 0) dcfg->auditlog_flag = AUDITLOG_OFF;
    else
    if (strcasecmp(p1, "RelevantOnly") == 0) dcfg->auditlog_flag = AUDITLOG_RELEVANT_ONLY;
    else
    if (strcasecmp(p1, "DynamicOrRelevant") == 0) dcfg->auditlog_flag = AUDITLOG_DYNAMIC_OR_RELEVANT;
    else
    return ap_psprintf(cmd->pool, "Unrecognized parameter value for SecAuditEngine: %s", p1);
    
    return NULL;
}

static const char *cmd_audit_log(cmd_parms *cmd, sec_dir_config * dcfg, char *p1) {
    dcfg->auditlog_name = ap_server_root_relative(cmd->pool, p1);

    #ifdef WIN32
    dcfg->auditlog_fd = open(dcfg->auditlog_name, O_WRONLY | O_APPEND | O_CREAT | O_BINARY, CREATEMODE); 
    #else
    dcfg->auditlog_fd = ap_popenf(cmd->pool, dcfg->auditlog_name, O_WRONLY | O_APPEND | O_CREAT | O_BINARY, CREATEMODE); 
    #endif    

    if (dcfg->auditlog_fd < 0) {
        return ap_psprintf(cmd->pool, "mod_security: Failed to open the audit log file: %s", dcfg->auditlog_name);
    }

    return NULL;
}

static const char *cmd_upload_dir(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    if (strcasecmp(p1, "none") == 0) dcfg->upload_dir = NULL;
    else dcfg->upload_dir = ap_server_root_relative(cmd->pool, p1);
    /* TODO does the directory exist? */
    return NULL;
}

static const char *cmd_upload_approve_script(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    if (strcasecmp(p1, "none") == 0) dcfg->upload_approve_script = NULL;
    else dcfg->upload_approve_script = p1;
    /* TODO does the script exist? */
    return NULL;
}

static const char *cmd_upload_keep_files(cmd_parms * cmd, sec_dir_config * dcfg, int flag) {
    dcfg->upload_keep_files = flag;
    return NULL;
}

static const char *cmd_scan_post(cmd_parms * cmd, sec_dir_config * dcfg, int flag) {
    dcfg->scan_post = flag;
    return NULL;
}

static const char *cmd_filter_check_url_encoding(cmd_parms * cmd, sec_dir_config *dcfg, int flag) {
    dcfg->check_encoding = flag;
    return NULL;
}

static const char *cmd_filter_check_unicode_encoding(cmd_parms * cmd, sec_dir_config *dcfg, int flag) {
    dcfg->check_unicode_encoding = flag;
    return NULL;
}

static const char *cmd_filter_force_byte_range(cmd_parms *cmd, sec_dir_config *dcfg, char *p1, char *p2) {
    dcfg->range_start = atoi(p1);
    dcfg->range_end = atoi(p2);
    return NULL;
}

static const char *cmd_default_action(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    dcfg->action = (actionset_t *)ap_pcalloc(cmd->pool, sizeof(actionset_t));
    return parse_action(p1, dcfg->action, cmd->pool);
}

static const char *cmd_filter(cmd_parms *cmd, sec_dir_config *dcfg, char *_p1, char *p2) {
    char *p1 = unescape_regex_hex_inplace(ap_pstrdup(cmd->pool, _p1));
    signature *sig;

    sig = ap_pcalloc(cmd->pool, sizeof(signature));
    if (sig == NULL) return FATAL_ERROR;
    
    /* p1 is the regular expression string */
    if (p1[0] == '!') {
        sig->is_allow = 1;
        sig->pattern = _p1;
        sig->regex = ap_pregcomp(cmd->pool, p1 + 1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    else {
        sig->pattern = _p1;
        sig->regex = ap_pregcomp(cmd->pool, p1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    
    if (sig->regex == NULL) {
        return ap_psprintf(cmd->pool, "Invalid regular expression: %s", sig->pattern);
    }
    
    /* p2 is an optional action string */
    if (p2 != NULL) {
        char *rc;
        
        sig->actionset = (actionset_t *)ap_pcalloc(cmd->pool, sizeof(actionset_t));
        rc = parse_action(p2, sig->actionset, cmd->pool);
        if (rc != NULL) return rc;
    }

    /* add the signature to the list of all signatures */
    *(signature **)ap_push_array(dcfg->signatures) = sig;

    return NULL;
}

static const char *cmd_filter_debug_log(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    dcfg->debuglog_name = ap_server_root_relative(cmd->pool, p1);

    #ifdef WIN32
    dcfg->debuglog_fd = open(dcfg->debuglog_name, O_WRONLY | O_APPEND | O_CREAT | O_BINARY, CREATEMODE); 
    #else
    dcfg->debuglog_fd = ap_popenf(cmd->pool, dcfg->debuglog_name, O_WRONLY | O_APPEND | O_CREAT | O_BINARY, CREATEMODE); 
    #endif   
    
    if (dcfg->debuglog_fd < 0) {
        return ap_psprintf(cmd->pool, "mod_security: Failed to open the debug log file: %s", dcfg->debuglog_name);
    }
    
    return NULL;
}

static const char *cmd_filter_debug_level(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    dcfg->filter_debug_level = atoi(p1);
    return NULL;
}

static const char *cmd_filter_selective(cmd_parms *cmd, sec_dir_config *dcfg, char *p1, char *_p2, char *p3) {
    char *p2 = unescape_regex_hex_inplace(ap_pstrdup(cmd->pool, _p2));
    char *p, *t;
    signature *sig;
    char *saveptr;
    
    /* initialise the structure first */
    sig = ap_pcalloc(cmd->pool, sizeof (signature));
    if (sig == NULL) return FATAL_ERROR;    
    
    sig->is_allow = 0;
    sig->is_selective = 1;
    sig->is_negative = 0;
    sig->requires_parsed_args = 0;
    sig->variables = ap_make_array(cmd->pool, 10, sizeof (variable *));
    
    if (p2[0] == '!') {
        sig->is_allow = 1;
        sig->pattern = _p2;
        sig->regex = ap_pregcomp(cmd->pool, p2 + 1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    else {
        sig->pattern = _p2;
        sig->regex = ap_pregcomp(cmd->pool, p2, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    
    if (sig->regex == NULL) {
        return ap_psprintf(cmd->pool, "Invalid regular expression: %s", sig->pattern);
    }
    
    /* split parameter 1 apart and extract variable names */
    
    p = strdup(p1);
    t = strtok_r(p, "|", &saveptr);
    while (t != NULL) {
        char *x = t;

        /* add the token to the list */
        variable *v = (variable *)ap_pcalloc(cmd->pool, sizeof(variable));
        if  (v == NULL) return FATAL_ERROR;
        v->type = VAR_UNKNOWN;
        v->name = NULL;

        /* when ! is the first character in the variable
         * name that means that the restrictions need to be
         * relaxed for that variable (within the filter scope)
         */
        if (t[0] == '!') {
            v->action = VAR_ACTION_ALLOW;
            sig->is_negative = 1;
            sig->requires_parsed_args = 1;
            x++;
        }
        else {
            v->action = VAR_ACTION_DENY;
        }

        /* arguments */
        if (strncmp(x, "ARG_", 4) == 0) {
            v->type = VAR_CUSTOM;
            v->name = ap_pstrdup(cmd->pool, x + 4);
            sig->requires_parsed_args = 1;
        }
        /* HTTP headers */
        else if (strncmp(x, "HTTP_", 5) == 0) {
            char *px;

            v->type = VAR_HEADER;
            v->name = ap_pstrdup(cmd->pool, x + 5);
    
            /* replace all "_" with "-" */
            px = v->name;
            while (*px != 0) {
                if (*px == '_') *px = '-';
                px++;
            }
        }
        /* COOKIES */
        else if (strncmp(x, "COOKIE_", 7) == 0) {
            v->type = VAR_COOKIE;
            v->name = ap_pstrdup(cmd->pool, x + 7);
        }
        /* environment variables */
        else if (strncmp(x, "ENV_", 4) == 0) {
            v->type = VAR_ENV;
            v->name = ap_pstrdup(cmd->pool, x + 4);
        }
        /* all arguments */
        else if (strcmp(x, "ARGS") == 0) {
            v->type = VAR_ARGS;
            v->name = ap_pstrdup(cmd->pool, x);
        }
        /* just the post payload */
        else if (strcmp(x, "POST_PAYLOAD") == 0) {
            v->type = VAR_POST_PAYLOAD;
            v->name = ap_pstrdup(cmd->pool, x);
        }
        /* everything else */
        else {
            char **vl = all_variables;
            int i = 0;

            while (*vl != NULL) {
                if (strcmp(*vl, x) == 0) {
                    v->type = i;
                    /* v->name = ap_pstrdup(cmd->pool, x); */
                    break;
                }

                i++;
                vl++;
            }
        }

        if (v->type == VAR_UNKNOWN) {
            v->name = ap_pstrdup(cmd->pool, "UKNOWN");
            return ap_psprintf(cmd->pool, "Unknown variable name: %s", x);
        }

        if ((v->type == VAR_ARGS_NAMES)||(v->type == VAR_ARGS_VALUES)) sig->requires_parsed_args = 1;

        *(variable **)ap_push_array(sig->variables) = v;

        /* and proceed to the next token */
        t = strtok_r(NULL, "|", &saveptr);
    }

    free(p);

    /* is default action parameter present? */
    if (p3 != NULL) {
        char *rc;
        
        sig->actionset = (actionset_t *)ap_pcalloc(cmd->pool, sizeof(actionset_t));
        rc = parse_action(p3, sig->actionset, cmd->pool);
        if (rc != NULL) return rc;
    }

    /* add the signature to the list of all signatures */
    *(signature **)ap_push_array(dcfg->signatures) = sig;

    return NULL;
}

static const char *cmd_chroot_dir(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    char cwd[1025] = "";

    if (cmd->server->is_virtual) {
        return "SecChrootDir not allowed in VirtualHost";
    }    
    
    scfg->chroot_dir = p1;
    
    if (getcwd(cwd, 1024) == NULL) {
        return "SecChrootDir: failed to get the current working directory";
    }
    
    if (chdir(scfg->chroot_dir) < 0) {
        return ap_psprintf(cmd->pool, "SecChrootDir: failed to chdir to \"%s\", errno=%d(%s)", scfg->chroot_dir, errno, strerror(errno));
    }
    
    if (chdir(cwd) < 0) {
        return ap_psprintf(cmd->pool, "SecChrootDir: failed to chdir to \"%s\", errno=%d(%s)", cwd, errno, strerror(errno));
    }
    
    return NULL;
}

static const char *cmd_chroot_lock(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    
    if (cmd->server->is_virtual) {
        return "SecChrootLock not allowed in VirtualHost";
    }
    
    scfg->chroot_lock = ap_server_root_relative(cmd->pool, p1);
    if (scfg->chroot_lock == NULL) {
        return "SecChrootLock: allocation failed";
    }
    
    return NULL;
}

static const char *cmd_server_signature(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    
    if (cmd->server->is_virtual) {
        return "SecServerSignature not allowed in VirtualHost";
    }
    
    scfg->server_signature = p1;
    return NULL;
}

static const char *cmd_normalize_cookies(cmd_parms * cmd, sec_dir_config * dcfg, int flag) {
    dcfg->normalize_cookies = flag;
    return NULL;
}

static const char *cmd_check_cookie_format(cmd_parms * cmd, sec_dir_config * dcfg, int flag) {
    dcfg->check_cookie_format = flag;
    return NULL;
}

static const char *cmd_charset(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    dcfg->charset_id = convert_charset_to_id(p1);
    if (dcfg->charset_id == -1) {
        return ap_psprintf(cmd->pool, "Unknown charset: %s", p1);
    }
    return NULL;
}

static const command_rec sec_cmds[] = {

     {
         "SecFilter",
         cmd_filter,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE12,
         "The filtering expression"
     },
     
     {
        "SecFilterDebugLog",
        cmd_filter_debug_log,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        TAKE1,
        "The filename of the filter debugging log file"
     },
          
     {
         "SecFilterDebugLevel",
         cmd_filter_debug_level,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         "The level of the debugging log file verbosity"
     },

     {
         "SecFilterSelective",
         cmd_filter_selective,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE23,
         "The variable representing areas where filtering is wanted, the filtering regular expression and optional action to take on match"
     },

     {
         "SecFilterEngine",
         cmd_filter_engine,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         "On, Off, or DynamicOnly to determine when will request be filtered"
     },
     
     {
        "SecServerResponseToken",
        cmd_server_response_token,
        NULL,
        RSRC_CONF,
        FLAG,
        "On or Off to set whether the mod_security token will appear in the server signature"
     },

     {
         "SecFilterScanPOST",
         cmd_scan_post,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         FLAG,
         "On or Off to set whether a request body will be processed"
     },

     {
         "SecFilterDefaultAction",
         cmd_default_action,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         "The default action to take on rule match"
     },

     {
        "SecFilterInheritance",
        cmd_filter_inheritance,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        FLAG,
        "On or Off to set whether rules from the parent context will be inherited"
     },

     {
         "SecAuditEngine",
         cmd_audit_engine,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         "On, Off, RelevantOnly or DynamicOrRelevent to determine the level of audit logging"
     },

     {
         "SecAuditLog",
         cmd_audit_log,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         "The filename of the audit log file"
     },
     
     {
         "SecUploadDir",
         cmd_upload_dir,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         "The path to the directory where uploaded files should be stored"
     },
     
     {
         "SecUploadKeepFiles",
         cmd_upload_keep_files,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         FLAG,
         "On or Off to choose whether to keep the uploaded files or not"
     },
     
     {
         "SecUploadApproveScript",
         cmd_upload_approve_script,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         "The path to the script that will be called to approve every uploaded file"
     },
     
     {
         "SecFilterCheckURLEncoding",
         cmd_filter_check_url_encoding,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         FLAG,
         "On or Off to set whether URL encoding validation will be performed"
     },
     
     {
         "SecFilterCheckUnicodeEncoding",
         cmd_filter_check_unicode_encoding,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         FLAG,
         "On or Off to set whether Unicode encoding validation will be performed"
     },
     
     {
         "SecFilterForceByteRange",
         cmd_filter_force_byte_range,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE2,
         "The first and the last byte value of the range that will be accepted"
     },
     
     {
         "SecChrootDir",
         cmd_chroot_dir,
         NULL,
         RSRC_CONF,
         TAKE1,
         "The path of the directory to which server will be chrooted"
     },
     
     {
         "SecChrootLock",
         cmd_chroot_lock,
         NULL,
         RSRC_CONF,
         TAKE1,
         "The filename of the lock file used during the chroot process, defaults to \"logs/modsec_chroot.lock\""
     },
     
     {
         "SecServerSignature",
         cmd_server_signature,
         NULL,
         RSRC_CONF,
         TAKE1,
         "The new signature of the server"
     },
     
     {
        "SecFilterNormalizeCookies",
        cmd_normalize_cookies,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        FLAG,
        "On or Off to determine whether cookie values will be normalized for testing, defaults to On"
     },
     
     {
        "SecFilterCheckCookieFormat",
        cmd_check_cookie_format,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        FLAG,
        "On or Off to determine whether cookie format will be checked. Defaults to On"
     },
     
     {
        "SecCharset",
        cmd_charset,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        TAKE1,
        "Configures the charset"
     },

     { NULL }
};

int sec_logger(request_rec *_r) {
    request_rec *r = _r;
    unsigned int o1size = 0, o2size = 0;
    char *o1 = NULL, *o2 = NULL;
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    const char *modsec_message = NULL;
    array_header *arr;
    table_entry *te;
    char *remote_user, *local_user, *t;
    char *escaped_the_request = real_debuglog_escape(r->pool, r->the_request);
    size_t nbytes;
    int is_post = 0;
    int i = 0;
    
    /* do nothing if not initialised properly */
    if ((dcfg == NULL)||(dcfg->filter_engine == NOT_SET)) {
        return DECLINED;
    }

    /* Since Apache 1.3.31 (I think) logger functions are called on
     * request line timeouts so we ignore that.
     */
    if (r->the_request == NULL) {
        return DECLINED;
    }
    
    if (escaped_the_request == NULL) {
        sec_debug_log(r, 1, "sec_logger: escaped_the_request is NULL");
        return DECLINED;
    }
    
    r = find_last_request(_r);

    sec_debug_log(r, 2, "sec_logger executed to perform audit log [_r=%x, r=%x]", _r, r);

    switch(dcfg->auditlog_flag) {
    
        case AUDITLOG_OFF :
            sec_debug_log(r, 2, "Audit log is off here");
            return OK;
            break;
            
        case AUDITLOG_DYNAMIC_OR_RELEVANT :
            modsec_message = ap_table_get(r->headers_in, NOTE_MESSAGE);
            if (((modsec_message == NULL)&&(r->handler == NULL))||(ap_table_get(_r->notes, NOTE_NOAUDITLOG) != NULL)) {
                sec_debug_log(r, 2, "Audit log: ignoring a non-dynamic and non-relevant request (content-type = \"%s\")", debuglog_escape(r->pool, (char *)r->content_type));
                return OK;
            }
            break;
            
        case AUDITLOG_RELEVANT_ONLY :
            modsec_message = ap_table_get(r->headers_in, NOTE_MESSAGE);
            if ((modsec_message == NULL)||(ap_table_get(_r->notes, NOTE_NOAUDITLOG) != NULL)) {
                sec_debug_log(r, 2, "Audit log: ignoring a non-relevant request (content-type = \"%s\")", debuglog_escape(r->pool, (char *)r->content_type));
                return OK;
            }
            break;
    }
            
    /* return immediatelly if we don't have a file to write to */
    if (dcfg->auditlog_fd == -1) {
        sec_debug_log(r, 1, "Audit log not found for URI \"%s\"", debuglog_escape(r->pool, r->uri));
        return OK;
    }
    
    /* ok, we now need to determine whether this is a POST request
     * or not; looking at r->method_number is not enough in cases
     * when ErrorDocument is used (it contains M_GET)
     */
    if (r->method_number == M_POST) is_post = 1;
    else
    if (strncmp(r->the_request, r->method, strlen(r->method)) != 0) is_post = 1;
    
    if (r->connection->remote_logname == NULL) {
        remote_user = "-";
    }
    else {
        remote_user = r->connection->remote_logname;
    }

    if (r->connection->user == NULL) {
        local_user = "-";
    }
    else {
        local_user = r->connection->user;
    }
    
    /* before allocating the first buffer, determine the size
     * of data; start with a reasonable number for the data we
     * ourselves produce and the overhead, add the_request twice,
     * and input headers
     */
    o1size = 1024 + strlen(remote_user) + strlen(local_user);
    o1size += strlen(escaped_the_request) * 3; /* the request line is used twice, once as is and once escaped*/
    arr = ap_table_elts(r->headers_in);
    te = (table_entry *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        o1size += strlen(te[i].key);
        o1size += strlen(te[i].val);
        o1size += 5;
    }

    o1 = ap_palloc(r->pool, o1size + 1);
    if ((o1 == NULL)||(o1size + 1 == 0)) {
        sec_debug_log(r, 1, "sec_logger: Could not allocate output buffer #1 [asked for %lu]", o1size + 1);
        return DECLINED;
    }
    
    strcpy(o1, "");
    strncat(o1, "========================================\n", o1size);
    
    t = (char *)get_env_var(r, "UNIQUE_ID");
    if (t != NULL) {
        t = ap_psprintf(r->pool, "UNIQUE_ID: %s\n", t);
        strncat(o1, t, o1size - strlen(o1));
    }

    /* TODO use r->request_time */
    t = ap_psprintf(r->pool, "Request: %s %s %s [%s] \"%s\" %i %li\n",
                     r->connection->remote_ip, remote_user, local_user,
                     current_logtime(r), debuglog_escape(r->pool, escaped_the_request), r->status,
                     r->bytes_sent);
             
    strncat(o1, t, o1size - strlen(o1));

    t = ap_psprintf (r->pool, "Handler: %s\n", r->handler);
    strncat(o1, t, o1size - strlen(o1));
    
    /* see if there is an error message stored in notes */
    t = (char *)ap_table_get(r->notes, "error-notes");
    if (t != NULL) {
        /* TODO handle newline character in the message */
        t = ap_psprintf(r->pool, "Error: %s\n", t);
        strncat(o1, t, o1size - strlen(o1));
    }

    strncat(o1, "----------------------------------------\n", o1size - strlen(o1));

    /* request line */
    t = ap_psprintf(r->pool, "%s\n", escaped_the_request);
    strncat(o1, t, o1size - strlen(o1));

    /* input headers */
    arr = ap_table_elts(r->headers_in);
    te = (table_entry *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = ap_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o1, t, o1size - strlen(o1));
    }

    strncat(o1, "\n", o1size - strlen(o1));
    
    /* determine the size of the second buffer */
    o2size = 1024;
    if (r->status_line != NULL) o2size += strlen(r->status_line);
    
    arr = ap_table_elts(r->headers_out);
    te = (table_entry *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        o2size += strlen(te[i].key);
        o2size += strlen(te[i].val);
        o2size += 5;
    }
    
    arr = ap_table_elts(r->err_headers_out);
    te = (table_entry *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        o2size += strlen(te[i].key);
        o2size += strlen(te[i].val);
        o2size += 5;
    }

    o2 = ap_palloc(r->pool, o2size + 1);
    if ((o2 == NULL)||(o2size + 1 == 0)) {
        sec_debug_log(r, 1, "sec_logger: Could not allocate output buffer #2 [asked for %lu]", o2size + 1);
        return DECLINED;
    }

    if (is_post) {
        strcpy(o2, "\n\n");
    }
    else {
        strcpy(o2, "");
    }

    /* status line */
    if (r->status_line != NULL) t = ap_psprintf(r->pool, "%s %s\n", r->protocol, r->status_line);
    else t = ap_psprintf(r->pool, "%s\n", r->protocol);
    strncat(o2, t, o2size - strlen(o2));

    /* output headers */
    arr = ap_table_elts(r->headers_out);
    te = (table_entry *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = ap_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o2, t, o2size - strlen(o2));
    }
    
    arr = ap_table_elts(r->err_headers_out);
    te = (table_entry *)arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = ap_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o2, t, o2size - strlen(o2));
    }

    /* write to the file */
    fd_lock(r, dcfg->auditlog_fd);
    
    write(dcfg->auditlog_fd, o1, strlen(o1));
    
    if (is_post) {
        request_body *rb = (request_body *)ap_table_get(_r->notes, NOTE);
        modsec_rec *msr = (modsec_rec *)ap_table_get(_r->notes, NOTE_MSR);
        int body_action = 0; /* not available by default */
        char *message = NULL, *filename = NULL;

        if ((rb != NULL)&&(msr != NULL)) {
            if ((msr->mpd == NULL)||(msr->mpd->tmp_file_name == NULL)) body_action = 1; /* write from memory */
            else {
                if (multipart_contains_files(msr->mpd) == 0) {
                    body_action = 1;
                } else {
                    /* we tell the multipart cleaner not to erase the file
                     * and we reference the existing file in the audit log
                     */
                    msr->mpd->create_tmp_file = MULTIPART_TMP_FILE_CREATE_LEAVE;
                    body_action = 2;
                    sec_debug_log(r, 4, "sec_logger: telling multipart_cleanup not to delete the request file \"%s\"", debuglog_escape(r->pool, msr->mpd->tmp_file_name));
                    
                    filename = strrchr(msr->mpd->tmp_file_name, '/');
                    if (filename == NULL) filename = msr->mpd->tmp_file_name;
                    else filename = filename + 1;
                }
            }
        }
        
        message = NULL;
        switch(body_action) {
            case 0 :
                message = "[POST payload not available]";
                nbytes = strlen(message);
                break;
            case 1 :
                message = rb->buffer;
                nbytes = rb->length;
                break;
            case 2 :
                message = ap_psprintf(r->pool, "[@file:%s]", filename);
                nbytes = strlen(message);
                break;
        }
        
        if (message != NULL) {
            char *o3 = ap_psprintf(r->pool, "%lu\n", (unsigned long)nbytes);
            write(dcfg->auditlog_fd, o3, strlen(o3));
            write(dcfg->auditlog_fd, message, nbytes);
        }
    }

    /* write the second part of the log */
    write(dcfg->auditlog_fd, o2, strlen(o2));
    fd_unlock(r, dcfg->auditlog_fd);

    return OK;
}

int sec_check_access(request_rec *r) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(r->server->module_config, &security_module);
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    int real_action, real_status;
    modsec_rec *msr;
    int rc;
    
    /* Finish the configuration process on the fly */
    sec_set_dir_defaults(dcfg);

    sec_debug_log(r, 2, "sec_check_access [path=%s]", debuglog_escape(r->pool, dcfg->path));
    if (is_filtering_on_here(r, dcfg) == 0) return DECLINED;

    msr = (modsec_rec *)ap_pcalloc(r->pool, sizeof(*msr));
    if (msr == NULL) {
        sec_debug_log(r, 1, "sec_check_access: Unable to allocate %i bytes", sizeof(*msr));
        return DECLINED;
    }
    msr->r = r;
    msr->scfg = scfg;
    msr->dcfg = dcfg;
    msr->_the_request = NULL;
    msr->_post_payload = NULL;
    msr->parsed_args = ap_make_table(r->pool, 10);
    msr->parsed_cookies = ap_make_table(r->pool, 10);
    
    /* store for sec_logger to find */
    ap_table_setn(r->notes, NOTE_MSR, (char *)msr);
    
    /* Initialize mod_security structures for this request.
     * As of 1.8.6 non-fatal default actions are not allowed
     * during the initialization phase.
     */
    real_action = msr->dcfg->action->action;
    real_status = msr->dcfg->action->status;
    if (msr->dcfg->action->action == ACTION_NONE) {
        msr->dcfg->action->action = ACTION_DENY;
    }
    if (msr->dcfg->action->status == 0) {
        msr->dcfg->action->status = HTTP_FORBIDDEN;
    }
    rc = sec_initialize(msr);
    msr->dcfg->action->action = real_action;
    msr->dcfg->action->status = real_status;
    
    /* Process rules only if there were no errors
     * in the initialization stage.
     */
    if (rc == DECLINED) {
        rc = sec_check_all_signatures(msr);
    }
    
    /* make a note for the logger */
    if (rc != DECLINED) {
        char *note = ap_psprintf(r->pool, "%i", rc);
        ap_table_setn(r->headers_in, NOTE_ACTION, note);
        ap_table_setn(r->subprocess_env, NOTE_ACTED, "1");
    } else {
        ap_table_unset(r->headers_in, NOTE_ACTION);
    }
    
    return rc;
}

char *current_logtime(request_rec *r) {
    int timz;
    struct tm *t;
    char tstr[100];
    char sign;

    t = ap_get_gmtoff(&timz);
    sign = (timz < 0 ? '-' : '+');
    if (timz < 0) {
        timz = -timz;
    }

    strftime(tstr, 80, "%d/%b/%Y:%H:%M:%S ", t);
    ap_snprintf(tstr + strlen(tstr), 80 - strlen(tstr), "%c%.2d%.2d", sign, timz / 60, timz % 60);
    return ap_pstrdup(r->pool, tstr);
}

char *current_filetime(request_rec *r) {
    int timz;
    struct tm *t;
    char tstr[100];
    char sign;

    t = ap_get_gmtoff(&timz);
    sign = (timz < 0 ? '-' : '+');
    if (timz < 0) {
        timz = -timz;
    }

    strftime(tstr, 80, "%Y%m%d-%H%M%S", t);
    return ap_pstrdup(r->pool, tstr);
}

/*
 * This function only escapes the quotation
 * mark when it appears in the string. The name
 * of the function is not appropriate any more
 * and it will be changed in one of the future
 * releases.
 */
char *debuglog_escape(pool *p, char *text) {
    const unsigned char *s = NULL;
    unsigned char *d = NULL;
    char *ret = NULL;
    
    if (text == NULL) return NULL;
    
    ret = ap_palloc(p, strlen(text) * 2 + 1);
    if (ret == NULL) return NULL;
    s = (const unsigned char *)text;
    d = (unsigned char *)ret;

    while (*s != 0) {
        if (*s == '"') {
            *d++ = '"';
            *d++ = '"';
        } else {
            *d++ = *s;
        }
        s++;
    }
    *d = 0;    
    
    return ret;
}

static const char c2x_table[] = "0123456789abcdef";

static unsigned char *c2x(unsigned what, unsigned char *where) {
    what = what & 0xff;
    *where++ = c2x_table[what >> 4];
    *where++ = c2x_table[what & 0x0f];
    return where;
}

char *real_debuglog_escape(pool *p, char *text) {
    const unsigned char *s = NULL;
    unsigned char *d = NULL;
    char *ret = NULL;
    
    if (text == NULL) return NULL;
    
    ret = ap_palloc(p, strlen(text) * 4 + 1);
    if (ret == NULL) return NULL;
    s = (const unsigned char *)text;
    d = (unsigned char *)ret;
    
    while (*s != 0) {
        if ((*s <= 0x1f)||(*s >= 0x7f)) {
            *d++ = '\\';
            *d++ = 'x';
            c2x(*s, d);
            d += 2;
        }
        else {
            switch(*s) {
                case '\b' :
                    *d++ = '\\';
                    *d++ = 'b';
                    break;
                case '\n' :
                    *d++ = '\\';
                    *d++ = 'n';
                    break;
                case '\r' :
                    *d++ = '\\';
                    *d++ = 'r';
                    break;
                case '\t' :
                    *d++ = '\\';
                    *d++ = 't';
                    break;
                case '\v' :
                    *d++ = '\\';
                    *d++ = 'v';
                    break;
                case '\\' :
                    *d++ = '\\';
                    *d++ = '\\';
                    break;
                default :
                    *d++ = *s;
                    break;
            }
        }
        s++;
    }
    *d = 0;
    
    return ret;
}

void sec_debug_log(request_rec *r, int level, const char *text, ...) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    va_list ap;
    char str1[1024] = "";
    char str2[1256] = "";
    
    /* Return immediately if we don't have where to write
     * or if the log level of the message is higher than
     * wanted in the log.
     */
    if ((level != 1)&&( (dcfg->debuglog_fd == -1) || (level > dcfg->filter_debug_level) )) return;

    va_start(ap, text);
    
    ap_vsnprintf(str1, sizeof(str1), text, ap);
    ap_snprintf(str2, sizeof(str2), "[%s] [%s/sid#%lx][rid#%lx][%s] %sN", current_logtime(r), ap_get_server_name(r), (unsigned long)(r->server), (unsigned long)r, r->uri, str1); 

    if ((dcfg->debuglog_fd != -1)&&(level <= dcfg->filter_debug_level)) {
        char *escaped_str2 = real_debuglog_escape(r->pool, str2);
        if ((escaped_str2 != NULL)&&(strlen(escaped_str2) != 0)) {
            /* convert the last character in the string into a new line */
            escaped_str2[strlen(escaped_str2) - 1] = '\n';
            
            fd_lock(r, dcfg->debuglog_fd);
            write(dcfg->debuglog_fd, escaped_str2, strlen(escaped_str2));
            fd_unlock(r, dcfg->debuglog_fd);
        }
    }
    
    if (level == 1) {
        char *unique_id = (char *)get_env_var(r, "UNIQUE_ID");
        char *hostname = (char *)r->hostname;
        
        if (unique_id != NULL) unique_id = ap_psprintf(r->pool, " [unique_id %s]", unique_id);
        else unique_id = "";
        
        if (hostname != NULL) hostname = ap_psprintf(r->pool, " [hostname \"%s\"]", debuglog_escape(r->pool, hostname));
        else hostname = "";

        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r->server, "[client %s] mod_security: %s%s [uri \"%s\"]%s", r->connection->remote_ip, str1, hostname, debuglog_escape(r->pool, r->unparsed_uri), unique_id);
    }        

    va_end(ap);
    return;
}

#if !(defined(WIN32)||defined(NETWARE))

int create_chroot_lock(server_rec *s, pool *p, char *lockfilename) {
    char buf[260] = "";
    int handle;
    
    handle = open(lockfilename, O_CREAT | O_TRUNC | O_RDWR | O_BINARY);
    if (handle == -1) {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, s, "mod_security: unable to create chroot lock \"%s\", errno=%d(%s)", lockfilename, errno, strerror(errno));
        return -1;
    }
    snprintf(buf, 255, "%i\n", getpid());
    if (write(handle, buf, strlen(buf)) != strlen(buf)) {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, s, "mod_security: error writing to the chroot lock file: \"%s\"", lockfilename);
        close(handle);
        return -1;
    }
    close(handle);
    
    return 1;
}

int is_time_to_chroot(server_rec *s, pool *p) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);
    int handle;
    char buf[260] = "";
    int ppid = getppid();

    /* always chroot if the ppid is 1 */
    if (ppid == 1) {
        unlink(scfg->chroot_lock);
        return 1;
    }    
    
    handle = open(scfg->chroot_lock, O_RDONLY | O_BINARY);
    if (handle == -1) {
        /* the file does not exist, this is probably the
         * first time we are called
         */
        if (create_chroot_lock(s, p, scfg->chroot_lock) < 0) return -1;
        
        /* not yet the right time for chroot */
        return 0;
    } else {
        /* the file exists, read the pid to see if it is
         * the right time to perform chroot
         */
         int i = read(handle, buf, 255);
         if (i >= 0) {
             if (i > 255) i = 255;
             buf[i] = 0;
         }
         close(handle);

         if (atoi(buf) == ppid) {
            /* the pid in the file matches our parent, we should chroot */
            unlink(scfg->chroot_lock);
            return 1;
         } else {
            /* the pid does not match our parent so
             * we assume the file was not erased when
             * it was supposed to - create the file again
             * then
             */
            if (create_chroot_lock(s, p, scfg->chroot_lock) < 0) return -1;
            return 0;
         }
    }
}

#endif

void sec_init(server_rec *s, pool *p) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);
    
    if (scfg->server_response_token) {    
        ap_add_version_component("mod_security/" MODULE_RELEASE);
    }

    /* Although this may seem not to make any sense we are
     * in fact allocating sufficient space in the signature
     * so we can change it later by using brute force.
     */    
    if (scfg->server_signature != NULL) {
        ap_add_version_component(scfg->server_signature);
    }
    
    change_server_signature(s, scfg);

    #if !(defined(WIN32)||defined(NETWARE))
    if (scfg->chroot_dir != NULL) {
        int rv;
    
        /* We do this here because modules are normally
         * initialised twice, and, as far as I could tell,
         * you can't keep any information between initialisations
         * to tell yourself whether the module is initalised for
         * the first or for the second time. Of course, calling
         * chroot during first initialisation is no good. However,
         * it seems that the parent pid of the Apache process is
         * always 1 during the second initialisation. So, fingers
         * crossed.
         */
        
        rv = is_time_to_chroot(s, p);
        if (rv < 0) {
            /* Error already reported */
            exit(1);
        }
        
        if (rv == 1) {
            ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, s, "mod_security: chroot checkpoint #2 (pid=%i ppid=%i)", getpid(), getppid());
        
            if (chdir(scfg->chroot_dir) < 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, s, "mod_security: chroot failed, unable to chdir to %s, errno=%d(%s)", scfg->chroot_dir, errno, strerror(errno));
                exit(1);
            }
        
            if (chroot(scfg->chroot_dir) < 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, s, "mod_security: chroot failed, path=%s, errno=%d(%s)", scfg->chroot_dir, errno, strerror(errno));
                exit(1);
            }
            
            if (chdir("/") < 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, s, "mod_security: chroot failed, unable to chdir to /, errno=%d(%s)", errno, strerror(errno));
                exit(1);
            }
            
            ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, s, "mod_security: chroot successful, path=%s", scfg->chroot_dir);
            scfg->chroot_completed = 1;
        } else {
            ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, s, "mod_security: chroot checkpoint #1 (pid=%i ppid=%i)", getpid(), getppid());
        }
    }
                                                
    #endif    
}

void sec_child_init(server_rec *s, pool *p) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);

    #if !(defined(WIN32)||defined(NETWARE))
    
    /* Refuse to work if chroot was requested but
     * not performed. Unfortunately, we can't perform
     * this check earlier, or at a better location.
     */
    if ((scfg->chroot_dir != NULL)&&(scfg->chroot_completed == 0)) {
        ap_log_error(APLOG_MARK, APLOG_EMERG | APLOG_NOERRNO, s, "mod_security: chroot requested but not completed! Exiting.");
        /* This is ugly but better than running a server
         * without a chroot when a chroot was configured.
         * We sleep a little (one sec) to prevent children
         * from dying too fast.
         */
        sec_sleep(1000);
        exit(1);
    }
    
    #endif
    
    change_server_signature(s, scfg);
}

#ifdef WIN32
#define USE_LOCKING
#endif

#ifndef USE_FCNTL
#ifndef USE_FLOCK
#ifndef USE_LOCKING
#ifndef NETWARE
#define USE_FCNTL
#endif
#endif
#endif
#endif

#ifdef USE_FCNTL
static struct flock   lock_it;
static struct flock unlock_it;
#endif

int fd_lock(request_rec *r, int fd) {
    int rc = 0;

    #ifdef USE_FCNTL
    lock_it.l_whence = SEEK_SET;
    lock_it.l_start = 0;
    lock_it.l_len = 0;
    lock_it.l_type = F_WRLCK;
    lock_it.l_pid = 0;

    while ( ((rc = fcntl(fd, F_SETLKW, &lock_it)) < 0) && (errno == EINTR) ) {
        continue;
    }
    #endif

    #ifdef USE_FLOCK
    while ( ((rc = flock(fd, LOCK_EX)) < 0) && (errno == EINTR) ) {
        continue;
    }
    #endif

    #ifdef USE_LOCKING
    lseek(fd, 0, SEEK_SET);
    rc = _locking(fd, _LK_LOCK, 1);
    lseek(fd, 0, SEEK_END);
    #endif

    #ifdef NETWARE
    if ((locking_sem != 0) && (TimedWaitOnLocalSemaphore(locking_sem, 10000) != 0)) rc = -1;
    else rc = 1;
    #endif

    if (rc < 0) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Locking failed %i [%s]", rc, strerror(errno));
        return -1;
    }
    
    return 1;
}

int fd_unlock(request_rec *r, int fd) {
    int rc = 0;

    #ifdef USE_FCNTL
    unlock_it.l_whence = SEEK_SET;
    unlock_it.l_start  = 0;
    unlock_it.l_len    = 0;
    unlock_it.l_type   = F_UNLCK;
    unlock_it.l_pid    = 0;

    while( ((rc = fcntl(fd, F_SETLKW, &unlock_it)) < 0) && (errno == EINTR) ) {
        continue;
    }
    #endif
    
    #ifdef USE_FLOCK
    while ( ((rc = flock(fd, LOCK_UN)) < 0) && (errno == EINTR) ) {
        continue;
    }
    #endif
    
    #ifdef USE_LOCKING
    lseek(fd, 0, SEEK_SET);
    rc = _locking(fd, _LK_UNLCK, 1);
    lseek(fd, 0, SEEK_END);
    #endif
    
    #ifdef NETWARE
    if (locking_sem) SignalLocalSemaphore(locking_sem);
    rc = 1;
    #endif

    if (rc < 0) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Unlocking failed %i", rc);
        return -1;
    }
    
    return 1;
}

char *get_temp_folder(void) {
    char *filename = getenv("TEMP");    
    if (filename != NULL) return filename;
    
    filename = getenv("TMP");
    if (filename != NULL) return filename;
    
    #if defined NETWARE
    return("sys:/tmp/");
    #elif defined WIN32
    return("");
    #else
    return("/tmp/");
    #endif
}

int multipart_finish(multipart_data *mpd) {
    sec_debug_log(mpd->r, 4, "multipart_finish");

    if (mpd->create_tmp_file != MULTIPART_TMP_FILE_NONE) {
        /* close the file */
        close(mpd->tmp_file_fd);
        
        /* delete the file if we are supposed to delete it */
        if (mpd->create_tmp_file == MULTIPART_TMP_FILE_CREATE) {
            if ((mpd->tmp_file_name != NULL)&&(unlink(mpd->tmp_file_name) < 0)) {
                sec_debug_log(mpd->r, 1, "multipart_finish: Failed to delete file (request) \"%s\" because %d(%s)", debuglog_escape(mpd->r->pool, mpd->tmp_file_name), errno, strerror(errno));
            } else {
                sec_debug_log(mpd->r, 2, "multipart_finish: Deleted file (request) \"%s\"", debuglog_escape(mpd->r->pool, mpd->tmp_file_name));
            }
        }
        
        /* otherwise leave the file where it is
         * MULTIPART_TMP_FILE_CREATE_LEAVE
         */
    }

    /* loop through the list of parts
     * and delete all temporary files, but only if
     * keeping the files is not requested
     */
    if (mpd->dcfg->upload_keep_files == 0) {
        multipart_part **parts;
        int i;
        
        parts = (multipart_part **)mpd->parts->elts;
        for(i = 0; i < mpd->parts->nelts; i++) {
            if (parts[i]->type == MULTIPART_FILE) {
                if (parts[i]->tmp_file_name != NULL) {
                    sec_debug_log(mpd->r, 4, "multipart_finish: deleting temporary file (part) [%s]", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name));
                    if (unlink(parts[i]->tmp_file_name) < 0) {
                        sec_debug_log(mpd->r, 1, "multipart_finish: Failed to delete file (part) \"%s\" because %d(%s)", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name), errno, strerror(errno));
                    } else {
                        sec_debug_log(mpd->r, 2, "multipart_finish: Deleted file (part) \"%s\"", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name));
                    }
                }
            }
        }
    } else {
        /* delete empty files */
        multipart_part **parts;
        int i;
        
        parts = (multipart_part **)mpd->parts->elts;
        for(i = 0; i < mpd->parts->nelts; i++) {
            if ((parts[i]->type == MULTIPART_FILE)&&(parts[i]->tmp_file_size == 0)) {
                if (parts[i]->tmp_file_name != NULL) {
                    sec_debug_log(mpd->r, 4, "multipart_finish: deleting temporary file (part) [%s]", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name));
                    if (unlink(parts[i]->tmp_file_name) < 0) {
                        sec_debug_log(mpd->r, 1, "multipart_finish: Failed to delete empty file (part) \"%s\" because %d(%s)", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name), errno, strerror(errno));
                    } else {
                        sec_debug_log(mpd->r, 2, "multipart_finish: Deleted empty file (part) \"%s\"", debuglog_escape(mpd->r->pool, parts[i]->tmp_file_name));
                    }
                }
            }
        }
    }
    
    return 1;
}

int multipart_init(multipart_data *mpd, request_rec *r) {
    mpd->dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    mpd->p = r->pool;
    mpd->r = r;
    
    if (mpd->create_tmp_file != MULTIPART_TMP_FILE_NONE) {
        char *folder = NULL;
        
        if (mpd->dcfg->upload_dir != NULL) folder = mpd->dcfg->upload_dir;
        else folder = get_temp_folder();

        mpd->tmp_file_name = ap_psprintf(r->pool, "%s/%s-%s-request_body-XXXXXX", folder, current_filetime(mpd->r), mpd->r->connection->remote_ip);
        if (mpd->tmp_file_name == NULL) {
            sec_debug_log(r, 1, "multipart_init: Memory allocation failed");
            return -1;
        }
    
        mpd->tmp_file_fd = sec_mkstemp(mpd->tmp_file_name);        
        if (mpd->tmp_file_fd < 0) {
            sec_debug_log(r, 1, "multipart_init: Failed to create file [%s] because %d(%s)", debuglog_escape(mpd->r->pool, mpd->tmp_file_name), errno, strerror(errno));
            return -1;
        }
    }
    
    mpd->parts = ap_make_array(mpd->p, 10, sizeof(multipart_part *));
    mpd->bufleft = MULTIPART_BUF_SIZE;
    mpd->bufptr = mpd->buf;
    mpd->buf_contains_line = 1;
    mpd->mpp = NULL;

    /* schedule resource cleanup for later */
    ap_register_cleanup(r->pool, (void *)mpd, (void (*)(void *))multipart_finish, ap_null_cleanup);
    
    return 1;
}

int multipart_process_chunk(multipart_data *mpd, char *buf, int size) {
    char *inptr = buf;
    unsigned int inleft = size;

    /* write the data to a temporary file if we are
     * required to do so
     */
    if (mpd->create_tmp_file != MULTIPART_TMP_FILE_NONE) {
        if (write(mpd->tmp_file_fd, buf, size) != size) {
            sec_debug_log(mpd->r, 1, "multipart_process_chunk: write to tmp file failed");
            return -1;
        }
    }

    /* here we loop through all the data available, byte by byte */
    while(inleft > 0) {
        char c = *inptr++;
        inleft = inleft - 1;
        
        *(mpd->bufptr) = c;
        mpd->bufptr++;
        mpd->bufleft--;

        /* until we either reach the end of the line
         * or the end of our internal buffer        
         */
        if ((c == 0x0a)||(mpd->bufleft == 0)) {
            *(mpd->bufptr) = 0;
            
            /* boundary preconditions: length of the line greater than
             * the length of the boundary + the first two characters
             * are dashes "-"
             */
            if ( mpd->buf_contains_line
                && (strlen(mpd->buf) > strlen(mpd->boundary) + 2)
                && (((*(mpd->buf) == '-'))&&(*(mpd->buf + 1) == '-')) ) {
                    
                if (strncmp(mpd->buf + 2, mpd->boundary, strlen(mpd->boundary)) == 0) {
                    if (multipart_process_boundary(mpd) < 0) return -1;
                }
            }
            else {
                if (multipart_process_data_chunk(mpd) < 0) return -1;
            }

            /* reset the pointer to the beginning of the buffer
             * and continue to accept input data            
             */
            mpd->bufptr = mpd->buf;
            mpd->bufleft = MULTIPART_BUF_SIZE;
            mpd->buf_contains_line = (c == 0x0a) ? 1 : 0;
        }
    }
    
    return 1;
}

char *multipart_construct_filename(multipart_data *mpd) {
    char c, *p, *q = mpd->mpp->filename;
    char *filename;

    /* find the last forward slash and consider the
     * filename to be only what's right from it
     */
    while((p = strstr(q, "\\")) != NULL) {
        q = p + 1;
    }
    
    /* do the same for the slash character, just in case */
    while((p = strstr(q, "/")) != NULL) {
        q = p + 1;
    }

    /* allow letters, digits and dots, replace
     * everything else with underscores    
     */
    p = filename = ap_pstrdup(mpd->p, q);
    while((c = *p) != 0) {    
        if ( !(isalnum(c)||(c == '.')) ) *p = '_';
        p++;
    }
    
    return filename;
}

int multipart_process_data_chunk(multipart_data *mpd) {
    sec_debug_log(mpd->r, 9, "multipart_process_data_chunk: state=%i, size=%i", mpd->mpp_state, MULTIPART_BUF_SIZE - mpd->bufleft);

    /* while mpp_state equals zero we are still reading
     * part headers
     */
    if (mpd->mpp_state == 0) {
        if (mpd->mpp == NULL) {
            /* the first line in the request must be a line
             * with a boundary but nothing prevents a malicuos
             * user to submit an invalid body
             */
            sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: data found but no boundary");
            /* return -1; */
        }
        else if (mpd->buf[1] == 0x0a) {
            sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: empty line, going to state 1");
            mpd->mpp_state = 1;
            
            /* check that we have all the information
             * we must have the name of the field
             */
            if (mpd->mpp->name == NULL) {
                sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: part name unknown");
                return -1;
            }
        }
        else {
            /* figure out what kind of part we have */
            if (strncasecmp(mpd->buf, "content-disposition: form-data", 30) == 0) {
                char *p1, *p2;
                
                p1 = strstr(mpd->buf + 30, "name=\"");
                if (p1 != NULL) {
                    p2 = p1 = p1 + 6;
                    while((*p2 != 0)&&(*p2 != '"')) p2++;
                    
                    mpd->mpp->name = ap_pcalloc(mpd->p, p2 - p1 + 1);
                    memcpy(mpd->mpp->name, p1, p2 - p1);
                    sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: got attribute name \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->name));
                }
                                
                p1 = strstr(mpd->buf + 30, "filename=\"");
                if (p1 != NULL) {
                    p2 = p1 = p1 + 10;
                    while((*p2 != 0)&&(*p2 != '"')) p2++;
                    
                    mpd->mpp->filename = ap_pcalloc(mpd->p, p2 - p1 + 1);
                    memcpy(mpd->mpp->filename, p1, p2 - p1);
                    sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: got attribute filename \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->filename));
                    mpd->mpp->type = MULTIPART_FILE;
                }
                
            }
            /* get the content type */
            else if (strncasecmp(mpd->buf, "content-type:", 13) == 0) {
                char *p;
                int i = 13;
                
                if (*(mpd->buf + 13) == 0x20) i = 14;
                p = mpd->mpp->content_type = ap_pstrdup(mpd->p, mpd->buf + i);
                
                /* get rid of that stuff at the end */
                p = mpd->mpp->content_type;
                while(*p != 0) {
                    if ((*p == 0x0a)||(*p == 0x0d)) *p = 0;
                    p++;
                }
                    
                sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: got content_type for part \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->content_type));
            }
            else {
                /* ignore header */
                sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: ignoring header \"%s\"", debuglog_escape(mpd->r->pool, mpd->buf));
            }
        }
    }
    else {
        char *p = mpd->buf + (MULTIPART_BUF_SIZE - mpd->bufleft) - 2;
        char localreserve[2];
        int bytes_reserved = 0;
        
        /* preserve the last two bytes for later */
        if (MULTIPART_BUF_SIZE - mpd->bufleft >= 2) {
            bytes_reserved = 1;
            localreserve[0] = *p;
            localreserve[1] = *(p + 1);
            mpd->bufleft += 2;
            *p = 0;
        }
        
        /* add data to the part we are building */
        if (mpd->mpp->type == MULTIPART_FILE) {
        
            /* only store individual files on disk if we are going
             * to keep them or if we need to have them approved later
             */
            if ((mpd->dcfg->upload_approve_script != NULL)||(mpd->dcfg->upload_keep_files > 0)) {
            
            /* first create a temporary file if we don't have it */
            if (mpd->mpp->tmp_file_fd == 0) {
                char *filename = multipart_construct_filename(mpd);
                
                /* the temp folder must be chosen in the configuration 
                 * create the filename first
                 */
                if (mpd->dcfg->upload_dir != NULL) {
                    mpd->mpp->tmp_file_name = ap_psprintf(mpd->p, "%s/%s-%s-%s", mpd->dcfg->upload_dir, current_filetime(mpd->r), mpd->r->connection->remote_ip, filename);
                }
                else {
                    mpd->mpp->tmp_file_name = ap_psprintf(mpd->p, "%s/%s-%s-%s", get_temp_folder(), current_filetime(mpd->r), mpd->r->connection->remote_ip, filename);
                }
                
                if ((mpd->mpp->tmp_file_fd = open(mpd->mpp->tmp_file_name, O_WRONLY | O_APPEND | O_CREAT | O_BINARY, CREATEMODE)) == -1) {
                    /* we've failed while opening the page, so we'll try
                     * again with a more unique filename
                     */
                    mpd->mpp->tmp_file_name = ap_pstrcat(mpd->p, mpd->mpp->tmp_file_name, "_XXXXXX", NULL);
                    mpd->mpp->tmp_file_fd = sec_mkstemp(mpd->mpp->tmp_file_name);
                }
        
                /* do we have an opened file? */
                if (mpd->mpp->tmp_file_fd < 0) {
                    sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: Failed to create file \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->tmp_file_name));
                    return -1;
                }
            }
        
            /* write data to the file */
            if (mpd->reserve[0] == 1) {
                if (write(mpd->mpp->tmp_file_fd, &mpd->reserve[1], 2) != 2) {
                    sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: writing to \"%s\" failed.", debuglog_escape(mpd->r->pool, mpd->mpp->tmp_file_name));
                }
                mpd->mpp->tmp_file_size += 2;
            }
            if (write(mpd->mpp->tmp_file_fd, mpd->buf, MULTIPART_BUF_SIZE - mpd->bufleft) != MULTIPART_BUF_SIZE - mpd->bufleft) {
                sec_debug_log(mpd->r, 1, "multipart_process_data_chunk: writing to \"%s\" failed.", debuglog_escape(mpd->r->pool, mpd->mpp->tmp_file_name));
            }
            mpd->mpp->tmp_file_size += (MULTIPART_BUF_SIZE - mpd->bufleft);

            }            
        }
        else if (mpd->mpp->type == MULTIPART_FORMDATA) {
            
            /* add data to the value we keep in memory */
            if (mpd->mpp->value == NULL) {
                mpd->mpp->value = ap_pstrdup(mpd->p, mpd->buf);
            }
            else {
                if (mpd->reserve[0] == 1) {
                    mpd->mpp->value = ap_pstrcat(mpd->p, mpd->mpp->value, &(mpd->reserve[1]), mpd->buf, NULL);
                }
                else {
                    mpd->mpp->value = ap_pstrcat(mpd->p, mpd->mpp->value, mpd->buf, NULL);
                }
            }
            
            sec_debug_log(mpd->r, 9, "Formdata variable value \"%s\"", debuglog_escape(mpd->r->pool, mpd->mpp->value));
        }
        else {
            sec_debug_log(mpd->r, 4, "multipart_process_data_chunk: unknown part type %i", mpd->mpp->type);
            return -1;
        }
        
        /* store the reserved bytes to the multipart
         * context so that they don't get lost
         */
        if (bytes_reserved) {
            mpd->reserve[0] = 1;
            mpd->reserve[1] = localreserve[0];
            mpd->reserve[2] = localreserve[1];
        }
        else {
            mpd->reserve[0] = 0;
        }
    }
    
    return 1;
}

int multipart_process_boundary(multipart_data *mpd) {
    sec_debug_log(mpd->r, 4, "multipart_process_boundary");

    /* if there was a part being built finish it */
    if (mpd->mpp != NULL) {
        /* close the temp file */
        if ((mpd->mpp->type == MULTIPART_FILE)&&(mpd->mpp->tmp_file_name != NULL)&&(mpd->mpp->tmp_file_fd != 0)) {
            close(mpd->mpp->tmp_file_fd);
        }

        /* add the part to the list of parts */
        *(multipart_part **)ap_push_array(mpd->parts) = mpd->mpp;
        sec_debug_log(mpd->r, 4, "multipart_process_boundary: added part %x to the list", mpd->mpp);
        mpd->mpp = NULL;
    }

    /* start building a new part */
    mpd->mpp = (multipart_part *)ap_pcalloc(mpd->p, sizeof(multipart_part));
    mpd->mpp->type = MULTIPART_FORMDATA;
    mpd->mpp_state = 0;    
    
    mpd->reserve[0] = 0;
    mpd->reserve[1] = 0;
    mpd->reserve[2] = 0;
    mpd->reserve[3] = 0;
    
    return 1;
}

int verify_uploaded_files(request_rec *r, multipart_data *mpd, char *approver_script, char **error_msg) {
    multipart_part **parts;
    int i;
    
    if (error_msg == NULL) return -1;
    *error_msg = NULL;
        
    parts = (multipart_part **)mpd->parts->elts;
    for(i = 0; i < mpd->parts->nelts; i++) {
        if (parts[i]->type == MULTIPART_FILE) {
            BUFF *p1, *p2, *p3;
            exec_data *ed = NULL;
            char buf[129];
            int j;
            
            ed = ap_pcalloc(r->pool, sizeof(exec_data));
            ed->r = r;
            ed->command = approver_script;
            ed->args = parts[i]->tmp_file_name;
            
            sec_debug_log(r, 4, "verify_uploaded_files: executing \"%s\" to verify \"%s\"", debuglog_escape(mpd->r->pool, ed->command), debuglog_escape(mpd->r->pool, ed->args));
            
            if (!ap_bspawn_child(r->main ? r->main->pool : r->pool, sec_exec_child, (void *)ed, kill_after_timeout, &p1, &p2, &p3)) {
                *error_msg = ap_psprintf(r->pool, "Couldn't spawn a child process \"%s\"", debuglog_escape(mpd->r->pool, approver_script));
                return -1;
            }

            /* read some output from the script */
            j = ap_bgets(buf, 128, p2);
            if (j > 0) {
                int k;
                
                buf[j] = 0;
                
                /* we only care about the first line of the result */
                for(k = 0; k < j; k++) if (buf[k] == 0x0a) buf[k] = 0;
                
                sec_debug_log(r, 4, "verify_uploaded_files: got result \"%s\"", debuglog_escape(mpd->r->pool, buf));
                
                /* and we also only care about the first character
                 * on the first line; unless that character is '1'
                 * we will not approve the file
                 */
                if (buf[0] != '1') {
                    *error_msg = ap_psprintf(r->pool, "File \"%s\" rejected by the approver script \"%s\"", debuglog_escape(mpd->r->pool, ed->args), debuglog_escape(mpd->r->pool, ed->command));
                    return 0;
                }
            }
                                   
            /* throw the rest away */
            while(ap_bgets(buf, 128, p2) > 0);
            while(ap_bgets(buf, 128, p3) > 0);
        }
    }
    
    return 1;
}

int multipart_get_variables(multipart_data *mpd, table *parsed_args, sec_dir_config *dcfg, char **error_msg) {
    multipart_part **parts;
    char *my_error_msg = NULL;
    int i;
    
    if (error_msg == NULL) return -1;
    *error_msg = NULL;
        
    parts = (multipart_part **)mpd->parts->elts;
    for(i = 0; i < mpd->parts->nelts; i++) {
        if (parts[i]->type == MULTIPART_FORMDATA) {
            char *name = NULL, *value = NULL;
            
            name = normalise_relaxed(mpd->r, dcfg, parts[i]->name, &my_error_msg);
            if (name == NULL) {
                *error_msg = ap_psprintf(mpd->r->pool, "Error normalizing parameter name: %s", my_error_msg);
                return -1;
            }
            
            value = normalise_relaxed(mpd->r, dcfg, parts[i]->value, &my_error_msg);
            if (value == NULL) {
                *error_msg = ap_psprintf(mpd->r->pool, "Error normalizing parameter value: %s", my_error_msg);
                return -1;
            }
            
            ap_table_add(parsed_args, name, value);
        }
    }
    
    return 1;
}

int multipart_contains_files(multipart_data *mpd) {
    multipart_part **parts;
    int i, file_count = 0;

    parts = (multipart_part **)mpd->parts->elts;
    for(i = 0; i < mpd->parts->nelts; i++) {
        if ((parts[i]->type == MULTIPART_FILE)&&(parts[i]->tmp_file_size != 0)) file_count++;
    }
    
    return file_count;
}

module MODULE_VAR_EXPORT security_module = {
    STANDARD_MODULE_STUFF,
    sec_init,                   /* module initializer                  */
    sec_create_dir_config,      /* create per-dir    config structures */
    sec_merge_dir_config,       /* merge  per-dir    config structures */
    sec_create_srv_config,      /* create per-server config structures */
    sec_merge_srv_config,       /* merge  per-server config structures */
    sec_cmds,                   /* table of config file commands       */
    NULL,                       /* [#8] MIME-typed-dispatched handlers */
    NULL,                       /* [#1] URI to filename translation    */
    NULL,                       /* [#4] validate user id from request  */
    NULL,                       /* [#5] check if the user is ok _here_ */
    NULL,                       /* [#3] check access by host address   */
    NULL,                       /* [#6] determine MIME type            */
    sec_check_access,           /* [#7] pre-run fixups                 */
    sec_logger,                 /* [#9] log a transaction              */
    NULL,                       /* [#2] header parser                  */
    sec_child_init,             /* child_init                          */
    NULL,                       /* child_exit                          */
    NULL                        /* [#0] post read-request              */
};
