/*
 * mod_security (apache 2.x), http://www.modsecurity.org/
 * Copyright (c) 2002,2003 Ivan Ristic <ivanr@webkreator.com>
 *
 * $Id: mod_security.c,v 1.47.2.6 2003/12/05 17:17:00 ivanr Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
 
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>

#if !(defined(WIN32) || defined(NETWARE))
#include <sys/types.h>
#include <unistd.h>
#include <unixd.h>
#endif

#include "ap_config.h"
#include "httpd.h"
#include "http_config.h"
#include "http_request.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "util_script.h"
#include "ap_mpm.h"

#include "apr.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_user.h"
#include "apr_lib.h"
#include "apr_signal.h"
#include "apr_global_mutex.h"

module AP_MODULE_DECLARE_DATA security_module;

static apr_global_mutex_t *modsec_debuglog_lock = NULL;
static apr_global_mutex_t *modsec_auditlog_lock = NULL;

static ap_filter_rec_t *global_sec_filter_in;
static ap_filter_rec_t *global_sec_filter_out;

#define MOD_SECURITY_SIGNATURE "mod_security/1.7.4"
#define AUDIT_BUF_SIZE 8192

#define MODSEC_SKIP                         -2000

#define UNICODE_ERROR_CHARACTERS_MISSING    -1
#define UNICODE_ERROR_INVALID_ENCODING      -2
#define UNICODE_ERROR_OVERLONG_CHARACTER    -3

#define NOT_SET                         -1
#define FILTERING_OFF                   0
#define FILTERING_ON                    1
#define FILTERING_DYNAMIC_ONLY          2

#define AUDITLOG_OFF                    0
#define AUDITLOG_ON                     1
#define AUDITLOG_DYNAMIC_OR_RELEVANT    2
#define AUDITLOG_RELEVANT_ONLY          3

#define ACTION_NONE                 0
#define ACTION_DENY                 1
#define ACTION_REDIRECT             2
#define ACTION_ALLOW                3
#define ACTION_SKIP                 4

#define VAR_ACTION_ALLOW            1
#define VAR_ACTION_DENY             0

#define VAR_UNKNOWN                 0
#define VAR_CUSTOM                  1
#define VAR_HEADER                  2
#define VAR_ENV                     3
#define VAR_ARGS                    4
#define VAR_POST_PAYLOAD            5
#define VAR_ARGS_NAMES              6
#define VAR_ARGS_VALUES             7
#define VAR_ARGS_SELECTIVE          8
#define VAR_OUTPUT                  9
#define VAR_COOKIES_NAMES           10
#define VAR_COOKIES_VALUES          11
#define VAR_COOKIE                  12
#define VAR_RESERVED_5              13
#define VAR_RESERVED_6              14
#define VAR_RESERVED_7              15
#define VAR_RESERVED_8              16
#define VAR_RESERVED_9              17
#define VAR_RESERVED_10             18
#define VAR_RESERVED_11             19
#define VAR_RESERVED_12             20

#define VAR_REMOTE_ADDR             21
#define VAR_REMOTE_HOST             22
#define VAR_REMOTE_USER             23   
#define VAR_REMOTE_IDENT            24
#define VAR_REQUEST_METHOD          25
#define VAR_SCRIPT_FILENAME         26
#define VAR_PATH_INFO               27
#define VAR_QUERY_STRING            28
#define VAR_AUTH_TYPE               29
#define VAR_DOCUMENT_ROOT           30
#define VAR_SERVER_ADMIN            31
#define VAR_SERVER_NAME             32
#define VAR_SERVER_ADDR             33
#define VAR_SERVER_PORT             34
#define VAR_SERVER_PROTOCOL         35
#define VAR_SERVER_SOFTWARE         36
#define VAR_TIME_YEAR               37
#define VAR_TIME_MON                38
#define VAR_TIME_DAY                39
#define VAR_TIME_HOUR               40
#define VAR_TIME_MIN                41
#define VAR_TIME_SEC                42
#define VAR_TIME_WDAY               43
#define VAR_TIME                    44
#define VAR_API_VERSION             45
#define VAR_THE_REQUEST             46
#define VAR_REQUEST_URI             47
#define VAR_REQUEST_FILENAME        48
#define VAR_IS_SUBREQ               49
#define VAR_HANDLER                 50

char *all_variables[] = {
    "UNKOWN",
    "CUSTOM",
    "HEADER",
    "ENV",
    "ARGS",
    "POST_PAYLOAD",
    "ARGS_NAMES",
    "ARGS_VALUES",
    "ARGS_SELECTIVE",
    "OUTPUT",
    "COOKIES_NAMES", // 10
    "COOKIES_VALUES",
    "COOKIE",
    "RESERVED_5",
    "RESERVED_6",
    "RESERVED_7",
    "RESERVED_8",
    "RESERVED_9",
    "RESERVED_10",
    "RESERVED_11",
    "RESERVED_12", // 20
    "REMOTE_ADDR",
    "REMOTE_HOST",
    "REMOTE_USER",
    "REMOTE_IDENT",
    "REQUEST_METHOD",
    "SCRIPT_FILENAME",
    "PATH_INFO",
    "QUERY_STRING",
    "AUTH_TYPE",
    "DOCUMENT_ROOT",
    "SERVER_ADMIN",
    "SERVER_NAME",
    "SERVER_ADDR",
    "SERVER_PORT",
    "SERVER_PROTOCOL",
    "SERVER_SOFTWARE",
    "TIME_YEAR",
    "TIME_MON",
    "TIME_DAY",
    "TIME_HOUR",
    "TIME_MIN",
    "TIME_SEC",
    "TIME_WDAY",
    "TIME",
    "API_VERSION",
    "THE_REQUEST",
    "REQUEST_URI",
    "REQUEST_FILENAME",
    "IS_SUBREQ",
    "HANDLER",
    NULL
};

typedef struct {
    char *name;
    int type;
    int action;
} variable;

typedef struct {
    request_rec *r;
    char *command;
} exec_data;

typedef struct {
    int log;
    int action;
    int status;
    char *redirect_url;
    int exec;
    char *exec_string;
    char *id;
    char *msg;
    char *pattern;
    regex_t *regex;
    int is_default_action_used;
    int is_selective;
    int is_negative;
    int is_allow;
    int is_output;
    int requires_parsed_args;
    apr_array_header_t *variables;
    int pause;
    int skip_count;
    int is_chained;
} signature;

typedef struct {
    int filter_engine;
    int scan_post;
    int scan_output;
    int is_action_default;
    signature action;
    apr_array_header_t *signatures;
    char *path;
    int auditlog_flag;
    char *auditlog_name;
    apr_file_t *auditlog_fd;
    int filter_debug_level;
    int filters_clear;
    int range_start;
    int range_end;
    int check_encoding;
    int check_unicode_encoding;
    char *scan_output_mimetypes;
} sec_dir_config;

typedef struct {
    int filter_engine;
    int scan_post;
    int scan_output;
    int is_action_default;
    signature action;
    apr_array_header_t *signatures;
    char *path;
    int auditlog_flag;
    char *auditlog_name;
    apr_file_t *auditlog_fd;
    int filter_debug_level;
    int server_response_token;
    char *debuglog_name;
    apr_file_t *debuglog_fd;
    int range_start;
    int range_end;
    int check_encoding;
    int check_unicode_encoding;
    char *chroot_dir;
    char *server_signature;
    char *scan_output_mimetypes;
} sec_srv_config;

typedef struct sec_filter_in_ctx_ {
    char *buffer;
    unsigned long int buflen; // max. size of the buffer
    unsigned long int bufleft; // space left in buffer
    unsigned int sofar; // data read sofar
    int access_check_performed;
    apr_bucket_brigade *pbbTmp;
    char *output_ptr; 
    unsigned long int output_sent;
    int done_reading;
    int done_writing;
} sec_filter_in_ctx;

typedef struct sec_filter_out_ctx_ {
    char *buffer;
    unsigned long int buflen;
    unsigned long int bufused;
    unsigned long int content_length;
    char *output_ptr; 
    unsigned long int output_sent;
    char *input_ptr;
    int done_reading;
    int done_writing;
} sec_filter_out_ctx;

typedef struct {
    request_rec *r;
    char *_the_request;
    char *_post_payload;
    sec_dir_config *dcfg;
    sec_srv_config *scfg;
    apr_table_t *parsed_args;
    apr_table_t *parsed_cookies;
    char *tmp_message;
    char *tmp_redirect_url;
    int tmp_log_message;
} modsec_rec;


int check_sig_against_string(modsec_rec *msr, signature *_sig, const char *s, int var_type);
static void sec_debug_log(request_rec *r, int level, const char *text, ...);
char *normalise_uri_inplace(request_rec *r, char *uri, int range_start, int range_end, int check_invalid_encoding, int check_invalid_unicode_encoding);
char *normalise_uri(request_rec *r, char *_uri, int range_start, int range_end, int check_invalid_encoding, int check_invalid_unicode_encoding);

const command_rec *ap_find_command_in_modules(const char *cmd_name, module **mod);

static char *current_logtime(request_rec *r) {
    apr_time_exp_t t;
    char tstr[80];
    apr_size_t len;
            
    apr_time_exp_lt(&t, apr_time_now());
                
    apr_strftime(tstr, &len, 80, "[%d/%b/%Y:%H:%M:%S ", &t);
    apr_snprintf(tstr + strlen(tstr), 80-strlen(tstr), "%c%.2d%.2d]",
        t.tm_gmtoff < 0 ? '-' : '+',
        t.tm_gmtoff / (60*60), t.tm_gmtoff % (60*60));
    return apr_pstrdup(r->pool, tstr);
}

int parse_cookies(request_rec *r, apr_table_t *parsed_cookies) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    const char *header = NULL;
    // const char *rvalue = NULL;
    int cookie_count = 0;
    
    header = (const char *)apr_table_get(r->headers_in, "Cookie");
    if (header != NULL) {
        char *header_copy = (char *)apr_pstrdup(r->pool, header);
        char *name, *value;
        
        sec_debug_log(r, 6, "Cookie header raw: %s", header);
        
        name = strtok(header_copy, ";");
        while(name != NULL) {
            while(*name == 32) name++;
            value = strstr(name, "=");
            if (value != NULL) {
                *value++ = 0;
            
                sec_debug_log(r, 6, "Individual cookie raw [%s][%s]", name, value);
                normalise_uri_inplace(r, value, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
                apr_table_add(parsed_cookies, name, value);
                cookie_count++;
            }
            
            name = strtok(NULL, ";");
        }
    }
    
    return cookie_count;
}

const char *get_variable(request_rec *r, variable *v, apr_table_t *parsed_args) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    time_t tc;
    struct tm *tm;
    static char resultbuf[254] = "";
    const char *result = NULL;

    switch (v->type) {

        case VAR_CUSTOM:

            if (parsed_args != NULL) {
                result = apr_table_get(parsed_args, v->name);
            }
            else {
                // log error
            }

            if (result == NULL) {
                *resultbuf = 0;
                result = resultbuf;
            }
            break;

        case VAR_HEADER:
            result = apr_table_get(r->headers_in, v->name);
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;

        case VAR_ENV:
            result = apr_table_get(r->notes, v->name);

            if (result == NULL) {
                result = apr_table_get(r->subprocess_env, v->name);
            }

            if (result == NULL) {
                result = getenv(v->name);
            }
            break;

        case VAR_ARGS:
            strcpy(resultbuf, "(internal error)");
            result = resultbuf;
            break;

        case VAR_REMOTE_ADDR:
            result = r->connection->remote_ip;
            break;

        case VAR_REMOTE_HOST:
            result = ap_get_remote_host(r->connection, r->per_dir_config, REMOTE_NAME, NULL);
            break;

        case VAR_REMOTE_USER:
            result = r->user;
            break;

        case VAR_REMOTE_IDENT:
            result = ap_get_remote_logname(r);
            break;

        case VAR_REQUEST_METHOD:
            result = r->method;
            break;

        case VAR_REQUEST_URI:
            result = r->uri;
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;

        case VAR_AUTH_TYPE:
            result = r->ap_auth_type;
            break;

        case VAR_IS_SUBREQ:
            result = (r->main != NULL ? "true" : "false");
            break;

        case VAR_DOCUMENT_ROOT:
            result = ap_document_root(r);
            break;

        case VAR_SERVER_ADMIN:
            result = r->server->server_admin;
            break;

        case VAR_SERVER_NAME:
            result = ap_get_server_name(r);
            break;

        case VAR_SERVER_ADDR:
            result = r->connection->local_ip;
            break;

        case VAR_SERVER_PORT:
            apr_snprintf(resultbuf, sizeof(resultbuf), "%u", ap_get_server_port(r));
            result = resultbuf;
            break;

        case VAR_SERVER_PROTOCOL:
            result = r->protocol;
            break;

        case VAR_SERVER_SOFTWARE:
            result = ap_get_server_version ();
            break;

        case VAR_API_VERSION:
            apr_snprintf(resultbuf, sizeof(resultbuf), "%d:%d", MODULE_MAGIC_NUMBER_MAJOR, MODULE_MAGIC_NUMBER_MINOR);
            result = resultbuf;
            break;

        case VAR_TIME_YEAR:
            tc = time (NULL);
            tm = localtime (&tc);
            apr_snprintf (resultbuf, sizeof(resultbuf), "%02d%02d", (tm->tm_year / 100) + 19, tm->tm_year % 100);
            result = resultbuf;
            break;

        case VAR_TIME:
            tc = time (NULL);
            tm = localtime (&tc);
            apr_snprintf (resultbuf, sizeof (resultbuf), "%02d%02d%02d%02d%02d%02d%02d",
                     (tm->tm_year / 100) + 19, (tm->tm_year % 100),
                     tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min,
                     tm->tm_sec);
             
            result = resultbuf;
            break;

        case VAR_TIME_WDAY:
            tc = time (NULL);
            tm = localtime (&tc);
            apr_snprintf(resultbuf, sizeof(resultbuf), "%d", tm->tm_wday);
            result = resultbuf;
            break;

        case VAR_TIME_SEC:
            tc = time (NULL);
            tm = localtime (&tc);
            apr_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_sec);
            result = resultbuf;
            break;

        case VAR_TIME_MIN:
            tc = time (NULL);       
            tm = localtime (&tc);
            apr_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_min);
            result = resultbuf;
            break;

        case VAR_TIME_HOUR:
            tc = time (NULL);
            tm = localtime (&tc);
            apr_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_hour);
            result = resultbuf;
            break;

        case VAR_TIME_MON:
            tc = time (NULL);
            tm = localtime (&tc);
            apr_snprintf (resultbuf, sizeof(resultbuf), "%02d", tm->tm_mon + 1);
            result = resultbuf;
            break;

        case VAR_TIME_DAY:
            tc = time (NULL);
            tm = localtime (&tc);
            apr_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_mday);
            result = resultbuf;
            break;

        case VAR_SCRIPT_FILENAME:
        case VAR_REQUEST_FILENAME:
            result = r->filename;
            break;

        case VAR_PATH_INFO:
            result = r->path_info;
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;

        case VAR_THE_REQUEST:
            result = r->the_request;
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;

        case VAR_QUERY_STRING:
            result = r->args;
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;
            
        case VAR_HANDLER :
            result = r->handler;
            break;
            
        case VAR_COOKIE:
            result = apr_table_get(parsed_args, v->name);
            break;
    }

    if (result == NULL) {
        *resultbuf = 0;
        result = resultbuf;
    }

    return result;
}

request_rec *find_last_request(request_rec *r) {
    // sec_debug_log(r, 5, "find_last_request: start from %x", r);
    while (r->next != NULL) {
        r = r->next;
        // sec_debug_log(r, 5, "find_last_request: one more %x", r);
    }
    return r;
}

static void *sec_create_srv_config(apr_pool_t *p, server_rec *s) {
    sec_srv_config *scfg = apr_pcalloc(p, sizeof(sec_srv_config));

    scfg->filter_engine = FILTERING_OFF;
    scfg->server_response_token = 0;
    scfg->scan_post = 0;
    scfg->scan_output = 0;

    scfg->action.log = 1;
    scfg->action.action = ACTION_DENY;
    scfg->action.status = HTTP_FORBIDDEN;
    scfg->action.pause = 0;

    scfg->signatures = apr_array_make(p, 10, sizeof(signature *));

    scfg->path = apr_pstrdup(p, ":server");

    scfg->auditlog_flag = 0;
    scfg->auditlog_name = NULL;
    scfg->auditlog_fd = NULL;
    
    scfg->filter_debug_level = 0;
    
    scfg->range_start = 0;
    scfg->range_end = 255;
    scfg->check_encoding = 1;
    scfg->check_unicode_encoding = 0;
    
    scfg->chroot_dir = NULL;
    scfg->server_signature = NULL;
    
    scfg->scan_output_mimetypes = NULL;

    return scfg;
}

static void *sec_create_dir_config(apr_pool_t *p, char *path) {
    sec_dir_config *dcfg = (sec_dir_config *)apr_pcalloc(p, sizeof(*dcfg));

    dcfg->filter_engine = NOT_SET;
    dcfg->scan_post = -1;
    dcfg->scan_output = -1;

    dcfg->is_action_default = 1;
    dcfg->action.log = 1;
    dcfg->action.action = ACTION_DENY;
    dcfg->action.status = HTTP_FORBIDDEN;
    dcfg->action.pause = 0;

    dcfg->signatures = apr_array_make(p, 10, sizeof(signature *));

    if (path == NULL) {
        dcfg->path = apr_pstrdup(p, ":null");
    }
    else {
        dcfg->path = apr_pstrdup(p, path);
    }

    dcfg->auditlog_flag = -1;
    dcfg->auditlog_name = NULL;
    dcfg->auditlog_fd = NULL;
    
    dcfg->filter_debug_level = 0;
    dcfg->filters_clear = 0;
    
    dcfg->range_start = 0;
    dcfg->range_end = 255;
    dcfg->check_encoding = 1;
    dcfg->check_unicode_encoding = 0;
    
    dcfg->scan_output_mimetypes = NULL;

    return dcfg;
}

static void *sec_merge_dir_config(struct apr_pool_t *p, void *_parent, void *_child) {
    sec_dir_config *parent = (sec_dir_config *)_parent;
    sec_dir_config *child = (sec_dir_config *)_child;

    // merge child & parent into new

    sec_dir_config *new = (sec_dir_config *)apr_pcalloc(p, sizeof(*new));

    memcpy(new, child, sizeof(*child));

    new->signatures = apr_array_copy(p, child->signatures);
    if (child->filters_clear == 0) apr_array_cat(new->signatures, parent->signatures);

    if (child->scan_post == -1) {
        new->scan_post = parent->scan_post;
    }
    else {
        new->scan_post = child->scan_post;
    }
    
    if (child->scan_output == -1) {
        new->scan_output = parent->scan_output;
    }
    else {
        new->scan_output = child->scan_output;
    }

    if (child->filter_engine == NOT_SET) {
        new->filter_engine = parent->filter_engine;
    }
    else {
        new->filter_engine = child->filter_engine;
    }

    if (child->auditlog_fd == NULL) {
        new->auditlog_fd = parent->auditlog_fd;
        new->auditlog_name = parent->auditlog_name;
    }
    else {
        new->auditlog_fd = child->auditlog_fd;
        new->auditlog_name = child->auditlog_name;
    }
    
    if (child->is_action_default == 1) {
        memcpy(&new->action, &parent->action, sizeof(parent->action));
        new->is_action_default = 0;
    }
    
    if (child->check_encoding == NOT_SET) {
        new->check_encoding = parent->check_encoding;
    }
    else {
        new->check_encoding = child->check_encoding;
    }
    
    if (child->check_unicode_encoding == NOT_SET) {
        new->check_unicode_encoding = parent->check_unicode_encoding;
    }
    else {
        new->check_unicode_encoding = child->check_unicode_encoding;
    }

    return new;
}

unsigned char x2c(unsigned char *what) {
    register unsigned char digit;

    digit = (what[0] >= 'A' ? ((what[0] & 0xdf) - 'A') + 10 : (what[0] - '0'));
    digit *= 16;
    digit += (what[1] >= 'A' ? ((what[1] & 0xdf) - 'A') + 10 : (what[1] - '0'));
    
    return(digit);
}

int detect_unicode_character(request_rec *r, unsigned char *p_read) {
    int unicode_len = 0;
    unsigned int d = 0;
    unsigned char c = *p_read;
    
        if ((c & 0xE0) == 0xC0) {
            // two byte unicode
            if (*(p_read + 1) == 0) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                unicode_len = 2;
                d = ((c & 0x1F) << 6) | (*(p_read + 1) & 0x3F);
                // sec_debug_log(r, 2, "2 Decoded character value: %x", d);
            }
        } else
        if ((c & 0xF0) == 0xE0) {
            // three byte unicode
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                unicode_len = 3;
                d = ((c & 0x0F) << 12) | ((*(p_read + 1) & 0x3F) << 6) | (*(p_read + 2) & 0x3F);
                // sec_debug_log(r, 2, "3 Decoded character value: %x", d);
            }
        } else
        if ((c & 0xF8) == 0xF0) {
            // four byte unicode
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                d = ((c & 0x07) << 18) | ((*(p_read + 1) & 0x3F) << 12) | ((*(p_read + 2) & 0x3F) < 6) | (*(p_read + 3) & 0x3F);
                unicode_len = 4;
                // sec_debug_log(r, 2, "4 Decoded character value: %x, *(p_read+3)=%x", d, *(p_read + 3));
            }
        } else
        if ((c & 0xFC) == 0xF8) {
            
            // five byte unicode
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)||(*(p_read + 4) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 4)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                d = ((c & 0x03) << 24) | ((*(p_read + 1) & 0x3F) << 18) | ((*(p_read + 2) & 0x3F) << 12) | ((*(p_read + 3) & 0x3F) << 6) | (*(p_read + 4) & 0x3F);
                unicode_len = 5;
                // sec_debug_log(r, 2, "5 Decoded character value: %x", d);
            }
        } else
        if ((c & 0xFE) == 0xFC) {
            // six byte unicode
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)||(*(p_read + 4) == 0)||(*(p_read + 5) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 4)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 5)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                d = ((c & 0x01) << 30) | ((*(p_read + 1) & 0x3F) << 24) | ((*(p_read + 2) & 0x3F) << 18) | ((*(p_read + 3) & 0x3F) << 12) | ((*(p_read + 4) & 0x3F) << 6) | (*(p_read + 5) & 0x3F);
                unicode_len = 6;
                // sec_debug_log(r, 2, "6 Decoded character value: %x", d);
            }
        }
        
        if ((unicode_len > 0)&&((d & 0xFF) == d)) {
            unicode_len = UNICODE_ERROR_OVERLONG_CHARACTER;
        }
        
    return(unicode_len);
}

char *normalise_uri_inplace(request_rec *r, char *uri, int range_start, int range_end, int check_invalid_encoding, int check_invalid_unicode_encoding) {
    unsigned char *p_read, *p_write, *p_slash;
    unsigned char c;
    int count;
    
    if (uri == NULL) return NULL;
    
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    
    // loop #1 - URL decoding    
    while ((c = *p_read) != 0) {
        
        // decode URL decoding
        if (c == '+') c = 32;
        else
        if (c == '%') {
        
            // see if there are enough bytes available
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) {
                if (check_invalid_encoding) {
                    sec_debug_log(r, 1, "Invalid URL encoding detected: not enough characters");
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Invalid URL encoding #1 detected.");
                    return NULL;
                } else c = 0;
            } else {
        
                // move onto the first hexadecimal letter
                p_read++;
                
                c = *p_read;
                
                if (check_invalid_encoding) {
                    if (! (((c >= '0')&&(c <= '9')) || ((c >= 'a')&&(c <= 'f')) || ((c >= 'A')&&(c <= 'F'))) ) {
                        sec_debug_log(r, 1, "Invalid URL encoding detected: invalid characters used");
                        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Invalid URL encoding #2 detected.");
                        return NULL;
                    }
                }
                
                c = *(p_read + 1);
                
                if (check_invalid_encoding) {
                    if (! (((c >= '0')&&(c <= '9')) || ((c >= 'a')&&(c <= 'f')) || ((c >= 'A')&&(c <= 'F'))) ) {
                        sec_debug_log(r, 1, "Invalid URL encoding detected: invalid characters used");
                        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Invalid URL encoding #2 detected.");
                        return NULL;
                    }
                }
                
                // decode two hexadecimal letters into a single byte
                c = x2c(p_read);
                p_read++;
            }
        }

        p_read++;
        
        if ((c < range_start)||(c > range_end)) {
            sec_debug_log(r, 1, "Invalid character detected [%i]", c);
            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Invalid character detected [%i]", c);
            return NULL;
        }
        
        if (c == 0) *p_write = ' ';
        else *p_write++ = c;
    }
    *p_write = 0;
    
    // loop #2 - decode everything else
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    p_slash = NULL;
    count = 0;
    while (*p_read != 0) {
        c = *p_read;

        if (check_invalid_unicode_encoding) {
            int urc = detect_unicode_character(r, p_read);
            
            // sec_debug_log(r, 1, "Got %i from Unicode check", urc);
            
            switch(urc) {
                case UNICODE_ERROR_CHARACTERS_MISSING :
                    sec_debug_log(r, 1, "Invalid Unicode encoding: not enough bytes");
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Invalid Unicode encoding: not enough bytes");
                    return NULL;
                    break;
                case UNICODE_ERROR_INVALID_ENCODING :
                    sec_debug_log(r, 1, "Invalid Unicode encoding: invalid byte value");
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Invalid Unicode encoding: invalid byte value");
                    return NULL;
                    break;
                case UNICODE_ERROR_OVERLONG_CHARACTER :
                    sec_debug_log(r, 1, "Invalid Unicode encoding: overlong character");
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Invalid Unicode encoding: overlong character");
                    return NULL;
                    break;
            }
            
        }
        
        switch (c) {
        
            #ifdef WIN32
            case '\\' :
            #endif
        
            case '/' :
                if (p_slash == NULL) {
    
                    // remove the occurencies of "./"
                    if ( (count > 1) && ((*(p_write - 1) == '.') && (*(p_write - 2) == '/')) ) {
                        count -= 2;
                        p_write -= 2;
                    }
                    
                    p_slash = p_read;
                    *p_write++ = '/';
                    p_read++;
                    count++;
                } else {
                    p_read++;
                }
                break;

            default:
                // p_slash is used to detect more than one
                // '/' character in a row
                p_slash = NULL;
        
                // c = *p_read;
                /*
                if ((c < range_start)||(c > range_end)) {
                    sec_debug_log(r, 1, "Invalid character detected [%i]", c);
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Invalid character detected [%i]", c);
                    return NULL;
                }
                */
        
                *p_write++ = c;
                
                p_read++;
                count++;
                
                break;
            }
    }

    *p_write = 0;
    return uri;
}
char *normalise_uri(request_rec *r, char *_uri, int range_start, int range_end, int check_invalid_encoding, int check_invalid_unicode_encoding) {
    char *uri;
    
    if (_uri == NULL) return NULL;
    
    uri = apr_pstrdup(r->pool, _uri);
    return normalise_uri_inplace(r, uri, range_start, range_end, check_invalid_encoding, check_invalid_unicode_encoding);
}

int is_filtering_on_here(request_rec *r) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    
    // refuse to work if the per dir config is null
    if (dcfg == NULL) {
        sec_debug_log(r, 2, "sec_pre: Filtering off, dcfg is null.");
        return 0;
    }
    
    // refuse to work if not initialised properly
    if (dcfg->filter_engine == NOT_SET) {
        return 0;
    }

    // refuse to work if this is a subrequest
    if (!ap_is_initial_req(r)) {
        sec_debug_log(r, 2, "sec_pre: Filtering off for a subrequest.");
        return 0;
    }

    // refuse to work if filtering is off
    if (dcfg->filter_engine == FILTERING_OFF) {
        sec_debug_log(r, 2, "sec_pre: Filtering off, switched off for path [%s]", dcfg->path);
        return 0;
    }
    
    if ((dcfg->filter_engine == FILTERING_DYNAMIC_ONLY)&&(r->handler == NULL)) {
        sec_debug_log(r, 2, "sec_pre: Filtering off for non-dynamic resources [content-type=%s]", r->content_type);
        return 0;
    }
    
    return 1;
}

static int read_post_payload(request_rec *r, char **p, long *postlen) {
    sec_filter_in_ctx *ctx = NULL;
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    
    *p = NULL;
    *postlen = 0;
    
    // sec_debug_log(r, 1, "read_post_payload here %x", r);
    
    if (is_filtering_on_here(r) == 0) return OK;
    
    if (r->method_number != M_POST) {
        sec_debug_log(r, 2, "read_post_payload: skipping a non-POST request");
        return OK;
    }
    
    if (dcfg->scan_post == 0) {
        sec_debug_log(r, 2, "read_post_payload: POST scanning is off here.");
        return OK;
    }
    
    {
        const char *lenp = apr_table_get(r->headers_in, "Content-Length");
        apr_bucket_brigade *bb;
		int seen_eos = 0;
		apr_status_t rv;
	
        if (lenp == NULL) {
            sec_debug_log(r, 1, "read_post_payload: Content-Length not available!");
            *p = NULL;
            return OK;
        }
        
        ctx = apr_pcalloc(r->pool, sizeof(*ctx));
        ctx->buflen = strtol(lenp, NULL, 10);
        
        ctx->buffer = apr_palloc(r->pool, ctx->buflen + 1);
        ctx->buffer[ctx->buflen] = 0;
        
        ctx->bufleft = ctx->buflen;
        ctx->sofar = 0;
        ctx->done_reading = 0;
        ctx->done_writing = 0;
        
        ctx->output_ptr = ctx->buffer;
        ctx->output_sent = 0;
        ctx->access_check_performed = 0;

		bb = apr_brigade_create(r->pool, r->connection->bucket_alloc);
		do {
			apr_bucket *bucket;
			rv = ap_get_brigade(r->input_filters,bb,AP_MODE_READBYTES,APR_BLOCK_READ,HUGE_STRING_LEN);
			if (rv != APR_SUCCESS)
				return rv;

			APR_BRIGADE_FOREACH(bucket,bb) {
				const char *data;
				apr_size_t len;

				if (APR_BUCKET_IS_EOS(bucket)) {
					seen_eos = 1;
					break;
				}

				if (APR_BUCKET_IS_FLUSH(bucket)) {
					continue;
				}

				apr_bucket_read(bucket, &data, &len, APR_BLOCK_READ);
		
                sec_debug_log(r, 5, "read_post_payload: read %i bytes", len);

                if (len <= ctx->bufleft) {
                    memcpy(ctx->buffer + ctx->sofar, data, len);
                    ctx->sofar += len;
                    ctx->bufleft -= len;
                } else {
                    sec_debug_log(r, 1, "read_post_payload: POST buffer overflow; %i bytes in buffer, %i bytes incoming", ctx->bufleft, len);
                    ctx->bufleft = 0;
                }
			}
			apr_brigade_cleanup(bb);
		} while (!seen_eos);

	    ctx->done_reading = 1;	

        // add note (needed for audit logging)
        apr_table_setn(r->notes, "mod_security-note", (char *)ctx);
        sec_debug_log(r, 2, "read_post_payload: Added mod_security-note to %x", r);
        
        ap_add_input_filter_handle(global_sec_filter_in, ctx, r, r->connection);
        *p = ctx->buffer;
        *postlen = ctx->buflen;
    }

    return OK;
}

char *unescape_url(unsigned char *url) {
    register int x, y;

    for (x = 0, y = 0; url[y]; ++x, ++y) {
        if ((url[x] = url[y]) == '%') {
                url[x] = x2c(&url[y + 1]);
                y += 2;
        }
    }
    
    url[x] = '\0';
    return (char *)url;
}

void parse_arguments(char *s, apr_table_t *parsed_args, request_rec *r, sec_dir_config *dcfg) {
    int inputlength;
    char *buf;
    int i, j, status;
    char *value = NULL;

    if (s == NULL) return;

    inputlength = strlen(s);

    buf = (char *)malloc(inputlength + 1);
    if (buf == NULL) {
        // log error
        return;
    }

    i = 0;
    j = 0;
    status = 0;
    while (i < inputlength) {
    
        while ((s[i] != '=') && (s[i] != '&') && (i < inputlength)) {
            if (s[i] == '+') buf[j] = 32;
            else buf[j] = s[i];

            j++;
            i++;
        }

        buf[j++] = '\0';

        if (status == 0) {
            status = 1;
            // unescape_url(buf);
            normalise_uri_inplace(r, buf, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            value = &buf[j];
        }
        else {
            status = 0;
            // unescape_url(value);
            normalise_uri_inplace(r, value, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            
            apr_table_add(parsed_args, buf, value);

            j = 0;
        }
        
        i++;
    }

    if (status == 1) {
        apr_table_add(parsed_args, buf, "");
    }

    free(buf);
    return;
}

char *remove_binary_content(request_rec *r, char *data, long size) {
    // long size = (long)r->remaining;
    char *src, *dst, *newdata;

    // make a copy of the payload first
    dst = src = newdata = apr_palloc(r->pool, size + 1);
    memcpy(newdata, data, size);
    newdata[size] = 0;
    
    // remove zeros from the payload
    while(size--) {
        if (*src != 0) *dst++ = *src++;
        else src++;
    }
    
    return newdata;
}

int check_single_signature(modsec_rec *msr, signature *sig) {
    int j, rs;

        // the idea behind non-selective filters is to apply them over
        // raw data, typically the complete first line of the request
        // and the complete POST payload
        
        if (sig->is_selective == 0) {
        
            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at THE_REQUEST", sig->pattern);
            
            rs = check_sig_against_string(msr, sig, msr->_the_request, VAR_THE_REQUEST);
            if (rs != OK) return rs;

            if (msr->_post_payload != NULL) {
                sec_debug_log(msr->r, 2, "Checking signature \"%s\" at POST_PAYLOAD", sig->pattern);

                // todo: will this regex check work over several lines or not?
                rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                if (rs != OK) return rs;
            }
        }
        else {
            variable **variables;
            const char *v;

            // this is a selective signature, this means that we need to
            // check only one part of the request and leave the rest alone
            
            // selective signatures can be negative and non-negative;
            // non-negative signatures consist of a list of variables
            // that represent parts of the request that need to be
            // checked; negative signatures apply only to request
            // arguments, when you want to exclude an argument from
            // a check            

            if (sig->is_negative == 0) {
                
                // loop through signature variables and
                // check them

                variables = (variable **)sig->variables->elts;
                for (j = 0; j < sig->variables->nelts; j++) {

                    if (variables[j]->type == VAR_ARGS) {
                    
                        // special handling for this variable; with POST
                        // request arguments can exist in the URL and in
                        // the body of the request

                        variables[j]->type = VAR_QUERY_STRING; 
                        v = get_variable(msr->r, variables[j], msr->parsed_args);
                        variables[j]->type = VAR_ARGS;
            
                        if (msr->_post_payload != NULL) {
                        
                            // check the encoding
                            // add _post_payload to the v
                            // todo: handle multipart/form-data encoding
                            v = apr_pstrcat(msr->r->pool, v, msr->_post_payload, NULL);
                        }
                        
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at VAR_ARGS", sig->pattern);
                        
                        rs = check_sig_against_string(msr, sig, (char *)v, VAR_ARGS);
                        if (rs != OK) return rs;
                    }
                    else if (variables[j]->type == VAR_POST_PAYLOAD) {

                        // only check the payload if this is a POST request
                        if (msr->r->method_number == M_POST) {
                            if (msr->_post_payload == NULL) {
                                ap_log_rerror (APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, msr->r, "mod_security: Filtering against POST payload requested but payload is not available.");
                                sec_debug_log(msr->r, 1, "Filtering against POST payload requested but payload is not available.");
                            }
                            else {
                                sec_debug_log(msr->r, 2, "Checking signature \"%s\" at POST_PAYLOAD", sig->pattern);
                                
                                rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                                if (rs != OK) return rs;
                            }
                        }
                    }
                    else if (variables[j]->type == VAR_ARGS_NAMES) {
                        apr_table_entry_t *te;
                        const apr_array_header_t *arr;
                        int k;
                    
                        // todo: do we have parsed_args?
                        
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_NAMES", sig->pattern);
                    
                        arr = apr_table_elts(msr->parsed_args);
                        te = (apr_table_entry_t *)arr->elts;
                        for (k = 0; k < arr->nelts; k++) {
                            rs = check_sig_against_string(msr, sig, te[k].key, VAR_ARGS_NAMES);
                            if (rs != OK) return rs;
                        }
                    }
                    else if (variables[j]->type == VAR_ARGS_VALUES) {
                        apr_table_entry_t *te;
                        const apr_array_header_t *arr;
                        int k;
                    
                        // todo: do we have parsed_args?
                        
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_VALUES", sig->pattern);
                    
                        arr = apr_table_elts(msr->parsed_args);
                        te = (apr_table_entry_t *)arr->elts;
                        for (k = 0; k < arr->nelts; k++) {
                            rs = check_sig_against_string(msr, sig, te[k].val, VAR_ARGS_VALUES);
                            if (rs != OK) return rs;
                        }
                    }
                    else if(variables[j]->type == VAR_COOKIES_NAMES) {
                        apr_table_entry_t *te;
                        const apr_array_header_t *arr;
                        int k;
                    
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at COOKIES_NAMES", sig->pattern);
                    
                        arr = apr_table_elts(msr->parsed_cookies);
                        te = (apr_table_entry_t *)arr->elts;
                        for (k = 0; k < arr->nelts; k++) {
                            sec_debug_log(msr->r, 5, "Cookie [%s][%s]", te[k].key, te[k].val);
                            rs = check_sig_against_string(msr, sig, te[k].key, VAR_COOKIES_NAMES);
                            if (rs != OK) return rs;
                        }
                    }
                    else if(variables[j]->type == VAR_COOKIES_VALUES) {
                        apr_table_entry_t *te;
                        const apr_array_header_t *arr;
                        int k;
                    
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at COOKIES_VALUES", sig->pattern);
                    
                        arr = apr_table_elts(msr->parsed_cookies);
                        te = (apr_table_entry_t *)arr->elts;
                        for (k = 0; k < arr->nelts; k++) {
                            sec_debug_log(msr->r, 5, "Cookie [%s][%s]", te[k].key, te[k].val);
                            rs = check_sig_against_string(msr, sig, te[k].val, VAR_COOKIES_VALUES);
                            if (rs != OK) return rs;
                        }
                    }
                    else {
                        // simple variable, get the value and check it

                        if (variables[j]->type == VAR_COOKIE) v = get_variable(msr->r, variables[j], msr->parsed_cookies);
                        else v = get_variable(msr->r, variables[j], msr->parsed_args);
                        if (v != NULL) {

                            if (variables[j]->name != NULL) {
                                sec_debug_log(msr->r, 2, "Checking signature \"%s\" at %s(%s)", sig->pattern, all_variables[variables[j]->type], variables[j]->name);
                            } else {                        
                                sec_debug_log(msr->r, 2, "Checking signature \"%s\" at %s", sig->pattern, all_variables[variables[j]->type]);
                            }
                            
                            sec_debug_log(msr->r, 5, "Variable value: [%s]", v);
                        
                            rs = check_sig_against_string(msr, sig, (char *)v, variables[j]->type);
                            if (rs != OK) return rs;
                        }
                        else {
                            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, msr->r, "mod_security: Variable not found [%s]", variables[j]->name);
                        }
                    }
                }
            }
            else {
                int k;
                apr_table_t *our_parsed_args;
                apr_table_entry_t *te;
                const apr_array_header_t *arr;
                char *v = NULL;
                unsigned int len = 0;

                if (msr->r->args != NULL) {
                    len = strlen(msr->r->args);
                }

                if (msr->_post_payload != NULL) {
                    len += strlen(msr->_post_payload);
                }

                if (len != 0) {
                    v = (char *)malloc(len + 1);
                    if (v == NULL) {
                        // error
                        // todo: could not allocate memory
                        return DECLINED;
                    }

                    *v = 0;
                    if (msr->parsed_args == NULL) {
                        ap_log_rerror (APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, msr->r, "mod_security: arguments are not parsed, internal error");
                        return DECLINED;
                    }

                    our_parsed_args = apr_table_copy(msr->r->pool, msr->parsed_args);

                    // find the unwanted variable names in the signature
                    // data and remove them from the variable list

                    variables = (variable **)sig->variables->elts;
                    for (j = 0; j < sig->variables->nelts; j++) {

                        if ((variables[j]->type == VAR_CUSTOM) && (variables[j]->action == VAR_ACTION_ALLOW)) {
                            apr_table_unset(our_parsed_args, variables[j]->name);
                        }
                    }
                    
                    // loop through remaining variables
                    // and create a single string out of them

                    arr = apr_table_elts(our_parsed_args);
                    te = (apr_table_entry_t *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {

                        if (*v != 0) {
                            strcat(v, "&");
                        }

                        strcat(v, te[k].key);
                        strcat(v, "=");
                        strcat(v, te[k].val);
                    }

                    // make the check against the compiled string   
                    rs = check_sig_against_string(msr, sig, (char *)v, VAR_ARGS_SELECTIVE);

                    free(v);
            
                    if (rs != OK) return rs;
                }
            }
        }
        
    return OK;
}

int sec_check_all_signatures(modsec_rec *msr, int is_output) {
    request_rec *r = msr->r;
    signature **signatures;
    // long _post_len = 0;
    int i;
    int mode = 0;
    int skip_count = 0;
    int rc = DECLINED;
    
        // loop through all signatures
        signatures = (signature **)msr->dcfg->signatures->elts;
        for (i = 0; i < msr->dcfg->signatures->nelts; i++) {

            // output rules are not processed here        
            if (signatures[i]->is_output != is_output) continue;
        
            // check if we need to skip this rule
            if (skip_count > 0) {
                skip_count--;
                continue;
            }

            // just clear the flag, we had to use the flag
            // to detect a case when the last rule in the
            // rule chain was marked as chained            
            if (mode == 2) mode = 0;
            
            // in mode 1, we are looking for the next filter
            // that is next chained, and then skip it to
            // execute the next filter in the chain
            if (mode == 1) {
                if (signatures[i]->is_chained == 0) mode = 0;
                continue;
            }
            
            msr->tmp_message = NULL;
            msr->tmp_redirect_url = NULL;
            msr->tmp_log_message = 0;
            
            rc = check_single_signature(msr, signatures[i]);
            sec_debug_log(r, 9, "Signature check returned %i", rc);
            
            if (msr->tmp_message != NULL) {
                sec_debug_log(r, 1, msr->tmp_message);
                apr_table_setn(r->headers_in, "mod_security-message", msr->tmp_message);
                if (msr->tmp_log_message) {
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: %s", msr->tmp_message);
                } else {
                    apr_table_setn(r->notes, "mod_security-noauditlog", "noauditlog");
                }
            }

            // DECLINED means that an allow action was called,
            // we need to pass the request through            
            if (rc == DECLINED) return DECLINED;

            // OK means there was no filter match, we
            // switch to mode 1, processing will continue
            // with the next filter chain
            if (rc == OK) {
                // we go into mode 1 (looking for the last rule
                // in the chain) if this rule is not it
                if (signatures[i]->is_chained == 1) mode = 1;
                continue;
            }
            
            // any status greater than zero means there
            // was a filter match, so we either stop execution
            // or proceed to the next rule if this rule was
            // chained
            if (rc > 0) {
                if (signatures[i]->is_chained == 1) {
                    mode = 2;
                    continue;
                }
                else {
rule_match:
                    if (msr->tmp_redirect_url != NULL) {
                        apr_table_setn(msr->r->headers_out, "Location", msr->tmp_redirect_url);
                    }
        
                    return rc;
                }
            }
            
            // if the return status is zero this
            // means skip some rules, so we skip
            if (rc == MODSEC_SKIP) {
                skip_count = signatures[i]->skip_count;
                continue;
            }
            
            sec_debug_log(r, 1, "Unprocessed return code [%i]", rc);
            return DECLINED;
        }
        
        // handle the case where there was a match on the
        // last filter so mode 2 check could not be done        
        // strickly speaking this should be a configuration error
        if (mode == 2) goto rule_match;

    return DECLINED;
}    
    
static int sec_exec_child(char *command, request_rec *r) {
    apr_procattr_t *procattr;
    apr_proc_t *procnew;
    apr_status_t rc = APR_SUCCESS;
    const char * const *env;
    // char **argv = NULL;
    
    ap_add_cgi_vars(r);
    ap_add_common_vars(r);
    
    // PHP hack, getting around its security checks
    apr_table_add(r->subprocess_env, "PATH_TRANSLATED", command);
    apr_table_add(r->subprocess_env, "REDIRECT_STATUS", "302");                                        
    
    env = (const char * const *)ap_create_environment(r->pool, r->subprocess_env);
    
    procnew = apr_pcalloc(r->pool, sizeof(*procnew));
    
    apr_procattr_create(&procattr, r->pool);
    
    rc = ap_os_create_privileged_process(r, procnew, command, NULL, env, procattr, r->pool);
    if (rc != APR_SUCCESS) {
        sec_debug_log(r, 1, "Failed to execute: \"%s\" (rc=%d)", command, rc);
        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "Failed to execute: \"%s\" (rc=%d)", command, rc);
    }
    
    apr_pool_note_subprocess(r->pool, procnew, APR_KILL_AFTER_TIMEOUT);
    apr_proc_wait(procnew, NULL, NULL, APR_WAIT);
    
    return rc;                                                                                                
}

int check_sig_against_string(modsec_rec *msr, signature *_sig, const char *s, int var_type) {
    signature *default_action = &msr->dcfg->action;
    request_rec *r = msr->r;
    int rc = OK;

    int regex_result = ap_regexec(_sig->regex, s, 0, NULL, 0);
    
    sec_debug_log(r, 4, "check_sig_against_string: string: %s regex_result: %i is_allow: %i", s, regex_result, _sig->is_allow);

    if ( ((regex_result == 0)&&(_sig->is_allow == 0)) || ((regex_result != 0)&&(_sig->is_allow == 1)) ) {

        signature *sig = _sig;
    
        if (sig->is_default_action_used == 1) sig = default_action;

        // execute the external script
        if (sig->exec) {
            sec_debug_log(r, 1, "Executing command \"%s\"", sig->exec_string);
            apr_table_setn(r->headers_in, "mod_security-executed", sig->exec_string);
            sec_exec_child(sig->exec_string, r);
        }
        
        if (sig->pause != 0) {
            sec_debug_log(r, 1, "Pausing [%s] for %i ms", r->uri, sig->pause);
            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: pausing [%s] for %i ms", r->uri, sig->pause);
            
            // apr_sleep accepts microseconds
            apr_sleep(sig->pause * 1000);
        }
        
        // perform action
        switch(sig->action) {
        
            case ACTION_SKIP :
                sec_debug_log(r, 2, "Skipping %i statements on pattern match \"%s\" at %s", sig->skip_count, _sig->pattern, all_variables[var_type]);
                rc = MODSEC_SKIP;
                break;
        
            case ACTION_ALLOW :
                // sec_debug_log(r, 2, "Access allowed based on pattern match \"%s\" at %s", _sig->pattern, all_variables[var_type]);
                msr->tmp_message = apr_psprintf(r->pool, "Access allowed based on pattern match \"%s\" at %s", _sig->pattern, all_variables[var_type]);
                
                // returning DECLINED will interrupt filter processing but
                // it won't do anything the the request
                rc = DECLINED;
                break;
        
            case ACTION_DENY :
                msr->tmp_message = apr_psprintf(r->pool, "Access denied with code %i. Pattern match \"%s\" at %s.", sig->status, _sig->pattern, all_variables[var_type]);
                rc = sig->status;
                break;
                
            case ACTION_REDIRECT :
                msr->tmp_message = apr_psprintf(r->pool, "Access denied with redirect to [%s]. Pattern match \"%s\" at %s.", sig->redirect_url, _sig->pattern, all_variables[var_type]);
                msr->tmp_redirect_url = sig->redirect_url;
                rc = HTTP_MOVED_TEMPORARILY;
                break;
                
            default :
                msr->tmp_message = apr_psprintf(r->pool, "Warning. Pattern match \"%s\" at %s.", _sig->pattern, all_variables[var_type]);
                break;
        }
        
        if ((msr->tmp_message != NULL)&&(sig->log)) msr->tmp_log_message = 1;
        // sec_debug_log(r, 1, "tmp_message=%s, tmp_log_message=%i", msr->tmp_message, msr->tmp_log_message);
    }

    return rc;
}


char *parse_action(char *p2, signature *sig, apr_pool_t *_pool) {
    char *t = apr_pstrdup(_pool, p2);
    char *p = strtok(t, ",");
    int found;

    while (p != NULL) {
        found = 0;
        
        if (strcmp (p, "log") == 0) {
            found = 1;
            sig->log = 1;
        }
        else if (strcmp(p, "nolog") == 0) {
            found = 1;
            sig->log = 0;
        }
        else if (strncmp(p, "status", 6) == 0) {
            found = 1;
            sig->action = ACTION_DENY;
            if (strlen(p) > 8) {
                sig->status = atoi(p + 7);
            }
        }
        else if (strcmp(p, "chain") == 0) {
            found = 1;
            sig->is_chained = 1;
        }
        else if (strcmp(p, "chained") == 0) {
            found = 1;
            sig->is_chained = 1;
        }
        else if (strncmp(p, "skipnext", 8) == 0) {
            found = 1;
            sig->action = ACTION_SKIP;
            if (strlen(p) > 9) {
                sig->skip_count = atoi(p + 9);
            }
        }
        else if (strncmp(p, "skip", 4) == 0) {
            found = 1;
            sig->action = ACTION_SKIP;
            if (strlen(p) > 9) {
                sig->skip_count = atoi(p + 9);
            }
        }
        else if (strcmp(p, "deny") == 0) {
            found = 1;
            sig->action = ACTION_DENY;
        }
        else if (strcmp(p, "allow") == 0) {
            found = 1;
            sig->action = ACTION_ALLOW;
        }
        else if (strcmp(p, "pass") == 0) {
            found = 1;
            sig->action = ACTION_NONE;
        }
        else if (strncmp(p, "exec", 4) == 0) {
            found = 1;
            sig->exec = 1;
            if (strlen(p) > 6) {
                sig->exec_string = apr_pstrdup(_pool, p + 5);
            }
        }
        else if (strncmp(p, "redirect", 8) == 0) {
            found = 1;
            sig->action = ACTION_REDIRECT;
            if (strlen(p) > 10) {
                sig->redirect_url = apr_pstrdup(_pool, p + 9);
            }
        }
        else if (strncmp(p, "msg", 3) == 0) {
            found = 1;
            if (strlen(p) > 5) {
                sig->msg = apr_pstrdup(_pool, p + 4);
            }
        }
        else if (strncmp(p, "id", 2) == 0) {
            found = 1;
            if (strlen(p) > 4) {
                sig->id = apr_pstrdup(_pool, p + 3);
            }
        } else if (strncmp(p, "pause", 5) == 0) {
            found = 1;
            if (strlen(p) > 7) {
                sig->pause = atoi(p + 6);
            }
        }
        
        if (found == 0) {
            return apr_psprintf(_pool, "Unknown mod_security action \"%s\"", p);
        }

        p = strtok(NULL, ",");
    }

    return NULL;
}

static const char *cmd_filter_check_encoding(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->check_encoding = flag;
    return NULL;
}

static const char *cmd_filter_check_unicode_encoding(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->check_unicode_encoding = flag;
    return NULL;
}

static const char *cmd_filter_force_byte_range(cmd_parms *cmd, void *in_dcfg, const char *p1, const char *p2) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->range_start = atoi(p1);
    dcfg->range_end = atoi(p2);
    return NULL;
}

static const char *cmd_filter_engine(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    
    if (strcasecmp(p1, "On") == 0) dcfg->filter_engine = FILTERING_ON;
    else
    if (strcasecmp(p1, "Off") == 0) dcfg->filter_engine = FILTERING_OFF;
    else
    if (strcasecmp(p1, "DynamicOnly") == 0) dcfg->filter_engine = FILTERING_DYNAMIC_ONLY;
    else
    return (const char *)apr_psprintf(cmd->pool, "Unrecognized parameter value for SecFilterEngine: %s", p1);
    
    return NULL;
}

static const char *cmd_filters_clear(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->filters_clear = flag;
    return NULL;
}

static const char *cmd_filter_inheritance(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    if (flag) dcfg->filters_clear = 0;
    else dcfg->filters_clear = 1;
    return NULL;
}

static const char *cmd_server_response_token(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    scfg->server_response_token = flag;
    return NULL;
}

static const char *cmd_audit_engine(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    
    if (strcasecmp(p1, "On") == 0) dcfg->auditlog_flag = AUDITLOG_ON;
    else
    if (strcasecmp(p1, "Off") == 0) dcfg->auditlog_flag = AUDITLOG_OFF;
    else
    if (strcasecmp(p1, "RelevantOnly") == 0) dcfg->auditlog_flag = AUDITLOG_RELEVANT_ONLY;
    else
    if (strcasecmp(p1, "DynamicOrRelevant") == 0) dcfg->auditlog_flag = AUDITLOG_DYNAMIC_OR_RELEVANT;
    else
    return (const char *)apr_psprintf(cmd->pool, "Unrecognized parameter value for SecAuditEngine: %s", p1);
                                    
    return NULL;
}

static const char *cmd_audit_log(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    int rc;

    dcfg->auditlog_name = ap_server_root_relative(cmd->pool, (char *)p1);
    
    rc = apr_file_open(&dcfg->auditlog_fd, dcfg->auditlog_name, 
                    APR_WRITE | APR_APPEND | APR_CREATE ,
                   APR_UREAD | APR_UWRITE | APR_GREAD | APR_WREAD, cmd->pool);

    if (rc != APR_SUCCESS) {
        return dcfg->auditlog_name;
    }

    return NULL;
}

static const char *cmd_scan_post(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->scan_post = flag;
    return NULL;
}

static const char *cmd_scan_output(cmd_parms *cmd, void *in_dcfg, int flag) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->scan_output = flag;
    return NULL;
}

static const char *cmd_default_action(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->is_action_default = 0;
    return parse_action((char *)p1, &dcfg->action, cmd->pool);
}

static const char *cmd_chroot_dir(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    scfg->chroot_dir = (char *)p1;
    return NULL;
}            

static const char *cmd_server_signature(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    scfg->server_signature = (char *)p1;        
    return NULL;
}            

char *strtolower(char *str) {
    char *c = str;
    
    if (str == NULL) return NULL;
    
    while(*c != 0) {
        *c = tolower(*c);
        c++;
    }
    
    return str;
}

static const char *cmd_filter_output_mimetypes(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->scan_output_mimetypes = apr_psprintf(cmd->pool, " %s ", p1);
    strtolower(dcfg->scan_output_mimetypes);
    return NULL;
}            

static const char *cmd_filter(cmd_parms *cmd, void *in_dcfg, const char *p1, const char *p2) {
    sec_dir_config *dcfg = in_dcfg;
    signature *sig;

    sig = apr_pcalloc(cmd->pool, sizeof(signature));
    sig->action = dcfg->action.action;
    sig->status = dcfg->action.status;
    sig->skip_count = 1;
    sig->is_chained = 0;
    
    // p1 is the regular expression string
    if (p1[0] == '!') {
        sig->is_allow = 1;
        sig->pattern = (char *)p1;
        sig->regex = ap_pregcomp(cmd->pool, p1 + 1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    } else {
        sig->pattern = (char *)p1;
        sig->regex = ap_pregcomp(cmd->pool, p1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    
    if (sig->regex == NULL) {
        return apr_psprintf(cmd->pool, "Invalid regular expression: %s", sig->pattern);
    }
    
    // p2 is an optional action string
    if (p2 != NULL) {
        char *rc;
        sig->is_default_action_used = 0;
        rc = parse_action((char *)p2, sig, cmd->pool);
        if (rc != NULL) return rc;
    }
    else {
        sig->is_default_action_used = 1;
    }

    // add the signature to the list of all signatures
    *(signature **)apr_array_push(dcfg->signatures) = sig;

    return NULL;
}

static const char *cmd_filter_debug_log(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    // sec_dir_config *dcfg = in_dcfg;
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    int rc;
    
    scfg->debuglog_name = ap_server_root_relative(cmd->pool, p1);
   
    rc = apr_file_open(&scfg->debuglog_fd, scfg->debuglog_name,
                   APR_WRITE | APR_APPEND | APR_CREATE,
                   APR_UREAD | APR_UWRITE | APR_GREAD | APR_WREAD, cmd->pool);

    if (rc != APR_SUCCESS) {
        return "mod_security: Failed to open the debug log file.";
    }
    
    return NULL;
}

static const char *cmd_filter_debug_level(cmd_parms *cmd, void *in_dcfg, const char *p1) {
    sec_dir_config *dcfg = in_dcfg;
    dcfg->filter_debug_level = atoi(p1);
    return NULL;
}

static const char *cmd_filter_selective(cmd_parms *cmd, void *in_dcfg, const char *p1, const char *p2, const char *p3) {
    sec_dir_config *dcfg = in_dcfg;
    char *p;
    char *t;
    signature *sig;
    
    // initialise the structure first    
    
    sig = apr_pcalloc(cmd->pool, sizeof(signature));
    sig->skip_count = 1;
    sig->is_chained = 0;
    sig->action = dcfg->action.action;
    sig->status = dcfg->action.status;
    sig->is_allow = 0;
    sig->is_selective = 1;
    sig->is_negative = 0;
    sig->requires_parsed_args = 0;
    sig->variables = apr_array_make(cmd->pool, 10, sizeof (variable *));
    
    if (p2[0] == '!') {
        sig->is_allow = 1;
        sig->pattern = (char *)p2;
        sig->regex = ap_pregcomp(cmd->pool, p2 + 1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    } else {
        sig->pattern = (char *)p2;
        sig->regex = ap_pregcomp(cmd->pool, p2, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    
    if (sig->regex == NULL) {
        return apr_psprintf(cmd->pool, "Invalid regular expression: %s", sig->pattern);
    }
    
    // split parameter 1 apart and extract variable names
    
    p = strdup(p1);
    t = strtok (p, "|");
    while (t != NULL) {

        // add the token to the list
        variable *v = (variable *)apr_palloc(cmd->pool, sizeof(variable));
        char *x = t;

        v->type = VAR_UNKNOWN;

        // when ! is the first character in the variable
        // name that means that the restrictions need to be
        // relaxed for that variable (within the filter scope)
        if (t[0] == '!') {
            v->action = VAR_ACTION_ALLOW;
            sig->is_negative = 1;
            sig->requires_parsed_args = 1;
            x++;
        }
        else {
            v->action = VAR_ACTION_DENY;
        }

        // arguments
        if (strncmp (x, "ARG_", 4) == 0) {
            v->type = VAR_CUSTOM;
            v->name = apr_pstrdup(cmd->pool, x + 4);
            sig->requires_parsed_args = 1;
        }
        // HTTP headers
        else if (strncmp (x, "HTTP_", 5) == 0) {
            char *px;

            v->type = VAR_HEADER;
            v->name = apr_pstrdup (cmd->pool, x + 5);
    
            // replace all "_" with "-"
            px = v->name;
            while (*px != 0) {
                if (*px == '_') *px = '-';
                px++;
            }
        }    
        // COOKIES
        else if (strncmp(x, "COOKIE_", 7) == 0) {
            // char *px;
            
            v->type = VAR_COOKIE;
            v->name = apr_pstrdup(cmd->pool, x + 7);
        }
        // environment variables
        else if (strncmp (x, "ENV_", 4) == 0) {
            v->type = VAR_ENV;
            v->name = apr_pstrdup(cmd->pool, x + 4);
        }
        // all arguments
        else if (strcmp (x, "ARGS") == 0) {
            v->type = VAR_ARGS;
            v->name = apr_pstrdup(cmd->pool, x);
        }
        // just the post payload
        else if (strcmp(x, "POST_PAYLOAD") == 0) {
            v->type = VAR_POST_PAYLOAD;
            v->name = apr_pstrdup(cmd->pool, x);
        }
        else if (strcmp(x, "OUTPUT") == 0) {
            v->type = VAR_OUTPUT;
            v->name = apr_pstrdup(cmd->pool, x);
            sig->is_output = 1;
        }
        // everything else
        else {
            char **vl = all_variables;
            int i = 0;

            while (*vl != NULL) {
                if (strcmp(*vl, x) == 0) {
                    v->type = i;
                    v->name = apr_pstrdup(cmd->pool, x);
                    break;
                }

                i++;
                vl++;
            }
        }

        if (v->type == VAR_UNKNOWN) {
            v->name = apr_pstrdup(cmd->pool, "UKNOWN");
            return apr_psprintf(cmd->pool, "Unknown variable name: %s", x);
        }

        if ((v->type == VAR_ARGS_NAMES)||(v->type == VAR_ARGS_VALUES)) sig->requires_parsed_args = 1;

        *(variable **)apr_array_push(sig->variables) = v;

        // and proceed to the next token
        t = strtok (NULL, "|");
    }

    free(p);

    // is default action parameter present?
    if (p3 != NULL) {
        char *rc;
        sig->is_default_action_used = 0;
        rc = parse_action((char *)p3, sig, cmd->pool);
        if (rc != NULL) return rc;
    }
    else {
        sig->is_default_action_used = 1;
    }

    // add the signature to the list of all signatures
    *(signature **) apr_array_push(dcfg->signatures) = sig;

    return NULL;
}


static const command_rec sec_cmds[] = {

     AP_INIT_TAKE12 (
         "SecFilter",
         cmd_filter,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),
     
     AP_INIT_TAKE1 (
        "SecFilterDebugLog",
        cmd_filter_debug_log,
        NULL,
        RSRC_CONF,
        ""
     ),
          
     AP_INIT_TAKE1 (
         "SecFilterDebugLevel",
         cmd_filter_debug_level,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),

     AP_INIT_TAKE23 (
         "SecFilterSelective",
         cmd_filter_selective,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),

     AP_INIT_TAKE1 (
         "SecFilterEngine",
         cmd_filter_engine,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),
     
     AP_INIT_FLAG (
        "SecServerResponseToken",
        cmd_server_response_token,
        NULL,
        RSRC_CONF,
        ""        
     ),

     AP_INIT_FLAG (
         "SecFilterScanPOST",
         cmd_scan_post,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),
     
     AP_INIT_FLAG (
         "SecFilterScanOutput",
         cmd_scan_output,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),

     AP_INIT_TAKE1 (
         "SecFilterDefaultAction",
         cmd_default_action,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),

     AP_INIT_TAKE1 (
         "SecAuditEngine",
         cmd_audit_engine,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),

     AP_INIT_TAKE1 (
         "SecAuditLog",
         cmd_audit_log,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),
     
     AP_INIT_TAKE1 (
         "SecChrootDir",
         cmd_chroot_dir,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),
     
     AP_INIT_TAKE1 (
         "SecServerSignature",
         cmd_server_signature,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),
     
     AP_INIT_TAKE1 (
         "SecFilterOutputMimeTypes",
         cmd_filter_output_mimetypes,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ),
     
     AP_INIT_FLAG (
        "SecFiltersClear",
        cmd_filters_clear,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        ""
     ),
     
     AP_INIT_FLAG (
        "SecFilterInheritance",
        cmd_filter_inheritance,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        ""
     ),
     
     AP_INIT_FLAG (
        "SecFilterCheckURLEncoding",
        cmd_filter_check_encoding,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        ""
     ),
     
     AP_INIT_FLAG (
        "SecFilterCheckUnicodeEncoding",
        cmd_filter_check_unicode_encoding,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        ""
     ),
     
     AP_INIT_TAKE2 (
         "SecFilterForceByteRange",
         cmd_filter_force_byte_range,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         ""
     ), 
         

    { NULL }
};


static int sec_logger(request_rec *_r) {
    request_rec *r = find_last_request(_r);
    char *o1 = apr_palloc(r->pool, AUDIT_BUF_SIZE + 1);
    char *o2 = apr_palloc(r->pool, AUDIT_BUF_SIZE + 1);
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    const char *modsec_message = NULL;
    const apr_array_header_t *arr;
    apr_table_entry_t *te;
    char *remote_user, *local_user, *t;
    int i = 0;
    apr_size_t nbytes;
    apr_status_t rv;

    // refuse to work if not initialised properly    
    if (dcfg->filter_engine == NOT_SET) {
        return DECLINED;
    }

    sec_debug_log(r, 2, "sec_logger: start");

    switch(dcfg->auditlog_flag) {
    
        case AUDITLOG_OFF :
            sec_debug_log(r, 2, "Audit log: off for [%s]", r->uri);
            return OK;
            break;
            
        case AUDITLOG_DYNAMIC_OR_RELEVANT :
            modsec_message = apr_table_get(r->headers_in, "mod_security-message");
            if (((modsec_message == NULL)&&(r->handler == NULL))||(apr_table_get(_r->notes, "mod_security-noauditlog") != NULL)) {
                sec_debug_log(r, 2, "Audit log: ignoring a non-dynamic and non-relevant request [content-type=%s]", r->content_type);
                return OK;
            }
            break;
            
        case AUDITLOG_RELEVANT_ONLY :
            modsec_message = apr_table_get(r->headers_in, "mod_security-message");
            if ((modsec_message == NULL)||(apr_table_get(_r->notes, "mod_security-noauditlog") != NULL)) {
                sec_debug_log(r, 2, "Audit log: ignoring a non-relevant request [content-type=%s]", r->content_type);
                return OK;
            }
            break;
    }
    
    // return immediatelly if we don't have a file to write to
    if (dcfg->auditlog_fd == NULL) {
        ap_log_rerror (APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: Audit log not found. [%s]", r->uri);
        sec_debug_log(r, 1, "Audit log: not found [%s]", r->uri);
        return OK;
    }
    
    strcpy (o1, "");
    strncat (o1, "========================================\n", AUDIT_BUF_SIZE);

    if (r->connection->remote_logname == NULL) {
        remote_user = "-";
    }
    else {
        remote_user = r->connection->remote_logname;
    }

    if (r->user == NULL) {
        local_user = "-";
    }
    else {
        local_user = r->user;
    }

    t = apr_psprintf(r->pool, "Request: %s %s %s [%s] \"%s\" %i %li\n",
                     r->connection->remote_ip, remote_user, local_user,
                     current_logtime(r), r->the_request, r->status,
                     r->bytes_sent);
             
    strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));

    t = apr_psprintf (r->pool, "Handler: %s\n", r->handler);
    strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));
    
    // see if there is an error message stored in notes
    t = (char *)apr_table_get(r->notes, "error-notes");
    if (t != NULL) {
        t = apr_psprintf(r->pool, "Error: %s\n", t);
        strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));
    }

    strncat(o1, "----------------------------------------\n", AUDIT_BUF_SIZE - strlen(o1));

    // input headers
    t = apr_psprintf (r->pool, "%s\n", r->the_request);
    strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));

    arr = apr_table_elts(r->headers_in);

    te = (apr_table_entry_t *) arr->elts;
    for (i = 0; i < arr->nelts; i++) {
            t = apr_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
            strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));
    }

    strncat(o1, "\n", AUDIT_BUF_SIZE - strlen(o1));

    if (r->method_number == M_POST) {
        strcpy(o2, "\n\n");
    } else {
        strcpy(o2, "");
    }

    // output headers

    // the status first
    t = apr_psprintf(r->pool, "%s %s\n", r->protocol, r->status_line);
    strncat(o2, t, AUDIT_BUF_SIZE - strlen(o2));


    // output headers
    arr = apr_table_elts(r->headers_out);

    te = (apr_table_entry_t *) arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = apr_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o2, t, AUDIT_BUF_SIZE - strlen(o2));
    }

    // write to the file
    
    // apr_file_lock(dcfg->auditlog_fd, APR_FLOCK_EXCLUSIVE);
    rv = apr_global_mutex_lock(modsec_auditlog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r, "apr_global_mutex_lock(modsec_auditlog_lock) failed");
    }    
    
    nbytes = strlen(o1);
    apr_file_write(dcfg->auditlog_fd, o1, &nbytes);
    
    if (r->method_number == M_POST) {
        sec_filter_in_ctx *ctx = (sec_filter_in_ctx *)apr_table_get(r->notes, "mod_security-note");
        
        if (ctx != NULL) {
            nbytes = ctx->buflen;
            apr_file_write(dcfg->auditlog_fd, ctx->buffer, &nbytes);
        } else {
            char *message = "[POST payload not available]";
            nbytes = strlen(message);
            apr_file_write(dcfg->auditlog_fd, message, &nbytes);
        }
    }

    // write the second part of the log    
    nbytes = strlen(o2);
    apr_file_write(dcfg->auditlog_fd, o2, &nbytes);
    
    // apr_file_unlock(dcfg->auditlog_fd);
    rv = apr_global_mutex_unlock(modsec_auditlog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r, "apr_global_mutex_unlock(modsec_auditlog_lock) failed");
    }    

    return OK;
}

static int sec_check_access(request_rec *r) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(r->server->module_config, &security_module);
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    modsec_rec *msr;
    int rc;
    long _post_len = 0;

    // refuse to work if the per dir config is null
    if (dcfg == NULL) {
        sec_debug_log(r, 2, "Filtering off, dcfg is null.");
        return DECLINED;
    }
    
    // refuse to work if not initialised properly
    if (dcfg->filter_engine == NOT_SET) {
        return DECLINED;
    }

    // refuse to work if this is a subrequest
    if (!ap_is_initial_req(r)) {
        sec_debug_log(r, 2, "Filtering off for a subrequest.");
        return DECLINED;
    }
    
    // refuse to work if filtering is off
    if (dcfg->filter_engine == FILTERING_OFF) {
        sec_debug_log(r, 2, "Filtering off, switched off for path [%s]", dcfg->path);
        return DECLINED;
    }
    
    if ((dcfg->filter_engine == FILTERING_DYNAMIC_ONLY)&&(r->handler == NULL)) {
        sec_debug_log(r, 2, "Filtering off for non-dynamic resources [content-type=%s]", r->content_type);
        return DECLINED;
    }

    sec_debug_log(r, 2, "Checking with per-dir-config [%s][%s]", dcfg->path, r->uri);
    
    msr = (modsec_rec *)apr_pcalloc(r->pool, sizeof(*msr));
    msr->r = r;
    msr->scfg = scfg;
    msr->dcfg = dcfg;
    msr->_the_request = NULL;
    msr->_post_payload = NULL;
    msr->parsed_args = apr_table_make(r->pool, 10);
    msr->parsed_cookies = apr_table_make(r->pool, 10);
    
    if (r->args == NULL) {
        msr->_the_request = apr_pstrdup(r->pool, r->uri);
    } else {
        msr->_the_request = apr_pstrcat(r->pool, r->uri, "?", r->args, NULL);
    }
    
    msr->_the_request = normalise_uri_inplace(r, msr->_the_request, msr->dcfg->range_start, msr->dcfg->range_end, msr->dcfg->check_encoding, msr->dcfg->check_unicode_encoding);
    if (msr->_the_request == NULL) {
        apr_table_setn(r->headers_in, "mod_security-message", "Invalid character detected");
        if (msr->dcfg->action.log == 0) apr_table_setn(r->notes, "mod_security-noauditlog", "noauditlog");
        return msr->dcfg->action.status;
    }
    
    if (r->args != NULL) parse_arguments(r->args, msr->parsed_args, r, msr->dcfg);
    parse_cookies(r, msr->parsed_cookies);    

    if (msr->dcfg->scan_post) {
        char *content_type;

        rc = read_post_payload(r, &msr->_post_payload, &_post_len);
        if (rc != OK) return rc;
        
        if (msr->_post_payload != NULL) {    
            content_type = (char *)apr_table_get(r->headers_in, "Content-Type");
            if (content_type != NULL) sec_debug_log(r, 3, "Content-Type=%s", content_type);
            else sec_debug_log(r, 3, "Content-Type is NULL");
            
            if ((content_type != NULL) && (strcmp(content_type, "application/x-www-form-urlencoded") == 0)) {
            
                // parse post payload
                if (msr->_post_payload != NULL) {
                    sec_debug_log(r, 3, "Parsing variables from POST payload");
                    parse_arguments(msr->_post_payload, msr->parsed_args, r, msr->dcfg);
                }
                
                msr->_post_payload = normalise_uri(r, msr->_post_payload, msr->dcfg->range_start, msr->dcfg->range_end, msr->dcfg->check_encoding, msr->dcfg->check_unicode_encoding);
            } else if ((content_type != NULL) && (strncmp(content_type, "multipart/form-data", 19) == 0)) {
                // todo: parse
                msr->_post_payload = remove_binary_content(r, msr->_post_payload, _post_len);
            } else {
                // remove binary content from the payload
                msr->_post_payload = remove_binary_content(r, msr->_post_payload, _post_len);
            }
            
            if (msr->_post_payload == NULL) {
                return msr->dcfg->action.status;
            }
        }
    }

    // 0 means OUTPUT filters will be ignored   
    rc = sec_check_all_signatures(msr, 0);
    
    // make a note for the logger
    if (rc != DECLINED) {
        char *note = apr_psprintf(r->pool, "%i", rc);
        apr_table_setn(r->headers_in, "mod_security-action", note);
    }
    
    return rc;
}

static void sec_debug_log(request_rec *r, int level, const char *text, ...) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(r->server->module_config, &security_module);
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    va_list ap;
    char str1[1024] = "";
    char str2[1256] = "";
    apr_size_t nbytes;
    apr_status_t rv;

    va_start(ap, text);
    
    if (scfg->debuglog_fd == NULL) {
        return;
    }
    
    if (level > dcfg->filter_debug_level) {
        return;
    }

    apr_vsnprintf(str1, sizeof(str1), text, ap);
    apr_snprintf(str2, sizeof(str2), "%s [%s/sid#%lx][rid#%lx][%s] %s\n", current_logtime(r), ap_get_server_name(r), (unsigned long)(r->server), (unsigned long)r, r->uri, str1); 

    // apr_file_lock(scfg->debuglog_fd, APR_FLOCK_EXCLUSIVE);
    rv = apr_global_mutex_lock(modsec_debuglog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r, "apr_global_mutex_lock(modsec_debuglog_lock) failed");
    }    

    nbytes = strlen(str2);
    apr_file_write(scfg->debuglog_fd, str2, &nbytes);

    // apr_file_unlock(scfg->debuglog_fd);
    rv = apr_global_mutex_unlock(modsec_debuglog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r, "apr_global_mutex_unlock(modsec_debuglog_lock) failed");
    }    

    va_end(ap);
    
    return;
}

static apr_status_t locks_remove(void *data) {
    if (modsec_debuglog_lock != NULL) apr_global_mutex_destroy(modsec_debuglog_lock);
    if (modsec_auditlog_lock != NULL) apr_global_mutex_destroy(modsec_auditlog_lock);
    return 0;
}

static int sec_init(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s) {                                                                     
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);
    apr_status_t rv;

    if (scfg->server_response_token) {    
        ap_add_version_component(p, MOD_SECURITY_SIGNATURE);
    }
    
    // this might not make sense at first glance but we are
    // actually just allocating space here; we will hack the
    // server name later
    if (scfg->server_signature) {
        ap_add_version_component(p, scfg->server_signature);
        {
            char *server_version = (char *)ap_get_server_version();
            if (strlen(server_version) >= strlen(scfg->server_signature)) {
                strcpy(server_version, scfg->server_signature);
            }
        }
    }

    #if !(defined(WIN32) || defined(NETWARE))
    if (scfg->chroot_dir != NULL) {
        if (getppid() == 1) {
            if (chroot(scfg->chroot_dir) < 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, s, "mod_security: chroot failed, path=%s, errno=%i", scfg->chroot_dir, errno);
            } else {
                ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, 0, s, "mod_security: performed chroot, path=%s", scfg->chroot_dir);
            }
        }
    }
    #endif
    
    if ((rv = apr_global_mutex_create(&modsec_debuglog_lock, NULL, APR_LOCK_DEFAULT, p)) != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s, "mod_security: Could not create modsec_debuglog_lock");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
                                                                                       
    #if APR_USE_SYSVSEM_SERIALIZE
    rv = unixd_set_global_mutex_perms(modsec_debuglog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s, "mod_security: Could not set permissions on modsec_debuglog_lock; check User and Group directives");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    #endif
    
    if ((rv = apr_global_mutex_create(&modsec_auditlog_lock, NULL, APR_LOCK_DEFAULT, p)) != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s, "mod_security: Could not create modsec_auditlog_lock");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
                                                                                       
    #if APR_USE_SYSVSEM_SERIALIZE
    rv = unixd_set_global_mutex_perms(modsec_auditlog_lock);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, s, "mod_security: Could not set permissions on modsec_auditlog_lock; check User and Group directives");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    #endif
    
    apr_pool_cleanup_register(p, (void *)s, locks_remove, apr_pool_cleanup_null);
    
    return OK;
}

static void sec_child_init(apr_pool_t *p, server_rec *s) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);

    if (scfg->server_signature != NULL) {
        char *server_version = (char *)ap_get_server_version();
        if (server_version != NULL) {
            // sec_init should make sure that we have enought
            // space for the signature but we are checking this
            // here just in case
            if (strlen(server_version) >= strlen(scfg->server_signature)) {
                strcpy(server_version, scfg->server_signature);
            }
        }
    }
}

void send_error_bucket(ap_filter_t *f, int status) {
    apr_bucket_brigade *bb;
    apr_bucket *e;
                
    bb = apr_brigade_create(f->r->pool, f->r->connection->bucket_alloc);
    e = ap_bucket_error_create(status, NULL, f->r->pool, f->r->connection->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, e);
    e = apr_bucket_eos_create(f->r->connection->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, e);
    
    ap_pass_brigade(f->next, bb);
}

/**
 * The purpose of this input filter is to break the
 * filter chain in those cases where we have already
 * read the POST data into our buffer. So, instead
 * of getting the data from the next filter in the
 * chain we serve it from our buffer.
 */
static apr_status_t sec_filter_in(ap_filter_t *f, apr_bucket_brigade *pbbOut, ap_input_mode_t eMode, apr_read_type_e eBlock, apr_off_t nBytes) {
    request_rec *r = f->r;
    conn_rec *c = r->connection;
    // sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    sec_filter_in_ctx *ctx = NULL;
    // apr_status_t ret;
    
    sec_debug_log(r, 4, "sec_filter_in: start");

    // the context will always be supplied to us
    if (!(ctx = f->ctx)) {
        sec_debug_log(r, 1, "sec_filter_in: context not found!");
        return ap_get_brigade(f->next, pbbOut, eMode, eBlock, nBytes);
    }
    
    // Skip the call if there isn't any work left for us
    if (ctx->done_writing == 1) {
        return ap_get_brigade(f->next, pbbOut, eMode, eBlock, nBytes);
    }
           
    // Send a chunk of data further down the filter chain
    if (ctx->output_sent < ctx->sofar) {
        apr_bucket *pbktOut;        
        unsigned int len = 4000;
        
        if (len > nBytes) len = nBytes;
        
        if (ctx->sofar - ctx->output_sent < len) len = ctx->sofar - ctx->output_sent;
        
        pbktOut = apr_bucket_heap_create(ctx->output_ptr, len, NULL, c->bucket_alloc);
        APR_BRIGADE_INSERT_TAIL(pbbOut, pbktOut);        
        
        ctx->output_ptr += len;
        ctx->output_sent += len;
        
        sec_debug_log(r, 4, "sec_filter_in: Sent %d bytes", len);
    }
    
    // are we done yet?
    if (ctx->output_sent == ctx->sofar) {
        // send an EOS bucket, we're done
        apr_bucket *pbktOut = apr_bucket_eos_create(c->bucket_alloc);
        APR_BRIGADE_INSERT_TAIL(pbbOut, pbktOut);
        
        sec_debug_log(r, 4, "sec_filter_in: Sent EOS bucket");
        ctx->done_writing = 1;
        
        // nothing left for us to do in this request
        ap_remove_input_filter(f);
    }
    
    return APR_SUCCESS;
}

static apr_status_t sec_filter_out(ap_filter_t *f, apr_bucket_brigade *pbbIn) {
    request_rec *r = f->r;
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config (r->per_dir_config, &security_module);
    sec_filter_out_ctx *ctx = NULL;
    conn_rec *c = r->connection;
    apr_bucket *pbktIn;
    // apr_bucket_brigade *pbbOut;
    // apr_status_t ret;
    
    sec_debug_log(r, 3, "sec_filter_out: start");
    
    // create a context if one does not already exist
    if (!(ctx = f->ctx)) {
        const char *s;
        
        f->ctx = ctx = apr_pcalloc(r->pool, sizeof(*ctx));
        ctx->content_length = -1;
        
        sec_debug_log(r, 3, "sec_filter_out: got Content-Type: %s", r->content_type);

        // if scan_output_mimetypes is not null that means output
        // filtering should be selective, using the content type value        
        if (dcfg->scan_output_mimetypes != NULL) {
            char *content_type;
            
            // check for the case when the content type is not set
            if (r->content_type == NULL) {
                if (strstr(dcfg->scan_output_mimetypes, "(null)") == NULL) {
                    ap_remove_output_filter(f);
                    return ap_pass_brigade(f->next, pbbIn);
                }
            } else {
                // in all other cases, see if the content type appears
                // on the acceptable mime types list            
                content_type = apr_psprintf(r->pool, " %s ", r->content_type);
                strtolower(content_type);
            
                if (strstr(dcfg->scan_output_mimetypes, content_type) == NULL) {
                    ap_remove_output_filter(f);
                    return ap_pass_brigade(f->next, pbbIn);
                }
            }
        }

        ctx->buflen = 0;
            
        // look up the Content-Length header to see if we know
        // the amount of data coming our way
        s = apr_table_get(r->headers_out, "Content-Length");
        
        // try this too, mod_cgi seems to put headers there
        if (s == NULL) s = apr_table_get(r->err_headers_out, "Content-Length");
        
        if (s != NULL) {
            ctx->buflen = ctx->content_length = atoi(s);
            sec_debug_log(r, 3, "sec_filter_out: got Content-Length %i", ctx->content_length);
        }

        // use the default buffer length if everything else fails        
        if (ctx->buflen <= 0) {
            // todo: move this value to a constant or to configuration
            ctx->buflen = 16384;
        }
        
        ctx->buffer = apr_palloc(f->r->pool, ctx->buflen + 1);
        ctx->input_ptr = ctx->buffer;
        
        if (ctx->buffer == NULL) {
            sec_debug_log(r, 1, "sec_filter_out: Failed to allocate buffer [%i]", ctx->buflen + 1);
            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: sec_filter_out: Failed to allocate buffer [%li]", ctx->buflen + 1);
            ap_remove_output_filter(f);
            return ap_pass_brigade(f->next, pbbIn);
        }        
    }

    // read data into our buffer    
    APR_BRIGADE_FOREACH(pbktIn, pbbIn) {        
        const char *data;
        apr_size_t len;
    
        if(APR_BUCKET_IS_EOS(pbktIn)) {
            sec_debug_log(r, 3, "sec_filter_out: done reading");
            ctx->buffer[ctx->bufused] = 0;
            ctx->done_reading = 1;
	        continue;
	    }
    
        apr_bucket_read(pbktIn, &data, &len, APR_BLOCK_READ);
        sec_debug_log(r, 3, "sec_filter_out: got %i bytes, bufused=%i, buflen=%i", len, ctx->bufused, ctx->buflen);

        if(ctx->bufused + len > ctx->buflen) {
            char *newbuffer;
            // todo: implement a smarter extension policy
            unsigned long int newsize = ctx->buflen * 2;
            
            // increase the size of the new buffer until the data fits
            while(ctx->bufused + len >= newsize) newsize = newsize * 2;
            
            sec_debug_log(r, 3, "sec_filter_out: expanding buffer to %i", newsize);
            
            // allocate a larger buffer
            newbuffer = apr_palloc(f->r->pool, newsize + 1);
            if (newbuffer == NULL) {
                sec_debug_log(r, 1, "sec_filter_out: Failed to allocate buffer [%i]", newsize + 1);
                ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: sec_filter_out: Failed to allocate buffer [%li]", newsize + 1);
                ap_remove_output_filter(f);
                return ap_pass_brigade(f->next, pbbIn);
            }
            memcpy(newbuffer, ctx->buffer, ctx->bufused);
            
            ctx->buffer = newbuffer;
            ctx->buflen = newsize;
            ctx->input_ptr = ctx->buffer + ctx->bufused;
        }

        memcpy(ctx->input_ptr, data, len);
        ctx->input_ptr += len;
        ctx->bufused += len;
        
        apr_bucket_delete(pbktIn);
    }

    // we wait patiently (for the next call) until we get
    // all of the output
    if (ctx->done_reading == 0) return APR_SUCCESS;
    
    // this shouldn't happen but doesn't hurt us to check
    if (ctx->done_writing == 1) {
        sec_debug_log(r, 1, "sec_filter_out: internal error, I shouldn't have been called at all");
        return ap_pass_brigade(f->next, pbbIn);
    }
    
    // todo: check output headers
    
    // check the output
    {
        modsec_rec msr;
        signature **signatures;
        int i, rv;

        /* TODO
        memset(&msr, 0, sizeof(msr));        
        msr.r = r;
        msr.dcfg = dcfg;
        
        // 1 means only check OUTPUT filters
        rc = sec_check_all_signatures(&msr, 1);
        if ((rc != OK)&&(rc != DECLINED)) {        
            // todo: headers must also be removed from the output!!
                
            // it seems that mod_proxy sets the status line
            // and it later overrides the real status
            //  in ap_send_error_response; so we kill it here
            r->status_line = NULL;
                
            send_error_bucket(f, rc);
                
            ctx->done_writing = 1;
            return rc;
        }
        */
        
        signatures = (signature **)dcfg->signatures->elts;
        for (i = 0; i < dcfg->signatures->nelts; i++) {
    
            if (signatures[i]->is_output != 1) continue;
            sec_debug_log(r, 2, "Checking signature \"%s\" at OUTPUT", signatures[i]->pattern);
            
            // quick hack, it will be ok for the time being,
            // until I get some serious code refactoring
            memset(&msr, 0, sizeof(msr));        
            msr.r = r;
            msr.dcfg = dcfg;
            
            // rv = check_sig_against_string(r, signatures[i], &dcfg->action, ctx->buffer, VAR_OUTPUT);
            rv = check_sig_against_string(&msr, signatures[i], ctx->buffer, VAR_OUTPUT);
            if ((rv != OK)&&(rv != DECLINED)) {
            
                if (msr.tmp_message != NULL) {
                    sec_debug_log(r, 1, msr.tmp_message);
                    apr_table_setn(r->headers_in, "mod_security-message", msr.tmp_message);
                    if (msr.tmp_log_message) {
                        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r, "mod_security: %s", msr.tmp_message);
                    } else {
                        apr_table_setn(r->notes, "mod_security-noauditlog", "noauditlog");
                    }
                }
    
                if (msr.tmp_redirect_url != NULL) {
                    apr_table_setn(msr.r->headers_out, "Location", msr.tmp_redirect_url);
                }

                // dirty hack, but filtering doesn't work with mod_deflate
                {
                    ap_filter_t *f = r->output_filters;
                    while(f != NULL) {
                        // sec_debug_log(r, 1, "Filter name: %s", f->frec->name);
                        if (strcasecmp(f->frec->name, "deflate") == 0) {
                            ap_remove_output_filter(f);
                            sec_debug_log(r, 2, "sec_filter_out: Removed deflate from the output_filters list");
                            break;
                        }
                        f = f->next;
                    }
                }
            
                // todo: headers must also be removed from the output!!
                
                // it seems that mod_proxy sets the status line
                // and it later overrides the real status
                //  in ap_send_error_response; so we kill it here
                r->status_line = NULL;
                
                send_error_bucket(f, rv);
                
                // free(ctx->buffer);
                ctx->done_writing = 1;
                return rv;
            }
        }
    }
    
    // if we're that means that all went well and that
    // we now need to send the output to the filter chain
    ctx->output_ptr = ctx->buffer;
    ctx->output_sent = 0;
    
    // sec_debug_log(r, 5, ctx->buffer);    

    while(ctx->output_sent < ctx->bufused) {
        apr_bucket_brigade *pbbTmp;
        apr_bucket *pbktTmp;
        unsigned int batch = 4000;
        
        // adjust the chunk size
        if (ctx->bufused - ctx->output_sent < batch) batch = ctx->bufused - ctx->output_sent;
            
        pbbTmp = apr_brigade_create(r->pool, c->bucket_alloc);
        pbktTmp = apr_bucket_heap_create(ctx->output_ptr, batch, NULL, c->bucket_alloc);
        APR_BRIGADE_INSERT_TAIL(pbbTmp, pbktTmp);
    
        ctx->output_ptr += batch;
        ctx->output_sent += batch;
            
        sec_debug_log(r, 3, "sec_filter_out: sent %i bytes", batch);
        
        ap_pass_brigade(f->next, pbbTmp);
    }

    {
        apr_bucket_brigade *pbbTmp;
        apr_bucket *pbktTmp;
        
        pbbTmp = apr_brigade_create(r->pool, c->bucket_alloc);
        pbktTmp = apr_bucket_eos_create(f->r->connection->bucket_alloc);
        APR_BRIGADE_INSERT_TAIL(pbbTmp, pbktTmp);
        ap_pass_brigade(f->next, pbbTmp);
        sec_debug_log(r, 3, "sec_filter_out: done writing");
    }
    
    ctx->done_writing = 1;
        
    ap_remove_output_filter(f);
    return APR_SUCCESS;
}

static void sec_pre(request_rec *r) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);

    if (is_filtering_on_here(r) == 0) return;    
    
    sec_debug_log(r, 3, "sec_pre: scan_output = %i", dcfg->scan_output);
    
    if (dcfg->scan_output == 1) {    
        ap_add_output_filter_handle(global_sec_filter_out, NULL, r, r->connection);
    } else {
        sec_debug_log(r, 2, "sec_pre: Output filtering off here.");
    }
}

static void register_hooks(apr_pool_t *p) {
    ap_hook_post_config(sec_init, NULL, NULL, APR_HOOK_REALLY_LAST);
    ap_hook_log_transaction(sec_logger, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_fixups(sec_check_access, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_child_init(sec_child_init, NULL, NULL, APR_HOOK_MIDDLE);
    
    ap_hook_insert_filter(sec_pre, NULL, NULL, APR_HOOK_FIRST);
    global_sec_filter_in = ap_register_input_filter("MODSEC_IN", sec_filter_in, NULL, AP_FTYPE_CONTENT_SET) ;
    global_sec_filter_out = ap_register_output_filter("MODSEC_OUT", sec_filter_out, NULL, AP_FTYPE_CONTENT_SET);
}

module AP_MODULE_DECLARE_DATA security_module = {
   STANDARD20_MODULE_STUFF,
   sec_create_dir_config,        /* create per-dir    config structures */
   sec_merge_dir_config,         /* merge  per-dir    config structures */
   sec_create_srv_config,        /* create per-server config structures */
   NULL,                         /* merge  per-server config structures */
   sec_cmds,                     /* table of config file commands       */
   register_hooks                /* register hooks                      */
};

