/*
 * mod_security (apache 1.x), http://www.modsecurity.org/
 * Copyright (c) 2002,2003 Ivan Ristic <ivanr@webkreator.com>
 *
 * $Id: mod_security.c,v 1.49.2.6 2003/12/05 17:17:00 ivanr Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */            
 


#include <stdarg.h>

#include "httpd.h"
#include "http_config.h"
#include "http_request.h"
#include "http_protocol.h"
#include "http_core.h"
#include "http_main.h"
#include "http_log.h"
#include "util_script.h"

#if defined(NETWARE)
#include <nwsemaph.h>
LONG locking_sem = 0;
#endif

#ifdef WIN32
#include <sys/locking.h>
#endif

#if defined(NETWARE)
#define CREATEMODE ( S_IREAD | S_IWRITE )
#elif defined(WIN32)
#define CREATEMODE ( _S_IREAD | _S_IWRITE )
#else
#define CREATEMODE ( S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH )
#endif

module MODULE_VAR_EXPORT security_module;

#define MOD_SECURITY_SIGNATURE "mod_security/1.7.4"
#define AUDIT_BUF_SIZE 8192

#define UNICODE_ERROR_CHARACTERS_MISSING    -1
#define UNICODE_ERROR_INVALID_ENCODING      -2
#define UNICODE_ERROR_OVERLONG_CHARACTER    -3

#define MODSEC_SKIP                     -2000

#define NOT_SET                         -1

#define FILTERING_OFF                   0
#define FILTERING_ON                    1
#define FILTERING_DYNAMIC_ONLY          2

#define AUDITLOG_OFF                    0
#define AUDITLOG_ON                     1
#define AUDITLOG_DYNAMIC_OR_RELEVANT    2
#define AUDITLOG_RELEVANT_ONLY          3

#define ACTION_NONE                 0
#define ACTION_DENY                 1
#define ACTION_REDIRECT             2
#define ACTION_ALLOW                3
#define ACTION_SKIP                 4

#define VAR_ACTION_ALLOW            1
#define VAR_ACTION_DENY             0

#define VAR_UNKNOWN                 0
#define VAR_CUSTOM                  1
#define VAR_HEADER                  2
#define VAR_ENV                     3
#define VAR_ARGS                    4
#define VAR_POST_PAYLOAD            5
#define VAR_ARGS_NAMES              6
#define VAR_ARGS_VALUES             7
#define VAR_ARGS_SELECTIVE          8
#define VAR_OUTPUT                  9
#define VAR_COOKIES_NAMES           10
#define VAR_COOKIES_VALUES          11
#define VAR_COOKIE                  12
#define VAR_RESERVED_5              13
#define VAR_RESERVED_6              14
#define VAR_RESERVED_7              15
#define VAR_RESERVED_8              16
#define VAR_RESERVED_9              17
#define VAR_RESERVED_10             18
#define VAR_RESERVED_11             19
#define VAR_RESERVED_12             20

#define VAR_REMOTE_ADDR             21
#define VAR_REMOTE_HOST             22
#define VAR_REMOTE_USER             23   
#define VAR_REMOTE_IDENT            24
#define VAR_REQUEST_METHOD          25
#define VAR_SCRIPT_FILENAME         26
#define VAR_PATH_INFO               27
#define VAR_QUERY_STRING            28
#define VAR_AUTH_TYPE               29
#define VAR_DOCUMENT_ROOT           30
#define VAR_SERVER_ADMIN            31
#define VAR_SERVER_NAME             32
#define VAR_SERVER_ADDR             33
#define VAR_SERVER_PORT             34
#define VAR_SERVER_PROTOCOL         35
#define VAR_SERVER_SOFTWARE         36
#define VAR_TIME_YEAR               37
#define VAR_TIME_MON                38
#define VAR_TIME_DAY                39
#define VAR_TIME_HOUR               40
#define VAR_TIME_MIN                41
#define VAR_TIME_SEC                42
#define VAR_TIME_WDAY               43
#define VAR_TIME                    44
#define VAR_API_VERSION             45
#define VAR_THE_REQUEST             46
#define VAR_REQUEST_URI             47
#define VAR_REQUEST_FILENAME        48
#define VAR_IS_SUBREQ               49
#define VAR_HANDLER                 50

char *all_variables[] = {
    "UNKOWN",
    "CUSTOM",
    "HEADER",
    "ENV",
    "ARGS",
    "POST_PAYLOAD",
    "ARGS_NAMES",
    "ARGS_VALUES",
    "ARGS_SELECTIVE",
    "OUTPUT",
    "COOKIES_NAMES", // 10
    "COOKIES_VALUES",
    "COOKIE",
    "RESERVED_5",
    "RESERVED_6",
    "RESERVED_7",
    "RESERVED_8",
    "RESERVED_9",
    "RESERVED_10",
    "RESERVED_11",
    "RESERVED_12", // 20
    "REMOTE_ADDR",
    "REMOTE_HOST",
    "REMOTE_USER",
    "REMOTE_IDENT",
    "REQUEST_METHOD",
    "SCRIPT_FILENAME",
    "PATH_INFO",
    "QUERY_STRING",
    "AUTH_TYPE",
    "DOCUMENT_ROOT",
    "SERVER_ADMIN",
    "SERVER_NAME",
    "SERVER_ADDR",
    "SERVER_PORT",
    "SERVER_PROTOCOL",
    "SERVER_SOFTWARE",
    "TIME_YEAR",
    "TIME_MON",
    "TIME_DAY",
    "TIME_HOUR",
    "TIME_MIN",
    "TIME_SEC",
    "TIME_WDAY",
    "TIME",
    "API_VERSION",
    "THE_REQUEST",
    "REQUEST_URI",
    "REQUEST_FILENAME",
    "IS_SUBREQ",
    "HANDLER",
    NULL
};



typedef struct {
    char *name;
    int type;
    int action;
} variable;

typedef struct {
    request_rec *r;
    char *command;
} exec_data;

typedef struct {
    char *buffer;
    char *sofar;
    long length;
    long remaining;
} request_body;

typedef struct {
    int log;
    int action;
    int status;
    char *redirect_url;
    int exec;
    char *exec_string;
    char *id;
    char *msg;
    char *pattern;
    regex_t *regex;
    int is_default_action_used;
    int is_selective;
    int is_negative;
    int is_allow;
    int requires_parsed_args;
    array_header *variables;
    int pause;
    int skip_count;
    int is_chained;
} signature;

typedef struct {
    int filter_engine;
    int scan_post;
    int is_action_default;
    signature action;
    array_header *signatures;
    char *path;
    int auditlog_flag;
    char *auditlog_name;
    int auditlog_fd;
    int filter_debug_level;
    int filters_clear;
    int range_start;
    int range_end;
    int check_encoding;
    int check_unicode_encoding;
} sec_dir_config;

typedef struct {
    int filter_engine;
    int scan_post;
    int is_action_default;
    signature action;
    array_header *signatures;
    char *path;
    int auditlog_flag;
    char *auditlog_name;
    int auditlog_fd;
    int filter_debug_level;
    int server_response_token;
    char *debuglog_name;
    int debuglog_fd;
    int range_start;
    int range_end;
    int check_encoding;
    int check_unicode_encoding;
    char *chroot_dir;
    char *server_signature;
} sec_srv_config;

typedef struct {
    request_rec *r;
    char *_the_request;
    char *_post_payload;
    sec_dir_config *dcfg;
    sec_srv_config *scfg;
    table *parsed_args;
    table *parsed_cookies;
    char *tmp_message;
    char *tmp_redirect_url;
    int tmp_log_message;
} modsec_rec;

int check_single_signature(modsec_rec *msr, signature *sig);
static int check_sig_against_string(modsec_rec *msr, signature *_sig, const char *s, int var_type);
static void fd_lock(request_rec *r, int fd);
static void fd_unlock(request_rec *r, int fd);
static void sec_debug_log(request_rec *r, int level, const char *text, ...);
char *normalise_uri_inplace(request_rec *r, char *uri, int range_start, int range_end, int check_invalid_encoding, int check_invalid_unicode_encoding);
char *normalise_uri(request_rec *r, char *_uri, int range_start, int range_end, int check_invalid_encoding, int check_invalid_unicode_encoding);

// we will need to access this value directly in
// order to change the server signature @ runtime
extern enum server_token_type ap_server_tokens;

int parse_cookies(request_rec *r, table *parsed_cookies) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    const char *header = NULL;
    // const char *rvalue = NULL;
    int cookie_count = 0;
    
    header = (const char *)ap_table_get(r->headers_in, "Cookie");
    if (header != NULL) {
        char *header_copy = (char *)ap_pstrdup(r->pool, header);
        char *name, *value;
        
        sec_debug_log(r, 6, "Cookie header raw: %s", header);
        
        name = strtok(header_copy, ";");
        while(name != NULL) {
            while(*name == 32) name++;
            value = strstr(name, "=");
            if (value != NULL) {
                *value++ = 0;
            
                sec_debug_log(r, 6, "Individual cookie raw [%s][%s]", name, value);
                normalise_uri_inplace(r, value, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
                ap_table_add(parsed_cookies, name, value);
                cookie_count++;
            }
            
            name = strtok(NULL, ";");
        }
    }
    
    return cookie_count;
}
const char *get_variable(request_rec *r, variable *v, table *parsed_args) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    time_t tc;
    struct tm *tm;
    static char resultbuf[254] = "";
    const char *result = NULL;

    switch (v->type) {

        case VAR_CUSTOM:

            if (parsed_args != NULL) {
                result = ap_table_get(parsed_args, v->name);
            }
            else {
                // log error
            }

            if (result == NULL) {
                *resultbuf = 0;
                result = resultbuf;
            }
            break;

        case VAR_HEADER:
            result = ap_table_get(r->headers_in, v->name);
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;

        case VAR_ENV:
            result = ap_table_get(r->notes, v->name);

            if (result == NULL) {
                result = ap_table_get(r->subprocess_env, v->name);
            }

            if (result == NULL) {
                result = getenv(v->name);
            }
            break;

        case VAR_ARGS:
            strcpy(resultbuf, "(internal error)");
            result = resultbuf;
            break;

        case VAR_REMOTE_ADDR:
            result = r->connection->remote_ip;
            break;

        case VAR_REMOTE_HOST:
            result = ap_get_remote_host(r->connection, r->per_dir_config, REMOTE_NAME);
            break;

        case VAR_REMOTE_USER:
            result = r->connection->user;
            break;

        case VAR_REMOTE_IDENT:
            result = ap_get_remote_logname(r);
            break;

        case VAR_REQUEST_METHOD:
            result = r->method;
            break;

        case VAR_REQUEST_URI:
            result = r->uri;
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;

        case VAR_AUTH_TYPE:
            result = r->connection->ap_auth_type;
            break;

        case VAR_IS_SUBREQ:
            result = (r->main != NULL ? "true" : "false");
            break;

        case VAR_DOCUMENT_ROOT:
            result = ap_document_root(r);
            break;

        case VAR_SERVER_ADMIN:
            result = r->server->server_admin;
            break;

        case VAR_SERVER_NAME:
            result = ap_get_server_name(r);
            break;

        case VAR_SERVER_ADDR:
            result = r->connection->local_ip;
            break;

        case VAR_SERVER_PORT:
            ap_snprintf(resultbuf, sizeof(resultbuf), "%u", ap_get_server_port(r));
            result = resultbuf;
            break;

        case VAR_SERVER_PROTOCOL:
            result = r->protocol;
            break;

        case VAR_SERVER_SOFTWARE:
            result = ap_get_server_version();
            break;

        case VAR_API_VERSION:
            ap_snprintf(resultbuf, sizeof(resultbuf), "%d:%d", MODULE_MAGIC_NUMBER_MAJOR, MODULE_MAGIC_NUMBER_MINOR);
            result = resultbuf;
            break;

        case VAR_TIME_YEAR:
            tc = time(NULL);
            tm = localtime(&tc);
            ap_snprintf(resultbuf, sizeof(resultbuf), "%02d%02d", (tm->tm_year / 100) + 19, tm->tm_year % 100);
            result = resultbuf;
            break;

        case VAR_TIME:
            tc = time(NULL);
            tm = localtime(&tc);
            ap_snprintf(resultbuf, sizeof(resultbuf), "%02d%02d%02d%02d%02d%02d%02d",
                     (tm->tm_year / 100) + 19, (tm->tm_year % 100),
                     tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min,
                     tm->tm_sec);
             
            result = resultbuf;
            break;

        case VAR_TIME_WDAY:
            tc = time(NULL);
            tm = localtime(&tc);
            ap_snprintf(resultbuf, sizeof(resultbuf), "%d", tm->tm_wday);
            result = resultbuf;
            break;

        case VAR_TIME_SEC:
            tc = time(NULL);
            tm = localtime(&tc);
            ap_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_sec);
            result = resultbuf;
            break;

        case VAR_TIME_MIN:
            tc = time(NULL);       
            tm = localtime(&tc);
            ap_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_min);
            result = resultbuf;
            break;

        case VAR_TIME_HOUR:
            tc = time(NULL);
            tm = localtime(&tc);
            ap_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_hour);
            result = resultbuf;
            break;

        case VAR_TIME_MON:
            tc = time(NULL);
            tm = localtime(&tc);
            ap_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_mon + 1);
            result = resultbuf;
            break;

        case VAR_TIME_DAY:
            tc = time(NULL);
            tm = localtime(&tc);
            ap_snprintf(resultbuf, sizeof(resultbuf), "%02d", tm->tm_mday);
            result = resultbuf;
            break;

        case VAR_SCRIPT_FILENAME:
        case VAR_REQUEST_FILENAME:
            result = r->filename;
            break;

        case VAR_PATH_INFO:
            result = r->path_info;
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;

        case VAR_THE_REQUEST:
            result = r->the_request;
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;

        case VAR_QUERY_STRING:
            result = r->args;
            if (result != NULL) result = normalise_uri(r, (char *)result, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            break;
            
        case VAR_HANDLER :
            result = r->handler;
            break;
            
        case VAR_COOKIE:
            result = ap_table_get(parsed_args, v->name);
            break;
    }

    if (result == NULL) {
        *resultbuf = 0;
        result = resultbuf;
    }

    return result;
}

request_rec *find_last_request(request_rec *r) {
    while (r->next != NULL) r = r->next;
    return r;
}

static void *sec_create_srv_config(pool *p, server_rec *s) {
    sec_srv_config *scfg = ap_pcalloc(p, sizeof(sec_srv_config));
    
    scfg->filter_engine = FILTERING_OFF;
    scfg->server_response_token = 0;
    scfg->scan_post = 0;

    scfg->action.log = 1;
    scfg->action.action = ACTION_DENY;
    scfg->action.status = HTTP_FORBIDDEN;

    scfg->signatures = ap_make_array(p, 10, sizeof(signature *));

    scfg->path = ap_pstrdup(p, ":server");

    scfg->auditlog_flag = 0;
    scfg->auditlog_name = NULL;
    scfg->auditlog_fd = -1;
    
    scfg->filter_debug_level = 0;
    
    scfg->range_start = 0;
    scfg->range_end = 255;
    scfg->check_encoding = 1;
    scfg->check_unicode_encoding = 0;
    scfg->chroot_dir = NULL;
    
    scfg->server_signature = NULL;

    return scfg;
}

static void *sec_create_dir_config(pool *p, char *path) {
    sec_dir_config *dcfg = (sec_dir_config *)ap_pcalloc(p, sizeof(*dcfg));
    
    dcfg->filter_engine = NOT_SET;
    dcfg->scan_post = -1;

    dcfg->is_action_default = 1;
    dcfg->action.log = 1;
    dcfg->action.action = ACTION_DENY;
    dcfg->action.status = HTTP_FORBIDDEN;

    dcfg->signatures = ap_make_array(p, 10, sizeof(signature *));

    if (path == NULL) {
        dcfg->path = ap_pstrdup(p, ":null");
    }
    else {
        dcfg->path = ap_pstrdup(p, path);
    }

    dcfg->auditlog_flag = -1;
    dcfg->auditlog_name = NULL;
    dcfg->auditlog_fd = -1;
    
    dcfg->filter_debug_level = 0;
    dcfg->filters_clear = 0;
    
    dcfg->range_start = 0;
    dcfg->range_end = 255;
    dcfg->check_encoding = 1;
    dcfg->check_unicode_encoding = -1;

    return dcfg;
}

static void *sec_merge_dir_config(struct pool *p, void *_parent, void *_child) {
    sec_dir_config *parent = (sec_dir_config *)_parent;
    sec_dir_config *child = (sec_dir_config *)_child;

    // merge child & parent into new

    sec_dir_config *new = (sec_dir_config *)ap_pcalloc(p, sizeof(*new));

    memcpy(new, child, sizeof(*child));

    new->signatures = ap_copy_array(p, child->signatures);
    if (child->filters_clear == 0) ap_array_cat(new->signatures, parent->signatures);

    if (child->scan_post == -1) {
        new->scan_post = parent->scan_post;
    }
    else {
        new->scan_post = child->scan_post;
    }

    if (child->filter_engine == NOT_SET) {
        new->filter_engine = parent->filter_engine;
    }
    else {
        new->filter_engine = child->filter_engine;
    }

    if (child->auditlog_fd == -1) {
        new->auditlog_fd = parent->auditlog_fd;
        new->auditlog_name = parent->auditlog_name;
    }
    else {
        new->auditlog_fd = child->auditlog_fd;
        new->auditlog_name = child->auditlog_name;
    }
    
    if (child->is_action_default == 1) {
        memcpy(&new->action, &parent->action, sizeof(parent->action));
        new->is_action_default = 0;
    }
    
    if (child->check_encoding == NOT_SET) {
        new->check_encoding = parent->check_encoding;
    }
    else {
        new->check_encoding = child->check_encoding;
    }
    
    if (child->check_unicode_encoding == NOT_SET) {
        new->check_unicode_encoding = parent->check_unicode_encoding;
    }
    else {
        new->check_unicode_encoding = child->check_unicode_encoding;
    }
        
    return new;
}

unsigned char x2c(unsigned char *what) {
    register unsigned char digit;

    digit = (what[0] >= 'A' ? ((what[0] & 0xdf) - 'A') + 10 : (what[0] - '0'));
    digit *= 16;
    digit += (what[1] >= 'A' ? ((what[1] & 0xdf) - 'A') + 10 : (what[1] - '0'));
    
    return(digit);
}

int detect_unicode_character(request_rec *r, char *p_read) {
    int unicode_len = 0;
    unsigned int d = 0;
    unsigned char c = *p_read;
    
        if ((c & 0xE0) == 0xC0) {
            // two byte unicode
            if (*(p_read + 1) == 0) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                unicode_len = 2;
                d = ((c & 0x1F) << 6) | (*(p_read + 1) & 0x3F);
                // sec_debug_log(r, 2, "2 Decoded character value: %x", d);
            }
        } else
        if ((c & 0xF0) == 0xE0) {
            // three byte unicode
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                unicode_len = 3;
                d = ((c & 0x0F) << 12) | ((*(p_read + 1) & 0x3F) << 6) | (*(p_read + 2) & 0x3F);
                // sec_debug_log(r, 2, "3 Decoded character value: %x", d);
            }
        } else
        if ((c & 0xF8) == 0xF0) {
            // four byte unicode
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                d = ((c & 0x07) << 18) | ((*(p_read + 1) & 0x3F) << 12) | ((*(p_read + 2) & 0x3F) < 6) | (*(p_read + 3) & 0x3F);
                unicode_len = 4;
                // sec_debug_log(r, 2, "4 Decoded character value: %x, *(p_read+3)=%x", d, *(p_read + 3));
            }
        } else
        if ((c & 0xFC) == 0xF8) {
            
            // five byte unicode
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)||(*(p_read + 4) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 4)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                d = ((c & 0x03) << 24) | ((*(p_read + 1) & 0x3F) << 18) | ((*(p_read + 2) & 0x3F) << 12) | ((*(p_read + 3) & 0x3F) << 6) | (*(p_read + 4) & 0x3F);
                unicode_len = 5;
                // sec_debug_log(r, 2, "5 Decoded character value: %x", d);
            }
        } else
        if ((c & 0xFE) == 0xFC) {
            // six byte unicode
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)||(*(p_read + 3) == 0)||(*(p_read + 4) == 0)||(*(p_read + 5) == 0)) unicode_len = UNICODE_ERROR_CHARACTERS_MISSING;
            else
            if (((*(p_read + 1)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 2)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 3)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 4)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else
            if (((*(p_read + 5)) & 0xC0) != 0x80) unicode_len = UNICODE_ERROR_INVALID_ENCODING;
            else {
                d = ((c & 0x01) << 30) | ((*(p_read + 1) & 0x3F) << 24) | ((*(p_read + 2) & 0x3F) << 18) | ((*(p_read + 3) & 0x3F) << 12) | ((*(p_read + 4) & 0x3F) << 6) | (*(p_read + 5) & 0x3F);
                unicode_len = 6;
                // sec_debug_log(r, 2, "6 Decoded character value: %x", d);
            }
        }
        
        if ((unicode_len > 0)&&((d & 0xFF) == d)) {
            unicode_len = UNICODE_ERROR_OVERLONG_CHARACTER;
        }
        
    return(unicode_len);
}

char *normalise_uri_inplace(request_rec *r, char *uri, int range_start, int range_end, int check_invalid_encoding, int check_invalid_unicode_encoding) {
    unsigned char *p_read, *p_write, *p_slash;
    unsigned char c;
    int count;
    
    if (uri == NULL) return NULL;
    
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    
    // loop #1 - URL decoding    
    while ((c = *p_read) != 0) {
        
        // decode URL decoding
        if (c == '+') c = 32;
        else
        if (c == '%') {
        
            // see if there are enough bytes available
            if ((*(p_read + 1) == 0)||(*(p_read + 2) == 0)) {
                if (check_invalid_encoding) {
                    sec_debug_log(r, 1, "Invalid URL encoding detected: not enough characters");
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Invalid URL encoding #1 detected.");
                    return NULL;
                } else c = 0;
            } else {
        
                // move onto the first hexadecimal letter
                p_read++;
                
                c = *p_read;
                
                if (check_invalid_encoding) {
                    if (! (((c >= '0')&&(c <= '9')) || ((c >= 'a')&&(c <= 'f')) || ((c >= 'A')&&(c <= 'F'))) ) {
                        sec_debug_log(r, 1, "Invalid URL encoding detected: invalid characters used");
                        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Invalid URL encoding #2 detected.");
                        return NULL;
                    }
                }
                
                c = *(p_read + 1);
                
                if (check_invalid_encoding) {
                    if (! (((c >= '0')&&(c <= '9')) || ((c >= 'a')&&(c <= 'f')) || ((c >= 'A')&&(c <= 'F'))) ) {
                        sec_debug_log(r, 1, "Invalid URL encoding detected: invalid characters used");
                        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Invalid URL encoding #2 detected.");
                        return NULL;
                    }
                }
                
                // decode two hexadecimal letters into a single byte
                c = x2c(p_read);
                p_read++;
            }
        }

        p_read++;
     
        if ((c < range_start)||(c > range_end)) {
            sec_debug_log(r, 1, "Invalid character detected [%i]", c);
            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Invalid character detected [%i]", c);
            return NULL;
        }
        
        if (c == 0) *p_write = ' ';
        else *p_write++ = c;
    }
    *p_write = 0;
    
    // loop #2 - decode everything else
    p_read = (unsigned char *)uri;
    p_write = (unsigned char *)uri;
    p_slash = NULL;
    count = 0;
    while (*p_read != 0) {
        c = *p_read;

        if (check_invalid_unicode_encoding) {
            int urc = detect_unicode_character(r, (char *)p_read);
            
            // sec_debug_log(r, 1, "Got %i from Unicode check", urc);
            
            switch(urc) {
                case UNICODE_ERROR_CHARACTERS_MISSING :
                    sec_debug_log(r, 1, "Invalid Unicode encoding: not enough bytes");
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Invalid Unicode encoding: not enough bytes");
                    return NULL;
                    break;
                case UNICODE_ERROR_INVALID_ENCODING :
                    sec_debug_log(r, 1, "Invalid Unicode encoding: invalid byte value");
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Invalid Unicode encoding: invalid byte value");
                    return NULL;
                    break;
                case UNICODE_ERROR_OVERLONG_CHARACTER :
                    sec_debug_log(r, 1, "Invalid Unicode encoding: overlong character");
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Invalid Unicode encoding: overlong character");
                    return NULL;
                    break;
            }
            
        }
        
        switch (c) {
        
            #ifdef WIN32
            case '\\' :
            #endif
        
            case '/' :
                if (p_slash == NULL) {
    
                    // remove the occurencies of "./"
                    if ( (count > 1) && ((*(p_write - 1) == '.') && (*(p_write - 2) == '/')) ) {
                        count -= 2;
                        p_write -= 2;
                    }
                    
                    p_slash = p_read;
                    *p_write++ = '/';
                    p_read++;
                    count++;
                } else {
                    p_read++;
                }
                break;

            default:
                // p_slash is used to detect more than one
                // '/' character in a row
                p_slash = NULL;
        
                // c = *p_read;
                if ((c < range_start)||(c > range_end)) {
                    sec_debug_log(r, 1, "Invalid character detected [%i]", c);
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Invalid character detected [%i]", c);
                    return NULL;
                }
        
                *p_write++ = c;
                
                p_read++;
                count++;
                
                break;
            }
    }

    *p_write = 0;
    return uri;
}

char *normalise_uri(request_rec *r, char *_uri, int range_start, int range_end, int check_invalid_encoding, int check_invalid_unicode_encoding) {
    char *uri;
    
    if (_uri == NULL) return NULL;
    
    uri = ap_pstrdup(r->pool, _uri);
    return normalise_uri_inplace(r, uri, range_start, range_end, check_invalid_encoding, check_invalid_unicode_encoding);
}

static int read_post_payload(request_rec *r, char **p) {
    request_body *rb = ap_palloc(r->pool, sizeof(request_body));
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);

    if (dcfg->scan_post != 1) {
        sec_debug_log(r, 1, "Not looking at POST, feature is disabled.");
        return DECLINED;
    }

    // is this a POST request?
    // if it is, we will handle it ourselves
    if (r->method_number == M_POST) {
        char *payload, *t;
        int i, rc, bufsize, payload_size = 0;

        if ((rc = ap_setup_client_block(r, REQUEST_CHUNKED_ERROR)) != OK) {
            sec_debug_log(r, 1, "ap_setup_client_block failed with %i", rc);
            return rc;
        }

        bufsize = r->remaining;
        *p = payload = t = ap_palloc(r->pool, r->remaining + 1);

        ap_hard_timeout("mod_security: receive request data", r);

        if (ap_should_client_block(r)) {
            while ((i = ap_get_client_block(r, t, bufsize)) > 0) {
                payload_size += i;
                bufsize -= i;
                t += i;
                ap_reset_timeout(r);
            }
        }

        ap_kill_timeout(r);

        payload[payload_size] = 0;

        rb->buffer = payload;
        rb->sofar = payload;
        rb->length = payload_size;
        rb->remaining = payload_size;
        
        // here we alter Apache internal structures
        // to point those pointers to our buffer
        // and to reset counters so that Apache will
        // think that the payload was not read at all
        r->connection->client->inptr = (unsigned char *)payload;
        r->connection->client->incnt = payload_size;
        r->read_length = 0;
        r->remaining = payload_size;

        ap_table_setn(r->notes, "mod_security-note", (char *)rb);
    
        sec_debug_log(r, 1, "Read %i bytes from POST.", rb->length);
    }

    return OK;
}

char *unescape_url(unsigned char *url) {
    register int x, y;

    for (x = 0, y = 0; url[y]; ++x, ++y) {
        if ((url[x] = url[y]) == '%') {
                url[x] = x2c(&url[y + 1]);
                y += 2;
        }
    }
    
    url[x] = '\0';
    return (char *)url;
}

void parse_arguments(char *s, table *parsed_args, request_rec *r, sec_dir_config *dcfg) {
    unsigned int inputlength, i, j;
    char *buf;
    int status;
    char *value;
    
    if (s == NULL) return;
    
    inputlength = strlen(s);

    buf = (char *)malloc(inputlength + 1);
    if (buf == NULL) {
        // log error
        return;
    }

    i = 0;
    j = 0;
    status = 0;
    while (i < inputlength) {
    
        while ((s[i] != '=') && (s[i] != '&') && (i < inputlength)) {
            if (s[i] == '+') buf[j] = 32;
            else buf[j] = s[i];

            j++;
            i++;
        }

        buf[j++] = '\0';

        if (status == 0) {
            status = 1;
            
            // unescape_url(buf);
            normalise_uri_inplace(r, buf, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);
            
            value = &buf[j];
        }
        else {
            status = 0;
            
            // unescape_url(value);
            normalise_uri_inplace(r, value, dcfg->range_start, dcfg->range_end, dcfg->check_encoding, dcfg->check_unicode_encoding);

            ap_table_add(parsed_args, buf, value);

            j = 0;
        }
        
        i++;
    }

    if (status == 1) {
        ap_table_add (parsed_args, buf, "");
    }

    free(buf);
    return;
}

char *remove_binary_content(request_rec *r, char *data) {
    long size = r->remaining;
    char *src, *dst, *newdata;

    // make a copy of the payload first
    dst = src = newdata = ap_palloc(r->pool, size + 1);
    memcpy(newdata, data, size);
    newdata[size] = 0;
    
    // remove zeros from the payload
    while(size--) {
        if (*src != 0) *dst++ = *src++;
        else src++;
    }
    
    return newdata;
}

static int sec_check_all_signatures(modsec_rec *msr) {
    request_rec *r = msr->r;
    signature **signatures;
    int i, rs;
    
    // todo: rename the variable below, it is misleading
    // msr->_the_request = NULL;
    
    if (r->args == NULL) {
        msr->_the_request = ap_pstrdup(r->pool, r->uri);
    } else {
        msr->_the_request = ap_pstrcat(r->pool, r->uri, "?", r->args, NULL);
    }

    // sec_debug_log(r, 1, "Check Unicode encoding: %i", msr->dcfg->check_unicode_encoding);
    msr->_the_request = normalise_uri_inplace(r, msr->_the_request, msr->dcfg->range_start, msr->dcfg->range_end, msr->dcfg->check_encoding, msr->dcfg->check_unicode_encoding);
    if (msr->_the_request == NULL) {
        ap_table_setn(r->headers_in, "mod_security-message", "Invalid character detected");
        if (msr->dcfg->action.log == 0) ap_table_setn(r->notes, "mod_security-noauditlog", "noauditlog");
        return msr->dcfg->action.status;
    }
    
    sec_debug_log(r, 4, "Normalised request: %s", msr->_the_request);
    
    sec_debug_log(r, 2, "Parsing arguments...");

    if (r->args != NULL) parse_arguments(r->args, msr->parsed_args, r, msr->dcfg);
    parse_cookies(r, msr->parsed_cookies);
    
    if (msr->dcfg->scan_post) {
        char *content_type;

        rs = read_post_payload(r, &msr->_post_payload);
        if (rs != OK) return rs;

        if (msr->_post_payload != NULL) {    
            content_type = (char *)ap_table_get(r->headers_in, "Content-Type");

            if (content_type != NULL) sec_debug_log(r, 3, "Content-Type=%s", content_type);
            else sec_debug_log(r, 3, "Content-Type is NULL");
            
            if ((content_type != NULL) && (strcmp(content_type, "application/x-www-form-urlencoded") == 0)) {
            
                // parse variables before normalising the bufffer
                if (msr->parsed_args != NULL) {
                    sec_debug_log(r, 3, "Parsing variables from POST payload");
                    parse_arguments(msr->_post_payload, msr->parsed_args, r, msr->dcfg);
                }
            
                msr->_post_payload = normalise_uri(r, msr->_post_payload, msr->dcfg->range_start, msr->dcfg->range_end, msr->dcfg->check_encoding, msr->dcfg->check_unicode_encoding);
            } else {
                // remove binary content from the payload
                sec_debug_log(r, 3, "Removing null bytes from POST payload");
                msr->_post_payload = remove_binary_content(r, msr->_post_payload);
            }
            
            if (msr->_post_payload == NULL) {
                return msr->dcfg->action.status;
            }
        }
    }
    
    {
        int mode = 0;
        int skip_count = 0;
        int rc = DECLINED;
    
        // loop through all signatures
        signatures = (signature **)msr->dcfg->signatures->elts;
        for (i = 0; i < msr->dcfg->signatures->nelts; i++) {
        
            // check if we need to skip this rule
            if (skip_count != 0) {
                skip_count--;
                continue;
            }

            // just clear the flag, we had to use the flag
            // to detect a case when the last rule in the
            // rule chain was marked as chained            
            if (mode == 2) mode = 0;
            
            // in mode 1, we are looking for the next filter
            // that is next chained, and then skip it to
            // execute the next filter in the chain
            if (mode == 1) {
                if (signatures[i]->is_chained == 0) mode = 0;
                continue;
            }
            
            msr->tmp_message = NULL;
            msr->tmp_redirect_url = NULL;
            msr->tmp_log_message = 0;
            
            rc = check_single_signature(msr, signatures[i]);
            // sec_debug_log(r, 5, "Rule rc = %i", rc);
            
            if (msr->tmp_message != NULL) {
                sec_debug_log(r, 1, msr->tmp_message);
                ap_table_setn(r->headers_in, "mod_security-message", msr->tmp_message);
                if (msr->tmp_log_message) {
                    ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: %s", msr->tmp_message);
                } else {
                    ap_table_setn(r->notes, "mod_security-noauditlog", "noauditlog");
                }
            }

            // DECLINED means that an allow action was called,
            // we need to pass the request through            
            if (rc == DECLINED) return DECLINED;

            // OK means there was no filter match, we
            // switch to mode 1, processing will continue
            // with the next filter chain
            if (rc == OK) {
                // we go into mode 1 (looking for the last rule
                // in the chain) if this rule is not it
                if (signatures[i]->is_chained == 1) mode = 1;
                continue;
            }
            
            // any status greater than zero means there
            // was a filter match, so we either stop execution
            // or proceed to the next rule if this rule was
            // chained
            if (rc > 0) {
                if (signatures[i]->is_chained == 1) {
                    mode = 2;
                    continue;
                }
                else {
rule_match:
                    if (msr->tmp_redirect_url != NULL) {
                        ap_table_setn(msr->r->headers_out, "Location", msr->tmp_redirect_url);
                    }
        
                    return rc;
                }
            }
            
            // if the return status is zero this
            // means skip some rules, so we skip
            if (rc == MODSEC_SKIP) {
                skip_count = signatures[i]->skip_count;
                continue;
            }
            
            sec_debug_log(r, 1, "Unprocessed return code [%i]", rc);
            return DECLINED;
        }
    
        // handle the case where there was a match on the
        // last filter so mode 2 check could not be done        
        // strickly speaking this should be a configuration error
        if (mode == 2) goto rule_match;
    }

    return DECLINED;
}

int check_single_signature(modsec_rec *msr, signature *sig) {
    int j, rs;

        // the idea behind non-selective filters is to apply them over
        // raw data, typically the complete first line of the request
        // and the complete POST payload
        
        if (sig->is_selective == 0) {
        
            sec_debug_log(msr->r, 2, "Checking signature \"%s\" at THE_REQUEST", sig->pattern);
            
            rs = check_sig_against_string(msr, sig, msr->_the_request, VAR_THE_REQUEST);
            if (rs != OK) return rs;

            if (msr->_post_payload != NULL) {
                sec_debug_log(msr->r, 2, "Checking signature \"%s\" at POST_PAYLOAD", sig->pattern);

                // todo: will this regex check work over several lines or not?
                rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                if (rs != OK) return rs;
            }
        }
        else {
            variable **variables;
            const char *v;

            // this is a selective signature, this means that we need to
            // check only one part of the request and leave the rest alone
            
            // selective signatures can be negative and non-negative;
            // non-negative signatures consist of a list of variables
            // that represent parts of the request that need to be
            // checked; negative signatures apply only to request
            // arguments, when you want to exclude an argument from
            // a check            

            if (sig->is_negative == 0) {
                
                // loop through signature variables and
                // check them

                variables = (variable **)sig->variables->elts;
                for (j = 0; j < sig->variables->nelts; j++) {

                    if (variables[j]->type == VAR_ARGS) {
                    
                        // special handling for this variable; with POST
                        // request arguments can exist in the URL and in
                        // the body of the request

                        variables[j]->type = VAR_QUERY_STRING; 
                        v = get_variable(msr->r, variables[j], msr->parsed_args);
                        variables[j]->type = VAR_ARGS;
            
                        if (msr->_post_payload != NULL) {
                        
                            // check the encoding
                            // add _post_payload to the v
                            // todo: handle multipart/form-data encoding
                            v = ap_pstrcat(msr->r->pool, v, msr->_post_payload, NULL);
                        }
                        
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at VAR_ARGS", sig->pattern);
                        
                        rs = check_sig_against_string(msr, sig, (char *)v, VAR_ARGS);
                        if (rs != OK) return rs;
                    }
                    else if (variables[j]->type == VAR_POST_PAYLOAD) {

                        // only check the payload if this is a POST request
                        if (msr->r->method_number == M_POST) {
                            if (msr->_post_payload == NULL) {
                                ap_log_rerror (APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, msr->r, "mod_security: Filtering against POST payload requested but payload is not available.");
                                sec_debug_log(msr->r, 1, "Filtering against POST payload requested but payload is not available.");
                            }
                            else {
                                sec_debug_log(msr->r, 2, "Checking signature \"%s\" at POST_PAYLOAD", sig->pattern);
                                
                                rs = check_sig_against_string(msr, sig, msr->_post_payload, VAR_POST_PAYLOAD);
                                if (rs != OK) return rs;
                            }
                        }
                    }
                    else if (variables[j]->type == VAR_ARGS_NAMES) {
                        table_entry *te;
                        array_header *arr;
                        int k;
                    
                        // todo: do we have parsed_args?
                        
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_NAMES", sig->pattern);
                    
                        arr = ap_table_elts(msr->parsed_args);
                        te = (table_entry *)arr->elts;
                        for (k = 0; k < arr->nelts; k++) {
                            rs = check_sig_against_string(msr, sig, te[k].key, VAR_ARGS_NAMES);
                            if (rs != OK) return rs;
                        }
                    }
                    else if (variables[j]->type == VAR_ARGS_VALUES) {
                        table_entry *te;
                        array_header *arr;
                        int k;
                    
                        // todo: do we have parsed_args?
                        
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at ARGS_VALUES", sig->pattern);
                    
                        arr = ap_table_elts(msr->parsed_args);
                        te = (table_entry *)arr->elts;
                        for (k = 0; k < arr->nelts; k++) {
                            rs = check_sig_against_string(msr, sig, te[k].val, VAR_ARGS_VALUES);
                            if (rs != OK) return rs;
                        }
                    }
                    else if(variables[j]->type == VAR_COOKIES_NAMES) {
                        table_entry *te;
                        array_header *arr;
                        int k;
                    
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at COOKIES_NAMES", sig->pattern);
                    
                        arr = ap_table_elts(msr->parsed_cookies);
                        te = (table_entry *)arr->elts;
                        for (k = 0; k < arr->nelts; k++) {
                            sec_debug_log(msr->r, 5, "Cookie [%s][%s]", te[k].key, te[k].val);
                            rs = check_sig_against_string(msr, sig, te[k].key, VAR_COOKIES_NAMES);
                            if (rs != OK) return rs;
                        }
                    }
                    else if(variables[j]->type == VAR_COOKIES_VALUES) {
                        table_entry *te;
                        array_header *arr;
                        int k;
                    
                        sec_debug_log(msr->r, 2, "Checking signature \"%s\" at COOKIES_VALUES", sig->pattern);
                    
                        arr = ap_table_elts(msr->parsed_cookies);
                        te = (table_entry *)arr->elts;
                        for (k = 0; k < arr->nelts; k++) {
                            sec_debug_log(msr->r, 5, "Cookie [%s][%s]", te[k].key, te[k].val);
                            rs = check_sig_against_string(msr, sig, te[k].val, VAR_COOKIES_VALUES);
                            if (rs != OK) return rs;
                        }
                    }
                    else {
                        // simple variable, get the value and check it

                        if (variables[j]->type == VAR_COOKIE) v = get_variable(msr->r, variables[j], msr->parsed_cookies);
                        else v = get_variable(msr->r, variables[j], msr->parsed_args);
                        if (v != NULL) {

                            if (variables[j]->name != NULL) {
                                sec_debug_log(msr->r, 2, "Checking signature \"%s\" at %s(%s)", sig->pattern, all_variables[variables[j]->type], variables[j]->name);
                            } else {                        
                                sec_debug_log(msr->r, 2, "Checking signature \"%s\" at %s", sig->pattern, all_variables[variables[j]->type]);
                            }
                            
                            sec_debug_log(msr->r, 5, "Variable value: [%s]", v);
                        
                            rs = check_sig_against_string(msr, sig, (char *)v, variables[j]->type);
                            if (rs != OK) return rs;
                        }
                        else {
                            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, msr->r, "mod_security: Variable not found [%s]", variables[j]->name);
                        }
                    }
                }
            }
            else {
                int k;
                table *our_parsed_args;
                table_entry *te;
                array_header *arr;
                char *v = NULL;
                unsigned int len = 0;

                if (msr->r->args != NULL) {
                    len = strlen(msr->r->args);
                }

                if (msr->_post_payload != NULL) {
                    len += strlen(msr->_post_payload);
                }

                if (len != 0) {
                    v = (char *)malloc(len + 1);
                    if (v == NULL) {
                        // error
                        // todo: could not allocate memory
                        return DECLINED;
                    }

                    *v = 0;
                    if (msr->parsed_args == NULL) {
                        ap_log_rerror (APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, msr->r, "mod_security: arguments are not parsed, internal error");
                        return DECLINED;
                    }

                    our_parsed_args = ap_copy_table(msr->r->pool, msr->parsed_args);

                    // find the unwanted variable names in the signature
                    // data and remove them from the variable list

                    variables = (variable **)sig->variables->elts;
                    for (j = 0; j < sig->variables->nelts; j++) {

                        if ((variables[j]->type == VAR_CUSTOM) && (variables[j]->action == VAR_ACTION_ALLOW)) {
                            ap_table_unset(our_parsed_args, variables[j]->name);
                        }
                    }
                    
                    // loop through remaining variables
                    // and create a single string out of them

                    arr = ap_table_elts(our_parsed_args);
                    te = (table_entry *)arr->elts;
                    for (k = 0; k < arr->nelts; k++) {

                        if (*v != 0) {
                            strcat(v, "&");
                        }

                        strcat(v, te[k].key);
                        strcat(v, "=");
                        strcat(v, te[k].val);
                    }

                    // make the check against the compiled string   
                    rs = check_sig_against_string(msr, sig, (char *)v, VAR_ARGS_SELECTIVE);

                    free(v);
            
                    if (rs != OK) return rs;
                }
            }
        }
        
    return OK;
}

static int sec_exec_child(void *_ed, child_info *pinfo) {
    char **env = NULL;
    int child_pid;
    exec_data *ed = (exec_data *)_ed;
    request_rec *r = ed->r;
      
    ap_add_cgi_vars(r);
    ap_add_common_vars(r);
    
    // todo: add mod_security specific environment variables
    
    // Hack to make PHP happy; PHP is often compiled to be
    // run from the command line and from the web server. It
    // analyses environment variables to determine how it is
    // supposed to behave. To make things worse, if it determines
    // that it is being run as a CGI script, there are certain
    // security checks it performs. In most cases, it simply
    // refuses to run when started. The following two lines
    // are a workaround around those security checks. 
    ap_table_add(r->subprocess_env, "PATH_TRANSLATED", ed->command);
    ap_table_add(r->subprocess_env, "REDIRECT_STATUS", "302");
    
    env = (char **)ap_create_environment(r->pool, r->subprocess_env);
    ap_error_log2stderr(r->server);
    
    r->filename = ed->command;
    
    // no arguments to minimise the risk of something going wrong
    // let the script get what is needed from environment variables
    r->args = NULL;
    
    ap_cleanup_for_exec();
    
    child_pid = ap_call_exec(r, pinfo, r->filename, env, 1);

#ifdef WIN32
    return(child_pid);
#else
    ap_log_error(APLOG_MARK, APLOG_ERR, NULL, "exec failed: %s", r->filename); 
    exit(0);
    
    return(0);
#endif
}
                             

int check_sig_against_string(modsec_rec *msr, signature *_sig, const char *s, int var_type) {
    signature *default_action = &msr->dcfg->action;
    request_rec *r = msr->r;
    BUFF *p1, *p2, *p3;
    int regex_result;
    int rc = OK;
    
    if (_sig->regex == NULL) {
        sec_debug_log(r, 1, "Compiled regex for pattern [%s] is NULL!", _sig->pattern);
        return OK;
    }
    
    sec_debug_log(r, 9, "check_sig_against_string: %s", s);

    regex_result = regexec(_sig->regex, s, 0, NULL, 0);
    if ( ((regex_result == 0)&&(_sig->is_allow == 0)) || ((regex_result == 1)&&(_sig->is_allow == 1)) ) {

        signature *sig = _sig;
        
        if (sig->is_default_action_used == 1) sig = default_action;

        // execute the external script
        if (sig->exec) {
            exec_data *ed = NULL;
            char buf[4097];
            
            ed = ap_palloc(r->pool, sizeof(exec_data));;
            ed->r = r;
            ed->command = sig->exec_string;
            
            sec_debug_log(r, 1, "Executing command \"%s\"", ed->command);
            
            ap_table_setn(r->headers_in, "mod_security-executed", ed->command);

            if (!ap_bspawn_child(r->main ? r->main->pool : r->pool, sec_exec_child, (void *)ed, kill_after_timeout, &p1, &p2, &p3)) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r->server, "mod_security: couldn't spawn child process: %s", sig->exec_string);
                return HTTP_INTERNAL_SERVER_ERROR;
            }
            
            while (ap_bgets(buf, 4096, p2) > 0) continue;
            while (ap_bgets(buf, 4096, p3) > 0) continue;
        }
        
        if (sig->pause != 0) {
            sec_debug_log(r, 1, "Pausing [%s] for %i ms", r->uri, sig->pause);
            ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: pausing [%s] for %i ms", r->uri, sig->pause);
            
            #ifdef WIN32
            _sleep(sig->pause);
            #elif defined NETWARE
            delay(sig->pause);
            #else
            usleep(sig->pause * 1000);
            #endif
        }
        
        // perform action
        switch(sig->action) {
        
            case ACTION_SKIP :
                sec_debug_log(r, 2, "Skipping %i statements on pattern match \"%s\" at %s", sig->skip_count, _sig->pattern, all_variables[var_type]);
                rc = MODSEC_SKIP;
                break;
        
            case ACTION_ALLOW :
                // sec_debug_log(r, 2, "Access allowed based on pattern match \"%s\" at %s", _sig->pattern, all_variables[var_type]);
                msr->tmp_message = ap_psprintf(r->pool, "Access allowed based on pattern match \"%s\" at %s", _sig->pattern, all_variables[var_type]);
                
                // returning DECLINED will interrupt filter processing but
                // it won't do anything the the request
                rc = DECLINED;
                break;
        
            case ACTION_DENY :
                msr->tmp_message = ap_psprintf(r->pool, "Access denied with code %i. Pattern match \"%s\" at %s.", sig->status, _sig->pattern, all_variables[var_type]);
                rc = sig->status;
                break;
                
            case ACTION_REDIRECT :
                msr->tmp_message = ap_psprintf(r->pool, "Access denied with redirect to [%s]. Pattern match \"%s\" at %s.", sig->redirect_url, _sig->pattern, all_variables[var_type]);
                msr->tmp_redirect_url = sig->redirect_url;
                rc = REDIRECT;
                break;
                
            default :
                msr->tmp_message = ap_psprintf(r->pool, "Warning. Pattern match \"%s\" at %s.", _sig->pattern, all_variables[var_type]);
                break;
        }
        
        if ((msr->tmp_message != NULL)&&(sig->log)) msr->tmp_log_message = 1;
    }

    return rc;
}


char *parse_action(char *p2, signature *sig, pool *_pool) {
    char *t = ap_pstrdup(_pool, p2);
    char *p = strtok(t, ",");
    int found;

    while (p != NULL) {
        found = 0;
        
        if (strcmp (p, "log") == 0) {
            found = 1;
            sig->log = 1;
        }
        else if (strcmp(p, "nolog") == 0) {
            found = 1;
            sig->log = 0;
        }
        else if (strncmp(p, "status", 6) == 0) {
            found = 1;
            if (strlen(p) > 8) {
                sig->status = atoi(p + 7);
            }
        }
        else if (strcmp(p, "deny") == 0) {
            found = 1;
            sig->action = ACTION_DENY;
        }        
        else if (strcmp(p, "pass") == 0) {
            found = 1;
            sig->action = ACTION_NONE;
        }
        else if (strcmp(p, "allow") == 0) {
            found = 1;
            sig->action = ACTION_ALLOW;
        }
        else if (strcmp(p, "chain") == 0) {
            found = 1;
            sig->is_chained = 1;
        }
        else if (strcmp(p, "chained") == 0) {
            found = 1;
            sig->is_chained = 1;
        }
        else if (strncmp(p, "skipnext", 8) == 0) {
            found = 1;
            sig->action = ACTION_SKIP;
            if (strlen(p) > 9) {
                sig->skip_count = atoi(p + 9);
            }
        }
        else if (strncmp(p, "skip", 4) == 0) {
            found = 1;
            sig->action = ACTION_SKIP;
            if (strlen(p) > 9) {
                sig->skip_count = atoi(p + 9);
            }
        }
        else if (strncmp(p, "exec", 4) == 0) {
            found = 1;
            sig->exec = 1;
            if (strlen(p) > 6) {
                sig->exec_string = ap_pstrdup(_pool, p + 5);
            }
        }
        else if (strncmp(p, "redirect", 8) == 0) {
            found = 1;
            sig->action = ACTION_REDIRECT;
            if (strlen(p) > 10) {
                sig->redirect_url = ap_pstrdup(_pool, p + 9);
            }
        }
        else if (strncmp(p, "msg", 3) == 0) {
            found = 1;
            if (strlen(p) > 5) {
                sig->msg = ap_pstrdup(_pool, p + 4);
            }
        }
        else if (strncmp(p, "id", 2) == 0) {
            found = 1;
            if (strlen(p) > 4) {
                sig->id = ap_pstrdup(_pool, p + 3);
            }
        }
        else if (strncmp(p, "pause", 5) == 0) {
            found = 1;
            if (strlen(p) > 7) {
                sig->pause = atoi(p + 6);
            }
        }
        
        if (found == 0) {
            return ap_psprintf(_pool, "Unknown mod_security action \"%s\"", p);
        }

        p = strtok(NULL, ",");
    }

    return NULL;
}

static const char *cmd_filter_engine(cmd_parms *cmd, sec_dir_config* dcfg, char *p1) {
    if (strcasecmp(p1, "On") == 0) dcfg->filter_engine = FILTERING_ON;
    else
    if (strcasecmp(p1, "Off") == 0) dcfg->filter_engine = FILTERING_OFF;
    else
    if (strcasecmp(p1, "DynamicOnly") == 0) dcfg->filter_engine = FILTERING_DYNAMIC_ONLY;
    else
    return ap_psprintf(cmd->pool, "Unrecognized parameter value for SecFilterEngine: %s", p1);
    
    return NULL;
}

static const char *cmd_filter_inheritance(cmd_parms *cmd, sec_dir_config* dcfg, int flag) {
    if (flag) dcfg->filters_clear = 0;
    else dcfg->filters_clear = 1;
    return NULL;
}

static const char *cmd_server_response_token(cmd_parms *cmd, sec_dir_config *dcfg, int flag) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    scfg->server_response_token = flag;
    return NULL;
}

static const char *cmd_filters_clear(cmd_parms *cmd, sec_dir_config *dcfg, int flag) {
    dcfg->filters_clear = flag;
    return NULL;
}

static const char *cmd_audit_engine(cmd_parms *cmd, sec_dir_config * dcfg, char *p1) {
    if (strcasecmp(p1, "On") == 0) dcfg->auditlog_flag = AUDITLOG_ON;
    else
    if (strcasecmp(p1, "Off") == 0) dcfg->auditlog_flag = AUDITLOG_OFF;
    else
    if (strcasecmp(p1, "RelevantOnly") == 0) dcfg->auditlog_flag = AUDITLOG_RELEVANT_ONLY;
    else
    if (strcasecmp(p1, "DynamicOrRelevant") == 0) dcfg->auditlog_flag = AUDITLOG_DYNAMIC_OR_RELEVANT;
    else
    return ap_psprintf(cmd->pool, "Unrecognized parameter value for SecAuditEngine: %s", p1);
    
    return NULL;
}

static const char *cmd_audit_log(cmd_parms *cmd, sec_dir_config * dcfg, char *p1) {
    dcfg->auditlog_name = ap_server_root_relative(cmd->pool, p1);

    #ifdef WIN32
    dcfg->auditlog_fd = open(dcfg->auditlog_name, O_WRONLY | O_APPEND | O_CREAT, CREATEMODE); 
    #else
    dcfg->auditlog_fd = ap_popenf(cmd->pool, dcfg->auditlog_name, O_WRONLY | O_APPEND | O_CREAT, CREATEMODE); 
    #endif    

    if (dcfg->auditlog_fd < 0) {
        return "mod_security: Failed to open the audit log file.";
    }

    return NULL;
}

static const char *cmd_scan_post(cmd_parms * cmd, sec_dir_config * dcfg, int flag) {
    dcfg->scan_post = flag;
    return NULL;
}

static const char *cmd_filter_check_url_encoding(cmd_parms * cmd, sec_dir_config *dcfg, int flag) {
    dcfg->check_encoding = flag;
    return NULL;
}

static const char *cmd_filter_check_unicode_encoding(cmd_parms * cmd, sec_dir_config *dcfg, int flag) {
    dcfg->check_unicode_encoding = flag;
    return NULL;
}

static const char *cmd_filter_force_byte_range(cmd_parms *cmd, sec_dir_config *dcfg, char *p1, char *p2) {
    dcfg->range_start = atoi(p1);
    dcfg->range_end = atoi(p2);
    return NULL;
}

static const char *cmd_default_action(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    dcfg->is_action_default = 0;
    return parse_action(p1, &dcfg->action, cmd->pool);
}

static const char *cmd_filter(cmd_parms *cmd, sec_dir_config *dcfg, char *p1, char *p2) {
    signature *sig;

    sig = ap_pcalloc(cmd->pool, sizeof(signature));
    sig->status = dcfg->action.status;
    sig->action = dcfg->action.action;
    sig->skip_count = 1;
    sig->is_chained = 0;
    
    // p1 is the regular expression string
    if (p1[0] == '!') {
        sig->is_allow = 1;
        sig->pattern = p1;
        sig->regex = ap_pregcomp(cmd->pool, p1 + 1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    } else {
        sig->pattern = p1;
        sig->regex = ap_pregcomp(cmd->pool, p1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    
    if (sig->regex == NULL) {
        return ap_psprintf(cmd->pool, "Invalid regular expression: %s", sig->pattern);
    }
    
    // p2 is an optional action string
    if (p2 != NULL) {
        char *rc;
        sig->is_default_action_used = 0;
        rc = parse_action(p2, sig, cmd->pool);
        if (rc != NULL) return rc;
    }
    else {
        sig->is_default_action_used = 1;
    }

    // add the signature to the list of all signatures
    *(signature **)ap_push_array(dcfg->signatures) = sig;

    return NULL;
}

static const char *cmd_filter_debug_log(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    
    scfg->debuglog_name = ap_server_root_relative(cmd->pool, p1);

    #ifdef WIN32
    scfg->debuglog_fd = open(scfg->debuglog_name, O_WRONLY | O_APPEND | O_CREAT, CREATEMODE); 
    #else
    scfg->debuglog_fd = ap_popenf(cmd->pool, scfg->debuglog_name, O_WRONLY | O_APPEND | O_CREAT, CREATEMODE); 
    #endif   
    
    if (scfg->debuglog_fd < 0) {
        return "mod_security: Failed to open the debug log file.";
    }
    
    return NULL;
}

static const char *cmd_filter_debug_level(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    dcfg->filter_debug_level = atoi(p1);
    return NULL;
}

static const char *cmd_filter_selective(cmd_parms *cmd, sec_dir_config *dcfg, char *p1, char *p2, char *p3) {
    char *p;
    char *t;
    signature *sig;
    
    // initialise the structure first    
    sig = ap_pcalloc(cmd->pool, sizeof (signature));
    
    sig->status = dcfg->action.status;
    sig->action = dcfg->action.action;
    sig->skip_count = 1;
    sig->is_chained = 0;
    sig->is_allow = 0;
    sig->is_selective = 1;
    sig->is_negative = 0;
    sig->requires_parsed_args = 0;
    sig->variables = ap_make_array(cmd->pool, 10, sizeof (variable *));
    
    if (p2[0] == '!') {
        sig->is_allow = 1;
        sig->pattern = p2;
        sig->regex = ap_pregcomp(cmd->pool, p2 + 1, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    } else {
        sig->pattern = p2;
        sig->regex = ap_pregcomp(cmd->pool, p2, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    }
    
    if (sig->regex == NULL) {
        return ap_psprintf(cmd->pool, "Invalid regular expression: %s", sig->pattern);
    }
    
    // split parameter 1 apart and extract variable names
    
    p = strdup(p1);
    t = strtok (p, "|");
    while (t != NULL) {

        // add the token to the list
        variable *v = (variable *)ap_pcalloc(cmd->pool, sizeof(variable));
        char *x = t;

        v->type = VAR_UNKNOWN;

        // when ! is the first character in the variable
        // name that means that the restrictions need to be
        // relaxed for that variable (within the filter scope)
        if (t[0] == '!') {
            v->action = VAR_ACTION_ALLOW;
            sig->is_negative = 1;
                sig->requires_parsed_args = 1;
            x++;
        }
        else {
            v->action = VAR_ACTION_DENY;
        }

        // arguments
        if (strncmp (x, "ARG_", 4) == 0) {
            v->type = VAR_CUSTOM;
            v->name = ap_pstrdup(cmd->pool, x + 4);
            sig->requires_parsed_args = 1;
        }
        // HTTP headers
        else if (strncmp (x, "HTTP_", 5) == 0) {
            char *px;

            v->type = VAR_HEADER;
            v->name = ap_pstrdup (cmd->pool, x + 5);
    
            // replace all "_" with "-"
            px = v->name;
            while (*px != 0) {
                if (*px == '_') *px = '-';
                px++;
            }
        }
        // COOKIES
        else if (strncmp(x, "COOKIE_", 7) == 0) {
            v->type = VAR_COOKIE;
            v->name = ap_pstrdup(cmd->pool, x + 7);
        }
        // environment variables
        else if (strncmp (x, "ENV_", 4) == 0) {
            v->type = VAR_ENV;
            v->name = ap_pstrdup(cmd->pool, x + 4);
        }
        // all arguments
        else if (strcmp (x, "ARGS") == 0) {
            v->type = VAR_ARGS;
                v->name = ap_pstrdup(cmd->pool, x);
        }
        // just the post payload
        else if (strcmp(x, "POST_PAYLOAD") == 0) {
            v->type = VAR_POST_PAYLOAD;
            v->name = ap_pstrdup(cmd->pool, x);
        }
        // everything else
        else {
            char **vl = all_variables;
            int i = 0;

            while (*vl != NULL) {
                if (strcmp(*vl, x) == 0) {
                    v->type = i;
                    v->name = ap_pstrdup(cmd->pool, x);
                    break;
                }

                i++;
                vl++;
            }
        }

        if (v->type == VAR_UNKNOWN) {
            v->name = ap_pstrdup(cmd->pool, "UKNOWN");
            return ap_psprintf(cmd->pool, "Unknown variable name: %s", x);
        }

        if ((v->type == VAR_ARGS_NAMES)||(v->type == VAR_ARGS_VALUES)) sig->requires_parsed_args = 1;

        *(variable **)ap_push_array(sig->variables) = v;

        // and proceed to the next token
        t = strtok (NULL, "|");
    }

    free(p);

    // is default action parameter present?
    if (p3 != NULL) {
        char *rc;
        sig->is_default_action_used = 0;
        rc = parse_action(p3, sig, cmd->pool);
        if (rc != NULL) return rc;
    }
    else {
        sig->is_default_action_used = 1;
    }

    // add the signature to the list of all signatures
    *(signature **) ap_push_array (dcfg->signatures) = sig;

    return NULL;
}

static const char *cmd_chroot_dir(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    scfg->chroot_dir = p1;
    return NULL;
}            

static const char *cmd_server_signature(cmd_parms *cmd, sec_dir_config *dcfg, char *p1) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(cmd->server->module_config, &security_module);
    scfg->server_signature = p1;
    return NULL;
}            

static const command_rec sec_cmds[] = {

     {
         "SecFilter",
         cmd_filter,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE12,
         ""
     },
     
     {
     
        "SecFilterDebugLog",
        cmd_filter_debug_log,
        NULL,
        RSRC_CONF,
        TAKE1,
        ""
     },
          
     {
         "SecFilterDebugLevel",
         cmd_filter_debug_level,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         ""
     },

     {
         "SecFilterSelective",
         cmd_filter_selective,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE23,
         ""
     },

     {
         "SecFilterEngine",
         cmd_filter_engine,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         ""
     },
     
     {
        "SecServerResponseToken",
        cmd_server_response_token,
        NULL,
        RSRC_CONF,
        FLAG,
        ""        
     },

     {
         "SecFilterScanPOST",
         cmd_scan_post,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         FLAG,
         ""
     },

     {
         "SecFilterDefaultAction",
         cmd_default_action,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         ""
     },
     
     {
         "SecFiltersClear",
         cmd_filters_clear,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         FLAG,
         ""
     },
     
     {
        "SecFilterInheritance",
        cmd_filter_inheritance,
        NULL,
        RSRC_CONF | OR_AUTHCFG,
        FLAG,
        ""
     },

     {
         "SecAuditEngine",
         cmd_audit_engine,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         ""
     },

     {
         "SecAuditLog",
         cmd_audit_log,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE1,
         ""
     },
     
     {
         "SecFilterCheckURLEncoding",
         cmd_filter_check_url_encoding,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         FLAG,
         ""
     },
     
     {
         "SecFilterCheckUnicodeEncoding",
         cmd_filter_check_unicode_encoding,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         FLAG,
         ""
     },
     
     {
         "SecFilterForceByteRange",
         cmd_filter_force_byte_range,
         NULL,
         RSRC_CONF | OR_AUTHCFG,
         TAKE2,
         ""
     },
     
     {
         "SecChrootDir",
         cmd_chroot_dir,
         NULL,
         RSRC_CONF,
         TAKE1,
         ""
     },
     
     {
         "SecServerSignature",
         cmd_server_signature,
         NULL,
         RSRC_CONF,
         TAKE1,
         ""
     },

    { NULL }
};

static int sec_logger(request_rec *_r) {
    request_rec *r = find_last_request(_r);
    char *o1 = ap_palloc(r->pool, AUDIT_BUF_SIZE + 1);
    char *o2 = ap_palloc(r->pool, AUDIT_BUF_SIZE + 1);
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    const char *modsec_message = NULL;
    array_header *arr;
    table_entry *te;
    char *remote_user, *local_user, *t;
    int i = 0;
    
    // do nothing if not initialised properly    
    if (dcfg->filter_engine == NOT_SET) {
        return DECLINED;
    }

    sec_debug_log(r, 2, "sec_logger executed to perform audit log.");

    switch(dcfg->auditlog_flag) {
    
        case AUDITLOG_OFF :
            sec_debug_log(r, 2, "Audit log is off for [%s]", r->uri);
            return OK;
            break;
            
        case AUDITLOG_DYNAMIC_OR_RELEVANT :
            modsec_message = ap_table_get(r->headers_in, "mod_security-message");
            if (((modsec_message == NULL)&&(r->handler == NULL))||(ap_table_get(_r->notes, "mod_security-noauditlog") != NULL)) {
                sec_debug_log(r, 2, "Audit log: ignoring a non-dynamic and non-relevant request [content-type=%s]", r->content_type);
                return OK;
            }
            break;
            
        case AUDITLOG_RELEVANT_ONLY :
            modsec_message = ap_table_get(r->headers_in, "mod_security-message");
            if ((modsec_message == NULL)||(ap_table_get(_r->notes, "mod_security-noauditlog") != NULL)) {
                sec_debug_log(r, 2, "Audit log: ignoring a non-relevant request [content-type=%s]", r->content_type);
                return OK;
            }
            break;
    }
            
    // return immediatelly if we don't have a file to write to
    if (dcfg->auditlog_fd == -1) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "mod_security: Audit log not found. [%s]", r->uri);
        sec_debug_log(r, 1, "Audit log not found [%s]", r->uri);
        return OK;
    }
    
    strcpy (o1, "");
    strncat (o1, "========================================\n", AUDIT_BUF_SIZE);

    if (r->connection->remote_logname == NULL) {
        remote_user = "-";
    }
    else {
        remote_user = r->connection->remote_logname;
    }

    if (r->connection->user == NULL) {
        local_user = "-";
    }
    else {
        local_user = r->connection->user;
    }

    // todo: use r->request_time
    t = ap_psprintf(r->pool, "Request: %s %s %s [%s] \"%s\" %i %li\n",
                     r->connection->remote_ip, remote_user, local_user,
                     ap_get_time(), r->the_request, r->status,
                     r->bytes_sent);
             
    strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));

    t = ap_psprintf (r->pool, "Handler: %s\n", r->handler);
    strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));
    
    // see if there is an error message stored in notes
    t = (char *)ap_table_get(r->notes, "error-notes");
    if (t != NULL) {
        t = ap_psprintf(r->pool, "Error: %s\n", t);
        strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));
    }

    strncat(o1, "----------------------------------------\n", AUDIT_BUF_SIZE - strlen(o1));

    // input headers
    t = ap_psprintf (r->pool, "%s\n", r->the_request);
    strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));

    arr = ap_table_elts(r->headers_in);

    te = (table_entry *) arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = ap_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o1, t, AUDIT_BUF_SIZE - strlen(o1));
    }

    strncat(o1, "\n", AUDIT_BUF_SIZE - strlen(o1));

    if (r->method_number == M_POST) {
        strcpy(o2, "\n\n");
    } else {
        strcpy(o2, "");
    }

    // output headers

    // the status first
    t = ap_psprintf(r->pool, "%s %s\n", r->protocol, r->status_line);
    strncat(o2, t, AUDIT_BUF_SIZE - strlen(o2));


    // output headers
    arr = ap_table_elts (r->headers_out);

    te = (table_entry *) arr->elts;
    for (i = 0; i < arr->nelts; i++) {
        t = ap_psprintf(r->pool, "%s: %s\n", te[i].key, te[i].val);
        strncat(o2, t, AUDIT_BUF_SIZE - strlen(o2));
    }

    // write to the file
    fd_lock(r, dcfg->auditlog_fd);
    
    write(dcfg->auditlog_fd, o1, strlen(o1));
    
    if (r->method_number == M_POST) {
        request_body *rb = (request_body *)ap_table_get(r->notes, "mod_security-note");
        
        if (rb != NULL) {
            write(dcfg->auditlog_fd, rb->buffer, rb->length);
            }
        else {
            char *message = "[POST payload not available]";
            write(dcfg->auditlog_fd, message, strlen(message));
        }
    }

    // write the second part of the log
    write(dcfg->auditlog_fd, o2, strlen(o2));
    fd_unlock(r, dcfg->auditlog_fd);

    return OK;
}

static int sec_check_access(request_rec *r) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(r->server->module_config, &security_module);
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    modsec_rec *msr;
    int rc;
    
    // refuse to work if the per dir config is null
    if (dcfg == NULL) {
        sec_debug_log(r, 2, "Filtering off, dcfg is null.");
        return DECLINED;
    }

    // also refuse to work if not configured properly    
    if (dcfg->filter_engine == NOT_SET) {
        return DECLINED;
    }    

    if (scfg->server_signature != NULL) {
        char *server_version = (char *)ap_get_server_version();
        if (server_version != NULL) {
            // sec_init should make sure that we have enought
            // space for the signature but we are checking this
            // here just in case
            if (strlen(server_version) >= strlen(scfg->server_signature)) {
                strcpy(server_version, scfg->server_signature);
            } else {
                sec_debug_log(r, 1, "SecServerSignature: not enough space");
            }
        }
    }

    // refuse to work if this is a subrequest
    if (!ap_is_initial_req(r)) {
        sec_debug_log(r, 2, "Filtering off for a subrequest.");
        return DECLINED;
    }

    // refuse to work if filtering is off
    if (dcfg->filter_engine == FILTERING_OFF) {
        sec_debug_log(r, 2, "Filtering off, switched off for path [%s]", dcfg->path);
        return DECLINED;
    }
    
    if ((dcfg->filter_engine == FILTERING_DYNAMIC_ONLY)&&(r->handler == NULL)) {
        sec_debug_log(r, 2, "Filtering off for non-dynamic resources [content-type=%s]", r->content_type);
        return DECLINED;
    }

    sec_debug_log(r, 2, "Checking with per-dir-config [%s][%s]", dcfg->path, r->uri);
    
    msr = (modsec_rec *)ap_pcalloc(r->pool, sizeof(*msr));
    msr->r = r;
    msr->scfg = scfg;
    msr->dcfg = dcfg;
    msr->_the_request = NULL;
    msr->_post_payload = NULL;
    msr->parsed_args = ap_make_table(r->pool, 10);
    msr->parsed_cookies = ap_make_table(r->pool, 10);
    
    rc = sec_check_all_signatures(msr);
    
    // make a note for the logger
    if (rc != DECLINED) {
        char *note = ap_psprintf(r->pool, "%i", rc);
        ap_table_setn(r->headers_in, "mod_security-action", note);
    }
    
    return rc;
}

static char *current_logtime(request_rec *r) {
    int timz;
    struct tm *t;
    char tstr[80];
    char sign;

    t = ap_get_gmtoff(&timz);
    sign = (timz < 0 ? '-' : '+');
    if (timz < 0) {
        timz = -timz;
    }

    strftime(tstr, 80, "[%d/%b/%Y:%H:%M:%S ", t);
    ap_snprintf(tstr + strlen(tstr), 80-strlen(tstr), "%c%.2d%.2d]", sign, timz / 60, timz % 60);
    return ap_pstrdup(r->pool, tstr);
}

static void sec_debug_log(request_rec *r, int level, const char *text, ...) {
    // ap_log_rerror(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r, "xxx: %s", text);
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(r->server->module_config, &security_module);
    sec_dir_config *dcfg = (sec_dir_config *)ap_get_module_config(r->per_dir_config, &security_module);
    va_list ap;
    char str1[1024] = "";
    char str2[1256] = "";

    va_start(ap, text);
    
    if (scfg->debuglog_fd < 0) {
        return;
    }
    
    if (level > dcfg->filter_debug_level) {
        return;
    }

    ap_vsnprintf(str1, sizeof(str1), text, ap);
    ap_snprintf(str2, sizeof(str2), "%s [%s/sid#%lx][rid#%lx][%s] %s\n", current_logtime(r), ap_get_server_name(r), (unsigned long)(r->server), (unsigned long)r, r->uri, str1); 

    fd_lock(r, scfg->debuglog_fd);
    write(scfg->debuglog_fd, str2, strlen(str2));
    fd_unlock(r, scfg->debuglog_fd);

    va_end(ap);
    
    return;
}

void sec_init(server_rec *s, pool *p) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);
    
    if (scfg->server_response_token) {    
        ap_add_version_component(MOD_SECURITY_SIGNATURE);
    }

    // this might not make sense at first glance but we are
    // actually just allocating space here; we will hack the
    // server name later
    if (scfg->server_signature) {
        ap_add_version_component(scfg->server_signature);
    }
    
    #if !(defined(WIN32)||defined(NETWARE))
    
    if (scfg->chroot_dir != NULL) {
    
        // We do this here because modules are normally
        // initialised twice, and, as far as I could tell,
        // you can't keep any information between initialisations
        // to tell yourself whether the module is initalised for
        // the first or for the second time. Of course, calling
        // chroot during first initialisation is no good. However,
        // it seems that the parent pid of the Apache process is
        // always 1 during the second initialisation. So, fingers
        // crossed.
        
        if (getppid() == 1) {
            if (chroot(scfg->chroot_dir) < 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, s, "mod_security: chroot failed, path=%s, errno=%i", scfg->chroot_dir, errno);
            } else {
                ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, s, "mod_security: performed chroot, path=%s", scfg->chroot_dir);
            }
        }
    }
                                                
    #endif    
}

void sec_child_init(server_rec *s, pool *p) {
    sec_srv_config *scfg = (sec_srv_config *)ap_get_module_config(s->module_config, &security_module);
    
    if (scfg->server_signature != NULL) {
        char *server_version = (char *)ap_get_server_version();
        if (server_version != NULL) {
            // sec_init should make sure that we have enought
            // space for the signature but we are checking this
            // here just in case
            if (strlen(server_version) >= strlen(scfg->server_signature)) {
                strcpy(server_version, scfg->server_signature);
            }
        }
    }
}

#ifdef WIN32
#define USE_LOCKING
#endif

#ifndef USE_FCNTL
#ifndef USE_FLOCK
#ifndef USE_LOCKING
#ifndef NETWARE
#define USE_FCNTL
#endif
#endif
#endif
#endif

#ifdef USE_FCNTL
static struct flock   lock_it;
static struct flock unlock_it;
#endif

static void fd_lock(request_rec *r, int fd) {
    int rc = 0;

    #ifdef USE_FCNTL
    lock_it.l_whence = SEEK_SET;
    lock_it.l_start = 0;
    lock_it.l_len = 0;
    lock_it.l_type = F_WRLCK;
    lock_it.l_pid = 0;

    while ( ((rc = fcntl(fd, F_SETLKW, &lock_it)) < 0) && (errno == EINTR) ) {
        continue;
    }
    #endif

    #ifdef USE_FLOCK
    while ( ((rc = flock(fd, LOCK_EX)) < 0) && (errno == EINTR) ) {
        continue;
    }
    #endif

    #ifdef USE_LOCKING
    lseek(fd, 0, SEEK_SET);
    rc = _locking(fd, _LK_LOCK, 1);
    lseek(fd, 0, SEEK_END);
    #endif

    #ifdef NETWARE
	if ((locking_sem != 0) && (TimedWaitOnLocalSemaphore(locking_sem, 10000) != 0)) {
		rc = -1;
	} else {
		rc = 1;
	}
    #endif

    if (rc < 0) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "mod_security: Locking failed");
        exit(1);
    }
    return;
}

static void fd_unlock(request_rec *r, int fd) {
    int rc = 0;

    #ifdef USE_FCNTL
    unlock_it.l_whence = SEEK_SET;
    unlock_it.l_start  = 0;
    unlock_it.l_len    = 0;
    unlock_it.l_type   = F_UNLCK;
    unlock_it.l_pid    = 0;

    while( ((rc = fcntl(fd, F_SETLKW, &unlock_it)) < 0) && (errno == EINTR) ) {
        continue;
    }
    #endif
    
    #ifdef USE_FLOCK
    while ( ((rc = flock(fd, LOCK_UN)) < 0) && (errno == EINTR) ) {
        continue;
    }
    #endif
    
    #ifdef USE_LOCKING
    lseek(fd, 0, SEEK_SET);
    rc = _locking(fd, _LK_UNLCK, 1);
    lseek(fd, 0, SEEK_END);
    #endif
    
    #ifdef NETWARE
	if (locking_sem) SignalLocalSemaphore(locking_sem);
	rc = 1;
    #endif

    if (rc < 0) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "mod_security: Unlocking failed");
        exit(1);
    }
}

module MODULE_VAR_EXPORT security_module = {

    STANDARD_MODULE_STUFF,
    sec_init,                   /* module initializer                  */
    sec_create_dir_config,      /* create per-dir    config structures */
    sec_merge_dir_config,       /* merge  per-dir    config structures */
    sec_create_srv_config,      /* create per-server config structures */
    NULL,                       /* merge  per-server config structures */
    sec_cmds,                   /* table of config file commands       */
    NULL,                       /* [#8] MIME-typed-dispatched handlers */
    NULL,                       /* [#1] URI to filename translation    */
    NULL,                       /* [#4] validate user id from request  */
    NULL,                       /* [#5] check if the user is ok _here_ */
    NULL,                       /* [#3] check access by host address   */
    NULL,                       /* [#6] determine MIME type            */
    sec_check_access,           /* [#7] pre-run fixups                 */
    sec_logger,                 /* [#9] log a transaction              */
    NULL,                       /* [#2] header parser                  */
    sec_child_init,             /* child_init                          */
    NULL,                       /* child_exit                          */
    NULL                        /* [#0] post read-request              */
};
