/*	$Id: mod_ldapvhost.c,v 1.10 2006/10/22 08:43:23 mbalmer Exp $	*/

/*
 * Copyright (c) 2004-2006 Marc Balmer <mbalmer@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
 
#include <err.h>
#include <ldap.h>

#include "httpd.h"
#include "http_config.h"
#include "http_request.h"
#include "http_core.h"
#include "http_config.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "util_script.h"
#include "http_conf_globals.h"

#define MODULE_VERSION "1.0.2"

MODULE_VAR_EXPORT module ldapvhost_module;

extern const char *ap_init_virtual_host(pool *, const char *, server_rec *,
    server_rec **);
extern const char *ap_handle_command(cmd_parms *, void *, const char *);

struct config {
	pool	*master_pool;
	char	*logfile;
	FILE	*logfp;
	char	*binddn;
	char	*ldaphost;
	char	*ldapport;
	int	 use_tls;
	char	*bindpasswd;
	char	*searchbase;
	LDAP	*ld;
};

struct server_config {
	char*	document_root;
	char	*server_name;
	char	*server_administrator;
	char	*access_log;
	char	*error_log;
};

struct ldap2apache {
	char	*ldap_attr;
	char	*apache_cmd;
};

static const struct ldap2apache entry_table[] = {
	{ "vhostServerAdmin",	"ServerAdmin" },
	{ "vhostDocumentRoot",	"DocumentRoot" },
	{ "vhostErrorLog",	"ErrorLog" },
	{ "vhostCustomLog",	"CustomLog" },
	{ "vhostAccessLog",	"AccessLog" },
	{ "vhostAlias",		"Alias" },
	{ "vhostServerAlias",	"ServerAlias" },
	{ "vhostRewriteEngine",	"RewriteEngine" },
	{ "vhostRewriteRule",	"RewriteRule" },
	{ "vhostCommand",	NULL },
	{ NULL, NULL }
};

static void
init_module(server_rec *s, pool *p)
{
    ap_add_version_component("mod_ldapvhost/" MODULE_VERSION);
}

static const char *
cmd_ldapvhosts(cmd_parms *cmd, void *dconf, char *str)
{
	struct config *cfg;
	server_rec *main_server = cmd->server, *s;
	pool *p = cmd->pool;
	char command[1024]; 
	const char *errmsg = NULL;
	char *attrs[] = { "vhost",
			  "vhostDocumentRoot",
			  "vhostServerAdmin",
			  "vhostAccessLog",
			  "vhostCustomLog",
			  "vhostErrorLog",
			  "vhostAlias",
			  "vhostServerAlias",
			  "vhostCommand",
			  NULL 
			};
	LDAPMessage *result, *e;
	char **vals;
	int i, n;
	int version;
	const struct ldap2apache *entry;

	cfg = (struct config *)ap_get_module_config(cmd->server->module_config,
	    &ldapvhost_module);

	if ((cfg->ld = ldap_init(cfg->ldaphost, atoi(cfg->ldapport))) == NULL) {
		warnx("Failed to initialize LDAP for server %s, %m",
		    cfg->ldaphost);
		return NULL;
	}

	version = LDAP_VERSION3;
	if (ldap_set_option(cfg->ld, LDAP_OPT_PROTOCOL_VERSION, &version)
	    != LDAP_OPT_SUCCESS) {
		warnx("Failed to set LDAP version 3 protocol");
		return NULL;
	}

	if (cfg->use_tls && ldap_start_tls_s(cfg->ld, NULL, NULL)
	    != LDAP_SUCCESS) {
		warnx("Failed to use TLS");
		return NULL;
	}

	if (ldap_simple_bind_s(cfg->ld, cfg->binddn, cfg->bindpasswd)
	    != LDAP_SUCCESS) {
		warnx("Failed to bind to directory server as %s, %s",
		    cfg->binddn, strerror(errno));
		/* Add an entry to the syslog or LDAP error log file */
		return NULL;
	}

	if (ldap_search_s(cfg->ld, cfg->searchbase, LDAP_SCOPE_SUBTREE, str,
	    attrs, 0, &result) != LDAP_SUCCESS) {
		ldap_perror(cfg->ld, "ldap_search_s");
		goto finish;
	}

	for (e = ldap_first_entry(cfg->ld, result), i = 1; e != NULL;
	    e = ldap_next_entry(cfg->ld, e), i++) {
		if ((vals = ldap_get_values(cfg->ld, e, "vhost")) == NULL)
			continue;

    		errmsg = ap_init_virtual_host(p, vals[0], main_server, &s);

    		if (errmsg)
			return errmsg;   

    		s->next = main_server->next;
    		main_server->next = s;
    		cmd->server = s;

		s->defn_name = "LDAP vhost entry";
    		s->defn_line_number = i;

		snprintf(command, sizeof(command), "ServerName %s", vals[0]);
		ap_handle_command(cmd, s->lookup_defaults, command);

 		ldap_value_free(vals);

		for (entry = &entry_table[0]; entry->ldap_attr != NULL;
		    entry++) {
			if ((vals = ldap_get_values(cfg->ld, e,
			    entry->ldap_attr)) == NULL)
				continue;

			for (n = 0; vals[n] != NULL; n++) {
				if (entry->apache_cmd != NULL)
					snprintf(command, sizeof(command),
					    "%s %s", entry->apache_cmd,
					    vals[n]);
				else
					snprintf(command, sizeof(command), "%s",
					    vals[n]);
				ap_handle_command(cmd, s->lookup_defaults,
				    command);
			}
			ldap_value_free(vals);
		}
 		cmd->server = main_server;
	}
finish:
	ldap_unbind(cfg->ld);

	return NULL;
}

static const char *
cmd_ldaperrorlog(cmd_parms *cmd, void *dconf, char *str)
{
	struct config *c;
	FILE *fp;

	c = (struct config *)ap_get_module_config(cmd->server->module_config,
	    &ldapvhost_module);
	c->logfile = ap_pstrdup(c->master_pool, str);
	if ((fp = fopen(str, "a")) != NULL) {
		fprintf(fp, "RESTART\n");
		fclose(fp);
	}
	return NULL;
}

static const char *
cmd_ldapbinddn(cmd_parms *cmd, void *dconf, char *str)
{
	struct config *c;

	c = (struct config *)ap_get_module_config(cmd->server->module_config,
	    &ldapvhost_module);
	c->binddn = ap_pstrdup(c->master_pool, str);
	return NULL;
}

static const char *
cmd_ldaphost(cmd_parms *cmd, void *dconf, char *str)
{
	struct config *c;

	c = (struct config *)ap_get_module_config(cmd->server->module_config,
	    &ldapvhost_module);
	c->ldaphost = ap_pstrdup(c->master_pool, str);
	return NULL;
}

static const char *
cmd_ldapport(cmd_parms *cmd, void *dconf, char *str)
{
	struct config *c;

	c = (struct config *)ap_get_module_config(cmd->server->module_config,
	    &ldapvhost_module);
	c->ldapport = ap_pstrdup(c->master_pool, str);
	return NULL;
}

static const char *
cmd_use_tls(cmd_parms *cmd, void *dconf, char *str)
{
	struct config *c;

	c = (struct config *)ap_get_module_config(cmd->server->module_config,
	    &ldapvhost_module);
	c->use_tls = strcmp(str, "True") == 0 ? 1 : 0;
	return NULL;
}

static const char *
cmd_ldapbindpasswd(cmd_parms *cmd, void *dconf, char *str)
{
	struct config *c;
	
	c = (struct config *)ap_get_module_config(cmd->server->module_config,
	    &ldapvhost_module);
	c->bindpasswd = ap_pstrdup(c->master_pool, str);
	return NULL;
}

static const char *
cmd_ldapsearchbase(cmd_parms *cmd, void *dconf, char *str)
{
	struct config *c;

	c = (struct config *)ap_get_module_config(cmd->server->module_config,
	    &ldapvhost_module);
	c->searchbase = ap_pstrdup(c->master_pool, str);
	return NULL;
}

void*
server_config(pool *p, server_rec *s)
{
	struct config *c;

	c = (struct config *)ap_pcalloc(p, sizeof(struct config));
	if (c != NULL)
		c->master_pool = p;
	return((void *)c);
}


static const command_rec command_table[] = {
	{
		"LDAPVirtualHosts", cmd_ldapvhosts, NULL, RSRC_CONF, TAKE1,
		"search string for virtual hosts, e.g. (objectClass=vhost)"
	},
	{
		"LDAPErrorLog", cmd_ldaperrorlog, NULL, RSRC_CONF, TAKE1,
		"Location of the LDAP error log file"
	},
	{
		"LDAPBindDN", cmd_ldapbinddn, NULL, RSRC_CONF, TAKE1,
		"DN to use for binding to the LDAP server"
	},
	{
		"LDAPHost", cmd_ldaphost, NULL, RSRC_CONF, TAKE1,
		"Name of the LDAP host"
	},
	{
		"LDAPPort", cmd_ldapport, NULL, RSRC_CONF, TAKE1,
		"Port number of the LDAP server"
	},
	{
		"LDAPUseTLS", cmd_use_tls, NULL, RSRC_CONF, TAKE1,
		"Use TLS to connect to LDAP server"
	},
	{
		"LDAPBindPasswd", cmd_ldapbindpasswd, NULL, RSRC_CONF, TAKE1,
		"Password to use when binding to the LDAP server"
	},
	{
		"LDAPSearchbase", cmd_ldapsearchbase, NULL, RSRC_CONF, TAKE1,
		"Searchbase for vhost lookups"
	},
	{ NULL }
};

module MODULE_VAR_EXPORT ldapvhost_module =
{
	STANDARD_MODULE_STUFF,
	init_module,		/* module initializer */
	NULL,			/* per-directory config creator */
	NULL,			/* dir config merger */
	server_config,		/* server config creator */
	NULL,			/* server config merger */
	command_table,		/* command table */
	NULL,			/* list of handlers */
	NULL,			/* URI-to-filename translation */
	NULL,			/* check/validate user_id */
	NULL,			/* check auth */
	NULL,			/* check access by host address */
	NULL,			/* type_checker */
	NULL,			/* fixups */
	NULL,			/* logger */
	NULL,			/* header parser */
	NULL,			/* child_init */
	NULL,			/* child_exit */
	NULL			/* post read-request */
};
