/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2008 MaxMind Inc  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        MaxMind (http://www.maxmind.com/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "MaxMind" and "GeoIP" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact support@maxmind.com.
 *
 * 5. Products derived from this software may not be called "GeoIP",
 *    nor may "MaxMind" appear in their name, without prior written
 *    permission of the MaxMind.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */

/*
 * Module definition information - the part between the -START and -END
 * lines below is used by Configure. This could be stored in a separate
 * instead.
 *
 * MODULE-DEFINITION-START
 * Name: geoip_module
 * ConfigStart
     GEOIP_LIB="-L/usr/local/lib -lGeoIP"
     if [ "X$GEOIP_LIB" != "X" ]; then
         LIBS="$LIBS $GEOIP_LIB"
         echo " + using $GEOIP_LIB for GeoIP support"
     fi
 * ConfigEnd
 * MODULE-DEFINITION-END
 */

/* geoip module
 *
 * Version 1.3.4
 *
 * This module sets an environment variable to the remote country
 * based on the requestor's IP address.  It uses the GeoIP library
 * to lookup the country by IP address.
 *
 * Copyright 2003 MaxMind LLC
 * June 26th, 2002

To use the module you have to compile it into the frontend part of
your server, I usually copy the module to apache-1.3/src/modules/extra/
and use APACI like:

  ./configure --prefix=/usr/local/apache \
     --activate-module=src/modules/extra/mod_geoip.c \
     [... more apaci options ...]    

You should also be able to compile and use this module as a
dynamically loaded module (DSO).
 
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "util_script.h"
#include "GeoIP.h"
#include "GeoIPCity.h"

typedef struct {
  int GeoIPEnable;
} geoip_dir_config_rec;


typedef struct {
  GeoIP **gips;
  int numGeoIPFiles;
  char **GeoIPFilenames;
  int GeoIPEnable;
  int GeoIPEnableUTF8;
  char GeoIPOutput;
  int GeoIPFlags;
  int *GeoIPFlags2;
  int scanProxyHeaders;
} geoip_server_cfg;

module MODULE_VAR_EXPORT geoip_module;

module MODULE_VAR_EXPORT proxy_add_uri_module;

static const int GEOIP_NONE    = 0;
static const int GEOIP_DEFAULT = 1;
static const int GEOIP_NOTES   = 2;
static const int GEOIP_ENV     = 4;
static const int GEOIP_ALL     = 6;
static const int GEOIP_INIT    = 7;

static const int GEOIP_UNKNOWN = -1;

char metrocodestr[100];
char areacodestr[100];
char latstr[100];
char lonstr[100];

/* create a disabled directory entry */
static void *geoip_create_dir_config(pool *p, char *d)
{
  
  geoip_dir_config_rec *dcfg;

  dcfg = (geoip_dir_config_rec *) ap_pcalloc(p, sizeof(geoip_dir_config_rec));
  dcfg->GeoIPEnable = 0;

  return dcfg;
}


static void    *
geoip_server_config(pool * p, server_rec * s)
{
        geoip_server_cfg *cfg;

	cfg = ap_pcalloc(p, sizeof(geoip_server_cfg));
	if (!cfg)
		return NULL;

	cfg->gips = NULL;
	cfg->numGeoIPFiles = 0;
	cfg->GeoIPFilenames = NULL;
	cfg->GeoIPEnable = 0;
	cfg->GeoIPEnableUTF8 = 0;
	cfg->GeoIPOutput = GEOIP_INIT;
	cfg->GeoIPFlags = GEOIP_STANDARD;
	cfg->GeoIPFlags2 = NULL;
	cfg->scanProxyHeaders = 0;
	return (void *) cfg;
}

static const char * 
geoip_enable(cmd_parms * cmd, void *dummy, int flag) 
{ 
  geoip_dir_config_rec *dcfg; 
  geoip_server_cfg *cfg; 
  server_rec     *s; 
  
  /* is per directory config? */ 
  if (cmd->path) { 
    dcfg = dummy; 
    dcfg->GeoIPEnable = flag; 
    return NULL; 
  } 

  /* no? then it is server config */ 
  s = cmd->server; 
  cfg = (geoip_server_cfg *) ap_get_module_config(s->module_config, &geoip_module); 
  cfg->GeoIPEnable = flag; 
  return NULL; 
}

static const char *geoip_enable_utf8(cmd_parms *cmd, void *dummy, int flag) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  cfg->GeoIPEnableUTF8 = flag;
  return NULL;
}

static const char *geoip_scanproxy(cmd_parms *cmd, void *dummy, int flag) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  cfg->scanProxyHeaders = flag;
  return NULL;
}

/* TODO we will have to change this to ITERATE2 to support multiple flags per database file */
static const char *geoip_set_filename(cmd_parms *cmd, void *dummy, char *GeoIPFilename, char *arg2) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  int i = cfg->numGeoIPFiles;
  cfg->numGeoIPFiles++;
  cfg->GeoIPFilenames = realloc(cfg->GeoIPFilenames, cfg->numGeoIPFiles * sizeof(char *));
  cfg->GeoIPFilenames[i] = GeoIPFilename;
  cfg->GeoIPFlags2 = realloc(cfg->GeoIPFlags2, cfg->numGeoIPFiles * sizeof(int));
  if (arg2 == NULL) {
    cfg->GeoIPFlags2[i] = GEOIP_UNKNOWN;
  } else if (!strcmp(arg2, "Standard")) {
    cfg->GeoIPFlags2[i] = GEOIP_STANDARD;
  } else if (!strcmp(arg2, "MemoryCache")) {
    cfg->GeoIPFlags2[i] = GEOIP_MEMORY_CACHE;
  } else if (!strcmp(arg2, "CheckCache")) {
    cfg->GeoIPFlags2[i] = GEOIP_CHECK_CACHE;
  } else if (!strcmp(arg2, "IndexCache")) {
    cfg->GeoIPFlags2[i] = GEOIP_INDEX_CACHE;
  } else if (!strcmp(arg2, "MMapCache")) {
    cfg->GeoIPFlags2[i] = GEOIP_MMAP_CACHE;
  }
  return NULL;
}

static const char *geoip_set_output(cmd_parms *cmd, void *dummy, const char *arg) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  if (cfg->GeoIPOutput & GEOIP_DEFAULT) {
    /* was set to default, clear so can be reset with user specified values */
    cfg->GeoIPOutput = GEOIP_NONE;
  }
  if (!strcmp(arg, "Notes")) {
    cfg->GeoIPOutput |= GEOIP_NOTES;
  } else if (!strcmp(arg, "Env")) {
    cfg->GeoIPOutput |= GEOIP_ENV;
  } else if (!strcmp(arg, "All")) {
    cfg->GeoIPOutput |= GEOIP_ALL;
  } else {
    ap_log_error(APLOG_MARK, APLOG_ERR, s, "[mod_geoip]: Invalid Value for GeoIPOutput: %s", arg);
  }
  return NULL;
}

/* TODO we will have to change this to ITERATE2 to support multiple flags per database file */
static const char *geoip_set_flags(cmd_parms *cmd, void *dummy, const char *arg) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  if (!strcmp(arg, "MemoryCache")) {
    cfg->GeoIPFlags &= GEOIP_MEMORY_CACHE;
  } else if (!strcmp(arg, "CheckCache")) {
    cfg->GeoIPFlags &= GEOIP_CHECK_CACHE;
  }
  return NULL;
}

static command_rec geoip_cmds[] = {
  {"GeoIPDBFile", geoip_set_filename, NULL, RSRC_CONF, TAKE12, "GeoIP Data Files"},
  {"GeoIPEnable", geoip_enable, NULL, RSRC_CONF | OR_FILEINFO, FLAG, "Enable mod_geoip"},
  {"GeoIPEnableUTF8", geoip_enable_utf8, NULL, RSRC_CONF, FLAG, "Enable utf8 charset instead of iso-8859-1"},
  {"GeoIPOutput", geoip_set_output, NULL, RSRC_CONF, ITERATE, "Specify output method(s)"},
  {"GeoIPFlags", geoip_set_flags, NULL, RSRC_CONF, ITERATE, "GeoIP flags"},
  {"GeoIPScanProxyHeaders", geoip_scanproxy, NULL, RSRC_CONF, FLAG, "Get IP from HTTP_CLIENT IP or X-Forwarded-For"},
  {NULL}
};

static int geoip_header_parser(request_rec *r);

static int 
geoip_post_read_request(request_rec * r)
{
  geoip_server_cfg *cfg = ap_get_module_config(r->server->module_config, &geoip_module);
  if (!cfg)
    return DECLINED;

  if (!cfg->GeoIPEnable)
    return DECLINED;

  return geoip_header_parser(r);
}

static int
geoip_per_dir(request_rec * r)
{

  geoip_dir_config_rec *dcfg;

  geoip_server_cfg *cfg =
  ap_get_module_config(r->server->module_config, &geoip_module);
  if (cfg && cfg->GeoIPEnable)
    return DECLINED;

  dcfg = ap_get_module_config(r->per_dir_config, &geoip_module);
  if (!dcfg)
    return DECLINED;

  if (!dcfg->GeoIPEnable)
    return DECLINED;

  return geoip_header_parser(r);
}

/**/
static void 
geoip_server_init(server_rec * s, pool * p)
{
  geoip_server_cfg *cfg;
  int             i;

  /* scan all servers */
  for ( ; s ; s = s->next ){

    cfg = (geoip_server_cfg *)
      ap_get_module_config(s->module_config, &geoip_module);

    if ( !cfg )
      continue;
    if ( !cfg->GeoIPEnable )
      continue;
    if (cfg->gips)
      continue;

    if (cfg->GeoIPFilenames != NULL) {
      cfg->gips = malloc(sizeof(GeoIP *) * cfg->numGeoIPFiles);
      for (i = 0; i < cfg->numGeoIPFiles; i++) {
	/*
	 * Use flags associated with filename, or if not available, use
	 * global flags
	 */
	cfg->gips[i] = GeoIP_open(cfg->GeoIPFilenames[i],
				  (cfg->GeoIPFlags2[i] == GEOIP_UNKNOWN) ? cfg->GeoIPFlags : cfg->GeoIPFlags2[i]);
	if (cfg->gips[i]) {
	  if (cfg->GeoIPEnableUTF8) {
	    GeoIP_set_charset(cfg->gips[i], GEOIP_CHARSET_UTF8);
	  }
	}
	else {
	  ap_log_error(APLOG_MARK, APLOG_ERR, s, "[mod_geoip]: Error while opening data file %s", cfg->GeoIPFilenames[i]);
          continue;
	}
      }
    }
    else {
      cfg->gips = malloc(sizeof(GeoIP *));
      cfg->gips[0] = GeoIP_new(cfg->GeoIPFlags);
      if (!cfg->gips[0]) {
	ap_log_error(APLOG_MARK, APLOG_ERR, s, "[mod_geoip]: Error while opening data file");
      }
      cfg->numGeoIPFiles = 1;
    } /* if */
  } /* for */
}

static int geoip_header_parser(request_rec *r) {
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(r->server->module_config, &geoip_module);
  char *ipaddr;
  char *orgorisp;
  short int country_id;
  GeoIPRecord * gir;
  GeoIPRegion * giregion;
  const char *continent_code, *country_code, *country_name, *region_name;
  const char *netspeedstring;
  unsigned char databaseType;
  int i;
  int netspeed = 0;

  /* For splitting proxy headers */
  char *ipaddr_ptr = 0;
  char *comma_ptr;

  if(!cfg->scanProxyHeaders)
    ipaddr = r->connection->remote_ip;
  else {
    ap_add_common_vars(r);
    if ( ap_table_get(r->subprocess_env, "HTTP_CLIENT_IP") )
      ipaddr_ptr = (char *) ap_table_get(r->subprocess_env, "HTTP_CLIENT_IP");
    else if ( ap_table_get(r->subprocess_env, "HTTP_X_FORWARDED_FOR") )
      ipaddr_ptr =
             (char *) ap_table_get(r->subprocess_env, "HTTP_X_FORWARDED_FOR");
    else if ( ap_table_get(r->headers_in, "X-Forwarded-For") )
      ipaddr_ptr = (char *) ap_table_get(r->headers_in, "X-Forwarded-For");
    else if ( ap_table_get(r->subprocess_env, "HTTP_REMOTE_ADDR") ) {
      ipaddr_ptr = (char *) ap_table_get(r->subprocess_env, "HTTP_REMOTE_ADDR");
    }

    if (!ipaddr_ptr) {
      ap_log_error(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r->server, "[mod_geoip]: Error while getting ipaddr from proxy headers. Using REMOTE_ADDR.");
      ipaddr = r->connection->remote_ip;
    } else {
      ap_log_error(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r->server, "[mod_geoip]: IPADDR_PTR: %s", ipaddr_ptr);
      /* Check to ensure that the HTTP_CLIENT_IP or X-Forwarded-For header is
       * not a comma separated list of addresses, which would
       * cause mod_geoip to return no country code. If the
       * header is a comma separated list, return the first
       * IP address in the list, which is (hopefully!) the
       * real client IP. */
      ipaddr = (char *) calloc(16, sizeof(char));
      strncpy(ipaddr, ipaddr_ptr, 15);
      comma_ptr = strchr(ipaddr, ',');
      if (comma_ptr != 0)
        *comma_ptr = '\0';
    }
  }
  if(!ipaddr)
    return DECLINED;

  ap_log_error(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r->server, "[mod_geoip]: IPADDR: %s", ipaddr);

  if (cfg->GeoIPOutput & GEOIP_NOTES) {
     ap_table_set(r->notes, "GEOIP_ADDR", ipaddr);
   }
   if (cfg->GeoIPOutput & GEOIP_ENV) {
     ap_table_set(r->subprocess_env, "GEOIP_ADDR", ipaddr);
   }

  for (i = 0; i < cfg->numGeoIPFiles; i++) {
          
    /* skip database handles that can not be opned for some reason */  
    if ( cfg->gips[i] == NULL )
            continue;
             
    databaseType = cfg->gips[ i ] ? GeoIP_database_edition( cfg->gips[ i ] ) : -1;  /* -1 is "magic value" */
    switch (databaseType) {
    case GEOIP_NETSPEED_EDITION:
      netspeed = GeoIP_id_by_addr (cfg->gips[i], ipaddr);	
      if (netspeed == GEOIP_UNKNOWN_SPEED) {
        netspeedstring = "unknown";
      }
      else if (netspeed == GEOIP_DIALUP_SPEED) {
        netspeedstring = "dialup";
      }
      else if (netspeed == GEOIP_CABLEDSL_SPEED) {
        netspeedstring = "cabledsl";
      }
      else if (netspeed == GEOIP_CORPORATE_SPEED) {
        netspeedstring = "corporate";
      }
      if (cfg->GeoIPOutput & GEOIP_NOTES){
        ap_table_set(r->notes,"GEOIP_NETSPEED",netspeedstring);
      }
      if (cfg->GeoIPOutput & GEOIP_ENV){
        ap_table_set(r->subprocess_env,"GEOIP_NETSPEED",netspeedstring);
      }
    break;
    case GEOIP_COUNTRY_EDITION:
      country_id = GeoIP_country_id_by_addr(cfg->gips[i], ipaddr);

      if (country_id > 0) {
	continent_code = GeoIP_country_continent[country_id];
	country_code = GeoIP_country_code[country_id];
	country_name = GeoIP_country_name[country_id];
	if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_CONTINENT_CODE", continent_code);
	  ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", country_code);
	  ap_table_set(r->notes, "GEOIP_COUNTRY_NAME", country_name);
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_CONTINENT_CODE", continent_code);
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", country_code);
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_NAME", country_name);
	}
      }
      break;
    case GEOIP_REGION_EDITION_REV0:
    case GEOIP_REGION_EDITION_REV1:
      giregion = GeoIP_region_by_addr (cfg->gips[i], ipaddr);
      if (giregion != NULL) {
	if ( giregion->country_code[0] ) {
	  region_name = GeoIP_region_name_by_code(giregion->country_code, giregion->region);
	}
			 
	if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  if ( giregion->country_code[0] ){
		    ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", giregion->country_code);
	  }
	  if (giregion->region[0]) {
	    ap_table_set(r->notes, "GEOIP_REGION", giregion->region);
	    if ( region_name != NULL ){
	      ap_table_set(r->notes, "GEOIP_REGION_NAME", region_name);
	    }
        }
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  if ( giregion->country_code[0] ){
	      ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", giregion->country_code);
	  }
	  if (giregion->region[0]) {
	    ap_table_set(r->subprocess_env, "GEOIP_REGION", giregion->region);
	    if ( region_name != NULL ){
	      ap_table_set(r->subprocess_env, "GEOIP_REGION_NAME", region_name);
	    }
    }
	}
	GeoIPRegion_delete(giregion);
      }
      break;
    case GEOIP_CITY_EDITION_REV0:
    case GEOIP_CITY_EDITION_REV1:
      gir = GeoIP_record_by_addr(cfg->gips[i], ipaddr);
      if (gir != NULL) {
	if ( gir->country_code != NULL ) {
	  region_name = GeoIP_region_name_by_code(gir->country_code, gir->region);
	}

        sprintf(metrocodestr,"%d",gir->dma_code);
        sprintf(areacodestr,"%d",gir->area_code);
        sprintf(latstr,"%f",gir->latitude);
        sprintf(lonstr,"%f",gir->longitude);
        if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_CONTINENT_CODE", gir->continent_code);
	  ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", gir->country_code);
	  ap_table_set(r->notes, "GEOIP_COUNTRY_NAME", gir->country_name);
	  if (gir->region != NULL){
	    ap_table_set(r->notes, "GEOIP_REGION", gir->region);
	    if ( region_name != NULL ){
	      ap_table_set(r->notes, "GEOIP_REGION_NAME", region_name);
	    }
	  } 
	  if (gir->city != NULL)
	    ap_table_set(r->notes, "GEOIP_CITY", gir->city);
          
	  ap_table_set(r->notes,"GEOIP_DMA_CODE",metrocodestr);
          ap_table_set(r->notes,"GEOIP_METRO_CODE",metrocodestr);
          ap_table_set(r->notes,"GEOIP_AREA_CODE",areacodestr);
          ap_table_set(r->notes,"GEOIP_LATITUDE",latstr);
          ap_table_set(r->notes,"GEOIP_LONGITUDE",lonstr);
          ap_table_set(r->notes,"GEOIP_POSTAL_CODE",gir->postal_code);
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_CONTINENT_CODE", gir->continent_code);
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", gir->country_code);
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_NAME", gir->country_name);
	  if (gir->region != NULL) {
	    ap_table_set(r->subprocess_env, "GEOIP_REGION", gir->region);
	    if ( region_name != NULL ){
	      ap_table_set(r->subprocess_env, "GEOIP_REGION_NAME", region_name);
	    }
	  }    	 
	  if (gir->city != NULL)
	    ap_table_set(r->subprocess_env, "GEOIP_CITY", gir->city);
          
	  ap_table_set(r->subprocess_env,"GEOIP_DMA_CODE",metrocodestr);
          ap_table_set(r->subprocess_env,"GEOIP_METRO_CODE",metrocodestr);
          ap_table_set(r->subprocess_env,"GEOIP_AREA_CODE",areacodestr);
          ap_table_set(r->subprocess_env,"GEOIP_LATITUDE",latstr);
          ap_table_set(r->subprocess_env,"GEOIP_LONGITUDE",lonstr);
          ap_table_set(r->subprocess_env,"GEOIP_POSTAL_CODE",gir->postal_code);
	}
	GeoIPRecord_delete(gir);
      }
      break;
    case GEOIP_ORG_EDITION:
      orgorisp = GeoIP_name_by_addr(cfg->gips[i], ipaddr);
      if (orgorisp != NULL) {
	if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_ORGANIZATION", orgorisp);
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_ORGANIZATION", orgorisp);
	}
      }
      break;
    case GEOIP_ISP_EDITION:
      orgorisp = GeoIP_name_by_addr(cfg->gips[i], ipaddr);
      if (orgorisp != NULL) {
	if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_ISP", orgorisp);
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_ISP", orgorisp);
	}
      }
      break;
    }
  }

  /* Cleanup splitting of Proxy Headers */
  if(ipaddr_ptr)
    free(ipaddr);

  return OK;
}

static void geoip_child_exit(server_rec *r, pool *p) {
  int i;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(r->module_config, &geoip_module);
  if(cfg->gips != NULL) {
    for (i = 0; i < cfg->numGeoIPFiles; i++)
      if(cfg->gips[i] != NULL)
	GeoIP_delete(cfg->gips[i]);
    free(cfg->gips);
    cfg->gips = NULL;
  }
  if(cfg->GeoIPFilenames != NULL) {
    free(cfg->GeoIPFilenames);
    cfg->GeoIPFilenames = NULL;
  }
  if(cfg->GeoIPFlags2 != NULL) {
    free(cfg->GeoIPFlags2);
    cfg->GeoIPFlags2 = NULL;
  }
}

/* map into the first apache */
static void 
geoip_post_config(server_rec * s, pool *p)
{
	geoip_server_init(s, p);
}


/**/
static void
geoip_child_init(server_rec * s, pool * p)
{
	geoip_server_cfg *cfg;
	int             i, flags;
	
	
	cfg = (geoip_server_cfg *)
    ap_get_module_config(s->module_config, &geoip_module);
	
	
	if (cfg->gips) {
		if (cfg->GeoIPFilenames != NULL) {
			for (i = 0; i < cfg->numGeoIPFiles; i++) {
				/*
				 * Use flags associated with filename, or if not available, use
				 * global flags
				 */
				flags = (cfg->GeoIPFlags2[i] == GEOIP_UNKNOWN) ? cfg->GeoIPFlags : cfg->GeoIPFlags2[i];
				if (flags & (GEOIP_MEMORY_CACHE | GEOIP_MMAP_CACHE))
					continue;

				/* reopen anything other than memory based files */
				if ( cfg->gips[i] )
					GeoIP_delete(cfg->gips[i]);
				cfg->gips[i] = GeoIP_open(cfg->GeoIPFilenames[i], flags);
				if (cfg->gips[i]) {
					if (cfg->GeoIPEnableUTF8) {
						GeoIP_set_charset(cfg->gips[i], GEOIP_CHARSET_UTF8);
					}
				}
				else {
					ap_log_error(APLOG_MARK, APLOG_ERR, s, "[mod_geoip]: Error while opening data file %s", cfg->GeoIPFilenames[i]);
					continue;
				}
			}
		}
		else {
			if (!(cfg->GeoIPFlags & (GEOIP_MEMORY_CACHE | GEOIP_MMAP_CACHE))) {
				/* reopen anything other than memory based files */
                        if ( cfg->gips[0] )
			        GeoIP_delete(cfg->gips[0]);
		        cfg->gips[0] = GeoIP_new(cfg->GeoIPFlags);
		        }
		        if (!cfg->gips[0]) {
				ap_log_error(APLOG_MARK, APLOG_ERR, s, "[mod_geoip]: Error while opening data file");
			}
			cfg->numGeoIPFiles = 1;
		}
	}
}


module MODULE_VAR_EXPORT geoip_module = {
    STANDARD_MODULE_STUFF,
    geoip_post_config,          /* initializer */
    geoip_create_dir_config,    /* dir config creater */
    NULL,                       /* dir merger --- default is to override */
    geoip_server_config,        /* server config */
    NULL,                       /* merge server configs */
    geoip_cmds,                 /* command table */
    NULL,                       /* handlers */
    NULL,                       /* filename translation */
    NULL,                       /* check_user_id */
    NULL,                       /* check auth */
    NULL,                       /* check access */
    NULL,                       /* type_checker */
    NULL,                       /* fixups */
    NULL,                       /* logger */
    geoip_per_dir,              /* header parser */
    geoip_child_init,           /* child_init */
    geoip_child_exit,           /* child_exit */
    geoip_post_read_request,    /* post read-request */
};
