/*
 *  Copyright (C) 2006  MakeHuman Project
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *  
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 *  File   : Image.cpp
 *  Project: MakeHuman <info@makehuman.org>, http://www.makehuman.org/
 *  Library: MHGUI
 *
 *  For individual developers look into the AUTHORS file.
 *   
 */

#include "../include/mhgui/Image.h"
#include "../include/mhgui/Tooltip.h"
#include "../include/mhgui/CGUtilities.h"
#include "../include/mhgui/ImageSysListener.h"
#include "../include/mhgui/ImageData.h"

#ifdef __APPLE__
    #include <GLUT/glut.h>
#else
  #ifdef USE_FREEGLUT     
    #include <GL/freeglut.h>
  #else
    #include <GL/glut.h>
  #endif
#endif

#include <iostream>

using std::cerr;
using std::endl;

namespace mhgui {

//constructor
Image::Image (uint32_t      inId,
              const string& inFilename,
              const Rect&   inGeometry)
    : Widget(inId, inGeometry),
    imageFilename(inFilename),

    texture(),

    textureIsInited(false),

    imageSysListener(new ImageSysListener()),
    alpha        (1.0),
    overlay (0,0,0,0),
    overlayEffect (false)
{

   setSysListener(imageSysListener);
}

Image::~Image()
{
  delete imageSysListener; // remove the listener again
}

void Image::setAlpha(float a)
{
  alpha = a;
}

void Image::show ()
{
  setVisible(true);

  lazyLoadTexture();
}

void Image::hide ()
{
  setVisible(false);
}

//Return the ID assigned
const Texture& Image::getTextures ()
{
  lazyLoadTexture();
  return texture;
}

void Image::setOverlayRectangle (const Color& c)
{
  overlay = c;
  overlayEffect = true;
}

void Image::setOverlayRectangle (bool overlayEffect)
{
  this->overlayEffect = overlayEffect;
}

void Image::drawOverlay()
{
  if (isVisible())
  {
    if (overlayEffect)
    {
      cgutils::enableBlend ();
      cgutils::drawSquareFill (getAbsoluteRect(), overlay);
      cgutils::disableBlend ();
    }
  }
}

//draw function
void Image::draw()
{
  if (isVisible())
  {  
    if (lazyLoadTexture())
    {
      cgutils::enableBlend ();
      cgutils::drawSquareFillTexture (getAbsoluteRect(), alpha, texture);
      cgutils::disableBlend ();
    }
    else
    {
      cgutils::enableBlend ();
      cgutils::drawSquareFill (getAbsoluteRect(), Color (1.0,1.0,1.0,alpha));
      cgutils::disableBlend ();
    }
  }
}


/* ==========================================================================
 * Private Methods
 * ========================================================================== */
/* ========================================================================== */
/** Load a image file lazy whose filename has been initialized in the
 *  constructor.
 *
 * Note: This call justtries to load a image once and remembers if it could be 
 * gathered successfully or not. Succeed calls just return if the image is 
 * available or not.
 *
 * @return true if the given Image file could be loaded, false otherwise.
 */
/* ========================================================================== */
bool Image::lazyLoadTexture()
{
  if (textureIsInited) // already loaded?
    return true;

  textureIsInited = true;

  if (imageFilename.empty())
    return false;

  //cout << "load: " << filename << endl;

  // read the PNG file using pngLoad
  // raw data from PNG file is in image buffer
  if (texture.load (imageFilename) == false)
  {
    cerr << "(pngLoad) " << imageFilename << " FAILED" << endl;;
    return false;
  }

  return true;
}

} // namespace mhgui
