/*           Copyright (C) 1999, 2000, 2001, 2002, 2003 Stijn van Dongen
 *
 * This file is part of MCL.  You can redistribute and/or modify MCL under the
 * terms of the GNU General Public License; either version 2 of the License or
 * (at your option) any later version.  You should have received a copy of the
 * GPL along with MCL, in the file COPYING.
*/

/* TODO
 *    parsing code is ugly. Some header parsing is now line-based,
 *    some other part is not.
*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include "io.h"
#include "iface.h"

#include "util/compile.h"
#include "util/types.h"
#include "util/err.h"
#include "util/minmax.h"
#include "util/alloc.h"
#include "util/ting.h"
#include "util/io.h"
#include "util/hash.h"
#include "util/array.h"

static const char *mclxar = "mclxReadAscii";

static mcxstatus mclxParseDimPart
(  mcxIO          *xf
,  mcxHash        *header
)  ;

static mcxstatus mclParseDomain
(  mcxIO        *xf
,  mclVector**  dompp
)  ;

static void mclxWriteAsciiHeader
(  const mclMatrix* mx
,  FILE* fp
)  ;


/* reads a vector (ascii format)
 * ensures it is in ascending format and has no negative entries
 * or repeated entries.
*/

static mcxstatus mclReaDaVec
(  mcxIO*      xf
,  mclVector*  vec
,  mclpAR*     ar
,  int         finalToken
,  mcxbits     bits              /* inherited from mcl{x,v}ReadAsciiRaw */
,  void      (*ivpmerge)(void* ivp1, const void* ivp2)
,  double    (*fltbinary)(pval val1, pval val2)
)  ;

static mcxstatus  mclxrDimPart
(  mcxIO          *xf
,  int            *pn_rows
,  int            *pn_cols
)  ;


mcxstatus  mclxFilePeek
(  mcxIO          *xf
,  int            *pn_cols
,  int            *pn_rows
,  mcxOnFail      ON_FAIL
)
   {  long f_pos

   ;  if (!xf->fp && mcxIOopen(xf, ON_FAIL) != STATUS_OK)
      return STATUS_FAIL

   ;  f_pos = ftell(xf->fp)

   ;  if
      (  xf->fp != stdin
      && mcxIOexpectCanary(xf->fp, mclxCanary)
      )
      {  fread(pn_cols, sizeof(long), 1, xf->fp)
      ;  fread(pn_rows, sizeof(long), 1, xf->fp)
   ;  }
      else if (mclxrDimPart(xf, pn_rows, pn_cols) != STATUS_OK)
      {  mcxErr("mclxFilePeek", "could not parse header")
      ;  if (ON_FAIL == RETURN_ON_FAIL)
         return STATUS_FAIL
      ;  else
         mcxExit(1)
   ;  }

      fseek(xf->fp, f_pos, SEEK_SET)
   ;  mcxIOrewind(xf)             /* fixme: temporary solution (? ?) */

   ;  return STATUS_OK
;  }


mclMatrix* mclxMaskedRead
(  mcxIO* xf
,  const mclVector* selector
,  mcxOnFail ON_FAIL
)
   {  mclMatrix* mx     =  NULL
   ;  int n_rows =  0
   ;  int n_cols =  0

   ;  mclxFormatFound = 'b'

   ;  if (!xf->fp && mcxIOopen(xf, ON_FAIL) != STATUS_OK)
      {  mcxIOerr(xf, "mclxMaskedRead", "can not be opened")
      ;  return NULL
   ;  }

      if
      (  xf->fp != stdin
      && mcxIOexpectCanary(xf->fp, mclxCanary)
      )
      {  mclVector *vec
      ;  mclVector* dom_cols, *dom_rows
      ;  int n_ivps  =  selector ? selector->n_ivps : 0
      
      ;  fread(&n_cols, sizeof(long), 1, xf->fp)
      ;  fread(&n_rows, sizeof(long), 1, xf->fp)

      ;  dom_cols = mclvCanonical(NULL, n_cols, 1.0)
      ;  dom_rows = mclvCanonical(NULL, n_rows, 1.0)

      ;  mx  =  mclxAllocZero(dom_cols, dom_rows)
      ;  vec =  mx->cols

      ;  if (selector)
         {  long f_pos  =  ftell(xf->fp)
         ;  int  k      =  0
         ;  long  v_pos

         ;  while (k < n_ivps)
            {  int vec_idx  =  selector->ivps[k].idx

            ;  fseek(xf->fp, f_pos + vec_idx * sizeof(long), SEEK_SET)
            ;  v_pos = mcxIOreadInteger(xf->fp)
            ;  fseek(xf->fp, v_pos, SEEK_SET)
            ;  mclvEmbedRead(vec + vec_idx, xf, EXIT_ON_FAIL)
            ;  k++
         ;  }

            /*
             * Move to end of matrix body
             *
            */            
         ;  fseek(xf->fp, f_pos + n_cols * sizeof(long), SEEK_SET)
         ;  v_pos = mcxIOreadInteger(xf->fp)
         ;  fseek(xf->fp, v_pos, SEEK_SET)
      ;  }
         else
         {  fseek(xf->fp, (1 + n_cols) * sizeof(long), SEEK_CUR)
         ;  while (--n_cols >= 0)
            mclvEmbedRead(vec++, xf, EXIT_ON_FAIL)
         ;
         }
      
         if (mclVerbosityIoImpala)
         mcxTell
         (  "mclIO"
         ,  "read native binary %ldx%ld matrix from stream <%s>"
         ,  (long) N_ROWS(mx)
         ,  (long) N_COLS(mx)
         ,  xf->fn->str
         )
   ;  }
      else
      {  mx = mclxReadAscii(xf, ON_FAIL)
      ;  mclxFormatFound = 'a'
   ;  }
      return mx
;  }


mcxstatus mclxWriteBinary
(  const mclMatrix*  mx
,  mcxIO*            xfOut
,  mcxOnFail         ON_FAIL
)
   {  long       n_cols   =  N_COLS(mx)
   ;  long       n_rows   =  N_ROWS(mx)
   ;  mclVector* vec      =  mx->cols
   ;  mcxstatus  status   =  0
   ;  long       v_pos    =  0
   ;  FILE*      fout     =  xfOut->fp

   ;  if (xfOut->fp == NULL && (mcxIOopen(xfOut, ON_FAIL) != STATUS_OK))
      return STATUS_FAIL

   ;  mcxIOwriteCanary(fout, mclxCanary)
   
   ;  fwrite(&n_cols, sizeof(long), 1, fout)
   ;  fwrite(&n_rows, sizeof(long), 1, fout)

      /*
      // Write vector offsets (plus one for end of matrix body)
      //
      */
   ;  v_pos = ftell(fout) + (1 + n_cols) * sizeof(long)
   ;  while (--n_cols >= 0)
      {  mcxIOwriteInteger(fout, v_pos)
      ;  v_pos += sizeof(long) + vec->n_ivps * sizeof(mclIvp)
      ;  vec++
   ;  }
      mcxIOwriteInteger(fout, v_pos)

      /*
      // Write cols
      //
      */   
   ;  n_cols      =  N_COLS(mx)
   ;  vec         =  mx->cols

   ;  while (--n_cols >= 0)
      {  status = mclvEmbedWrite(vec, xfOut)
      ;  if (status == STATUS_FAIL) break
      ;  vec++
   ;  }
   
      if (mclVerbosityIoImpala)
      mcxTell
      (  "mclIO"
      ,  "wrote native binary %ldx%ld matrix to stream <%s>"
      ,  (long) N_ROWS(mx)
      ,  (long) N_COLS(mx)
      ,  xfOut->fn->str
      )
   ;  return status
;  }


/* reads single required part, so does not read too far */
static mcxstatus mclxrDimPart
(  mcxIO*   xf
,  int      *pn_rows
,  int      *pn_cols
)
   {  mcxHash* header      =  mcxHashNew(4, mcxTingHash, mcxTingCmp)
   ;  mcxTing* txtmx       =  mcxTingNew("mcltype")
   ;  mcxTing* txtdim      =  mcxTingNew("dimensions")
   ;  mcxKV    *kvtp, *kvdim
   ;  mcxstatus status     =  STATUS_OK

   ;  if(mcxIOfind(xf, "(mclheader", RETURN_ON_FAIL) != STATUS_OK)
      {  mcxHashFree(&header, NULL, NULL) /* hash still empty */
      ;  return STATUS_FAIL
   ;  }

      mclxParseDimPart(xf, header)  /* fills hash */
   /* fixme; check return status etc; (errors are noticed below though) */

   ;  kvtp  =  mcxHashSearch(txtmx, header, MCX_DATUM_FIND)
   ;  kvdim =  mcxHashSearch(txtdim, header, MCX_DATUM_FIND)

   ;  mcxTingFree(&txtmx)
   ;  mcxTingFree(&txtdim)

   ;  if (!kvtp)
      {  mcxErr(mclxar, "expected <mcltype matrix> specification not found")
      ;  mcxIOpos(xf, stderr)
      ;  status =  STATUS_FAIL
   ;  }
      else if
      (  !kvdim
      || (  sscanf
            (  ((mcxTing*) kvdim->val)->str
            ,  "%dx%d"
            ,  pn_rows
            ,  pn_cols
            )
            < 2
         )
      )
      {  mcxErr(mclxar, "expected <dimensions MxN> specification not found")
      ;  mcxIOpos(xf, stderr)
      ;  status =  STATUS_FAIL
   ;  }
      else if (*pn_rows < 0 || *pn_cols < 0)
      {  mcxErr
         (  mclxar
         ,  "each dimension must be nonnegative (found %ldx%ld pair)"
         ,  (long) *pn_rows
         ,  (long) *pn_cols
         )
      ;  status =  STATUS_FAIL
   ;  }

      mcxHashFree(&header, mcxTingFree_v, mcxTingFree_v)
   ;  return status
;  }


/* may read too far, hence returns line */
mcxTing* mclxrDomPart
(  mcxIO          *xf
,  mclVector**  dom_colspp
,  mclVector**  dom_rowspp
,  mcxstatus*  statusp
)
   {  mcxTing*    line     =  mcxTingEmpty(NULL, 30)
   ;  mclVector*  dom_cols =  NULL
   ;  mclVector*  dom_rows =  NULL

   ;  while ((line = mcxIOreadLine(xf, line, MCX_READLINE_CHOMP)))
      {  if (strncmp(line->str, "(mcl", 4))
         continue

      ;  if (!strncmp(line->str, "(mclcols", 8))
         {  if (dom_cols || mclParseDomain(xf, &dom_cols) == STATUS_FAIL)
            {  mcxErr(mclxar, "error parsing column domain")
            ;  goto fail
         ;  }
         }
         else if (!strncmp(line->str, "(mclrows", 8))
         {  if (dom_rows || mclParseDomain(xf, &dom_rows) == STATUS_FAIL)
            {  mcxErr(mclxar, "error parsing row domain")
            ;  goto fail
         ;  }
         }
         else if (!strncmp(line->str, "(mcldoms", 8))
         {  if
            (  dom_cols
            || dom_rows
            || mclParseDomain(xf, &dom_cols) == STATUS_FAIL
            )
            {  mcxErr(mclxar, "error parsing row domain")
            ;  goto fail
         ;  }
            dom_rows = mclvNew(dom_cols->ivps, dom_cols->n_ivps)
         ;  break
      ;  }
         else if (!strncmp(line->str, "(mclmatrix", 10))
         break
      ;  else
         {  mcxErr(mclxar, "unknown header type <%s>", line->str)
         ;  goto fail
      ;  }
   ;  }

      if (0)
      {  fail
      :  *statusp = STATUS_FAIL
      ;  mclvFree(&dom_cols)
      ;  mclvFree(&dom_rows)
      ;  mcxTingFree(&line)
   ;  }
      else
      *statusp = STATUS_OK

   ;  *dom_colspp = dom_cols  /* possibly NULL */
   ;  *dom_rowspp = dom_rows  /* possibly NULL */
   ;  return line
;  }


/* may read too far, hence returns line */
mcxTing* mclxReadAsciiHeader
(  mcxIO* xf
,  mclVector **dom_colspp
,  mclVector **dom_rowspp
,  mcxstatus *statusp
)
   {  int n_cols = 0, n_rows = 0
   ;  mcxTing* line  = NULL  
   ;  mclVector* dom_cols = NULL
   ;  mclVector* dom_rows = NULL

   ;  *statusp = STATUS_OK
   ;  *dom_colspp = NULL
   ;  *dom_rowspp = NULL

   ;  if (mclxrDimPart(xf, &n_rows, &n_cols) != STATUS_OK)
      {  mcxErr(mclxar, "error parsing dimension part")
      ;  goto fail
   ;  }

      line = mclxrDomPart(xf, dom_colspp, dom_rowspp, statusp)

   ;  if (*statusp != STATUS_OK)
      {  mcxErr(mclxar, "error constructing domains")
      ;  goto fail
   ;  }
      dom_rows = *dom_rowspp
   ;  dom_cols = *dom_colspp

   ;  if (!dom_rows)
      {  dom_rows = mclvCanonical(NULL, n_rows, 1.0)
      ;  *dom_rowspp = dom_rows
   ;  }
      else if (dom_rows->n_ivps != n_rows)
      {  mcxErr
         (  mclxar
         ,  "row domain count <%ld> != dimension <%ld>"
         ,  (long) dom_rows->n_ivps
         ,  (long) n_rows
         )
      ;  goto fail
   ;  }

      if (!dom_cols)
      {  dom_cols = mclvCanonical(NULL, n_cols, 1.0)
      ;  *dom_colspp = dom_cols
   ;  }
      else if (dom_cols->n_ivps != n_cols)
      {  mcxErr
         (  mclxar
         ,  "col domain count <%ld> != dimension <%ld>"
         ,  (long) dom_cols->n_ivps
         ,  (long) n_cols
         )
      ;  goto fail
   ;  }

      if (0)
      {  fail
      :  *statusp = STATUS_FAIL
      ;  mclvFree(&dom_cols)
      ;  mclvFree(&dom_rows)
      ;  mcxTingFree(&line)
   ;  }
      else
      *statusp = STATUS_OK

   ;  *dom_colspp = dom_cols
   ;  *dom_rowspp = dom_rows
   ;  return line
;  }


mclMatrix* mclxReadAscii
(  mcxIO          *xf
,  mcxOnFail      ON_FAIL
)
   {  mclVector*  dom_cols =  NULL
   ;  mclVector*  dom_rows =  NULL
   ;  mcxstatus   status   =  STATUS_FAIL
   ;  mcxTing*    line     =  NULL
   ;  mclMatrix*  mx       =  NULL
   ;  mcxbits     bits     =  MCLV_WARN_REPEAT

   ;  if (xf->fp == NULL && (mcxIOopen(xf, ON_FAIL) != STATUS_OK))
      goto fail

   ;  line = mclxReadAsciiHeader(xf, &dom_cols, &dom_rows, &status)
   ;  if (status != STATUS_OK)
      goto fail

   ;  while (line && strncmp(line->str, "(mclmatrix", 10))
      line = mcxIOreadLine(xf, line, MCX_READLINE_CHOMP)
      /* fixme should add section parsing [ delimited by ^(mcl .. ^) ] */

   ;  if (!line)
      {  mcxErr(mclxar, "(mclmatrix section not found")
      ;  goto fail
   ;  }

      if (mcxIOfind(xf, "begin", RETURN_ON_FAIL) == STATUS_FAIL)
      {  mcxErr(mclxar, "begin token not found in matrix specification")
      ;  goto fail
   ;  }

      mx = mclxAllocZero(dom_cols, dom_rows)

   ;  if
      (  mclxReadAsciiRaw
         (  xf, mx, ON_FAIL, ')', bits, mclpMergeLeft, fltLeft
         )
         != STATUS_OK
      )
      {  mx = NULL      /* twas freed by mclxReadAsciiRaw */
      ;  goto fail
   ;  }

      if (mclVerbosityIoImpala)
      mcxTell
      (  "mclIO"
      ,  "read native ascii %ldx%ld matrix from stream <%s>"
      ,  (long) N_ROWS(mx)
      ,  (long) N_COLS(mx)
      ,  xf->fn->str
      )
   ;  mcxTingFree(&line)
   ;

      if (0)
      {  fail:
      ;  if (ON_FAIL == RETURN_ON_FAIL)
         {  mcxTingFree(&line)
         ;  mclxFree(&mx)
         ;  return NULL
      ;  }
         else
         mcxExit(1)
   ;  }

      return mx
;  }


/* fixme; can't I make this more general,
 * with callback and void* argument ?
 *
 * fixme; remove offset == vid constraint [allready fixed?].
*/

mcxstatus mclxTaggedWrite
(  const mclMatrix*     mx
,  const mclMatrix*     el2dom
,  mcxIO                *xfOut
,  int                  valdigits
,  mcxOnFail            ON_FAIL
)
   {  int   i
   ;  FILE* fp
   ;  const char* me = "mclxTaggedWrite"  

   ;  if (!xfOut->fp && mcxIOopen(xfOut, ON_FAIL) != STATUS_OK)
      {  mcxErr(me, "cannot open stream <%s>", xfOut->fn->str)
      ;  return STATUS_FAIL
   ;  }

      fp =  xfOut->fp
   ;  mclxWriteAsciiHeader(mx, fp)

   ;  for (i=0;i<N_COLS(mx);i++)
      {  mclVector*  mvec  =  mx->cols+i
      ;  mclVector*  dvec  =  mclxGetVector
                              (  el2dom, mvec->vid, RETURN_ON_FAIL, NULL)
                             /*  fixme; make more efficient */
      ;  long tag = dvec && dvec->n_ivps ? dvec->ivps[0].idx : -1
      ;  int j

      ;  if (!mvec->n_ivps)
         continue

      ;  fprintf(fp, "%ld(%ld)  ", (long) mvec->vid, (long) tag)

      ;  for (j=0;j<mvec->n_ivps;j++)
         {  long  hidx  =  (mvec->ivps+j)->idx
         ;  double hval =  (mvec->ivps+j)->val

         ;  dvec  =  mclxGetVector(el2dom, hidx, RETURN_ON_FAIL, NULL)
         ;  tag   =  dvec && dvec->n_ivps ? dvec->ivps[0].idx : -1

         ;  if (valdigits > -1)
            fprintf
            (  fp
            ,  " %ld(%ld):%.*f"
            ,  (long) hidx
            ,  (long) tag
            ,  (int) valdigits
            ,  (double) hval
            )
         ;  else
            fprintf
            (  fp
            ,  " %ld(%ld)"
            ,  (long) hidx
            ,  (long) tag
            )
      ;  }
         fprintf(fp, " $\n")
   ;  }

      fprintf(fp, ")\n")
   ;  if (mclVerbosityIoImpala)
      mcxTell
      (  "mclIO"
      ,  "wrote native ascii %ldx%ld matrix to stream <%s>"
      ,  (long) N_ROWS(mx)
      ,  (long) N_COLS(mx)
      ,  xfOut->fn->str
      )
   ;  return STATUS_OK
;  }


void mclxWriteAsciiHeader
(  const mclMatrix* mx
,  FILE* fp
)
   {  int  idxwidth =  ((int) log10(N_ROWS(mx)+1)) + 1

   ;  fprintf
      (  fp
      ,  "(mclheader\nmcltype matrix\ndimensions %ldx%ld\n)\n"
      ,  (long) N_ROWS(mx)
      ,  (long) N_COLS(mx)
      )

   ;  if
      (  !mcldIsCanonical(mx->dom_rows)
      || !mcldIsCanonical(mx->dom_cols)
      )
      {  if (mcldEquate(mx->dom_rows, mx->dom_cols, MCL_DOM_EQUAL))
         {  fputs("(mcldoms\n", fp)
         ;  mclvDumpAscii
            (  mx->dom_cols
            ,  fp
            ,  idxwidth
            ,  -1
            ,  FALSE
            )
         ;  fputs(")\n", fp)
      ;  }
         else
         {  if (!mcldIsCanonical(mx->dom_rows))
            {  fputs("(mclrows\n", fp)
            ;  mclvDumpAscii
               (  mx->dom_rows
               ,  fp
               ,  idxwidth
               ,  -1
               ,  FALSE
               )
            ;  fputs(")\n", fp)
         ;  }
            if (!mcldIsCanonical(mx->dom_cols))
            {  fputs("(mclcols\n", fp)
            ;  mclvDumpAscii
               (  mx->dom_cols
               ,  fp
               ,  idxwidth
               ,  -1
               ,  FALSE
               )
            ;  fputs(")\n", fp)
         ;  }
         }
      }

      fputs("(mclmatrix\nbegin\n", fp)
;  }


mcxstatus mclxWriteAscii
(  const mclMatrix*        mx
,  mcxIO*                  xfOut
,  int                     valdigits
,  mcxOnFail               ON_FAIL
)
   {  int   i
                  /* fixme; need more sanity checks on N_ROWS(mx) ? ? */
   ;  int   idxwidth    =  ((int) log10(N_ROWS(mx)+1)) + 1
   ;  FILE* fp
   ;  const char* me    =  "mclxWriteAscii"

   ;  if (!xfOut->fp && mcxIOopen(xfOut, RETURN_ON_FAIL) != STATUS_OK)
      {  mcxErr(me, "cannot open stream <%s>", xfOut->fn->str)
      ;  return STATUS_FAIL
   ;  }

      fp =  xfOut->fp
   ;  mclxWriteAsciiHeader(mx, fp)

   ;  for (i=0;i<N_COLS(mx);i++)
      if ((mx->cols+i)->n_ivps)
      mclvDumpAscii
      (  mx->cols+i
      ,  fp
      ,  idxwidth
      ,  valdigits
      ,  FALSE
      )

   ;  fprintf(fp, ")\n")

   ;  if (mclVerbosityIoImpala)
      mcxTell
      (  "mclIO"
      ,  "wrote native ascii %ldx%ld matrix to stream <%s>"
      ,  (long) N_ROWS(mx)
      ,  (long) N_COLS(mx)
      ,  xfOut->fn->str
      )
   ;  return STATUS_OK
;  }


void mcxPrettyPrint
(  const mclMatrix*        mx
,  FILE*                   fp
,  int                     width
,  int                     digits
,  const char              msg[]
)
   {  int   i
   ;  char     bgl[]       =  " [ "
   ;  char     eol[]       =  "  ]"
   ;  mclMatrix*  tp       =  mclxTranspose(mx)
   ;  char  voidstring[20]

   ;  width                =  MAX(2, width)
   ;  width                =  MIN(width, 15)

   ;  memset(voidstring, ' ', width-2)
   ;  *(voidstring+width-2) = '\0'

   ;  for (i=0;i<N_COLS(tp);i++)
      {  mclVector*  rowVec   =  tp->cols+i
      ;  mclIvp*  domIvp      =  tp->dom_rows->ivps - 1
      ;  mclIvp*  domIvpMax   =  tp->dom_rows->ivps + tp->dom_rows->n_ivps

      ;  fprintf(fp, "%s", bgl)

      ;  while (++domIvp < domIvpMax)
         {  mclIvp* ivp = mclvGetIvp(rowVec, domIvp->idx, NULL)
         ;  if (!ivp)
            fprintf(fp, " %s--", voidstring)
         ;  else
            fprintf(fp, " %*.*f", (int) width, (int) digits, (double) ivp->val)
      ;  }
         fprintf(fp, "%s\n", eol)
   ;  }

      mclxFree(&tp)
   ;  if (msg)
      fprintf(fp, "^ %s\n", msg)
;  }


void mclxBoolPrint
(  mclMatrix*     mx
,  int            mode
)
   {  int      i, t                 
   ;  const char  *space   =  mode & 1 ? "" : " "
   ;  const char  *empty   =  mode & 1 ? " " : "  "

   ;  fprintf(stdout, "\n  ")        
   ;  for (i=0;i<N_ROWS(mx);i++)    
      fprintf(stdout, "%d%s", (int) i % 10, space)   
   ;  fprintf(stdout, "\n")

   ;  for (i=0;i<N_COLS(mx);i++)
      {  int         last        =  0
      ;  mclIvp*     ivpPtr      =  (mx->cols+i)->ivps
      ;  mclIvp*     ivpPtrMax   =  ivpPtr + (mx->cols+i)->n_ivps
      ;  fprintf(stdout, "%d ", (int) i%10)
                                    
      ;  while (ivpPtr < ivpPtrMax) 
         {  for (t=last;t<ivpPtr->idx;t++) fprintf(stdout, "%s", empty)
         ;  fprintf(stdout, "@%s", space)
         ;  last = (ivpPtr++)->idx + 1
      ;  }        
         for (t=last;t<N_ROWS(mx);t++) fprintf(stdout, "%s", empty)
      ;  fprintf(stdout, " %d\n", (int) i%10)   
   ;  }           

      fprintf(stdout, "  ")
   ;  for (i=0;i<N_ROWS(mx);i++)
      fprintf(stdout, "%d%s", (int) i%10, space)
   ;  fprintf(stdout, "\n")
;  }


void mclvDumpAscii
(  const mclVector*  vec
,  FILE*    fp
,  int      idxwidth
,  int      valdigits
,  mcxbool  doHeader
)
   {  int vid = vec->vid
   ;  int nr_chars   =     0
   ;  int fieldwidth =     idxwidth+1
   ;  const char* eov =    " $\n"
   ;  int i
                                          /* works for 0.xxx 1.xxx .. */
   ;  if (valdigits >= 0)
      fieldwidth += valdigits + 3

   ;  if (doHeader)
      {  fprintf(fp , "(mclheader\nmcltype vector\n)\n" "(mclvector\nbegin\n")
      ;  eov = " $\n)\n"
   ;  }

      if (vid>=0)
      {  fprintf(fp, "%-*ld  ", (int) idxwidth, (long) vid)
      ;  nr_chars = idxwidth + 2
   ;  }

      for (i=0; i<vec->n_ivps;i++)
      {  if (valdigits > -1)
         {  fprintf
            (  fp
            ,  " %*ld:%-*.*f"
            ,  (int) idxwidth
            ,  (long) (vec->ivps+i)->idx
            ,  (int) valdigits+2
            ,  (int) valdigits
            ,  (double) (vec->ivps+i)->val
            )
         ;  nr_chars += idxwidth + valdigits + 4   /* 4 chars: [01]\.\:\ */
      ;  }
         else
         {  fprintf(fp, " %*ld", (int) idxwidth, (long) (vec->ivps+i)->idx)
         ;  nr_chars += idxwidth + 1
      ;  }

         if
         (  (  (i+2 < vec->n_ivps)
            && (nr_chars + fieldwidth > 78) 
            )
         ||
            (  (i+2 == vec->n_ivps)
            && (nr_chars + fieldwidth + 2 > 78)    /* mq what's this? */
            )
         )
         {  int j
         ;  fputc('\n', fp)
                                       /* fixme below is _very_ stupid */
         ;  if (vid >= 0)
            {  for (j=0;j<idxwidth+2;j++)
               fputc(' ', fp)
            ;  nr_chars =  idxwidth+2
         ;  }
            else
            nr_chars = 0
      ;  }
      }
      fputs(eov, fp)
;  }


static void report_vector_size
(  const char*             action
,  const mclVector*           vec
)
   {  char                 report[80]

   ;  sprintf
      (  report, "%s %ld pair%s\n"
      ,  action
      ,  (long) vec->n_ivps
      ,  vec->n_ivps == 1 ? "" : "s"
      )
   ;  mcxTell(NULL, report)
;  }


mcxstatus mclvEmbedRead
(  mclVector*     vec
,  mcxIO*         xf
,  mcxOnFail      ON_FAIL
)
   {  int n_ivps =  0      /* fixme; check vec             */
                           /* fixme; check retval of fread */
   ;  n_ivps = mcxIOreadInteger(xf->fp)

   ;  if (n_ivps && mclvResize(vec, n_ivps))
      {  fread(vec->ivps, sizeof(mclIvp), n_ivps, xf->fp)
      ;  mclvSort(vec, NULL)
      ;  mclvUniqueIdx(vec, mclpMergeAdd)
            /*
             * fixme; param mclpMerge fixme; this should be true vector, so no
             * need for sorting and dedupping This code is near-dead btw.
            */
      ;  return STATUS_OK
   ;  }
      else
      mclvResize(vec, 0)

   ;  return STATUS_OK
;  }


mclpAR* mclpReaDaList
(  mcxIO    *xf
,  mclpAR*  ar
,  int      *sortbits
,  int finalToken
)
   {  int n_ivps = 0
   ;  const char* me = "mclpReaDaList"
   ;  mcxbool ok = FALSE
   ;  int sorted = 1
   ;  int noduplicates = 1
   ;  long previdx = -1

   ;  if (sortbits)
      *sortbits = 0

   ;  if (!ar)
      ar = mclpARresize(NULL, 100)

   ;  while (1)
      {  long idx
      ;  double val
      ;  mclIvp* ivp
      ;  int c = mcxIOskipSpace(xf)  /* c is ungotten */

      ;  if (c == finalToken)
         {  mcxIOstep(xf)  /* discard '$' or EOF etc */
         ;  ok = TRUE
         ;  break
      ;  }
         else if (c == '#')
         {  mcxIOdiscardLine(xf)
         ;  continue
      ;  }

         if (mcxIOexpectNum(xf, &idx, RETURN_ON_FAIL) == STATUS_FAIL)
         {  mcxErr(me, "expected row index")
         ;  break
      ;  }
         else if (idx < 0)
         {  mcxErr(me, "found negative index <%ld>", (long) idx)
         ;  break
      ;  }

         if (idx < previdx)
         sorted = 0
      ;  if (idx == previdx)
         noduplicates = 0
      ;
         n_ivps++
      ;
      expect_val

      :  if (':' == (c = mcxIOskipSpace(xf)))
         {  mcxIOstep(xf) /* discard ':' */
         ;  if (mcxIOexpectReal(xf, &val, RETURN_ON_FAIL) == STATUS_FAIL)
            {  mcxErr(me, "expected value after row index <%ld>", (long) idx)
            ;  break
         ;  }
         }
         else if ('(' == c)
         {  if (mcxIOfind(xf, ")", RETURN_ON_FAIL) == STATUS_FAIL)
            {  mcxErr(me, "could not skip over s-expression <%ld>", (long) idx)
            ;  break
         ;  }
            goto expect_val
      ;  }
         else
         val = 1.0

      ;  if (ar->n_alloc <= n_ivps)
         mcxResize
         (  &(ar->ivps)
         ,  sizeof(mclp)
         ,  &(ar->n_alloc)
         ,  n_ivps * 2
         ,  EXIT_ON_FAIL   /* fixme; respect ON_FAIL */
         )

      ;  ivp      =  ar->ivps + n_ivps - 1
      ;  ivp->val =  val
      ;  ivp->idx =  idx
      ;  previdx  =  idx
   ;  }

      if (!ok)
      {  mclpARfree(&ar)
      ;  return NULL
   ;  }

      if (sortbits && sorted)
      {  *sortbits |= 1
      ;  if (noduplicates)
         *sortbits |= 2
   ;  }
      /* don't set noduplicates unless the thing is sorted */

      ar->n_ivps = n_ivps
   ;  return ar
;  }


static mcxstatus mclReaDaVec
(  mcxIO*      xf
,  mclv*       dst
,  mclpAR*     ar
,  int         finalToken
,  mcxbits     warn_repeat
,  void (*ivpmerge)(void* ivp1, const void* ivp2)
,  double (*fltbinary)(pval val1, pval val2)
)
   {  mclpAR* arcp = ar
   ;  int sortbits = 0  /* 1: in sorted order, 2: no duplicates present. */

   ;  if (!(ar = mclpReaDaList(xf, ar, &sortbits, finalToken)))
      return STATUS_FAIL

   ;  mclvFromIvps_x
      (dst,ar->ivps,ar->n_ivps,warn_repeat, sortbits, ivpmerge, fltbinary)
   ;  if (!arcp)
      mclpARfree(&ar)
   ;  return STATUS_OK
;  }


mcxstatus mclvEmbedWrite
(  const mclVector*     vec
,  mcxIO*               xfOut
)
   {  mcxIOwriteInteger(xfOut->fp, vec->n_ivps)
   ;  if (vec->n_ivps)
      fwrite(vec->ivps, sizeof(mclIvp), vec->n_ivps, xfOut->fp)

   ;  return STATUS_OK
;  }


mcxstatus mclvWrite
(  const mclVector      *vec
,  mcxIO                *xfOut
,  mcxOnFail            ON_FAIL
)
   {  mcxstatus         status

   ;  if (xfOut->fp == NULL && mcxIOopen(xfOut, ON_FAIL) != STATUS_OK)
      {  mcxErr("mclvWrite", "cannot open stream <%s>", xfOut->fn->str)
      ;  return STATUS_FAIL
   ;  }

      mcxIOwriteCanary(xfOut->fp, mclvCanary)
   ;  status = mclvEmbedWrite(vec, xfOut)
   ;  if (status == STATUS_OK)
      report_vector_size("wrote", vec)

   ;  return status
;  }


static mcxstatus mclParseDomain
(  mcxIO        *xf
,  mclVector**  dompp
)
   {  mclVector *dom = *dompp

   ;  if (!dom)
      dom = mclvInit(NULL)

   ;  *dompp = dom

   ;  if
      (  mclReaDaVec
         (  xf
         ,  dom
         ,  NULL
         ,  '$'
         ,  MCLV_WARN_REPEAT
         ,  mclpMergeLeft
         ,  NULL
         )
         == STATUS_OK
      )
      {  if (')' == mcxIOskipSpace(xf))
         {  mcxIOstep(xf) /* discard ')' */
         ;  return STATUS_OK
      ;  }
         return STATUS_FAIL
   ;  }
      return STATUS_FAIL
;  }


static mcxstatus mclxParseDimPart
(  mcxIO        *xf
,  mcxHash      *header
)
   {  int  n
   ;  mcxTing   *keyTxt  =  (mcxTing*) mcxTingInit(NULL)
   ;  mcxTing   *valTxt  =  (mcxTing*) mcxTingInit(NULL)
   ;  mcxTing   *line    =  (mcxTing*) mcxTingInit(NULL)

   ;  while ((line = mcxIOreadLine(xf, line, MCX_READLINE_CHOMP)))
      {  if (*(line->str+0) == ')')
         break

      ;  mcxTingEnsure(keyTxt, line->len)
      ;  mcxTingEnsure(valTxt, line->len)

      ;  n = sscanf(line->str, "%s%s", keyTxt->str, valTxt->str)

      ;  if (n < 2)
         continue
      ;  else
         {  mcxTing* key   =  mcxTingNew(keyTxt->str)
         ;  mcxTing* val   =  mcxTingNew(valTxt->str)
         ;  mcxKV*   kv    =  mcxHashSearch(key, header, MCX_DATUM_INSERT)
         ;  kv->val        =  val
      ;  }
      }

      mcxTingFree(&line)
   ;  mcxTingFree(&valTxt)
   ;  mcxTingFree(&keyTxt)
   ;  return STATUS_OK
;  }



mclMatrix* mclxRead
(  mcxIO       *xf
,  mcxOnFail   ON_FAIL
)  
   {  return mclxMaskedRead(xf, NULL, ON_FAIL)
;  }


void mclFlowPrettyPrint
(  const mclMatrix*  mx
,  FILE*             fp
,  int               digits
,  const char        msg[]
)
   {  mcxPrettyPrint
      (  mx
      ,  fp
      ,  digits+2
      ,  digits
      ,  msg
      )
;  }


void mclvWriteAscii
(  const mclVector*  vec
,  FILE*             fp
,  int               valdigits
)  
   {  mclvDumpAscii
      (  vec
      ,  fp
      ,  1
      ,  valdigits
      ,  TRUE
      )
;  }



mclpAR *mclpReadAsciiRaw
(  mcxIO       *xf
,  mcxOnFail   ON_FAIL
,  int         finalToken     /* e.g. EOF or '$' */
)
   {  mclpAR* ar = mclpReaDaList(xf, NULL, NULL, finalToken)
   ;  if (!ar && ON_FAIL != RETURN_ON_FAIL)
      mcxExit(1)
   ;  return ar
;  }



/* fixme; add expect_vid argument */
/* fixme;? add ar buffer argument */
mclVector* mclvReadAsciiRaw
(  mcxIO          *xf
,  mclpAR*        ar
,  mcxOnFail      ON_FAIL
,  int            finalToken     /* e.g. EOF or '$' */
,  mcxbits        warn_repeat
,  void (*ivpmerge)(void* ivp1, const void* ivp2)
)
   {  mclVector* vec = mclvInit(NULL)  /* cannot use create; vec must be ok */
   ;  if
      (  mclReaDaVec(xf, vec, ar, finalToken, warn_repeat, ivpmerge, NULL)
      != STATUS_OK
      )
      {  mcxErr("mclvReadAsciiRaw", "read failed in <%s>", xf->fn->str)
      ;  if (ON_FAIL == EXIT_ON_FAIL)
         mcxExit(1)
      ;  return NULL
   ;  }
      if (mclVerbosityIoImpala)
      mcxTell
      (  "mclIO"
      ,  "read raw ascii <%ld> vector from stream <%s>"
      ,  (long) vec->n_ivps
      ,  xf->fn->str
      )
   ;  return vec
;  }


mcxstatus mclxReadAsciiRaw
(  mcxIO          *xf
,  mclMatrix      *mx
,  mcxOnFail      ON_FAIL
,  int            finalToken     /* e.g. EOF or ')' */
,  mcxbits        bits
,  void (*ivpmerge)(void* ivp1, const void* ivp2)
,  double (*fltbinary)(pval val1, pval val2)
)
   {  mclVector*  dom_rows =  mx->dom_rows
   ;  const char* me       =  "mclxReadAsciiRaw"
   ;  mclpAR*     ar       =  mclpARresize(NULL, 100)

   ;  int         N_cols   =  N_COLS(mx)
   ;  int         n_cols   =  0
   ;  int         n_mod    =  MAX(1+(N_cols-1)/40, 1)
   ;  mcxbool     progress =  isatty(fileno(stderr))
                              && N_cols > 1
                              && mclVerbosityIoImpala

   ;  if (progress)
         fprintf(stderr, "[mclIO] reading <%s> ", xf->fn->str)
      ,  fflush(NULL)

   ;  if (xf->fp == NULL && (mcxIOopen(xf, ON_FAIL) != STATUS_OK))
      goto fail

   ;  while (1)
      {  long           cidx
      ;  mclVector*     vec
      ;  int a = mcxIOskipSpace(xf)

      ;  if (a == finalToken)
         break
      ;  else if (a == '#')
         {  mcxIOdiscardLine(xf)
         ;  continue
      ;  }

         if (mcxIOexpectNum(xf, &cidx, RETURN_ON_FAIL) == STATUS_FAIL)
         {  mcxErr(me, "expected column index")
         ;  goto fail
      ;  }

         vec = mclxGetVector(mx, cidx, RETURN_ON_FAIL, NULL)
      ;  if (!vec)
         {  mcxErr(me, "found alien col index <%ld>", (long) cidx)
         ;  goto fail
      ;  }
         vec->vid = cidx;

      ;  if
         (  mclReaDaVec(xf, vec, ar, '$', bits, ivpmerge, fltbinary)
         == STATUS_FAIL
         )
         {  mcxErr(me, "vector read failed for column <%ld>", (long) cidx)
         ;  goto fail
      ;  }

         if (mcldCountSet(vec, dom_rows, MCL_DOM_LDIF))
         {  mclv* ldif = mcldMinus(vec, dom_rows, NULL)
         ;  mcxErr
            (  me
            ,  "alien row indices in column <%ld> - (a total of %ld)"
            ,  (long) cidx
            ,  (long) ldif->n_ivps
            )
         ;  mcxErr(me, "the first is <%ld>", (long) ldif->ivps[0].idx)
         ;  goto fail
      ;  }
         n_cols++
      ;  if (progress && n_cols % n_mod == 0)
            fputc('.', stderr)
         ,  fflush(NULL)
   ;  }
      if (progress)
      fputc('\n', stderr)

   ;  if (mclVerbosityIoImpala)
      mcxTell
      (  "mclIO"
      ,  "read raw ascii %ldx%ld matrix from stream <%s>"
      ,  (long) N_ROWS(mx)
      ,  (long) N_COLS(mx)
      ,  xf->fn->str
      )
   ;  mclpARfree(&ar)
   ;

      if (0)
      {  fail
      :  if (ON_FAIL == RETURN_ON_FAIL)
         {  mclxFree(&mx)  
         ;  return STATUS_FAIL
      ;  }
         else
         mcxExit(1)
   ;  }

      return STATUS_OK
;  }

