/*
 * Show factors of numbers around a specified number.
 * Works on up to 14 digit numbers.
 *
 * Copyright (C) 2004 George Gesslein II.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <time.h>
#include <assert.h>

#define	true	1
#define	false	0

void factor_one(double start);
int positive_integer(double arg);
void try_factor(double arg);
void usage(void);

double nn, vv;
double huge_value = 1.0e14;
double sq[] = {
	10, 2, 4, 2, 4, 6, 2, 6,
	 4, 2, 4, 6, 6, 2, 6, 4,
	 2, 6, 4, 6, 8, 4, 2, 4,
	 2, 4, 8, 6, 4, 6, 2, 4,
	 6, 2, 6, 6, 4, 2, 4, 6,
	 2, 6, 4, 2, 4, 2,10, 2
};

double	unique[64];
int	ucnt[64];
int	uno;

char	*prog_name;

int
main(int argc, char *argv[])
{
	int		i;
	double		start;
	double		stop_at;

	prog_name = argv[0];
askformore:
	if (argc <= 1) {
		printf("Enter integer to factor around (0 to exit): ");
		if ((scanf("%lf", &start) != 1) || (start == 0.0))
			exit(0);
		stop_at = start + 10;
		start -= 10;
		if (start < 1.0) {
			start = 1.0;
		}
	} else {
		if (argc > 3) {
			usage();
		}
		i = sscanf(argv[1], "%lf-%lf", &start, &stop_at);
		if (i == 2) {
			if (argc != 2) {
				usage();
			}
		} else if (i == 1) {
			if (argc == 3) {
				if (sscanf(argv[2], "%lf", &stop_at) != 1) {
					usage();
				}
			} else {
				stop_at = start + 10;
				start -= 10;
				if (start < 1.0) {
					start = 1.0;
				}
			}
		} else {
			usage();
		}
	}
	if (!positive_integer(start) || !positive_integer(stop_at)) {
		fprintf(stderr, "Number must be a positive integer.\n");
		goto endloop;
	}
	if (start >= huge_value || stop_at >= huge_value) {
		fprintf(stderr, "Number is too large.\n");
		goto endloop;
	}
	for (; start <= stop_at; start++) {
		printf("%.0f = ", start);
		factor_one(start);
		for (i = 0; i < uno; i++) {
			if (i > 0) {
				printf(" * ");
			}
			printf("%.0f", unique[i]);
			if (ucnt[i] != 1) {
				printf("^%d", ucnt[i]);
			}
		}
		printf("\n");
	}
endloop:
	if (argc <= 1)
		goto askformore;
	exit(0);
}

/*
 * Factor "start" into the unique array.
 */
void
factor_one(double start)
{
	int		j;
	double		ii;

	nn = start;
	uno = 0;
	vv = 1.0 + sqrt(nn);
	try_factor(2.0);
	try_factor(3.0);
	try_factor(5.0);
	try_factor(7.0);
	ii = 1.0;
	while (ii <= vv) {
		for (j = 0; j < 48; j++) {
			ii += sq[j];
			try_factor(ii);
		}
	}
	if (nn > 1.0) {
		try_factor(nn);
	}
	assert(nn == 1.0);
}

/*
 * Return true if passed double is a positive integer.
 */
int
positive_integer(double arg)
{
	return(arg > 0.0 && fmod(arg, 1.0) == 0.0);
}

/*
 * See if "arg" is a factor of "nn".
 * If so, save it and remove it from "nn".
 */
void
try_factor(double arg)
{
	while (fmod(nn, arg) == 0.0) {
		if (uno > 0 && unique[uno-1] == arg) {
			ucnt[uno-1]++;
		} else {
			unique[uno] = arg;
			ucnt[uno] = 1;
			uno++;
		}
		nn = nn/arg;
		vv = 1.0 + sqrt(nn);
	}
}

void
usage()
{
	fprintf(stderr,
	    "\nUsage: %s number1 [number2]\n\n", prog_name);
	fprintf(stderr,
	    "Display prime factors of integers around the specified number.\n");
	fprintf(stderr,
	    "Works on up to 14 digit integers.\n");
	fprintf(stderr,
	    "A number range may be specified on the command line.\n");
	exit(1);
}
