/*
 * Expression parsing routines for Mathomatic.
 */

#include "includes.h"

/*
 * This is a simple mathematical expression parser.
 *
 * Returns the new string position or NULL on error.
 */
char	*
parse_section(equation, np, cp)
token_type	*equation;	/* where the parsed expression is stored */
int		*np;		/* pointer to parsed expression size */
char		*cp;		/* string to parse */
{
	int		n;
	int		cur_level;
	int		operand;
	char		*startp;
	char		*cp_start;
	char		*cp1;
	double		d;
	int		abs_count;
	int		abs_array[10];

	if (cp == NULL)
		return(NULL);
	cp_start = cp;
	n = 0;
	*np = 0;
	cur_level = 1;
	abs_count = 0;
	operand = false;
	for (;; cp++) {
		switch (*cp) {
		case ' ':
		case '\t':
			continue;
		case '(':
		case '[':
			cur_level++;
			if (operand) {
				goto syntax_error;
			}
			continue;
		case ')':
		case ']':
			cur_level--;
			if (cur_level <= 0
			    || (abs_count > 0 && cur_level < abs_array[abs_count-1])) {
#if	!SILENT
				put_up_arrow((int) (cp - cp_start));
				printf(_("Too many right parentheses.\n"));
#endif
				return(NULL);
			}
			if (!operand) {
				goto syntax_error;
			}
			continue;
		case '=':
		case 0:
		case '\n':
			goto p_out;
		}
		if (n > (n_tokens - 6)) {
			error_huge();
		}
		operand = !operand;
		switch (*cp) {
		case '|':
			if (operand) {
				if (abs_count >= ARR_CNT(abs_array)) {
#if	!SILENT
					printf(_("Too many nested absolute values.\n"));
#endif
					return(NULL);
				}
				cur_level += 2;
				abs_array[abs_count++] = cur_level;
				operand = false;
			} else {
				if (abs_count <= 0
				    || cur_level != abs_array[--abs_count]) {
					goto syntax_error;
				}
				cur_level--;
				equation[n].level = cur_level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = POWER;
				n++;
				equation[n].level = cur_level;
				equation[n].kind = CONSTANT;
				equation[n].token.constant = 2.0;
				n++;
				equation[n].level = cur_level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = POWER;
				n++;
				equation[n].level = cur_level;
				equation[n].kind = CONSTANT;
				equation[n].token.constant = 0.5;
				n++;
				cur_level--;
				operand = true;
			}
			break;
		case '!':
			if (operand) {
				goto syntax_error;
			}
			equation[n].level = cur_level;
			equation[n].kind = OPERATOR;
			equation[n].token.operatr = FACTORIAL;
			n++;
			equation[n].level = cur_level;
			equation[n].kind = CONSTANT;
			equation[n].token.constant = 0.0;
			n++;
			operand = true;
			break;
		case '^':
			if (operand) {
				goto syntax_error;
			}
			equation[n].level = cur_level;
			equation[n].kind = OPERATOR;
			equation[n].token.operatr = POWER;
			n++;
			break;
		case '*':
		case '/':
		case '%':
			if (operand) {
				goto syntax_error;
			}
			equation[n].level = cur_level;
			equation[n].kind = OPERATOR;
			switch (*cp) {
			case '*':
				equation[n].token.operatr = TIMES;
				break;
			case '/':
				equation[n].token.operatr = DIVIDE;
				break;
			case '%':
				equation[n].token.operatr = MODULUS;
				break;
			}
			n++;
			break;
		case '+':
		case '-':
			if (!operand) {
				equation[n].level = cur_level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = ((*cp == '+') ? PLUS : MINUS);
				n++;
			}
			if (strncasecmp(cp, "+/-", 3) == 0) {
				equation[n].level = cur_level;
				equation[n].kind = VARIABLE;
				next_sign(&equation[n].token.variable);
				n++;
				equation[n].level = cur_level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = TIMES;
				n++;
				cp += 2;
				operand = false;
				break;
			}
			if (!operand)
				break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
		case '.':
			if (!operand) {
				goto syntax_error;
			}
			if (*cp == '-' && !((isascii(cp[1]) && isdigit(cp[1]))
			    || cp[1] == '.')) {
				equation[n].kind = CONSTANT;
				equation[n].token.constant = -1.0;
				equation[n].level = cur_level;
				n++;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = NEGATE;
				equation[n].level = cur_level;
				n++;
				operand = false;
				continue;
			}
			startp = cp;
			errno = 0;
			d = strtod(startp, &cp);
			if (errno) {
#if	!SILENT
				put_up_arrow((int) (startp - cp_start));
				printf(_("Constant out of range.\n"));
#endif
				return(NULL);
			}
			if (cp == startp) {
#if	!SILENT
				put_up_arrow((int) (startp - cp_start));
				printf(_("Error parsing constant.\n"));
#endif
				return(NULL);
			}
			equation[n].kind = CONSTANT;
			equation[n].token.constant = d;
			equation[n].level = cur_level;
			n++;
			cp--;
			break;
		default:
			if (!isvarchar(*cp)) {
				goto syntax_error;
			}
			if (!operand) {
				operand = true;
				equation[n].level = cur_level;
				equation[n].kind = OPERATOR;
				equation[n].token.operatr = TIMES;
				n++;
			}
			if (strncasecmp(cp, "inf", 3) == 0) {
				equation[n].kind = CONSTANT;
				equation[n].token.constant = HUGE_VAL;
				if (strncasecmp(cp, "infinity", 8) == 0) {
					cp += 8;
				} else {
					cp += 3;
				}
			} else {
				equation[n].kind = VARIABLE;
				cp1 = cp;
				cp = parse_var(&equation[n].token.variable, cp);
				if (cp == NULL) {
#if	!SILENT
					put_up_arrow((int) (cp1 - cp_start));
					printf(_("Invalid variable.\n"));
#endif
					return(NULL);
				}
			}
			cp--;
			equation[n].level = cur_level;
			n++;
			break;
		}
	}
p_out:
	if (abs_count != 0 || (n && !operand)) {
		goto syntax_error;
	}
	if (cur_level != 1) {
#if	!SILENT
		put_up_arrow((int) (cp - cp_start));
		printf(_("Missing right parenthesis.\n"));
#endif
		return(NULL);
	}
	if (*cp == '=')
		cp++;
	*np = n;
	if (n) {
		handle_negate(equation, *np);
		prior_sub(equation, *np);
		organize(equation, np);
	}
	return cp;

syntax_error:
#if	!SILENT
	put_up_arrow((int) (cp - cp_start));
	printf(_("Syntax error.\n"));
#endif
	return(NULL);
}

/*
 * Parse variable name pointed to by "cp".
 * Variable name is converted to Mathomatic format and stored in "*vp".
 *
 * Return new string position, or NULL on failure.
 */
char	*
parse_var(vp, cp)
long	*vp;
char	*cp;
{
	int	i, j;
	long	vtmp;
	char	buf[MAX_VAR_LEN+1];
	char	*cp1;
	int	len;

	if (vp == NULL || cp == NULL) {
		return(NULL);
	}
	if (!isvarchar(*cp)) {
		return(NULL);
	}
	cp1 = cp;
	for (i = 0; *cp1;) {
		if (!isvarchar(*cp1)) {
			break;
		}
		if (i >= MAX_VAR_LEN) {
			return(NULL);
		}
		buf[i++] = *cp1++;
	}
	buf[i] = '\0';
	if (strcasecmp(buf, "sign") == 0) {
		vtmp = SIGN;
		cp += 4;
	} else if (strcasecmp(buf, "integer") == 0) {
		vtmp = V_INTEGER;
		cp += 7;
	} else if (strcasecmp(buf, "temp") == 0) {
		vtmp = V_TEMP;
		cp += 4;
	} else {
		if (strncasecmp(cp, "i#", 2) == 0) {
			*vp = IMAGINARY;
			return(cp + 2);
		}
		if (strncasecmp(cp, "e#", 2) == 0) {
			*vp = V_E;
			return(cp + 2);
		}
		if (strncasecmp(cp, "p#", 2) == 0 || strcasecmp(buf, "pi") == 0) {
			*vp = V_PI;
			return(cp + 2);
		}
		for (i = 0; *cp;) {
			if (!isvarchar(*cp) && !(isascii(*cp) && isdigit(*cp))) {
				break;
			}
			if (i >= MAX_VAR_LEN) {
				return(NULL);
			}
			buf[i++] = *cp++;
		}
		buf[i] = '\0';
		vtmp = 0;
		for (i = 0; var_names[i]; i++) {
			if (case_sensitive_flag) {
				if (strcmp(buf, var_names[i]) == 0) {
					vtmp = i + VAR_OFFSET;
					break;
				}
			} else {
				if (strcasecmp(buf, var_names[i]) == 0) {
					vtmp = i + VAR_OFFSET;
					break;
				}
			}
		}
		if (vtmp == 0) {
			if (i >= (MAX_VAR_NAMES - 1)) {
#if	!SILENT
				printf(_("Maximum number of variable names reached.\n"));
				printf(_("Please restart or use \"clear all\".\n"));
#endif
				return(NULL);
			}
			len = strlen(buf) + 1;
			var_names[i] = (char *) malloc(len);
			if (var_names[i] == NULL) {
#if	!SILENT
				printf(_("Out of memory!  (can't malloc()).\n"));
#endif
				return(NULL);
			}
			blt(var_names[i], buf, len);
			vtmp = i + VAR_OFFSET;
			var_names[i+1] = NULL;
		}
	}
	if (isascii(*cp) && isdigit(*cp)) {
		j = strtol(cp, &cp, 10);
		if (j < 0 || j > MAX_SUBSCRIPT) {
			return(NULL);
		}
		if (vtmp == SIGN) {
			sign_array[j+1] = true;
		}
		vtmp += ((long) (j + 1)) << VAR_SHIFT;
	}
	while (*cp == '\'') {
		cp++;
		vtmp += PRIME_INCREMENT;
		if (vtmp < 0) {
			return(NULL);
		}
	}
	*vp = vtmp;
	return cp;
}

/*
 * Return true if passed variable is a constant.
 * Return value of constant in "*dp".
 */
int
var_is_const(v, dp)
long	v;
double	*dp;
{
	switch (v) {
	case V_E:
		*dp = E;
		return true;
	case V_PI:
		*dp = PI;
		return true;
	}
	return false;
}

/*
 * Substitute E and PI variables with their respective constants.
 */
int
subst_constants(equation, np)
token_type	*equation;
int		*np;
{
	int	i;
	int	modified;
	double	d;

	modified = false;
	for (i = 0; i < *np; i += 2) {
		if (equation[i].kind == VARIABLE) {
			if (var_is_const(equation[i].token.variable, &d)) {
				equation[i].kind = CONSTANT;
				equation[i].token.constant = d;
				modified = true;
			}
		}
	}
	return modified;
}

/*
 * Parenthesize an operator.
 */
binary_parenthesize(equation, n, i)
token_type	*equation;	/* expression */
int		n;		/* length of expression */
int		i;		/* location of operator to parenthesize in expression */
{
	int	j;
	int	level;

#if	!SILENT
	if (equation[i].kind != OPERATOR || (n % 2) != 1) {
		printf("Invalid use of binary_parenthesize().\n");
	}
#endif
	level = equation[i].level++;
	if (equation[i-1].level++ > level) {
		for (j = i - 2; j >= 0; j--) {
			if (equation[j].level <= level)
				break;
			equation[j].level++;
		}
	}
	if (equation[i+1].level++ > level) {
		for (j = i + 2; j < n; j++) {
			if (equation[j].level <= level)
				break;
			equation[j].level++;
		}
	}
}

handle_negate(equation, n)
token_type	*equation;
int		n;
{
	int	i;

	for (i = 1; i < n; i += 2) {
		if (equation[i].token.operatr == NEGATE) {
			equation[i].token.operatr = TIMES;
			binary_parenthesize(equation, n, i);
		}
	}
}

/*
 * Give different operators on the same level
 * in an expression the correct priority.
 *
 * organize() should be called after this.
 */
prior_sub(equation, n)
token_type	*equation;
int		n;
{
	int	i;

	for (i = 1; i < n; i += 2) {
		if (equation[i].token.operatr == FACTORIAL) {
			binary_parenthesize(equation, n, i);
		}
	}
	for (i = 1; i < n; i += 2) {
		if (equation[i].token.operatr == POWER) {
			binary_parenthesize(equation, n, i);
		}
	}
	for (i = 1; i < n; i += 2) {
		switch (equation[i].token.operatr) {
		case TIMES:
		case DIVIDE:
		case MODULUS:
			binary_parenthesize(equation, n, i);
			break;
		}
	}
}

/*
 * Convert all alphabetic characters in a string to lower case.
 */
str_tolower(cp)
char	*cp;
{
	for (; *cp; cp++) {
		if (isascii(*cp) && isupper(*cp))
			*cp = tolower(*cp);
	}
}

/*
 * Return true if character is a valid starting variable character.
 */
int
isvarchar(ch)
int	ch;
{
	if (ch == '_' || ch == '\\')
		return true;
	if (isascii(ch) && isalpha(ch))
		return true;
	return false;
}

#ifndef	my_strlcpy
/*
 * A very efficient strlcpy().
 *
 * Copy up to (n - 1) characters from string in src
 * to dest and null-terminate the result.
 *
 * Return length of src.
 */
int
my_strlcpy(dest, src, n)
char	*dest, *src;
int	n;
{
	int	len, len_src;

	len_src = strlen(src);
	len = min(n - 1, len_src);
	memmove(dest, src, len);
	dest[len] = '\0';
	return len_src;
}
#endif

#ifndef	my_strlcat
/*
 * strlcat() from OpenBSD.
 */
int
my_strlcat(char *dst, const char *src, int siz)
{
	char *d = dst;
	const char *s = src;
	int n = siz;
	int dlen;

	/* Find the end of dst and adjust bytes left but don't go past end */
	while (n-- != 0 && *d != '\0')
		d++;
	dlen = d - dst;
	n = siz - dlen;

	if (n == 0)
		return(dlen + strlen(s));
	while (*s != '\0') {
		if (n != 1) {
			*d++ = *s;
			n--;
		}
		s++;
	}
	*d = '\0';

	return(dlen + (s - src));	/* count does not include NUL */
}
#endif
