/*
 * This file contains main() and startup code for Mathomatic.
 *
 * Mathomatic Copyright (c) 1996 George Gesslein II.
 *
 * Output to stderr is only done in this file.  The rest of the code
 * should not output to stderr.
 *
 * This program only supports binary and unary operators.
 * Unary operators are implemented as a binary operation with a dummy operand.
 *
 * In the storage format, each level of parenthesis is indicated
 * by a level number (origin 1).  The deeper the level, the
 * higher the level number.
 *
 * The storage format for expressions is a fixed size array of elements
 * "token_type", which may be a CONSTANT, VARIABLE, or OPERATOR.
 * The array always alternates between operand (CONSTANT or VARIABLE)
 * and OPERATOR.  There is a separate integer for each array which
 * contains the current length of the expression stored in the array.
 * This length is always odd.
 *
 * Only one POWER operator is allowed per level in the storage format,
 * and no other operators may be on that level.  Same with the FACTORIAL
 * and MODULUS operators.
 *
 * Any number of TIMES and DIVIDE operators may be on the same
 * level, because they are simple multiplicative class operators.
 * The same for PLUS and MINUS.
 */

#include "includes.h"
#include <signal.h>
#if	UNIX || CYGWIN
#include <sys/ioctl.h>
#include <termios.h>
#endif

static void usage();

#if	CYGWIN
char	*
dirname1(cp)
char	*cp;
{
	int	i;

	i = strlen(cp);
	while (i >= 0 && cp[i] != '\\' && cp[i] != '/')
		i--;
	if (i < 0)
		return(".");
	cp[i] = '\0';
	return(cp);
}
#endif

int
main(argc, argv)
int	argc;
char	**argv;
{
	extern char	*optarg;
	extern int	optind;

	int		i;
	char		*cp;
	double		numerator, denominator;
	double		multiplier;
	int		coption;
#if	UNIX || CYGWIN
	struct winsize	ws;
#endif

#if	UNIX
	prog_name = strdup(basename(argv[0]));
#endif
#if	CYGWIN
	dir_path = strdup(dirname1(argv[0]));
#endif
	piping_in_flag = !isatty(0);
	init_gvars();
	gfp = stdout;
	setbuf(stdout, NULL);		/* make standard output unbuffered */

#if	UNIX || CYGWIN
	ws.ws_col = 0;
	ws.ws_row = 0;
	ioctl(1, TIOCGWINSZ, &ws);
	if (ws.ws_col) {
		screen_columns = ws.ws_col;
	}
	if (ws.ws_row) {
		screen_rows = ws.ws_row;
	}
#endif
	coption = false;
	while ((i = getopt(argc, argv, "tchm:")) >= 0) {
		switch (i) {
		case 'c':
			coption = true;
			break;
		case 'h':
			html_flag = true;
			break;
		case 'm':
			multiplier = atof(optarg);
			if (multiplier <= 0.0 || (n_tokens = (int) (multiplier * DEFAULT_N_TOKENS)) <= 0) {
				fprintf(stderr, _("Invalid memory size multiplier specified!\n"));
				usage();
			}
			break;
		case 't':
			test_mode = true;
			break;
		default:
			usage();
		}
	}
	if (test_mode || html_flag) {
		screen_columns = 0;
		screen_rows = 0;
	}
	if (html_flag) {
		printf("<pre>\n");
	}
	if (!test_mode) {
		printf(_("Mathomatic Version %s (www.mathomatic.com)\n"), VERSION);
		printf(_("Copyright (C) 1987-2005 George Gesslein II.\n"));
	}
	if (!init_mem()) {
		fprintf(stderr, _("%s: Not enough memory.\n"), prog_name);
		exit(1);
	}
	if (!test_mode) {
		printf(_("%d equation spaces available.\n"), N_EQUATIONS);
	}
#if	!SECURE
	if (!test_mode && !load_rc()) {
		fprintf(stderr, _("Error loading set options from \"%s\".\n"), RC_FILE);
	}
#endif
	if (test_mode) {
		color_flag = false;
	} else if (coption) {
		color_flag = !color_flag;
	}
	printf("\n");
	default_color();
	if ((i = setjmp(jmp_save)) != 0) {
		clean_up();
		switch (i) {
		case 3:
			break;
		case 13:
			printf(_("Operation abruptly aborted.\n"));
			break;
		case 14:
			error(_("Expression too big!"));
		default:
			printf(_("Operation aborted.\n"));
		}
	} else {
		signal(SIGFPE, fphandler);
		signal(SIGINT, inthandler);
#if	UNIX || CYGWIN
		signal(SIGWINCH, resizehandler);
#endif
#if	TIMEOUT_SECONDS
		signal(SIGALRM, alarmhandler);
		alarm(TIMEOUT_SECONDS);
#endif
		if (!f_to_fraction(0.5, &numerator, &denominator)
		    || !f_to_fraction(1.0/3.0, &numerator, &denominator)) {
			fprintf(stderr, _("Cannot convert floating point values to fractions.\n"));
		}
#if	!SECURE
		for (i = optind; i < argc; i++) {
			if (!read_cmd(argv[i])) {
				exit_program(1);
			}
		}
#endif
	}
	for (;;) {
		snprintf(prompt_str, sizeof(prompt_str), "%d%s", cur_equation + 1, html_flag ? HTML_PROMPT : PROMPT);
		if ((cp = getstring((char *) tlhs, n_tokens * sizeof(token_type))) == NULL)
			break;
		process(cp);
	}
	exit_program(0);
}

static void
usage()
{
	fprintf(stderr, "Usage: %s [ options ] [ input_files ]\n", prog_name);
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "  -c               Toggle color mode.\n");
	fprintf(stderr, "  -h               Enable HTML mode.\n");
	fprintf(stderr, "  -m number        Specify a memory size multiplier.\n");
	fprintf(stderr, "  -t               Used when testing.\n");
	exit(1);
}

void
fphandler(sig)
int	sig;
{
	error(_("Floating point exception."));
	signal(SIGFPE, fphandler);
	longjmp(jmp_save, 2);
}

/*
 * Control-C handler.
 */
void
inthandler(sig)
int	sig;
{
	exit_program(0);
}

#if	TIMEOUT_SECONDS
/*
 * Alarm signal handler.
 */
void
alarmhandler(sig)
int	sig;
{
	printf(_("timeout\n"));
	exit_program(0);
}
#endif

#if	UNIX || CYGWIN
/*
 * Window resize handler.
 */
void
resizehandler(sig)
int	sig;
{
	struct winsize	ws;

	ws.ws_col = 0;
	ws.ws_row = 0;
	ioctl(1, TIOCGWINSZ, &ws);
	if (ws.ws_col && screen_columns) {
		screen_columns = ws.ws_col;
	}
	if (ws.ws_row && screen_rows) {
		screen_rows = ws.ws_row;
	}
}
#endif

/*
 * Exit this program and return to the Operating System.
 */
void
exit_program(exit_value)
int	exit_value;
{
	printf("\n");
	reset_attr();
	printf(_("Thank you for using Mathomatic!\n"));
	if (html_flag) {
		printf("</pre>\n");
	}
	exit(exit_value);
}
