/*
 * Floating point complex number library and routines.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "includes.h"

/*
 * Zero out very small real or imaginary parts.
 */
complex_fixup(ap)
complexs	*ap;
{
	if (fabs(ap->re * epsilon) > fabs(ap->im)) {
		ap->im = 0.0;
	} else if (fabs(ap->im * epsilon) > fabs(ap->re)) {
		ap->re = 0.0;
	}
}

#if	false
/*
 * Add two complex numbers and return result.
 */
complexs
complex_add(a, b)
complexs	a, b;
{
	a.re += b.re;
	a.im += b.im;
	return(a);
}

/*
 * Subtract two complex numbers and return result.
 */
complexs
complex_sub(a, b)
complexs	a, b;
{
	a.re -= b.re;
	a.im -= b.im;
	return(a);
}
#endif

/*
 * Multiply two complex numbers and return result.
 */
complexs
complex_mult(a, b)
complexs	a;
complexs	b;
{
	complexs	r;

	r.re = a.re * b.re - a.im * b.im;
	r.im = a.re * b.im + a.im * b.re;
	return(r);
}

/*
 * Divide complex "a" by complex "b" and return result.
 */
complexs
complex_div(a, b)
complexs	a;	/* dividend */
complexs	b;	/* divisor */
{
	complexs	r, num;
	double		denom;

	b.im = -b.im;
	num = complex_mult(a, b);
	denom = b.re * b.re + b.im * b.im;
	r.re = num.re / denom;
	r.im = num.im / denom;
	return r;
}

/*
 * Take the natural logarithm of a complex number and return result.
 */
complexs
complex_log(a)
complexs	a;
{
	complexs	r;

	errno = 0;
	r.re = log(a.re * a.re + a.im * a.im) / 2.0;
	r.im = atan2(a.im, a.re);
	check_err();
	return(r);
}

/*
 * Raise the natural number (e) to the power of a complex number
 * and return result.
 */
complexs
complex_exp(a)
complexs	a;
{
	complexs	r;
	double		m;

	errno = 0;
	m = exp(a.re);
	r.re = m * cos(a.im);
	r.im = m * sin(a.im);
	check_err();
	return(r);
}

/*
 * Raise a complex number to the power of a complex number and return result.
 */
complexs
complex_pow(a, b)
complexs	a;
complexs	b;
{
	complexs	r;

	r = complex_log(a);
	r = complex_mult(r, b);
	r = complex_exp(r);
	complex_fixup(&r);
	return(r);
}

/*
 * The roots command.
 */
int
roots_cmd(cp)
char	*cp;
{
#define	MAX_ROOT	1000.0

	complexs	c, c2, check;
	double		d, k;
	double		root;
	double		radius, theta;
	double		radius_root;
	char		*cp1, buf[MAX_CMD_LEN];

	if (extra_garbage(cp))
		return false;
	for (;;) {
		my_strlcpy(prompt_str, _("Enter root (positive integer): "), sizeof(prompt_str));
		if ((cp1 = getstring(buf, sizeof(buf))) == NULL)
			return false;
		if (*cp1 == '\0') {
			return false;
		}
		root = strtod(cp1, &cp1);
		if (*cp1 || root <= 0.0 || root >= MAX_ROOT || fmod(root, 1.0) != 0.0) {
			printf(_("Root must be a positive integer less than %.12g\n"), MAX_ROOT);
			continue;
		}
		break;
	}
	do {
		my_strlcpy(prompt_str, _("Enter real part: "), sizeof(prompt_str));
		if ((cp1 = getstring(buf, sizeof(buf))) == NULL)
			return false;
		c.re = strtod(cp1, &cp1);
	} while (*cp1);
	do {
		my_strlcpy(prompt_str, _("Enter imaginary part: "), sizeof(prompt_str));
		if ((cp1 = getstring(buf, sizeof(buf))) == NULL)
			return false;
		c.im = strtod(cp1, &cp1);
	} while (*cp1);
/* convert to polar coordinates */
	errno = 0;
	radius = sqrt(c.re * c.re + c.im * c.im);
	theta = atan2(c.im, c.re);
	radius_root = pow(radius, 1.0 / root);
	check_err();
	if (c.im == 0.0) {
		fprintf(gfp, _("The %.12g roots of %.12g^(1/%.12g) are:\n\n"), root, c.re, root);
	} else {
		fprintf(gfp, _("The %.12g roots of (%.12g %+.12g*i#)^(1/%.12g) are:\n\n"), root, c.re, c.im, root);
	}
	for (k = 0.0; k < root; k += 1.0) {
		c2.re = radius_root * cos((theta + 2.0 * k * PI) / root);
		c2.im = radius_root * sin((theta + 2.0 * k * PI) / root);
		complex_fixup(&c2);
		if (c2.im == 0.0) {
			fprintf(gfp, "%.12g\n", c2.re);
		} else {
			fprintf(gfp, "%.12g %+.12g*i#\n", c2.re, c2.im);
		}
		check = c2;
		for (d = 1.0; d < root; d += 1.0) {
			check = complex_mult(check, c2);
		}
		complex_fixup(&check);
		if (check.im == 0.0) {
			printf(_("Inverse Check: %.12g\n\n"), check.re);
		} else {
			printf(_("Inverse Check: %.12g %+.12g*i#\n\n"), check.re, check.im);
		}
	}
	return true;
}

/*
 * Approximate roots of complex numbers (complex^real
 * and real^complex and complex^complex).
 * Only gives one root, so this is not perfect.
 *
 * Returns true if expression was modified.
 */
int
complex_root_simp(equation, np)
token_type	*equation;
int		*np;
{
	int		i, j;
	int		level;
	int		len;
	complexs	c, p;

	for (i = 1; i < *np; i += 2) {
		if (equation[i].token.operatr != POWER)
			continue;
		level = equation[i].level;
		for (j = i + 2; j < *np && equation[j].level >= level; j += 2)
			;
		len = j - (i + 1);
		if (!parse_complex_exp(&equation[i+1], len, &p))
			continue;
		for (j = i - 1; j >= 0 && equation[j].level >= level; j--)
			;
		j++;
		if (!parse_complex_exp(&equation[j], i - j, &c))
			continue;
		if (c.im == 0.0 && p.im == 0.0)
			continue;
		i += len + 1;
		debug_string(0, _("Warning: Complex number root taken."));
		c = complex_pow(c, p);
		if (*np + 5 - (i - j) > n_tokens) {
			error_huge();
		}
		blt(&equation[j+5], &equation[i], (*np - i) * sizeof(token_type));
		*np += 5 - (i - j);
		equation[j].level = level;
		equation[j].kind = CONSTANT;
		equation[j].token.constant = c.re;
		j++;
		equation[j].level = level;
		equation[j].kind = OPERATOR;
		equation[j].token.operatr = PLUS;
		j++;
		equation[j].level = level + 1;
		equation[j].kind = CONSTANT;
		equation[j].token.constant = c.im;
		j++;
		equation[j].level = level + 1;
		equation[j].kind = OPERATOR;
		equation[j].token.operatr = TIMES;
		j++;
		equation[j].level = level + 1;
		equation[j].kind = VARIABLE;
		equation[j].token.variable = IMAGINARY;
		return true;
	}
	return false;
}

/*
 * Get a constant, if the passed expression is a constant.
 *
 * Return true if successful.
 */
int
get_constant(equation, n, dp)
token_type	*equation;	/* expression */
int		n;		/* length of expression */
double		*dp;		/* pointer to returned double */
{
	if (n != 1)
		return false;
	switch (equation[0].kind) {
	case CONSTANT:
		*dp = equation[0].token.constant;
		return true;
	case VARIABLE:
		if (var_is_const(equation[0].token.variable, dp)) {
			return true;
		}
	}
	return false;
}

/*
 * Parse a complex number expression.
 *
 * If successful return true.
 */
int
parse_complex_exp(equation, n, cp)
token_type	*equation;	/* expression */
int		n;		/* length of expression */
complexs	*cp;		/* pointer to returned complex number */
{
	int		j;
	int		imag_cnt, plus_cnt, times_cnt;
	complexs	c;
	int		level2;

	if (get_constant(equation, n, &c.re)) {
		c.im = 0.0;
		*cp = c;
		return true;
	}
	c.re = 0.0;
	c.im = 1.0;
	imag_cnt = 0;
	plus_cnt = 0;
	times_cnt = 0;
	for (j = n - 1; j >= 0; j--) {
		switch (equation[j].kind) {
		case CONSTANT:
			break;
		case VARIABLE:
			if (equation[j].token.variable != IMAGINARY)
				return false;
			imag_cnt++;
			break;
		case OPERATOR:
			switch (equation[j].token.operatr) {
			case TIMES:
				if (++times_cnt > 1)
					return false;
				level2 = equation[j].level;
				if (equation[j-1].level != level2
				    || equation[j+1].level != level2)
					return false;
				if (equation[j-1].kind == VARIABLE
				    && equation[j-1].token.variable == IMAGINARY) {
					if (equation[j+1].kind != CONSTANT)
						return false;
					c.im = equation[j+1].token.constant;
					continue;
				}
				if (equation[j+1].kind == VARIABLE
				    && equation[j+1].token.variable == IMAGINARY) {
					if (equation[j-1].kind != CONSTANT)
						return false;
					c.im = equation[j-1].token.constant;
					continue;
				}
				return false;
			case MINUS:
				if (imag_cnt) {
					c.im = -c.im;
				}
			case PLUS:
				if (++plus_cnt > 1)
					return false;
				level2 = equation[j].level;
				if (equation[j-1].level == level2
				    && equation[j-1].kind == CONSTANT) {
					c.re = equation[j-1].token.constant;
					continue;
				}
				if (equation[j+1].level == level2
				    && equation[j+1].kind == CONSTANT) {
					c.re = equation[j+1].token.constant;
					if (equation[j].token.operatr == MINUS)
						c.re = -c.re;
					continue;
				}
			}
		default:
			return false;
		}
	}
	if (imag_cnt != 1)
		return false;
	*cp = c;
	return true;
}
