/* Copyright (c) 2002,2003 Sam Trenholme
 *
 * TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * This software is provided 'as is' with no guarantees of correctness or
 * fitness for purpose.
 */

/* This is the core DNS server */

/* Language specific labels */
#include "MaraDNS_locale.h"

/* Include stuff needed to be a UDP server */

#include "../libs/MaraHash.h"
#include "../MaraDns.h"
#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#ifdef __FreeBSD__
#include <sys/time.h>
#endif
#include <sys/types.h>
#ifndef DARWIN
#include <sys/resource.h>
#endif
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "../dns/functions_dns.h"
#include "../parse/functions_parse.h"
#include "functions_server.h"

extern int js_readuint16();
extern int make_ip_acl();
extern int init_cache();
extern int init_spammers();
extern int init_crypto();
extern int setgroups();
extern int hname_translate();
extern int how_many_threads();
extern int js_tell_memory_allocated();
extern int cache_elements();
extern int init_rng();
extern int set_min_ttl();
extern int debug_show_ip();
extern int init_rlog_level();
extern int decomp_init();
/* Timestamp junk */
extern int set_timestamp();
extern int show_timestamp();

/* Our global variables */
mhash *bighash;
int log_level = 1; /* 0: No messages except for fatal errors which stop
                         MaraDNS, 1: Startup and shutdown messages,
                  2: Log queries that generate errors, 3: Log all queries */
int no_fingerprint = 0; /* 0: Have some MaraDNS-specific features, such as
                              DDIP expansion and a special query that
                              tells you the version number of the server.
                           1: Attempts to have as generic an interface as
                              possible. */
int rrany_set = 3; /* (Determined with default_rrany_set in mararc file)
                      3: A request for RR_ANY will only return A and MX
                         records for a given node.
                      15: A request for RR_ANY will return A, MX, NS, and
                         SOA records */
int max_ar_chain = 1; /* Maximum number of records we show of a chain
                         of A records in the additional section of an
                         answer */
int max_chain = 8; /* Total maximum number of records in any chain of
                      records */
int max_total = 20; /* Total maximum number of records we will show */
int debug_delay = 0; /* Delay before sending a reply; only used for
                        debugging MaraDNS */

/* Some variables used to assist in the housekeeping making sure we
   do not display a given RR in the Additional records section twice */
rr *seenlist[256];
int seenlist_where = 0;

int total_count = 0; /* This has to be global to handle udpany's use of it */

int debug_msg_level = 1; /* The level of debug messages to allow */

/* A list of who is and who is not allowed to make recursive DNS queries */
ipv4pair recurse_acl[512];
/* A list of the IP addresses we bind MaraDNS to (the netmask portion is
   ignored) */
ipv4pair bind_addresses[512];

/* Signal handler for SIGPIPE, so we don't terminate */
void handle_sigpipe() {
    if(log_level > 1)
        printf("%s%s",L_CAUGHT_PIPE,L_N); /* "Caught SIGPIPE" */
    return;
    }

int got_hup_signal = 0;

/* Signal handler for HUP signal */
void handle_hup() {
    got_hup_signal = 1;
    return;
    }

/* Signal handler for other signals */
void handle_signal() {
    if(log_level > 1)
        printf("%s%s",L_CAUGHT_SIG,L_N); /* "Caught Signal" */
    return;
    }

#ifdef DEBUG
/* Signal handler which tells us about unfreed memory before exiting */
void display_unfreed() {
    if(log_level > 0)
        js_show_leaks();
    exit(64);
    }
#endif /* DEBUG */

/* Print out log messages
   Input: Null-terminated string with the message to log
   Output: JS_SUCCESS on success, JS_ERROR on error
*/

int mlog(char *logmessage) {

    if(log_level == 0)
        return JS_SUCCESS;

    if(logmessage == 0)
        return JS_ERROR;
    show_timestamp();
    printf("%s%s%s",L_LOG,logmessage,L_N);
          /* "Log: ", logmessage, "\n" */

    /* Unbuffered output */
    fflush(stdout);

    return JS_SUCCESS;
    }

/* Print out log messages of js_string messages
   Input: js_string object to log
   Output: JS_SUCCESS on success, JS_ERROR on error
*/

int jlog(js_string *logmessage) {

    int ret;

    if(log_level == 0)
        return JS_SUCCESS;

    printf("%s",L_LOG); /* "Log: " */
    ret = show_esc_stdout(logmessage);
    printf("%s",L_N); /* "\n" */

    /* Unbuffered output */
    fflush(stdout);

    return ret;
    }

/* Print out log message of a Null-terminated string followed by a js_string
   Input: Null-terminated string, js_string
   Output: JS_SUCCESS on success, JS_ERROR on error */

int zjlog(char *left, js_string *right) {
    int ret;
    if(log_level == 0)
        return JS_SUCCESS;
    if(left == 0)
        return JS_ERROR;
    printf("%s%s",L_LOG,left); /* "Log: ", left */
    ret = show_esc_stdout(right);
    printf("%s",L_N); /* "\n" */

    /* Unbuffered output */
    fflush(stdout);

    return ret;
    }

/* Handler to handle fatal errors.
   Input: Pointer to null-terminalted string with fatal error
   Output: MaraDNS exits
*/

void harderror(char *why) {
    printf("%s%s%s",L_FATAL,why,L_N); /* "Fatal Error: ", why, "\n" */

    /* Unbuffered output */
    fflush(stdout);

    exit(3);
    }

/* Handler to handle system fatal errors.
   Input: Pointer to null-terminalted string with fatal error
   Output: MaraDNS exits
*/

void sys_harderror(char *why) {
    printf("%s%s%s",L_FATAL,why,L_N); /* "Fatal Error: ", why, "\n" */
    printf("%s: %s%s",L_SYSERROR,strerror(errno),L_N);
    /* This outputs to stderr, which duende can not catch (I gave up
       trying to catch stderr messages after trying for two days) */
    /*perror(L_SYSERROR);*/ /* "System said: " */

    /* Unbuffered output */
    fflush(stdout);

    exit(3);
    }

/* Calculate the TTL age given the expire time (absolute time) and
   the ttl (relative time)
   Input: Exprire time, TTL in question
   Output: The TTL we should give, taking TTL aging in to account
 */

uint32 determine_ttl(time_t expire,uint32 ttl) {
    time_t now;

    if(expire == 0) {
        return ttl;
        }
    now = time(0);

    /* If this is a record being resurrected from the expired records, we
       make the TTL 29 seconds */
    if(expire < (now - 10)) { 
        return 29;
        }
 
    if(expire - now > 30) {
        return expire - now;
        }
    return 30;
    }

/* Return a packet indicating that there was an error in the received
   packet
   input: socket number,
          a js_string object that we get the data from the first two
          bytes from, a sockaddr of who to send the error to,
          the question the error-generating query asked, the error
          to give them in the RCODE part of the header,
          the reason for the error, the minimim log_level to log this
          error (with reason) with
   output: JS_ERROR on error, JS_SUCCESS on success
*/

int udperror(int sock,js_string *raw, struct sockaddr_in *from,
             js_string *question, int error,
             char *why,int min_log_level) {

    q_header header;
    js_string *reply;
    int len_inet = sizeof(struct sockaddr);

    if(log_level >= min_log_level) {
        show_timestamp();
        zjlog(L_BAD_QUERY,raw); /* "Bad query received: " */
        }
    if(log_level >= 2) /* Tell them why */
        mlog(why);

    if(raw->unit_count < 2 || raw->max_count < 3)
        return JS_ERROR;

    if((reply = js_create(96,1)) == 0)
        return JS_ERROR;

    /* Fill out the header */
    header.id = ((*(raw->string) & 0xff) << 8) | (*(raw->string + 1) & 0xff);
    header.qr = 1;
    header.opcode = 0;
    header.aa = 0; /* Errors are never authoritative (unless they are
                      NXDOMAINS, which this is not) */
    header.tc = 0;
    header.rd = 0;
    header.ra = 0;
    header.z = 0;
    header.rcode = error;
    header.qdcount = 0;
    header.ancount = 0;
    header.nscount = 0;
    header.arcount = 0;

    /* Make that raw UDP data */
    if(make_hdr(&header,reply) == JS_ERROR) {
        js_destroy(reply);
        return JS_ERROR;
    }

    /* Append the question, if there is one */
    if(question != 0) {
       if(js_append(question,reply) != JS_SUCCESS) {
           js_destroy(reply);
           return JS_ERROR;
           }
       if(js_adduint16(reply,1) != JS_SUCCESS) {
           js_destroy(reply);
           return JS_ERROR;
           }
       }

    /* Send them the reply */
    sendto(sock,reply->string,reply->unit_count,0,
            (struct sockaddr *)from,len_inet);
    js_destroy(reply);
    return JS_SUCCESS;

    }

/* If we successfully found a record, spit out that record on the
   udp packet.
   Input: Where a pointer to the rr in question is, the id of the
          query they sent us, the socket the
          UDP bind is on, the sockaddr of the client who sent us the message,
          a js_string containing the query (dname + type), whether to show
          an A record along with a CNAME record
   Output: JS_ERROR on error, JS_SUCCESS on success
*/

int udpsuccess(rr *where, int id, int sock, struct sockaddr_in *client,
               js_string *query, void **rotate_point, int show_cname_a) {
    js_string *most, *ar; /* Most of the data then the additional records */

    uint16 first_rr_type;
    int in_ns = 0;
    int length_save;
    int len_inet = sizeof(struct sockaddr);
    rr *ipwhere;
    /* The following are used for round robin rotation */
    rr *rotate_1st = 0, *rotate_2nd = 0, *rotate_last = 0;
    /* Counters in ensure that we don't give out more than the maximum
       number of A, AN (A), chain, or total records */
    int a_count = 0, an_count = 0;
    fila *zap_point;
    /* These two variables are added to handle PTR records */
    int seen_ptr_record = 0;
    rr *top = where;

    q_header header;

    /* Initialize the total count */
    total_count = 0;
    /* Initialize the js_string objects */
    if((most = js_create(1024,1)) == 0)
        return JS_ERROR;
    if((ar = js_create(1024,1)) == 0) {
        js_destroy(most);
        return JS_ERROR;
        }

    /* Make the header a placeholder for now */
    init_header(&header);
    header.id = id;
    if(make_hdr(&header,most) == JS_ERROR)
        goto giveerror;

    /* Sanity check */
    if(where == 0) {
        goto giveerror;
        }
    if(where->query == 0) {
        goto giveerror;
        }
    if(js_has_sanity(where->query) == JS_ERROR) {
        goto giveerror;
        }
    if(where->data == 0) {
        goto giveerror;
        }
    if(js_has_sanity(where->data) == JS_ERROR) {
        goto giveerror;
        }
    if(js_has_sanity(query) == JS_ERROR) {
        goto giveerror;
        }
    first_rr_type = get_rtype(query);

    /* With the cache, this may be a rtype of RR_ANY.  We need to handle
       this special case */

    /* We have to add this header here--authoritative depends on the
       authorative status of the first record we find */
    header.aa = where->authoritative;

    /* The data must be between 0 and 65535 bytes in length (16-bit
       unsigned value) */
    if(where->data->unit_count < 0 || where->data->unit_count > 65535) {
        goto giveerror;
        }

    /* Append the question to the answer */
    if(js_append(query,most) == JS_ERROR) {
        goto giveerror;
        }

    /* Append the class (in) to the answer */
    if(js_adduint16(most,1) == JS_ERROR) {
        goto giveerror;
        }

    /* We will increment the ancount, nscount, an arcount, starting at 0 */
    header.ancount = 0;
    header.nscount = 0;
    header.arcount = 0;

    /* Initialize some temporary pointers used for round robin rotation */
    rotate_1st = where;
    rotate_2nd = where->next;
    /* We do not round robin if there is but a single record */
    if(rotate_2nd != 0 && first_rr_type != RR_NS &&
       rotate_2nd->rr_type == RR_NS)
        rotate_2nd = 0;

    /* OK, we now add the answers */
    while(where != 0) {
        /* Increment the number of answers -or- ns records */
        if(first_rr_type != RR_NS && where->rr_type == RR_NS && in_ns == 0) {
            /* Due to the data structure MaraDNS currently uses, the behavior
               is buggy if we round-robin rotate data when we allow more than
               one additional record to be create per answer/authoritative
               record.  */
            if(rotate_2nd != 0 && max_ar_chain == 1) {
                /* If it makes sense to do a round-robin rotation, do so */
                rotate_1st->next = where;
                rotate_last->next = rotate_1st;
                *rotate_point = rotate_2nd;
                rotate_2nd = 0; /* Make sure we can not rotate again */
                }
            in_ns = 1;
            a_count = 0; /* The NS chain is different than the AN
                            chain of answers: If we only allow eight
                            answers in a chain, we can still have 16
                            answers: 8 records in the answer section then
                            8 records in the authority section */
            }
        if(a_count < max_chain && total_count < max_total) {
            a_count++;
            total_count++;
            if(!in_ns)
                header.ancount++;
            else
                header.nscount++;
            /* Append the name for this answer to the answer */
            if(js_append(where->query,most) == JS_ERROR)
                goto giveerror;
            /* Append the class (in) to the answer */
            if(js_adduint16(most,1) == JS_ERROR)
                goto giveerror;
            /* Append the ttl to the answer */
            if(js_adduint32(most,determine_ttl(where->expire,where->ttl))
               == JS_ERROR)
                goto giveerror;
            /* Add the rdlength to the answer */
            if(js_adduint16(most,where->data->unit_count) == JS_ERROR)
                goto giveerror;
            /* Add the record itself to the answer */
            if(js_append(where->data,most) == JS_ERROR)
                goto giveerror;
            /* If there is an IP, and this is *not* a CNAME record,
               append the IP of the answer to the AR section */
            if(where->ip != 0 && where->rr_type != RR_CNAME) {
                /* Reset the number of an records we have seen */
                an_count = 0;
                ipwhere = where->ip;
                while(ipwhere != 0 && ipwhere->rr_type != RR_NS) {
                    /* We only show a given additional record once */
                    if(ipwhere->seen == 1) { /* If we have displayed this RR
                                                already */
                        /* Go to the next link in the linked list */
                        ipwhere = ipwhere->next;
                        continue;
                        }
                    /* Stop showing records if we have exceeded our limit */
                    if(an_count >= max_ar_chain || total_count >= max_total)
                        break;
                    an_count++;
                    total_count++;
                    /* Increment the number of additional records */
                    header.arcount++;
                    /* Append the name for this answer to the ip */
                    if(js_append(ipwhere->query,ar) == JS_ERROR)
                        goto giveerror;
                    /* Append the class (in) to the ip */
                    if(js_adduint16(ar,1) == JS_ERROR)
                        goto giveerror;
                    /* Append the TTL to the ip */
                    if(js_adduint32(ar,
                      determine_ttl(ipwhere->expire,ipwhere->ttl)) == JS_ERROR)
                        goto giveerror;
                    /* Add the rdlength to the ip */
                    if(js_adduint16(ar,ipwhere->data->unit_count) == JS_ERROR)
                        goto giveerror;
                    /* Add the record itself to the ip */
                    if(js_append(ipwhere->data,ar) == JS_ERROR)
                        goto giveerror;
                    /* Mark that we have seen this record already */
                    if(seenlist_where < 250) {
                        ipwhere->seen = 1;
                        seenlist[seenlist_where] = ipwhere;
                        seenlist_where++;
                        }
                    /* Go to the next link in the linked list */
                    ipwhere = ipwhere->next;
                    }
                }
            /* If there is an IP, and this is a CNAME record, and
               show_cname_a is set to one (argument to this function)
               append the IP in question to the answer section */
            if(where->ip != 0 && where->rr_type == RR_CNAME
               && show_cname_a == RR_A) {
                /* Reset the number of an records we have seen */
                an_count = 0;
                ipwhere = where->ip;
                while(ipwhere != 0 && ipwhere->rr_type != RR_NS) {
                    /* We only show a given additional record once */
                    if(ipwhere->seen == 1) { /* If we have displayed this RR
                                                already */
                        /* Go to the next link in the linked list */
                        ipwhere = ipwhere->next;
                        continue;
                        }
                    /* If the IP in question is 255.255.255.255, we do
                       not show the data in question */
                    if(ipwhere->rr_type == RR_A &&
                       ipwhere->data->unit_count == 4 &&
                       *(ipwhere->data->string) == 0xff &&
                       *(ipwhere->data->string + 1) == 0xff &&
                       *(ipwhere->data->string + 2) == 0xff &&
                       *(ipwhere->data->string + 3) == 0xff) {
                        ipwhere = ipwhere->next;
                        continue;
                        }
                    /* Stop showing records if we have exceeded our limit */
                    if(an_count >= max_ar_chain || total_count >= max_total)
                        break;
                    an_count++;
                    total_count++;
                    /* Increment the number of answer records */
                    header.ancount++;
                    /* Append the name for this answer to the ip */
                    if(js_append(ipwhere->query,most) == JS_ERROR)
                        goto giveerror;
                    /* Append the class (in) to the ip */
                    if(js_adduint16(most,1) == JS_ERROR)
                        goto giveerror;
                    /* Append the TTL to the ip */
                    if(js_adduint32(most,
                      determine_ttl(ipwhere->expire,ipwhere->ttl)) == JS_ERROR)
                        goto giveerror;
                    /* Add the rdlength to the ip */
                    if(js_adduint16(most,ipwhere->data->unit_count)
                       == JS_ERROR)
                        goto giveerror;
                    /* Add the record itself to the ip */
                    if(js_append(ipwhere->data,most) == JS_ERROR)
                        goto giveerror;
                    /* Mark that we have seen this record already */
                    if(seenlist_where < 250) {
                        ipwhere->seen = 1;
                        seenlist[seenlist_where] = ipwhere;
                        seenlist_where++;
                        }
                    /* Go to the next link in the linked list */
                    ipwhere = ipwhere->next;
                    }
                }
            /* If there is an PTR, and this is a CNAME record, and
               show_cname_a is set to one (argument to this function)
               append the IP in question to the answer section */
            else if(top->ptr != 0 && top->rr_type == RR_CNAME
               && show_cname_a == RR_PTR && seen_ptr_record == 0) {
                    /* Mark that we have seen this record already */
                    seen_ptr_record = 1;
                    /* Increment the total number of answers seen */
                    total_count++;
                    /* Increment the number of answer records */
                    header.ancount++;
                    /* Append the name for this answer to the ip */
                    if(js_append(top->data,most) == JS_ERROR)
                        goto giveerror;
                    /* Append the type for this query */
                    if(js_adduint16(most,RR_PTR) == JS_ERROR)
                        goto giveerror;
                    /* Append the class (in) to the ip */
                    if(js_adduint16(most,1) == JS_ERROR)
                        goto giveerror;
                    /* Append the TTL to the ip */
                    if(js_adduint32(most,
                      determine_ttl(top->expire,top->ttl)) == JS_ERROR)
                        goto giveerror;
                    /* Add the rdlength to the ip */
                    if(js_adduint16(most,top->ptr->unit_count)
                       == JS_ERROR)
                        goto giveerror;
                    /* Add the record itself to the ip */
                    if(js_append(top->ptr,most) == JS_ERROR)
                        goto giveerror;
               }
            }
        /* Go on to the next record in the linked list */
        rotate_last = where;
        where = where->next;
        /* If it makes sense to do a round-robin rotation, do so */
        if(where == 0 && rotate_2nd != 0 && max_ar_chain == 1) {
            /* For records in the cache, we need to make sure that
               the custodian properly points to the first record
               in the chain or we will leak memory */
            if(rotate_1st->zap != 0) {
                zap_point = rotate_1st->zap;
                rotate_1st->zap = 0;
                rotate_2nd->zap = zap_point;
                zap_point->record = rotate_2nd;
                }
            rotate_1st->next = 0;
            rotate_last->next = rotate_1st;
            *rotate_point = rotate_2nd;
            rotate_2nd = 0; /* Make sure we can not rotate again */
            }
        }

    /* Customize the header */
    /* header.id already set */
    header.qr = 1;
    header.opcode = 0;
    header.tc = 0;
    header.rd = 0;
    header.ra = 0;
    header.z = 0;
    header.rcode = 0; /* No error */
    header.qdcount = 1;

    /* OBhack: Tack on the header at the beginning without molesting the
       rest of the string */
    length_save = most->unit_count;
    make_hdr(&header,most);
    most->unit_count = length_save;

    /* Add the ar records to the end */
    if(js_append(ar,most) == JS_ERROR) {
        goto giveerror;
        }

    /* Compress "most" and place the compressed data in "ar" */
    if(compress_data(most,ar) == JS_ERROR) {
        /* This is a bit of a kludge to work around the news.com.com
           problem for the 1.0.00 release */
        show_timestamp();
        printf("%s",L_COMPRESS_ERROR);
        /* Compress error */
        show_esc_stdout(most);
        printf("\n");
        if(js_copy(most,ar) == JS_ERROR) {
            js_destroy(ar);
            udperror(sock,most,client,0,SERVER_FAIL,"Compress failure",2);
            js_destroy(most);
            return JS_ERROR;
            }
        }

    /* Check to make sure the data fits in under 512 bytes */
    if(ar->unit_count > 512) {
        /* We handle truncation by truncating everything except the
           12-byte header */
        header.tc = 1;
        make_hdr(&header,ar);
        }

    /* Success! Put out the good data */
    sendto(sock,ar->string,ar->unit_count,0,
            (struct sockaddr *)client,len_inet);

    js_destroy(most);
    js_destroy(ar);

    /* Clean up the seenlist_where list */
    while(seenlist_where > 0) {
        seenlist_where--;
        if(seenlist[seenlist_where] != 0)
            (seenlist[seenlist_where])->seen = 0;
        }

    return JS_SUCCESS;

    /* We use gotos to make up for C's lack of error trapping */
    giveerror:
        js_destroy(ar);
        udperror(sock,most,client,0,SERVER_FAIL,"giveerror in udpsuccess",2);
        js_destroy(most);

        /* Clean up the seenlist_where list */
        while(seenlist_where > 0) {
            seenlist_where--;
            if(seenlist[seenlist_where] != 0)
                (seenlist[seenlist_where])->seen = 0;
            }
        return JS_ERROR;

    }

/* If we successfully found a record, add the answer to the A record,
   if applicable, add the NS data too, and add the appropriate
   additional records.
   Input: Where a pointer to the rr in question is, a pointer to the
          string where we add answers, pointer to ns data string, a pointer
          to the string where we add additional records, a pointer to the
          number containing the number of answers, a pointer to the
          number containing the number of authority records, a pointer to
          the number containing the number of additional records, whether
          to add records to the authority (ns) section.
   Output: JS_ERROR on error, JS_SUCCESS on success
*/

int add_answer(rr *where,js_string *most, js_string *ns, js_string *ar,
               uint16 *ancount, uint16 *nscount, uint16 *arcount,
               int add_ns, void **rotate_point) {

    uint16 first_rr_type;
    rr *ipwhere;
    int in_ns = 0;
    int a_count = 0; /* Number of records displayed for a given chain */
    int ar_count = 0; /* Number of records displayed for a chain in the AR
                         section */
    int rotate_done = 0; /* We only rotate data once per call
                            to this function */
    /* The following are used for round robin rotation */
    rr *rotate_1st = 0, *rotate_2nd = 0, *rotate_last = 0;

    /* Sanity check */
    if(where == 0) {
        goto giveerror;
        }
    if(where->query == 0) {
        goto giveerror;
        }
    if(js_has_sanity(where->query) == JS_ERROR) {
        goto giveerror;
        }
    if(where->data == 0) {
        goto giveerror;
        }
    if(js_has_sanity(where->data) == JS_ERROR) {
        goto giveerror;
        }
    first_rr_type = get_rtype(where->query);

    /* The data must be between 0 and 65535 bytes in length (16-bit
       unsigned value) */
    if(where->data->unit_count < 0 || where->data->unit_count > 65535) {
        goto giveerror;
        }

    /* Initialize some temporary pointers used for round robin rotation */
    rotate_1st = where;
    rotate_2nd = where->next;
    /* We do not round robin if there is but a single record */
    if(rotate_2nd != 0 && first_rr_type != RR_NS &&
       rotate_2nd->rr_type == RR_NS)
        rotate_2nd = 0;

    /* OK, we now add the answers */
    while(where != 0) {
        /* Increment the number of answers -or- ns records */
        if(first_rr_type != RR_NS && where->rr_type == RR_NS) {

            /* Due to the data structure MaraDNS currently uses, the behavior
               is buggy if we round-robin rotate data when we allow more than
               one additional record to be create per answer/authoritative
               record.  */
            if(rotate_2nd != 0 && max_ar_chain == 1 && rotate_done == 0) {
                rotate_done = 1;
                /* If it makes sense to do a round-robin rotation */
                rotate_1st->next = where;
                rotate_last->next = rotate_1st;
                *rotate_point = rotate_2nd;
                }

            a_count = 0; /* The NS chain is different than the AN
                            chain of answers: If we only allow eight
                            answers in a chain, we can still have 16
                            answers: 8 records in the answer section then
                            8 records in the authority section */
            if(add_ns == 1)
                in_ns = 1;
            else
                return JS_SUCCESS;
            }
        if(!in_ns && a_count < max_chain && total_count < max_total) {
            a_count++; /* Counter so we don't exceed the maximum number
                          of records allowed to be seen in a chain */
            total_count++;
            (*ancount)++; /* This goes in the header of the reply */
            /* Append the name for this answer to the answer */
            if(js_append(where->query,most) == JS_ERROR) {
                goto giveerror;
                }
            /* Append the class (in) to the answer */
            if(js_adduint16(most,1) == JS_ERROR) {
                goto giveerror;
                }
            /* Append the ttl to the answer */
            if(js_adduint32(most,determine_ttl(where->expire,where->ttl))
                == JS_ERROR) {
                goto giveerror;
                }
            /* Add the rdlength to the answer */
            if(js_adduint16(most,where->data->unit_count) == JS_ERROR) {
                goto giveerror;
                }
            /* Add the record itself to the answer */
            if(js_append(where->data,most) == JS_ERROR) {
                goto giveerror;
                }
            }
        else if(a_count < max_chain && total_count < max_total) {
            a_count++; total_count++; /* The counters that make sure we do
                                         not have more than, say, eight
                                         records for a given answer */
            /* Append the name for this answer to the answer */
            if(js_append(where->query,ns) == JS_ERROR) {
                goto giveerror;
                }
            /* Append the class (in) to the answer */
            if(js_adduint16(ns,1) == JS_ERROR) {
                goto giveerror;
                }
            /* Append the ttl to the answer */
            if(js_adduint32(ns,determine_ttl(where->expire,where->ttl))
                == JS_ERROR) {
                goto giveerror;
                }
            /* Add the rdlength to the answer */
            if(js_adduint16(ns,where->data->unit_count) == JS_ERROR) {
                goto giveerror;
                }
            /* Add the record itself to the answer */
            if(js_append(where->data,ns) == JS_ERROR) {
                goto giveerror;
                }
            (*nscount)++;
            }
        /* If there is an IP, and this is *not* a CNAME record,
           append the IP of the answer to the AR section */
        if(where->ip != 0 && where->rr_type != RR_CNAME) {
            ipwhere = where->ip;
            ar_count = 0; /* Reset for each instance of showing AR
                             records */
            while(ipwhere != 0 && ipwhere->rr_type != RR_NS &&
                  ar_count < max_ar_chain && total_count < max_total) {
                ar_count++; /* Counter so we don't exceed maximum number
                               of AN records allowed to be displayed */
                total_count++; /* Similar to ar_count */
                /* We only show a given additional record once */
                if(ipwhere->seen == 1) { /* If we have displayed this RR
                                            already */
                    /* Go to the next link in the linked list */
                    ipwhere = ipwhere->next;
                    continue;
                    }
                /* Increment the number of additional records */
                (*arcount)++;
                /* Append the name for this answer to the ip */
                if(js_append(ipwhere->query,ar) == JS_ERROR) {
                    goto giveerror;
                    }
                /* Append the class (in) to the ip */
                if(js_adduint16(ar,1) == JS_ERROR) {
                    goto giveerror;
                    }
                /* Append the TTL to the ip */
                if(js_adduint32(ar,determine_ttl(ipwhere->expire,ipwhere->ttl))
                   == JS_ERROR) {
                    goto giveerror;
                    }
                /* Add the rdlength to the ip */
                if(js_adduint16(ar,ipwhere->data->unit_count) == JS_ERROR) {
                    goto giveerror;
                    }
                /* Add the record itself to the ip */
                if(js_append(ipwhere->data,ar) == JS_ERROR) {
                    goto giveerror;
                    }
                /* Mark that we have seen this record already */
                if(seenlist_where < 250) {
                    ipwhere->seen = 1;
                    seenlist[seenlist_where] = ipwhere;
                    seenlist_where++;
                    }
                ipwhere = ipwhere->next;
                }
            }

        /* We do not chase CNAME records in an "RR_ALL" query */

        /* Make a note of this node for round-robin rotation purposes */
        rotate_last = where;
        /* Go on to the next record in the linked list */
        where = where->next;
        }

    return JS_SUCCESS;

    /* We use gotos to make up for C's lack of error trapping */
    giveerror:
        return JS_ERROR;

    }

/* If they asked for a RR_ANY record, do this:
   1. See if MX records exist.  If so, add it to the answer to give.
   2. See if A records record exist.  If so, add it.
   3. If neither A nor MX exist, look for a CNAME record.  If so,
      return just the CNAME record.
   4. Otherwise, return "query denied".
   Input: ID of the iquery they sent us, socket of the request, a sockaddr
          with their address and port number on it, a js_string containing
          the query (dname + type), the rr_set to return (3: A and MX,
          15: A, MX, SOA, and NS), the hash to look for data in
   Output: JS_ERROR on error, JS_SUCCESS on success, 0 if no records were
           found
*/

int udpany(int id,int sock,struct sockaddr_in *client, js_string *query,
           int rr_set, mhash *bighash) {
    js_string *most, *ns, *ar; /* The answers, the ns records, the ar records*/

    int length_save;
    int len_inet = sizeof(struct sockaddr);
    int found = 0;
    int authoritative = 1;
    rr *where;
    mhash_e spot_data;

    q_header header;
    /* Initialize the js_string objects */
    if((most = js_create(1024,1)) == 0)
        return JS_ERROR;
    if((ar = js_create(1024,1)) == 0) {
        js_destroy(most);
        return JS_ERROR;
        }
    if((ns = js_create(1024,1)) == 0) {
        js_destroy(most); js_destroy(ar);
        return JS_ERROR;
        }

    /* Initialize the total number of RRs displayed to the DNS client */
    total_count = 0;

    /* Make the header a placeholder for now */
    header.id = id;
    if(make_hdr(&header,most) == JS_ERROR)
        goto giveerror;

    /* Append the question to the answer */
    if(js_append(query,most) == JS_ERROR) {
        goto giveerror;
        }

    /* Append the class (in) to the answer */
    if(js_adduint16(most,1) == JS_ERROR) {
        goto giveerror;
        }

    /* We will increment the ancount, nscount, an arcount, starting at 0 */
    header.ancount = 0;
    header.nscount = 0;
    header.arcount = 0;

    /* Start synthesizing the reply */

    /* Look for an A record with the same name as the query */
    if(change_rtype(query,RR_A) == JS_ERROR)
        goto giveerror;
    spot_data = mhash_get(bighash,query);
    /* If found, add the data to our records */
    if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
        found = 1;
        where = spot_data.value;
        authoritative = where->authoritative;
        if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                   &(header.nscount),&(header.arcount),1,
                   spot_data.point) == JS_ERROR)
            goto giveerror;
        }
    /* Look for MX record with the same name as the query */
    if(change_rtype(query,RR_MX) == JS_ERROR)
        goto giveerror;
    spot_data = mhash_get(bighash,query);
    if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
        if(found == 1) {
            if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
               &(header.nscount),&(header.arcount),0,
               spot_data.point) == JS_ERROR)
                goto giveerror;
            }
        else {
            where = spot_data.value;
            authoritative = where->authoritative;
            if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
               &(header.nscount),&(header.arcount),1,
               spot_data.point) == JS_ERROR)
                goto giveerror;
            }
        found = 1;
        }

    /* We optionally look for NS and SOA on an RR_ANY query */
    if(rr_set == 15) {
      /* Look for NS record with the same name as the query */
      if(change_rtype(query,RR_NS) == JS_ERROR)
        goto giveerror;
      spot_data = mhash_get(bighash,query);
      if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
        if(found == 1) {
            if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
               &(header.nscount),&(header.arcount),0,
               spot_data.point) == JS_ERROR)
                goto giveerror;
            }
        else {
            where = spot_data.value;
            authoritative = where->authoritative;
            if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
               &(header.nscount),&(header.arcount),1,
               spot_data.point) == JS_ERROR)
                goto giveerror;
            }
        found = 1;
        }
      /* Look for SOA record with the same name as the query */
      if(change_rtype(query,RR_SOA) == JS_ERROR)
        goto giveerror;
      spot_data = mhash_get(bighash,query);
      if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
        if(found == 1) {
            if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
               &(header.nscount),&(header.arcount),0,
               spot_data.point) == JS_ERROR)
                goto giveerror;
            }
        else {
            where = spot_data.value;
            authoritative = where->authoritative;
            if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
               &(header.nscount),&(header.arcount),1,
               spot_data.point) == JS_ERROR)
                goto giveerror;
            }
        found = 1;
        }
      }

    /* If not found, look for lower-case version of the same query */
    if(found != 1) {
        found = fold_case(query);
        if(found == JS_ERROR)
            goto giveerror;
        if(found == 1) /* Case folded */ {
            found = 0;
            /* Look for lower case version of A record */
            if(change_rtype(query,RR_A) == JS_ERROR)
                goto giveerror;
            spot_data = mhash_get(bighash,query);
            /* If A record of lower-case found... */
            if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                found = 1;
                where = spot_data.value;
                authoritative = where->authoritative;
                if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                   &(header.nscount),&(header.arcount),1,
                   spot_data.point) == JS_ERROR)
                    goto giveerror;
                }
            if(change_rtype(query,RR_MX) == JS_ERROR)
                goto giveerror;
            spot_data = mhash_get(bighash,query);
            /* If MX record of lower-case found... */
            if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                if(found == 1) {
                    if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                       &(header.nscount),&(header.arcount),0,
                       spot_data.point) == JS_ERROR)
                        goto giveerror;
                    }
                else {
                    where = spot_data.value;
                    authoritative = where->authoritative;
                    if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                       &(header.nscount),&(header.arcount),1,
                       spot_data.point) == JS_ERROR)
                        goto giveerror;
                    }
                }
            /* Optionally look for SOA and NS records */
            if(rr_set == 15) {
              if(change_rtype(query,RR_NS) == JS_ERROR)
                goto giveerror;
              spot_data = mhash_get(bighash,query);
              /* If NS record of lower-case found... */
              if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                if(found == 1) {
                    if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                       &(header.nscount),&(header.arcount),0,
                       spot_data.point) == JS_ERROR)
                        goto giveerror;
                    }
                else {
                    where = spot_data.value;
                    authoritative = where->authoritative;
                    if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                       &(header.nscount),&(header.arcount),1,
                       spot_data.point) == JS_ERROR)
                        goto giveerror;
                    }
                }
              if(change_rtype(query,RR_SOA) == JS_ERROR)
                goto giveerror;
              spot_data = mhash_get(bighash,query);
              /* If SOA record of lower-case found... */
              if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                if(found == 1) {
                    if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                       &(header.nscount),&(header.arcount),0,
                       spot_data.point) == JS_ERROR)
                        goto giveerror;
                    }
                else {
                    where = spot_data.value;
                    authoritative = where->authoritative;
                    if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                       &(header.nscount),&(header.arcount),1,
                       spot_data.point) == JS_ERROR)
                        goto giveerror;
                    }
                }
              }
            }
        }

    /* If nothing found, look for a CNAME record with the same name as
       the query */
    if(found == 0) {
        if(change_rtype(query,RR_CNAME) == JS_ERROR)
            goto giveerror;
        spot_data = mhash_get(bighash,query);
        /* If found, add the data to our records */
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
            found = 1;
            if(add_answer(spot_data.value,most,ns,ar,&(header.ancount),
                         &(header.nscount),&(header.arcount),1,
                         spot_data.point) == JS_ERROR)
                goto giveerror;
            }
        }

    /* If nothing found, look for DDIP notation */
    if(found == 0) {
        if(change_rtype(query,RR_ANY) == JS_ERROR)
            goto giveerror;
        found = ddip_check(id,sock,(struct sockaddr *)client,query);
        if(found == JS_ERROR)
            goto giveerror;
        if(found == JS_SUCCESS) {
            js_destroy(ar);
            js_destroy(ns);
            js_destroy(most);
            return JS_SUCCESS;
            }
        }

    /* Return with exit code of 0 if no answer was found */
    if(header.ancount == 0) {
        js_destroy(ar);
        js_destroy(ns);
        js_destroy(most);
        return 0;
        }

    /* Customize the header */
    /* header.id already set */
    header.qr = 1;
    header.opcode = 0;
    header.tc = 0;
    header.rd = 0;
    header.ra = 0;
    header.aa = authoritative;
    header.z = 0;
    header.rcode = 0; /* No error */
    header.qdcount = 1;

    /* OBhack: Tack on the header at the beginning without molesting the
       rest of the string */
    length_save = most->unit_count;
    make_hdr(&header,most);
    most->unit_count = length_save;

    /* Add the ns and ar records to the end */
    if(js_append(ns,most) == JS_ERROR) {
        goto giveerror;
        }
    if(js_append(ar,most) == JS_ERROR) {
        goto giveerror;
        }

    /* Compress "most" and place the compressed data in "ar" */
    if(compress_data(most,ar) == JS_ERROR) {
        js_destroy(ar);
        js_destroy(ns);
        udperror(sock,most,client,0,SERVER_FAIL,"compression failure",2);
        js_destroy(most);
        return JS_ERROR;
        }

    /* Check to make sure the data fits in under 512 bytes */
    if(ar->unit_count > 512) {
        /* We handle truncation by truncating everything except the
           12-byte header */
        header.tc = 1;
        make_hdr(&header,ar);
        }

    /* Success! Put out the good data */
    sendto(sock,ar->string,ar->unit_count,0,
            (struct sockaddr *)client,len_inet);

    js_destroy(ar);
    js_destroy(ns);
    js_destroy(most);

    /* Clean up the seenlist_where list (list marking which ARs we gave out) */
    while(seenlist_where > 0) {
        --seenlist_where;
        if(seenlist[seenlist_where] != 0)
            (seenlist[seenlist_where])->seen = 0;
        }


    return JS_SUCCESS;

    /* We use gotos to make up for C's lack of error trapping */
    giveerror:
        js_destroy(ar);
        js_destroy(ns);
        udperror(sock,most,client,0,SERVER_FAIL,"giveerror in udpany",2);
        js_destroy(most);

        /* Clean up the seenlist_where list
           (list marking which ARs we gave out) */
        while(seenlist_where > 0) {
            --seenlist_where;
            if(seenlist[seenlist_where] != 0)
                (seenlist[seenlist_where])->seen = 0;
            }

        return JS_ERROR;

    }

/* OK, there are a handful of record types which MaraDNS gives special
   treatment to when a TXT record is asked for the host name in question.
   This routine handles these special domain names.
   Input: ID of the query they sent us, socket of the request, a sockaddr
          with their address and port on it, a js_string containing
          the query (dname + type), The host name that is given special
          treatment (in a pre-hname2rfc1035 format), query type to convert,
          2 strings whose data is dependent on the the query_type to
          convert.
*/

int easter_egg(int id,int sock,struct sockaddr_in *client, js_string *query,
               char *hname, uint16 type, char *opt1, char *opt2) {
    js_string *reply, *hname_js, *data; /* The reply, the query, the answer */
    q_header header;
    int result;

    /* Sanity checks */
    if(js_has_sanity(query) == JS_ERROR)
        return JS_ERROR;
    if(hname == 0 || opt1 == 0)
        return JS_ERROR;

    if((reply = js_create(512,1)) == 0)
        return JS_ERROR;
    if((hname_js = js_create(256,1)) == 0) {
        js_destroy(reply);
        return JS_SUCCESS;
        }
    if((data = js_create(256,1)) == 0) {
        js_destroy(reply); js_destroy(hname_js);
        return JS_SUCCESS;
        }

    /* Make sure that this is the query that they asked for */
    hname_js->encoding = query->encoding;

    if(js_qstr2js(hname_js,hname) == JS_ERROR)
        goto cleanup;

    if(hname_2rfc1035(hname_js) <= 0)
        goto cleanup;

    if(js_adduint16(hname_js,type) == JS_ERROR)
        goto cleanup;

    result = js_issame(hname_js,query);
    if(result == JS_ERROR)
        goto cleanup;

    if(result != 1) {
        js_destroy(reply); js_destroy(hname_js); js_destroy(data);
        return 0;
        }

    /* OK, the hostname matches the "easter egg" name, now we form
       the "easter egg" reply */

    /* Get the data from the options */
    /* If we ever support easter eggs for anything besides TXT
       records, this will become a switch statement */
    if(type != RR_TXT) {
        js_destroy(reply); js_destroy(hname_js); js_destroy(data);
        return 0;
        }

    if(opt2 == 0)
        goto cleanup;

    /* With TXT records, we take the string in opt1, add the string in
       opt2 to the string, and make that the data.  hname_js is used
       as a "throwaway" string */
    if(js_qstr2js(hname_js,"") == JS_ERROR)
        goto cleanup;
    if(js_qappend(opt1,hname_js) == JS_ERROR)
        goto cleanup;
    if(js_qappend(opt2,hname_js) == JS_ERROR)
        goto cleanup;
    if(js_qstr2js(data,"") == JS_ERROR)
        goto cleanup;
    if(hname_js->unit_count > 255)
        goto cleanup;
    if(js_addbyte(data,hname_js->unit_count) == JS_ERROR)
        goto cleanup;
    if(js_append(hname_js,data) == JS_ERROR)
        goto cleanup;

    /* Build up the header for this reply */
    if(id > 0 && id < 65535)
        header.id = id;
    else
        goto cleanup;

    header.qr = 1; /* Reply */
    header.opcode = 0; /* Normal DNS */
    header.aa = 0; /* DDIP to A translations are never authoritative */
    header.tc = 0; /* A labels are too short to be truncated */
    header.rd = 0; /* Recursion not desired */
    header.ra = 0; /* Recursion not available */
    header.z = 0; /* This must be 0 unless we are EDNS aware (we aren't) */
    header.rcode = 0; /* Success! */
    header.qdcount = 1;
    header.ancount = 1;
    header.nscount = 0;
    header.arcount = 0;

    /* Make a header of the reply */
    if(make_hdr(&header,reply) == JS_ERROR)
        goto cleanup;

    /* Add the question they asked to the reply */
    if(js_append(query,reply) == JS_ERROR)
        goto cleanup;

    /* Add the class (in) to the answer */
    if(js_adduint16(reply,1) == JS_ERROR)
        goto cleanup;

    /* We will now add out manufactured reply */
    if(js_append(query,reply) == JS_ERROR)
        goto cleanup;
    /* Append the class (in) to the answer */
    if(js_adduint16(reply,1) == JS_ERROR)
        goto cleanup;
    /* Append a bogus TTL to the answer */
    if(js_adduint32(reply,770616) == JS_ERROR) /* Was 770616 */
        goto cleanup;
    /* Add the rdlength to the answer */
    if(js_adduint16(reply,data->unit_count) == JS_ERROR)
        goto cleanup;
    /* Add the actual data to the answer */
    if(js_append(data,reply) == JS_ERROR)
        goto cleanup;

    /* Send the reply out */
    sendto(sock,reply->string,reply->unit_count,0,(struct sockaddr *)client,
           sizeof(struct sockaddr));

    /* And, we are done */
    js_destroy(reply);
    js_destroy(hname_js);
    js_destroy(data);
    return JS_SUCCESS;

    /* We use gotos to work around C's lack of error trapping */
    cleanup:
        js_destroy(reply);
        js_destroy(hname_js);
        js_destroy(data);
        return JS_ERROR;

    }

/* If we successfully found a star record, spit out that record on the
   udp packet.
   Input: Where a pointer to the rr in question is, the id of the
          query they sent us, the socket the
          UDP bind is on, the sockaddr of the client who sent us the message,
          a js_string containing the query (dname + type),
          a js_string containing the answer.
   Output: JS_ERROR on error, JS_SUCCESS on success
*/

int udpstar(rr *where,int id,int sock,struct sockaddr_in *client,
               js_string *query, js_string *answer) {
    js_string *most, *ar; /* Most of the data then the additional records */

    uint16 first_rr_type;
    int in_ns = 0;
    int length_save;
    int len_inet = sizeof(struct sockaddr);

    q_header header;
    /* Initialize the js_string objects */
    if((most = js_create(1024,1)) == 0)
        return JS_ERROR;
    if((ar = js_create(1024,1)) == 0) {
        js_destroy(most);
        return JS_SUCCESS;
        }

    /* Make the header a placeholder for now */
    header.id = id;
    if(make_hdr(&header,most) == JS_ERROR)
        goto giveerror;

    /* Sanity check */
    if(where == 0) {
        goto giveerror;
        }
    if(where->query == 0) {
        goto giveerror;
        }
    if(js_has_sanity(where->query) == JS_ERROR) {
        goto giveerror;
        }
    if(where->data == 0) {
        goto giveerror;
        }
    if(js_has_sanity(where->data) == JS_ERROR) {
        goto giveerror;
        }
    if(js_has_sanity(query) == JS_ERROR) {
        goto giveerror;
        }
    first_rr_type = get_rtype(query);

    /* We have to add this header here--authoritative depends on the
       authorative status of the first record we find */
    header.aa = where->authoritative;

    /* The data must be between 0 and 65535 bytes in length (16-bit
       unsigned value) */
    if(where->data->unit_count < 0 || where->data->unit_count > 65535) {
        goto giveerror;
        }

    /* Append the question to the answer */
    if(js_append(query,most) == JS_ERROR) {
        goto giveerror;
        }

    /* Append the class (in) to the answer */
    if(js_adduint16(most,1) == JS_ERROR) {
        goto giveerror;
        }

    /* We will increment the ancount, nscount, an arcount, starting at 0 */
    header.ancount = 0;
    header.nscount = 0;
    header.arcount = 0;

    /* OK, we now add the answers */
    while(where != 0) {
        /* Increment the number of answers -or- ns records */
        if(first_rr_type != RR_NS && where->rr_type == RR_NS)
            in_ns = 1;
        if(!in_ns)
            header.ancount++;
        else
            header.nscount++;
        /* Append the name for the user's "canonical" query to the answer */
        if(!in_ns) {
            if(js_append(answer,most) == JS_ERROR) {
                goto giveerror;
                }
            }
        /* (Unless we are telling them the NS records for this RR) */
        else {
            if(js_append(where->query,most) == JS_ERROR) {
                goto giveerror;
                }
            }
        /* Append the class (in) to the answer */
        if(js_adduint16(most,1) == JS_ERROR) {
            goto giveerror;
            }
        /* Append the ttl to the answer */
        if(js_adduint32(most,determine_ttl(where->expire,where->ttl))
            == JS_ERROR) {
            goto giveerror;
            }
        /* Add the rdlength to the answer */
        if(js_adduint16(most,where->data->unit_count) == JS_ERROR) {
            goto giveerror;
            }
        /* Add the record itself to the answer */
        if(js_append(where->data,most) == JS_ERROR) {
            goto giveerror;
            }
        /* If there is an IP, and this is *not* a CNAME record,
           append the IP of the answer to the AR section */
        if(where->ip != 0 && where->rr_type != RR_CNAME) {
            /* Increment the number of additional records */
            header.arcount++;
            /* Append the name for this answer to the ip */
            if(js_append(where->ip->query,ar) == JS_ERROR) {
                goto giveerror;
                }
            /* Append the class (in) to the ip */
            if(js_adduint16(ar,1) == JS_ERROR) {
                goto giveerror;
                }
            /* Append the TTL to the ip */
            if(js_adduint32(ar,determine_ttl(where->ip->expire,where->ip->ttl))
                == JS_ERROR) {
                goto giveerror;
                }
            /* Add the rdlength to the ip */
            if(js_adduint16(ar,where->ip->data->unit_count) == JS_ERROR) {
                goto giveerror;
                }
            /* Add the record itself to the ip */
            if(js_append(where->ip->data,ar) == JS_ERROR) {
                goto giveerror;
                }
            }
        /* To do: A records attached to CNAMES are added as a second AN
                  record if the originally requested query was not a CNAME
        */
        /* Go on to the next record in the linked list */
        where = where->next;
        }

    /* Customize the header */
    /* header.id already set */
    header.qr = 1;
    header.opcode = 0;
    header.tc = 0; /* To do: truncation handling */
    header.rd = 0;
    header.ra = 0;
    header.z = 0;
    header.rcode = 0; /* No error */
    header.qdcount = 1;

    /* OBhack: Tack on the header at the beginning without molesting the
       rest of the string */
    length_save = most->unit_count;
    make_hdr(&header,most);
    most->unit_count = length_save;

    /* Add the ar records to the end */
    if(js_append(ar,most) == JS_ERROR) {
        goto giveerror;
        }

    /* Compress "most" and place the compressed data in "ar" */
    if(compress_data(most,ar) == JS_ERROR) {
        js_destroy(ar);
        udperror(sock,most,client,0,SERVER_FAIL,"compression failure",2);
        js_destroy(most);
        return JS_ERROR;
        }

    /* Check to make sure the data fits in under 512 bytes */
    if(ar->unit_count > 512) {
        /* We handle truncation by truncating everything except the
           12-byte header */
        header.tc = 1;
        make_hdr(&header,ar);
        }

    /* Success! Put out the good data */
    sendto(sock,ar->string,ar->unit_count,0,
            (struct sockaddr *)client,len_inet);

    js_destroy(most);
    js_destroy(ar);

    return JS_SUCCESS;

    /* We use gotos to make up for C's lack of error trapping */
    giveerror:
        js_destroy(ar);
        udperror(sock,most,client,0,SERVER_FAIL,"giveerror in udpstar",2);
        js_destroy(most);
        return JS_ERROR;

    }

/* If we have a NXDOMAIN, deliver that record on the udp packet.
   Input: Where a pointer to the rr in question is, the id of the
          query they sent us, the socket the
          UDP bind is on, the sockaddr of the client who sent us the message,
          a js_string containing the query (dname + type),
          if the qtype of the question is different than the desired qtype of
          the answer, then specify the qtype of the original question,
          otherwise specify 0 for the qtype
   Output: JS_ERROR on error, JS_SUCCESS on success
*/

int udpnotfound(rr *where, int id, int sock, struct sockaddr_in *client,
                js_string *query, int qtype) {
    js_string *most, *compressed; /* Most of the data */

    uint16 first_rr_type;
    int length_save, qtype_save = JS_ERROR;
    int len_inet = sizeof(struct sockaddr);

    q_header header;
    /* Initialize the js_string objects */
    if((most = js_create(1024,1)) == 0)
        return JS_ERROR;
    if((compressed = js_create(1024,1)) == 0) {
        js_destroy(most);
        return JS_ERROR;
        }

    /* Make the header a placeholder for now */
    init_header(&header);
    header.id = id;
    if(make_hdr(&header,most) == JS_ERROR) {
        js_destroy(most); js_destroy(compressed);
        return JS_ERROR;
        }

    /* Sanity check */
    if(where == 0)
        goto giveerror;
    if(where->query == 0) {
        goto giveerror;
        }
    if(js_has_sanity(where->query) == JS_ERROR) {
        goto giveerror;
        }
    if(where->data == 0) {
        goto giveerror;
        }
    if(js_has_sanity(where->data) == JS_ERROR) {
        goto giveerror;
        }
    if(js_has_sanity(query) == JS_ERROR) {
        goto giveerror;
        }
    first_rr_type = get_rtype(query);

    /* We have to add this header here--authoritative depends on the
       authorative status of the first record we find */
    header.aa = where->authoritative;

    /* The data must be between 0 and 65535 bytes in length (16-bit
       unsigned value) */
    if(where->data->unit_count < 0 || where->data->unit_count > 65535) {
        goto giveerror;
        }

    /* If they specified that the qtype of the quesiton is differnet than
       the qtype of the answer (this is used in the case of RR_ANY), then
       temporarily change the qtype when we give out the answer */
    if(qtype != 0) {
        qtype_save = get_rtype(query);
        if(qtype_save == JS_ERROR)
            goto giveerror;
        if(change_rtype(query,qtype) == JS_ERROR)
            goto giveerror;
        }

    /* Append the question to the answer */
    if(js_append(query,most) == JS_ERROR) {
        change_rtype(query,qtype_save);
        goto giveerror;
        }

    /* Restore the qtype of the query */
    if(change_rtype(query,qtype) == JS_ERROR)
        goto giveerror;

    /* Append the class (in) to the answer */
    if(js_adduint16(most,1) == JS_ERROR) {
        goto giveerror;
        }

    /* These three values stay at zero */
    header.ancount = 0;
    header.nscount = 1;
    header.arcount = 0;

    /* Append the name for this answer to the answer */
    if(js_append(where->query,most) == JS_ERROR) {
        goto giveerror;
        }
    /* Append the class (in) to the answer */
    if(js_adduint16(most,1) == JS_ERROR) {
        goto giveerror;
        }
    /* Append the ttl to the answer */
    if(js_adduint32(most,determine_ttl(where->expire,where->ttl)) ==
        JS_ERROR) {
        goto giveerror;
        }
    /* Add the rdlength to the answer */
    if(js_adduint16(most,where->data->unit_count) == JS_ERROR) {
        goto giveerror;
        }
    /* Add the record itself to the answer */
    if(js_append(where->data,most) == JS_ERROR) {
        goto giveerror;
        }

    /* Customize the header */
    /* header.id already set */
    header.qr = 1;
    header.opcode = 0;
    header.tc = 0; /* To do: truncation handling */
    header.rd = 0;
    header.ra = 0;
    header.z = 0;
    /* TO DO: Code that verifies that this host does not exist in
       any class.  If so, then we give them a rcode of NXDOMAIN_RCODE.
       Otherwise, we give them a rcode of 0 */
    header.rcode = 0;
    header.qdcount = 1;

    /* OBhack: Tack on the header at the beginning without molesting the
       rest of the string */
    length_save = most->unit_count;
    make_hdr(&header,most);
    most->unit_count = length_save;

    /* Compress "most" and place the compressed data in "compressed" */
    if(compress_data(most,compressed) == JS_ERROR) {
        js_destroy(compressed);
        udperror(sock,most,client,0,SERVER_FAIL,"Compression failure",2);
        js_destroy(most);
        return JS_ERROR;
        }

    /* Check to make sure the data fits in under 512 bytes */
    if(compressed->unit_count > 512) {
        /* We handle truncation by truncating everything except the
           12-byte header */
        header.tc = 1;
        make_hdr(&header,compressed);
        }

    /* Success! Put out the good data */
    sendto(sock,compressed->string,compressed->unit_count,0,
            (struct sockaddr *)client,len_inet);

    js_destroy(most);
    js_destroy(compressed);

    return JS_SUCCESS;


    /* We use gotos to make up for C's lack of error trapping */
    giveerror:
        js_destroy(compressed);
        udperror(sock,most,client,0,SERVER_FAIL,"giveerror in udpnotfound",2);
        js_destroy(most);
        return JS_ERROR;

    }

/* Given a domain-label starting with a star record ('_') change this label
   in-place so that the first domain label after the star record is lopped
   off of it.  Eg. '_\003sub\007example\003com\000" becomes
   "_\007example\003com\000"
   input: A pointer to the js_string object in question
   output: JS_ERROR on error, JS_SUCCESS on success, 0 if the label is
           zero-length already
*/

int bobbit_starlabel(js_string *js) {
    int counter = 1;
    unsigned char length;

    if(js->unit_size != 1)
        return JS_ERROR;
    if(js->unit_count >= js->max_count)
        return JS_ERROR;
    if(js->unit_count < 2)
        return JS_ERROR;
    if(*(js->string) != '_')
        return JS_ERROR;
    length = *(js->string + 1);

    if(length + 2 > js->unit_count || length > 63)
        return JS_ERROR;
    else if(length == 0)
        return 0;

    length++;

    while(counter < (js->unit_count - length) + 1) {
        *(js->string + counter) = *(js->string + counter + length);
        counter++;
        }

    js->unit_count -= length;

    return JS_SUCCESS;

    }

/* Given a domain-label without a star record ('_'), change the first
   domain label in to a star record ('_') Eg. "\003www\007example\003com\000"
   becomes "_\007example\003com\000"
   input: A pointer to the js_string object in question
   output: JS_ERROR on error, JS_SUCCESS on success, 0 if the label is
           a star record already
*/

int make_starlabel(js_string *js) {
    int counter = 1;
    unsigned char length;

    if(js->unit_size != 1)
        return JS_ERROR;
    if(js->unit_count >= js->max_count)
        return JS_ERROR;
    if(js->unit_count < 2)
        return JS_ERROR;
    if(*(js->string) == '_')
        return 0;
    length = *(js->string);
    *(js->string) = '_';

    if(length > js->unit_count || length > 63)
        return JS_ERROR;
    if(length == 0) /* We don't convert a "." domain-label */
        return 0;

    while(counter < js->unit_count - length) {
        *(js->string + counter) = *(js->string + counter + length);
        counter++;
        }

    js->unit_count -= length;

    return JS_SUCCESS;

    }

/* Given a domain-label ending with (or without) a star record ('_'),
   change the label
   in-place so that the first domain label before the star record is lopped
   off of it.  Eg. "\003name\007example\003com\000" becomes
   "\003name\007example\003com_", and "\003name\007example\003com_"
   becomes "\003name\007example_"
   input: A pointer to the js_string object in question
   output: JS_ERROR on error, JS_SUCCESS on success, 0 if the label is
           zero-length already
*/

/* CODE HERE: Make sure this works */

int bobbit_starlabel_end(js_string *js) {
    int counter = 1;
    unsigned char toread;
    int length;

    if(js->unit_size != 1)
        return JS_ERROR;
    if(js->unit_count >= js->max_count)
        return JS_ERROR;
    if(js->unit_count < 2)
        return JS_ERROR;
    counter = dlabel_length(js,0);
    counter--;
    if(counter < 0)
        return JS_ERROR;

    /* If this is not a starlabel-at-end label yet, convert it */
    if(*(js->string + counter) == '\0') {
        *(js->string + counter) = '_';
        return JS_SUCCESS;
        }

    /* Otherwise, lop off the last label */
    length = 0;
    toread = *(js->string);
    counter = 0;
    while(length < 256 && toread > 0 && toread != '_') {
        if(toread > 63)
            return JS_ERROR; /* No EDNS nor compressed label support */
        length += toread + 1;
        /* Go to the next jump */
        if(length > js->unit_count || length >= js->max_count)
            return JS_ERROR;
        counter = toread;
        toread = *(js->string + length);
        }
    if(length >= 256) {
        return JS_ERROR;
        }

    counter++;
    if(counter > js->unit_count || counter > length) {
        return JS_ERROR;
        }

    js->unit_count -= counter;
    if(js->unit_count < 1) {
        return JS_ERROR;
        }
    *(js->string + js->unit_count - 1) = '_';

    return JS_SUCCESS;

    }

/* Convert a domain-name query in to its lower-case equivalent
   Input: Pointer to the js string object with the query
   Output: JS_ERROR on error, JS_SUCCESS on sucess, 0 on
           success if no change was made to the string */

int fold_case(js_string *js) {
    int counter = 0;
    int ret = 0;

    if(js->max_count <= js->unit_count) {
        return JS_ERROR;
        }
    if(js->unit_size != 1) {
        return JS_ERROR;
        }
    if(js->unit_count < 2) {
        return JS_ERROR;
        }
    while(counter + 2 < js->unit_count) {
        /* Since A-Z never happen in a domain length label, we can speed
           things up a bit */
        if(*(js->string + counter) >= 'A' && *(js->string + counter) <= 'Z') {
            *(js->string + counter) += 32;
            ret = 1;
            }
        counter++;
        }

    return ret;

    }

/* Check to see if the IP in question is a ddip (e.g.
   "<03>127<01>0<01>0<03>1<00>"), and, if so, convert it in to
   a bare A record
   input: Pointer to js_string object with the query
   output: JS_ERROR on fatal error, 0 on non-ddip query,
           JS_SUCCESS if it was a ddip
*/

int ddip_check(int id, int sock, struct sockaddr *from, js_string *query) {
    unsigned char ip[4];
    unsigned char length, val;
    int counter, critter, lenl, value;
    js_string *reply;
    q_header header;

    /* Sanity checks */
    if(query->unit_size != 1)
        return JS_ERROR;
    if(query->unit_count >= query->max_count)
        return JS_ERROR;

    /* We presently only do ddip translation for A and ANY requests
       (DJB only supports this in Dnscache) */
    if(get_rtype(query) != RR_A && get_rtype(query) != RR_ANY)
        return 0;

    if(query->unit_count < 9) /* The minimum possible length for a
                                 ddip domain label */
        return 0;

    lenl = 0;
    for(counter=0;counter<4;counter++) {
        length = *(query->string + lenl);
        if(length < 1 || length > 3)
            return 0;
        critter = lenl + 1;
        lenl += length + 1;
        if(lenl > query->unit_count)
            return JS_ERROR;
        for(value = 0;critter < lenl;critter++) {
            val = *(query->string + critter);
            if(val > '9' || val < '0')
                return 0;
            value *= 10;
            value += val - '0';
            }
        if(value < 0 || value > 255)
            return 0;
        ip[counter] = value;
        }

    if(*(query->string + lenl) != 0)
        return 0;

    /* OK, it is definitely a ddip label.  Convert the ip in to a DNS reply */

    if((reply = js_create(512,1)) == 0)
        return JS_ERROR;

    /* Build up the header for this reply */
    if(id > 0 && id < 65535)
        header.id = id;
    else
        goto cleanup;

    header.qr = 1; /* Reply */
    header.opcode = 0; /* Normal DNS */
    header.aa = 0; /* DDIP to A translations are never authoritative */
    header.tc = 0; /* A labels are too short to be truncated */
    header.rd = 0; /* Recursion not desired */
    header.ra = 0; /* Recursion not available */
    header.z = 0; /* This must be 0 unless we are EDNS aware (we aren't) */
    header.rcode = 0; /* Success! */
    header.qdcount = 1;
    header.ancount = 1;
    header.nscount = 0;
    header.arcount = 0;

    /* Make a header of the reply */
    if(make_hdr(&header,reply) == JS_ERROR)
        goto cleanup;

    /* Add the question they asked to the reply */
    if(js_append(query,reply) == JS_ERROR)
        goto cleanup;

    /* Add the class (in) to the answer */
    if(js_adduint16(reply,1) == JS_ERROR)
        goto cleanup;

    /* Make sure the answer is an A record type */
    if(change_rtype(query,RR_A) == JS_ERROR)
        goto cleanup;

    /* We will now add out manufactured A reply */
    if(js_append(query,reply) == JS_ERROR)
        goto cleanup;
    /* Append the class (in) to the answer */
    if(js_adduint16(reply,1) == JS_ERROR)
        goto cleanup;
    /* Append a bogus TTL to the answer */
    if(js_adduint32(reply,19770616) == JS_ERROR)
        goto cleanup;
    /* Add the rdlength to the answer */
    if(js_adduint16(reply,4) == JS_ERROR)
        goto cleanup;
    /* Add the actual 4-byte reply to the answer */
    for(counter = 0; counter < 4; counter++) {
        if(js_addbyte(reply,ip[counter]) == JS_ERROR)
            goto cleanup;
        }

    /* Send the reply out */
    sendto(sock,reply->string,reply->unit_count,0,from,
           sizeof(struct sockaddr));

    /* And, we are done */
    js_destroy(reply);
    return JS_SUCCESS;

    /* We use gotos to work around C's lack of error trapping */
    cleanup:
        js_destroy(reply);
        return JS_ERROR;

    }

/* Determine if a given IP has authority to perform recursive DNS lookups
   Input: IP of where they come from
   Ouput: 0 if they do not have authority, 1 if they do
   Global variables used: The recurse_acl array
*/

int check_recursive_acl(uint32 ip) {
    int counter = 0, ret = 0;
    while(counter < 500 && (recurse_acl[counter]).ip != 0xffffffff) {
        if((ip & (recurse_acl[counter]).mask) ==
               ((recurse_acl[counter]).ip & (recurse_acl[counter]).mask)) {
            /* They are authorized */
            ret = 1;
            break;
            }
        counter++;
        }
    return ret;
    }

/* Look for both the upper and lower case versions of a given query.
   Input: The query, and, to give to udpsuccess:
          The ID of this query
          The socket ID of this query
          Where this ID came from
          The original query (to echo in the question)
   Output: 0 on not found, JS_ERROR on error, JS_SUCCESS on success
 */

int hunt_single_query(js_string *query, int id, int sock,
                      struct sockaddr_in *from, js_string *question) {
    mhash_e spot_data;
    int qtype_o, qtype_q;
    js_string *lower;

    qtype_o = get_rtype(question);
    qtype_q = get_rtype(query);

    if(qtype_o == JS_ERROR || qtype_q == JS_ERROR) {
        return JS_ERROR;
        }

    spot_data = mhash_get(bighash,query);
    /* If found, give back the response */
    if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
        if(qtype_o == RR_A || qtype_q == RR_CNAME) {
            udpsuccess(spot_data.value,id,sock,from,question,
                       spot_data.point,1);
            }
        else {
            udpsuccess(spot_data.value,id,sock,from,question,
                       spot_data.point,0);
             }
        return JS_SUCCESS;
        }
    /* Try a lower-case version of the query */
    if((lower = js_create(258,1)) == 0) {
        return JS_ERROR;
        }
    if(js_copy(query,lower) == JS_ERROR) {
        js_destroy(lower);
        return JS_ERROR;
        }
    if(fold_case(lower) == 1) {
        spot_data = mhash_get(bighash,lower);
        js_destroy(lower);
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
            if(qtype_o == RR_A || qtype_q == RR_CNAME) {
                udpsuccess(spot_data.value,id,sock,from,question,
                           spot_data.point,1);
                }
            else {
                udpsuccess(spot_data.value,id,sock,from,question,
                           spot_data.point,0);
                }
            return JS_SUCCESS;
            }
        }
    else {
        js_destroy(lower);
        return 0;
        }
    return JS_ERROR;
    }

/* Process the DNS query that comes in from the 'net
   Input: uncompressed form of incoming UDP query, IP address of where
          this query came from, socket number of this socket
   Output: JS_ERROR on error, JS_SUCCESS on success
*/

int proc_query(js_string *raw, struct sockaddr_in *from, int sock) {

    q_header header; /* Header of the question */
    js_string *lookfor; /* What to look for in the big hash */
    js_string *origq; /* Original query asked by the user */
    js_string *lc; /* Lower-case version of query asked by the user */
    rr *nxstore = 0; /* A pointer to the SOA we return when we hit a
                        NXDOMAIN */
    int length, case_folded, result_code = 0, qtype;
    int has_recursive_authority = 0;
    mhash_e spot_data;
    int have_authority = 0; /* Do we have authority for this record?
                               (must be 1 to return a NXDOMAIN) */
    rr *point;
    uint32 ip;
    int desires_recursion = 0; /* Do they desire recursion? */
    char *num_string; /* The string to put the number of thread running
                         in */
    unsigned int mem_usage; /* The amount of memory a maradns process has
                               allocated */

    /* Sanity checks */
    if(js_has_sanity(raw) == JS_ERROR)
        return JS_SUCCESS;
    if(raw->unit_size != 1)
        return JS_SUCCESS;

    /* Get the header */
    if(read_hdr(raw,&header) == JS_ERROR) { /* Something went wrong,
                                               return error "Format error" */
        udperror(sock,raw,from,0,FORMAT_ERROR,"Couldn't get header",2);
        return JS_SUCCESS;
        }

    /* We only answer questions (Thanks to Roy Arends for pointing out this
       security flaw) */
    if(header.qr != 0) {
        return JS_SUCCESS;
        }

    /* We only support a qdcount of 1 */
    if(header.qdcount != 1) {
        if(no_fingerprint != 1)
            udperror(sock,raw,from,0,NOT_IMPLEMENTED,"Qdcount not 1",2);
        return JS_SUCCESS;
        }

    /* See if they desire recursion or not */
    desires_recursion = header.rd;

    /* Get the question from the stream */
    if(raw->unit_count < 14) {
        if(no_fingerprint != 1)
            udperror(sock,raw,from,0,FORMAT_ERROR,"bad question hdr",2);
        return JS_SUCCESS;
        }

    /* Determine the length of the domain label in the question */
    length = dlabel_length(raw,12);
    if(length < 0 || length > 255) {
        if(no_fingerprint != 1)
            udperror(sock,raw,from,0,FORMAT_ERROR,"bad question length",2);
        return JS_SUCCESS;
        }

    if(raw->unit_count < 16 + length) { /* 16 because 12 for the header,
                                           and 4 for the type and class */
        if(no_fingerprint != 1)
            udperror(sock,raw,from,0,FORMAT_ERROR,"question doesn't fit",2);
        return JS_SUCCESS;
        }

    /* Create the lookfor string, returning error if appropriate */
    if((lookfor = js_create(256,1)) == 0) {
        if(no_fingerprint != 1)
            udperror(sock,raw,from,0,SERVER_FAIL,
                     "can't create lookfor string",2);
        return JS_ERROR;
        }
    if((origq = js_create(256,1)) == 0) {
        js_destroy(lookfor);
        udperror(sock,raw,from,0,SERVER_FAIL,"can't create origq string",2);
        return JS_ERROR;
        }
    if((lc = js_create(256,1)) == 0) {
        js_destroy(lookfor); js_destroy(origq);
        udperror(sock,raw,from,0,SERVER_FAIL,"can't create lc string",2);
        return JS_ERROR;
        }
    if(js_set_encode(lc,JS_US_ASCII) == JS_ERROR) { /* ASCII because we
                                                     only fold the case of
                                                     A-Z */
        goto serv_fail;
        }

    /* Get the query we will look for from their raw query */
    if(js_substr(raw,lookfor,12,length + 2) == JS_ERROR) {
        goto serv_fail;
        }

    /* We only support an opcode of 0 (standard query)
       (this check is down here so we can echo the question) */
    if(header.opcode != 0) {
        /* Since TinyDNS also returns NOT_IMPLEMENTED here, no need for
           a fingerprint check.  Note that tinydns, unlike MaraDNS, echos
           the question. */
        udperror(sock,raw,from,lookfor,NOT_IMPLEMENTED,"non-0 opcode",2);
        return JS_SUCCESS;
        }

    /* Return "not implemented" if the class is not 1 (Internet class) */
    /* Down here so we can echo the question */
    if(*(raw->string + length + 14) != 0 &&
       *(raw->string + length + 15) != 1) {
        if(no_fingerprint != 1)
            udperror(sock,raw,from,lookfor,NOT_IMPLEMENTED,"Class not 1",2);
        return JS_ERROR;
        }

    /* And copy it over to the "original query" */
    if(js_copy(lookfor,origq) == JS_ERROR) {
        goto serv_fail;
        }

    /* Get the type of query the client desires */
    qtype = get_rtype(origq);
    if(qtype == JS_ERROR) {
        goto serv_fail;
        }

    if(qtype >= 250 && qtype <= 254) { /* IXFR, AXFR, and 2 more */
        goto not_impl;
        }

    /* See if they have permission to perform a recursive query */
    ip = htonl((from->sin_addr).s_addr);
    has_recursive_authority = check_recursive_acl(ip);
    if(has_recursive_authority == JS_ERROR)
        goto serv_fail;

    /* Handle the case of RR_ANY */
    if(qtype == RR_ANY &&
       (desires_recursion == 0 || has_recursive_authority == 0)) {
        result_code = udpany(header.id,sock,from,origq,rrany_set,bighash);
        if(result_code == JS_SUCCESS) {
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        else if(result_code == JS_ERROR)
            goto serv_fail;
        /* Otherwise, no RR_ANY information was found.  We will return,
           if appropriate, the expected "no such host" SOA reply or
           NS delegation reply.  Hack: Since there *should not* be
           any elements in the hash proper with "ANY" as the desired
           record type, we go ahead and perform the various normal
           searches. */
        }

    /* OK, start the complicated domain look up routine */
    /* Look for upper and lower case versions of the query as
       they asked it */
    if(hunt_single_query(lookfor,header.id,sock,from,origq) != 0) {
        js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
        return JS_SUCCESS;
        }

    /* OK, if not found, maybe there is a CNAME with the same domain label */
    if(change_rtype(lookfor,RR_CNAME) == JS_ERROR) {
        goto serv_fail;
        }
    if(hunt_single_query(lookfor,header.id,sock,from,origq) != 0) {
        js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
        return JS_SUCCESS;
        }

    /* CODE HERE: Franky's request to have "administrative IPs" */

#ifdef VERSION
        /* A TXT query to "erre-con-erre-cigarro.maradns.org" will
           return the version of MaraDNS being run.  This only
           works if we are not authoritative for "maradns.org", since
           the real "erre-con-erre-cigarro.maradns.org" says
           "MaraDNS version number not available" in the TXT record.
           Note: This is disabled if no_fingerprint is 1 or if
           debug_msg_level is less than one */
        if(origq->unit_count == 37 && *(origq->string) == 21
           && no_fingerprint != 1 && debug_msg_level >= 1) {
            result_code = easter_egg(header.id,sock,from,origq,
               "Terre-con-erre-cigarro.maradns.org.",RR_TXT,"MaraDNS version ",
               VERSION);
            if(result_code == JS_SUCCESS) {
                js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                return JS_SUCCESS;
                }
            if(result_code == JS_ERROR) {
                goto serv_fail;
                }
            }
#endif /* VERSION */

        /* A TXT query to "numthreads." tells us the number of
           threads that MaraDNS is running; this is only enabled if
           no_fingerprint is 0 and if debug_msg_level is 2 or greater
        */
       if(origq->unit_count == 14 && *(origq->string) == 10
          && no_fingerprint != 1 && debug_msg_level >= 2) {
           /* Allocate a string to put the number of threads running in */
           if((num_string = js_alloc(32,1)) == 0) {
               js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
               return JS_ERROR;
               }
           snprintf(num_string,10,"%d",how_many_threads());
           result_code = easter_egg(header.id,sock,from,origq,
            "Tnumthreads.",RR_TXT,"Number threads running: ",
            num_string);
           js_dealloc(num_string);
           if(result_code == JS_SUCCESS) {
               js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
               return JS_SUCCESS;
               }
           if(result_code == JS_ERROR) {
               goto serv_fail;
               }
           }

        /* A TXT query to "memusage." tells us the number of
           threads that MaraDNS is running; this is only enabled if
           no_fingerprint is 0 and if debug_msg_level is 2 or greater
        */
       if(origq->unit_count == 12 && *(origq->string) == 8
          && no_fingerprint != 1 && debug_msg_level >= 2) {
           /* Allocate a string to put the number of threads running in */
           mem_usage = js_tell_memory_allocated();
           if(mem_usage > 0) {
               if((num_string = js_alloc(32,1)) == 0) {
                   js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                   return JS_ERROR;
                   }
               snprintf(num_string,14,"%d",mem_usage);
               result_code = easter_egg(header.id,sock,from,origq,
                "Tmemusage.",RR_TXT,"Memory usage, in bytes: ",
                num_string);
               js_dealloc(num_string);
               }
           else {
               result_code = easter_egg(header.id,sock,from,origq,
                "Tmemusage.",RR_TXT,"Memory usage unknown; ",
                "try compiling with make debug");
               }
           if(result_code == JS_SUCCESS) {
               js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
               return JS_SUCCESS;
               }
           if(result_code == JS_ERROR) {
               goto serv_fail;
               }
           }

        /* A TXT query to "timestamp." tells us the time
           on the system MaraDNS is running on; this is only enabled if
           no_fingerprint is 0 and if debug_msg_level is 2 or greater
        */
       if(origq->unit_count == 13 && *(origq->string) == 9
          && no_fingerprint != 1 && debug_msg_level >= 3) {
           /* Allocate a string to put the number of threads running in */
           mem_usage = (int)time(0);
           if(mem_usage > 0) {
               if((num_string = js_alloc(32,1)) == 0) {
                   js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                   return JS_ERROR;
                   }
               snprintf(num_string,14,"%d",mem_usage);
               result_code = easter_egg(header.id,sock,from,origq,
                "Ttimestamp.",RR_TXT,"Timestamp: ",
                num_string);
               js_dealloc(num_string);
               }
           else {
               result_code = easter_egg(header.id,sock,from,origq,
                "Tmemusage.",RR_TXT,"Memory usage unknown; ",
                "try compiling with make debug");
               }
           if(result_code == JS_SUCCESS) {
               js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
               return JS_SUCCESS;
               }
           if(result_code == JS_ERROR) {
               goto serv_fail;
               }
           }

        /* A TXT query to "cache-elements." tells us the number of
           elements in the DNS cache; this is only enabled if
           no_fingerprint is 0 and if debug_msg_level is 2 or greater
        */
       if(origq->unit_count == 18 && *(origq->string) == 14
          && no_fingerprint != 1 && debug_msg_level >= 2) {
           /* Allocate a string to put the number of threads running in */
           if((num_string = js_alloc(32,1)) == 0) {
               js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
               return JS_ERROR;
               }
           snprintf(num_string,10,"%d",cache_elements());
           result_code = easter_egg(header.id,sock,from,origq,
            "Tcache-elements.",RR_TXT,"Elements in DNS cache: ",
            num_string);
           js_dealloc(num_string);
           if(result_code == JS_SUCCESS) {
               js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
               return JS_SUCCESS;
               }
           if(result_code == JS_ERROR) {
               goto serv_fail;
               }
           }

    /* OK, if not found, maybe there is a *nonauthoritative* NS record with
       the same domain label */
    if(change_rtype(lookfor,RR_NS) == JS_ERROR) {
        goto serv_fail;
        }

    spot_data = mhash_get(bighash,lookfor);
    point = spot_data.value;
    /* If the non-authoritative NS was found, return the NS infomation */
    if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR &&
       point->authoritative == 0) {
        /* In the case of them desiring recursion, and the client having
           authority to be recursive, we go recursive */
        if(desires_recursion == 1 && has_recursive_authority == 1) {
            launch_thread(header.id,sock,*from,origq);
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        /* Otherwise, we return a NS server delegation */
        udpsuccess(spot_data.value,header.id,sock,from,origq,spot_data.point,0);
        js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
        return JS_SUCCESS;
        }

    /* Maybe a lower-case version of the same ns label */
    if(case_folded==1) { /* If this domain label had uppercase letters in it */
        if(change_rtype(lc,RR_NS) == JS_ERROR) {
            goto serv_fail;
            }

        spot_data = mhash_get(bighash,lc);
        point = spot_data.value;
        /* If the non-authoritative NS was found, return the NS infomation */
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR &&
           point->authoritative == 0) {
            /* In the case of them desiring recursion, and the client having
               authority to be recursive, we go recursive */
            if(desires_recursion == 1 && has_recursive_authority == 1) {
                launch_thread(header.id,sock,*from,origq);
                js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                return JS_SUCCESS;
                }
            /* Otherwise, we return a NS server delegation */
            udpsuccess(spot_data.value,header.id,sock,from,origq,
                       spot_data.point,0);
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        }

    /* See if it is a dotted-decimal IP */
    if(no_fingerprint != 1) {
        result_code = ddip_check(header.id,sock,(struct sockaddr *)from,origq);
        if(result_code == JS_SUCCESS) {
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        if(result_code == JS_ERROR) {
            goto serv_fail;
            }
        }

    /* Look for a NS record at the same level or above.  E.G., if they
       ask for somthing.below.sub.example.com. and we have knowledge
       that sub.example.com is a NS record, act as appropriate.  Send
       them the NS record -or- go recursive if the NS record is
       non-authoritative (we're not handling the zone), otherwise return
       a "host not there" if the NS record is authoritative
     */

    nxstore = NULL;

    do {
        spot_data = mhash_get(bighash,lookfor);
        point = spot_data.value;
        /* We stop going up the tree if we have found an authoritative NS
           record */
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR &&
           point->authoritative != 0) {
            have_authority = 1;
            /* Look for a SOA record of the same type to prepare for
               a NXDOMAIN reply */
            if(change_rtype(lookfor,RR_SOA) == JS_ERROR) {
                goto serv_fail;
                }
            spot_data = mhash_get(bighash,lookfor);
            if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                nxstore = spot_data.value;
                }
            break;
            }
        /* Return the NS record we found "up the tree", if appropriate */
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
            /* In the case of them desiring recursion, and the client having
               authority to be recursive, we go recursive */
            if(desires_recursion == 1 && has_recursive_authority == 1) {
                launch_thread(header.id,sock,*from,origq);
                js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                return JS_SUCCESS;
                }
            /* Otherwise, we return a NS server delegation */
            udpsuccess(spot_data.value,header.id,sock,from,origq,
                       spot_data.point,0);
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        } while(bobbit_label(lookfor) > 0);

    /* Same thing, with case-insensitivity */
    if(nxstore == NULL && case_folded==1) {
        /* If nothing foung and this domain label had
           uppercase letters in it */
        do {
            spot_data = mhash_get(bighash,lc);
            point = spot_data.value;
            /* We stop going up the tree if we have found an authoritative NS
               record */
            if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR &&
               point->authoritative != 0) {
                have_authority = 1; /* We are allowed to give a NXDOMAIN */
                /* Look for a SOA record of the same type to prepare for
                   a NXDOMAIN reply */
                if(change_rtype(lc,RR_SOA) == JS_ERROR) {
                    goto serv_fail;
                    }
                spot_data = mhash_get(bighash,lc);
                if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                    nxstore = spot_data.value;
                    }
                break;
                }
            if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                /* In the case of them desiring recursion, and the client
                   having authority to be recursive, we go recursive */
                if(desires_recursion == 1 && has_recursive_authority == 1) {
                    launch_thread(header.id,sock,*from,origq);
                    js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                    return JS_SUCCESS;
                    }
                /* Otherwise, we return a NS server delegation */
                udpsuccess(spot_data.value,header.id,sock,from,origq,
                           spot_data.point,0);
                js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                return JS_SUCCESS;
                }
            } while(bobbit_label(lc) > 0);
        }

    /* If we do not have authority for this record... */
    if(have_authority == 0) {
        /* Ask other DNS servers for RRs which we do not have authoity
           for.  */

        /* Launch a separate thread to recursivly determine the
           host name in question */
        if(has_recursive_authority == 1 && desires_recursion == 1)
            launch_thread(header.id,sock,*from,origq);
        else
            udperror(sock,raw,from,0,REFUSED,
            "I'm sorry Dave (recurse attempt)",3);
        js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
        return JS_SUCCESS;
        }

    /* Maybe it is a star record they are looking for */

    /* We need to restore "lookfor" and "lc" because we shredded both
       strings looking for a NS sub-delegation */
    if(js_copy(origq,lookfor) == JS_ERROR) {
        goto serv_fail;
        }
    /* Convert lookfor in to a star label */
    if(make_starlabel(lookfor) == JS_ERROR) {
        goto serv_fail;
        }

    /* Look for the star record in the big hash */
    spot_data = mhash_get(bighash,lookfor);
    if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
        udpstar(spot_data.value,header.id,sock,from,origq,origq);
        js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
        return JS_SUCCESS;
        }

    /* Then look for the lowercase star record */

    /* Make a 'lc' (lower case) version of the query if appropriate */
    if(case_folded==1) { /* Naturally, we only deal with lc if we need to */
        if(js_copy(lookfor,lc) == JS_ERROR) {
            goto serv_fail;
            }
        if(fold_case(lc) == JS_ERROR) {
            goto serv_fail;
            }
        spot_data = mhash_get(bighash,lc);
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
            /* We have to fold the case of the query answer to lowercase */
            if(js_copy(origq,lc) == JS_ERROR) {
                goto serv_fail;
                }
            if(fold_case(lc) == JS_ERROR) {
                goto serv_fail;
                }
            udpstar(spot_data.value,header.id,sock,from,origq,lc);
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        }

    /* OK, maybe there is a star record "above".  In other words,
       handle the case when they ask for foo.bar.example.com and we have
       a record for *.example.com */
    while(bobbit_starlabel(lookfor) > 0) {
        spot_data = mhash_get(bighash,lookfor);
        point = spot_data.value;
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
            udpstar(spot_data.value,header.id,sock,from,origq,origq);
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        }

    /* lower-case version of looking for a label "above" */
    if(case_folded==1) {
        while(bobbit_starlabel(lc) > 0) {
            spot_data = mhash_get(bighash,lc);
            point = spot_data.value;
            if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                /* We have to fold the case of the query answer to lowercase */
                if(js_copy(origq,lc) == JS_ERROR) {
                    goto serv_fail;
                    }
                if(fold_case(lc) == JS_ERROR) {
                    goto serv_fail;
                    }
                udpstar(spot_data.value,header.id,sock,from,origq,lc);
                js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                return JS_SUCCESS;
                }
            }
        }

    /* Perhaps they have a star record which points to a CNAME (yes,
       some people actually do this) */

    /* We need to restore "lookfor" and "lc" because, again, we shredded
       both strings looking for a star label that was not a CNAME */
    if(js_copy(origq,lookfor) == JS_ERROR)
        goto serv_fail;
    /* First, we make it a star label */
    if(make_starlabel(lookfor) == JS_ERROR)
        goto serv_fail;
    /* Then we make it a CNAME rtype */
    if(change_rtype(lookfor,RR_CNAME) == JS_ERROR)
        goto serv_fail;

    /* Look for the star CNAME record in the big hash */
    spot_data = mhash_get(bighash,lookfor);
    if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
        /* We have to make a form of origq that is a cname */
        if(js_copy(origq,lookfor) == JS_ERROR)
            goto serv_fail;
        if(change_rtype(lookfor,RR_CNAME) == JS_ERROR)
            goto serv_fail;
        udpstar(spot_data.value,header.id,sock,from,origq,lookfor);
        js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
        return JS_SUCCESS;
        }

    /* And now look for a lowecase star CNAME record in the big hash */
    if(case_folded==1) { /* Only deal with lower-case if needed */
        if(js_copy(lookfor,lc) == JS_ERROR)
            goto serv_fail;
        if(fold_case(lc) == JS_ERROR)
            goto serv_fail;
        spot_data = mhash_get(bighash,lc);
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
            /* We have to fold the case of the query answer to lowercase */
            if(js_copy(origq,lc) == JS_ERROR)
                goto serv_fail;
            if(fold_case(lc) == JS_ERROR)
                goto serv_fail;
            if(change_rtype(lc,RR_CNAME) == JS_ERROR)
                goto serv_fail;
            udpstar(spot_data.value,header.id,sock,from,origq,lc);
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        }

    /* Look for a star record "above" when it is a CNAME */
    while(bobbit_starlabel(lookfor) > 0) {
        spot_data = mhash_get(bighash,lookfor);
        if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
            if(js_copy(origq,lookfor) == JS_ERROR)
                goto serv_fail;
            if(change_rtype(lookfor,RR_CNAME) == JS_ERROR)
                goto serv_fail;
            udpstar(spot_data.value,header.id,sock,from,origq,lookfor);
            js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
            return JS_SUCCESS;
            }
        }

    /* Look for a lower-case version of the star record if appropriate */
    if(case_folded == 1) {
        while(bobbit_starlabel(lc) > 0) {
            spot_data = mhash_get(bighash,lc);
            point = spot_data.value;
            if(spot_data.value != 0 && spot_data.datatype == MARA_DNSRR) {
                /* We have to copy over the original query, change the RR to
                   CNAME, and fold the case of it. */
                if(js_copy(origq,lc) == JS_ERROR)
                    goto serv_fail;
                if(fold_case(lc) == JS_ERROR)
                    goto serv_fail;
                if(change_rtype(lookfor,RR_CNAME) == JS_ERROR)
                    goto serv_fail;
                udpstar(spot_data.value,header.id,sock,from,origq,lc);
                js_destroy(lookfor); js_destroy(origq); js_destroy(lc);
                return JS_SUCCESS;
                }
            }
        }

    /* CODE HERE: Perhaps they have something which ends in a star record;
       look for it. */

    /* Currently, MaraDNS will not support star records for NS
       subdelegation.  Code in ParseCsv1.c warns the user of this fact. */

    udpnotfound(nxstore,header.id,sock,from,origq,0);
    js_destroy(lookfor); js_destroy(origq); js_destroy(lc);

    return JS_SUCCESS;

    /* Work around C's lack of error handling and garbage collection with
       gotos */
    serv_fail:
        js_destroy(origq);
        js_destroy(lc);
        if(no_fingerprint != 1)
            udperror(sock,raw,from,lookfor,SERVER_FAIL,
                     "serv_fail in proc_query",2);
        js_destroy(lookfor);
        return JS_ERROR;

    not_impl:
        js_destroy(origq);
        js_destroy(lc);
        if(no_fingerprint != 1)
            udperror(sock,raw,from,lookfor,NOT_IMPLEMENTED,
                     "not_impl in proc_query",2);
        js_destroy(lookfor);
        return JS_ERROR;

    }

/* Bind to UDP port 53. (or DNS_PORT if debugging MaraDNS on a system where
                         I do not have root, and theirfore can not bind to
                         a low port number)
   Input:  pointer to socket to bind on, js_string with the dotted-decimal
           ip address to bind to
   Output: JS_ERROR on error, JS_SUCCESS on success
*/

int udpbind(int *sockets, ipv4pair *addresses) {
    int len_inet; /* Length */
    struct sockaddr_in dns_udp;
    int counter;

    /* Sanity checks */
    if(sockets == 0)
        return JS_ERROR;

    counter = 0;

    /* Create a socket address to use with bind() */
    while(counter < 500 && addresses[counter].ip != 0xffffffff) {
        /* Create a raw UDP socket */
        if((sockets[counter] = socket(AF_INET,SOCK_DGRAM,0)) == -1) {
            return JS_ERROR;
            }

        memset(&dns_udp,0,sizeof(dns_udp));
        dns_udp.sin_family = AF_INET;
        /* DNS_PORT is usually 53, but can be another port.  Defined in
           MaraDNS.h */
        dns_udp.sin_port = htons(DNS_PORT);
        if((dns_udp.sin_addr.s_addr = htonl(addresses[counter].ip))
           == INADDR_NONE)
            return JS_ERROR;

        len_inet = sizeof(dns_udp);

        /* Bind to the socket.  Note that we usually have to be root to
           do this */
        if(bind(sockets[counter],(struct sockaddr *)&dns_udp,len_inet) == -1)
            return JS_ERROR;

        counter++;
        }

    /* We are now bound to UDP port 53. (Or whatever DNS_PORT is) Leave */
    return JS_SUCCESS;
    }

/* Get information from a previously binded UDP socket
   Input:  list of UDP bound sockets, list of addresses we are bound to,
           pointer to sockaddr structure that will contain
           the IP of the system connecting to us, pointer to js_string
           object that will have the data in question, maximum allowed
           length of data we receive
   Output: JS_ERROR on error, socket we got packet from on success
*/

int getudp(int *sock,ipv4pair *addr,struct sockaddr *client,
           js_string *data, int max_len) {
    int len_inet, counter, len;
    fd_set rx_fd;
    int select_output;
    int max_socket;
    struct timeval timeout;

    /* Sanity checks */
    if(client == 0 || data == 0)
        return JS_ERROR;
    if(js_has_sanity(data) == JS_ERROR)
        return JS_ERROR;
    if(data->unit_size != 1)
        return JS_ERROR;
    if(max_len < 0 || max_len >= data->max_count)
        return JS_ERROR;

    len_inet = sizeof(struct sockaddr);

    FD_ZERO(&rx_fd);
    counter = 0;
    max_socket = 0;
    while(counter < 500 && addr[counter].ip != 0xffffffff) {
        FD_SET(sock[counter],&rx_fd);
        if((sock[counter] + 1) > max_socket) {
            max_socket = sock[counter] + 1;
            }
        counter++;
        }
    if(max_socket == 0) /* No sockets */ {
        return JS_ERROR;
        }
 
    timeout.tv_sec = 1; /* Check for HUP signal every second */
    timeout.tv_usec = 0;

    /* OK, wait for activity on any of those sockets */
    select_output = select(max_socket,&rx_fd,NULL,NULL,&timeout);

    if(select_output <= 0) { /* 0: Timeout; less than 0: error */
        return JS_ERROR;
	}

    /* Figure out which socket gave us something */
    counter = 0;
    while(counter < 500 && addr[counter].ip != 0xffffffff) {
        if(FD_ISSET(sock[counter],&rx_fd)) {
            len = recvfrom(sock[counter],data->string,max_len,0,
	                   client,&len_inet);
            if(len < 0)
                return JS_ERROR;

            data->unit_count = len;
	     
	    return sock[counter];
            }
        counter++;
        }

    return JS_ERROR;

    }

/* Routine which reads a numeric kvar from the database of values set
   in the mararc file (this can not be used for dictionary variables).

   Input: A null-terminated string with the desired variable name,
          the default value for the kvar in question (if not set)

   Output: The numeric value in question (always positive or zero)
           -1 (JS_ERROR) if a fatal error happened

 */

int read_numeric_kvar(char *name,int default_value) {
    js_string *kvar_name;
    js_string *kvar_value;
    int ret,status;

    if((kvar_name = js_create(64,1)) == 0) {
        printf("Aieeeeeee!\n");
        exit(1);
        return JS_ERROR;
        }

    if((kvar_value = js_create(256,1)) == 0) {
        printf("Aieeeeeeee!\n");
        exit(1);
        js_destroy(kvar_name);
        return JS_ERROR;
        }

    js_set_encode(kvar_name,MARA_LOCALE);
    js_set_encode(kvar_value,MARA_LOCALE);

    if(js_qstr2js(kvar_name,name) == JS_ERROR) {
        js_destroy(kvar_name);
        js_destroy(kvar_value);
        harderror(L_KVAR_Q); /* "Could not create kvar_query" */
        }

    status = read_kvar(kvar_name,kvar_value);

    if(status == JS_ERROR) { /* Fatal error parsing it */
        js_destroy(kvar_name);
        js_destroy(kvar_value);
        show_timestamp();
        printf("%s%s\n","Error processing value for ",name);
        return default_value;
        }

    if(status == 0) { /* Variable not set in mararc */
        js_destroy(kvar_name);
        js_destroy(kvar_value);
        return default_value;
        }

    ret = js_atoi(kvar_value,0);

    js_destroy(kvar_name);
    js_destroy(kvar_value);
    return ret;

    }

/* The core of the DNS server */

int main(int argc, char **argv) {

    js_string *mararc_loc, *errors,
              *kvar_query, *bind_address, *incoming, *uncomp, *verbstr;
    unsigned char chroot_zt[255];
    uid_t uid;
    gid_t gid;
    int errorn, value, maxprocs, counter;
    int sock[512];
    int cache_size;
    int min_ttl_n = 300, min_ttl_c = 300;
    int timestamp_type = 0; /* Type of timestamp */
    int recursion_enabled = 0; /* Whether we have recursion */
    int max_glueless; /* Maximum allowed glueless level */
    int max_q_total; /* Maximum total queries in attempt to resolve hostname */
    int timeout; /* Maximum time to wait for a remote server when performing
                    a recursive query */
    struct sockaddr client;
    struct sockaddr_in *clin; /* So we can log the IP */
#ifndef DARWIN
    struct rlimit rlim;
#endif

    clin = (struct sockaddr_in *)&client;

    /* Initialize the strings (allocate memory for them, etc.) */
    if((mararc_loc = js_create(256,1)) == 0)
        harderror(L_MLC); /* "Could not create mararc_loc string" */
    if(js_set_encode(mararc_loc,MARA_LOCALE) == JS_ERROR)
        harderror(L_MLL); /* "Could not set locale for mararc_loc string" */
    if((errors = js_create(256,1)) == 0)
        harderror(L_EC); /* "Could not create errors string" */
    if(js_set_encode(errors,MARA_LOCALE) == JS_ERROR)
        harderror(L_EL); /* "Could not set locale for errors string" */
    if((verbstr = js_create(256,1)) == 0)
        harderror(L_VC); /* "Could not create verbstr string" */
    if(js_set_encode(verbstr,MARA_LOCALE) == JS_ERROR)
        harderror(L_VL); /* "Could not set locale for verbstr string" */
    if((kvar_query = js_create(256,1)) == 0)
        harderror(L_KQC); /* "Could not create kvar_query string" */
    if(js_set_encode(kvar_query,MARA_LOCALE) == JS_ERROR)
        harderror(L_KQL); /* "Could not set locale for kvar_query string" */
    if((bind_address = js_create(64,1)) == 0)
        harderror(L_BAC); /* "Could not create bins_address string" */
    if(js_set_encode(bind_address,MARA_LOCALE) == JS_ERROR)
        harderror(L_BAL); /* "Could not set locale for bind_address string" */
    if((incoming = js_create(768,1)) == 0)
        harderror(L_IC); /* "Could not create incoming string" */
    if(js_set_encode(incoming,MARA_LOCALE) == JS_ERROR)
        harderror(L_IL); /* "Could not set locale for incoming string" */
    if((uncomp = js_create(768,1)) == 0)
        harderror(L_UCC); /* "Could not create uncomp string" */
    if(js_set_encode(uncomp,MARA_LOCALE) == JS_ERROR)
        harderror(L_UCL); /* "Could not set locale for uncomp string" */

    /* First, find the mararc file */
    if(argc == 1) { /* No arguments */
        if(find_mararc(mararc_loc) == JS_ERROR)
            harderror(L_LOC_MARARC); /* "Error locating mararc file" */
        }
    else if(argc==2) { /* maradns -v or maradns --version */
        printf("%s %s\n%s %s\n%s\n",L_THISIS,VERSION,L_COMPILED,
               COMPILED,L_RTFM); /* "This is MaraDNS version blah blah blah */
        exit(0);
        }
    else if(argc==3) { /* maradns -f /wherever/mararc */
        if(js_qstr2js(mararc_loc,argv[2]) == JS_ERROR)
            harderror(L_MARARC_ARG); /* "Could not get mararc from command line" */
        }
    else
        harderror(L_USAGE); /* "Usage: maradns [-f mararc_location]" */

    /* Then parse that file */
    if(read_mararc(mararc_loc,errors,&errorn) == JS_ERROR) {
        harderror(L_MARARC_PARSE); /* "Error parsing contents of mararc file" */
        }
    if(errorn != 0) {
        /* Print this out at log level 0 because it is a fatal error */
        if(errorn != -1)
          /* "Error parsing contents of mararc file on line " */
          printf("%s%d%s",L_MARARC_LINE,errorn,L_N); /* errorn, "\n" */
        printf("%s",L_ERROR_CODE); /* "Error code: " */
        js_show_stdout(errors);
        printf("%s",L_N); /* "\n" */
        exit(2);
        }

    /* For the Windows version, start it as a service if requested */
    if ( 0 != read_numeric_kvar("win9x_service",0) ) {
#ifdef __CYGWIN__
        start_win9x_service();
#else
        printf("I'm not on Windows, so I can't run as a Windows service\n");
#endif
        }

    /* There are too many greedy lawyers in the US */
    if(js_qstr2js(kvar_query,"hide_disclaimer") == JS_ERROR)
        harderror(L_KVAR_Q); /* "Could not create kvar_query" */
    if(read_kvar(kvar_query,verbstr) != JS_SUCCESS) {
        printf("%s","THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS OR\n");
        printf("%s","IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES\n");
        printf("%s","OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.\n");
        printf("%s","IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,\n");
        printf("%s","INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES\n");
        printf("%s","(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR\n");
        printf("%s","SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)\n");
        printf("%s","HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,\n");
        printf("%s","STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING\n");
        printf("%s","IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE\n");
        printf("%s","POSSIBILITY OF SUCH DAMAGE.\n");
        printf("\nTo not display this message, add the follwing to your ");
        printf("mararc file:\n\nhide_disclaimer = \"YES\"\n\n");
        }
    /* Get in to a state of least privledge ASAP */

    /* Limit the maximum number of processes */
    maxprocs = read_numeric_kvar("maxprocs",64);
    if(maxprocs == 0) {
        maxprocs = 64;
        }

    if(maxprocs > 500) {
        maxprocs = 500;
        mlog(L_MAXPROC_MAX); /* "Maxprocs can not be greater than 500" */
        }
#ifndef DARWIN
    rlim.rlim_cur = rlim.rlim_max = maxprocs;

    /* If this OS supports setrlimit and if setrlimit fails, bail (the ENOSYS
       check is there so OSes w/o setrlimit support can still run MaraDNS) */
#ifdef RLIMIT_NPROC
    if(setrlimit(RLIMIT_NPROC,&rlim) != 0 && errno != ENOSYS)
        sys_harderror(L_MAXPROC_SET); /* "Unable to set maximum number of processes" */
#endif /* SOLARIS */
#endif /* DARWIN */

    /* Determine the level of error reporting */
    log_level = read_numeric_kvar("verbose_level",1);
    init_rlog_level(log_level);

    /* Set the timestamp type */
    timestamp_type = read_numeric_kvar("timestamp_type",0);
    set_timestamp(timestamp_type);

    /* Get the minttl values from the kvar database (if there) */
    min_ttl_n = read_numeric_kvar("min_ttl",300);
    min_ttl_c = read_numeric_kvar("min_ttl_cname",min_ttl_n);
    /* Set the values */
    set_min_ttl(min_ttl_n,min_ttl_c);

    /* Determine if we should make a "best faith" effort to have
       no MaraDNS-specific features */
    no_fingerprint = read_numeric_kvar("no_fingerprint",0);

    /* Determine if we should return NS and SOA records when given a RR_ANY
       query */
    rrany_set = read_numeric_kvar("default_rrany_set",3);

    /* There are 3 user-customizable parameters which determine the maximum
       number of records we spit out for various chains of records */
    /* Maximum number of A records we show for a given host name in the
       additional section */
    max_ar_chain = read_numeric_kvar("max_ar_chain",1);
    /* Maximum number of records we show for any non-A chain in the answer
       or additional section */
    max_chain = read_numeric_kvar("max_chain",8);
    /* Maximum number of records we show total */
    max_total = read_numeric_kvar("max_total",20);

    /* Determine how many elements the cache can have */
    cache_size = read_numeric_kvar("maximum_cache_elements",1024);
    if(cache_size < 32 || cache_size > 268435455) /* 2^28 - 1 */ {
        mlog(L_INVALID_CACHESIZE);
        cache_size = 1024;
        }

    /* Determine what the ACL is for recursive queries */
    /* Initialize the ACL list */
    for(counter = 0; counter < 511; counter++)
        recurse_acl[counter].ip = 0xffffffff;
    /* Read in the ACL list from the mararc file */
    if(js_qstr2js(kvar_query,"recursive_acl") == JS_ERROR)
        harderror(L_KVAR_Q);
    if(read_kvar(kvar_query,verbstr) == JS_SUCCESS) {
        recursion_enabled = 1;
        /* If recursive ACL is set, then we set all the variables
           which use recursion */
        if(make_ip_acl(verbstr,recurse_acl,500,0) == JS_ERROR)
            harderror(L_ACL_LIST_RECURSE); /* "Could not make ip ACL list" */

        /* Determine what the maximum glueless level is */
        max_glueless = read_numeric_kvar("max_glueless_level",10);
        if(max_glueless == 0)
            harderror(L_INVALID_MAXGLUE); /* max_glueless_level needs to be a number, and be greater than zero */

        /* Determine the total numer of queries to perform in a recursive
           query */
        max_q_total = read_numeric_kvar("max_queries_total",32);
        if(max_q_total == 0)
            harderror(L_INVALID_MAXQTOTAL); /* max_queries_total needs to be a number, and be greater than zero */

        /* Determine the maximum time to wait for a remote server (in
           seconds) */
        timeout = read_numeric_kvar("timeout_seconds",2);
        if(timeout < 1)
            harderror(L_INVALID_TIMEOUT); /* timeout_seconds needs to be a number, and be greater than zero */

        /* Load the "seed" data in to the DNS cache */
        counter = init_cache(cache_size,maxprocs,max_glueless,max_q_total,
                             timeout,read_numeric_kvar("resurrections",1));
        if(counter < 0) {
            switch(counter) {
                case -7:
                    harderror(L_SETROOTNS); /* root_servers["."] must be set in the mararc file; e.g. root_servers["."] = "198.41.0.4" */
                case -11:
                    harderror(L_BADROOTNS); /* root_servers["."] in the mararc file is invalid.; Example good value: root_servers["."] = "198.41.0.4" */
                case -14:
                    harderror(L_CHOOSEONE); /* Both root_servers and upstream_servers are set in the mararc file... please choose one  */
                default:
                    printf(L_ERROR_VALUE); /* "Error value (for software devlopers): " */
                    printf("%d\n",counter);
                    harderror(L_INITCACHE_FAIL); /* Init_cache() failed */
                }
            }

    /* Read in the list of spam-friendly DNS servers, which we will
       refuse to get data from */
    if(js_qstr2js(kvar_query,"spammers") == JS_ERROR)
        harderror(L_KVAR_Q);
    /* If there is a spam-friendly list, read it in */
    if(read_kvar(kvar_query,verbstr) == JS_SUCCESS) {
        if(init_spammers(verbstr) == JS_ERROR)
            harderror(L_INIT_SPAMMERS); /* "Could not make spammers list" */
        }
    /* Otherwise, make sure the spammers list is an empty list */
    else {
        if(init_spammers(0) == JS_ERROR)
            harderror(L_INIT_SPAMMERS); /* "Could not make spammers list" */
        }

    /* BEGIN RNG USING CODE */
    /* Determine which file to read the key from */
    if(js_qstr2js(kvar_query,"random_seed_file") == JS_ERROR)
        harderror(L_KVAR_Q);
    if(read_kvar(kvar_query,verbstr) == JS_SUCCESS) {
        counter = init_rng(verbstr);
        if(counter < 0) {
            switch(counter) {
                case -2:
                    sys_harderror(L_OPENSEED_FAIL); /* Could not open the random_seed_file */
                case -3:
                    harderror(L_NEED16BYTES); /* The random_seed_file needs to be 16 bytes or longer */
                default:
                    harderror(L_INITCRYPTO_FAIL); /* "Init_crypto() failed" */
                }
            }
        }
    else if((recurse_acl[0]).ip != 0xffffffff) {
        harderror(L_RANDOM_IF_RECURSE); /* "random_seed_file must be set if recursive DNS is enabled" */
        }
    /* END RNG USING CODE */
    }

    /* Anything after this does not need recursion enabled for the
       kvar in question to be read */

    /* Determine whether to wait before sending a reply (used only
       for debugging) */
    debug_delay = read_numeric_kvar("debug_response_delay",0);

    /* Set the debug_msg_level to the Debug message level they want */
    debug_msg_level = read_numeric_kvar("debug_msg_level",1);

    /* Determine if we are root */
    if(geteuid() == 0) {

        /* Change the root directory */
        if(js_qstr2js(kvar_query,"chroot_dir") == JS_ERROR)
            harderror(L_KVAR_Q); /* "Could not create kvar_query" */
        if(read_kvar(kvar_query,verbstr) == JS_ERROR)
            harderror(L_CHROOT_KVAR); /* "Problem getting chroot kvar.\nYou must have chroot_dir set if you start this as root" */
        if(js_js2str(verbstr,chroot_zt,200) == JS_ERROR)
            harderror(L_CHROOT_NT); /* "Problem making chroot nt string.\nMake sure the chroot directory is 200 chars or less" */
        if(chdir(chroot_zt) != 0)
            sys_harderror(L_CHROOT_CHANGE); /* "Problem changing to chroot dir.\nMake sure chroot_dir points to a valid directory" */
        if(chroot(chroot_zt) != 0)
            sys_harderror(L_CHROOT_DO);  /* "Problem changing the root directory." */

        mlog(L_CHROOT_SUCCESS); /* "Root directory changed" */

        /* Bind to port 53
           To Do: use capset to give us privledged bind abilities without
                  needing to be root.
        */
        if(js_qstr2js(kvar_query,"bind_address") == JS_ERROR)
            harderror(L_KVAR_Q); /* "Could not create kvar_query" */
        if(read_kvar(kvar_query,bind_address) == JS_ERROR)
            harderror(L_NOBIND); /* "Problem getting chroot kvar.\nYou must have bind_address set to the IP maradns will listen on" */
        for(counter = 0; counter < 512 ; counter++)
            bind_addresses[counter].ip = 0xffffffff;
        if(make_ip_acl(bind_address,bind_addresses,500,0) == JS_ERROR)
            harderror("XXX we need an error message here 3305");
        if(udpbind(sock,bind_addresses) == JS_ERROR)
            sys_harderror(L_BINDFAIL); /* "Problem binding to port 53.\nMost likely, another process is already listening on port 53" */
        zjlog(L_BIND2ADDR,bind_address); /* "Binding to address " */
        mlog(L_BIND_SUCCESS);  /* "Socket opened on UDP port 53" */

        /* Drop the elevated privledges */
        /* First, change the GID */
        gid = read_numeric_kvar("maradns_gid",99);
#ifndef __CYGWIN__
        /* Drop all supplemental groups */
        setgroups(1,&gid);
#endif
        /* Set the group ID */
        setgid(gid);

        /* Next, change the UID */
        uid = read_numeric_kvar("maradns_uid",99);
        if(uid < 10)
            harderror(L_BADUID); /* "maradns_uid is less than 10 or not a number.\nThis uid must have a value of 10 or more" */
        if(setuid(uid) != 0)
            sys_harderror(L_NODROP); /* "Could not drop root uid" */
        /* Workaround for known Linux kernel security problem circa
           early 2000 */
        if(setuid(0) == 0)
            sys_harderror(L_STILL_ROOT);  /* "We seem to still be root" */

        mlog(L_DROP_SUCCESS); /* "Root privledges dropped" */

        }
    else {

        /* Bind to port 53 as a non-root user */
        if(js_qstr2js(kvar_query,"bind_address") == JS_ERROR)
            harderror(L_KVAR_Q); /* "Could not create kvar_query" */
        if(read_kvar(kvar_query,bind_address) == JS_ERROR)
            harderror(L_NOBIND); /* "Problem getting chroot kvar.\nYou must have bind_address set to the IP maradns will listen on" */
        if(udpbind(sock,bind_addresses) == JS_ERROR)
            sys_harderror(L_BEROOT); /* "Problem binding to port 53.\nYou should run this as root" */
        mlog(L_BIND_SUCCESS);  /* "Socket opened on UDP port 53" */
        }

    /* Create the big hash */
    bighash = 0;
    bighash = mhash_create(8);
    if(bighash == 0)
        harderror(L_NOBIGHASH); /* "Could not create big hash" */

    value = populate_main(bighash,errors,recursion_enabled);
    if(value == JS_ERROR)
        harderror(L_NOPOPULATE); /* "Error running populate_main program" */
    else if(value == -2) {
        js_show_stdout(errors);
        printf("%s",L_N); /* "\n" */
        harderror(L_POPULATE_FATAL); /* "This error in populate hash is fatal" */
        }

    mlog(L_RRS_LOADED);  /* "All RRs have been loaded" */

    /* Enable signal handler for SIGPIPE */
    /* (Signals disabled until I can determine if the "signal race
       condition" attack affects MaraDNS) */
    /*signal(SIGPIPE,handle_sigpipe);*/
    /* Some other catchable signals */
    /*signal(SIGALRM,handle_signal);*/
    /*signal(SIGUSR1,handle_signal);*/
    /*signal(SIGUSR2,handle_signal);*/
    /* Right now, all we do after getting a HUP signal is exit with a code
       of 8.  The static DNS database is too tangley for me to figure out 
       how to clear all of the memory it uses quickly; there is too much 
       chance of having a memory leak with this, so I don't feel comfortable 
       doing the right thing after getting a HUP signal until Franky comes
       back to help make sure HUP handling is memory-leak free */
    signal(SIGHUP,handle_hup); /* All this does is change the got_hup_signal
                                  global variable */
#ifdef DEBUG
    signal(SIGTERM,display_unfreed);
    signal(SIGINT,display_unfreed);
#endif

    /* Initialize the new decompression code */
    /* Disabled until I can get the bugs worked out */
    decomp_init(log_level);

    /* Flush out any messages that have already appeared */
    fflush(stdout);

    if(log_level >= 3)
        mlog(L_DATAWAIT); /* "Awaiting data on port 53" */
    /* Listen for data on the UDP socket */
    for(;;) {
        int sock_num;
        /* Make sure we never got a HUP signal */
	if(got_hup_signal != 0) {
            printf("HUP signal sent to MaraDNS process\n");
            printf("Exiting with return value of 8\n");
	    exit(8);
	    }
        /* Update the timestamp; this needs to be run once a second */
	josa_set_time();
        if(log_level >= 50) /* This happens once a second */
            mlog(L_DATAWAIT); /* "Awaiting data on port 53" */
        sock_num = getudp(sock,bind_addresses,
	                  (struct sockaddr *)&client,incoming,512); 
        if(sock_num == JS_ERROR)
            continue;
        if(log_level >= 3)
            mlog(L_GOTDATA);     /* "Message received, processing" */
        if(decompress_data(incoming,uncomp) == JS_ERROR) {
            if(log_level >= 4) {
                show_timestamp();
                printf("%s ","Query from");
                debug_show_ip(ntohl(clin->sin_addr.s_addr));
                printf("has decompression error: ");
                show_esc_stdout(incoming);
                printf("\n");
                }
            continue;
            }
        if(log_level >= 5) {
            show_timestamp();
            printf("Decompressed packet: ");
            show_esc_stdout(uncomp);
            printf("\n");
            }
        if(log_level >= 3 && uncomp->unit_count > 12) {
            /* Show them the query */
            counter = dlabel_length(uncomp,12);
            value = js_readuint16(uncomp,12+counter);
            if(js_substr(uncomp,incoming,12,counter) != JS_ERROR) {
                hname_translate(incoming,value);
                /* Yes, I know, put these in the "to localize" header file */
                show_timestamp();
                printf("%s: ","Query from");
                debug_show_ip(ntohl(clin->sin_addr.s_addr));
                printf(" ");
                js_show_stdout(incoming);
                printf("\n");
                }
            }
        /* Delay the processing the request, as needed */
        if(debug_delay > 0)
            sleep(debug_delay);
        proc_query(uncomp,(struct sockaddr_in *)&client,sock_num);
        }

    /* We should never end up here */

    exit(7); /* Exit code 7: Broke out of loop somehow */

    }

