/* $Id: output_apropos.c,v 1.12 2003/04/17 03:03:41 jtalkington Exp $ */

#include "common.h"
#include "output_apropos.h"

#ifndef DISABLE_APROPOS
/* the tree of all of the apropos_node s
 */
static struct section_node *apropos_tree = NULL;

/* output_apropos()
 * outputs the results of apropos sorted by section and with references back
 * to the program's URL
 */
void
output_apropos(char *keyword) {
	FILE *ap_fh = NULL;
	char line[BUFSIZ];
	char *command = NULL;

	if((Config.apropos != NULL) && ((Config.disabled & D_APROPOS) == 0)) {
		command = strdup(Config.apropos);
	} else {
		output_apropos_denied();
		return;
	}


	command = str_app_s(command, " ");
	command = str_app_s(command, keyword);

	output_header("Keyword search", keyword);
	if((ap_fh = popen(command, "r")) == NULL) {
		printf("<h2>Error opening pipe to: %s</h2>\n", command);
		output_footer();
		return;
	} else {
		while(fgets(line, sizeof(line), ap_fh) != NULL) {
			apr_add_line(line);
		}
		
		printf("<h2>Keyword search for %s results:</h2>\n<hr />\n", keyword);
		printf("<table width=\"100%%\">\n");
		output_section_node(&apropos_tree);
		printf("</table>\n");
		output_footer();
		pclose(ap_fh);
#ifdef M2W_CLEANUP
		cleanup_apropos_section(&apropos_tree);
		apropos_tree = NULL;
#endif /* M2W_CLEANUP */
	}

	free(command);

}

/* break down the line and add links if we have are not in command mode */
void
apr_add_line(char *line) {
	char *linked_line = NULL;
	char *program = NULL;
	char *section = NULL;
	char *prog_start = NULL;
	char *prog_end = NULL;
	char *sec_start = NULL;
	char *sec_end = NULL;
	char *white_start = NULL;
	char *script_name = get_query_value(SCRIPT_NAME);
	char *prog_extra = calloc(1,1);
	
	if(line == NULL) {
		return;
	}

	/* kill the newline */
	memset(line + strlen(line) - 1, '\0', 1);
	
	prog_start = line;
	prog_end = prog_start;

	/* get the program name */
	while(*prog_end != ' ' && *prog_end != '\t' && *prog_end != '(') {
		prog_end++;
	}

	program = strndup(prog_start, prog_end - prog_start);

	/* extract the section */
	sec_start = strchr(prog_end, '(');

	if(sec_start != NULL) {

		sec_end = strchr(sec_start, ')');

		if(sec_end != NULL) {

			section = strndup(sec_start + 1, sec_end - (sec_start + 1));

			white_start = sec_start;

			while(white_start > prog_end && (*(white_start - 1) == ' ' || *(white_start - 1) == '\t')) {
				white_start--;
			}

			/* get any non whitespace extras after the program name but before
			 * the section start (for entries with [foo] after the program name
			 */

			while(prog_end < white_start) {
				prog_extra = str_app_c(prog_extra, *prog_end);
				prog_end++;
			}

			/* get to the start of the description (-) */
			while(*sec_end != '-') {
				sec_end++;
			}

			if(script_name != NULL) {
				linked_line = calloc(1, strlen(APROPOS_HREF) + (2 * (strlen(program) + 1)) + (strlen(prog_extra) + 1) + (strlen(section) + 1) + (strlen(script_name) + 1) + strlen(line) + 1);
				sprintf(linked_line, APROPOS_HREF, script_name, program, section, program, prog_extra, section, sec_end);

			} else {
				linked_line = strdup(line);
			}
		} else {
			linked_line = strdup(line);
		}
	} else {
		linked_line = strdup(line);
	}

	apr_add_to_tree(&apropos_tree, program, section, linked_line);

	if(program != NULL) {
		free(program);
		program = NULL;
	}

	if(prog_extra != NULL) {
		free(prog_extra);
		prog_extra = NULL;
	}

	if(section != NULL) {
		free(section);
		section = NULL;
	}

	free(linked_line);
	linked_line = NULL;
}

/* apr_add_to_tree()
 * adds an apropos section node and program node to the tree
 */
void
apr_add_to_tree(struct section_node **sec_node, char *program, char *section, char *line) {
	int result = 0;

	if((*sec_node) == NULL) {
		(*sec_node) = calloc(1, sizeof(struct section_node));
		
		(*sec_node)->left = NULL;
		(*sec_node)->right = NULL;
		(*sec_node)->data = NULL;
		if(section != NULL) {
			(*sec_node)->name = strdup(section);
		} else {
			(*sec_node)->name = strdup(program);
		};
		apr_add_prog(&(*sec_node)->data, program, line);
	} else {
		result = strcmp((*sec_node)->name, section);
		
		if(result == 0) {
			apr_add_prog(&(*sec_node)->data, program, line);
		} else if(result < 0) {
			apr_add_to_tree(&(*sec_node)->right, program, section, line);
		} else {
			apr_add_to_tree(&(*sec_node)->left, program, section, line);
		}
	}
}

/* apr_add_prog()
 * adds a program to an apropos_node tree
 */
void
apr_add_prog(struct apropos_node **node, char *program, char *line) {
	int result = 0;
	
	if((*node) == NULL) {
		(*node) = calloc(1, sizeof(struct apropos_node));

		(*node)->left = NULL;
		(*node)->right = NULL;
		(*node)->name = strdup(program);
		(*node)->line = strdup(line);
	} else {
		result = strcmp((*node)->name, program);

		if(result == 0) {
			/* it already exists, so just dump it */
			return;
		} else if(result < 0) {
			apr_add_prog(&(*node)->right, program, line);
		} else {
			apr_add_prog(&(*node)->left, program, line);
		}
	}
}

/* output_section_node()
 * outputs all of the elements of the apropos_tree recursively (and therefore
 * sorted by section
 */
void
output_section_node(struct section_node **section) {
	
	if((*section) == NULL) {
		return;
	}

	if((*section)->left != NULL) {
		output_section_node(&(*section)->left);
	}
	
	output_apropos_node(&(*section)->data);

	if((*section)->right != NULL) {
		output_section_node(&(*section)->right);
	}
}

/* output_apropos_node()
 * outputs the program node of a section recursively
 */

void
output_apropos_node(struct apropos_node **node) {
	
	if((*node) == NULL) {
		return;
	}

	if((*node)->left != NULL) {
		output_apropos_node(&(*node)->left);
	}
	
	printf("%s\n", (*node)->line);

	output_apropos_node(&(*node)->right);
}

#ifdef M2W_CLEANUP

/* cleanup_apropos_section()
 * cleans up a section node of the apropos_tree
 */
void
cleanup_apropos_section(struct section_node **node) {
	struct section_node *right = NULL;

	if((*node) == NULL) {
		return;
	} else {
		right = (*node)->right;
	}

	if((*node)->left != NULL) {
		cleanup_apropos_section(&(*node)->left);
		(*node)->left = NULL;
	} 
	
	cleanup_apropos_node(&(*node)->data);
	(*node)->data = NULL;

	free((*node)->name);
	(*node)->name = NULL;

	free(*node);

	if(right != NULL) {
		cleanup_apropos_section(&right);
	} 
}

/* cleanup_apropos_node()
 * cleans up the apropos_node part of a section node
 */
void
cleanup_apropos_node(struct apropos_node **node) {
	struct apropos_node *right = NULL;
	
	if((*node) == NULL) {
		return;
	} else {
		right = (*node)->right;
	}

	if((*node)->left != NULL) {
		cleanup_apropos_node(&(*node)->left);
		(*node)->left = NULL;
	}

	free((*node)->name);
	(*node)->name = NULL;

	free((*node)->line);
	(*node)->line = NULL;

	free(*node);

	if(right != NULL) {
		cleanup_apropos_node(&right);
	}
}
#endif /* M2W_CLEANUP */

#else /* DISABLE_APROPOS */

/* wrapper for if apropos is disabled */
void
output_apropos(char *keyword) {
	output_apropos_denied();
}

#endif /* DISABLE_APROPOS */

/* prints out a page if apropos searching is disabled */
void
output_apropos_denied() {
	output_header("Keyword Search Disabled", NULL);
	printf("<h2>Keyword (apropos) search disabled</h2>\n");
	output_footer();
}
