/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 */
/*

  See the README file in this directory for info on compiling, using
  and the state of this program. If the readme file is for some reason
  not there, it can be found at the MAL website:

  http://corp.avantgo.com/mal/


*/
#include <stdio.h>
#include <signal.h>

#include <AGUtil.h>
#include <AGClientProcessor.h>
#include <AGProtocol.h>
#include <AGBufferReader.h>
#include <AGPalmProtocol.h>
#include <AGUserConfig.h>
#include <AGServerConfig.h>
#include <AGSyncCommon.h>
#include <AGCommandProcessor.h>
#include <AGDesktopInfoPalm.h>
#include <AGTimeoutMessageBox.h>
#include <AGMobileLinkSettings.h>


#include <pi-source.h>
#include <pi-socket.h>
#include <pi-file.h>
#include <pi-dlp.h>
#include <pi-version.h>

#define DEFAULT_CARD_NUM 0

static void Connect();
static char *device = "/dev/pilot";
static char *httpProxy      = NULL;
static int   httpProxyPort  = 0;
static char *socksProxy     = NULL;
static int   socksProxyPort = 0;
static char *proxyUsername  = NULL;
static char *proxyPassword  = NULL;
static char *progname       = NULL;
#ifdef _DEBUG
#define debug_print(x) printf("%s\n", (x));
#else
#define debug_print(x)
#endif

typedef struct {

    AGDeviceInfo * deviceInfo;
    AGUserConfig * userConfig;
    AGServerConfig * currentServerConfig;
    AGClientProcessor * clientProcessor;
    AGPlatformCalls * platform;
    AGRecord * record;
    AGDBConfig * currentDb;
    AGCommandProcessor *commandProcessor;
    AGBool quit;
    
    /* Pilot-specific */
    AGBool currentDBIsResourceType;
    int pilot_rHandle;
    int pilot_RecIndex;
    recordid_t id;
    uint8  *pilot_buffer;
    int pilot_buffer_size;
    
    /* Secure Network Library Stuff */
    AGBool          hasseclib;

} PalmSyncInfo;

static int sd = 0;

static int32 initAndOpenDatabase(void *out, AGDBConfig *theDatabase, 
                                 int32 *errCode);
static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                   int32 *errCode);
static int32 getNextRecord(void * out, AGRecord ** record,
                           int32 *errCode);

static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask,
                     AGBool bufferable);
static int32 cmdITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem);
static int32 cmdDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname);
static int32 cmdOPENDATABASE(void *out, int32 *returnErrorCode,
                                           char *dbname);
static int32 cmdCLOSEDATABASE(void *out, int32 *returnErrorCode);
static int32 cmdCLEARMODS(void *out, int32 *returnErrorCode);
static int32 cmdGOODBYE(void *out, int32 *returnErrorCode,
                        AGSyncStatus syncStatus,
                        int32 errorCode,
                        char *errorMessage);
static int32 cmdRECORD(void *out, int32 *returnErrorCode,
                       int32 *newUID,
                       int32 uid,
                       AGRecordStatus mod,
                       int32 recordDataLength,
                       void *recordData,
                       int32 platformDataLength,
                       void *platformData);

static int processCommandLine(int argc, char *argv[], PalmSyncInfo *pi);

/* ----------------------------------------------------------------------------*/
void syncInfoFree(PalmSyncInfo * pInfo)
{

    if (NULL != pInfo) {

        if (NULL != pInfo->platform)
            free(pInfo->platform);

        if (NULL != pInfo->userConfig)
            AGUserConfigFree(pInfo->userConfig);

        if (NULL != pInfo->pilot_buffer)
            free(pInfo->pilot_buffer);

        if (NULL != pInfo->commandProcessor)
            AGCommandProcessorFree(pInfo->commandProcessor);

        free(pInfo);
    }
}

/* ----------------------------------------------------------------------------*/
PalmSyncInfo * syncInfoNew()
{
    PalmSyncInfo * pInfo;

    /* Allocate the PalmSyncInfo structure. */
    pInfo = (PalmSyncInfo *)malloc(sizeof(PalmSyncInfo));
    if (NULL != pInfo) {

        const int pbs = 65535;

        bzero(pInfo, sizeof(PalmSyncInfo));

        pInfo->pilot_buffer_size    = pbs;
        pInfo->pilot_buffer         = (uint8 *)malloc(pbs);
        if (NULL == pInfo->pilot_buffer)
            goto fail;

        /* Allocate the platform calls record. */
        pInfo->platform = (AGPlatformCalls *)malloc(sizeof(AGPlatformCalls));
        bzero(pInfo->platform, sizeof(AGPlatformCalls));
        if (NULL == pInfo->platform)
            goto fail;

        return pInfo;
    }

fail:
    if (NULL != pInfo)
        syncInfoFree(pInfo);

    return NULL;
}
/* ----------------------------------------------------------------------------*/
static AGBool setupPlatformCalls(PalmSyncInfo * pInfo)
{
    pInfo->platform->out = pInfo;
    pInfo->platform->nextModifiedRecordFunc = getNextModifiedRecord;
    pInfo->platform->nextRecordFunc = getNextRecord;
    pInfo->platform->openDatabaseFunc =  initAndOpenDatabase;
    return FALSE;
}
/* ----------------------------------------------------------------------------*/
static int readDeviceInfo(PalmSyncInfo * pInfo)
{
    AGDeviceInfo * devInfo;
    struct SysInfo sysInfo;
    struct CardInfo cardInfo;
    int majorver, minorver;
    char osverstring[24];

    /* Start with clean slate. */
    devInfo = pInfo->deviceInfo;

    if (dlp_ReadSysInfo(sd, &sysInfo) < 0)
        return -1;

    cardInfo.card = DEFAULT_CARD_NUM;
    if (dlp_ReadStorageInfo(sd, DEFAULT_CARD_NUM, &cardInfo) < 0)
        return -1;

    /* Find out version of operating system. */
    majorver = sysInfo.romVersion >> 24;
    minorver = 0xff & (sysInfo.romVersion >> 16);
    if (minorver < 10) 
        minorver = 0;
    
    snprintf(osverstring, 24, "%d.%d", majorver, minorver);

    devInfo->availableBytes   = cardInfo.ramFree;
    devInfo->serialNumber     = strdup("");
    devInfo->osName           = strdup("PALM_OS");
    devInfo->osVersion        = strdup(osverstring);
    devInfo->screenWidth      = 150;
    devInfo->screenHeight     = 150;
    if(majorver > 2)
       devInfo->colorDepth    = 2; 
    else
       devInfo->colorDepth    = 1; 

    return 0;
}
/* ----------------------------------------------------------------------------*/
static AGBool getPalmDatabaseCreationInfo(AGDBConfig * db, uint32 * creator,
                                          uint32 * flags, uint32 * type)
{
    AGBufferReader * r = NULL;
    debug_print("GetPalmDatabaseCreationInfo()\n");

    if (NULL == db)
        return FALSE;

    if (0 == db->platformDataLength || NULL == db->platformData)
        return FALSE;

    r = AGBufferReaderNew((uint8*)db->platformData);
    if (NULL == r)
        return FALSE;
    
    AGPalmReadDBConfigPlatformData((AGReader*)r, creator, type, flags);
    AGBufferReaderFree(r);
    return TRUE;
}

/* ----------------------------------------------------------------------------*/
static int createDatabase(AGDBConfig *db)
{
    int dbhandle;
    long creator;
    int flags;
    int cardNo = DEFAULT_CARD_NUM;
    long type;
    int version = 0;
    int err;
    
    if (!db)
        return 0;

#ifdef _DEBUG
    prinf("createDatabase\n");
#endif
    
    getPalmDatabaseCreationInfo(db, (uint32*)&creator, 
                                (uint32*)&flags, (uint32*)&type);

    if ((err = dlp_CreateDB(sd,  creator,  type, cardNo, flags,  
                            version, db->dbname, &dbhandle)) < 0) {
#ifdef _DEBUG
        prinf("createDatabase: dlp_CreateDB failed err = %d\n", err);
#endif
        
        dbhandle = 0;
    }
    
    return dbhandle;
}

/* ----------------------------------------------------------------------------*/
static long openDatabase(PalmSyncInfo *pInfo, char *dbname, AGBool create)
{

    long result;
    
    if (!dbname || !pInfo) {
        debug_print("\n");
        return -1;
    }
    debug_print("... opening '");
    debug_print(dbname);
    debug_print("' ... ");

    pInfo->currentDb =
        AGServerConfigGetDBConfigNamed(pInfo->currentServerConfig, dbname);
    
    result = dlp_OpenDB(sd, DEFAULT_CARD_NUM, dlpOpenRead|dlpOpenWrite,  
                        dbname, &pInfo->pilot_rHandle);
    
    if ((result < 0) && create)
        pInfo->pilot_rHandle = createDatabase(pInfo->currentDb);

    if (pInfo->pilot_rHandle) {
        uint32 creator, flags, type;
        if (getPalmDatabaseCreationInfo(pInfo->currentDb, &creator, &flags, 
                                        &type) && (flags & 0x01) != 0) {
            pInfo->currentDBIsResourceType = TRUE;
        } else
            pInfo->currentDBIsResourceType = FALSE;
        debug_print("successfully.\n");
    } else {
        debug_print("unsuccessfully.\n");
        pInfo->currentDBIsResourceType = FALSE;
        pInfo->currentDb = NULL;
    }

    return result;
}

/* ----------------------------------------------------------------------------*/
static long closeDatabase(PalmSyncInfo * pInfo)
{
    int result;

    result = dlp_CloseDB(sd, pInfo->pilot_rHandle);
    pInfo->pilot_rHandle = 0;
    pInfo->currentDb = NULL;
    pInfo->currentDBIsResourceType = FALSE;
    return result;
}

/* ----------------------------------------------------------------------------*/
static void deleteDatabase(char * dbname)
{
    debug_print("deleteDatabase()\n");
    dlp_DeleteDB(sd, DEFAULT_CARD_NUM, dbname);
}
/* ----------------------------------------------------------------------------*/
static void clearMods(long dbHandle)
{
    debug_print("clearMods()\n");
    dlp_CleanUpDatabase(sd, dbHandle);
    dlp_ResetSyncFlags(sd, dbHandle);

}

/* ----------------------------------------------------------------------------*/
static long openUserConfigDatabase(void)
{
    long result;
    int userConfigDBHandle = 0;

    result = dlp_OpenDB(sd, DEFAULT_CARD_NUM, dlpOpenRead|dlpOpenWrite,  
                        DEVICE_USERCONFIG_DB_NAME, 
                        &userConfigDBHandle);

    if (result < 0) {

        result = dlp_CreateDB(sd, DEVICE_USERCONFIG_DB_CREATOR, 
                              DEVICE_USERCONFIG_DB_TYPE, DEFAULT_CARD_NUM,
                              0, 0, DEVICE_USERCONFIG_DB_NAME, 
                              &userConfigDBHandle);
        if (result < 0)
            userConfigDBHandle = 0;

    }
    return userConfigDBHandle;
}
#define BUFFERSIZE 0xFFFF
/* ----------------------------------------------------------------------------*/
static int32 readDeviceUserConfig(int userConfigDBHandle,
                                  AGUserConfig **deviceUserConfig)
{
    recordid_t id;
    int bufferSize = BUFFERSIZE;
    int attr = 0;
    int cat  = 0;
    int rc;
    uint8 buffer[BUFFERSIZE];
    AGBufferReader * r = NULL;

    rc = dlp_ReadRecordByIndex(sd, userConfigDBHandle, 0, (void *)buffer, 
                               &id, &bufferSize, &attr, &cat);
    
    if (rc < 0) {
#ifdef _DEBUG
        printf("readDeviceUserConfig: dlp_ReadRecordByIndex failed, err = %d\n",
               rc);
#endif
        return 0;
    }
    
    r = AGBufferReaderNew(buffer);
    if (r) {
        *deviceUserConfig = AGUserConfigNewAndReadData((AGReader*)r);
        AGBufferReaderFree(r);
        return id;
    } else
        return 0;
}

/* ----------------------------------------------------------------------------*/
static void writeDeviceUserConfig(int userConfigDBHandle,
                                  AGUserConfig * deviceUserConfig,
                                  recordid_t *recID)
{
    AGBufferWriter * w = NULL;
    w = AGBufferWriterNew(0);
    if (w) {
        long result;
        AGUserConfigWriteData(deviceUserConfig, (AGWriter*)w);
        result =  dlp_WriteRecord(sd, userConfigDBHandle, 0,
                                  *recID, 0, (void *)AGBufferWriterGetBuffer(w), 
                                  AGBufferWriterGetBufferSize(w), 
                                  recID);
        AGBufferWriterFree(w);
    }
}

/* ----------------------------------------------------------------------------*/
static AGUserConfig *getUserConfig(uint32 * pilotID)
{

    /* The device is the truth. There is no way to set these values on 
       the desktop */
    AGUserConfig * deviceUserConfig = NULL;
    int userConfigDBHandle = 0;

    /* Get device's record. */
    userConfigDBHandle = openUserConfigDatabase();
    if (userConfigDBHandle) {
        
        /* Retrieve device's idea of current userConfig. */
        *pilotID = readDeviceUserConfig(userConfigDBHandle,
                                        &deviceUserConfig);
        
        /* Done with database for now, so close it. */
        dlp_CloseDB(sd, userConfigDBHandle);
        

    }
    return deviceUserConfig;
}

/* ----------------------------------------------------------------------------*/
static void storeDeviceUserConfig(AGUserConfig *userConfig, recordid_t id)
{
    int userConfigDBHandle = openUserConfigDatabase();
    if (0 != userConfigDBHandle) {
        writeDeviceUserConfig(userConfigDBHandle,
                              userConfig, &id);
        dlp_CloseDB(sd, userConfigDBHandle);
    }
}

/* ----------------------------------------------------------------------------*/
static void doStartServer(PalmSyncInfo * pInfo,
                          AGServerConfig *sc,
                          int32 *errCode)
{
    pInfo->currentServerConfig = sc;
    if(pInfo->commandProcessor) {
        AGCommandProcessorFree(pInfo->commandProcessor);
        pInfo->commandProcessor = NULL;
    }
    pInfo->commandProcessor = AGCommandProcessorNew(sc);
    pInfo->platform->performCommandOut = pInfo->commandProcessor;
    pInfo->platform->performCommandFunc = 
                    AGCommandProcessorGetPerformFunc(pInfo->commandProcessor);

    pInfo->commandProcessor->commands.out = pInfo;

    pInfo->commandProcessor->commands.performTaskFunc = cmdTASK;
    pInfo->commandProcessor->commands.performItemFunc = cmdITEM;
    pInfo->commandProcessor->commands.performDeleteDatabaseFunc = cmdDELETEDATABASE;
    pInfo->commandProcessor->commands.performOpenDatabaseFunc = cmdOPENDATABASE;
    pInfo->commandProcessor->commands.performCloseDatabaseFunc = cmdCLOSEDATABASE;
    pInfo->commandProcessor->commands.performClearModsFunc = cmdCLEARMODS;
    pInfo->commandProcessor->commands.performGoodbyeFunc = cmdGOODBYE;
    pInfo->commandProcessor->commands.performRecordFunc = cmdRECORD;
    
}

/* ----------------------------------------------------------------------------*/
static void doEndServer(PalmSyncInfo * pInfo,
                        int32 *errCode)
{
    pInfo->currentServerConfig = NULL;
    if(pInfo->commandProcessor) {
        AGCommandProcessorFree(pInfo->commandProcessor);
        pInfo->commandProcessor = NULL;
    }
}
/* ----------------------------------------------------------------------------*/
static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask, 
                     AGBool bufferable)
{
    if (currentTask)
        printf("%s\n", currentTask);
    return AGCLIENT_CONTINUE;
}
/* ----------------------------------------------------------------------------*/
static int32 cmdITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem)
{
    printf(".");
    fflush(stdout);
    if (currentItemNumber == totalItemCount)
        printf("\n");
    
    return AGCLIENT_CONTINUE;
}
/* ----------------------------------------------------------------------------*/
static int32 cmdDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname)
{
    debug_print("doCmdAG_DELETEDATABASE_CMD()\n");

    if (NULL != dbname) {
        debug_print("... trying to delete database ");
        debug_print(dbname);
        debug_print(" from device.\n");
        deleteDatabase(dbname);
    }
    return AGCLIENT_CONTINUE;
}
/*----------------------------------------------------------------------------*/
static int32 cmdOPENDATABASE(void *out, int32 *returnErrorCode, char *dbname)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_OPENDATABASE_CMD()\n");

    if (NULL != dbname) {
        long result;
        /* We assign a result code here only for debugging.  It's ok for an
        open to fail here. */
        result = openDatabase(pInfo, dbname, TRUE);
    }
    return AGCLIENT_CONTINUE;
}
/*----------------------------------------------------------------------------*/
static int32 cmdCLOSEDATABASE(void *out, int32 *returnErrorCode)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_CLOSEDATABASE_CMD()\n");
    closeDatabase(pInfo);
    return AGCLIENT_CONTINUE;
}
/*----------------------------------------------------------------------------*/
static int32 cmdCLEARMODS(void *out, int32 *returnErrorCode)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_CLEARMODS_CMD()\n");
    clearMods(pInfo->pilot_rHandle);
    return AGCLIENT_CONTINUE;
}
/*----------------------------------------------------------------------------*/
static int32 cmdGOODBYE(void *out, int32 *returnErrorCode,
                        AGSyncStatus syncStatus,
                        int32 errorCode,
                        char *errorMessage)
{
    debug_print("doCmdAG_GOODBYE_CMD()\n");

    if (errorMessage)
        printf("%s\n", errorMessage);

    return AGCLIENT_CONTINUE;
}
/*----------------------------------------------------------------------------*/
int getIndexFromPlatformData(uint8 *platformData)
{
    int16 recIndex;
    AGBufferReader reader;

    if (!platformData)
        return 0;

    AGBufferReaderInit(&reader, platformData);
    AGPalmReadRecordPlatformData((AGReader *)&reader, &recIndex);
    AGBufferReaderFinalize(&reader);
    return (int)recIndex;
}
/*----------------------------------------------------------------------------*/
static int32 cmdRECORD(void *out, int32 *returnErrorCode,
                       int32 *newUID,
                       int32 uid,
                       AGRecordStatus mod,
                       int32 recordDataLength,
                       void *recordData,
                       int32 platformDataLength,
                       void *platformData)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    
    debug_print("doCmdAG_RECORD_CMD()\n");

    if(mod == AG_RECORD_NEW_TEMPORARY_UID) {
        uid = 0; 
    }

    if (AG_RECORD_DELETED == mod) {

        dlp_DeleteRecord(sd, pInfo->pilot_rHandle, 0, uid);

    } else if (recordDataLength <= 0x0000ffff) {

        if (pInfo->currentDBIsResourceType) {
            
            dlp_WriteRecord(sd, pInfo->pilot_rHandle, dlpDBFlagResource,
                            uid, 0, (void *)recordData,
                            recordDataLength, (recordid_t *)newUID);
            debug_print("doCmdAG_RECORD_CMD()\n");


        } else {

            dlp_WriteRecord(sd, pInfo->pilot_rHandle, 0,
                            uid, 0, (void *)recordData,
                            recordDataLength, (recordid_t *)newUID);
        }

    }
    return AGCLIENT_CONTINUE;
}

static int32 initAndOpenDatabase(void *_pInfo, AGDBConfig *db,
                                    int32 *errCode)
{
    long result;
    PalmSyncInfo * pInfo = (PalmSyncInfo *)_pInfo;

    if (NULL == db->dbname) {
        *errCode = AGCLIENT_OPEN_ERR;
        return AGCLIENT_ERR;
    }

    result = openDatabase(pInfo, db->dbname, FALSE);
    if (result < 0) {
        if (result == dlpErrNotFound)
            *errCode = AGCLIENT_OPEN_ERR;
        else
            *errCode = AGCLIENT_UNKNOWN_ERR;
        return AGCLIENT_ERR;
    }

    pInfo->pilot_RecIndex = 0;
    pInfo->record = AGRecordNew(0, AG_RECORD_UNMODIFIED, 0, 0, 0, 0);
    if (!pInfo->record) {
        *errCode = AGCLIENT_OPEN_ERR;
        return AGCLIENT_ERR;
    }

    return AGCLIENT_IDLE;
}

static int32 leaveGetRecord(PalmSyncInfo * pInfo, int32 result)
{
    if (pInfo->record) {

        /* Set recordData to NULL so that AGRecordFree doesn't try
        to free it (we own that buffer and will release it when the
        program ends). */
        pInfo->record->recordData = NULL;

        AGRecordFree(pInfo->record);
        pInfo->record = NULL;
    }
    return result;
}

static int32 getRecordBase(PalmSyncInfo * pInfo, AGBool modonly, 
                           AGRecord ** record, int32 * errCode)
{
    int32 result;
    int att = 0;
    int cat = 0;
    int size = pInfo->pilot_buffer_size;
    int idx   = pInfo->pilot_RecIndex++;

    result = (modonly) ?
        dlp_ReadNextModifiedRec (sd, pInfo->pilot_rHandle, pInfo->pilot_buffer,
                                 &pInfo->id, &idx,
                                 &size, &att, &cat)
        :
        dlp_ReadRecordByIndex(sd, pInfo->pilot_rHandle, idx,
                              pInfo->pilot_buffer, &pInfo->id,
                              &size, &att, &cat);
    
    if (result < 0) {
        closeDatabase(pInfo);
        if (result == dlpErrNotFound) {
            debug_print("(successfully reached end of records ...)\n");
            return leaveGetRecord(pInfo, AGCLIENT_IDLE);
        }
        else {
            *errCode = AGCLIENT_UNKNOWN_ERR;
            return leaveGetRecord(pInfo, AGCLIENT_ERR);
        }
    }
    pInfo->record = AGRecordInit(pInfo->record, pInfo->id,
                                 AGPalmPilotAttribsToMALMod((uint8)att),
                                 size, pInfo->pilot_buffer, 0, NULL);

    *record = pInfo->record;
    return AGCLIENT_CONTINUE;
}

static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                    int32 *errCode)
{
    debug_print("GetNextModifiedRecord()\n");

    return getRecordBase((PalmSyncInfo *)out, TRUE, record, errCode);
}

static int32 getNextRecord(void * out, AGRecord ** record, int32 *errCode)
{
    debug_print("GetNextRecord()\n");

    return getRecordBase((PalmSyncInfo *)out, FALSE, record, errCode);
}

static AGBool doClientProcessorLoop(PalmSyncInfo * pInfo, AGNetCtx *ctx)
{
    int32 dummyError;
    int32 cpResult;
    int32 syncCount;
    int32 i, n;
    AGBool cancelled = FALSE;
    AGLocationConfig *lc = NULL;
    
    n = AGUserConfigCount(pInfo->userConfig);

    for (i = 0; i < n; ++i) {

        AGServerConfig * sc =
            AGUserConfigGetServerByIndex(pInfo->userConfig, i);
        
        if (cancelled)
            continue;

        if (NULL == sc)
            continue;

        if (sc->disabled)
            continue;

        if (NULL == sc->serverName || sc->serverPort <= 0)
            continue;

        syncCount = 0;
        doStartServer(pInfo, sc, &dummyError);
 
        do {
            AGCommandProcessorStart(pInfo->commandProcessor);
            
            pInfo->deviceInfo = AGDeviceInfoNew();
            /* If this fails, we're totally dead.  Exit with fail code. */
            if(!pInfo->deviceInfo)
                return FALSE;
                
            readDeviceInfo(pInfo);

            if (httpProxy && httpProxyPort) {
                lc = AGLocationConfigNew();
                lc->HTTPUseProxy = 1;
                lc->HTTPName = httpProxy;
                lc->HTTPPort = httpProxyPort;
                if (proxyUsername && proxyPassword) {
                    lc->HTTPUseAuthentication = 1;
                    lc->HTTPUsername = proxyUsername;
                    lc->HTTPPassword = proxyPassword;
                }
            }

            if (socksProxy && socksProxyPort) {
                if (!lc)
                    lc = AGLocationConfigNew();
                lc->SOCKSUseProxy = 1;
                lc->SOCKSName = socksProxy;
                lc->SOCKSPort = socksProxyPort;
            }
            
            pInfo->clientProcessor =
                AGClientProcessorNew(pInfo->currentServerConfig,
                                     pInfo->deviceInfo, 
                                     lc,
                                     pInfo->platform, 
                                     TRUE,
                                     ctx);
                                        
            /* If this fails, we're totally dead.  Exit with fail code. */
            if (NULL == pInfo->clientProcessor) {
                AGDeviceInfoFree(pInfo->deviceInfo);
                return FALSE;
            }

            AGClientProcessorSetBufferServerCommands(pInfo->clientProcessor,
                                                     FALSE);
            
            AGClientProcessorSync(pInfo->clientProcessor);


            cpResult = AGCLIENT_CONTINUE;
            while (AGCLIENT_CONTINUE == cpResult) {
                cpResult = AGClientProcessorProcess(pInfo->clientProcessor);
                
                if (AGCLIENT_CONTINUE == cpResult && pInfo->quit) {
                    cancelled = TRUE;
                    cpResult = AGCLIENT_IDLE;
                }
                if (dlp_OpenConduit(sd)<0) {
                    fprintf(stderr, "Exiting on cancel, data not retrieved.\n");
                    exit(1);
                }
                
            }


            if(cpResult == AGCLIENT_ERR) {

                char *msg = AGGetMsg(pInfo->clientProcessor->errStringId);
                if (msg)
                    fprintf(stderr,"%s\n", msg);
                else
                    fprintf(stderr,"Unknown error\n");
                    
            }

            AGClientProcessorFree(pInfo->clientProcessor);
            AGDeviceInfoFree(pInfo->deviceInfo);

        } while (!cancelled
                 && AGCommandProcessorShouldSyncAgain(pInfo->commandProcessor) 
                 && syncCount++ < 3);
        doEndServer(pInfo, &dummyError);

        /* It's possible to cancel in the middle of an open database,
           so if that happened, close it. */
        if (pInfo->pilot_rHandle)
            closeDatabase(pInfo);
        
    }

    return TRUE; /* success */
}
void Disconnect(void)
{
    if(sd == 0)
        return;
    
    dlp_EndOfSync(sd, 0);
    pi_close(sd);
    sd = 0;
}
int main(int argc, char *argv[])
{
    
    PalmSyncInfo * pInfo;
    uint32 pilotID;
    AGNetCtx *ctx;
    

    
    pInfo = syncInfoNew();
    if (NULL == pInfo)
        return -1;

    processCommandLine(argc, argv, pInfo);

    Connect(pInfo); 

    if ( dlp_OpenConduit(sd) < 0) {
        fprintf(stderr, "Exiting on cancel\n");
        exit(1);
    }

    ctx = (AGNetCtx *)malloc(sizeof(AGNetCtx));
    AGNetInit(ctx);

    if (setupPlatformCalls(pInfo))
           return -1;

    pInfo->userConfig = getUserConfig(&pilotID);

    doClientProcessorLoop(pInfo, ctx);

    storeDeviceUserConfig(pInfo->userConfig, pilotID);
    
    AGNetClose(ctx);

    Disconnect(); 
    syncInfoFree(pInfo);
    free(ctx);

    return 0;
}
RETSIGTYPE SigHandler(int signal)
{
  puts("Abort on signal!");
  Disconnect();
  exit(3);
}

static void Connect(PalmSyncInfo *pi) 
{
    struct pi_sockaddr addr;
    int ret;
    
    if (sd != 0)
        return;

#ifndef WIN32
    signal(SIGHUP, SigHandler);
    signal(SIGINT, SigHandler);
    signal(SIGSEGV, SigHandler);
#endif
    
    if (!(sd = pi_socket(PI_AF_SLP, PI_SOCK_STREAM, PI_PF_PADP))) {
        perror("pi_socket");
        exit(1);
    }
    
    addr.pi_family = PI_AF_SLP;
    strcpy(addr.pi_device, device);
    
    ret = pi_bind(sd, (struct sockaddr*)&addr, sizeof(addr));
    if(ret == -1) {
        fprintf(stderr, "Unable to bind to port '%s'.\n", device);
        exit(1);
    }
    
    printf("Waiting for connection on %s (press the HotSync button now)...\n", 
           device);
    
    ret = pi_listen(sd,1);
    if(ret == -1) {
        perror("pi_listen");
        exit(1);
    }
    
    sd = pi_accept(sd,0,0);
    if(sd == -1) {
        perror("pi_accept");
        exit(1);
    }
    
    puts("Connected");
}
void Help(void)
{
      printf("Usage: %s command(s)\n\n",progname);
      printf("Where a command is one or more of: -p   proxy address\n");
      printf("                                   -r   proxy port\n");
      printf("                                   -u   proxy username\n");
      printf("                                   -d   proxy password\n");
      printf("                                   -s   socks proxy\n");
      printf("                                   -o   socks port\n");
      printf("                                   -h   prints this\n");
      printf("\n");
      printf("The serial port to connect to may be specified by the PILOTPORT\n");
      printf("environment variable. If not specified it will default to \n");
      printf("/dev/pilot \n");
      printf("\n");
      printf("The baud rate to connect with may be specified by the PILOTRATE\n");
      printf("environment variable. If not specified, it will default to 9600.\n");
      printf("Please use caution setting it to higher values, as several types\n");
      printf("of workstations have problems with higher rates.\n");
      printf("\n");
      exit(0);
}
static int processCommandLine(int argc, char *argv[], PalmSyncInfo *pi)
{
    char *str;
    int i;

    str = getenv("PILOTPORT");
    if (str != NULL)
        device = str;

    progname = argv[0];
    for(i=1;i<argc;i++) {
  	if (argv[i][0] == '-') {
            if (argv[i][1] == 'h') {
                Help();
            }
            else if (argv[i][1] == 'p') {
                if (++i >= argc) {
                    fprintf(stderr, "%s: Options '%s' requires argument\n",
                            argv[0], argv[i-1]);
                    Help();
                }
                httpProxy = strdup(argv[i]);
            }
            else if (argv[i][1] == 'r') {
                if (++i >= argc) {
                    fprintf(stderr, "%s: Options '%s' requires argument\n",
                            argv[0], argv[i-1]);
                    Help();
                }
                httpProxyPort = atoi(argv[i]);
            }
            else if (argv[i][1] == 's') {
                if (++i >= argc) {
                    fprintf(stderr, "%s: Options '%s' requires argument\n",
                            argv[0], argv[i-1]);
                    Help();
                }
                socksProxy = strdup(argv[i]);
            }
            else if (argv[i][1] == 'o') {
                if (++i >= argc) {
                    fprintf(stderr, "%s: Options '%s' requires argument\n",
                            argv[0], argv[i-1]);
                    Help();
                }
                socksProxyPort = atoi(argv[i]);
            }
            else if (argv[i][1] == 'u') {
                if (++i >= argc) {
                    fprintf(stderr, "%s: Options '%s' requires argument\n",
                            argv[0], argv[i-1]);
                    Help();
                }
                proxyUsername = strdup(argv[i]);
            }
            else if (argv[i][1] == 'd') {
                if (++i >= argc) {
                    fprintf(stderr, "%s: Options '%s' requires argument\n",
                            argv[0], argv[i-1]);
                    Help();
                }
                proxyPassword = strdup(argv[i]);
            } else {
                fprintf(stderr, "%s: Unknown option '%s'\n", argv[0], argv[i]);
                Help();
            }
        } else {
            fprintf(stderr, "%s: Unknown option '%s'\n", argv[0], argv[i]);
            Help();
        }
    }
    return 0;
    
}

