/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

#include <stdio.h>
#include <AGUtil.h>
#include <AGNet.h>
#include <AGBufferWriter.h>
#include <AGProtocol.h>
#include <AGDBConfig.h>
#include <AGBufferReader.h>
#include <AGArray.h>
#include <AGPalmProtocol.h>
#include <AGDigest.h>

#ifdef _WIN32
#define XMLTOKAPI   __declspec(dllimport) 
#define XMLPARSEAPI __declspec(dllimport)
#else
#define XMLTOKAPI   
#define XMLPARSEAPI 
#endif

#include <xmlparse.h>

#define SERVER_NAME "localhost"
#define SERVER_PORT 80
#define MALSERVER_XMLPREFS_FILENAME    "malserver.xml"

// Should the server send a sample EXPANSION cmd down to the client?
#define SEND_TEST_EXPANSION_CMD 

// Should the server insert sleep() calls to simulate server processing lags?
//#define SIMULATE_SERVER_DELAYS 

// Converts a 4 character string to a Long suitable for use as a 
// creatorId
#define AGStrToLong(__theStringPtr__)      \
    (  (uint32)                                 \
       (                                            \
        (((uint32)__theStringPtr__[0]) << 24) + \
        (((uint32)__theStringPtr__[1]) << 16) + \
        (((uint32)__theStringPtr__[2]) << 8) +  \
        ((uint32)__theStringPtr__[3])           \
       )                                            \
    )


typedef struct MalDatabase {
    AGDBConfig dbconfig;
    AGBool isDeleted;
    AGArray records;

    uint32 palmCreator;
    uint32 palmType;
    uint32 palmFlags;
} MalDatabase;

typedef struct MalServer {
    char *serverName;
    uint16 port;
    char *cookie;
    char *userUrl;
    char *message;
    char *serverUri;
    AGArray dbs;
} MalServer;

static void malServerInit(char *filename, MalServer *malServer);
static void malServerFinalize(MalServer *malServer);
static AGSocket *listenForConnections(AGSocket *listener,
                                      char *serverName, 
                                      int16 serverPort);
static void sendXMLCommands(AGSocket *soc, MalServer *malServer);
static int32 sendDefaultCommands(AGSocket *soc);
static int32 parseHeaders(AGSocket *soc, int32 *contentLen);
static void writeDebugEXPANSIONCMD(AGWriter *w);

static int32 parseClientCommands(AGSocket *soc, int32 commandLen);
static void printHELLO(AGReader *r);
static void printHELLO2(AGReader *r);
static void printDEVICEINFO(AGReader *r);
static void printOPENDATABASE(AGReader *r);
static void printRECORD(AGReader *r);
static void printEXPANSION(AGReader *r, int32 len);
static void printUNKNOWNDATABASE(AGReader *r);
static void printDebugReaderData(AGReader *r, uint32 maxLen);
static void printNEWIDS(AGReader *r);
static int32 sendHttpHeader(AGSocket *soc);
static void printPING(AGReader *r);
static void printXMLDATA(AGReader *r);
static AGNetCtx ctx;

static void *userCookie = NULL;

int main (int argc, char *argv[])
{
    AGSocket *soc, *listener;
    int32 rc;
    int32 contentLen;
    MalServer malServer;
    char *filename;

    AGNetInit(&ctx); 

    filename = MALSERVER_XMLPREFS_FILENAME;
    if(argc > 1)
        filename = argv[1];
    else
        printf("usage: malserver [serverprefs.xml]\n");

    malServerInit(filename, &malServer);

    if ((listener = AGNETSOCKETNEW(&ctx)) == NULL)
        return 0;
    listener->saddr.sin_family      = AF_INET;
    listener->saddr.sin_port        = htons(malServer.port);
    listener->saddr.sin_addr.s_addr = INADDR_ANY;

    rc = bind(listener->fd, 
            (struct sockaddr *)&listener->saddr, 
            sizeof(struct sockaddr_in));
        
    printf("malserver running: %s:%d\n", malServer.serverName, malServer.port);
    printf("accepting connections...\n");

    rc = listen(listener->fd, SOMAXCONN);

    while(TRUE) {
        if(userCookie != NULL) {
            free(userCookie);
            userCookie = NULL;
        }

        soc = listenForConnections(listener, malServer.serverName, malServer.port);

        printf("User Connected:\n");
        contentLen = 0;
        parseHeaders(soc, &contentLen);
        if(contentLen > 0) {
            parseClientCommands(soc, contentLen);
        }
        printf("\n");
        sendHttpHeader(soc);
        if(AGArrayCount(&malServer.dbs) < 1)
            sendDefaultCommands(soc);
        else
            sendXMLCommands(soc, &malServer);
        AGNETSOCKETFREE(&ctx, soc);
    }

    malServerFinalize(&malServer);
    AGNETSOCKETCLOSE(&ctx, listener);
    AGNetClose(&ctx);
    
}
  
static AGSocket *listenForConnections(AGSocket *listener,
                                      char *serverName, int16 serverPort)
{
    AGSocket *soc;
    int32 r;
    
    r = accept(listener->fd, NULL, NULL);

    //PENDING(klobad) some socket magic here per tom
    soc = calloc(1, sizeof(AGSocket));
    soc->fd = r;
    soc->state = AG_SOCKET_NEW;
    return soc;
}

#define HEADER_LINE_LEN 200
#define CONTENT_LEN_STR "Content-length: "
static int32 parseHeaders(AGSocket *soc, int32 *contentLen)
{
    char headerline[HEADER_LINE_LEN];
    int32 rc, br;
    char *clen;

    *contentLen = 0;
    while(TRUE) {
        bzero(headerline, HEADER_LINE_LEN);
        rc = AGNetGets(&ctx, soc, headerline, HEADER_LINE_LEN, &br,
                       TRUE);

        if(rc < 1)
            return rc;

        if(rc == 2
            && headerline[0] == '\r'
            && headerline[1] == '\n') {
            break;
        }
        printf("\t %s", headerline);

        clen = strstr(headerline, CONTENT_LEN_STR);
        if(clen == NULL)
            continue;

        *contentLen = atoi(clen + strlen(CONTENT_LEN_STR));
    }


    return 0;
}

static int32 parseClientCommands(AGSocket *soc, int32 commandLen)
{
    uint8 *buf;
    int32 rc;
    AGBufferReader aReader;
    AGReader *reader;
    int32 command, len;
    AGBool done = FALSE;
    uint16 magic;
    int8 majorVersion, minorVersion;
    char *name;

    buf = malloc(commandLen);
    if(buf == NULL)
        return 0;

    rc = AGNETRECV(&ctx, soc, buf, commandLen, TRUE);
    if(rc != commandLen) {
        free(buf);
        return 0;
    }
        
    AGBufferReaderInit(&aReader, buf);
    reader = (AGReader *)&aReader;

    AGReadMAGIC(reader, &magic);
    AGReadMAJORVERSION(reader, &majorVersion);
    AGReadMINORVERSION(reader, &minorVersion);

    if (magic != ((AG_PROTOCOL_MAGIC_HIGH << 8) | (AG_PROTOCOL_MAGIC_LOW))) {
        printf("*** ERROR wrong magic %d != %d\n", 
                        magic, ((AG_PROTOCOL_MAGIC_HIGH << 8) |
                        (AG_PROTOCOL_MAGIC_LOW)));
        AGBufferReaderFinalize(&aReader);
        free(buf);
        return 0;
    }
    if (majorVersion != AG_PROTOCOL_MAJOR_VERSION) {
        printf("*** ERROR wrong major version client talking %d != server talking %d\n",
                    majorVersion, AG_PROTOCOL_MAJOR_VERSION);
        AGBufferReaderFinalize(&aReader);
        free(buf);
        return 0;
    }
    if (minorVersion != AG_PROTOCOL_MINOR_VERSION) {
        printf("WARNING: wrong minor version client talking %d != server talking %d\n", 
                    minorVersion, AG_PROTOCOL_MINOR_VERSION);
    }

    do {        
        command = AGReadCompactInt(reader);
        len = AGReadCompactInt(reader);

        name = AGProtocolCommandName(command);
        if(name == NULL) {
            name = "UNKNOWN";
        }
    
        printf("%s(%u, %u):\n", name, command, len);
        switch(command) {
        case AG_END_CMD:
            AGReadEND(reader);
            done = TRUE;
            break;
        case AG_EXPANSION_CMD:
            printEXPANSION(reader, len);
            break;
        case AG_HELLO_CMD:
            if(majorVersion == 0
                && minorVersion == 0)
                printHELLO(reader);
            else
                printHELLO2(reader);
            break;
        case AG_DEVICEINFO_CMD:
            printDEVICEINFO(reader);
            break;
        case AG_OPENDATABASE_CMD:
            printOPENDATABASE(reader);
            break;
        case AG_CLOSEDATABASE_CMD:
            break;
        case AG_RECORD_CMD:
            printRECORD(reader);
            break;
        case AG_UNKNOWNDATABASE_CMD:
            printUNKNOWNDATABASE(reader);
            break;
        case AG_NEWIDS_CMD:
            printNEWIDS(reader);
            break;
        case AG_PING_CMD:
            printPING(reader);
            break;
        case AG_XMLDATA_CMD:
            printXMLDATA(reader);
            break;
        default:
            printDebugReaderData(reader, commandLen);
            done = TRUE;
            break;
        }
    } while (!done);

    AGBufferReaderFinalize(&aReader);
    free(buf);
    return 0;
}

static void printHELLO(AGReader *r)
{
    char *username;
    uint8 digestAuth[16], nonce[16];
    int32 availableBytes, cookieLength;

    if(userCookie != NULL) {
        free(userCookie);
        userCookie = NULL;
    }
    
    AGDigestSetToNull(digestAuth);
    AGDigestSetToNull(nonce);
    AGReadHELLO(r, &username, digestAuth, nonce, &availableBytes, 
                &cookieLength, &userCookie);

    printf("\t username: %s\n", username);
    if (!AGDigestNull(digestAuth)) {
        printf("\t Auth sent up by client\n");
    } else
        printf("\t No auth sent up by client\n");
    if (!AGDigestNull(nonce)) {
        printf("\t Nonce sent up by client\n");
    } else
        printf("\t No nonce sent up by client\n");
    printf("\t availableBytes: %d\n", availableBytes);
    printf("\t cookieLength: %d\n", cookieLength);
    printf("\t cookie: %s\n", userCookie);

    free(username);
}

static void printHELLO2(AGReader *r)
{
    char *username;
    uint8 digestAuth[16], nonce[16];
    int32 availableBytes, cookieLength;
    uint32 uid;

    if(userCookie != NULL) {
        free(userCookie);
        userCookie = NULL;
    }
    
    AGDigestSetToNull(digestAuth);
    AGDigestSetToNull(nonce);
    AGReadHELLO2(r, &username, digestAuth, nonce, &availableBytes, 
                &cookieLength, &userCookie, &uid);

    printf("\t username: %s\n", username);
    if (!AGDigestNull(digestAuth)) {
        printf("\t Auth sent up by client\n");
    } else
        printf("\t No auth sent up by client\n");
    if (!AGDigestNull(nonce)) {
        printf("\t Nonce sent up by client\n");
    } else
        printf("\t No nonce sent up by client\n");
    printf("\t availableBytes: %d\n", availableBytes);
    printf("\t cookieLength: %d\n", cookieLength);
    printf("\t cookie: %s\n", userCookie);
    printf("\t uid: %d\n", uid);

    free(username);
}

static void printDEVICEINFO(AGReader *r)
{
    char *osName, *osVersion, *serialNumber, *language, *charset;
    int32 colorDepth, screenWidth, screenHeight, platformDataLength;
    void *platformData;

    AGReadDEVICEINFO(r, &osName, &osVersion, 
            &colorDepth, &screenWidth, &screenHeight, 
            &serialNumber, &language, &charset,
            &platformDataLength, &platformData);
    printf("\t osName: %s\n", osName);
    printf("\t osVersion: %s\n", osVersion);
    printf("\t colorDepth: %d\n", colorDepth);
    printf("\t screenWidth: %d\n", screenWidth);
    printf("\t screenHeight: %d\n", screenHeight);
    printf("\t serialNumber: %s\n", serialNumber);
    printf("\t language: %s\n", language);
    printf("\t charset: %s\n", charset);
    printf("\t platformDataLength: %d\n", platformDataLength);
    printf("\t platformData: %s\n", platformData);
    free(osName);
    free(osVersion);
    free(serialNumber);
    free(language);
    free(charset);
    free(platformData);
}

static void printOPENDATABASE(AGReader *r)
{
    char *dbname;

    AGReadOPENDATABASE(r, &dbname);
    printf("\t dbname: %s\n", dbname);
    free(dbname);
}

static void printNEWIDS(AGReader *r)
{
    AGArray *newids = NULL;
    int32 i, count;

    AGReadNEWIDS(r, &newids);
    if(newids != NULL) {
        count = AGArrayCount(newids);
        if(count < 1 || count % 2 > 0)
            printf("\t invalid number of newids: %d\n", count);
        else {
            for(i = 0; i < count; i+=2) {
                printf("\t %d = %d\n", 
                            (int32)AGArrayElementAt(newids, i),
                            (int32)AGArrayElementAt(newids, i+1));
            }
        }
        AGArrayFree(newids);
    } else {
        printf("\t NEWIDS command send with NO newids. This is not entirely optimal.\n");
    }
}

static void printRECORD(AGReader *r)
{
    int32 uid;
    AGRecordStatus mod;
    int32 recordDataLength;
    void *recordData;
    int32 platformDataLength;
    void *platformData;

    AGReadRECORD(r, &uid, &mod,
        &recordDataLength, &recordData,
        &platformDataLength, &platformData);
    printf("\t uid: %d\n", uid);
    printf("\t mod: %d\n", mod);
    printf("\t recordDataLength: %d\n", recordDataLength);
    printf("\t recordData: %s\n", recordData);
    printf("\t platformDataLength: %d\n", platformDataLength);
    printf("\t platformData: %d\n", platformData);

    free(recordData);
    free(platformData);
}

static void printEXPANSION(AGReader *r, int32 len)
{
    int32 command;
    int32 commandLen;
    void *buf = NULL;

    AGReadEXPANSION(r, &command, &commandLen, &buf);
    printf("\t command: %d\n", command);
    printf("\t commandLen: %d\n", commandLen);
    printf("\t command bytes: %s\n", buf);
    if(buf)
       free(buf);
}

static void printUNKNOWNDATABASE(AGReader *r)
{
    char *dbname;

    AGReadUNKNOWNDATABASE(r, &dbname);
    printf("\t dbname: %s\n", dbname);
    free(dbname);
}

static void printPING(AGReader *r)
{
    AGReadPING(r);
}

static void printXMLDATA(AGReader *r)
{
    int32 xmlLen = 0;
    void *xmlData = NULL;

    AGReadXMLDATA(r, &xmlLen, &xmlData);
    printf("\t xmlLen: %d\n", xmlLen);
    if(xmlLen > 0 && xmlData) {
        printf("\t xmlData: %s\n", xmlData);
        free(xmlData);
    }
}

static void printDebugReaderData(AGReader *r, uint32 maxLen)
{
    AGBufferReader *reader = (AGBufferReader *)r;
    uint32 b;

    b = reader->currentIndex;
    b -= 20; //note it's a uint32
    printf("invalid command data parsed\n");
    printf("current parse index: %d\n", reader->currentIndex);
    printf("max parse index: %d\n", maxLen - 1);
    printf("nearby bytes:\n");
    while(b < maxLen && (reader->currentIndex + 21 > b)) {
        if(b == reader->currentIndex)
            printf("----> ");
        printf("%4d:%02X\n", b, reader->buffer[b]);
        b++;
    }
    printf("shutting down malserver\n");
    exit(0);
}

/*--------------------------------------------------------------------------*/
void xmlEndElement(void *userData, const char *name)
{
    MalServer *malServer;
    MalDatabase *mdb;
    int32 count;
    AGBufferWriter w;
    uint8 *ptr;
    int32 len;

    // only interested in PalmDatabaseInfo currently
    if(strcmp("PalmDatabaseInfo", name) != 0) 
        return; 

    malServer = (MalServer *)userData;
    count = AGArrayCount(&malServer->dbs);
    if(count < 1)
        return;

    mdb = AGArrayElementAt(&malServer->dbs, count - 1);
    if(mdb == NULL)
        return;

    if (mdb->palmCreator != 0) {
        AGBufferWriterInit(&w, 16);
        AGPalmWriteDBConfigPlatformData((AGWriter *)&w, mdb->palmCreator, mdb->palmType, mdb->palmFlags);
        ptr = AGBufferWriterGetBuffer(&w);
        len = AGBufferWriterGetBufferSize(&w);
        w.buffer = NULL;
        AGDBConfigSetPlatformData(&mdb->dbconfig, len, ptr);
        AGBufferWriterFinalize(&w);
    }
    mdb->palmCreator = 0;
    mdb->palmType = 0;
    mdb->palmFlags = 0;
}

void xmlStartElement(void *userData, const char *name, const char **atts)
{
    MalServer *malServer;
    int i = 0;
    char *attr;

    malServer = (MalServer *)userData;
    if (strcmp("Server", name) == 0) {
        // look for "hostname", "port", "cookie", "userurl", "message", "serveruri"
        while (atts[i] != 0) {
            attr = (char *)atts[i++];
            if (strcmp("hostname", attr) == 0) {
                if (malServer->serverName != NULL)
                    free(malServer->serverName);
                malServer->serverName = strdup(atts[i++]);
            } else if (strcmp("port", attr) == 0) {
                malServer->port = atoi(atts[i++]);
            } else if (strcmp("cookie", attr) == 0) {
                if (malServer->cookie != NULL)
                    free(malServer->cookie);
                malServer->cookie = strdup(atts[i++]);
            } else if (strcmp("userurl", attr) == 0) {
                if (malServer->userUrl != NULL)
                    free(malServer->userUrl);
                malServer->userUrl = strdup(atts[i++]);
            } else if (strcmp("message", attr) == 0) {
                if (malServer->message != NULL)
                    free(malServer->message);
                malServer->message = strdup(atts[i++]);
            } else if (strcmp("serveruri", attr) == 0) {
                if (malServer->serverUri != NULL)
                    free(malServer->serverUri);
                malServer->serverUri = strdup(atts[i++]);
            } 
        }
    } else if (strcmp("Database", name) == 0) {
        // look for "name", "isdeleted", "status", "sendrecordplatformdata"
        MalDatabase *mdb;

        mdb = malloc(sizeof(MalDatabase));
        bzero(mdb, sizeof(MalDatabase));
        AGArrayInit(&mdb->records, AGUnownedPointerElements, 0);
        AGArrayAppend(&malServer->dbs, mdb);
        while (atts[i] != 0) {
            attr = (char *)atts[i++];
            if (strcmp("name", attr) == 0) {
                if (mdb->dbconfig.dbname != NULL)
                    free(mdb->dbconfig.dbname);
                mdb->dbconfig.dbname = strdup(atts[i++]);
            } else if (strcmp("isdeleted", attr) == 0) {
                if(0 == strcmp(atts[i], "TRUE"))
                    mdb->isDeleted = TRUE;
                else
                    mdb->isDeleted = FALSE;
                i++;
            } else if (strcmp("status", attr) == 0) {
                if(0 == strcmp(atts[i], "AG_SENDMODS_CFG"))
                    mdb->dbconfig.type = AG_SENDMODS_CFG;
                else if(0 == strcmp(atts[i], "AG_DONTSEND_CFG"))
                    mdb->dbconfig.type = AG_DONTSEND_CFG;
                else 
                    mdb->dbconfig.type = AG_SENDALL_CFG;
                i++;
            } else if (strcmp("sendrecordplatformdata", attr) == 0) {
                if(0 == strcmp(atts[i++], "TRUE"))
                    mdb->dbconfig.sendRecordPlatformData = TRUE;
                else
                    mdb->dbconfig.sendRecordPlatformData = FALSE;
            } 
        }
    } else if ((strcmp("Record", name) == 0) && AGArrayCount(&malServer->dbs) > 0) {
        // look for "uid", "status", "data"
        MalDatabase *mdb;
        AGRecord *record;
        int32 count;

        count = AGArrayCount(&malServer->dbs);            
        mdb = AGArrayElementAt(&malServer->dbs, count - 1);
        record = malloc(sizeof(AGRecord));
        bzero(record, sizeof(AGRecord));
        AGArrayAppend(&mdb->records, record);
        while (atts[i] != 0) {
            attr = (char *)atts[i++];
            if (strcmp("uid", attr) == 0) {
                record->uid = atoi(atts[i++]);
            } else if (strcmp("status", attr) == 0) {
                if(0 == strcmp(atts[i], "AG_RECORD_NEW"))
                    record->status = AG_RECORD_NEW;
                else if(0 == strcmp(atts[i], "AG_RECORD_UPDATED"))
                    record->status = AG_RECORD_UPDATED;
                else if(0 == strcmp(atts[i], "AG_RECORD_DELETED"))
                    record->status = AG_RECORD_DELETED;
                else if(0 == strcmp(atts[i], "AG_RECORD_NEW_TEMPORARY_UID"))
                    record->status = AG_RECORD_NEW_TEMPORARY_UID;
                else 
                    record->status = AG_RECORD_UNMODIFIED;
                i++;
            } else if (strcmp("data", attr) == 0) {
                if (record->recordData != NULL)
                    free(record->recordData);
                record->recordData = strdup(atts[i++]);
                record->recordDataLength = strlen(record->recordData) + 1;
            }
        }
    } else if ((strcmp("PalmDatabaseInfo", name) == 0) && AGArrayCount(&malServer->dbs) > 0) {
        // look for "creator", "type", and "flags"
        MalDatabase *mdb;
        int32 count;
        const uint8 *ptr;

        count = AGArrayCount(&malServer->dbs);            
        mdb = AGArrayElementAt(&malServer->dbs, count - 1);
        while (atts[i] != 0) {
            attr = (char *)atts[i++];
            if (strcmp("flags", attr) == 0) {
                mdb->palmFlags = atoi(atts[i++]);
            } else if (strcmp("creator", attr) == 0) {
                ptr = atts[i++];
                mdb->palmCreator = AGStrToLong(ptr);
            } else if (strcmp("type", attr) == 0) {
                ptr = atts[i++];
                mdb->palmType = AGStrToLong(ptr);
            }
        }
    } else {
        // elements that we are not looking for, dumping to console
        while (atts[i] != 0) {
            puts(atts[i++]);
            putchar('\t');
            puts(atts[i++]);
        }
    }
}

static int32 readConfigFromFile(char *filename, MalServer *malServer)
{
    char buffer[BUFSIZ];
    XML_Parser parser;
    FILE *stream;
    int32 done;
    AGBool errored = FALSE;
    int32 dbCount;
    MalDatabase *mdb;

    stream = fopen(filename, "rb");
    if (stream == NULL)
        return -1;

    parser = XML_ParserCreate(NULL);
    XML_SetUserData(parser, malServer);
    XML_SetElementHandler(parser, xmlStartElement, xmlEndElement);

    do {
        size_t len = fread(buffer, 1, sizeof(buffer), stream);
        if (ferror(stream)) {
            errored = TRUE;
            break;
        }
        done = len < sizeof(buffer);
        if (!XML_Parse(parser, buffer, len, done)) {
            fprintf(stderr, "%s at line %d\n",
                XML_ErrorString(XML_GetErrorCode(parser)),
                XML_GetCurrentLineNumber(parser));
            errored = TRUE;
            break;
        }
    } while (!done);

    XML_ParserFree(parser);

    fclose(stream);
    if (errored)
        return -1;

    printf("server data from: %s\n", filename);
    dbCount = AGArrayCount(&malServer->dbs);
    while(dbCount--) {
        mdb = AGArrayElementAt(&malServer->dbs, dbCount);
        printf("\t name:%s #recs:%d\n", 
                        mdb->dbconfig.dbname, 
                        AGArrayCount(&mdb->records));
    }
    return 0;
}

static void malServerInit(char *filename, MalServer *malServer) 
{
    memset(malServer, 0, sizeof(MalServer));
    AGArrayInit(&malServer->dbs, AGUnownedPointerElements, 0);
    malServer->serverName = strdup(SERVER_NAME);
    malServer->port = SERVER_PORT;

    readConfigFromFile(filename, malServer);
}

static void malServerFinalize(MalServer *malServer)
{
    int32 dbCount, recCount;
    AGRecord *record;
    MalDatabase *mdb;

    if(malServer->serverName != NULL)
        free(malServer->serverName);
    if(malServer->cookie != NULL)
        free(malServer->cookie);
    if(malServer->userUrl != NULL)
        free(malServer->userUrl);
    if(malServer->message != NULL)
        free(malServer->message);
    if(malServer->serverUri != NULL)
        free(malServer->serverUri);

    dbCount = AGArrayCount(&malServer->dbs);

    while(dbCount--) {
        mdb = AGArrayElementAt(&malServer->dbs, dbCount);
        AGDBConfigFinalize(&mdb->dbconfig);
        recCount = AGArrayCount(&mdb->records);
        while(recCount--) {
            record = AGArrayElementAt(&mdb->records, recCount);
            AGRecordFree(record);
        }
        AGArrayFinalize(&mdb->records);
    }
    AGArrayFinalize(&malServer->dbs);
}

static void sendXMLCommands(AGSocket *soc, MalServer *malServer)
{
    int32 dbCount, recCount, maxRecCount;
    AGRecord *record;
    MalDatabase *mdb;
    AGBufferWriter writer;
    AGBool deviceShouldHashPasswords = TRUE;
    AGBool allowSecureDeviceConnect = FALSE;
    uint32 deviceConnectTimeout, deviceWriteTimeout, deviceReadTimeout;

    deviceConnectTimeout = 30;
    deviceWriteTimeout = 30;
    deviceReadTimeout = 120;

    AGBufferWriterInit(&writer, 1024);

    AGWriteMAGIC((AGWriter *)&writer);
    AGWriteMAJORVERSION((AGWriter *)&writer, AG_PROTOCOL_MAJOR_VERSION);
    AGWriteMINORVERSION((AGWriter *)&writer, AG_PROTOCOL_MINOR_VERSION);

    // Send the serverConfig and DatabaseConfig info if
    // the cookie is out of whack
    dbCount = AGArrayCount(&malServer->dbs);
    if(userCookie == NULL || 0 != strcmp(userCookie, malServer->cookie)) {
        AGWriteSERVERCONFIG((AGWriter *)&writer, "MAL Test Server", malServer->userUrl, 
                            malServer->message, malServer->serverUri, 
                            deviceShouldHashPasswords,
                            allowSecureDeviceConnect, deviceConnectTimeout, 
                            deviceWriteTimeout, deviceReadTimeout);
        // turn on the device info sending since cookies don't match
        // we need to get it new and then store it
        AGWriteSENDDEVICEINFO((AGWriter *)&writer, TRUE);
        while(dbCount--) {
            mdb = AGArrayElementAt(&malServer->dbs, dbCount);
#ifdef SIMULATE_SERVER_DELAYS
        printf("SIMULATE_SERVER_DELAYS ON - sleeping for: %ld\n",20);
        AGSleepMillis(20);
#endif
            AGWriteDATABASECONFIG((AGWriter *)&writer, 
                            mdb->dbconfig.dbname, mdb->dbconfig.type,
                            mdb->dbconfig.sendRecordPlatformData, 
                            mdb->dbconfig.platformDataLength, 
                            mdb->dbconfig.platformData);
        }
        AGWriteCOOKIE((AGWriter *)&writer, 
                strlen(malServer->cookie) + 1, 
                (void *)malServer->cookie);
#ifdef SIMULATE_SERVER_DELAYS
        printf("SIMULATE_SERVER_DELAYS ON - sleeping for: %ld\n",15);
        AGSleepMillis(15);
#endif
        AGWriteGOODBYE((AGWriter *)&writer, AG_CALLAGAIN_STATUS, 0, "Invalid User Cookie");
        AGWriteEND((AGWriter *)&writer);
        AGNETSEND(&ctx, soc, 
                AGBufferWriterGetBuffer(&writer),
                AGBufferWriterGetBufferSize(&writer),
                TRUE);

        AGBufferWriterFinalize(&writer);
        return;
    }

    // turn off the device info sending since we should have stored it by now
    AGWriteSENDDEVICEINFO((AGWriter *)&writer, FALSE);

    // send any delete database commands
    dbCount = AGArrayCount(&malServer->dbs);
    AGWriteTASK((AGWriter *)&writer, "Deleting old DBs", TRUE);
    while(dbCount--) {
        mdb = AGArrayElementAt(&malServer->dbs, dbCount);
        if(mdb->isDeleted) {
            AGWriteDELETEDATABASE((AGWriter *)&writer, mdb->dbconfig.dbname);
        }
    }

    // now send the records for the databases
    dbCount = AGArrayCount(&malServer->dbs);
    AGWriteTASK((AGWriter *)&writer, "Updating databases...", TRUE);
    while(dbCount--) {
        mdb = AGArrayElementAt(&malServer->dbs, dbCount);
        if(mdb->isDeleted) {
            continue;
        }
        maxRecCount = AGArrayCount(&mdb->records);
        if(maxRecCount < 1)
            continue;
        AGWriteOPENDATABASE((AGWriter *)&writer, mdb->dbconfig.dbname);

        for(recCount = 0; recCount < maxRecCount; recCount++) {
            record = AGArrayElementAt(&mdb->records, recCount);
#ifdef SIMULATE_SERVER_DELAYS
        printf("SIMULATE_SERVER_DELAYS ON - sleeping for: %ld\n",(record->recordDataLength / 1000)+1);
        AGSleepMillis((record->recordDataLength / 100)+1);
#endif
            AGWriteITEM((AGWriter *)&writer, recCount, maxRecCount, NULL);
            AGWriteRECORD((AGWriter *)&writer, record->uid, record->status, 
                    record->recordDataLength, (void *)record->recordData, 0, NULL);
        }
        AGWriteCLEARMODS((AGWriter *)&writer);
        AGWriteCLOSEDATABASE((AGWriter *)&writer);
    }

    AGWriteCOOKIE((AGWriter *)&writer, 
                    strlen(malServer->cookie) + 1, 
                    (void *)malServer->cookie);
#ifdef SIMULATE_SERVER_DELAYS
        printf("SIMULATE_SERVER_DELAYS ON - sleeping for: %ld\n",20);
        AGSleepMillis(20);
#endif

    writeDebugEXPANSIONCMD((AGWriter *)&writer);
    AGWriteGOODBYE((AGWriter *)&writer, AG_DONE_STATUS, 0, NULL);
    AGWriteEND((AGWriter *)&writer);

    AGNETSEND(&ctx, soc, AGBufferWriterGetBuffer(&writer),
              AGBufferWriterGetBufferSize(&writer), TRUE);

    AGBufferWriterFinalize(&writer);
}
#define HTTP_RESPONSE "HTTP/1.1 200 OK\r\n\r\n"
static int32 sendHttpHeader(AGSocket *soc)
{

    return AGNETSEND(&ctx, soc, HTTP_RESPONSE, strlen(HTTP_RESPONSE),
                     TRUE);
    
}
static int32 sendDefaultCommands(AGSocket *soc)
{
const char recordData1[] = "1 THIS IS SOME RECORD DATA 1";
const char recordData2[] = "2 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 22 THIS IS SOME RECORD DATA 2";
const char recordData3[] = "3 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 33 THIS IS SOME RECORD DATA 3";
const char recordData4[] = "4 THIS IS SOME RECORD DATA 4";
const char cookie[] = "malServerCookie";
uint8 nonce[16] =  "1234561234567890";
    AGBool deviceShouldHashPasswords = TRUE;
    uint32 deviceConnectTimeout, deviceWriteTimeout, deviceReadTimeout;    
    AGBufferWriter writer;

    deviceConnectTimeout = 30;
    deviceWriteTimeout = 30;
    deviceReadTimeout = 120;

    AGBufferWriterInit(&writer, 1024);

    AGWriteMAGIC((AGWriter *)&writer);
    AGWriteMAJORVERSION((AGWriter *)&writer, AG_PROTOCOL_MAJOR_VERSION);
    AGWriteMINORVERSION((AGWriter *)&writer, AG_PROTOCOL_MINOR_VERSION);

    if(userCookie == NULL || 0 != strcmp(userCookie, cookie)) {
        AGWriteDATABASECONFIG((AGWriter *)&writer, "ModDbBeta", AG_SENDMODS_CFG, TRUE, 0, NULL);
        AGWriteSERVERCONFIG((AGWriter *)&writer, "MAL Test Server", 
            "http://www.avantgo.com", "server message", "/sync", 
            deviceShouldHashPasswords, FALSE, deviceConnectTimeout, 
            deviceWriteTimeout, deviceReadTimeout);
        AGWriteCOOKIE((AGWriter *)&writer, strlen(cookie) + 1, (void *)cookie);
        AGWriteNONCE((AGWriter *)&writer, nonce);
        AGWriteGOODBYE((AGWriter *)&writer, AG_CALLAGAIN_STATUS, 0,
                       "Invalid User Cookie");
        AGWriteEND((AGWriter *)&writer);
        AGNETSEND(&ctx, soc, AGBufferWriterGetBuffer(&writer),
                  AGBufferWriterGetBufferSize(&writer), TRUE);

        AGBufferWriterFinalize(&writer);
        return 0;
    }

    /* pending(dim) this need to be implemented

    if( check whether digest auth from client == digest auth computed by server ) {
          send call again status = "Bad Password"
    }

    */
    
    AGWriteTASK((AGWriter *)&writer, "Deleting old DBs", TRUE);
    AGWriteDELETEDATABASE((AGWriter *)&writer, "DeletedDb1");
    AGWriteDELETEDATABASE((AGWriter *)&writer, "DeletedDb2");

    AGWriteTASK((AGWriter *)&writer, "Updating databases...", TRUE);
    AGWriteOPENDATABASE((AGWriter *)&writer, "NewDbAlpha");
    AGWriteITEM((AGWriter *)&writer, 0, 4, NULL);
    AGWriteRECORD((AGWriter *)&writer, 1, AG_RECORD_NEW, 
                    strlen(recordData1) + 1, (void *)recordData1, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 1, 4, NULL);
    AGWriteRECORD((AGWriter *)&writer, 2, AG_RECORD_NEW, 
                    strlen(recordData2) + 1, (void *)recordData2, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 2, 4, NULL);
    AGWriteRECORD((AGWriter *)&writer, 3, AG_RECORD_NEW, 
                    strlen(recordData3) + 1, (void *)recordData3, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 3, 4, NULL);
    AGWriteRECORD((AGWriter *)&writer, 4, AG_RECORD_NEW, 
                    strlen(recordData4) + 1, (void *)recordData4, 0, NULL);
    AGWriteCLEARMODS((AGWriter *)&writer);
    AGWriteCLOSEDATABASE((AGWriter *)&writer);
    
    AGWriteOPENDATABASE((AGWriter *)&writer, "ModDbBeta");
    AGWriteITEM((AGWriter *)&writer, 0, 4, NULL);
    AGWriteRECORD((AGWriter *)&writer, 1, AG_RECORD_UPDATED, 
                    strlen(recordData1) + 1, (void *)recordData1, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 1, 4, NULL);
    AGWriteRECORD((AGWriter *)&writer, 2, AG_RECORD_UPDATED, 
                    strlen(recordData2) + 1, (void *)recordData2, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 2, 4, NULL);
    AGWriteRECORD((AGWriter *)&writer, 3, AG_RECORD_UPDATED, 
                    strlen(recordData3) + 1, (void *)recordData3, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 3, 4, NULL);
    AGWriteRECORD((AGWriter *)&writer, 4, AG_RECORD_UPDATED, 
                    strlen(recordData4) + 1, (void *)recordData4, 0, NULL);
    AGWriteCLEARMODS((AGWriter *)&writer);
    AGWriteCLOSEDATABASE((AGWriter *)&writer);
    
    AGWriteOPENDATABASE((AGWriter *)&writer, "DeleteDbGamma");
    AGWriteITEM((AGWriter *)&writer, 0, 4, "adding 3");
    AGWriteRECORD((AGWriter *)&writer, 1, AG_RECORD_DELETED, 
                    strlen(recordData1) + 1, (void *)recordData1, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 1, 4, "adding 3");
    AGWriteRECORD((AGWriter *)&writer, 2, AG_RECORD_DELETED, 
                    strlen(recordData2) + 1, (void *)recordData2, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 2, 4, "adding 3");
    AGWriteRECORD((AGWriter *)&writer, 3, AG_RECORD_DELETED, 
                    strlen(recordData3) + 1, (void *)recordData3, 0, NULL);
    AGWriteITEM((AGWriter *)&writer, 3, 4, "adding 3");
    AGWriteRECORD((AGWriter *)&writer, 4, AG_RECORD_DELETED, 
                    strlen(recordData4) + 1, (void *)recordData4, 0, NULL);
    AGWriteCLEARMODS((AGWriter *)&writer);
    AGWriteCLOSEDATABASE((AGWriter *)&writer);

    AGWriteCOOKIE((AGWriter *)&writer, strlen(cookie) + 1, (void *)cookie);
    AGWriteNONCE((AGWriter *)&writer, nonce);
    AGWriteGOODBYE((AGWriter *)&writer, AG_DONE_STATUS, 0, NULL);
    AGWriteEND((AGWriter *)&writer);
    AGNETSEND(&ctx, soc, AGBufferWriterGetBuffer(&writer),
              AGBufferWriterGetBufferSize(&writer), TRUE);

    AGBufferWriterFinalize(&writer);
    return 0;
}

static void writeDebugEXPANSIONCMD(AGWriter *w)
{
#ifdef SEND_TEST_EXPANSION_CMD
    char *commandData = "Data for a new server Command";
    int32 newCommand = 50;
    AGWriteEXPANSION(w, newCommand, strlen(commandData) + 1, commandData);
#endif
}
