/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

#include <AGServerConfig.h>
#include <AGDeviceInfo.h>
#include <stdio.h>
#include <AGProtocol.h>
#include <AGBufferReader.h>
#include <AGArray.h>
#include <AGClientProcessor.h>
#include <AGUtil.h>
#include <AGWriter.h> // for AGCompactSize()
#include <AGCommandProcessor.h>
#include <AGMD5.h>

#ifdef _WIN32
#define XMLTOKAPI   __declspec(dllimport) 
#define XMLPARSEAPI __declspec(dllimport)
#else
#define XMLTOKAPI    
#define XMLPARSEAPI
#endif
#include <xmlparse.h>

#define MALCLIENT_XMLPREFS_FILENAME    "malclient.xml"

//#define SEND_TEST_EXPANSION_CMD

typedef struct ClientState {
    AGServerConfig *serverConfig;
    AGDeviceInfo *deviceInfo;
    AGCommandProcessor *cp;
#ifdef SEND_TEST_EXPANSION_CMD
    void *expansionData;
#endif
} ClientState;

static ClientState *clientStateNew(int argc, char* argv[]);
static void clientStateFree(ClientState *state);
static int32 readConfigFromFile(char *filename, 
                                AGServerConfig *serverConfig, 
                                AGDeviceInfo *deviceInfo);

static int32 getNextModifiedRecord(void *out, AGRecord **record, int32 *errCode); 
static int32 getNextRecord(void *out,AGRecord **record, int32 *errCode) ;
static int32 openDatabase(void *out, AGDBConfig *db, int32 *errCode);
static int32 nextExpansionCmd(void *out,
                               int32 *newCommand,
                               int32 *commandLength,
                               void **commandBytes);

static int32 printTASK(void *out, int32 *returnErrorCode, char *currentTask,
                       AGBool bufferable);
static int32 printITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem);
static int32 printDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname);
static int32 printOPENDATABASE(void *out, int32 *returnErrorCode,
                                           char *dbname);
static int32 printCLOSEDATABASE(void *out, int32 *returnErrorCode);
static int32 printCLEARMODS(void *out, int32 *returnErrorCode);
static int32 printGOODBYE(void *out, int32 *returnErrorCode,
                                      AGSyncStatus syncStatus,
                                      int32 errorCode,
                                      char *errorMessage);
static int32 printRECORD(void *out, int32 *returnErrorCode,
                                     int32 *newUID,
                                     int32 uid,
                                     AGRecordStatus mod,
                                     int32 recordDataLength,
                                     void *recordData,
                                     int32 platformDataLength,
                                     void *platformData);
static int32 printEXPANSION(void *out, int32 *returnErrorCode,
                                        int32 expansionCommand, 
                                        int32 commandLength,
                                        void *commandBytes);

int main(int argc, char* argv[]) {
    int16 syncCount = 0;
    AGPlatformCalls calls;
    AGClientProcessor *clientProcessor;
    ClientState *state;
    int32 result;
    AGBool pingFirst = TRUE;
    AGNetCtx ctx;

    AGNetInit(&ctx);

    state = clientStateNew(argc, argv);
    bzero(&calls, sizeof(AGPlatformCalls));
    calls.out = state;
    calls.openDatabaseFunc = openDatabase;
    calls.nextModifiedRecordFunc = getNextModifiedRecord;
    calls.nextRecordFunc = getNextRecord;
    calls.nextExpansionCommandFunc = nextExpansionCmd;

    calls.performCommandOut = state->cp;
    calls.performCommandFunc = AGCommandProcessorGetPerformFunc(state->cp);

    do {
        if(pingFirst) {
            printf("Pinging %s:%d\n",
                    state->serverConfig->serverName,
                    state->serverConfig->serverPort);
        } else {
            printf("Connecting to %s:%d as %s\n",
                    state->serverConfig->serverName,
                    state->serverConfig->serverPort,
                    state->serverConfig->userName);
            printf("\t cookie: %s\n", state->serverConfig->sequenceCookie);
        }
        AGCommandProcessorStart(state->cp);
        clientProcessor = AGClientProcessorNew(state->serverConfig, 
                                               state->deviceInfo,
                                               NULL,
                                               &calls,
                                               TRUE, &ctx);
        AGClientProcessorSetBufferServerCommands(clientProcessor, FALSE);
 
        if(pingFirst)
            AGClientProcessorPing(clientProcessor);
        else
            AGClientProcessorSync(clientProcessor);
        while ((result = AGClientProcessorProcess(clientProcessor))
                    == AGCLIENT_CONTINUE)
        {
        }

        if(result == AGCLIENT_ERR) {
            if(AGGetMsg(clientProcessor->errStringId))
                printf("Error during processing %s\n", 
                            AGGetMsg(clientProcessor->errStringId));
            else
                printf("Unknown Error during processing %u\n", 
                            clientProcessor->errStringId);
        }

        AGClientProcessorFree(clientProcessor);

        if(pingFirst) {
            syncCount--;
            pingFirst = FALSE;
            state->cp->syncAgain = TRUE; //HACK to sync again
        }

    } while (AGCommandProcessorShouldSyncAgain(state->cp) && syncCount++ < 2);

    if(syncCount > 2)
        printf("Cancelled because of syncCount: %d\n", syncCount);

    clientStateFree(state);
    AGNetClose(&ctx);
    return 0;
}

static ClientState *clientStateNew(int argc, char* argv[])
{
    ClientState *state = malloc(sizeof(ClientState));

    bzero(state, sizeof(ClientState));
    state->serverConfig = AGServerConfigNew();
    state->deviceInfo = AGDeviceInfoNew();

    // setup default values
    state->serverConfig->serverName = strdup("localhost");
    state->serverConfig->serverPort = 80;
    state->serverConfig->userName = strdup("nobody");
    state->serverConfig->disabled = FALSE;
    state->serverConfig->sequenceCookieLength = 0;
    state->serverConfig->sequenceCookie = NULL;

    state->deviceInfo->availableBytes = 1000000;
    state->deviceInfo->serialNumber = strdup("123321");
    state->deviceInfo->osName = strdup("WinCE");
    state->deviceInfo->osVersion = strdup("2.01");
    state->deviceInfo->screenWidth = 240;
    state->deviceInfo->screenHeight = 320;
    state->deviceInfo->colorDepth = 2;

    if(argc < 3)
        printf("usage: malclient [-f pref.xml] "
                "[serverName serverPort login password]\n");

    if(argc == 1) { // malclient.exe\n
        // if !MALCLIENT_XMLPREFS_FILENAME will use the defaults above
        readConfigFromFile(MALCLIENT_XMLPREFS_FILENAME, 
                           state->serverConfig, state->deviceInfo);
    } else {
        if(argc > 1) {
            if(0 == strcmp("-f", argv[1]) && argc > 2) {
                // malclient -f foo.xml
                readConfigFromFile(argv[2], state->serverConfig,
                                   state->deviceInfo);
            } else {
                /* malclient serverFoo 80 userfoo passwordfoo  */
                free(state->serverConfig->serverName);
                state->serverConfig->serverName = strdup(argv[1]);
                if(argc > 2) {
                    state->serverConfig->serverPort = atoi(argv[2]);
                }
                if(argc > 3) {
                    free(state->serverConfig->userName);
                    state->serverConfig->userName = strdup(argv[3]);
                }
                if(argc > 4) {
                    AGMd5((char *)argv[4], strlen(argv[4]), 
                                            state->serverConfig->password);
                }
            }
        }
    }

    state->cp = AGCommandProcessorNew(state->serverConfig);
    state->cp->commands.out = state;
    state->cp->commands.performTaskFunc = printTASK;
    state->cp->commands.performItemFunc = printITEM;
    state->cp->commands.performDeleteDatabaseFunc = printDELETEDATABASE;
    state->cp->commands.performOpenDatabaseFunc = printOPENDATABASE;
    state->cp->commands.performCloseDatabaseFunc = printCLOSEDATABASE;
    state->cp->commands.performClearModsFunc = printCLEARMODS;
    state->cp->commands.performGoodbyeFunc = printGOODBYE;
    state->cp->commands.performRecordFunc = printRECORD;
    state->cp->commands.performExpansionFunc = printEXPANSION;

    return state;
}

static void clientStateFree(ClientState *state)
{
    AGServerConfigFree(state->serverConfig);
    AGDeviceInfoFree(state->deviceInfo);
    AGCommandProcessorFree(state->cp);
    free(state);
}

// This function is passed into to AGPlatformCalls struct
// when we initialize the AGClientProcessor
//
// The AGClientProcessor calls this record when the 
// AGDBConfig says that the server wants just the mod the records
// from a database. 
static int32 getNextModifiedRecord(void *out, AGRecord **record, 
                                    int32 *errCode) 
{
    //PENDING(klobad) send up some bogus records??
    return AGCLIENT_IDLE;
}

// This function is passed into to AGPlatformCalls struct
// when we initialize the AGClientProcessor
//
// The AGClientProcessor calls this record when the 
// AGDBConfig says that the server wants all the records
// from a database. 
static int32 getNextRecord(void *out, AGRecord **record,
                                    int32 *errCode) 
{
    *errCode = AGCLIENT_OPEN_ERR;
    return AGCLIENT_ERR;
    //PENDING(klobad) send up some bogus records??
//    return AGCLIENT_IDLE;
}

static int32 openDatabase(void *out, AGDBConfig *db, int32 *errCode) 
{
    ClientState *state = (ClientState *)out;

    return AGCLIENT_IDLE;
}

static int32 nextExpansionCmd(void *out,
                               int32 *newCommand,
                               int32 *commandLength,
                               void **commandBytes)
{
    ClientState *state = (ClientState *)out;

#ifdef SEND_TEST_EXPANSION_CMD
    if(!state->expansionData) {
        AGBufferWriter writer;
        char *data = "This is an expansion client command";
        state->expansionData = strdup(data);

        AGBufferWriterInit(&writer, 512);
        AGWriteCompactInt((AGWriter *)&writer, 
            strlen(state->expansionData) + 1);
        AGWriteBytes((AGWriter *)&writer, data, 
            strlen(state->expansionData) + 1);

        *newCommand = AG_XMLDATA_CMD;
        *commandLength = AGBufferWriterGetBufferSize(&writer);
        *commandBytes = AGBufferWriterGetBuffer(&writer);
        writer.buffer = NULL;
        AGBufferWriterFinalize(&writer);
        return AGCLIENT_CONTINUE;
    } else {
        free(state->expansionData);
        state->expansionData = NULL;
        return AGCLIENT_IDLE;
    }
#endif

    return AGCLIENT_IDLE;
}

static int32 printTASK(void *out, int32 *returnErrorCode, char *currentTask, 
                       AGBool bufferable)
{
    printf("AG_TASK_CMD\n");
    printf("\t currentTask: %s\n", currentTask);
    printf("\t bufferable: %d\n", bufferable);
    return AGCLIENT_CONTINUE;
}

static int32 printITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem)
{
    printf("AG_ITEM_CMD\n");
    printf("\t currentItemNumber: %d\n", currentItemNumber);
    printf("\t totalItemCount: %d\n", totalItemCount);
    printf("\t currentItem: %s\n", currentItem);
    return AGCLIENT_CONTINUE;
}

static int32 printDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname)
{
    printf("AG_DELETEDATABASE_CMD\n");
    printf("\t dbname: %s\n", dbname);
    return AGCLIENT_CONTINUE;
}

static int32 printOPENDATABASE(void *out, int32 *returnErrorCode,
                                           char *dbname)
{
    printf("AG_OPENDATABASE_CMD\n");
    printf("\t dbname: %s\n", dbname);
    return AGCLIENT_CONTINUE;
}

static int32 printCLOSEDATABASE(void *out, int32 *returnErrorCode)
{
    printf("AG_CLOSEDATABASE_CMD\n");
    return AGCLIENT_CONTINUE;
}

static int32 printCLEARMODS(void *out, int32 *returnErrorCode)
{
    printf("AG_CLEARMODS_CMD\n");
    return AGCLIENT_CONTINUE;
}

static int32 printGOODBYE(void *out, int32 *returnErrorCode,
                                      AGSyncStatus syncStatus,
                                      int32 errorCode,
                                      char *errorMessage)
{
    printf("AG_GOODBYE_CMD\n");
    printf("\t syncStatus: %d\n", syncStatus);
    printf("\t errorCode: %d\n", errorCode);
    printf("\t errorMsg: %s\n", errorMessage);
    return AGCLIENT_CONTINUE;
}

static int32 printRECORD(void *out, int32 *returnErrorCode,
                                     int32 *newUID,
                                     int32 uid,
                                     AGRecordStatus mod,
                                     int32 recordDataLength,
                                     void *recordData,
                                     int32 platformDataLength,
                                     void *platformData)
{
    ClientState *cs = (ClientState *)out;
    AGCommandProcessor *cp = cs->cp;

    printf("AG_RECORD_CMD\n");
    if(mod != AG_RECORD_NEW_TEMPORARY_UID) {
        printf("\t uid: %d\n", uid);
    } else {
        //NOTE: This would really save the record and ask the device what id was
        //assigned. It needs to record this mapping between the temporary and
        //the real uid to the dbconfig.
        printf("\t server sent temporary uid: %d\n", uid);
        printf("\t assigning it to new uid: %d\n", uid/2); 
        *newUID = uid/2;
        //HACK HACK HACK 
        //This causes the top level while loop to loop forever 
        //(well three times really)
        //so we can see what the NEWIDS are. 
        //DO NOT DO THIS IN A REAL CLIENT!!!!
        cp->syncAgain = TRUE;
    }
    printf("\t mod: %d\n", mod);
    printf("\t recordDataLength: %d\n", recordDataLength);
    printf("\t recordData: %s\n", recordData);
    printf("\t platformDataLength: %d\n", platformDataLength);
    printf("\t platformData: %d\n", platformData);
    
    return AGCLIENT_CONTINUE;
}

static int32 printEXPANSION(void *out, int32 *returnErrorCode,
                                        int32 expansionCommand, 
                                        int32 commandLength,
                                        void *commandBytes)
{
    printf("AG_EXPANSION_CMD\n");
    printf("\t command: %d\n", expansionCommand);
    printf("\t commandLen: %d\n", commandLength);
    printf("\t command bytes: %s\n", commandBytes);
    return AGCLIENT_CONTINUE;
}

/*---------------------------------------------------------------------------*/
typedef struct Glue {
    AGServerConfig *config;
    AGDeviceInfo *deviceInfo;
} Glue;

void xmlElement(void *userData, const char *name, const char **atts)
{
    Glue *glue;
    int i = 0;
    char *attr;
    const char *tmp;

    glue = (Glue *)userData;
    if (strcmp("Server", name) == 0) {
        /* look for "hostname", "port", "username", "password", "disabled"
          "nonce"
        */
        while (atts[i] != 0) {
            attr = (char *)atts[i++];
            if (strcmp("hostname", attr) == 0) {
                if (glue->config->serverName != NULL)
                    free(glue->config->serverName);
                glue->config->serverName = strdup(atts[i++]);
            } else if (strcmp("port", attr) == 0) {
                glue->config->serverPort = atoi(atts[i++]);
            } else if (strcmp("username", attr) == 0) {
                if (glue->config->userName != NULL)
                    free(glue->config->userName);
                glue->config->userName = strdup(atts[i++]);
            } else if (strcmp("password", attr) == 0) {
                tmp = atts[i++];
                AGMd5((char *)tmp, strlen(tmp), glue->config->password);
            } else if (strcmp("disabled", attr) == 0) {
                if (strcmp("TRUE", atts[i++]) == 0)
                    glue->config->disabled = TRUE;
                else
                    glue->config->disabled = FALSE;
            }
        }
    } else if (strcmp("Device", name) == 0) {
        // look for "os", "version", "screenWidth", "screenHeight", 
        // "colorDepth", "serialNumber", "availableBytes"
        while (atts[i] != 0) {
            attr = (char *)atts[i++];
            if (strcmp("os", attr) == 0) {
                if (glue->deviceInfo->osName != NULL)
                    free(glue->deviceInfo->osName);
                glue->deviceInfo->osName = strdup(atts[i++]);
            } else if (strcmp("version", attr) == 0) {
                if (glue->deviceInfo->osVersion != NULL)
                    free(glue->deviceInfo->osVersion);
                glue->deviceInfo->osVersion = strdup(atts[i++]);
            } else if (strcmp("screenWidth", attr) == 0) {
                glue->deviceInfo->screenWidth = atoi(atts[i++]);
            } else if (strcmp("screenHeight", attr) == 0) {
                glue->deviceInfo->screenHeight = atoi(atts[i++]);
            } else if (strcmp("colorDepth", attr) == 0) {
                glue->deviceInfo->colorDepth = atoi(atts[i++]);
            } else if (strcmp("serialNumber", attr) == 0) {
                if (glue->deviceInfo->serialNumber != NULL)
                    free(glue->deviceInfo->serialNumber);
                glue->deviceInfo->serialNumber = strdup(atts[i++]);
            } else if (strcmp("availableBytes", attr) == 0) {
                glue->deviceInfo->availableBytes = atoi(atts[i++]);
            }
        }
    } else {
        while (atts[i] != 0) {
            puts(atts[i++]);
            putchar('\t');
            puts(atts[i++]);
        }
    }


}

static int32 readConfigFromFile(char *filename, 
                                AGServerConfig *serverConfig, 
                                AGDeviceInfo *deviceInfo)
{
    char buffer[BUFSIZ];
    XML_Parser parser;
    Glue *glueData;
    FILE *stream;
    int32 done;
    AGBool errored = FALSE;

    stream = fopen(filename, "rb");
    if (stream == NULL) {
        printf("failed to load data from: %s\n", filename);
        return -1;
    }

    glueData = (Glue *)malloc(sizeof(Glue));
    memset(glueData, 0, sizeof(*glueData));
    glueData->config = serverConfig;
    glueData->deviceInfo = deviceInfo;
    parser = XML_ParserCreate(NULL);
    XML_SetUserData(parser, glueData);
    XML_SetElementHandler(parser, xmlElement, NULL);

    do {
        size_t len = fread(buffer, 1, sizeof(buffer), stream);
        if (ferror(stream)) {
            errored = TRUE;
            break;
        }
        done = len < sizeof(buffer);
        if (!XML_Parse(parser, buffer, len, done)) {
            fprintf(stderr, "%s at line %d\n",
                XML_ErrorString(XML_GetErrorCode(parser)),
                XML_GetCurrentLineNumber(parser));
            errored = TRUE;
            break;
        }
    } while (!done);

    XML_ParserFree(parser);
    free(glueData);

    fclose(stream);
    if (errored) {
        printf("failed to load data from: %s\n", filename);
        return -1;
    }

    printf("malclient data from: %s\n", filename);
    return 0;
}

