/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <windowsx.h>
#include <AGProtocol.h>
#include <AGUtil.h>
#include <AGDeviceInfo.h>
#include <AGServerConfig.h>
#include <AGClientProcessor.h>
#include <AGBufferReader.h>
#include <AGSyncCEDeviceDLLTest.h>
#include <AGSyncCEStatusDialog.h>
#include <AGSyncCEResourceWinCE.h>

typedef struct _CESyncInfo {

    AGDeviceInfo * deviceInfo;
    AGServerConfig * serverConfig;
    AGRecord * record;
    AGSyncCEStatusDialog * statusDialog;
    HANDLE hFile;
    CEOID oDB;
    long currentIndex;

} CESyncInfo;

static HMODULE v_hInst = NULL;

static AGBool createDatabase(char * dbname)
{
	SORTORDERSPEC sorts[2];
    BOOLEAN ret;
    LPWSTR wBuf = NULL;

    sorts[0].propid = UID_ID;
	sorts[0].dwFlags = 0;
	sorts[1].propid = STATUS_ID;
	sorts[1].dwFlags = 0;

    // Create the database
    AnsiToUnicode(dbname, &wBuf);
    if (CeCreateDatabase(wBuf, 1, 2, sorts))
        ret = TRUE;
    else
        ret = FALSE;
    free((void *)wBuf);

    return TRUE;
}

static long openDatabase(CESyncInfo * pInfo, char * dbname, AGBool create)
{
    LPWSTR wBuf = NULL;
    AnsiToUnicode(dbname, &wBuf);
    pInfo->hFile = CeOpenDatabase(&pInfo->oDB, wBuf, UID_ID, 0, 0);

    if (pInfo->hFile == INVALID_HANDLE_VALUE && create) {
        createDatabase(dbname);
        pInfo->hFile = CeOpenDatabase(&pInfo->oDB, wBuf, UID_ID, 0, 0);
    }

    free((void *)wBuf);

    return (pInfo->hFile == INVALID_HANDLE_VALUE) ? 0 : 1;

    return 1;
}

static void closeDatabase(CESyncInfo * pInfo)
{
    CloseHandle(pInfo->hFile);
    pInfo->hFile = INVALID_HANDLE_VALUE;
    pInfo->oDB = NULL;

}

static AGBool deleteDatabase(CESyncInfo * pInfo, char * dbname)
{
    LPWSTR wBuf = NULL;
    AnsiToUnicode(dbname, &wBuf);

    CEOID oid = NULL;
	HANDLE handle = CeOpenDatabase(&oid, wBuf, 0, NULL, 0);
    free((void *)wBuf);

	if (handle != INVALID_HANDLE_VALUE) {
		CloseHandle(handle);
		return (AGBool)CeDeleteDatabase(oid);
	}

    return TRUE;
}

static void writeRecord(CESyncInfo * pInfo,
                        long uid,
                        AGRecordStatus mod, 
                        long recordDataLength,
                        void * recordData)
{
	CEPROPVAL prop;
	unsigned long index;

	prop.propid = UID_ID;
	prop.wFlags = 0;
	prop.val.lVal = uid;
	CEOID oid = CeSeekDatabase(pInfo->hFile, CEDB_SEEK_VALUEFIRSTEQUAL,
        (long)&prop, &index);
	CEPROPVAL props[4];

	props[0].propid = UID_ID;
	props[1].propid = STATUS_ID;
    props[2].propid = BYTES_ID;

    props[0].wFlags = 0;
	props[0].val.ulVal = uid;
    if (oid) {
        if (AG_RECORD_DELETED == mod) {

	        props[1].wFlags = 0;
	        props[1].val.ulVal = AG_RECORD_DELETED;

	        props[2].wFlags = 0;
            props[2].val.blob.dwCount = 0;
            props[2].val.blob.lpb = (unsigned char *)0;

	        CeWriteRecordProps(pInfo->hFile, oid, 3, props);
        } else {
            DWORD numberOfBytesWritten = 0;

	        props[1].wFlags = 0;
	        props[1].val.uiVal = (unsigned short)0;

            props[2].wFlags = 0;
            props[2].val.blob.dwCount = recordDataLength;
            props[2].val.blob.lpb = (unsigned char *)recordData;

            CeWriteRecordProps(pInfo->hFile, oid, 3, props);
        }
    }

}

static void doCmdAG_DATABASECONFIG_CMD(CESyncInfo * pInfo, AGReader *r)
{
    AGDBConfig * db;
    char * dbname;
    AGDBConfigType config;
    AGBool sendRecordPlatformData;
    int32 platformDataLength;
    void *platformData = NULL;

    AGReadDATABASECONFIG(r, &dbname, &config, &sendRecordPlatformData,
        &platformDataLength, &platformData);

    if (NULL == dbname)
        return;

	if(config == AG_DONTSEND_CFG) {
		db = AGServerConfigDeleteDBConfigNamed(
											pInfo->serverConfig, dbname);
		if(db)
			AGDBConfigFree(db);
		free(dbname);
		if(platformDataLength)
			free(platformData);
	} else {
	    db = AGDBConfigNew(dbname, config, 
	                            sendRecordPlatformData, 
	                            platformDataLength, platformData, NULL);
	    AGServerConfigAddDBConfig(pInfo->serverConfig, db);
	}
}

static void doCmdAG_SERVERCONFIG_CMD(CESyncInfo * pInfo, AGReader *r)
{
    if (NULL != pInfo->serverConfig->userUrl) {
        free(pInfo->serverConfig->userUrl);
        pInfo->serverConfig->userUrl = NULL;
    }
    if (NULL != pInfo->serverConfig->description) {
        free(pInfo->serverConfig->description);
        pInfo->serverConfig->description = NULL;
    }
    if (NULL != pInfo->serverConfig->serverUri) {
        free(pInfo->serverConfig->serverUri);
        pInfo->serverConfig->serverUri = NULL;
    }

    AGReadSERVERCONFIG(r,
        &pInfo->serverConfig->friendlyName,
        &pInfo->serverConfig->userUrl,
        &pInfo->serverConfig->description,
        &pInfo->serverConfig->serverUri,
        &pInfo->serverConfig->hashPassword,
        &pInfo->serverConfig->connectTimeout,
        &pInfo->serverConfig->writeTimeout,
        &pInfo->serverConfig->readTimeout);
}

static void doCmdAG_COOKIE_CMD(CESyncInfo * pInfo, AGReader *r)
{
    if (NULL != pInfo->serverConfig->sequenceCookie) {
        free(pInfo->serverConfig->sequenceCookie);
        pInfo->serverConfig->sequenceCookie = NULL;
    }
    AGReadCOOKIE(r, &pInfo->serverConfig->sequenceCookieLength,
        (void**)&pInfo->serverConfig->sequenceCookie);
}

static void doCmdAG_TASK_CMD(CESyncInfo * pInfo, AGReader *r)
{
    char *currentTask;
    AGBool bufferable;

    AGReadTASK(r, &currentTask, &bufferable);
    AGSyncCEStatusDialogSetTask(pInfo->statusDialog, currentTask);
    if (NULL != currentTask)
        free(currentTask);
}

static void doCmdAG_ITEM_CMD(CESyncInfo * pInfo, AGReader *r)
{
    int32 currentItemNumber;
    int32 totalItemCount;
    char * currentItem;

    AGReadITEM(r, &currentItemNumber, &totalItemCount, &currentItem);
    AGSyncCEStatusDialogSetTask(pInfo->statusDialog, currentItem);
    AGSyncCEStatusDialogSetBar(pInfo->statusDialog, currentItemNumber,
        totalItemCount);
    if (NULL != currentItem)
        free(currentItem);
}

static void doCmdAG_DELETEDATABASE_CMD(CESyncInfo * pInfo, AGReader *r)
{
    char * dbname = NULL;

    AGReadDELETEDATABASE(r, &dbname);
    if (NULL != dbname) {
        deleteDatabase(pInfo, dbname);
        free(dbname);
    }
}

static void doCmdAG_OPENDATABASE_CMD(CESyncInfo * pInfo, AGReader *r)
{
    char * dbname = NULL;

    AGReadOPENDATABASE(r, &dbname);
    if (NULL != dbname) {
        openDatabase(pInfo, dbname, TRUE);
        free(dbname);
    }
}

static void doCmdAG_CLOSEDATABASE_CMD(CESyncInfo * pInfo, AGReader *r)
{
    AGReadCLOSEDATABASE(r);
    closeDatabase(pInfo);
}

static void doCmdAG_CLEARMODS_CMD(CESyncInfo * pInfo, AGReader *r)
{
    AGReadCLEARMODS(r);
    // Pending...clear mods
}

static void doCmdAG_SENDDEVICEINFO_CMD(CESyncInfo * pInfo, AGReader *r)
{
    AGReadSENDDEVICEINFO(r, &pInfo->serverConfig->sendDeviceInfo);
    // Pending...send device info
}

static void doCmdAG_GOODBYE_CMD(CESyncInfo * pInfo, AGReader *r)
{
    AGSyncStatus syncStatus;
    int32 errorCode;
    char * errorMsg;

    AGReadGOODBYE(r, &syncStatus, &errorCode, &errorMsg);
    if (NULL != errorMsg)
        free(errorMsg);
}

static void doCmdAG_NONCE_CMD(CESyncInfo * pInfo, AGReader *r)
{
    AGReadNONCE(r, pInfo->serverConfig->nonce);
}

static void doCmdAG_UNKNOWN_CMD(CESyncInfo * pInfo,
                                uint32 length,
                                AGReader *r)
{
    if (length > 0) {
        void * buf = malloc(length);
        AGReadBytes(r, buf, length);
        free(buf);
    }
}

static void doCmdAG_RECORD_CMD(CESyncInfo * pInfo, AGReader *r)
{
    int32 uid;
    AGRecordStatus mod;
    int32 recordDataLength;
    void * recordData;
    int32 platformDataLength;
    void * platformData;

    AGReadRECORD(r, &uid, &mod, &recordDataLength, &recordData,
        &platformDataLength, &platformData);

    writeRecord(pInfo, uid, mod, recordDataLength, recordData);

    if (NULL != recordData)
        free(recordData);
    if (NULL != platformData)
        free(platformData);
}

#define HANDLE_COMMAND(message, buf, r)    \
    case (message): doCmd##message((CESyncInfo*)buf, (AGReader*)r); break
int32 PerformCommand(void * out, AGCommand command, uint32 length,
                     void * bytes, int32 *errCode)
{
    CESyncInfo * pInfo = (CESyncInfo *)out;
    AGBufferReader * reader = AGBufferReaderNew((uint8*)bytes);
    int32 result = AGCLIENT_CONTINUE;

    switch(command) {
        HANDLE_COMMAND(AG_SERVERCONFIG_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_SENDDEVICEINFO_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_TASK_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_ITEM_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_DATABASECONFIG_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_OPENDATABASE_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_RECORD_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_CLEARMODS_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_CLOSEDATABASE_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_DELETEDATABASE_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_COOKIE_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_NONCE_CMD, pInfo, reader);
        HANDLE_COMMAND(AG_GOODBYE_CMD, pInfo, reader);
        case AG_END_CMD:
            AGReadEND((AGReader*)reader);
            result = AGCLIENT_IDLE;
            break;
        default:
            doCmdAG_UNKNOWN_CMD(pInfo, length, (AGReader*)reader);
            result = AGCLIENT_CONTINUE;
            break;
    }
    AGBufferReaderFree(reader);

    return result;
}

static PCEPROPVAL getPropVal(PCEPROPVAL props,
                             DWORD nProps,
                             unsigned long id)
{
    unsigned i;

    for (i = 0; i < nProps; i++) {
        PCEPROPVAL prop = props + i;

        if (prop->propid == id) {
            return prop;
        }
    }
    return NULL;
}


static AGBool readRecord(CESyncInfo * pInfo)
{
    PCEPROPVAL props;
	unsigned short nProps;
	DWORD propsLength = 0;
	CEOID oid = CeReadRecordProps(pInfo->hFile, CEDB_ALLOWREALLOC,
					        &nProps,     // return count of properties
					        NULL,        // retrieve all properties
					        (LPBYTE*)&props, // buffer to return prop data
					        (LPDWORD)&propsLength);

    if (oid == 0)
        return FALSE;

    PCEPROPVAL uidProp = getPropVal(props, nProps, UID_ID), 
               statusProp = getPropVal(props, nProps, STATUS_ID),
               bytesProp = getPropVal(props, nProps, BYTES_ID);

    if (uidProp)
        pInfo->record->uid = uidProp->val.ulVal;
    if (statusProp)
        pInfo->record->status = (AGRecordStatus)statusProp->val.ulVal;
    if (bytesProp) {
        pInfo->record->recordDataLength = bytesProp->val.blob.dwCount;
        pInfo->record->recordData = bytesProp->val.blob.lpb;
    }

    return TRUE;
}

int32 OpenDatabase(void *out, AGDBConfig * db,
                   int32 * errCode)
{
    CESyncInfo * pInfo = (CESyncInfo *)out;

    long result = openDatabase(pInfo, db->dbname, FALSE);

    if (!result)
        return AGCLIENT_ERR;

    pInfo->currentIndex = 0;
    CeSeekDatabase(pInfo->hFile, CEDB_SEEK_BEGINNING, 0, NULL);

    return AGCLIENT_IDLE;
}

static int32 leaveGetRecord(CESyncInfo * pInfo, int32 result)
{
    return result;
}

static int32 getRecordBase(CESyncInfo * pInfo, AGBool modonly,
                           AGRecord ** record, int32 * errCode)
{
    int32 result = AGCLIENT_NO_ERR;
    if (modonly) {
        PCEPROPVAL props;
	    unsigned short nProps;
	    DWORD propsLength = 0;
        AGBool done = FALSE;

        while (!done) {
            pInfo->currentIndex++;
            CeSeekDatabase(pInfo->hFile, CEDB_SEEK_BEGINNING,
                pInfo->currentIndex, NULL);
            CEOID oid = CeReadRecordProps(pInfo->hFile, CEDB_ALLOWREALLOC,
					        &nProps,     // return count of properties
					        NULL,        // retrieve all properties
					        (LPBYTE*)&props,    // buffer to return prop data
					        (LPDWORD)&propsLength);
            if (oid == 0)
                result = AGCLIENT_IDLE;

            if (props) {
                PCEPROPVAL statusProp = getPropVal(props, nProps, STATUS_ID);

                if ((AGRecordStatus)statusProp->val.ulVal !=
                    AG_RECORD_UNMODIFIED) {
                    readRecord(pInfo);
                    done = TRUE;
                }
            } else {
                done = TRUE;
            }
        }
    } else {
        pInfo->currentIndex++;
        CeSeekDatabase(pInfo->hFile, CEDB_SEEK_BEGINNING,
            pInfo->currentIndex, NULL);
        if (!readRecord(pInfo))
            result = AGCLIENT_IDLE;
    }
    if (AGCLIENT_NO_ERR != result) {
        closeDatabase(pInfo);
        return leaveGetRecord(pInfo, AGCLIENT_IDLE);
    }
    return leaveGetRecord(pInfo, AGCLIENT_CONTINUE);
}

int32 GetNextModifiedRecord(void *out, AGRecord **record, 
                            int32 *errCode) 
{
    return getRecordBase((CESyncInfo *)out, TRUE, record, errCode);
}

int32 GetNextRecord(void *out, AGRecord **record,
                    int32 *errCode) 
{
    return getRecordBase((CESyncInfo *)out, FALSE, record, errCode);
}

void * Initialize(AGServerConfig * serverConfig)
{
    CESyncInfo * pInfo;

    pInfo = (CESyncInfo *)malloc(sizeof(CESyncInfo));
    if (NULL == pInfo)
        return NULL;
    memset(pInfo, 0, sizeof(CESyncInfo));

    pInfo->serverConfig = serverConfig;
    pInfo->statusDialog = AGSyncCEStatusDialogNew(v_hInst);

    return pInfo;

}

void Terminate(void * out)
{
    CESyncInfo * pInfo = (CESyncInfo *)out;

    if (NULL != pInfo) {

        if (NULL != pInfo->statusDialog) {
            AGSyncCEStatusDialogFree(pInfo->statusDialog);
            pInfo->statusDialog = NULL;
        }

        free(pInfo);
        pInfo = NULL;

    }
}

BOOL WINAPI DllMain (HANDLE hInst, ULONG ulReason, LPVOID lpReserved)
{
    switch(ulReason)
    {
    case DLL_PROCESS_ATTACH:
        v_hInst = hInst;
        break;
    }
    return TRUE;
}
