/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

#include <AGServerConfig.h>
#include <AGBufferReader.h>
#include <AGBufferWriter.h>
#include <AGUtil.h>
#include <AGDigest.h>

#ifndef REMOVE_SYNCHRONIZE_FEATURE
#include <AGSynchronize.h>
#endif

/* Version 0 - original creation */
#define RECORD_VERSION_0 (0)

/* Version 1 - added allowSecureConnection */
#define RECORD_VERSION_1 (1)

/* This is the version number that is written to new records */
#define CURRENT_RECORD_VERSION (RECORD_VERSION_1)

ExportFunc AGServerConfig *AGServerConfigNew()
{
    AGServerConfig *config;

    config = malloc(sizeof(AGServerConfig));
    AGServerConfigInit(config);
    return config;
}

ExportFunc AGServerConfig *AGServerConfigInit(AGServerConfig *config)
{
    bzero(config, sizeof(*config));

    /* pending(miket):  Until we implement cleartext handling,
    default to hashing all the time. */
    config->hashPassword = TRUE;

    return config;
}

static void freeDBConfigArray(AGArray * array)
{
    if (array) {
        int32 count;
        AGDBConfig *dbconfig;

        count = AGArrayCount(array);
        while(count--) {
            dbconfig = AGArrayElementAt(array, count);
            AGDBConfigFree(dbconfig);
        }
        AGArrayFree(array);
    }
}

ExportFunc void AGServerConfigFinalize(AGServerConfig *config)
{
    if (config->serverName)
        free(config->serverName);
    if (config->userName)
        free(config->userName);
    if (config->cleartextPassword)
        free(config->cleartextPassword);
    if (config->friendlyName)
        free(config->friendlyName);
    if (config->userUrl)
        free(config->userUrl);
    if (config->description)
        free(config->description);
    if (config->serverUri)
        free(config->serverUri);
    if (config->sequenceCookie)
        free(config->sequenceCookie);

    if(config->dbconfigs)
       freeDBConfigArray(config->dbconfigs);

    bzero(config, sizeof(*config));
}

ExportFunc void AGServerConfigFree(AGServerConfig *config)
{
    AGServerConfigFinalize(config);
    free(config);
}

ExportFunc AGServerConfig *AGServerConfigCopy(AGServerConfig *dstConfig, 
                                              AGServerConfig *srcConfig)
{
    int32 i, n;

    dstConfig->uid = srcConfig->uid;
    dstConfig->status = srcConfig->status;

    if (NULL != srcConfig->serverName)
        dstConfig->serverName = strdup(srcConfig->serverName);
    dstConfig->serverPort = srcConfig->serverPort;
    if (NULL != srcConfig->userName)
        dstConfig->userName = strdup(srcConfig->userName);
    if (NULL != srcConfig->cleartextPassword)
        dstConfig->cleartextPassword = strdup(srcConfig->cleartextPassword);
    memcpy(dstConfig->password, srcConfig->password, 16);
    dstConfig->disabled = srcConfig->disabled;

    if (NULL != srcConfig->friendlyName)
        dstConfig->friendlyName = strdup(srcConfig->friendlyName);
    if (NULL != srcConfig->userUrl)
        dstConfig->userUrl = strdup(srcConfig->userUrl);
    if (NULL != srcConfig->description)
        dstConfig->description = strdup(srcConfig->description);
    if (NULL != srcConfig->serverUri)
        dstConfig->serverUri = strdup(srcConfig->serverUri);
    dstConfig->sequenceCookieLength = srcConfig->sequenceCookieLength;
    if (NULL != srcConfig->sequenceCookie) {
        dstConfig->sequenceCookie = malloc(srcConfig->sequenceCookieLength);
        if (NULL != dstConfig->sequenceCookie)
            memcpy(dstConfig->sequenceCookie, srcConfig->sequenceCookie,
                srcConfig->sequenceCookieLength);
    }
    else
        dstConfig->sequenceCookie = NULL;

    if (NULL != dstConfig->dbconfigs)
        AGArrayFree(dstConfig->dbconfigs);
    dstConfig->dbconfigs = AGArrayNew(AGUnownedPointerElements, 1);

    if (NULL != srcConfig->dbconfigs) {
        n = AGArrayCount(srcConfig->dbconfigs);
        for (i = 0; i < n; ++i)
            AGServerConfigAddDBConfig(dstConfig,
                AGDBConfigDup((AGDBConfig *)AGArrayElementAt(
                srcConfig->dbconfigs, i)));
    }

    memcpy(dstConfig->nonce, srcConfig->nonce, 16);

    dstConfig->sendDeviceInfo = srcConfig->sendDeviceInfo;

    dstConfig->hashPassword = srcConfig->hashPassword;
    dstConfig->connectTimeout = srcConfig->connectTimeout;
    dstConfig->writeTimeout = srcConfig->writeTimeout;
    dstConfig->readTimeout = srcConfig->readTimeout;

    dstConfig->connectSecurely = srcConfig->connectSecurely;
    dstConfig->allowSecureConnection = srcConfig->allowSecureConnection;

    return dstConfig;
}

ExportFunc AGServerConfig *AGServerConfigDup(AGServerConfig *srcConfig)
{
    AGServerConfig * dstConfig = malloc(sizeof(AGServerConfig));

    if (NULL != dstConfig) {
        bzero(dstConfig, sizeof(AGServerConfig));
        return AGServerConfigCopy(dstConfig, srcConfig);
    }
    return NULL;
}

ExportFunc void AGServerConfigReadData(AGServerConfig *config, AGReader *r)
{
    int32 count, i;
    int16 recordVersion;
    AGDBConfig *dbconfig;

    recordVersion = AGReadInt16(r);
    config->uid = AGReadInt32(r);
    config->status = (AGRecordStatus)AGReadInt16(r);
    config->serverName = AGReadCString(r);
    config->serverPort = AGReadInt16(r);

    config->userName = AGReadCString(r);
    config->cleartextPassword = AGReadCString(r);
    if (AGReadInt8(r))
        AGReadBytes(r, config->password, 16);

    /* read in nonce */
    if (AGReadInt8(r))
        AGReadBytes(r, config->nonce, 16);

    if (AGReadInt8(r))
        config->disabled = TRUE;
    else
        config->disabled = FALSE;

    config->friendlyName = AGReadCString(r);
    config->userUrl = AGReadCString(r);
    config->description = AGReadCString(r);
    config->serverUri = AGReadCString(r);

    config->sequenceCookieLength = AGReadInt32(r);
    if (config->sequenceCookieLength > 0) {
        config->sequenceCookie = malloc(config->sequenceCookieLength);
        AGReadBytes(r, config->sequenceCookie, config->sequenceCookieLength);
    }

    count = AGReadInt32(r);
    config->dbconfigs = AGArrayNew(AGUnownedPointerElements, count);
    for (i = 0; i < count; i++) {
        dbconfig = AGDBConfigNewAndReadData(r);
        AGArrayAppend(config->dbconfigs, dbconfig);
    }
    if (AGReadInt8(r))
        config->sendDeviceInfo = TRUE;
    else
        config->sendDeviceInfo = FALSE;

    config->hashPassword = AGReadBoolean(r);
    config->connectTimeout = AGReadCompactInt(r);
    config->writeTimeout = AGReadCompactInt(r);
    config->readTimeout = AGReadCompactInt(r);

    config->connectSecurely = AGReadBoolean(r);

    /* Record version 0:  in theory, all serverconfigs that should exist
    in the world have at least the data above. So if we're reading version
    zero, that's all we get -- initialize the remaining fields and return. */
    if (recordVersion == RECORD_VERSION_0) {
        config->allowSecureConnection = FALSE;
        return;
    }

    /* Record version 1:  added allowSecureConnection. */
    config->allowSecureConnection = AGReadBoolean(r);

    if (recordVersion == RECORD_VERSION_1)
        return;

}

ExportFunc AGServerConfig * AGServerConfigNewAndReadData(AGReader *r)
{
    AGServerConfig * result;

    result = (AGServerConfig *)malloc(sizeof(AGServerConfig));
    bzero(result, sizeof(AGServerConfig));
    AGServerConfigReadData(result, r);
    return result;
}

//PENDING(klobad) this is also in AGDigest, but
//I wanted to reduce the dependancies for the device
static AGBool digestIsNull(uint8 a[16])
{
    int i;
    for(i=0;i<16;i++)
        if(a[i])
            return 0;
    return 1;
}

static void digestSetToNull(uint8 a[16])
{
    int i;
    for(i=0;i<16;i++)
        a[i]=0;
}

ExportFunc void AGServerConfigWriteData(AGServerConfig *config, AGWriter *w)
{
    int32 i, count;
    AGDBConfig *dbconfig;

    AGWriteInt16(w, CURRENT_RECORD_VERSION);
    AGWriteInt32(w, config->uid);
    AGWriteInt16(w, (uint16)config->status);
    AGWriteCString(w, config->serverName);
    AGWriteInt16(w, config->serverPort);

    AGWriteCString(w, config->userName);
    AGWriteCString(w, config->cleartextPassword);

    if (digestIsNull(config->password))
        AGWriteInt8(w, 0);
    else {
        AGWriteInt8(w, 16);
        AGWriteBytes(w, config->password, 16);
    }

    if (digestIsNull(config->nonce))
        AGWriteInt8(w, 0);
    else {
        AGWriteInt8(w, 16);
        AGWriteBytes(w, config->nonce, 16);
    }

    if (config->disabled)
        AGWriteInt8(w, 1);
    else
        AGWriteInt8(w, 0);

    AGWriteCString(w, config->friendlyName);
    AGWriteCString(w, config->userUrl);
    AGWriteCString(w, config->description);
    AGWriteCString(w, config->serverUri);

    AGWriteInt32(w, config->sequenceCookieLength);
    if (config->sequenceCookieLength > 0) {
        AGWriteBytes(w, config->sequenceCookie, 
            config->sequenceCookieLength);
    }

    count = AGArrayCount(config->dbconfigs);
    AGWriteInt32(w, count);
    for (i = 0; i < count; i++) {
        dbconfig = AGArrayElementAt(config->dbconfigs, i);
        AGDBConfigWriteData(dbconfig, w);
    }
    if (config->sendDeviceInfo)
        AGWriteInt8(w, 1);
    else
        AGWriteInt8(w, 0);

    AGWriteBoolean(w, config->hashPassword);
    AGWriteCompactInt(w, config->connectTimeout);
    AGWriteCompactInt(w, config->writeTimeout);
    AGWriteCompactInt(w, config->readTimeout);

    AGWriteBoolean(w, config->connectSecurely);
    AGWriteBoolean(w, config->allowSecureConnection);

}

static void getDBConfigNamed(AGServerConfig *config,  char *dbname,
                             AGDBConfig **dbconfig, uint32 *index)
{
    AGDBConfig *result = NULL;
    int32 i;
    int32 n;

    if(dbconfig)
        *dbconfig = NULL;
    if(index)
        *index = -1;

    if (NULL == config->dbconfigs || NULL == dbname)
        return;

    n = AGArrayCount(config->dbconfigs);
    for (i = 0; i < n; i++) {
        result = (AGDBConfig *)AGArrayElementAt(config->dbconfigs, i);
        //pending(miket): Are database names case-sensitive?
        if (!strcmp(result->dbname, dbname)) {
            if(dbconfig)
                *dbconfig = result;
            if(index)
                *index = i;
            return;
        }
    }
    return;
}

ExportFunc AGDBConfig *AGServerConfigGetDBConfigNamed(AGServerConfig *config, 
                                                    char *dbname)
{
    AGDBConfig *result = NULL;
    getDBConfigNamed(config, dbname, &result, NULL);
    return result;
}

ExportFunc AGDBConfig *AGServerConfigDeleteDBConfigNamed(AGServerConfig *config, 
                                                    char *dbname)
{
    uint32 i;
    AGDBConfig *db;

    getDBConfigNamed(config, dbname, &db, &i);

    if (i == -1 || db == NULL)
        return NULL;

    AGArrayRemoveAt(config->dbconfigs, i);
    return db;
}

ExportFunc void AGServerConfigAddDBConfig(AGServerConfig *config, 
                                                    AGDBConfig *dbconfig)
{
    AGDBConfig *oldConfig;

    if(!config->dbconfigs)
        config->dbconfigs = AGArrayNew(AGUnownedPointerElements, 1);
    if(!config->dbconfigs)
        return;

    oldConfig = AGServerConfigDeleteDBConfigNamed(config, dbconfig->dbname);
    if(oldConfig)
        AGDBConfigFree(oldConfig);
    AGArrayAppend(config->dbconfigs, dbconfig);
}

#ifndef REMOVE_SYNCHRONIZE_FEATURE
static AGArray * dupDBConfigArray(AGArray * src)
{
    int32 i, count;
    AGArray * dest;
    
    count = AGArrayCount(src);
    dest = AGArrayNew(AGUnownedPointerElements, 0);
    for (i=0; i<count; ++i)
        AGArrayInsertAt(dest,
            i,
            AGDBConfigDup((AGDBConfig *)AGArrayElementAt(src, i)));
    return dest;
}

AGServerConfig * AGServerConfigSynchronize(AGServerConfig *agreed,
                                           AGServerConfig *device,
                                           AGServerConfig *desktop)
{
    AGServerConfig * result;
    
    result = malloc(sizeof(AGServerConfig));

    if (NULL != result) {

        AGBool userNameChanged = FALSE;

        bzero(result, sizeof(AGServerConfig));

        result->uid = AGSynchronizeInt32(agreed->uid,
            device->uid,
            desktop->uid);

        result->status = (AGRecordStatus)AGSynchronizeInt32(
            agreed->status,
            device->status,
            desktop->status);

        result->serverName = AGSynchronizeString(agreed->serverName,
            device->serverName,
            desktop->serverName);

        result->serverPort = AGSynchronizeInt16(agreed->serverPort,
            device->serverPort,
            desktop->serverPort);

        result->userName = AGSynchronizeString(agreed->userName,
            device->userName,
            desktop->userName);
        if (NULL == device->userName)
            userNameChanged = TRUE;
        else if (strcmp(device->userName, result->userName))
            userNameChanged = TRUE;

        result->cleartextPassword =
            AGSynchronizeString(agreed->cleartextPassword,
                device->cleartextPassword,
                desktop->cleartextPassword);

        AGSynchronizeStackStruct((void*)result->password,
            (void*)agreed->password,
            (void*)device->password,
            (void*)desktop->password,
            16);

        result->disabled = AGSynchronizeBoolean(agreed->disabled,
            device->disabled,
            desktop->disabled);

        result->friendlyName = AGSynchronizeString(agreed->friendlyName,
            device->friendlyName,
            desktop->friendlyName);

        result->userUrl = AGSynchronizeString(agreed->userUrl,
            device->userUrl,
            desktop->userUrl);

        result->description = AGSynchronizeString(agreed->description,
            device->description,
            desktop->description);

        result->serverUri = AGSynchronizeString(agreed->serverUri,
            device->serverUri,
            desktop->serverUri);

        /* We almost always take the device's cookie, no matter what.
        If the username changed, we should reset the cookie instead.
        This is the same as doing nothing (we zeroed out the result
        structure already). */
        if (device->sequenceCookieLength > 0 && !userNameChanged) {
            result->sequenceCookie = malloc(device->sequenceCookieLength);
            if (NULL != result->sequenceCookie) {
                memcpy(result->sequenceCookie,
                    device->sequenceCookie,
                    device->sequenceCookieLength);
                result->sequenceCookieLength = device->sequenceCookieLength;
            }
        }

        /* We always take the device's set of DBConfigs, no matter what. */
        if (NULL != device->dbconfigs)
            result->dbconfigs = dupDBConfigArray(device->dbconfigs);

        if (userNameChanged)
            digestSetToNull((void*)result->nonce);
        else
            AGSynchronizeStackStruct((void*)result->nonce,
                (void*)agreed->nonce,
                (void*)device->nonce,
                (void*)desktop->nonce,
                16);

        result->hashPassword = AGSynchronizeBoolean(agreed->hashPassword,
            device->hashPassword,
            desktop->hashPassword);

        result->sendDeviceInfo = AGSynchronizeBoolean(agreed->sendDeviceInfo,
            device->sendDeviceInfo,
            desktop->sendDeviceInfo);

        result->connectTimeout = AGSynchronizeBoolean(agreed->connectTimeout,
            device->connectTimeout, desktop->connectTimeout);

        result->writeTimeout = AGSynchronizeBoolean(agreed->writeTimeout,
            device->writeTimeout, desktop->writeTimeout);

        result->readTimeout = AGSynchronizeBoolean(agreed->readTimeout,
            device->readTimeout, desktop->readTimeout);

        result->connectSecurely =
            AGSynchronizeBoolean(agreed->connectSecurely,
                device->connectSecurely,
                desktop->connectSecurely);

        result->allowSecureConnection =
            AGSynchronizeBoolean(agreed->allowSecureConnection,
                device->allowSecureConnection,
                desktop->allowSecureConnection);
    }


    return result;

}
#endif /*#ifndef REMOVE_SYNCHRONIZE_FEATURE */

//PENDING(klobad) confirm these are valid validity checks
ExportFunc AGBool AGServerConfigIsValid(AGServerConfig *config)
{
    if(config == NULL)
        return FALSE;

    if(config->friendlyName == NULL)
        return FALSE;

    if(0 == strlen(config->friendlyName))
        return FALSE;

    if(config->serverName == NULL)
        return FALSE;

    if(0 == strlen(config->serverName))
        return FALSE;

    if(config->serverPort <= 0)
        return FALSE;

    if(config->userName == NULL)
        return FALSE;

    if(0 == strlen(config->userName))
        return FALSE;

    return TRUE;
}

void AGServerConfigResetCookie(AGServerConfig *config)
{
    config->sequenceCookieLength = 0;
    if (NULL != config->sequenceCookie) {
        free(config->sequenceCookie);
        config->sequenceCookie = NULL;
    }
}

void AGServerConfigResetNonce(AGServerConfig *config)
{
    // Wanted to remove the depandancy on AGDigestSetToNull
    // so I didn't need AGDigest on the client.
    digestSetToNull(config->nonce);
}
