/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <AGConfigUtil.h>
#include <AGConfigDialogBase.h>
#include <AGUserConfig.h>
#include <AGConfigResource.h>
#include <AGSyncCommon.h>
#include <AGShlapi.h>
#include <AGMobileLinkSettings.h>
#include <stdio.h>
#include <windowsx.h>

extern HINSTANCE g_hInstance;

/* ----------------------------------------------------------------------------
*/
char * AGConfigWindowTextDup(HWND hwnd)
{
    int len;
    char * result = NULL;
    len = GetWindowTextLength(hwnd) + 1;
    if (len != 0) {
        result = (char*)malloc(len);
        GetWindowText(hwnd, result, len);
    }
    return result;
}

/* ----------------------------------------------------------------------------
*/
int AGConfigAddToolTip(HWND toolTip, int itemID, int stringID)
{
    TOOLINFO tooltip;

    tooltip.cbSize = sizeof(TOOLINFO);
    tooltip.uFlags = TTF_IDISHWND | TTF_SUBCLASS; 
    tooltip.hwnd = GetParent(toolTip);
    tooltip.rect;
    tooltip.lParam = 0;
    
    tooltip.uId = (UINT)GetDlgItem(GetParent(toolTip), itemID);
    tooltip.lpszText = MAKEINTRESOURCE(stringID);
    return SendMessage(toolTip,
        TTM_ADDTOOL,
        (WPARAM)0,
        (LPARAM)(LPTOOLINFO)&tooltip);
}

/* ----------------------------------------------------------------------------
*/
static AGBool isFilledIn(HWND hwndCtl)
{
    return (GetWindowTextLength(hwndCtl) > 0);
}

/* ----------------------------------------------------------------------------
*/
static AGBool checkButtons(uint32 nHwnd, HWND * hwnd)
{
    uint32 i;
    AGBool ok = TRUE;

    for (i = 0; i < nHwnd; ++i) {

        if (!isFilledIn(hwnd[i]))
            ok = FALSE;
    }

    return ok;
}

/* ----------------------------------------------------------------------------
*/
static int32 doExclude(int32 source, int32 exclude)
{
    return source & (~exclude);
}

/* ----------------------------------------------------------------------------
*/
void AGConfigUtilCheckWizButtons(uint32 nHwnd, HWND * hwnd, int32 exclude)
{
    if (0 == nHwnd)
        PropSheet_SetWizButtons(GetParent(hwnd[0]),
            doExclude(PSWIZB_BACK | PSWIZB_NEXT, exclude));
    else
        PropSheet_SetWizButtons(GetParent(GetParent(hwnd[0])),
            doExclude(PSWIZB_BACK | (checkButtons(nHwnd, hwnd)
                ? PSWIZB_NEXT
                : 0), exclude));
}

/* ----------------------------------------------------------------------------
*/
void AGConfigUtilCheckDialogButtons(uint32 nHwnd, HWND * hwnd)
{
    if (0 == nHwnd)
        EnableWindow(GetDlgItem(hwnd[0], IDOK), TRUE);
    else
        EnableWindow(GetDlgItem(GetParent(hwnd[0]), IDOK),
            checkButtons(nHwnd, hwnd));
}

/* ----------------------------------------------------------------------------
*/
AGBool AGConfigUtilIsServerInList(char * userConfigPath,
                                  char * serverName,
                                  int16 serverPort)
{
    AGBool result = FALSE;
    AGUserConfig * userConfig;

    userConfig = AGReadUserConfigFromDisk(userConfigPath);

    if (NULL == userConfig)
        return FALSE;

    result = (NULL != AGUserConfigGetServerByNameAndPort(userConfig,
        serverName, serverPort, AGUSERCONFIG_FIND));

    AGUserConfigFree(userConfig);

    return result;
}

/* ----------------------------------------------------------------------------
    disallow flag indicates that the error message will tell the user that
    duplicates are not allowed, and that the message box will simply inform
    the user of that fact.  If it's false, it will ask the user if it's ok
    to replace the previous instance.
*/
int AGConfigUtilDisplayDuplicateErrorMessage(HWND hwnd, AGBool disallow)
{
    int c;
    char message[MAX_PATH], title[MAX_PATH];

    c = LoadString(g_hInstance, IDS_DUPLICATE_SERVER_TITLE,
        title, MAX_PATH);

    c = LoadString(g_hInstance,
        disallow ? IDS_DUPLICATE_SERVER_MESSAGE_DISALLOW
            : IDS_DUPLICATE_SERVER_MESSAGE,
        message,
        MAX_PATH);

    return MessageBox(hwnd,
        message,
        title,
        (disallow ? MB_OK : MB_OKCANCEL) | MB_ICONEXCLAMATION);
}

/* ----------------------------------------------------------------------------
*/
void AGConfigUtilAddServerToUserConfigFile(AGServerConfig * sc,
                                           char * filename)
{
    HANDLE f;
    AGUserConfig * userConfig;

    userConfig = AGReadUserConfigFromDiskAtomically(filename, &f);

    if (NULL == userConfig)
        userConfig = AGUserConfigNew();

    AGUserConfigAddServerToDesktop(userConfig, sc);

    AGWriteUserConfigToDiskAtomically(userConfig, f);

    AGUserConfigFree(userConfig);
}

/* ----------------------------------------------------------------------------
*/
void AGConfigUtilGetCurrentUserConfigFilename(char * userConfig,
                                              char * syncUserConfig)
{
    AGDeviceEntry * deviceEntry;
    char * filename;

    deviceEntry = AGMobileLinkGetCurrentDevice();

    if (NULL != deviceEntry) {

        if (NULL != deviceEntry->prefsPath) {

            if (NULL != userConfig) {

                filename = (char*)AGSyncCommonGetStringConstant(
                    agPreferencesFilename, FALSE);
                PathCombine(userConfig,
                    deviceEntry->prefsPath,
                    filename);
                free(filename);

            }

            if (NULL != syncUserConfig) {

                filename = (char*)AGSyncCommonGetStringConstant(
                    agSynchronizedPreferencesFilename, FALSE);
                PathCombine(syncUserConfig,
                    deviceEntry->prefsPath,
                    filename);
                free(filename);

            }

        }

        AGMobileLinkDeviceEntryFree(deviceEntry);

    }

}

/* ----------------------------------------------------------------------------
*/
void AGConfigUtilRemoveServerFromUserConfigFile(char * serverName,
                                                int16 serverPort,
                                                char * filename)
{
    HANDLE f;
    AGUserConfig * userConfig;

    userConfig = AGReadUserConfigFromDiskAtomically(filename, &f);

    if (NULL != userConfig)
        AGUserConfigRemoveServerFromDesktop(userConfig, serverName,
            serverPort);

    AGWriteUserConfigToDiskAtomically(userConfig, f);

    if (NULL != userConfig)
        AGUserConfigFree(userConfig);
}

HWND AGConfigUtilAddToolTips(HWND parent, const AGToolTip * tips, int n)
{
    HWND result = CreateWindowEx(0,
        (LPSTR)TOOLTIPS_CLASS,
        (LPSTR)NULL,
        WS_POPUP | TTS_ALWAYSTIP,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        CW_USEDEFAULT, 
        parent,
        (HMENU)NULL,
        g_hInstance,
        NULL);

    if (IsWindow(result)) {

        for (int i = 0; i < n; ++i) {

            AGConfigAddToolTip(result,
                tips[i].controlID,
                tips[i].stringID);

        }

    }

    return result;

}

/* ----------------------------------------------------------------------------
*/
static BOOL onInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
    char buf[MAX_PATH];
    char buf2[MAX_PATH];
    AGServerConfig * serverConfig =
        (AGServerConfig *)lParam;

    GetDlgItemText(hwnd, IDC_WARN_SERVER_INFO, buf, MAX_PATH);

    _snprintf(buf2,
        MAX_PATH,
        buf,
        serverConfig->friendlyName,
        serverConfig->serverName,
        serverConfig->serverPort);

    SetDlgItemText(hwnd, IDC_WARN_SERVER_INFO, buf2);

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static void onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    switch (codeNotify) {
        case BN_CLICKED:
            switch (id) {
                case IDOK:
                case IDCANCEL:
                    EndDialog(hwnd, id);
                    return;
                case ID_MORE_INFO: {

                    char message[MAX_PATH * 3], caption[MAX_PATH];
                    LoadString(g_hInstance,
                        IDS_MORE_INFO_CAPTION,
                        caption,
                        MAX_PATH);

                    LoadString(g_hInstance,
                        IDS_MORE_INFO_MESSAGE,
                        message,
                        MAX_PATH * 3);

                    MessageBox(hwnd,
                        message,
                        caption,
                        MB_OK | MB_ICONINFORMATION);

                }

                default:
                    break;
            }
            break;
        default:
            break;

    }
    FORWARD_WM_COMMAND(hwnd, id, hwndCtl,
        codeNotify, AGConfigDialogBaseProc);
}

/* ----------------------------------------------------------------------------
*/
static BOOL CALLBACK warnDlgProc(HWND hwnd, UINT message,
                                 WPARAM wParam, LPARAM lParam)
{
    switch (message) {
        HANDLE_MSG(hwnd, WM_INITDIALOG, onInitDialog);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
    }

    return 0;   // except for WM_INITDIALOG, returning zero means
                // we didn't process the message.
}

/* ----------------------------------------------------------------------------
*/
AGBool AGConfigUtilWarnUser(HWND parent, AGServerConfig * serverConfig)
{
    return (IDOK == DialogBoxParam(g_hInstance,
        MAKEINTRESOURCE(IDD_WARN_USER),
        parent,
        warnDlgProc, (LONG)serverConfig));
}