/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <AGServerPing.h>
#include <AGSyncCommon.h>
#include <AGConfigUtil.h>
#include <AGDigest.h>
#include <AGMD5.h>
#include <AGConfigServerWizard.h>
#include <AGConfigAddServerDialog.h>
#include <AGProgressDialog.h>
#include <AGUtil.h>
#include <windowsx.h>
#include <prsht.h>

const int NUM_PAGES = 4;

static BOOL reentrantPing = FALSE;

typedef struct swWindowInfo {
    AGBool userPressedFinish;
    AGBool userPressedExpert;
    AGServerConfig * serverConfig;
    char * userConfigPath;
    char * serverName;
    int16 serverPort;
} swWindowInfo;

/* ----------------------------------------------------------------------------
*/
static BOOL introOnInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
    PROPSHEETPAGE * psp = (PROPSHEETPAGE *)lParam;
    SetWindowLong(hwnd, GWL_USERDATA, psp->lParam);

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static LRESULT introOnNotify(HWND hwnd, NMHDR * nmh)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (nmh->code) {

        case PSN_KILLACTIVE:
            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);
            break;

        case PSN_RESET:
            /* reset to the original values */
            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);
            break;

        case PSN_SETACTIVE: {
            HWND h = GetDlgItem(hwnd, IDC_SERVER_WIZ_URL_EDIT);
            AGConfigUtilCheckWizButtons(1, &h, PSWIZB_BACK);

            SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_URL_EDIT),
                info->serverName);
            break;
        }
        case PSN_WIZNEXT: {
            /* the Next button was pressed */

            char * stringConstant;
            AGLocationConfig * locationConfig;
            AGProgressDialog * progressDialog;
            char buffer[MAX_PATH], sbuffer[MAX_PATH];

            if (reentrantPing) {
                SetWindowLong(hwnd, DWL_MSGRESULT, -1);
                return TRUE;
            }

            reentrantPing = TRUE;

            if (NULL != info->serverName) {
                free(info->serverName);
                info->serverName = NULL;
            }
            info->serverName = AGConfigWindowTextDup(GetDlgItem(hwnd,
                IDC_SERVER_WIZ_URL_EDIT));

            stringConstant =
                (char*)AGSyncCommonGetStringConstant(agLocationConfigPath,
                FALSE);
            locationConfig = AGReadLocationConfigFromDisk(stringConstant);
            free(stringConstant);

            if (NULL != info->serverConfig) {
                AGServerConfigFree(info->serverConfig);
                info->serverConfig = NULL;
            }

            LoadString(g_hInstance, IDS_PINGING_SERVER, buffer, MAX_PATH);
            sprintf(sbuffer,
                buffer,
                info->serverName,
                info->serverPort);
            progressDialog = AGProgressDialogCreate(hwnd, sbuffer);

            info->serverConfig = AGServerPingDoPing(info->serverName,
                info->serverPort,
                locationConfig);

            AGProgressDialogStopMotion(progressDialog);

            AGProgressDialogTerminate(progressDialog);

            AGLocationConfigFree(locationConfig);

            reentrantPing = FALSE;

            break;
        }
        default:
            return FALSE;

    }

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static void introOnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (codeNotify) {
        case EN_CHANGE:
            switch (id) {
                case IDC_SERVER_WIZ_URL_EDIT: {
                    AGConfigUtilCheckWizButtons(1, &hwndCtl, PSWIZB_BACK);
                    return;
                }
                break;
            }
            break;
        case BN_CLICKED:
            switch (id) {
                case IDC_EXPERT:
                    info->userPressedExpert = TRUE;
                    PropSheet_PressButton(GetParent(hwnd), PSBTN_CANCEL);               
                break;
            }
            break;
        default:
            break;
    }
}

/* ----------------------------------------------------------------------------
*/
static BOOL APIENTRY intro(HWND hDlg, UINT message, UINT wParam, LONG lParam)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hDlg, GWL_USERDATA);

    switch (message)
    {
        HANDLE_MSG(hDlg, WM_INITDIALOG, introOnInitDialog);
        HANDLE_MSG(hDlg, WM_COMMAND, introOnCommand);
        case WM_NOTIFY:
            return introOnNotify(hDlg, (NMHDR*)lParam);
    }
    return FALSE;
}

/* ----------------------------------------------------------------------------
*/
static void addressCheckSowFields(HWND hwnd, swWindowInfo * info)
{
    char buf[32];
    char buffer[MAX_PATH];

    if (NULL == info->serverConfig) {

        LoadString(g_hInstance,
            IDS_SERVER_WIZ_PING_FAILED,
            buffer,
            MAX_PATH);

        SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_URL_EDIT),
            info->serverName);
        itoa(info->serverPort, buf, 10);
        SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_PORT_EDIT), buf);

        info->serverConfig = AGServerConfigNew();

    } else {

        LoadString(g_hInstance,
            IDS_SERVER_WIZ_PING_SUCCEEDED,
            buffer,
            MAX_PATH);

        SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_NAME_EDIT),
            info->serverConfig->friendlyName);
        SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_URL_EDIT),
            info->serverConfig->serverName);
        itoa(info->serverConfig->serverPort, buf, 10);
        SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_PORT_EDIT), buf);

    }
    SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_SUCCESS_STATIC),
        buffer);

}

/* ----------------------------------------------------------------------------
*/
static BOOL addressCheckOnInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
    PROPSHEETPAGE * psp = (PROPSHEETPAGE *)lParam;
    SetWindowLong(hwnd, GWL_USERDATA, psp->lParam);

    return TRUE;
}

static void addressCheckButtons(HWND hwnd)
{
    HWND hwnds[3];

    hwnds[0] = GetDlgItem(hwnd, IDC_SERVER_WIZ_NAME_EDIT);
    hwnds[1] = GetDlgItem(hwnd, IDC_SERVER_WIZ_URL_EDIT);
    hwnds[2] = GetDlgItem(hwnd, IDC_SERVER_WIZ_PORT_EDIT);

    AGConfigUtilCheckWizButtons(3, hwnds, 0);
}

/* ----------------------------------------------------------------------------
*/
static void addressCheckDoOK(HWND hwnd, swWindowInfo * info)
{
    char buf[32];
    HWND hwndControl;

    if (NULL != info->serverConfig->friendlyName)
        free(info->serverConfig->friendlyName);
    info->serverConfig->friendlyName =
        AGConfigWindowTextDup(GetDlgItem(hwnd,
        IDC_SERVER_WIZ_NAME_EDIT));

    if (NULL != info->serverConfig->serverName)
        free(info->serverConfig->serverName);
    info->serverConfig->serverName =
        AGConfigWindowTextDup(GetDlgItem(hwnd,
        IDC_SERVER_WIZ_URL_EDIT));

    hwndControl = GetDlgItem(hwnd, IDC_SERVER_WIZ_PORT_EDIT);
    GetWindowText(hwndControl, buf, 31);
    info->serverConfig->serverPort = atoi(buf);
}
       
/* ----------------------------------------------------------------------------
*/
static LRESULT addressCheckOnNotify(HWND hwnd, NMHDR * nmh)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (nmh->code) {

        case PSN_KILLACTIVE:
            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);
            break;

        case PSN_RESET:
            /* reset to the original values */
            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);
            break;

        case PSN_SETACTIVE:
            addressCheckButtons(hwnd);
            addressCheckSowFields(hwnd, info);
            break;

        case PSN_WIZNEXT: {
            /* the Next button was pressed */

            AGBool nextOK = TRUE;

            addressCheckDoOK(hwnd, info);

            if (AGConfigUtilIsServerInList(info->userConfigPath,
                info->serverConfig->serverName,
                info->serverConfig->serverPort)) {

                AGConfigUtilDisplayDuplicateErrorMessage(hwnd, TRUE);
                nextOK = FALSE;

            }

            if (nextOK)
                SetWindowLong(hwnd, DWL_MSGRESULT, 0);
            else
                SetWindowLong(hwnd, DWL_MSGRESULT, -1);

            break;
        }

        case PSN_WIZBACK: {
            /* the Back button was pressed */
            break;
        }

        default:
            return FALSE;

    }

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static void addressCheckOnCommand(HWND hwnd,
                                  int id,
                                  HWND hwndCtl,
                                  UINT codeNotify)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (codeNotify) {
        case EN_CHANGE:
            addressCheckButtons(hwnd);
            return;
        default:
            break;
    }
}

/* ----------------------------------------------------------------------------
*/
static BOOL APIENTRY addressCheck(HWND hDlg, UINT message,
                                  UINT wParam, LONG lParam)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hDlg, GWL_USERDATA);

    switch (message)
    {
        HANDLE_MSG(hDlg, WM_INITDIALOG, addressCheckOnInitDialog);
        HANDLE_MSG(hDlg, WM_COMMAND, addressCheckOnCommand);
        case WM_NOTIFY:
            return addressCheckOnNotify(hDlg, (NMHDR*)lParam);
    }
    return FALSE;
}

/* ----------------------------------------------------------------------------
*/
static BOOL usernameAndPasswordOnInitDialog(HWND hwnd,
                                            HWND hwndFocus,
                                            LPARAM lParam)
{
    PROPSHEETPAGE * psp = (PROPSHEETPAGE *)lParam;
    SetWindowLong(hwnd, GWL_USERDATA, psp->lParam);

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static LRESULT usernameAndPasswordOnNotify(HWND hwnd, NMHDR * nmh)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (nmh->code) {

        case PSN_KILLACTIVE:
            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);
            break;

        case PSN_RESET:
            /* reset to the original values */
            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);
            break;

        case PSN_SETACTIVE: {
            HWND h = GetDlgItem(hwnd, IDC_SERVER_WIZ_USERNAME_EDIT);
            AGConfigUtilCheckWizButtons(1, &h, 0);
            SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_USERNAME_EDIT),
                info->serverConfig->userName);
            SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_PASSWORD_EDIT),
                "");
            break;
        }

        case PSN_WIZNEXT: {
            /* the Next button was pressed */
            char plaintext[MAX_PATH];

            if (NULL != info->serverConfig->userName)
                free(info->serverConfig->userName);
            info->serverConfig->userName =
                AGConfigWindowTextDup(GetDlgItem(hwnd,
                IDC_SERVER_WIZ_USERNAME_EDIT));

            GetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_PASSWORD_EDIT),
                plaintext, MAX_PATH);

            if (info->serverConfig->hashPassword) {

                if (strcmp(KEEPPASSWORD, plaintext)) {

                    if (strlen(plaintext) > 0)
                        AGMd5((uint8*)plaintext, strlen(plaintext),
                            info->serverConfig->password);
                    else
                        AGDigestSetToNull(info->serverConfig->password);

                }

            } else {

                if (NULL != info->serverConfig->cleartextPassword)
                    free(info->serverConfig->cleartextPassword);

                if (strlen(plaintext) > 0) {

                    info->serverConfig->cleartextPassword =
                        strdup(plaintext);

                } else {

                    info->serverConfig->cleartextPassword = NULL;

                }

            }

            break;
        }

        case PSN_WIZBACK: {
            /* the Back button was pressed */
            break;
        }

        default:
            return FALSE;

    }

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static void usernameAndPasswordOnCommand(HWND hwnd,
                                         int id,
                                         HWND hwndCtl,
                                         UINT codeNotify)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (codeNotify) {
        case EN_CHANGE:
            switch (id) {
                case IDC_SERVER_WIZ_USERNAME_EDIT: {
                    AGConfigUtilCheckWizButtons(1, &hwndCtl, 0);
                    return;
                }
                break;
            }
            break;
        default:
            break;
    }
}

/* ----------------------------------------------------------------------------
*/
static BOOL APIENTRY usernameAndPassword(HWND hDlg, UINT message,
                                         UINT wParam, LONG lParam)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hDlg, GWL_USERDATA);

    switch (message)
    {
        HANDLE_MSG(hDlg, WM_INITDIALOG, usernameAndPasswordOnInitDialog);
        HANDLE_MSG(hDlg, WM_COMMAND, usernameAndPasswordOnCommand);
        case WM_NOTIFY:
            return usernameAndPasswordOnNotify(hDlg, (NMHDR*)lParam);
    }
    return FALSE;
}

/* ----------------------------------------------------------------------------
*/
static BOOL completeOnInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
    PROPSHEETPAGE * psp = (PROPSHEETPAGE *)lParam;
    SetWindowLong(hwnd, GWL_USERDATA, psp->lParam);

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static char * safeString(char * s)
{
    if (NULL == s)
        return "<none>";
    return s;
}

/* ----------------------------------------------------------------------------
*/
static LRESULT completeOnNotify(HWND hwnd, NMHDR * nmh)
{
    swWindowInfo * info;
    info = (swWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (nmh->code) {

        case PSN_KILLACTIVE:
            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);
            break;

        case PSN_RESET:
            /* reset to the original values */
            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);
            break;

        case PSN_SETACTIVE: {

            char buffer[2048], sourceString[512];

            PropSheet_SetWizButtons(GetParent(hwnd),
                PSWIZB_BACK | PSWIZB_FINISH);

            LoadString(g_hInstance, IDS_SERVER_WIZ_SUMMARY, sourceString, 512);

            sprintf(buffer,
                sourceString,
                safeString(info->serverConfig->friendlyName),
                safeString(info->serverConfig->serverName),
                info->serverConfig->serverPort,
                safeString(info->serverConfig->userName),
                AGDigestNull(info->serverConfig->password)
                ? "<not assigned>"
                : "<assigned>",
                safeString(info->serverConfig->description),
                safeString(info->serverConfig->userUrl));

            SetWindowText(GetDlgItem(hwnd, IDC_SERVER_WIZ_COMPLETE_SUMMARY),
                buffer);

            break;
        }

        case PSN_WIZNEXT: {
            /* the Next button was pressed */
            break;
        }

        case PSN_WIZBACK: {
            /* the Back button was pressed */
            break;
        }

        case PSN_WIZFINISH: {
            /* the Finish button was pressed */
            info->userPressedFinish = TRUE;
            break;
        }
        default:
            return FALSE;

    }

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static BOOL APIENTRY complete(HWND hDlg, UINT message,
                              UINT wParam, LONG lParam)
{
    swWindowInfo * info = (swWindowInfo *)GetWindowLong(hDlg, GWL_USERDATA);

    switch (message)
    {
        HANDLE_MSG(hDlg, WM_INITDIALOG, completeOnInitDialog);
        case WM_NOTIFY:
            return completeOnNotify(hDlg, (NMHDR*)lParam);
    }
    return FALSE;
}

/* ----------------------------------------------------------------------------
*/
int AGConfigServerWizardDo(const HWND hwndOwner,
                           AGServerConfig * serverConfig,
                           char * userConfigPath)
{
    PROPSHEETPAGE psp[NUM_PAGES];
    PROPSHEETHEADER psh;
    swWindowInfo info;
    int result = 0;

    bzero(&psh, sizeof(PROPSHEETHEADER));

    info.userPressedFinish = FALSE;
    info.userPressedExpert = FALSE;
    info.serverConfig = NULL;
    info.serverName = strdup("");
    info.serverPort = AG_DEFAULT_PORT;
    info.userConfigPath = userConfigPath;

    int ppage = 0;
    AGConfigWizardFillInPropertyPage((LPARAM)&info, &psp[ppage++],
        IDD_SERVER_WIZ_INTRO, intro);
    AGConfigWizardFillInPropertyPage((LPARAM)&info, &psp[ppage++],
        IDD_SERVER_WIZ_URL_CHECK, addressCheck);
    AGConfigWizardFillInPropertyPage((LPARAM)&info, &psp[ppage++],
        IDD_SERVER_WIZ_USERNAME_AND_PASSWORD, usernameAndPassword);
    AGConfigWizardFillInPropertyPage((LPARAM)&info, &psp[ppage++],
        IDD_SERVER_WIZ_COMPLETE, complete);

    psh.dwSize = sizeof(PROPSHEETHEADER);
    psh.dwFlags = PSH_PROPSHEETPAGE | PSH_WIZARD | PSH_NOAPPLYNOW;
    psh.hInstance = g_hInstance;
    psh.hIcon = NULL;
    psh.hwndParent = hwndOwner;
    psh.pszCaption = NULL;
    psh.nPages = sizeof(psp) / sizeof(PROPSHEETPAGE);
    psh.nStartPage = 0;
    psh.ppsp = (LPCPROPSHEETPAGE) &psp;
    psh.pfnCallback = NULL;

    ::PropertySheet(&psh);

    if (info.userPressedExpert) {
        result = -1;
    } else if (info.userPressedFinish) {
        AGServerConfigCopy(serverConfig, info.serverConfig);
        result = 1;
    }

    if (NULL != info.serverName) {
        free(info.serverName);
        info.serverName = NULL;
    }

    if (NULL != info.serverConfig)
        AGServerConfigFree(info.serverConfig);

    return result;
}
