/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */
 
/* Owner:  miket */

#include <AGConfigProxySearch.h>
#include <AGLocationConfig.h>
#include <AGConfigDomainExclusionDialog.h>
#include <windows.h>
#include <stdio.h>
namespace netscape {
#include <reg.h> /* mozilla */
#include <nsreg.h> /* mozilla */
}
#include <wininet.h>
#include <AGBase64.h>
#include <crtdbg.h>
#include <AGUtil.h>

static const LPTSTR kProxyRegistryKey  = TEXT("Software\\Mobile Application Link\\proxy"); 
static const LPTSTR kIESettingsKey     = TEXT("Software\\Microsoft\\Windows\\CurrentVersion\\Internet Settings");
static const LPTSTR kUseProxy          = TEXT("UseProxy");
static const LPTSTR kHttpProxy         = TEXT("ProxyServer");
static const LPTSTR kHttpProxyPort     = TEXT("ProxyServerPort");
static const LPTSTR kUseAuth           = TEXT("UseAuth");
static const LPTSTR kUseSocks          = TEXT("UseSocks");
static const LPTSTR kHttpProxyUsername = TEXT("ProxyServerUsername");
static const LPTSTR kHttpProxyPassword = TEXT("ProxyServerPassword");
static const LPTSTR kSocksProxy        = TEXT("SocksServer");
static const LPTSTR kSocksProxyPort    = TEXT("SocksServerPort");
static const LPTSTR DEFAULT_PROXY_PORT  = TEXT("80");
static const LPTSTR DEFAULT_SOCKS_PORT  = TEXT("1080");

#ifndef _WIN32_WCE

/* ----------------------------------------------------------------------------
*/
int findWindowsInetInternetInformation(AGLocationConfig * locationConfig)
{
    typedef HINTERNET (CALLBACK* INETOPEN)(LPCSTR lpszAgent,
        DWORD dwAccessType,
        LPCSTR lpszProxyName,
        LPCSTR lpszProxyBypass,
        DWORD dwFlags);

    typedef BOOL (CALLBACK* INETQUERYOPTION)(HINTERNET hInternet,
        DWORD dwOption,
        LPVOID lpBuffer,
        LPDWORD lpdwBufferLength);

    typedef BOOL (CALLBACK* INETCLOSEHANDLE)(HINTERNET hInet);

    int ret = AG_PROXY_NONE_FOUND;
    HMODULE hInetLib = NULL;
    HINTERNET hInet = NULL;
    INETOPEN InetOpen = NULL;
    INETQUERYOPTION InetQuery = NULL;
    INETCLOSEHANDLE InetClose = NULL;

    __try {

        hInetLib = LoadLibrary(TEXT("WININET.DLL"));

        if (NULL == hInetLib)
            __leave;

        /* Load up the functions we need to find out info from Wininet. */

        InetOpen = (INETOPEN) GetProcAddress(
            hInetLib, TEXT("InternetOpenA"));

#ifdef UNICODE
        InetQuery = (INETQUERYOPTION) GetProcAddress(
            hInetLib, L"InternetQueryOptionW");
#else
        InetQuery = (INETQUERYOPTION) GetProcAddress(
            hInetLib, "InternetQueryOptionA");
#endif /* !UNICODE */

        InetClose = (INETCLOSEHANDLE) GetProcAddress(
            hInetLib, TEXT("InternetCloseHandle"));

        if (NULL == InetOpen || NULL == InetQuery || NULL == InetClose)
            __leave;

        /* Query Wininet for information. */

        hInet = InetOpen("Mobile Link", INTERNET_OPEN_TYPE_PRECONFIG,
            NULL, NULL, 0);

        if (NULL != hInet) {

            const int bufSize = 1024;
            char buffer[bufSize];
            DWORD bufLen;
            HKEY hkey = NULL;

            /* Find out proxy information. */
            bufLen = bufSize;
            if (!InetQuery(NULL, INTERNET_OPTION_PROXY, &buffer, &bufLen))
                __leave;
            
            LPINTERNET_PROXY_INFO inetProxyInfo =
                (LPINTERNET_PROXY_INFO)buffer;

            locationConfig->HTTPUseProxy = FALSE;
            locationConfig->SOCKSUseProxy = FALSE;

            /* Is there an autoproxy URL? */
            RegOpenKeyEx(HKEY_CURRENT_USER,
                kIESettingsKey,
                0,
                KEY_READ,
                &hkey);
            if (NULL != hkey) {

                DWORD dwType;
                LONG hr;

                bufLen = bufSize;
                hr = RegQueryValueEx(hkey,
                    "AutoConfigURL",
                    0,
                    &dwType,
                    (LPBYTE)buffer,
                    &bufLen);
                if (ERROR_SUCCESS == hr) {
                    if (NULL != locationConfig->autoConfigProxyURL) {
                        free(locationConfig->autoConfigProxyURL);
                        locationConfig->autoConfigProxyURL = NULL;
                    }
                    if (strlen(buffer) > 0)
                        locationConfig->autoConfigProxyURL = strdup(buffer);
                }
                RegCloseKey(hkey);
            }

            AGConfigDomainExclusionFreeArray(
                locationConfig->exclusionServers);
            locationConfig->exclusionServers = NULL;

            if (INTERNET_OPEN_TYPE_PROXY == inetProxyInfo->dwAccessType) {
                
                char * cptr, * dptr;
                char sptr[128];
                char buf[MAX_PATH];
                char proxylist[4096];

                /* If there's no qualifier to tell us which kind of
                proxy server this is, then we need to insert one to
                make it an http proxy server. pending(miket): this
                may not be right! It may be that Windows simply tries
                HTTP and SOCKS if one or the other isn't specified. */
                if (NULL == strstr(inetProxyInfo->lpszProxy, "="))
                    strcpy(proxylist, "http=");
                else
                    strcpy(proxylist, "");
                strcat(proxylist, inetProxyInfo->lpszProxy);

                /* Look for an HTTP proxy */
                strcpy(sptr, "http=");
                cptr = strstr(proxylist, sptr);
                if (NULL != cptr) {

                    locationConfig->HTTPUseProxy = TRUE;

                    cptr += strlen(sptr);
                    dptr = buf;
                    while (*cptr != '\0' && *cptr != ' ' && *cptr != ':')
                        *dptr++ = *cptr++;
                    *dptr = '\0';

                    /* Get port number. */
                    if (*cptr == ':') {

                        char number[16];
                        cptr++;
                        dptr = number;
                        while (*cptr != '\0' && *cptr != ' ')
                            *dptr++ = *cptr++;
                        *dptr = '\0';

                        locationConfig->HTTPPort = atoi(number);

                    }

                    if (locationConfig->HTTPUseProxy &&
                        locationConfig->HTTPPort <= 0)
                        locationConfig->HTTPPort = 8080;

                    if (NULL != locationConfig->HTTPName)
                        free(locationConfig->HTTPName);
                    locationConfig->HTTPName = strdup(buf);

                }

                /* Look for a SOCKS proxy */
                strcpy(sptr, "socks=");
                cptr = strstr(proxylist, sptr);
                if (NULL != cptr) {

                    locationConfig->SOCKSUseProxy = TRUE;

                    cptr += strlen(sptr);
                    dptr = buf;
                    while (*cptr != '\0' && *cptr != ' ' && *cptr != ':')
                        *dptr++ = *cptr++;
                    *dptr = '\0';

                    /* Get port number. */
                    if (*cptr == ':') {

                        char number[16];
                        cptr++;
                        dptr = number;
                        while (*cptr != '\0' && *cptr != ' ')
                            *dptr++ = *cptr++;
                        *dptr = '\0';

                        locationConfig->SOCKSPort = atoi(number);

                    }

                    if (locationConfig->SOCKSUseProxy &&
                        locationConfig->SOCKSPort <= 0)
                        locationConfig->SOCKSPort = 1080;

                    if (NULL != locationConfig->SOCKSName)
                        free(locationConfig->SOCKSName);
                    locationConfig->SOCKSName = strdup(buf);
                }

                if (locationConfig->SOCKSUseProxy
                    || locationConfig->HTTPUseProxy) {

                    locationConfig->exclusionServers = 
                        AGConfigDomainExclusionFill((char*)
                        inetProxyInfo->lpszProxyBypass);

                    /* pending(miket):  handle <local> */

                }

            }

            ret = AG_PROXY_NO_ERROR;

            if (NULL != hInet && NULL != InetClose)
                InetClose(hInet);

        }

    }

    __finally {

        if (NULL != hInetLib) 
            ::FreeLibrary(hInetLib);

    }
    
    return ret;

}

/* ----------------------------------------------------------------------------
*/
void getTokenWithoutQuotes(FILE * f, char * buf)
{
    fscanf(f, "%[^\"]", buf);
    fgetc(f);
    fscanf(f, "%[^\"]", buf);
}

/* ----------------------------------------------------------------------------
*/
static void findNetscapeUserPrefFilename(char * buffer, uint32 buflen)
{
    AGBool success = FALSE;
    netscape::HREG hReg = NULL;

    __try {

        if (REGERR_OK != netscape::NR_RegOpen(NULL, &hReg))
            __leave;

        netscape::RKEY rKey;
        char buf[MAX_PATH];

        if (REGERR_OK != netscape::NR_RegGetKey(hReg, ROOTKEY_COMMON,
            "Netscape", &rKey))
            __leave;

        if (REGERR_OK != netscape::NR_RegGetKey(hReg, rKey,
            "ProfileManager", &rKey))
            __leave;

        /* Get name of last user who used Netscape. */
        if (REGERR_OK != netscape::NR_RegGetEntryString(hReg, rKey,
            "LastNetscapeUser", buf, MAX_PATH))
            __leave;

        /* Get profile location for this user. */
        if (REGERR_OK != netscape::NR_RegGetKey(hReg,
            ROOTKEY_USERS,
            buf,
            &rKey))
            __leave;

        if (REGERR_OK != netscape::NR_RegGetEntryString(hReg, rKey,
            "ProfileLocation", buffer, buflen))
            __leave;

        success = TRUE;
    }

    __finally {

        if (NULL != hReg)
            netscape::NR_RegClose(hReg);

    }

    /* Reading 4.5 registry failed.  Try looking in known registry location
    for older browsers. */
    if (!success) {

        HKEY hKey = NULL;
        HKEY hSubKey = NULL;

        __try {

            char buf[MAX_PATH];

            if (ERROR_SUCCESS != RegOpenKeyEx(HKEY_LOCAL_MACHINE,
                "SOFTWARE\\Netscape\\Netscape Navigator\\Users",
                0,
                KEY_READ,
                &hKey))
                __leave;

            DWORD size = MAX_PATH;
            if (ERROR_SUCCESS != RegQueryValueEx(hKey,
                "CurrentUser",
                0, NULL, (LPBYTE)buf, &size))
                __leave;

            if (ERROR_SUCCESS != RegOpenKeyEx(hKey,
                buf,
                0,
                KEY_READ,
                &hSubKey))
                __leave;

            size = buflen;
            if (ERROR_SUCCESS != RegQueryValueEx(hSubKey,
                "DirRoot",
                0, NULL, (LPBYTE)buffer, &size))
                __leave;

            success = TRUE;
        }

        __finally {

            if (NULL != hKey)
                RegCloseKey(hKey);
            if (NULL != hSubKey)
                RegCloseKey(hSubKey);
        }

    }

    if (!success)
        strcpy(buffer, "");
}

/* ----------------------------------------------------------------------------
*/
static int findNetscapeInternetInformation(AGLocationConfig * locationConfig)
{
    int ret = AG_PROXY_NONE_FOUND;

    char buf[MAX_PATH];

    /* Open up Netscape's proprietary registry and query it. */

    findNetscapeUserPrefFilename(buf, MAX_PATH);
    if (strlen(buf) == 0)
        return ret;
    
    /* At this point, we know the location of the prefs file for the user
    who last used Netscape. That's a reasonable guess for picking the
    right prefs file to search for proxy information. */

    strcat(buf, "\\prefs.js");
    FILE * f = fopen(buf, "r");
    int c = 0;
    int httpPort = -1;
    int socksPort = -1;
    int proxyType = 0;
    
    AGConfigDomainExclusionFreeArray(
        locationConfig->exclusionServers);

    locationConfig->exclusionServers = NULL;

    do {

        c = fscanf(f, "%s", &buf);

        if (!strcmp(buf,
            "user_pref(\"network.proxy.type\",")) {
            fscanf(f, "%d", &proxyType);
            continue;
        }

        if (!strcmp(buf, "user_pref(\"network.hosts.socks_server\",")) {
            getTokenWithoutQuotes(f, buf);
            if (NULL != locationConfig->SOCKSName)
                free(locationConfig->SOCKSName);
            locationConfig->SOCKSName = strdup(buf);
            if (socksPort < 0)
                socksPort = 1080;
            locationConfig->SOCKSUseProxy = TRUE;
            continue;
        }

        if (!strcmp(buf, "user_pref(\"network.proxy.http\",")) {
            getTokenWithoutQuotes(f, buf);
            if (NULL != locationConfig->HTTPName)
                free(locationConfig->HTTPName);
            locationConfig->HTTPName = strdup(buf);
            if (httpPort < 0)
                httpPort = 80;
            locationConfig->HTTPUseProxy = TRUE;
            continue;
        }

        if (!strcmp(buf, "user_pref(\"network.proxy.autoconfig_url\",")) {
            getTokenWithoutQuotes(f, buf);
            if (NULL != locationConfig->autoConfigProxyURL)
                free(locationConfig->autoConfigProxyURL);
            locationConfig->autoConfigProxyURL = strdup(buf);
            continue;
        }

        if (!strcmp(buf, "user_pref(\"network.proxy.no_proxies_on\",")) {
            getTokenWithoutQuotes(f, buf);

            locationConfig->exclusionServers = 
                AGConfigDomainExclusionFill(buf);

            continue;
        }

        if (!strcmp(buf,
            "user_pref(\"network.hosts.socks_serverport\",")) {
            fscanf(f, "%d", &socksPort);
            continue;
        }

        if (!strcmp(buf, "user_pref(\"network.proxy.http_port\",")) {
            fscanf(f, "%d", &httpPort);
            continue;
        }

    } while (EOF != c);

    fclose(f);

    /* At this point, we have successfully parsed the Netscape
    user preferences. */
    ret = AG_PROXY_NO_ERROR;

    if (socksPort >= 0)
        locationConfig->SOCKSPort = socksPort;

    if (httpPort >= 0)
        locationConfig->HTTPPort = httpPort;

    /* Note:  Netscape apparently doesn't have a way to specify that
    all local addresses should bypass the proxy, so we just set it false
    in all cases here. */
    locationConfig->bypassLocal = FALSE;

    /* Proxy type zero (which isn't ever put in the source file as far
    as I can tell) means don't use proxy servers.
    
    Proxy type 1 is either socks or HTTP. */
    if (proxyType != 1) {
        locationConfig->HTTPUseProxy = FALSE;
        locationConfig->SOCKSUseProxy = FALSE;
    }

    /* Proxy type 2 means Netscape's automatic proxy configuration. We
    don't support that at present. */
/*    if (proxyType == 2)
        ret = AG_PROXY_NONE_FOUND;*/

    if (!(locationConfig->HTTPUseProxy || locationConfig->SOCKSUseProxy)) {

        AGConfigDomainExclusionFreeArray(
            locationConfig->exclusionServers);

        locationConfig->exclusionServers = NULL;

    }

    return ret;
}

/* ----------------------------------------------------------------------------
*/
int32 AGConfigSearchBrowserSettings(AGLocationConfig * locationConfig)
{
    int ret = AG_PROXY_NONE_FOUND;
    HKEY hKey = NULL;

    __try {

        if (NULL == locationConfig) {
            ret = AG_PROXY_NULL_POINTER;
            __leave;
        }

        char buf[MAX_PATH];

        /* This registry key looks for whoever's registered to handle http
         documents.  In English, this is the default browser. */
        if (ERROR_SUCCESS != RegOpenKeyEx(HKEY_CLASSES_ROOT,
            "http\\shell\\open\\ddeexec\\Application", 0, KEY_EXECUTE, &hKey))
            __leave;

        DWORD size = MAX_PATH;
        if (ERROR_SUCCESS != RegQueryValueEx(hKey,
            NULL, 0, NULL, (LPBYTE)buf, &size))
            __leave;

        int preferredBrowser = -1;

        /* See which browser is the default. */
        if (!strcmp(buf, "NSShell"))
            preferredBrowser = AG_PROXY_Netscape;
        else
            preferredBrowser = AG_PROXY_MSIE;

        enum {
            AG_TRY_PREFERRED,
            AG_TRY_OTHER,
            AG_FOUND,
            AG_NONE
        };

        /* In the following state machine, we cycle until we've (a) found
        the proxy information, or (b) failed with every browser we know
        about.  We try in order of preference. */
        int state = AG_TRY_PREFERRED;
        while (1) {

            switch (state) {
                case AG_TRY_PREFERRED: {
                    if (AG_PROXY_MSIE == preferredBrowser) {
                        locationConfig->source = AG_PROXY_MSIE;
                        ret = findWindowsInetInternetInformation(locationConfig);
                    }
                    else {
                        locationConfig->source = AG_PROXY_Netscape;
                        ret = findNetscapeInternetInformation(locationConfig);
                    }

                    if (AG_PROXY_NO_ERROR == ret)
                    {
                        locationConfig->source =
                            (AG_PROXY_MSIE == preferredBrowser) ?
                            AG_PROXY_MSIE : AG_PROXY_Netscape;
                        state = AG_FOUND;
                    }
                    else
                        state = AG_TRY_OTHER;
                }
                    break;
                case AG_TRY_OTHER: {
                    if (AG_PROXY_MSIE == preferredBrowser) {
                        locationConfig->source = AG_PROXY_Netscape;
                        ret = findNetscapeInternetInformation(locationConfig);
                    }
                    else {
                        locationConfig->source = AG_PROXY_MSIE;
                        ret = findWindowsInetInternetInformation(locationConfig);
                    }

                    if (AG_PROXY_NO_ERROR == ret)
                        state = AG_FOUND;
                    else
                        state = AG_NONE;
                }
                    break;
                case AG_FOUND:
                    __leave;
                    break;
                case AG_NONE:
                    locationConfig->source = AG_PROXY_FAILED;
                    __leave;
                    break;
            }
        }
    }
    __finally {

        if (NULL != hKey)
            RegCloseKey(hKey);

        return ret;
    }
}

#endif

/* ----------------------------------------------------------------------------
*/
int AGConfigLoad(AGLocationConfig * locationConfig)
{
    int ret = 0;
    HKEY hKey = NULL;
    char * buf = NULL;
    int fieldSize = MAX_PATH;
    int dwordbuf;
    DWORD size, type = REG_SZ;

    if (NULL == locationConfig) {
        ret = AG_PROXY_NULL_POINTER;
        goto error;
    }

    /* Open our proxy settings key. */
    ret = RegOpenKeyEx(HKEY_CURRENT_USER,
        kProxyRegistryKey, 0, KEY_EXECUTE, &hKey);

    if (ERROR_SUCCESS != ret)
        goto error;


    buf = (char*)malloc(fieldSize);
    if (NULL == buf)
        goto error;

    AGLocationConfigInit(locationConfig);

    type = REG_DWORD;
    size = sizeof(DWORD);
    dwordbuf = 0;
    RegQueryValueEx(hKey, kUseProxy, 0, &type, 
        (LPBYTE)&dwordbuf, &size);
    locationConfig->HTTPUseProxy = dwordbuf;

    type = REG_SZ;
    size = fieldSize;
    buf[0] = '\0';
    RegQueryValueEx(hKey, kHttpProxy, 0, &type, 
        (LPBYTE)buf, &size);
    locationConfig->HTTPName = strdup(buf);

    type = REG_DWORD;
    size = sizeof(DWORD);
    dwordbuf = 8080;
    RegQueryValueEx(hKey, kHttpProxyPort, 0, &type, 
        (LPBYTE)&dwordbuf, &size);
    locationConfig->HTTPPort = dwordbuf;

    type = REG_DWORD;
    size = sizeof(DWORD);
    dwordbuf = 0;
    RegQueryValueEx(hKey, kUseAuth, 0, &type, 
        (LPBYTE)&dwordbuf, &size);
    locationConfig->HTTPUseAuthentication = dwordbuf;

    type = REG_SZ;
    size = fieldSize;
    buf[0] = '\0';
    RegQueryValueEx(hKey, kHttpProxyUsername, 0, &type, 
        (LPBYTE)buf, &size);
    locationConfig->HTTPUsername = strdup(buf);

    type = REG_SZ;
    size = fieldSize;
    buf[0] = '\0';
    RegQueryValueEx(hKey, kHttpProxyPassword, 0, &type, 
        (LPBYTE)buf, &size);
    /* Note: AGBase64Decode() allocates memory for us. */
    int32 dummy;
    locationConfig->HTTPPassword = (char*)AGBase64Decode(buf, &dummy);

    type = REG_DWORD;
    size = sizeof(DWORD);
    dwordbuf = 0;
    RegQueryValueEx(hKey, kUseSocks, 0, &type, 
        (LPBYTE)&dwordbuf, &size);
    locationConfig->SOCKSUseProxy = dwordbuf;

    type = REG_SZ;
    size = fieldSize;
    buf[0] = '\0';
    RegQueryValueEx(hKey, kSocksProxy, 0, &type, 
        (LPBYTE)buf, &size);
    locationConfig->SOCKSName = strdup(buf);

    type = REG_DWORD;
    size = sizeof(DWORD);
    dwordbuf = 1080;
    RegQueryValueEx(hKey, kSocksProxyPort, 0, &type, 
        (LPBYTE)&dwordbuf, &size);
    locationConfig->SOCKSPort = dwordbuf;

    ret = 0;
error:
    if (NULL != buf)
        free(buf);

    if (NULL != hKey)
        RegCloseKey(hKey);

    return ret;

}

/* ----------------------------------------------------------------------------
*/
int AGConfigSave(AGLocationConfig * locationConfig)
{
    int ret = 0;
    HKEY hKey = NULL;
    char *encodedPWD;

    if (NULL == locationConfig) {
        ret = AG_PROXY_NULL_POINTER;
        goto error;
    }

    /* Open our proxy settings key. */
    ret = RegOpenKeyEx(HKEY_CURRENT_USER,
        kProxyRegistryKey, 0, KEY_ALL_ACCESS, &hKey);

    if (ERROR_SUCCESS != ret)
        goto error;

    ret = RegSetValueEx(hKey, kUseProxy, 0, REG_DWORD, 
        (LPBYTE)&locationConfig->HTTPUseProxy, sizeof(DWORD));

    ret = RegSetValueEx(hKey, kHttpProxy, 0, REG_SZ, 
        (LPBYTE)locationConfig->HTTPName,
        strlen(locationConfig->HTTPName));

    ret = RegSetValueEx(hKey, kHttpProxyPort, 0, REG_DWORD, 
        (LPBYTE)&locationConfig->HTTPPort, sizeof(DWORD));

    ret = RegSetValueEx(hKey, kUseAuth, 0, REG_DWORD, 
        (LPBYTE)&locationConfig->HTTPUseAuthentication, sizeof(DWORD));

    ret = RegSetValueEx(hKey, kHttpProxyUsername, 0, REG_SZ, 
        (LPBYTE)locationConfig->HTTPUsername,
        strlen(locationConfig->HTTPUsername));

    encodedPWD = AGBase64Encode((uint8*)locationConfig->HTTPPassword,
        strlen(locationConfig->HTTPPassword));
    ret = RegSetValueEx(hKey, kHttpProxyPassword, 0, REG_SZ,
        (LPBYTE)encodedPWD, strlen(encodedPWD));
    free(encodedPWD);

    ret = RegSetValueEx(hKey, kUseSocks, 0, REG_DWORD,
        (LPBYTE)&locationConfig->SOCKSUseProxy, sizeof(DWORD));

    ret = RegSetValueEx(hKey, kSocksProxy, 0, REG_SZ, 
        (LPBYTE)locationConfig->SOCKSName,
        strlen(locationConfig->SOCKSName));

    ret = RegSetValueEx(hKey, kSocksProxyPort, 0, REG_DWORD, 
        (LPBYTE)&locationConfig->SOCKSPort, sizeof(DWORD));

    ret = 0;
error:
    if (NULL != hKey)
        RegCloseKey(hKey);

    return ret;

}
