/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <AGConfigMainDialog.h>
#include <AGConfigUtil.h>
#include <AGConfigAddServerDialog.h>
#include <AGConfigServerSettingsDialog.h>
#include <AGConfigServerWizard.h>
#include <AGUserConfig.h>
#include <AGConfigProxyWizard.h>
#include <AGConfigProxyDialog.h>
#include <AGSyncCommon.h>
#include <windowsx.h>
#include <stdio.h>
#include <crtdbg.h>
#include <AGUtil.h>
#include <AGMobileLinkSettings.h>
#include <AGDeviceUserDropDown.h>
#include <AGConnectionTesterUI.h>

typedef struct {
    char * serverName;
    int16 serverPort;
    AGBool contentValid;
} serverListEntry;

typedef struct mdWindowInfo {

    HWND hwnd;
    AGArray * serverArray;
    AGArray * deviceArray;
    HWND toolTipControl;
    AGBool somethingIsSelected;
    AGBool testConnectionAvailable;

} mdWindowInfo;

extern "C" {
    extern HWND hMainDialog;
    extern AGBool selfShotInHead;
}

const AGToolTip tips [] = {

    IDC_ADD, IDS_TIP_ADD,
    IDC_DEVICE_COMBO, IDS_TIP_DEVICE_COMBO,
    IDC_SERVER_LIST, IDS_TIP_SERVER_LIST,
    IDC_PROXY, IDS_TIP_PROXY,
    IDOK, IDS_TIP_MAIN_OK,
    IDC_REMOVE, IDS_TIP_REMOVE,
    IDC_CONTENT, IDS_TIP_CONTENT,
    IDC_CONNECTION, IDS_TIP_CONNECTION

};

/* ----------------------------------------------------------------------------
*/
static void rebuildDeviceList(HWND deviceCombo,
                              mdWindowInfo * pInfo)
{
    if (NULL != pInfo->deviceArray)
        AGDeviceUserDropDownReleaseDeviceList(pInfo->deviceArray);

    pInfo->deviceArray = AGDeviceUserDropDownBuild(deviceCombo, NULL);
}

/* ----------------------------------------------------------------------------
    static AGBool urlIsValid(char * url)

    Returns true only if the url is valid (as defined by Windows function)
    and begins with "http://," which attempts to prevent malicious servers
    from sending down destructive things to open.

*/
static AGBool urlIsValid(char * url)
{
    return (NULL != url && PathIsURL(url) && !strncmp("http://", url, 7));
}

/* ----------------------------------------------------------------------------
*/
static void releaseServerList(AGArray * array)
{
    uint32 i, n;

    n = AGArrayCount(array);

    for (i = 0; i < n; ++i) {

        serverListEntry * se;

        se = (serverListEntry *)AGArrayElementAt(array, i);

        if (NULL != se) {

            if (NULL != se->serverName)
                free(se->serverName);

            free(se);

        }

    }

    if (n > 0)
        AGArrayRemoveAll(array);

    if (NULL != array)
        AGArrayFree(array);
}

/* ----------------------------------------------------------------------------
*/
static int32 addServerToServerList(AGArray * array,
                                   char * name,
                                   int16 port,
                                   AGBool contentValid)
{
    serverListEntry * se;

    se = (serverListEntry *)malloc(sizeof(serverListEntry));

    if (NULL != se) {

        se->serverName = name;
        se->serverPort = port;
        se->contentValid = contentValid;

        AGArrayAppend(array, se);

        return AGArrayIndexOf(array, se, 0);

    }
    else
        return -1;

}

/* ----------------------------------------------------------------------------
*/
static AGBool getCurrentSelectionInfo(HWND hwnd,
                                      mdWindowInfo * pInfo,
                                      char ** serverName,
                                      int16 * serverPort,
                                      AGBool * contentValid)
{
    HWND hListBox;
    serverListEntry * se;
    int index;

    hListBox = GetDlgItem(hwnd, IDC_SERVER_LIST);
    if (NULL == hListBox)
        return FALSE;

    index = ListBox_GetCurSel(hListBox);

    if (LB_ERR == index)
        return FALSE;

    se = (serverListEntry *)AGArrayElementAt(pInfo->serverArray, index);
    if (NULL != serverName)
        *serverName = (NULL != se->serverName)
            ? strdup(se->serverName)
            : strdup("");
    if (NULL != serverPort)
        *serverPort = se->serverPort;
    if (NULL != contentValid)
        *contentValid = se->contentValid;

    return TRUE;
}

static HANDLE imageForServer(serverListEntry *se) {
    static HANDLE image = NULL;

    if (!image) {
        LPTSTR prototype = (LPSTR)AGSyncCommonGetStringConstant(agProgressBitmapFilename, FALSE);

        if (prototype) {
            TCHAR *filename = (TCHAR *)malloc(strlen(prototype) + 10);

            sprintf(filename, prototype, 0);
            image = LoadImage(NULL,
                            filename,
                            IMAGE_BITMAP,
                            0,
                            0,
                            LR_LOADFROMFILE |
                            LR_LOADMAP3DCOLORS |
                            LR_LOADTRANSPARENT);
            free(filename);
            free(prototype);
        }
    }
    return image;
}

/* ----------------------------------------------------------------------------
*/
static void enableServerButtons(HWND hwnd, AGBool addOK)
{
    BOOL enable;
    HWND hListBox;
    AGBool contentValid = FALSE;
    mdWindowInfo * pInfo;
    int selectedServer;

    pInfo = (mdWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    hListBox = GetDlgItem(hwnd, IDC_SERVER_LIST);
    if (NULL == hListBox)
        enable = FALSE;
    else {
        selectedServer = SendMessage(hListBox, LB_GETCURSEL, 0, 0);
        enable = (LB_ERR != selectedServer);
    }

    if (enable) {
        serverListEntry *se;
        HANDLE image;

        getCurrentSelectionInfo(hwnd, pInfo, NULL, NULL, &contentValid);
        se = (serverListEntry *)AGArrayElementAt(pInfo->serverArray, selectedServer);
        image = imageForServer(se);
        if (image) {
            AGUserConfig * userConfig;
            char tempstring[MAX_PATH];
            AGServerConfig *sc;

            AGConfigUtilGetCurrentUserConfigFilename(tempstring, NULL);
            userConfig = AGReadUserConfigFromDisk(tempstring);
            sc = AGUserConfigGetServerByNameAndPort(userConfig, se->serverName, se->serverPort, AGUSERCONFIG_FIND);

            SendMessage(GetDlgItem(hwnd, IDC_SERVERNAME), WM_SETTEXT, 0, (WPARAM)sc->friendlyName);
            SendMessage(GetDlgItem(hwnd, IDC_SERVERIMAGE), STM_SETIMAGE, IMAGE_BITMAP, (LPARAM)image);
            ShowWindow(GetDlgItem(hwnd, IDC_SERVERIMAGE), SW_SHOW);
            AGUserConfigFree(userConfig);
        } else {
            SendMessage(GetDlgItem(hwnd, IDC_SERVERNAME), WM_SETTEXT, 0, (LPARAM)"");
            ShowWindow(GetDlgItem(hwnd, IDC_SERVERIMAGE), SW_HIDE);
        }
    } else {
        SendMessage(GetDlgItem(hwnd, IDC_SERVERNAME), WM_SETTEXT, 0, (LPARAM)"");
        ShowWindow(GetDlgItem(hwnd, IDC_SERVERIMAGE), SW_HIDE);
    }
    UpdateWindow(hwnd);

    EnableWindow(GetDlgItem(hwnd, IDC_ADD), addOK);
    EnableWindow(GetDlgItem(hwnd, IDC_REMOVE), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_CONTENT), contentValid);
    EnableWindow(GetDlgItem(hwnd, IDC_CONNECTION), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_TESTCONN), enable
        & pInfo->testConnectionAvailable);
}

/* ----------------------------------------------------------------------------
*/
static void setupServerList(HWND hwnd, mdWindowInfo * pInfo)
{
    HWND hListBox;
    char tempstring[MAX_PATH];
    AGUserConfig * userConfig;

    hListBox = GetDlgItem(hwnd, IDC_SERVER_LIST);
    if (NULL == hListBox)
        return;
    ListBox_ResetContent(hListBox);

    if (NULL != pInfo->serverArray)
        releaseServerList(pInfo->serverArray);
    pInfo->serverArray = AGArrayNew(AGUnownedPointerElements, 0);
    if (NULL == pInfo->serverArray)
        return;

    AGConfigUtilGetCurrentUserConfigFilename(tempstring, NULL);
    userConfig = AGReadUserConfigFromDisk(tempstring);

    if (NULL != userConfig) {

        AGUserConfigEnumerateState * state;
        AGServerConfig * sc;
        AGBool invalidEntries = FALSE;
        int index = 0;

        state = NULL;

        do {

            sc = AGUserConfigEnumerate(userConfig, &state);

            if (NULL != sc) {

                if (AGServerConfigIsValid(sc)) {

                    char serverListString[MAX_PATH];
                    AGBool contentValid;

                    contentValid = urlIsValid(sc->userUrl);

                    addServerToServerList(
                        pInfo->serverArray,
                        strdup(sc->serverName),
                        sc->serverPort,
                        contentValid);

                    sprintf(serverListString,
                        "%s (%s:%d, %s)",
                        sc->friendlyName,
                        sc->serverName,
                        sc->serverPort,
                        (sc->disabled) ? "disabled" : "enabled");

                    ListBox_InsertString(hListBox,
                        index,
                        (LPARAM)(serverListString));

                    ++index;

                } else {
                    /* pending(miket): we got a bad serverConfig in the
                    list.  Do something with it. */
                    invalidEntries = TRUE;
                }
            }

        } while (NULL != sc);

        if (invalidEntries) {
            /* pending(miket): do something here to inform the user
            that the serverConfig was messed up. */
        }

        AGUserConfigFree(userConfig);
        if (index > 0) {
            ListBox_SetCurSel(hListBox, 0);
            SetFocus(hListBox);
        }

    }

    enableServerButtons(hwnd, pInfo->somethingIsSelected);

    // pending(miket): hack
    ShowWindow(GetDlgItem(hwnd, IDC_DEVICE_COMBO), SW_SHOWNORMAL);
    ShowWindow(GetDlgItem(hwnd, IDC_DEVICE_EDIT), SW_HIDE);

}

/* ----------------------------------------------------------------------------
*/
static BOOL onInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
    mdWindowInfo * pInfo;
    LONG windowStyle;
    int count;
    char * testname;

    SetWindowLong(hwnd, GWL_USERDATA, lParam);
    pInfo = (mdWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    windowStyle = GetWindowLong(hwnd, GWL_EXSTYLE);
    SetWindowLong(hwnd, GWL_EXSTYLE, windowStyle | WS_EX_APPWINDOW);

    hMainDialog = hwnd;

    pInfo->deviceArray =
        AGDeviceUserDropDownBuild(GetDlgItem(hwnd, IDC_DEVICE_COMBO),
        NULL);

    count = AGArrayCount(pInfo->deviceArray);
    if (count < 2) {

        AGBool nodevice = TRUE;

        if (1 == count) {

            if (AG_NO_DEVICE_TYPE !=
                ((AGDeviceEntry *)
                AGArrayElementAt(pInfo->deviceArray, 0))->deviceType) {

                nodevice = FALSE;

            }

        }

        if (nodevice)
            AGDeviceUserDropDownNoDeviceErrorMessage(hwnd);

    }

    pInfo->toolTipControl = AGConfigUtilAddToolTips(hwnd,
        &tips[0],
        tipdim(tips));

    testname = (char*)AGSyncCommonGetStringConstant(agExtensionDLLName, FALSE);
    if (NULL != testname) {
        pInfo->testConnectionAvailable = TRUE;
        free(testname);
    }
    else
        pInfo->testConnectionAvailable = FALSE;

    return FORWARD_WM_INITDIALOG(hwnd, hwndFocus,
        lParam, AGConfigDialogBaseProc);
}

/* ----------------------------------------------------------------------------
*/
static void removeServerFromUserConfigFile(HWND hwnd,
                                           char * serverName,
                                           int16 serverPort)
{
    char tempstring[MAX_PATH];

    AGConfigUtilGetCurrentUserConfigFilename(tempstring, NULL);

    AGConfigUtilRemoveServerFromUserConfigFile(serverName,
        serverPort,
        tempstring);
}

/* ----------------------------------------------------------------------------
*/
static void handleRemoveButton(HWND hwnd, mdWindowInfo * pInfo)
{
    int c, result;
    char buf[MAX_PATH], message[MAX_PATH], title[MAX_PATH];
    char * serverName;
    int16 serverPort;

    if (!getCurrentSelectionInfo(hwnd, pInfo, &serverName, &serverPort, NULL))
        return;

    c = LoadString(g_hInstance, IDS_CONFIRM_SERVER_REMOVAL,
        buf, MAX_PATH);
    _ASSERT(c);

    sprintf(message, buf, serverName);

    c = LoadString(g_hInstance, IDS_CONFIRM_SERVER_REMOVAL_TITLE,
        title, MAX_PATH);
    _ASSERT(c);

    result = MessageBox(hwnd, message, title,
        MB_OKCANCEL | MB_ICONEXCLAMATION);

    if (IDOK == result && !selfShotInHead) {

        removeServerFromUserConfigFile(hwnd, serverName, serverPort);

        setupServerList(hwnd, pInfo);
    }

    if (NULL != serverName)
        free(serverName);

    enableServerButtons(hwnd, pInfo->somethingIsSelected);

}

/* ----------------------------------------------------------------------------
*/
static void handleTestButton(HWND hwnd, mdWindowInfo * pInfo)
{
    char * serverName;
    int16 serverPort;
    AGUserConfig * userConfig;
    AGServerConfig * sc;
    char tempstring[MAX_PATH];

    /* Find out which server's selected. */
    if (!getCurrentSelectionInfo(hwnd, pInfo, &serverName, &serverPort, NULL))
        return;

    /* Read the server into memory. */
    AGConfigUtilGetCurrentUserConfigFilename(tempstring, NULL);
    userConfig = AGReadUserConfigFromDisk(tempstring);
    if (NULL != userConfig) {
        sc = AGServerConfigDup(AGUserConfigGetServerByNameAndPort(userConfig,
            serverName,
            serverPort,
            AGUSERCONFIG_FIND));
        AGUserConfigFree(userConfig);
    }
    if (NULL != serverName)
        free(serverName);

    /* Did we find a serverConfig?  If so, test it. */
    if (NULL != sc) {
        AGConnectionTesterUIDo(hwnd, sc);
        AGServerConfigFree(sc);
    }

}

/* ----------------------------------------------------------------------------
*/
static void handleResetButton(HWND hwnd, mdWindowInfo * pInfo)
{
    char tempstring[MAX_PATH];
    int c, result;
    char message[MAX_PATH], title[MAX_PATH];
    HANDLE f;
    AGUserConfig * userConfig;
    c = LoadString(g_hInstance, IDS_CONFIRM_COOKIE_REMOVAL_TITLE,
        title, MAX_PATH);
    _ASSERT(c);

    c = LoadString(g_hInstance, IDS_CONFIRM_COOKIE_REMOVAL_MESSAGE,
        message, MAX_PATH);
    _ASSERT(c);

    result = MessageBox(hwnd, message, title,
        MB_OKCANCEL | MB_ICONEXCLAMATION);

    if (IDOK != result || selfShotInHead)
        return;

    AGConfigUtilGetCurrentUserConfigFilename(tempstring, NULL);
    userConfig = AGReadUserConfigFromDiskAtomically(tempstring, &f);

    if (NULL != userConfig)
        AGUserConfigResetCookies(userConfig);
    else
        userConfig = AGUserConfigNew();

    userConfig->resetCookiesAtNextSync = TRUE;

    AGWriteUserConfigToDiskAtomically(userConfig, f);

}

/* ----------------------------------------------------------------------------
*/
static void handleAddButton(HWND hwnd, mdWindowInfo * pInfo)
{
    AGServerConfig * sc;
    int32 result;
    char tempstring[MAX_PATH];
    AGBool done = FALSE;

    AGConfigUtilGetCurrentUserConfigFilename(tempstring, NULL);

    sc = (AGServerConfig *)malloc(sizeof(AGServerConfig));

    if (NULL != sc) {

        bzero(sc, sizeof(AGServerConfig));

        while (!done) {

            if (AGMobileLinkGetUseWizards())
                result = AGConfigServerWizardDo(hwnd, sc, tempstring);
            else
                result = AGConfigAddServerDialogDo(hwnd, sc, tempstring);

            if (result == 1)
                done = TRUE;

            if (result == 0)
                done = TRUE;

            if (result == -1)
                AGMobileLinkSetUseWizards(!AGMobileLinkGetUseWizards());

        }

        if (result != 0 && !selfShotInHead) {
            AGConfigUtilAddServerToUserConfigFile(sc, tempstring);
            setupServerList(hwnd, pInfo);
        }
        else {
            AGServerConfigFinalize(sc);
            free(sc);
        }

    }

}
 
/* ----------------------------------------------------------------------------
*/
static void handleProxyButton(HWND hwnd, mdWindowInfo * pInfo)
{
    char * stringPointer;
    AGLocationConfig * lc;
    int result;
    AGBool done = FALSE;

    stringPointer = (char*)AGSyncCommonGetStringConstant(agLocationConfigPath,
        FALSE);
    lc = AGReadLocationConfigFromDisk(stringPointer);
    if (NULL == lc)
        lc = AGLocationConfigNew();

    while (!done) {

        if (AGMobileLinkGetUseWizards())
            result = AGConfigProxyWizardDo(hwnd, lc);
        else
            result = AGConfigProxyDialogDo(hwnd, lc);

        if (result == 1)
            done = TRUE;

        if (result == 0)
            done = TRUE;

        if (result == -1)
            AGMobileLinkSetUseWizards(!AGMobileLinkGetUseWizards());

    }

    if (0 != result && !selfShotInHead)
        AGWriteLocationConfigToDisk(stringPointer, lc);
    AGLocationConfigFree(lc);
    free(stringPointer);
}
 
/* ----------------------------------------------------------------------------
*/
static void handleSettingsButton(HWND hwnd, mdWindowInfo * pInfo)
{
    HANDLE f;
    char tempstring[MAX_PATH];
    char * serverName;
    char * oldUsername = NULL;
    int16 serverPort;
    AGUserConfig * userConfig;
    AGServerConfig * sc;
    int result;

    if (!getCurrentSelectionInfo(hwnd, pInfo, &serverName, &serverPort, NULL))
        return;

    AGConfigUtilGetCurrentUserConfigFilename(tempstring, NULL);
    userConfig = AGReadUserConfigFromDisk(tempstring);

    if (NULL != userConfig) {
        
        sc = AGServerConfigDup(AGUserConfigGetServerByNameAndPort(userConfig,
            serverName,
            serverPort,
            AGUSERCONFIG_FIND));

        oldUsername = strdup(sc->userName);

        AGUserConfigFree(userConfig);

    }

    if (NULL != sc) {
 
        result = AGConfigServerSettingsDialogDo(hwnd, sc);

        if (result == IDOK && !selfShotInHead) {

            /* If the server and port match, just copy the data from the
            new serverConfig to the area of memory occupied by the
            old one. */
            if (serverPort == sc->serverPort
                && !strcmp(serverName, sc->serverName)) {

                AGServerConfig * scToBlowAway;

                userConfig = AGReadUserConfigFromDiskAtomically(tempstring,
                    &f);

                if (NULL != userConfig) {
                    
                    scToBlowAway = AGUserConfigGetServerByNameAndPort(
                        userConfig,
                        serverName,
                        serverPort,
                        AGUSERCONFIG_FIND);

                    /* Check whether the cookie should be reset.
                    This should happen if the user changed the username. */
                    if (NULL != oldUsername) {
                        if (stricmp(sc->userName, oldUsername)) {
                            AGServerConfigResetCookie(sc);
                        }
                    }

                    if (NULL != scToBlowAway) {

                        AGServerConfigFinalize(scToBlowAway);
                        AGServerConfigCopy(scToBlowAway, sc);
                        AGServerConfigFree(sc);

                    }

                }
                else
                    userConfig = AGUserConfigNew();

                AGWriteUserConfigToDiskAtomically(userConfig, f);

                AGUserConfigFree(userConfig);

            }
            else {

                /* The user changed either the servername or the port, or
                both.  We must delete the previous one and add the
                new one. */

                userConfig = AGReadUserConfigFromDiskAtomically(tempstring,
                    &f);

                if (NULL != userConfig) {
                    
                    AGUserConfigRemoveServerFromDesktop(userConfig,
                        serverName, serverPort);
                    
                    AGUserConfigAddServerToDesktop(userConfig, sc);

                }
                else
                    userConfig = AGUserConfigNew();

                /* The cookie should be reset if the user changed the
                servername or port. */
                AGServerConfigResetCookie(sc);

                AGWriteUserConfigToDiskAtomically(userConfig, f);

                AGUserConfigFree(userConfig);

            }

        }
        else
            AGServerConfigFree(sc);

    }

    if (NULL != serverName)
        free(serverName);

    if (NULL != oldUsername)
        free(oldUsername);

    setupServerList(hwnd, pInfo);
}
 
/* ----------------------------------------------------------------------------
*/
static void handleChangeButton(HWND hwnd, mdWindowInfo * pInfo)
{
    char tempstring[MAX_PATH];
    char * serverName;
    int16 serverPort;
    AGUserConfig * userConfig;

    if (!getCurrentSelectionInfo(hwnd, pInfo, &serverName, &serverPort, NULL))
        return;

    AGConfigUtilGetCurrentUserConfigFilename(tempstring, NULL);
    userConfig = AGReadUserConfigFromDisk(tempstring);

    if (NULL != userConfig) {
        
        AGServerConfig * sc;

        sc = AGUserConfigGetServerByNameAndPort(userConfig,
            serverName,
            serverPort,
            AGUSERCONFIG_FIND);

        if (urlIsValid(sc->userUrl))
            ShellExecute(hwnd, "open", sc->userUrl, NULL, NULL, 1);
        else
            handleSettingsButton(hwnd, pInfo);

        AGUserConfigFree(userConfig);

    }

    if (NULL != serverName)
        free(serverName);

}
 
/* ----------------------------------------------------------------------------
*/
static void handleCloseButton(HWND hwnd, mdWindowInfo * pInfo)
{
    PostQuitMessage(0);
}
 
/* ----------------------------------------------------------------------------
*/
static void doServerListSelectionChanged(HWND hwndServerList,
                                         mdWindowInfo * pInfo)
{
    enableServerButtons(GetParent(hwndServerList),
        pInfo->somethingIsSelected);
}

/* ----------------------------------------------------------------------------
*/
static void doDeviceSelectionChanged(HWND hwndDeviceList,
                                     mdWindowInfo * pInfo)
{
    pInfo->somethingIsSelected = AGDeviceUserDropDownHandleSelection(
        hwndDeviceList,
        pInfo->deviceArray);

    setupServerList(GetParent(hwndDeviceList), pInfo);
}

/* ----------------------------------------------------------------------------
*/
static void onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    mdWindowInfo * pInfo = (mdWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (id) {
        case IDOK:
            handleCloseButton(hwnd, pInfo);
            break; // fallthrough to base dialog handler
        case IDC_ADD:
            handleAddButton(hwnd, pInfo);
            return;
        case IDC_PROXY:
            handleProxyButton(hwnd, pInfo);
            return;
        case IDC_REMOVE:
            if (GetAsyncKeyState(VK_CONTROL) & 0x8000)
                handleResetButton(hwnd, pInfo);
            else
                handleRemoveButton(hwnd, pInfo);
            return;
        case IDC_TESTCONN:
            handleTestButton(hwnd, pInfo);
            return;
        case IDC_CONNECTION:
            handleSettingsButton(hwnd, pInfo);
            return;
        case IDC_CONTENT:
            handleChangeButton(hwnd, pInfo);
            return;
        case IDC_SERVER_LIST:
            switch (codeNotify) {
                case LBN_SELCHANGE:
                    doServerListSelectionChanged(hwndCtl, pInfo);
                    break;
                case LBN_DBLCLK:
                    handleChangeButton(hwnd, pInfo);
                    break;
                default:
                    break;
            }
            break;
        case IDC_DEVICE_COMBO:
            switch (codeNotify) {
                case LBN_SELCHANGE:
                    doDeviceSelectionChanged(hwndCtl, pInfo);
                    enableServerButtons(hwnd, pInfo->somethingIsSelected);
                    break;
                case CBN_DROPDOWN:
                    AGDeviceUserDropDownDoSelectionDropDown(
                        pInfo->deviceArray,
                        hwndCtl);
                    break;
                default:
                    break;
            }
            break;
        default:
            break;
    }
    FORWARD_WM_COMMAND(hwnd, id, hwndCtl,
        codeNotify, AGConfigDialogBaseProc);
}

/* ----------------------------------------------------------------------------
*/
static void onDestroy(HWND hwnd)
{
    mdWindowInfo * pInfo = (mdWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);
    PostQuitMessage(0);
    hMainDialog = NULL;
}

/* ----------------------------------------------------------------------------
*/
static void onShowWindow(HWND hwnd, BOOL fShow, UINT status)
{
    if (fShow)
        setupServerList(hwnd,
            (mdWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA));
}

/* ----------------------------------------------------------------------------
*/
static void onActivate(HWND hwnd,
                       UINT state,
                       HWND hwndActDeact,
                       BOOL fMinimized)
{
    if (state == WA_ACTIVE || state == WA_CLICKACTIVE) {

        mdWindowInfo * pInfo;
        HWND deviceCombo;
        
        pInfo = (mdWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

        setupServerList(hwnd, pInfo);

        deviceCombo = GetDlgItem(hwnd, IDC_DEVICE_COMBO);

        AGDeviceUserDropDownSelectCurrentDevice(deviceCombo,
            pInfo->deviceArray);
        doDeviceSelectionChanged(deviceCombo, pInfo);

    }
}

static LRESULT onVKeyToItem(HWND hwnd, UINT vk)
{
    mdWindowInfo * pInfo;

    pInfo = (mdWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (vk) {

        case VK_INSERT:
            handleAddButton(hwnd, pInfo);
            return -2;

        case VK_DELETE:
            handleRemoveButton(hwnd, pInfo);
            return -2;

    }

    return -1;
}

/* ----------------------------------------------------------------------------
*/
static BOOL CALLBACK dialogProc(HWND hwnd, UINT message,
                                WPARAM wParam, LPARAM lParam)
{
    switch (message) {
        HANDLE_MSG(hwnd, WM_INITDIALOG, onInitDialog);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_SHOWWINDOW, onShowWindow);
        HANDLE_MSG(hwnd, WM_ACTIVATE, onActivate); 
        case WM_AG_CONFIG_UPDATE_SERVER_LIST:
            onActivate(hwnd, WA_ACTIVE, NULL, FALSE);
            break;
        case WM_VKEYTOITEM:
            return onVKeyToItem(hwnd, LOWORD(wParam));
    }

    return 0;   /* except for WM_INITDIALOG, returning zero means
                we didn't process the message. */
}

/* ----------------------------------------------------------------------------
*/
int AGConfigMainDialogDo(const HWND parentHwnd)
{
    int ret;
    mdWindowInfo info = { parentHwnd, NULL, NULL, NULL, FALSE };

    ret = DialogBoxParam(g_hInstance,
        MAKEINTRESOURCE(IDD_MAIN),
        parentHwnd,
        dialogProc,
        (LPARAM)&info);

    if (NULL != info.serverArray)
        releaseServerList(info.serverArray);

    if (NULL != info.deviceArray)
        AGDeviceUserDropDownReleaseDeviceList(info.deviceArray);

    if (NULL != info.toolTipControl)
        DestroyWindow(info.toolTipControl);

    return ret;
}

void AGConfigSelectCEUser(HWND hwnd, ATOM name)
{
    char namebuf[MAX_PATH];
    mdWindowInfo * pInfo;
    HWND deviceCombo;

    namebuf[0] = '\0';

    pInfo = (mdWindowInfo *)GetWindowLong(hwnd, GWL_USERDATA);

    deviceCombo = GetDlgItem(hwnd, IDC_DEVICE_COMBO);

    GlobalGetAtomName(name, namebuf, MAX_PATH);

    if (strlen(namebuf) > 0) {

        rebuildDeviceList(deviceCombo, pInfo);

        /* Move current drop down selection to right device. */
        AGDeviceUserDropDownSelectDevice(deviceCombo,
            pInfo->deviceArray,
            namebuf,
            AG_CE_DEVICE_TYPE);

        /* Write that setting to the registry. */
        AGDeviceUserDropDownHandleSelection(deviceCombo,
            pInfo->deviceArray);

    }

}